"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const reference_1 = require("./reference");
const CFN_REFERENCE_SYMBOL = Symbol.for('@aws-cdk/cdk.CfnReference');
/**
 * A Token that represents a CloudFormation reference to another resource
 *
 * If these references are used in a different stack from where they are
 * defined, appropriate CloudFormation `Export`s and `Fn::ImportValue`s will be
 * synthesized automatically instead of the regular CloudFormation references.
 *
 * Additionally, the dependency between the stacks will be recorded, and the toolkit
 * will make sure to deploy producing stack before the consuming stack.
 *
 * This magic happens in the prepare() phase, where consuming stacks will call
 * `consumeFromStack` on these Tokens and if they happen to be exported by a different
 * Stack, we'll register the dependency.
 */
class CfnReference extends reference_1.Reference {
    constructor(value, displayName, target) {
        if (typeof (value) === 'function') {
            throw new Error('Reference can only hold CloudFormation intrinsics (not a function)');
        }
        // prepend scope path to display name
        super(value, `${target.node.id}.${displayName}`, target);
        this.originalDisplayName = displayName;
        this.replacementTokens = new Map();
        this.producingStack = target.node.stack;
        Object.defineProperty(this, CFN_REFERENCE_SYMBOL, { value: true });
    }
    /**
     * Check whether this is actually a Reference
     */
    static isCfnReference(x) {
        return x[CFN_REFERENCE_SYMBOL] === true;
    }
    /**
     * Return the CfnReference for the indicated target
     *
     * Will make sure that multiple invocations for the same target and intrinsic
     * return the same CfnReference. Because CfnReferences accumulate state in
     * the prepare() phase (for the purpose of cross-stack references), it's
     * important that the state isn't lost if it's lazily created, like so:
     *
     *     new Token(() => new CfnReference(...))
     */
    static for(target, attribute) {
        return CfnReference.singletonReference(target, attribute, () => {
            const cfnInstrinsic = attribute === 'Ref' ? { Ref: target.logicalId } : { 'Fn::GetAtt': [target.logicalId, attribute] };
            return new CfnReference(cfnInstrinsic, attribute, target);
        });
    }
    /**
     * Return a CfnReference that references a pseudo referencd
     */
    static forPseudo(pseudoName, scope) {
        return CfnReference.singletonReference(scope, `Pseudo:${pseudoName}`, () => {
            const cfnInstrinsic = { Ref: pseudoName };
            return new CfnReference(cfnInstrinsic, pseudoName, scope);
        });
    }
    /**
     * Get or create the table
     */
    static singletonReference(target, attribKey, fresh) {
        let attribs = CfnReference.referenceTable.get(target);
        if (!attribs) {
            attribs = new Map();
            CfnReference.referenceTable.set(target, attribs);
        }
        let ref = attribs.get(attribKey);
        if (!ref) {
            ref = fresh();
            attribs.set(attribKey, ref);
        }
        return ref;
    }
    resolve(context) {
        // If we have a special token for this consuming stack, resolve that. Otherwise resolve as if
        // we are in the same stack.
        const token = this.replacementTokens.get(context.scope.node.stack);
        if (token) {
            return token.resolve(context);
        }
        else {
            return super.resolve(context);
        }
    }
    /**
     * Register a stack this references is being consumed from.
     */
    consumeFromStack(consumingStack, consumingConstruct) {
        if (this.producingStack && consumingStack.node.root !== this.producingStack.node.root) {
            throw this.newError(`Cannot reference across apps. ` +
                `Consuming and producing stacks must be defined within the same CDK app.`);
        }
        // tslint:disable-next-line:max-line-length
        if (this.producingStack && this.producingStack !== consumingStack && !this.replacementTokens.has(consumingStack)) {
            // We're trying to resolve a cross-stack reference
            consumingStack.addDependency(this.producingStack, `${consumingConstruct.node.path} -> ${this.target.node.path}.${this.originalDisplayName}`);
            this.replacementTokens.set(consumingStack, this.exportValue(this, consumingStack));
        }
    }
    /**
     * Export a Token value for use in another stack
     *
     * Works by mutating the producing stack in-place.
     */
    exportValue(tokenValue, consumingStack) {
        const producingStack = this.producingStack;
        if (producingStack.env.account !== consumingStack.env.account || producingStack.env.region !== consumingStack.env.region) {
            throw this.newError('Can only reference cross stacks in the same region and account.');
        }
        // Ensure a singleton "Exports" scoping Construct
        // This mostly exists to trigger LogicalID munging, which would be
        // disabled if we parented constructs directly under Stack.
        // Also it nicely prevents likely construct name clashes
        const exportsName = 'Exports';
        let stackExports = producingStack.node.tryFindChild(exportsName);
        if (stackExports === undefined) {
            stackExports = new construct_1.Construct(producingStack, exportsName);
        }
        // Ensure a singleton CfnOutput for this value
        const resolved = producingStack.node.resolve(tokenValue);
        const id = 'Output' + JSON.stringify(resolved);
        let output = stackExports.node.tryFindChild(id);
        if (!output) {
            output = new cfn_output_1.CfnOutput(stackExports, id, { value: tokenValue });
        }
        // We want to return an actual FnImportValue Token here, but Fn.importValue() returns a 'string',
        // so construct one in-place.
        return new token_1.Token({ 'Fn::ImportValue': output.obtainExportName() });
    }
}
/**
 * Static table where we keep singleton CfnReference instances
 */
CfnReference.referenceTable = new Map();
exports.CfnReference = CfnReference;
const cfn_output_1 = require("./cfn-output");
const construct_1 = require("./construct");
const token_1 = require("./token");
//# sourceMappingURL=data:application/json;base64,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