"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_element_1 = require("./cfn-element");
const construct_1 = require("./construct");
class CfnOutput extends cfn_element_1.CfnElement {
    /**
     * Creates an CfnOutput value for this stack.
     * @param scope The parent construct.
     * @param props CfnOutput properties.
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props.value === undefined) {
            throw new Error(`Missing value for CloudFormation output at path "${this.node.path}"`);
        }
        this.description = props.description;
        this._value = props.value;
        this.condition = props.condition;
        this.disableExport = props.disableExport !== undefined ? props.disableExport : false;
        if (props.export && this.disableExport) {
            throw new Error('Cannot set `disableExport` and specify an export name');
        }
        this.export = props.export;
        if (props.export) {
            this.export = props.export;
        }
    }
    /**
     * The value of the property returned by the aws cloudformation describe-stacks command.
     * The value of an output can include literals, parameter references, pseudo-parameters,
     * a mapping value, or intrinsic functions.
     */
    get value() {
        return this._value;
    }
    /**
     * Returns an FnImportValue bound to this export name.
     */
    makeImportValue() {
        return fn().importValue(this.obtainExportName());
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Outputs: {
                [this.logicalId]: {
                    Description: this.description,
                    Value: this.value,
                    Export: this.export != null ? { Name: this.export } : undefined,
                    Condition: this.condition ? this.condition.logicalId : undefined
                }
            }
        };
    }
    get ref() {
        throw new Error('Outputs cannot be referenced');
    }
    /**
     * Allocate an export name for this `CfnOutput` if not already done.
     */
    obtainExportName() {
        if (!this.export && this.disableExport) {
            throw new Error('Cannot create an ImportValue; `disableExport` has been set.');
        }
        if (!this.export) {
            this.export = this.uniqueOutputName();
        }
        return this.export;
    }
    /**
     * Automatically determine an output name for use with FnImportValue
     *
     * This gets called in case the user hasn't specified an export name but is
     * taking an action that requires exporting. We namespace with the stack name
     * to reduce chances of collissions between CDK apps.
     */
    uniqueOutputName() {
        // prefix export name with stack name since exports are global within account + region.
        const stackName = this.node.stack.name;
        return (stackName ? stackName + ':' : '') + this.logicalId;
    }
}
exports.CfnOutput = CfnOutput;
/**
 * An output for a list of strings.
 *
 * Exports a list of Tokens via an CfnOutput variable, and return a list of Tokens
 * that selects the imported values for them.
 */
class StringListCfnOutput extends construct_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.separator = props.separator || ',';
        this.length = props.values.length;
        this.output = new CfnOutput(this, 'Resource', {
            description: props.description,
            condition: props.condition,
            disableExport: props.disableExport,
            export: props.export,
            value: fn().join(this.separator, props.values)
        });
    }
    /**
     * Return an array of imported values for this CfnOutput
     */
    makeImportValues() {
        const combined = this.output.makeImportValue();
        const ret = [];
        for (let i = 0; i < this.length; i++) {
            ret.push(fn().select(i, fn().split(this.separator, combined)));
        }
        return ret;
    }
}
exports.StringListCfnOutput = StringListCfnOutput;
function fn() {
    // Lazy loading of "Fn" module to break dependency cycles on startup
    return require('./fn').Fn;
}
//# sourceMappingURL=data:application/json;base64,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