"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const construct_1 = require("./construct");
const token_1 = require("./token");
const LOGICAL_ID_MD = 'aws:cdk:logicalId';
/**
 * An element of a CloudFormation stack.
 */
class CfnElement extends construct_1.Construct {
    /**
     * Returns `true` if a construct is a stack element (i.e. part of the
     * synthesized cloudformation template).
     *
     * Uses duck-typing instead of `instanceof` to allow stack elements from different
     * versions of this library to be included in the same stack.
     *
     * @returns The construct as a stack element or undefined if it is not a stack element.
     */
    static isCfnElement(construct) {
        return ('logicalId' in construct && '_toCloudFormation' in construct);
    }
    /**
     * Creates an entity and binds it to a tree.
     * Note that the root of the tree must be a Stack object (not just any Root).
     *
     * @param scope The parent construct
     * @param props Construct properties
     */
    constructor(scope, id) {
        super(scope, id);
        this.node.addMetadata(LOGICAL_ID_MD, new (require("./token").Token)(() => this.logicalId), this.constructor);
        this._logicalId = this.node.stack.logicalIds.getLogicalId(this);
        this.logicalId = new token_1.Token(() => this._logicalId, `${notTooLong(this.node.path)}.LogicalID`).toString();
    }
    /**
     * Overrides the auto-generated logical ID with a specific ID.
     * @param newLogicalId The new logical ID to use for this stack element.
     */
    overrideLogicalId(newLogicalId) {
        this._logicalId = newLogicalId;
    }
    /**
     * @returns the stack trace of the point where this Resource was created from, sourced
     *      from the +metadata+ entry typed +aws:cdk:logicalId+, and with the bottom-most
     *      node +internal+ entries filtered.
     */
    get creationStackTrace() {
        const trace = this.node.metadata.find(md => md.type === LOGICAL_ID_MD).trace;
        if (!trace) {
            return undefined;
        }
        return filterStackTrace(trace);
        function filterStackTrace(stack) {
            const result = Array.of(...stack);
            while (result.length > 0 && shouldFilter(result[result.length - 1])) {
                result.pop();
            }
            // It's weird if we filtered everything, so return the whole stack...
            return result.length === 0 ? stack : result;
        }
        function shouldFilter(str) {
            return str.match(/[^(]+\(internal\/.*/) !== null;
        }
    }
    /**
     * Return the path with respect to the stack
     */
    get stackPath() {
        return this.node.ancestors(this.node.stack).map(c => c.node.id).join(construct_1.PATH_SEP);
    }
    /**
     * Automatically detect references in this CfnElement
     */
    prepare() {
        try {
            // Note: it might be that the properties of the CFN object aren't valid.
            // This will usually be preventatively caught in a construct's validate()
            // and turned into a nicely descriptive error, but we're running prepare()
            // before validate(). Swallow errors that occur because the CFN layer
            // doesn't validate completely.
            //
            // This does make the assumption that the error will not be rectified,
            // but the error will be thrown later on anyway. If the error doesn't
            // get thrown down the line, we may miss references.
            this.node.recordReference(...resolve_1.findTokens(this, () => this._toCloudFormation()));
        }
        catch (e) {
            if (e.type !== 'CfnSynthesisError') {
                throw e;
            }
        }
    }
}
exports.CfnElement = CfnElement;
/**
 * Base class for referenceable CloudFormation constructs which are not Resources
 *
 * These constructs are things like Conditions and Parameters, can be
 * referenced by taking the `.ref` attribute.
 *
 * Resource constructs do not inherit from CfnRefElement because they have their
 * own, more specific types returned from the .ref attribute. Also, some
 * resources aren't referenceable at all (such as BucketPolicies or GatewayAttachments).
 */
class CfnRefElement extends CfnElement {
    /**
     * Returns a token to a CloudFormation { Ref } that references this entity based on it's logical ID.
     */
    get ref() {
        return this.referenceToken.toString();
    }
    /**
     * Return a token that will CloudFormation { Ref } this stack element
     */
    get referenceToken() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
}
exports.CfnRefElement = CfnRefElement;
function notTooLong(x) {
    if (x.length < 100) {
        return x;
    }
    return x.substr(0, 47) + '...' + x.substr(x.length - 47);
}
const cfn_reference_1 = require("./cfn-reference");
const resolve_1 = require("./resolve");
//# sourceMappingURL=data:application/json;base64,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