import { Construct, IConstruct } from "./construct";
import { Token } from './token';
/**
 * An element of a CloudFormation stack.
 */
export declare abstract class CfnElement extends Construct {
    /**
     * Returns `true` if a construct is a stack element (i.e. part of the
     * synthesized cloudformation template).
     *
     * Uses duck-typing instead of `instanceof` to allow stack elements from different
     * versions of this library to be included in the same stack.
     *
     * @returns The construct as a stack element or undefined if it is not a stack element.
     */
    static isCfnElement(construct: IConstruct): construct is CfnElement;
    /**
     * The logical ID for this CloudFormation stack element. The logical ID of the element
     * is calculated from the path of the resource node in the construct tree.
     *
     * To override this value, use `overrideLogicalId(newLogicalId)`.
     *
     * @returns the logical ID as a stringified token. This value will only get
     * resolved during synthesis.
     */
    readonly logicalId: string;
    private _logicalId;
    /**
     * Creates an entity and binds it to a tree.
     * Note that the root of the tree must be a Stack object (not just any Root).
     *
     * @param scope The parent construct
     * @param props Construct properties
     */
    constructor(scope: Construct, id: string);
    /**
     * Overrides the auto-generated logical ID with a specific ID.
     * @param newLogicalId The new logical ID to use for this stack element.
     */
    overrideLogicalId(newLogicalId: string): void;
    /**
     * @returns the stack trace of the point where this Resource was created from, sourced
     *      from the +metadata+ entry typed +aws:cdk:logicalId+, and with the bottom-most
     *      node +internal+ entries filtered.
     */
    readonly creationStackTrace: string[] | undefined;
    /**
     * Return the path with respect to the stack
     */
    readonly stackPath: string;
    /**
     * Automatically detect references in this CfnElement
     */
    protected prepare(): void;
}
/**
 * Base class for referenceable CloudFormation constructs which are not Resources
 *
 * These constructs are things like Conditions and Parameters, can be
 * referenced by taking the `.ref` attribute.
 *
 * Resource constructs do not inherit from CfnRefElement because they have their
 * own, more specific types returned from the .ref attribute. Also, some
 * resources aren't referenceable at all (such as BucketPolicies or GatewayAttachments).
 */
export declare abstract class CfnRefElement extends CfnElement {
    /**
     * Returns a token to a CloudFormation { Ref } that references this entity based on it's logical ID.
     */
    readonly ref: string;
    /**
     * Return a token that will CloudFormation { Ref } this stack element
     */
    readonly referenceToken: Token;
}
