"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fn_1 = require("./fn");
const token_1 = require("./token");
const util_1 = require("./util");
/**
 * Creates an ARN from components.
 *
 * If `partition`, `region` or `account` are not specified, the stack's
 * partition, region and account will be used.
 *
 * If any component is the empty string, an empty string will be inserted
 * into the generated ARN at the location that component corresponds to.
 *
 * The ARN will be formatted as follows:
 *
 *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}{resource-name}
 *
 * The required ARN pieces that are omitted will be taken from the stack that
 * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
 * can be 'undefined'.
 */
function arnFromComponents(components, stack) {
    const partition = components.partition !== undefined ? components.partition : stack.partition;
    const region = components.region !== undefined ? components.region : stack.region;
    const account = components.account !== undefined ? components.account : stack.accountId;
    const sep = components.sep !== undefined ? components.sep : '/';
    const values = ['arn', ':', partition, ':', components.service, ':', region, ':', account, ':', components.resource];
    if (sep !== '/' && sep !== ':' && sep !== '') {
        throw new Error('resourcePathSep may only be ":", "/" or an empty string');
    }
    if (components.resourceName != null) {
        values.push(sep);
        values.push(components.resourceName);
    }
    return values.join('');
}
exports.arnFromComponents = arnFromComponents;
/**
 * Given an ARN, parses it and returns components.
 *
 * If the ARN is a concrete string, it will be parsed and validated. The
 * separator (`sep`) will be set to '/' if the 6th component includes a '/',
 * in which case, `resource` will be set to the value before the '/' and
 * `resourceName` will be the rest. In case there is no '/', `resource` will
 * be set to the 6th components and `resourceName` will be set to the rest
 * of the string.
 *
 * If the ARN includes tokens (or is a token), the ARN cannot be validated,
 * since we don't have the actual value yet at the time of this function
 * call. You will have to know the separator and the type of ARN. The
 * resulting `ArnComponents` object will contain tokens for the
 * subexpressions of the ARN, not string literals. In this case this
 * function cannot properly parse the complete final resourceName (path) out
 * of ARNs that use '/' to both separate the 'resource' from the
 * 'resourceName' AND to subdivide the resourceName further. For example, in
 * S3 ARNs:
 *
 *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
 *
 * After parsing the resourceName will not contain
 * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
 * because there is no slicing functionality in CloudFormation templates.
 *
 * @param sep The separator used to separate resource from resourceName
 * @param hasName Whether there is a name component in the ARN at all. For
 * example, SNS Topics ARNs have the 'resource' component contain the topic
 * name, and no 'resourceName' component.
 *
 * @returns an ArnComponents object which allows access to the various
 * components of the ARN.
 *
 * @returns an ArnComponents object which allows access to the various
 *      components of the ARN.
 */
function parseArn(arn, sepIfToken = '/', hasName = true) {
    if (token_1.Token.isToken(arn)) {
        return parseToken(arn, sepIfToken, hasName);
    }
    const components = arn.split(':');
    if (components.length < 6) {
        throw new Error('ARNs must have at least 6 components: ' + arn);
    }
    const [arnPrefix, partition, service, region, account, sixth, ...rest] = components;
    if (arnPrefix !== 'arn') {
        throw new Error('ARNs must start with "arn:": ' + arn);
    }
    if (!service) {
        throw new Error('The `service` component (3rd component) is required: ' + arn);
    }
    if (!sixth) {
        throw new Error('The `resource` component (6th component) is required: ' + arn);
    }
    let resource;
    let resourceName;
    let sep;
    let sepIndex = sixth.indexOf('/');
    if (sepIndex !== -1) {
        sep = '/';
    }
    else if (rest.length > 0) {
        sep = ':';
        sepIndex = -1;
    }
    if (sepIndex !== -1) {
        resource = sixth.substr(0, sepIndex);
        resourceName = sixth.substr(sepIndex + 1);
    }
    else {
        resource = sixth;
    }
    if (rest.length > 0) {
        if (!resourceName) {
            resourceName = '';
        }
        else {
            resourceName += ':';
        }
        resourceName += rest.join(':');
    }
    // "|| undefined" will cause empty strings to be treated as "undefined"
    return util_1.filterUndefined({
        service: service || undefined,
        resource: resource || undefined,
        partition: partition || undefined,
        region: region || undefined,
        account: account || undefined,
        resourceName,
        sep
    });
}
exports.parseArn = parseArn;
/**
 * Given a Token evaluating to ARN, parses it and returns components.
 *
 * The ARN cannot be validated, since we don't have the actual value yet
 * at the time of this function call. You will have to know the separator
 * and the type of ARN.
 *
 * The resulting `ArnComponents` object will contain tokens for the
 * subexpressions of the ARN, not string literals.
 *
 * WARNING: this function cannot properly parse the complete final
 * resourceName (path) out of ARNs that use '/' to both separate the
 * 'resource' from the 'resourceName' AND to subdivide the resourceName
 * further. For example, in S3 ARNs:
 *
 *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
 *
 * After parsing the resourceName will not contain 'path/to/exampleobject.png'
 * but simply 'path'. This is a limitation because there is no slicing
 * functionality in CloudFormation templates.
 *
 * @param arnToken The input token that contains an ARN
 * @param sep The separator used to separate resource from resourceName
 * @param hasName Whether there is a name component in the ARN at all.
 * For example, SNS Topics ARNs have the 'resource' component contain the
 * topic name, and no 'resourceName' component.
 * @returns an ArnComponents object which allows access to the various
 * components of the ARN.
 */
function parseToken(arnToken, sep = '/', hasName = true) {
    // Arn ARN looks like:
    // arn:partition:service:region:account-id:resource
    // arn:partition:service:region:account-id:resourcetype/resource
    // arn:partition:service:region:account-id:resourcetype:resource
    // We need the 'hasName' argument because {Fn::Select}ing a nonexistent field
    // throws an error.
    const components = fn_1.Fn.split(':', arnToken);
    const partition = fn_1.Fn.select(1, components).toString();
    const service = fn_1.Fn.select(2, components).toString();
    const region = fn_1.Fn.select(3, components).toString();
    const account = fn_1.Fn.select(4, components).toString();
    if (sep === ':') {
        const resource = fn_1.Fn.select(5, components).toString();
        const resourceName = hasName ? fn_1.Fn.select(6, components).toString() : undefined;
        return { partition, service, region, account, resource, resourceName, sep };
    }
    else {
        const lastComponents = fn_1.Fn.split(sep, fn_1.Fn.select(5, components));
        const resource = fn_1.Fn.select(0, lastComponents).toString();
        const resourceName = hasName ? fn_1.Fn.select(1, lastComponents).toString() : undefined;
        return { partition, service, region, account, resource, resourceName, sep };
    }
}
//# sourceMappingURL=data:application/json;base64,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