"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const construct_1 = require("./construct");
const runtime_info_1 = require("./runtime-info");
const synthesis_1 = require("./synthesis");
const APP_SYMBOL = Symbol.for('@aws-cdk/cdk.App');
/**
 * Represents a CDK program.
 */
class App extends construct_1.Construct {
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Initializes a CDK application.
     * @param request Optional toolkit request (e.g. for tests)
     */
    constructor(props = {}) {
        super(undefined, '');
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        // both are reverse logic
        this.runtimeInfo = this.node.getContext(cxapi.DISABLE_VERSION_REPORTING) ? false : true;
        this.outdir = props.outdir || process.env[cxapi.OUTDIR_ENV];
        const autoRun = props.autoRun !== undefined ? props.autoRun : cxapi.OUTDIR_ENV in process.env;
        if (autoRun) {
            // run() guarantuees it will only execute once, so a default of 'true' doesn't bite manual calling
            // of the function.
            process.once('beforeExit', () => this.run());
        }
    }
    /**
     * Runs the program. Output is written to output directory as specified in the
     * request.
     *
     * @returns a `CloudAssembly` which includes all the synthesized artifacts
     * such as CloudFormation templates and assets.
     */
    run() {
        // this app has already been executed, no-op for you
        if (this._assembly) {
            return this._assembly;
        }
        const synth = new synthesis_1.Synthesizer();
        const assembly = synth.synthesize(this, {
            outdir: this.outdir,
            runtimeInfo: this.runtimeInfo ? runtime_info_1.collectRuntimeInformation() : undefined
        });
        this._assembly = assembly;
        return assembly;
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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