__all__ = ['expand_path']

import os
import re
import subprocess
import tempfile
import pandas as pd

EPLUS_EXEC = "energyplus-9.4.0"


IDF_GENERATED_FILE_NAME = "generated_input.idf"

def expand_path(path):
    """Expand `~` constructs and return an absolute path.

    On Unix operating systems (Linux, MacOSX, ...),
    `~` is a shortcut for the curent user's home directory
    (e.g. "/home/jack" for "jack").

    Example
    -------

    Lets assume we are on a Unix system,
    the current user is "jack" and the current path is "/bar".

    >>> expand_path("~/foo")
    "/home/jack/foo"

    >>> expand_path("baz")
    "/bar/baz"

    >>> expand_path("../tmp")
    "/tmp"

    Parameters
    ----------
    path : str
        The `path` to expand.

    Returns
    -------
    str
        The absolute and expanded `path`.
    """
    if path is not None:
        path = os.path.expanduser(path)  # to handle "~/..." paths
        path = os.path.abspath(path)     # to handle relative paths
        return path
    else:
        return None


def make_idf_and_run_eplus(orig_idf_path, weather_file_path, sub_dict={}):

    orig_idf_path = expand_path(orig_idf_path)

    # READ THE ORIGINAL INPUT FILE ############################################

    idf_str = None

    with open(orig_idf_path) as fd:
        idf_str = fd.read()

    # MODIFY THE INPUT FILE ###################################################

    for (section, field), value in sub_dict.items():
        idf_str = re_substitute(idf_str, section, field, value)

    # WRITE THE MODIFIED INPUT FILE ###########################################

    home_path = expand_path("~")

    with tempfile.TemporaryDirectory(dir=home_path, prefix=".", suffix="_test") as temp_dir_path:
        dst_idf_path = os.path.join(temp_dir_path, IDF_GENERATED_FILE_NAME)

        print(dst_idf_path)

        with open(dst_idf_path, "w") as fd:
            fd.write(idf_str)

        print(idf_str)

        df = run_eplus(dst_idf_path, weather_file_path=weather_file_path)
    
    return df


def re_substitute(s, section, field, value):
    return re.sub(r'(?P<begin>' + section + r'.*?)\d+.\d*(?P<end>, +!- ' + field + r')', r'\g<begin>' + str(value) + r'\g<end>,', s, flags=re.MULTILINE|re.DOTALL)


def run_eplus(idf_file_path, weather_file_path, output_directory_path="."):
    """
    energyplus --help
    EnergyPlus, Version 9.4.0-998c4b761e
    PythonLinkage: Linked to Python Version: "3.6.9 (default, Jul 17 2020, 12:50:27) 
    [GCC 8.4.0]"
    Usage: energyplus [options] [input-file]
    Options:
    -a, --annual                 Force annual simulation
    -c, --convert                Output IDF->epJSON or epJSON->IDF, dependent on
                                input file type
    -D, --design-day             Force design-day-only simulation
    -d, --output-directory ARG   Output directory path (default: current
                                directory)
    -h, --help                   Display help information
    -i, --idd ARG                Input data dictionary path (default: Energy+.idd
                                in executable directory)
    -m, --epmacro                Run EPMacro prior to simulation
    -p, --output-prefix ARG      Prefix for output file names (default: eplus)
    -r, --readvars               Run ReadVarsESO after simulation
    -s, --output-suffix ARG      Suffix style for output file names (default: L)
                                    L: Legacy (e.g., eplustbl.csv)
                                    C: Capital (e.g., eplusTable.csv)
                                    D: Dash (e.g., eplus-table.csv)
    -v, --version                Display version information
    -w, --weather ARG            Weather file path (default: in.epw in current
                                directory)
    -x, --expandobjects          Run ExpandObjects prior to simulation
    --convert-only                 Only convert IDF->epJSON or epJSON->IDF,
                                dependent on input file type. No simulation
    Example: energyplus -w weather.epw -r input.idf

    """

    idf_file_path = expand_path(idf_file_path)
    weather_file_path = expand_path(weather_file_path)
    output_directory_path = expand_path(output_directory_path)

    cmd = [
            EPLUS_EXEC,
            "-w", weather_file_path,
            "-d", output_directory_path,
            "-r",
            idf_file_path
          ]

    print(" ".join(cmd))

    subprocess.call(cmd)

    df = pd.read_csv(os.path.join(output_directory_path, "eplusout.csv"))

    return df