"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SPADeploy = void 0;
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config !== 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false,
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config, isForCloudFront) {
        const bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true,
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true || isForCloudFront === true) {
            bucketConfig.publicReadAccess = false;
            if (typeof config.blockPublicAccess !== 'undefined') {
                bucketConfig.blockPublicAccess = config.blockPublicAccess;
            }
        }
        const bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true && isForCloudFront === false) {
            if (typeof this.globalConfig.ipList === 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(`${bucket.bucketArn}/*`);
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList,
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, accessIdentity, cert) {
        const cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket,
                        originAccessIdentity: accessIdentity,
                    },
                    behaviors: config.cfBehaviors ? config.cfBehaviors : [{ isDefaultBehavior: true }],
                },
            ],
            // We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                }],
        };
        if (typeof config.certificateARN !== 'undefined' && typeof config.cfAliases !== 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases,
            };
        }
        if (typeof config.zoneName !== 'undefined' && typeof cert !== 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website.
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config, false);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        const cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl,
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName === 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
        return { websiteBucket };
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution It will also setup error forwarding and unauth forwarding back to indexDoc.
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.domainName,
        });
        return { websiteBucket, distribution };
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto configured by using the details in the hosted zone provided.
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        const cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName: config.zoneName,
            region: 'us-east-1',
        });
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: config.zoneName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution)),
        });
        new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
            zone,
            recordNames: [`www.${config.zoneName}`],
            targetDomain: config.zoneName,
        });
        return { websiteBucket, distribution };
    }
}
exports.SPADeploy = SPADeploy;
//# sourceMappingURL=data:application/json;base64,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