# cmake-build-extension

Setuptools extension to build and package CMake projects.

A project `MyProject` with C++ files that can be compiled with CMake (here `MyClass.{h,cpp}`) 
and a Python package (here `my_python_package`) can be structured as follows:

```
myproject/
|-- CMakeLists.txt
|-- include
|   `-- MyClass.h
|-- python
|   `-- my_python_package
|       `-- __init__.py
|-- setup.py
`-- src
    `-- MyClass.cpp
```

The C++ and Python could be independent, even though this hybrid setup usually
applies to project that want to provide Python bindings of the C++ libraries.

This extension enables the following:

- Streamlines the interaction of Python packaging and CMake.
- Enables building and packaging a generic CMake project through `setuptools`.
- Simplifies distributing and consuming C++ / Python hybrid projects.

The following lines of the `setup.py` script add a `build_ext` step to the pipeline that
compiles and installs the CMake project in the root of the resulting Python package:

```python
from setuptools import find_packages, setup
from cmake_build_extension import BuildExtension, CMakeExtension

setup(
    name="my-python-package",
    packages=find_packages("python"),
    package_dir={'': "python"},
    # ...
    ext_modules=[
        CMakeExtension(name="MyProject",
                       install_prefix="my_python_package",
                       cmake_configure_options=[
                           "-DBUILD_SHARED_LIBS:BOOL=OFF",
                       ]),
    ],
    cmdclass=dict(build_ext=BuildExtension),
)
```

Note: If CMake also exports the targets, downstream projects can import them by 
adding the directory of the installed Python package to `CMAKE_PREFIX_PATH`.

The resulting Python install tree will be something similar to what follows:

```
site-packages/
`-- my_python_package
    |-- __init__.py
    `-- lib
        |-- cmake
        |   |-- MyProject.cmake
        |   |-- MyProjectTargets.cmake
        |   `-- MyProjectVersion.cmake
        `-- libMyProject.so
```

This way, if the C++ project also includes Python bindings, they can be generated by your favourite tool
and handled with CMake. This is often convenient since CMake typically has powerful modules that
simplify the building process, especially for big projects.

Check out [`examples/swig`](examples/swig) for a minimal working example.

### Usage

Once both CMake project and `setup.py` are correctly configured, the following
commands can be used:

```bash
# Install with pip
pip install .
pip install --editable .

# Install with setup.py
python setup.py install .
python setup.py develop .

# Install with pip passing custom CMake options
pip install --global-option="build_ext" --global-option="-DBAR=Foo;VAR=TRUE" .

# Install with setup.py passing custom CMake options
python setup.py install build_ext -D"BAR=Foo;VAR=TRUE"

# Create wheel with pip
pip wheel -w dist .

# Create wheel with setup.py
python setup.py bdist_wheel

# Create wheel with setup.py passing custom CMake options
python setup.py bdist_wheel build_ext -D"BAR=Foo;VAR=TRUE"
```

If the Python project is compliant with PEP517 and PEP518, `pip` will use an isolated environment.
Note that CMake's `find_package` will still find resources from the filesystem and it will
not be isolated. 
