"""
Created on 10 Feb 2023
Stream tests for pygpsclient.spartnreader

@author: semuadmin
"""

import os
import sys
import unittest
from io import StringIO
from pyspartn.exceptions import SPARTNMessageError, SPARTNParseError
from pyspartn.spartnreader import SPARTNReader, SPARTNMessage
from pyspartn.spartntypes_core import ERRRAISE, ERRIGNORE, ERRLOG


class StreamTest(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None
        dirname = os.path.dirname(__file__)
        self.streamSPARTN = open(os.path.join(dirname, "spartn_mqtt.log"), "rb")
        self.streamBADCRC = open(os.path.join(dirname, "spartn_badcrc.log"), "rb")
        self.streamBADPRE = open(os.path.join(dirname, "spartn_badpreamble.log"), "rb")
        self.spartntransport = b"s\x00\x12\xe2\x00|\x10[\x12H\xf5\t\xa0\xb4+\x99\x02\x15\xe2\x05\x85\xb7\x83\xc5\xfd\x0f\xfe\xdf\x18\xbe\x7fv \xc3`\x82\x98\x10\x07\xdc\xeb\x82\x7f\xcf\xf8\x9e\xa3ta\xad"
        self.spartnbadcrc = b"s\x00\x12\xe2\x00|\x10[\x12H\xf5\t\xa0\xb4+\x99\x02\x15\xe2\x05\x85\xb7\x83\xc5\xfd\x0f\xfe\xdf\x18\xbe\x7fv \xc3`\x82\x98\x10\x07\xdc\xeb\x82\x7f\xcf\xf8\x9e\xa3ta\xa1"
        self.badpayload = b"x\x00\x12\xe2\x00|\x10[\x12H\xf5\t\xa0\xb4+\x99\x02\x15"

    def tearDown(self):
        self.streamSPARTN.close()
        self.streamBADCRC.close()
        self.streamBADPRE.close()

    def catchio(self):
        """
        Capture stdout as string.
        """

        self._saved_stdout = sys.stdout
        self._strout = StringIO()
        sys.stdout = self._strout

    def restoreio(self) -> str:
        """
        Return captured output and restore stdout.
        """

        sys.stdout = self._saved_stdout
        return self._strout.getvalue().strip()

    def testSerialize(self):  # test serialize()
        msg1 = SPARTNReader.parse(self.spartntransport)
        msg2 = SPARTNMessage(transport=self.spartntransport)
        res = msg1.serialize()
        self.assertEqual(res, self.spartntransport)
        res1 = msg2.serialize()
        self.assertEqual(res1, self.spartntransport)

    def testsetattr(self):  # test immutability
        EXPECTED_ERROR = (
            "Object is immutable. Updates to eaf not permitted after initialisation."
        )
        with self.assertRaisesRegex(SPARTNMessageError, EXPECTED_ERROR):
            msg = SPARTNReader.parse(self.spartntransport)
            msg.eaf = 0

    def testrepr(self):  # test repr, check eval recreates original object
        EXPECTED_RESULT = "SPARTNMessage(transport=b's\\x00\\x12\\xe2\\x00|\\x10[\\x12H\\xf5\\t\\xa0\\xb4+\\x99\\x02\\x15\\xe2\\x05\\x85\\xb7\\x83\\xc5\\xfd\\x0f\\xfe\\xdf\\x18\\xbe\\x7fv \\xc3`\\x82\\x98\\x10\\x07\\xdc\\xeb\\x82\\x7f\\xcf\\xf8\\x9e\\xa3ta\\xad')"
        msg1 = SPARTNReader.parse(self.spartntransport)
        self.assertEqual(repr(msg1), EXPECTED_RESULT)
        msg2 = eval(repr(msg1))
        self.assertEqual(str(msg1), str(msg2))

    def testpayload(self):  # test payload
        EXPECTED_RESULT = b"\xf5\t\xa0\xb4+\x99\x02\x15\xe2\x05\x85\xb7\x83\xc5\xfd\x0f\xfe\xdf\x18\xbe\x7fv \xc3`\x82\x98\x10\x07\xdc\xeb\x82\x7f\xcf\xf8\x9e\xa3"
        msg = SPARTNReader.parse(self.spartntransport)
        self.assertEqual(msg.payload, EXPECTED_RESULT)
        self.assertEqual(msg.nData, len(msg.payload))

    def testnopayload(self):  # test null payload
        EXPECTED_ERROR = "Transport must be provided"
        with self.assertRaisesRegex(SPARTNMessageError, EXPECTED_ERROR):
            msg = SPARTNMessage(transport=None)

    def testbadpayload2(self):  # test null payload
        EXPECTED_ERROR = "Unknown message preamble 120"
        with self.assertRaisesRegex(SPARTNParseError, EXPECTED_ERROR):
            msg = SPARTNMessage(transport=self.badpayload)

    def testbadpayload(self):  # test null payload
        EXPECTED_ERROR = "Unknown message preamble 120"
        with self.assertRaisesRegex(SPARTNParseError, EXPECTED_ERROR):
            msg = SPARTNReader.parse(self.badpayload)

    def testbadcrc(self):  # test bad CRC
        EXPECTED_ERROR = "Invalid CRC 7627169"
        with self.assertRaisesRegex(SPARTNMessageError, EXPECTED_ERROR):
            msg = SPARTNReader.parse(self.spartnbadcrc)

    def testERRRAISE(self):  # test stream of SPARTN messages with quitonerror = 2
        EXPECTED_ERROR = "Invalid CRC 15632804"
        with self.assertRaisesRegex(SPARTNParseError, EXPECTED_ERROR):
            spr = SPARTNReader(self.streamBADCRC, quitonerror=ERRRAISE)
            for raw, parsed in spr:
                pass

    def testERRRAISE2(self):  # test stream of SPARTN messages with quitonerror = 2
        EXPECTED_ERROR = "Unknown protocol b'\xaa'"
        with self.assertRaises(SPARTNParseError):
            spr = SPARTNReader(self.streamBADPRE, quitonerror=ERRRAISE)
            for raw, parsed in spr:
                pass

    def testERRLOG(self):  # test stream of SPARTN messages with quitonerror = 1
        EXPECTED_OUTPUT = "Invalid CRC 15632804"
        self.catchio()
        spr = SPARTNReader(self.streamBADCRC, quitonerror=ERRLOG)
        for raw, parsed in spr:
            pass
        output = self.restoreio()
        self.assertEqual(output, EXPECTED_OUTPUT)

    def testERRIGNORE(self):  # test stream of SPARTN messages with quitonerror = 1
        EXPECTED_OUTPUT = ""
        self.catchio()
        spr = SPARTNReader(self.streamBADCRC, quitonerror=ERRIGNORE)
        for raw, parsed in spr:
            pass
        output = self.restoreio()
        self.assertEqual(output, EXPECTED_OUTPUT)

    def testERRHandler(self):  # test stream of SPARTN messages with quitonerror = 1
        def igor(err):
            print(f"The error was ({err})")

        EXPECTED_OUTPUT = "The error was (Invalid CRC 15632804)"
        self.catchio()
        spr = SPARTNReader(self.streamBADCRC, quitonerror=ERRLOG, errorhandler=igor)
        for raw, parsed in spr:
            pass
        output = self.restoreio()
        self.assertEqual(output, EXPECTED_OUTPUT)

    def testSPARTNLOG(
        self,
    ):  # test stream of SPARTN messages
        EXPECTED_RESULTS = (
            "<SPARTN(SPARTN-1X-OCB-GPS, msgType=0, nData=37, eaf=1, crcType=2, frameCrc=2, msgSubtype=0, timeTagtype=0, gnssTimeTag=3940, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=3, authInd=1, embAuthLen=0, crc=7556915, payload=b'\\xc3\\x14\\xe4\\x0a\\xd4\\x2a\\x98\\x87\\xff\\x4e\\xe1\\xd7\\x44\\xa1\\xc3\\xfe\\x2e\\xd5\\x93\\x2b\\x64\\x12\\x2f\\xb4\\x80\\xc0\\xdd\\x9b\\x26\\x72\\xc9\\x6e\\x94\\x16\\x39\\xb8\\x3b')>",
            "<SPARTN(SPARTN-1X-OCB-GLO, msgType=0, nData=33, eaf=1, crcType=2, frameCrc=3, msgSubtype=1, timeTagtype=0, gnssTimeTag=14722, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=42, authInd=1, embAuthLen=0, crc=13784453, payload=b'\\xd5\\x52\\x56\\x23\\xeb\\x12\\xf4\\x49\\x1d\\x0f\\xe8\\x36\\x9f\\x91\\x64\\x34\\xf9\\x3f\\x48\\xc8\\xee\\x3f\\x23\\x39\\xa2\\x38\\x33\\xfe\\xd7\\x17\\x0a\\x66\\xb5')>",
            "<SPARTN(SPARTN-1X-OCB-GAL, msgType=0, nData=34, eaf=1, crcType=2, frameCrc=3, msgSubtype=2, timeTagtype=0, gnssTimeTag=3940, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=63, authInd=1, embAuthLen=0, crc=15726580, payload=b'\\xdc\\x9d\\xff\\xac\\xf5\\x34\\xc2\\xb3\\x54\\xdd\\xc8\\x7e\\xab\\xba\\x21\\x97\\xb9\\xd6\\x77\\x74\\x00\\xdf\\x19\\xe8\\xc6\\xec\\xb4\\x4f\\xc1\\xc8\\x7f\\x83\\x55\\xc5')>",
            "<SPARTN(SPARTN-1X-OCB-GPS, msgType=0, nData=37, eaf=1, crcType=2, frameCrc=2, msgSubtype=0, timeTagtype=1, gnssTimeTag=413903145, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=4, authInd=1, embAuthLen=0, crc=6997525, payload=b'\\x50\\xa8\\x12\\xa7\\x8c\\xfd\\x98\\x0e\\x91\\x6d\\xb8\\x0c\\x55\\x1f\\xaa\\xd8\\xf2\\x11\\x93\\x1d\\x9a\\xf0\\x5b\\xc5\\x6b\\xdd\\xe1\\x2d\\x73\\xb4\\x06\\xa3\\x32\\xad\\xcb\\x8f\\xf4')>",
            "<SPARTN(SPARTN-1X-OCB-GLO, msgType=0, nData=33, eaf=1, crcType=2, frameCrc=3, msgSubtype=1, timeTagtype=1, gnssTimeTag=413913927, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=43, authInd=1, embAuthLen=0, crc=11358619, payload=b'\\xa9\\xb4\\x58\\xe6\\x5a\\x11\\xc6\\x96\\x27\\x77\\x08\\xd7\\x34\\x4d\\xc0\\xd9\\x1a\\xf3\\x2e\\x3c\\xda\\xc8\\x3f\\xd7\\x47\\xc7\\x81\\x99\\x34\\x0d\\x7c\\x95\\x4b')>",
            "<SPARTN(SPARTN-1X-OCB-GAL, msgType=0, nData=34, eaf=1, crcType=2, frameCrc=3, msgSubtype=2, timeTagtype=1, gnssTimeTag=413903145, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=0, authInd=1, embAuthLen=0, crc=16183954, payload=b'\\xed\\xfd\\x97\\x98\\xc6\\xf1\\xfd\\x06\\x44\\x34\\x4b\\xe5\\x44\\xfd\\x27\\x21\\x03\\xa3\\x9c\\x16\\xa0\\x79\\x99\\x30\\x9b\\x7d\\x67\\x83\\x10\\x01\\x52\\x69\\x6a\\xc8')>",
            "<SPARTN(SPARTN-1X-OCB-GPS, msgType=0, nData=37, eaf=1, crcType=2, frameCrc=2, msgSubtype=0, timeTagtype=0, gnssTimeTag=3950, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=5, authInd=1, embAuthLen=0, crc=9417614, payload=b'\\x1e\\x68\\xd2\\xfa\\xed\\xed\\x9d\\xd0\\xbb\\x85\\xcb\\x94\\x9a\\x18\\xd3\\xe2\\xdd\\x80\\xd6\\x2f\\xb7\\xf7\\x4a\\x54\\x51\\xd6\\x1c\\x71\\xbc\\x30\\x75\\x43\\x2a\\x18\\xc9\\xc3\\x72')>",
            "<SPARTN(SPARTN-1X-OCB-GLO, msgType=0, nData=33, eaf=1, crcType=2, frameCrc=3, msgSubtype=1, timeTagtype=0, gnssTimeTag=14732, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=44, authInd=1, embAuthLen=0, crc=2885277, payload=b'\\x4c\\x4e\\xf7\\xa1\\xda\\xfb\\x8f\\xc4\\x06\\x26\\x4e\\x10\\xb9\\xf3\\x0d\\xd9\\x02\\x41\\xdf\\xfe\\x41\\x41\\x0f\\x30\\xfe\\xc2\\xf0\\x02\\xbe\\x83\\x75\\x6b\\x29')>",
            "<SPARTN(SPARTN-1X-OCB-GAL, msgType=0, nData=34, eaf=1, crcType=2, frameCrc=3, msgSubtype=2, timeTagtype=0, gnssTimeTag=3950, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=1, authInd=1, embAuthLen=0, crc=7937704, payload=b'\\x6d\\x18\\x37\\xf6\\xab\\xb1\\x31\\xb1\\xce\\x25\\xdc\\x4b\\xe7\\x13\\xfe\\x3d\\xc5\\xd1\\x34\\x8c\\x39\\xd9\\x47\\xfa\\xf3\\x5b\\xc1\\x59\\x3a\\x9d\\xa8\\x34\\xdd\\x24')>",
            "<SPARTN(SPARTN-1X-OCB-GPS, msgType=0, nData=37, eaf=1, crcType=2, frameCrc=2, msgSubtype=0, timeTagtype=0, gnssTimeTag=3955, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=6, authInd=1, embAuthLen=0, crc=2323099, payload=b'\\xc4\\x55\\x12\\xaf\\x45\\xf1\\xff\\x15\\x85\\x21\\x6e\\xc5\\x6b\\x29\\x21\\xc9\\xcf\\xd2\\x99\\x9c\\xc9\\xf7\\xdd\\x9c\\xcd\\xb5\\x36\\x0a\\x1a\\xe5\\xf3\\x51\\x40\\xb4\\x7a\\xd1\\x95')>",
            "<SPARTN(SPARTN-1X-OCB-GLO, msgType=0, nData=33, eaf=1, crcType=2, frameCrc=3, msgSubtype=1, timeTagtype=0, gnssTimeTag=14737, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=45, authInd=1, embAuthLen=0, crc=6930276, payload=b'\\x30\\x69\\xf2\\x73\\x2d\\xbe\\xe7\\xa4\\xfb\\x24\\xce\\xb8\\xc1\\x34\\x7d\\x3d\\x0f\\x88\\xdd\\x9c\\x60\\xd0\\x80\\x09\\xfc\\xd1\\x6d\\x1c\\xea\\x52\\x93\\x55\\x2a')>",
            "<SPARTN(SPARTN-1X-OCB-GAL, msgType=0, nData=34, eaf=1, crcType=2, frameCrc=3, msgSubtype=2, timeTagtype=0, gnssTimeTag=3955, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=2, authInd=1, embAuthLen=0, crc=1602694, payload=b'\\xc0\\xcd\\x19\\x08\\x56\\xc5\\x95\\x1c\\xec\\x2f\\x1b\\xa4\\x0e\\x21\\x20\\x84\\xc5\\xa8\\x26\\xa7\\xa7\\xd8\\x55\\xce\\x4c\\xf6\\x1e\\x2e\\x9f\\xaf\\x15\\xe6\\xbb\\x65')>",
            "<SPARTN(SPARTN-1X-GAD, msgType=2, nData=191, eaf=1, crcType=2, frameCrc=14, msgSubtype=0, timeTagtype=0, gnssTimeTag=3960, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=63, authInd=1, embAuthLen=0, crc=13757653, payload=b'\\x30\\x23\\x09\\x83\\x7c\\xd3\\xf0\\x26\\xa9\\x4b\\xe9\\xba\\x2c\\x12\\xf0\\x25\\x35\\xa7\\xda\\x8c\\x64\\x8e\\x3f\\xed\\xb7\\x4e\\x9d\\x2f\\xa2\\x19\\x4d\\xad\\xbb\\x44\\x0c\\x53\\xe9\\x40\\x16\\xf5\\xb4\\xc5\\x1d\\x5b\\x6d\\x7f\\x91\\x97\\xae\\x18\\xe4\\xd2\\xdb\\xbf\\xa1\\x03\\x32\\xed\\x0c\\x08\\xd8\\x3a\\x20\\x74\\x37\\x7c\\x8f\\x5a\\xee\\x4e\\x7c\\x17\\xe4\\xfe\\x94\\xb8\\xe6\\xbe\\xa5\\xba\\xfd\\xb5\\x85\\xa9\\x6e\\x16\\xe8\\x35\\x12\\x50\\x97\\x1a\\x19\\x75\\x9a\\x19\\x89\\x54\\xeb\\xf6\\xf3\\xea\\x46\\x46\\x1a\\x26\\xea\\x29\\xdd\\x20\\x42\\x98\\xb0\\x69\\x91\\x9a\\x91\\xce\\x5a\\x41\\x2f\\xd4\\x7a\\x8a\\x57\\xcd\\x25\\xf5\\x80\\x98\\xe8\\xf1\\xee\\x64\\x69\\x46\\xfe\\x5b\\x96\\x4e\\x62\\x36\\x12\\xab\\x35\\x2b\\x8f\\xf3\\x5a\\x47\\x9f\\x28\\x2d\\xad\\x8a\\xe7\\x00\\xa4\\xe7\\xbf\\x9b\\x46\\xba\\x29\\x17\\x85\\x92\\x70\\xed\\xab\\x1c\\x3b\\x78\\xfd\\xa2\\x1e\\x94\\x26\\xd3\\x11\\xc9\\xd0\\x72\\x77\\x31\\x2e\\xf1\\x1d\\xe7\\xa6\\x10')>",
            "<SPARTN(SPARTN-1X-GAD, msgType=2, nData=50, eaf=1, crcType=2, frameCrc=2, msgSubtype=0, timeTagtype=0, gnssTimeTag=3960, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=0, authInd=1, embAuthLen=0, crc=11582036, payload=b'\\x8c\\x9f\\xe1\\x52\\x85\\x34\\x0c\\x28\\x34\\x4f\\x09\\xd0\\x3e\\x7f\\xc2\\x68\\xda\\xf2\\xbe\\x85\\x9e\\x36\\x7d\\x65\\xf4\\x4a\\xa8\\x91\\xce\\x61\\x14\\xbb\\x95\\xe7\\xb7\\xf3\\x28\\x15\\x53\\x78\\xec\\x34\\x50\\x1c\\x9e\\xb7\\x8e\\xff\\x23\\x1b')>",
            "<SPARTN(SPARTN-1X-HPAC-GPS, msgType=1, nData=590, eaf=1, crcType=2, frameCrc=12, msgSubtype=0, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=52, authInd=1, embAuthLen=0, crc=7879592, payload=b'\\x1a\\xeb\\xde\\xad\\x19\\xda\\xa7\\xec\\x12\\x78\\xad\\x17\\xb1\\xf8\\x59\\x3c\\xf2\\x14\\x6e\\xa3\\xe9\\xf2\\x43\\x1a\\x66\\x96\\xae\\x75\\xfe\\x4f\\xc8\\xd3\\xc2\\xd9\\x31\\x1a\\x2b\\x87\\x31\\x77\\x1b\\xa1\\x4f\\xc2\\x63\\x6f\\x1b\\x4f\\x13\\x59\\x5b\\x9b\\xf4\\x08\\x1f\\xde\\x8f\\x2d\\x54\\x20\\x46\\xc6\\xa3\\x3c\\x14\\x27\\xe9\\x5d\\x79\\xc6\\x54\\x2a\\xb7\\x9e\\x1e\\xdb\\xdb\\x55\\x54\\xe1\\xc8\\x5e\\xd6\\x23\\x97\\xf7\\x30\\x06\\xf7\\xd1\\x69\\x6d\\xcb\\xad\\x7b\\x93\\x0d\\x0b\\x8d\\x60\\x5b\\x34\\x96\\x05\\x2a\\xda\\xaf\\x23\\x1c\\xc0\\xc8\\x33\\x7b\\x1e\\x65\\x52\\x3e\\x59\\x68\\xae\\x2c\\xe0\\x50\\x8b\\xfc\\xf8\\x92\\xff\\xbf\\x45\\x81\\x6b\\x6c\\xec\\xb1\\x87\\x56\\x33\\x46\\x8a\\xdb\\xc6\\x5b\\xae\\x00\\xda\\xe5\\xd1\\xb9\\x3e\\x9a\\x99\\x95\\x52\\x44\\xd3\\x1e\\xba\\x1b\\xa4\\x11\\xed\\xb7\\xac\\x47\\x00\\xcd\\x63\\xbd\\x89\\x1b\\x0e\\x57\\x1c\\xc2\\xe3\\xd4\\x35\\xff\\x02\\x42\\xdf\\x18\\xb3\\xd1\\x16\\xa0\\xdf\\x03\\x32\\xc4\\xf0\\x5b\\xed\\x74\\x8e\\xd6\\x4e\\x75\\xac\\x98\\x78\\x7e\\x30\\x46\\x31\\xed\\xf5\\x33\\x86\\x65\\x1d\\xd4\\xd6\\x5e\\x2d\\x6c\\xce\\x29\\xa4\\xd7\\x5f\\x21\\x3c\\xd3\\x93\\x07\\x44\\xc1\\xad\\xb7\\xe3\\xf5\\xd1\\x15\\x37\\xde\\x8d\\x78\\xc4\\x6c\\x86\\x51\\x1c\\x99\\x3b\\x7c\\xa4\\xb8\\x6c\\xa9\\x2e\\xc2\\xaf\\xd5\\x50\\xfe\\xdc\\xa7\\x4c\\xa6\\xda\\xe4\\x27\\x1c\\x86\\x29\\x38\\x78\\x92\\xc0\\x87\\xd6\\x9b\\xab\\x46\\xd5\\x42\\x70\\xb9\\xa5\\x96\\x55\\x20\\xd1\\x09\\x27\\x90\\x0c\\xf9\\xf7\\x41\\xfd\\x93\\x79\\x1b\\xf7\\x7b\\x6e\\xd7\\x6b\\x24\\x48\\x6d\\xae\\x78\\xa1\\x78\\xac\\xb7\\x76\\x18\\xfd\\x2a\\x45\\xc4\\xbb\\x35\\xb0\\x62\\xfe\\xae\\x9a\\xaa\\x4c\\xc7\\xa0\\xd0\\x3d\\xf1\\x4f\\x34\\xa4\\xfc\\xf1\\xb9\\x17\\xa0\\x2e\\xda\\x3f\\x9f\\x82\\x51\\x6c\\x52\\x20\\x62\\x81\\x46\\xcb\\x59\\x8b\\x99\\xa9\\x70\\x8d\\xd2\\x30\\xf6\\x70\\x45\\xb5\\x63\\xa4\\x34\\x4e\\x0a\\x59\\xab\\x8f\\xe9\\x1c\\xfa\\x3c\\x8f\\xf3\\x27\\xef\\xd4\\x6c\\x31\\x7a\\x95\\xd1\\xe5\\x07\\x54\\x9d\\xf6\\x7e\\xa7\\x7c\\x97\\x97\\x1e\\x58\\x76\\x1c\\x51\\x85\\xa8\\x4a\\x9c\\xe9\\x73\\xf5\\x34\\xf2\\x76\\x31\\x46\\x1a\\xb9\\xc5\\x3c\\x10\\x6d\\x75\\xdd\\x4e\\xb2\\xb1\\x77\\x1b\\x8e\\xaa\\xcc\\x92\\xa8\\x31\\xcd\\xfd\\xde\\x5d\\x56\\xee\\x9f\\x39\\x6d\\x8d\\x8c\\x51\\xa5\\x40\\xe2\\x90\\xc0\\x4d\\x72\\x44\\xe9\\x81\\xc3\\xdc\\xa7\\x32\\x44\\x51\\x4c\\xcb\\x01\\x9a\\x14\\xa7\\xca\\x3e\\x26\\x4f\\x53\\xe8\\x2c\\x6c\\x31\\xc8\\x00\\xf2\\x93\\x66\\x95\\x7e\\x55\\x93\\x25\\x82\\xf8\\x97\\x30\\x20\\x28\\x56\\x08\\x0f\\x95\\x69\\x65\\xde\\xcb\\x65\\x42\\x7e\\xdf\\xce\\xf0\\xc0\\x5f\\x76\\x46\\x19\\x20\\x36\\x7e\\xb2\\xc3\\x62\\x1e\\x57\\xe5\\x74\\x9e\\xdf\\x09\\x94\\xcb\\xcb\\x92\\xc3\\x3e\\x57\\x05\\x87\\x21\\xd1\\x10\\x7f\\xa2\\x10\\x95\\xed\\xf0\\x73\\x2c\\xa1\\x9f\\x81\\x93\\x10\\xad\\x2c\\x4c\\xf1\\x18\\x26\\xda\\xc4\\x34\\x52\\x64\\xa2\\x11\\x39\\x30\\x97\\x7e\\x21\\x67\\x8d\\x5b\\xf2\\x38\\xca\\xf1\\xb4\\x41\\x79\\x94\\xa5\\x2e\\x49\\x63\\x5c\\x7f\\x49\\xd7\\xaf\\x1c\\x85\\xbb\\x3a\\x9b')>",
            "<SPARTN(SPARTN-1X-HPAC-GPS, msgType=1, nData=584, eaf=1, crcType=2, frameCrc=4, msgSubtype=0, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=53, authInd=1, embAuthLen=0, crc=5046464, payload=b'\\xef\\x7e\\x59\\x86\\x87\\x3d\\x37\\x82\\x9e\\xea\\xf5\\x64\\x99\\xc3\\x8d\\x6d\\xc2\\x5f\\xe6\\xb1\\x47\\xe8\\x5e\\x11\\x92\\x9a\\x0d\\x9a\\xa8\\x0e\\xda\\x9d\\x7e\\xf8\\x24\\x26\\x41\\x8e\\x57\\x55\\x03\\x1d\\xef\\x61\\xa5\\x23\\x3c\\x71\\x79\\x83\\xbb\\x3c\\x3b\\x96\\x98\\xf3\\xf7\\x25\\x14\\x23\\x79\\xb1\\x93\\x8a\\xd1\\x84\\x2f\\xa5\\x5a\\xc8\\x39\\xc5\\x63\\xd3\\x39\\x83\\xfc\\xea\\x94\\x78\\x60\\x9f\\xae\\xa1\\x27\\xb5\\x97\\xd7\\x44\\x73\\x88\\xa0\\xf2\\x8a\\xc7\\x13\\x5b\\xae\\x63\\x1e\\x1d\\x7f\\x24\\xe7\\x6e\\x49\\xb9\\x25\\xd9\\xbc\\xdd\\x69\\xee\\xbe\\x82\\x9e\\x0e\\x29\\x54\\xaf\\xb8\\x2f\\x2e\\xfb\\xe0\\x61\\xd5\\xc1\\xb4\\x73\\xec\\x50\\x7c\\x13\\x6b\\xa6\\x86\\x5e\\xc3\\x63\\x44\\x12\\x1b\\x12\\xd0\\x28\\x8d\\xb6\\xfd\\xa7\\xd8\\xf3\\x09\\x56\\x18\\x3f\\xd9\\x36\\xca\\x18\\x6f\\x77\\xef\\x4e\\x3d\\x19\\xcb\\x59\\xbd\\x98\\x2e\\x7c\\xd9\\xd9\\xa1\\x43\\x9b\\x3c\\xe1\\x9a\\xd5\\x73\\xbb\\x2a\\xbf\\xe7\\xce\\x18\\x91\\x82\\xbe\\x6e\\x06\\x9c\\x0b\\x89\\x9d\\x9e\\xdd\\x86\\x5d\\xae\\xaa\\x0b\\xbe\\x1b\\x89\\xfc\\xee\\x3a\\x55\\x90\\xaf\\x5d\\xac\\xbc\\x81\\xe1\\x09\\x1a\\xfa\\x76\\x09\\xd3\\xf7\\x8e\\xa1\\x41\\xe9\\xb7\\x98\\x70\\xef\\x2c\\x97\\xf3\\xf5\\xed\\x02\\x13\\x5a\\xaf\\x26\\x08\\xa6\\x9e\\xe5\\x43\\xd9\\xf3\\x08\\xf7\\x9f\\x49\\x1b\\x7f\\x85\\x5c\\x3c\\xe8\\xf0\\x7c\\x62\\x3c\\xcd\\x28\\x44\\x34\\x42\\x08\\xc7\\xb0\\x3e\\x01\\xd8\\x84\\x16\\x73\\x1b\\x8a\\xe4\\xf8\\xcc\\xf1\\x4c\\xef\\xdc\\xdf\\x24\\xda\\x34\\xea\\x37\\xb5\\xf9\\x55\\xd3\\x28\\x78\\x56\\x33\\x8c\\x32\\x5e\\x5c\\x9f\\xbe\\x91\\x49\\x30\\x92\\x5b\\x65\\x86\\xd4\\x95\\x38\\xc7\\x0a\\xb6\\x39\\x26\\x45\\x4d\\x8a\\xed\\xbd\\x85\\x47\\x9d\\x02\\xfa\\x17\\x91\\x29\\x85\\x77\\xa4\\xf2\\x04\\x52\\x24\\x8e\\xde\\x57\\x27\\x8a\\x58\\x57\\x2d\\x78\\xca\\x1f\\xbe\\xd0\\xff\\x25\\x90\\xe8\\x8b\\xc5\\x40\\xb3\\xb6\\x7f\\x85\\xfe\\xe1\\x77\\x93\\x03\\x5d\\x2c\\xfb\\x01\\x5e\\x6f\\x11\\xf4\\xa3\\xcc\\xa5\\x97\\xdd\\x42\\xf8\\x76\\xa5\\xec\\x83\\x40\\x76\\x9f\\xc7\\xd6\\x2a\\xa1\\x92\\x91\\x21\\x64\\x83\\xb0\\xc3\\x0c\\x71\\x5e\\x7f\\xe1\\x82\\xca\\xed\\xaf\\x0a\\xc8\\xdd\\xe7\\x42\\x89\\xed\\x1a\\xfc\\x49\\x0b\\x31\\x67\\x00\\x5b\\x27\\x3c\\xfc\\xb4\\xc2\\xcf\\x33\\x9f\\x2f\\x96\\xb9\\x03\\xf7\\x9b\\x98\\x9c\\xb7\\xa9\\x15\\xdd\\xd8\\x3a\\x37\\x7a\\x7b\\x10\\xa2\\x6f\\x0c\\x9b\\x6b\\x2c\\x13\\xd9\\x7a\\x6f\\x9a\\x17\\x3b\\x09\\xc1\\x0b\\x9d\\x9d\\x55\\x22\\xfa\\x58\\xa0\\x04\\x78\\xd5\\x40\\xde\\x4e\\x30\\xa8\\xb7\\x63\\x80\\xfb\\x81\\xdd\\xca\\x1b\\xef\\x99\\x20\\xbe\\x34\\x46\\xf8\\xf1\\xc2\\xe9\\x75\\x66\\x0c\\x13\\xe4\\x71\\x82\\x2e\\xc2\\x13\\xbf\\xdd\\xe8\\x6d\\xdf\\x4e\\x55\\xdd\\x6f\\x69\\xb0\\xba\\x24\\xa2\\x9d\\x94\\x60\\x1f\\x8b\\x24\\x8d\\x77\\xc2\\x95\\xe8\\x63\\xc0\\xfc\\x0e\\x4d\\x00\\xeb\\x48\\xe3\\x8d\\xd2\\x4d\\xd0\\xa6\\x65\\x37\\xd6\\xe7\\x19\\x7c\\x27\\x50\\xa6\\xcb\\xab\\xc2\\x09\\x58\\x0c\\xf0\\x1d\\x94\\x45\\x79\\x2b\\x6b\\x7b\\xa7\\xd0\\x03\\xa2\\x41\\xa6\\x87\\x9e\\x07')>",
            "<SPARTN(SPARTN-1X-HPAC-GPS, msgType=1, nData=554, eaf=1, crcType=2, frameCrc=6, msgSubtype=0, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=54, authInd=1, embAuthLen=0, crc=14377135, payload=b'\\x0c\\xe9\\x25\\x36\\xb6\\xef\\x65\\xec\\xc9\\xe6\\x38\\x78\\xe5\\xbf\\x8d\\x0c\\xe6\\x56\\x1a\\xe2\\xa6\\x92\\x7d\\x9f\\x12\\xd6\\x45\\x4c\\xb7\\x1b\\x67\\x68\\x5a\\x87\\x12\\xca\\x11\\x8a\\x60\\x97\\xf4\\x61\\xf3\\x99\\xc0\\xe3\\x56\\x0b\\x1e\\xb0\\x12\\x70\\xb8\\x03\\x85\\x88\\xd9\\xdf\\x59\\x52\\x99\\xaa\\xe9\\x35\\x6d\\x0e\\xae\\x4a\\x44\\x7f\\x11\\xae\\x1d\\x95\\xf8\\x51\\x6a\\x54\\x23\\xd3\\x7a\\xa4\\x68\\xfb\\x40\\x3f\\xa9\\x3f\\x23\\x81\\x90\\x48\\x44\\xd2\\x7b\\xa7\\x1a\\xb9\\x9c\\x46\\x8e\\x64\\x65\\xf8\\x6a\\x44\\xcc\\x35\\x05\\xc3\\xd1\\xd7\\x7f\\x9a\\xe6\\x2a\\xfe\\x60\\xb0\\xac\\x2b\\x96\\x0e\\xce\\xb7\\x5f\\xd4\\x82\\x1c\\x00\\x1d\\xe3\\xae\\x8d\\x93\\xb2\\x0e\\x2a\\xdc\\xa9\\x85\\xbc\\xb3\\x94\\x62\\x8b\\x18\\xd7\\x15\\xf6\\x7d\\x05\\x33\\x7f\\xb0\\x98\\xb5\\x40\\xa7\\x6d\\xe1\\x28\\x20\\xdb\\x4e\\xa2\\xea\\xe7\\x20\\x23\\xe6\\x27\\x0a\\x96\\x3d\\x87\\x70\\xca\\x3c\\x11\\xdb\\x68\\x2a\\xec\\x95\\x0e\\x1d\\x19\\x69\\xc7\\xb6\\xe1\\x0c\\x47\\xe1\\x1c\\x7c\\x6a\\xbf\\xc6\\x62\\x9c\\xdd\\xea\\xd7\\x6e\\x5d\\xf4\\xb2\\x76\\x91\\x04\\x45\\x0a\\x24\\x7e\\x1b\\x7d\\xe4\\xbc\\x0d\\x94\\x80\\xdd\\x17\\x11\\x8d\\xa1\\xfb\\xb3\\x4d\\xad\\x7f\\xea\\x04\\xd0\\x8a\\x47\\x0c\\xc6\\xdb\\xd6\\x01\\xf2\\xdb\\x21\\x9c\\x81\\x6f\\x15\\x5e\\xee\\x57\\xea\\x64\\x4a\\x0c\\x59\\xd0\\x39\\x41\\xb9\\xc4\\x27\\xef\\x07\\xb6\\x63\\x2e\\xee\\x6c\\xf2\\x6d\\xd3\\x9c\\x78\\xd5\\x19\\x76\\x08\\x95\\x90\\x4e\\xd2\\x44\\xca\\x78\\xd8\\x27\\x71\\xb1\\x88\\x6c\\xbb\\x88\\x8b\\x58\\x8b\\x6d\\x5c\\xd9\\xcb\\x31\\x74\\x9f\\x91\\x5b\\x04\\x2c\\x25\\xeb\\x8d\\x8e\\xc3\\xe5\\x9f\\x40\\xf3\\xaf\\x86\\x8b\\xcd\\x31\\x4b\\x55\\x5f\\x76\\xe1\\x70\\x83\\x20\\x95\\x6d\\xdc\\xf9\\x20\\xbc\\xde\\xb8\\x5d\\x1e\\xfc\\x92\\x3c\\x70\\xcd\\x82\\xbc\\x44\\x03\\x89\\xbb\\xe4\\xd1\\x1c\\x2d\\x45\\xda\\xef\\x05\\x56\\x33\\x9c\\xfd\\x36\\xc1\\xdb\\x64\\x4b\\x06\\xdb\\xba\\x4d\\xdf\\x8f\\x7d\\xec\\x0b\\x87\\xe5\\xac\\x56\\xe1\\x97\\x75\\x0f\\x33\\xd9\\x27\\xc0\\x78\\x35\\xa3\\x86\\xf8\\x4a\\x4c\\x9d\\x38\\xfb\\xc3\\xc5\\xb8\\x75\\xf6\\xc6\\x5a\\x5f\\x59\\x60\\xa2\\xab\\xa9\\x19\\x34\\x92\\x41\\x91\\x69\\x9c\\x99\\xf0\\x45\\x3e\\x37\\xbc\\xb3\\x61\\xa6\\x47\\x2c\\xfd\\xbb\\x5e\\x6d\\x2e\\x6f\\x0d\\x3d\\x88\\xef\\x2a\\x47\\xcc\\x40\\x43\\x3a\\xf2\\x91\\xfe\\x17\\xde\\x96\\x02\\x2d\\x98\\xdd\\x73\\xf6\\x7b\\xdc\\xbb\\x37\\x0a\\x7c\\x7f\\xa1\\x4c\\x60\\x47\\x12\\x64\\x67\\x74\\x8d\\x18\\xa8\\xff\\x9f\\x8b\\x90\\x91\\x13\\x84\\xde\\x43\\x43\\x9a\\x60\\xf8\\x10\\x20\\xa6\\x39\\xbd\\x4b\\x11\\x54\\x85\\x6e\\x5d\\x8b\\xdf\\x2b\\x13\\x8c\\x22\\x30\\xcc\\xca\\x9d\\x6a\\xaf\\x18\\xe4\\x7f\\x33\\x7d\\xa1\\x0e\\x25\\xc9\\xb8\\x88\\xbb\\xbe\\x77\\xc9\\x68\\xbd\\xa3\\x8e\\x1a\\x88\\x2d\\xca\\x56\\xf7\\x88\\x4c\\xab\\xef\\x29\\x9c\\xa6\\x28\\x16\\x4b\\x01\\x01\\x9a\\x03\\xb9\\xff')>",
            "<SPARTN(SPARTN-1X-HPAC-GPS, msgType=1, nData=554, eaf=1, crcType=2, frameCrc=6, msgSubtype=0, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=55, authInd=1, embAuthLen=0, crc=5226642, payload=b'\\x07\\x77\\x81\\x42\\x05\\xb1\\x14\\xa2\\x09\\xef\\x87\\x13\\x37\\xde\\xac\\xf4\\x9b\\x67\\x8e\\xdf\\xfe\\xfa\\xe6\\x1a\\x36\\x10\\x76\\xf1\\x0b\\x71\\xf6\\x9a\\x99\\x3e\\xf4\\x8a\\x75\\x9d\\x2a\\xd5\\x7a\\x31\\x0a\\x09\\xe2\\x4d\\xf8\\xb0\\x79\\x22\\x53\\xc9\\xca\\x68\\x04\\xf8\\x15\\x6b\\x25\\x2e\\xe9\\x95\\x1c\\x6f\\xbc\\x27\\xfd\\x7a\\xc0\\xa1\\xc0\\xf1\\x4b\\xbf\\x55\\x69\\x88\\x98\\xd5\\x5d\\x8c\\xad\\x1f\\xab\\xff\\x11\\x32\\x46\\x57\\x89\\x4d\\x8a\\xb2\\xa1\\xfc\\x77\\x07\\x21\\xc9\\x52\\xf4\\x5c\\x55\\x72\\x50\\x0e\\x9c\\xa5\\xcd\\x1a\\xd8\\xb4\\x9e\\x93\\xf5\\x8d\\x24\\x5c\\x09\\xf0\\xa5\\x7e\\x83\\x32\\xad\\x87\\xb2\\xcd\\x28\\xa3\\x46\\x76\\x4a\\x09\\x2f\\x22\\xc0\\x70\\x48\\x16\\x62\\xdb\\xcf\\xc7\\xe0\\xd0\\xab\\x7d\\x7e\\xfe\\xdc\\x08\\x44\\x35\\xf4\\xde\\x5e\\x4f\\x08\\x3a\\xe5\\x67\\xb6\\x82\\x9d\\xaa\\x01\\xab\\xa2\\x08\\xfd\\x23\\x74\\x70\\xe0\\x31\\x4f\\x9b\\xa8\\x76\\xeb\\x0d\\x65\\x1b\\x90\\x49\\xdc\\xa8\\x8b\\xdb\\xbf\\xdb\\xd3\\xb2\\x8c\\x4d\\x23\\x44\\x81\\xac\\xac\\x27\\x68\\x6d\\xc0\\x98\\xe2\\x9e\\xfd\\x48\\xbc\\xbb\\x37\\x68\\x93\\xf0\\xad\\x0e\\x1e\\x1a\\x65\\x41\\x26\\x51\\x9b\\x64\\x6f\\x15\\x4a\\xc2\\xb4\\x74\\x2e\\x49\\xc4\\xfc\\x36\\x42\\x42\\xa1\\x87\\xec\\x1a\\x32\\x34\\x22\\xe2\\xa2\\x5b\\x56\\x8c\\x1a\\x8c\\x30\\x57\\x84\\xcd\\xa4\\x79\\x8a\\xd1\\x82\\x5a\\x03\\x34\\x75\\x4e\\xab\\x98\\x97\\xc2\\xf8\\x3a\\x56\\x5c\\x39\\x7a\\xaa\\xfa\\x0a\\x11\\x0c\\x86\\x3f\\x25\\x6a\\x9d\\xe8\\x65\\x19\\x03\\x34\\xa6\\xcb\\xc4\\x2d\\xaf\\x81\\xc8\\xd4\\xe9\\x20\\xcf\\x87\\xfa\\x41\\x99\\xbc\\x0b\\x87\\xf6\\x46\\x51\\x29\\x5d\\xcd\\x95\\x60\\x66\\x26\\x4d\\x52\\xc7\\x00\\x58\\x6d\\xb9\\x58\\x3c\\x80\\x55\\xef\\xe3\\xb6\\xbb\\x4e\\x35\\x01\\xf9\\x0c\\x91\\xbb\\xa0\\x3f\\x1b\\xd2\\x99\\xd7\\xc1\\x76\\x02\\xe2\\xb4\\x5b\\xd2\\x17\\xc3\\x0c\\x84\\xc4\\x8b\\x43\\xbf\\x3d\\xe0\\xd2\\xac\\x4b\\x1c\\x16\\xc1\\x2c\\xa4\\x2e\\x79\\xdd\\x43\\x80\\x03\\xb8\\xe8\\xe5\\xb3\\x4f\\x5d\\x4c\\xde\\x87\\xd6\\x77\\xd5\\xfa\\xee\\xe6\\x25\\x89\\xf6\\x6b\\x90\\xfc\\x48\\x4b\\xcf\\x1b\\x11\\x2d\\xed\\x76\\xd5\\xd8\\x0e\\xad\\xde\\x8e\\xff\\x08\\x19\\x61\\x95\\x41\\xcf\\xfa\\xa3\\xd7\\x8b\\xf4\\x23\\x10\\x65\\xd9\\x8c\\x3a\\xc4\\x3a\\x1d\\x8a\\x3a\\x4c\\x40\\x56\\x18\\xd2\\xe0\\xf3\\x4d\\x9d\\xcb\\x72\\x89\\x9c\\x84\\x75\\xda\\x99\\x77\\xc9\\x1e\\x58\\xc2\\xdc\\x09\\x3f\\x71\\xba\\xbc\\xe8\\xff\\x66\\x41\\x5b\\x08\\xb1\\x0d\\x83\\x6c\\x9d\\xb2\\x62\\x97\\x63\\x35\\x5b\\x9c\\x75\\x8e\\x0d\\x97\\x0b\\xc1\\x4c\\xd5\\x1c\\x0d\\x7b\\x24\\x17\\x5a\\x4f\\xb4\\x97\\x0f\\x53\\x68\\x7f\\x97\\x50\\xcb\\x51\\x76\\xd0\\x0f\\x98\\x0b\\x3e\\xfb\\x04\\xab\\x3e\\xca\\xf4\\x18\\xf8\\x7d\\x5b\\x0b\\x5a\\x81\\x6a\\x8b\\x78\\x3b\\x02\\x8d\\x5c\\x83\\x0f\\xd9\\x34\\xb2\\x03\\x92\\xdb\\x01\\xb5\\x5e\\x32\\x1f\\x70\\xaf\\xcf\\x0d\\x0a\\x63\\x76')>",
            "<SPARTN(SPARTN-1X-HPAC-GLO, msgType=1, nData=456, eaf=1, crcType=2, frameCrc=2, msgSubtype=1, timeTagtype=1, gnssTimeTag=413913942, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=11, authInd=1, embAuthLen=0, crc=4825390, payload=b'\\x82\\xbf\\x91\\x87\\x1c\\xc6\\xf3\\x8d\\x43\\xb3\\x75\\x8c\\xb8\\xd6\\x97\\x3e\\x4e\\xda\\x88\\xdd\\x90\\x5c\\x5a\\x32\\x64\\x1b\\x5b\\xb6\\x43\\x9b\\x31\\x66\\xf2\\x75\\xce\\xc7\\x4a\\x52\\x87\\x8b\\x4e\\x5a\\x86\\xff\\x83\\xf4\\xb1\\xae\\x9a\\x74\\x8a\\x41\\xab\\xe8\\x09\\x9e\\x9c\\x2d\\xe1\\x46\\x71\\xc8\\x91\\x68\\x96\\xd0\\xeb\\x26\\x63\\x9a\\x3f\\x65\\x9b\\x02\\xa0\\x76\\xf5\\xbd\\x5d\\x1c\\xeb\\x85\\x7a\\x68\\x18\\x0f\\x50\\x36\\xab\\xf7\\x45\\x6e\\x6c\\x48\\xf0\\x6f\\xc2\\x01\\x87\\x16\\x81\\x3e\\xf5\\x2b\\xc6\\x64\\x5d\\x27\\xeb\\xb2\\x89\\x34\\x4a\\x67\\x1c\\xea\\xa1\\xd3\\xf5\\x5d\\x2e\\xf1\\xa6\\xa9\\xc0\\x70\\x68\\x0c\\x47\\x84\\x12\\x2c\\x6d\\x5d\\x64\\x03\\x2f\\x5b\\x3b\\x0e\\x92\\x16\\xcc\\x6a\\x3b\\xb2\\x65\\xee\\xf2\\x05\\x75\\x66\\x36\\xad\\xd2\\x09\\x06\\xff\\x5e\\x90\\x2c\\x5b\\xe4\\xee\\xf8\\x1a\\x16\\x10\\xfe\\xc2\\xc2\\x94\\x24\\xfd\\x31\\x7c\\xcf\\xe1\\x22\\x50\\x98\\x44\\x3d\\x45\\x9b\\x0a\\x95\\x46\\x71\\x63\\x05\\x8a\\xd8\\x5f\\x67\\xcb\\x2d\\xea\\x0c\\x7a\\x50\\x0b\\x5a\\x02\\xe6\\xe3\\x66\\x59\\xb2\\x39\\xc2\\xdb\\x5f\\x0d\\x1b\\xed\\x98\\xd8\\x21\\xa4\\xf6\\xad\\xb3\\xd2\\x01\\xc2\\xf1\\x6e\\x5e\\x51\\xc3\\x6f\\x00\\xa7\\x2f\\xc6\\x2a\\x81\\x9f\\x1e\\x5d\\xec\\xc8\\xeb\\xa9\\xd5\\x49\\x14\\x08\\xab\\xe2\\x86\\xe0\\x09\\x0a\\xdc\\x13\\xf3\\x58\\x1e\\x9c\\x5b\\x6d\\xbc\\x09\\xfb\\x07\\x6e\\x5a\\xf6\\x1d\\x63\\x5b\\x2c\\x28\\x01\\xc2\\x01\\x12\\x41\\xfb\\xd7\\xf9\\x4b\\x72\\xae\\xa4\\xca\\x46\\x33\\xb5\\x06\\x8b\\xe4\\x43\\x46\\xed\\xfe\\xd0\\xce\\xa5\\x08\\x7e\\xf2\\xc9\\xf7\\xf4\\xaf\\x2f\\x75\\x63\\xf0\\xd1\\x21\\xf2\\x91\\x23\\xb2\\xed\\xb5\\xfc\\x66\\x1b\\x30\\xd8\\x2c\\xbc\\xab\\xc0\\x7b\\x3c\\xab\\x2e\\x32\\x63\\xe2\\xdb\\xfc\\x2c\\x8b\\xab\\x67\\x69\\xda\\x39\\x2f\\xf0\\x24\\x74\\xd0\\x1a\\xa0\\xd8\\xe0\\x8a\\x55\\xb2\\x34\\x15\\x0b\\x4a\\xb3\\x4e\\xb0\\x4f\\x53\\x1b\\x5c\\x04\\x8b\\x34\\x13\\x20\\x37\\xe4\\x63\\x2a\\x38\\xbe\\x9f\\x8e\\xe4\\xb9\\x36\\xaf\\xc8\\x53\\xcf\\xef\\x21\\x19\\x8c\\x53\\x44\\x82\\xd1\\x5f\\x8f\\x8b\\x01\\xb5\\xeb\\xd7\\xa9\\xc0\\xa7\\x53\\xc1\\x63\\x13\\x53\\xfa\\x7c\\xab\\xa5\\xe6\\x9b\\x8f\\x05\\x8c\\x31\\x41\\x5a\\xc1\\x93\\x39\\xad\\xbd\\x94\\x9d\\x12\\x0e\\x83\\x49\\x8c\\xa1\\x50\\x48\\x7b\\xab\\xe6\\xac\\x8e\\xc2\\xe9\\x85\\xdb\\x32\\x0c\\x75\\x4e\\xc4\\xf4\\x47\\x5a\\xad')>",
            "<SPARTN(SPARTN-1X-HPAC-GLO, msgType=1, nData=415, eaf=1, crcType=2, frameCrc=4, msgSubtype=1, timeTagtype=1, gnssTimeTag=413913942, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=12, authInd=1, embAuthLen=0, crc=2661822, payload=b'\\x9c\\xf3\\xb9\\xc8\\x33\\xfc\\xba\\x76\\xd0\\x7d\\x18\\x0c\\x30\\xa7\\x2b\\xad\\x19\\x78\\x6d\\x6f\\x79\\xf0\\x5c\\x5f\\x12\\xe4\\xcd\\xdf\\x37\\x04\\x19\\x97\\x02\\x00\\xb5\\xf9\\x62\\x38\\xe5\\xf3\\xe6\\xa7\\x05\\xb3\\xc8\\x3b\\x63\\x29\\xe3\\x01\\x84\\x0e\\x66\\x25\\x0d\\xc0\\x16\\x72\\xdc\\xe7\\xda\\xcc\\x58\\x28\\x66\\xbb\\xce\\x98\\xa8\\xb6\\xc1\\xba\\xc1\\x05\\xc6\\x7e\\x48\\x74\\xed\\x12\\xec\\x37\\x64\\xc7\\x2d\\x4e\\xc9\\x92\\xd0\\xf9\\x84\\xaf\\xf9\\x20\\x69\\x98\\x6c\\xfc\\xc2\\x72\\x37\\x3b\\x57\\x6c\\x1d\\xa4\\xa3\\xc4\\xf5\\x6e\\xd5\\xdb\\xd7\\x26\\x93\\x06\\xfd\\x3e\\xa4\\xbe\\x60\\x57\\xaa\\xea\\x81\\x4c\\xa6\\xa7\\x4d\\xd1\\x10\\xb6\\x7f\\xef\\xd7\\x08\\x30\\x46\\xc5\\xbd\\xe0\\x22\\x7b\\x39\\x30\\x2b\\x8e\\xca\\x5f\\x88\\x43\\xf0\\x38\\x09\\xc2\\x21\\xd0\\x28\\x27\\x2f\\xa2\\x74\\xef\\x96\\xd3\\x3e\\x0b\\x92\\x80\\x46\\x4e\\xfc\\xbf\\xea\\x77\\x34\\x0a\\x9d\\x8d\\x7d\\x82\\x19\\x31\\xce\\xb5\\x54\\x9c\\x75\\xed\\x23\\xc3\\x92\\xe1\\xf0\\x3b\\x34\\x46\\x83\\x85\\x35\\xed\\xfe\\x63\\xaf\\xa4\\x16\\x99\\xb7\\xfd\\xe2\\xe8\\x68\\xb3\\x59\\xd8\\x00\\x36\\x74\\xe2\\x4c\\x09\\xf4\\x86\\x0d\\x30\\xfb\\x98\\xa0\\xb7\\x1e\\x8f\\x79\\x87\\x5f\\x8e\\xf9\\x14\\x06\\x26\\x14\\x05\\x79\\xbc\\x66\\xb6\\xfa\\xe1\\xe7\\xf6\\x16\\xaa\\x7d\\x07\\x00\\x92\\xfd\\x0e\\x97\\xf0\\x5a\\xcd\\xc7\\x01\\xbd\\x52\\xf8\\xdc\\x34\\x97\\xaf\\x69\\x7e\\x2c\\xd6\\x11\\x5c\\x1f\\x98\\xde\\x1f\\x71\\xec\\xe2\\xc0\\x2e\\x8d\\x28\\xe8\\x70\\x72\\xf1\\x4a\\x02\\x5e\\x67\\x95\\x37\\x28\\x82\\x7c\\x03\\xcf\\x4c\\x49\\xc7\\xa7\\xd8\\xe2\\xe3\\x17\\x33\\x65\\x3e\\x2d\\x7d\\xae\\xc2\\x2b\\x41\\xd8\\x71\\x2f\\x1b\\x90\\xbd\\xd6\\xeb\\xa1\\xbb\\x1e\\xa0\\x1e\\x04\\xe4\\x03\\x0e\\x2b\\x80\\xc0\\xc0\\xd2\\xef\\xfe\\x35\\x98\\x2c\\xa2\\xab\\x24\\x42\\x6b\\x72\\xa2\\xd4\\xaa\\xf6\\xe5\\x7e\\x5c\\xa3\\xdf\\x0e\\x1c\\xa8\\xd4\\xcc\\xba\\x33\\x9f\\xef\\x27\\xf8\\xc0\\x95\\x4d\\xd1\\xd3\\x94\\x0a\\x8d\\xa7\\xf9\\x27\\x54\\x41\\x37\\x0c\\x92\\x28\\x97\\xe6\\xf8\\xdc\\xca\\x1c\\xa9\\x30\\x03\\x1d\\xcc\\x74\\x24\\xed\\x86\\x31\\x5a\\xab\\xdd\\x6f\\x96\\x2b\\x47\\x43\\x96\\x9c')>",
            "<SPARTN(SPARTN-1X-HPAC-GLO, msgType=1, nData=433, eaf=1, crcType=2, frameCrc=6, msgSubtype=1, timeTagtype=1, gnssTimeTag=413913942, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=13, authInd=1, embAuthLen=0, crc=4661009, payload=b'\\xa8\\x0d\\xa9\\xb3\\xa9\\x46\\xe2\\x2b\\x73\\x82\\x38\\x22\\x87\\x0e\\x00\\xfa\\xc3\\x81\\x6e\\xd9\\xb5\\xdc\\xb3\\x05\\x54\\x30\\x94\\x59\\x17\\x45\\x12\\x07\\x14\\x44\\x6c\\x38\\x6f\\x1e\\xea\\x70\\x3f\\x9b\\xd1\\x65\\x66\\x10\\x07\\x72\\x80\\x5c\\xf8\\x48\\x05\\xf0\\x3f\\x81\\x06\\x89\\xbe\\x0f\\x99\\xf2\\x9f\\x36\\xc3\\x87\\xa3\\x93\\xf8\\xe1\\x7b\\x1e\\x82\\xe2\\x14\\xe0\\x92\\x5d\\xf3\\x99\\x04\\xf7\\x57\\x3f\\x47\\x55\\x6d\\x3c\\x4a\\x19\\x3f\\xfb\\x71\\x3a\\x54\\x6e\\x2d\\x76\\x24\\x9c\\xd9\\x87\\x92\\x8d\\xdd\\x03\\x52\\xaf\\xd9\\x36\\xa9\\xfd\\xc6\\xda\\x14\\xdd\\xe1\\x5b\\x94\\x7f\\x21\\xf9\\x96\\x85\\xa2\\x35\\xc5\\x08\\x30\\xaa\\x6f\\x02\\x28\\x59\\xa6\\x62\\x69\\x62\\xf4\\x99\\x25\\x0d\\xaf\\xdb\\x71\\x70\\x69\\xf8\\xf7\\x6d\\xa4\\x1d\\xb1\\xd7\\xfd\\xfe\\x2c\\xd6\\xcb\\x29\\x37\\x4c\\x81\\x25\\x9a\\xfe\\x3e\\xa2\\xcd\\xc8\\x48\\x42\\x2c\\x78\\x3e\\x0e\\x56\\x80\\xc8\\x9c\\x29\\xa1\\xb0\\x5c\\xb6\\xed\\xeb\\x31\\x4d\\x00\\x1e\\x8c\\xd0\\x71\\xe2\\xe1\\xf8\\xfd\\x09\\x05\\x64\\xf6\\xbf\\xae\\x65\\x1a\\xbc\\x2a\\xc5\\xe3\\xc1\\xf3\\xac\\x7a\\xff\\xef\\xca\\xd9\\xdd\\xc8\\x24\\xaa\\x29\\x05\\xa4\\x80\\x8a\\xa3\\xfc\\x36\\x41\\x0c\\x09\\xf1\\x3f\\x18\\xfd\\xd8\\x1c\\x77\\x49\\x51\\x3e\\xd1\\x9d\\xba\\x93\\x35\\x5a\\x91\\xfd\\x4e\\x3e\\xe2\\x43\\x37\\x7b\\xa4\\xf7\\x61\\x6e\\xa9\\x1e\\xe6\\x81\\x20\\x35\\x53\\x5f\\xc4\\xc9\\x3c\\x68\\xd0\\xe1\\x54\\xb4\\x38\\x1a\\x8c\\x39\\x79\\xda\\x4b\\x76\\x7c\\x9c\\xb0\\xc2\\xec\\x8b\\x5d\\x5f\\xff\\x50\\xfd\\xad\\x84\\x98\\xee\\x84\\x5c\\x2b\\x97\\x82\\x73\\x1f\\xa9\\x85\\x0b\\xb8\\x76\\xf5\\x3c\\x7d\\xa0\\x7d\\x63\\xa9\\xb7\\x6f\\x76\\x37\\x70\\x79\\x1a\\xd2\\x8f\\xc0\\x06\\xe9\\xa8\\x0a\\xac\\xd4\\xbb\\xa7\\x66\\xa2\\xb4\\x25\\x9e\\xf8\\x76\\xcd\\xd8\\x87\\xd1\\x9c\\x26\\x85\\x01\\x42\\x76\\x6c\\x3e\\x23\\x95\\xb4\\xa3\\x2e\\x1f\\x4b\\x05\\x71\\x23\\x76\\x50\\xc0\\x54\\x5a\\xa5\\x3f\\xfc\\x3c\\xa6\\x1d\\x2b\\x95\\xc0\\x70\\xa8\\xc2\\x73\\xd8\\xc9\\x8e\\xf6\\x7e\\xb9\\x96\\x05\\xfb\\xe0\\x81\\xc5\\xc8\\xb9\\xf7\\x40\\xe1\\xf1\\x5a\\x83\\x67\\x21\\xf1\\xe9\\x82\\xfc\\xd1\\xc0\\x06\\xfd\\x09\\x93\\xd2\\x67\\x28\\x23\\xb2\\xd6\\xe7\\xf8\\xae\\x3e\\x8c\\x0a\\xef\\xd9\\x95\\x20\\x15')>",
            "<SPARTN(SPARTN-1X-HPAC-GLO, msgType=1, nData=380, eaf=1, crcType=2, frameCrc=9, msgSubtype=1, timeTagtype=1, gnssTimeTag=413913942, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=14, authInd=1, embAuthLen=0, crc=6432064, payload=b'\\x1b\\x92\\xbf\\x45\\x93\\xee\\x2c\\x61\\xb8\\x89\\xfb\\x3f\\xd0\\x43\\x6c\\x6e\\x3d\\x18\\xc3\\x66\\x5d\\xc4\\x40\\x5d\\x55\\xdf\\x39\\xba\\xce\\x56\\xd9\\x6b\\xab\\x0f\\x0d\\xff\\x0e\\x6d\\xc3\\x2c\\x54\\x6d\\x78\\x07\\xb9\\xdd\\x95\\xe9\\x7f\\x85\\x4e\\x7a\\x7a\\xb5\\x35\\xce\\xa2\\x3d\\xd9\\xe0\\x02\\x93\\x70\\x54\\xce\\x09\\x71\\xf4\\x1b\\x9e\\x82\\x23\\xa1\\xd5\\x08\\xc4\\x1e\\x90\\x4c\\x66\\x22\\xe3\\x23\\xdc\\x9d\\xa0\\x05\\x05\\xa7\\x06\\xc1\\xe7\\x37\\xbd\\x3f\\x33\\xdb\\xe8\\x0c\\x2d\\x35\\x0e\\x5e\\x59\\x86\\xd6\\x73\\x07\\x3d\\x70\\x02\\xaf\\x1f\\x60\\x8a\\x1d\\x38\\x54\\x7f\\xc9\\x14\\x2f\\xd1\\xb5\\xce\\x95\\xcd\\xea\\x5d\\x8a\\x2d\\x2f\\x87\\x1c\\x80\\x94\\xd8\\xf9\\xe6\\x75\\x6c\\x75\\xec\\x45\\x13\\x72\\x7a\\x66\\x6c\\x9d\\x52\\x5e\\x93\\x80\\xec\\xfc\\x53\\xd9\\x6a\\x59\\x5f\\xd3\\x4f\\xd0\\xfd\\x8d\\x3a\\xf6\\xc4\\x19\\x90\\x0d\\xf6\\xaf\\xf0\\x51\\x4d\\xdf\\x81\\x06\\x70\\x53\\x1e\\xfb\\x3e\\x64\\x2a\\x69\\xea\\xe2\\x17\\xa9\\x83\\xf9\\xaf\\xb0\\xc2\\x3b\\x2a\\xfc\\xfb\\x56\\x12\\x5d\\x77\\x7d\\x14\\x44\\xc5\\x62\\x72\\xd6\\xfa\\x43\\xb9\\xec\\xe3\\x0f\\xef\\xa7\\x26\\x67\\x1d\\xa5\\xc6\\xfc\\x2a\\x1e\\xcf\\xd7\\xdf\\xf5\\xfa\\x76\\xb0\\xcc\\xf8\\x17\\x17\\x55\\x5d\\x92\\xca\\x25\\x0c\\x0e\\x78\\x0b\\xe0\\xdf\\x93\\xf3\\x57\\x4d\\x95\\x8b\\x90\\xc2\\xf1\\xea\\x50\\xbe\\x39\\xd5\\x8c\\xb0\\xf8\\xac\\x3c\\x5b\\xd1\\xf2\\x9f\\xfc\\xe8\\x30\\xb2\\x4e\\xbd\\x9f\\xba\\xda\\x75\\x01\\xc8\\x05\\x5e\\x35\\x7b\\x1a\\x07\\x95\\xd3\\x93\\xf7\\xd5\\x7a\\x65\\xc1\\xa3\\x62\\xe4\\x10\\x8a\\x37\\x21\\x9c\\xd8\\x6c\\xd2\\xb9\\xd8\\x11\\xde\\xa4\\xbc\\x28\\xd4\\xc9\\xef\\x7e\\x61\\xe2\\x8e\\x67\\x10\\x32\\x8f\\x1c\\xc5\\xb9\\x8a\\x4f\\x0d\\xda\\x87\\x78\\x30\\x53\\x27\\x13\\x91\\x53\\x27\\xdc\\x96\\x21\\x7c\\xee\\xe1\\xa2\\xe1\\xb6\\xa3\\x52\\x2a\\x41\\xac\\xea\\x1c\\xc5\\x89\\xff\\xf0\\x45\\x49\\x10\\x2d\\x9c\\x14\\x83\\x61\\xd0\\x4c\\x2b\\x39\\x9c\\xb8\\xd6\\xf9')>",
            "<SPARTN(SPARTN-1X-HPAC-GAL, msgType=1, nData=565, eaf=1, crcType=2, frameCrc=1, msgSubtype=2, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=15, authInd=1, embAuthLen=0, crc=9900363, payload=b'\\x9f\\x0c\\xe2\\x5d\\x45\\xd0\\x34\\x9b\\x66\\x98\\x3e\\x97\\x4a\\x16\\x42\\x2c\\x36\\x08\\x65\\x1b\\x94\\x58\\xea\\x98\\x90\\x72\\x92\\x51\\xbb\\x48\\x6d\\xdb\\xa8\\x94\\xf8\\x85\\xd4\\x09\\x6f\\x75\\x92\\x06\\xab\\x26\\xd7\\xa3\\xd7\\x61\\xfb\\x38\\x39\\x9e\\xe8\\x11\\xfd\\x39\\xc1\\x36\\xf9\\x9c\\x81\\x8f\\xb5\\xea\\x73\\x2b\\xfc\\xfb\\x30\\x2b\\x17\\xf4\\x3f\\x54\\x98\\x7a\\xdc\\x78\\x6b\\xb8\\xf2\\x81\\xd3\\x29\\xe7\\xbc\\xac\\x69\\x58\\x3a\\x49\\x50\\x8f\\x9a\\x45\\x3e\\x0d\\xa6\\xf7\\xf2\\xc2\\x55\\x10\\x30\\x07\\x93\\x2c\\xda\\x7c\\x5d\\xd3\\x59\\x8b\\x4b\\xdd\\xc0\\x2f\\x23\\x71\\x72\\x58\\x10\\x6a\\xdb\\x77\\x6b\\x9f\\x58\\x8e\\x2a\\x69\\x1e\\xb7\\x5c\\x19\\xb3\\x0b\\x61\\x5c\\x9f\\xdc\\x3b\\x00\\xee\\x0a\\x20\\xa8\\xbe\\x30\\xa9\\x8e\\x2c\\x07\\x03\\x37\\x6b\\x01\\x17\\xbc\\xd0\\x5f\\x21\\xc0\\xb9\\x1c\\x17\\x06\\x2f\\x88\\xa1\\x50\\x71\\x22\\x2d\\xa6\\x20\\x04\\xd3\\x1c\\x6c\\x94\\x30\\xaf\\x67\\xb3\\x98\\x4d\\xf3\\xb9\\x50\\x79\\xe6\\x59\\x09\\x04\\xad\\xca\\xae\\x03\\x81\\x6f\\x50\\x1f\\x32\\xae\\xba\\xf6\\x73\\xd3\\x21\\x51\\x37\\x09\\x81\\x87\\x1c\\x15\\xd6\\x78\\x2e\\x98\\x26\\xca\\xa6\\x3d\\xf8\\x4d\\xd6\\xf4\\xad\\xb4\\xd3\\x11\\x4a\\x19\\x5d\\xee\\xf9\\x7d\\x0a\\xa0\\xe2\\xe7\\x1f\\x62\\x9b\\xcc\\x25\\xc4\\x54\\xef\\x9f\\xa9\\x98\\x65\\xb9\\xb4\\x7a\\xf1\\x93\\x9a\\x56\\x06\\xcf\\x76\\x34\\xe6\\x2c\\xbb\\x01\\x08\\xd1\\x48\\x42\\xc4\\x15\\xfd\\x37\\xa7\\x66\\x26\\x72\\x1c\\x70\\x2a\\xe1\\xb4\\x58\\x28\\x97\\xa4\\x16\\x87\\x20\\x78\\x10\\xaa\\x89\\x53\\xb5\\x71\\xb0\\x34\\xb8\\x8c\\x58\\x76\\xfa\\x08\\x89\\x00\\xe1\\xb0\\xd7\\xf4\\x4c\\xc0\\xe2\\x73\\xf9\\xc4\\x1f\\x8d\\xea\\x72\\xa4\\x09\\x10\\xb6\\x30\\xe6\\x9e\\x2a\\x50\\xfe\\x8d\\x00\\x9f\\x34\\x13\\x2c\\xab\\x88\\xc2\\x17\\xc8\\x99\\xc1\\xd6\\x8e\\xa2\\xe6\\x47\\x1f\\xaf\\x21\\x2c\\xc3\\x96\\x2d\\xfa\\x5f\\xa9\\x30\\x45\\xec\\x1d\\x3e\\xca\\x8e\\x62\\x6d\\xeb\\xa5\\x06\\xb5\\xfb\\x24\\x80\\xea\\x12\\x80\\x34\\x98\\xa0\\xe2\\x3f\\xa7\\x41\\xd4\\x12\\x94\\x5f\\x1e\\x62\\xf3\\x58\\x8f\\x85\\x33\\xbe\\xc4\\x7e\\xef\\x8a\\xb1\\x31\\xb5\\x3e\\x8d\\xe3\\x9a\\x63\\xd5\\x6e\\x13\\x66\\x43\\xdf\\xff\\x2d\\x5a\\xc2\\x92\\x6b\\xea\\xc5\\x6e\\xed\\x18\\x76\\xf4\\x91\\x24\\xbd\\x7f\\x6b\\xdc\\x49\\x32\\xcf\\x0a\\x62\\x2c\\x24\\x0a\\x14\\x3b\\x62\\xa4\\xec\\x12\\xed\\xad\\x4d\\x04\\x1c\\x1a\\x10\\x3c\\xd1\\x87\\x80\\x84\\xe6\\x20\\xea\\x9d\\x10\\x85\\x8a\\x32\\xcc\\xd4\\x9f\\xd1\\x63\\xe9\\x31\\xf5\\xc3\\x1b\\x24\\xa2\\x73\\x2c\\xf9\\xb2\\x8a\\x98\\x17\\x75\\x52\\xa9\\xdd\\x9f\\xce\\xa2\\x7c\\x81\\xf3\\x17\\x5b\\x0f\\x30\\x29\\xba\\x4d\\x42\\x36\\xc8\\x47\\x94\\x13\\xbe\\xd4\\x4d\\x59\\x1c\\x69\\xe5\\xd3\\x8c\\x6f\\x0e\\x03\\x27\\xa1\\x68\\xdf\\x0a\\x61\\x8e\\xc3\\xd2\\x87\\xbf\\xd8\\x0c\\x3e\\x22\\x7f\\xbe\\x55\\x57\\xd9\\xdf\\xc6\\x7e\\x37\\xf9\\xa8\\x78\\x45\\xaa\\x45\\x33\\x79\\xb6\\x84\\x23\\xa5\\x77\\xad\\xd8')>",
            "<SPARTN(SPARTN-1X-HPAC-GAL, msgType=1, nData=565, eaf=1, crcType=2, frameCrc=1, msgSubtype=2, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=16, authInd=1, embAuthLen=0, crc=3171880, payload=b'\\xa3\\x5f\\x15\\xa9\\x67\\xe5\\xa9\\xf3\\xcf\\xfb\\xa7\\x7d\\x17\\x78\\xae\\xb8\\xba\\x1d\\xb3\\x33\\x7a\\x75\\xba\\x90\\x63\\xca\\x37\\xe8\\x07\\x13\\xc0\\x8a\\x3d\\x7a\\x0b\\x20\\x48\\x6e\\x2a\\x65\\xed\\x73\\x76\\x58\\x13\\xec\\xd8\\xca\\xf3\\x8a\\x09\\x29\\x6f\\xeb\\x9e\\xbb\\x68\\x47\\x67\\xd1\\xf3\\x70\\x47\\xc2\\x03\\x48\\x73\\x83\\x66\\xaf\\x0d\\xbc\\xeb\\xa9\\x26\\x96\\x5f\\xa0\\x92\\x14\\x09\\x06\\x70\\xa9\\x3d\\xdb\\xaf\\x31\\xd5\\xeb\\x1a\\xb6\\x2e\\xcb\\x5f\\x61\\x8e\\x69\\xe0\\xe0\\xbc\\xc4\\xde\\x43\\xda\\x83\\xe1\\x3a\\xf8\\xf1\\x22\\x73\\xcb\\xa7\\xb3\\xb7\\xb9\\xc7\\x72\\xfa\\x15\\x2c\\x57\\x82\\xfd\\x29\\x99\\xb0\\xc7\\xf7\\xad\\xee\\x42\\x64\\xdf\\x88\\xb7\\x87\\x44\\x62\\xd5\\xcc\\x82\\x4e\\x42\\x1b\\x0c\\x3e\\xe2\\x12\\x4d\\xed\\xbc\\xe4\\xff\\xad\\x9b\\x64\\x75\\xdc\\xa8\\xa1\\xab\\xaa\\xd6\\x85\\xeb\\xab\\x0c\\x0a\\x8b\\xb0\\x22\\x6f\\xba\\xfb\\x6c\\x0f\\x02\\x43\\x3a\\x00\\x2a\\x99\\xf0\\x58\\x26\\xab\\x62\\xa5\\x94\\xec\\x9f\\xf5\\x18\\x16\\xbf\\xda\\x3d\\xe0\\xd9\\x6e\\x9b\\x1d\\xde\\x28\\xca\\x65\\x92\\xa6\\xb4\\x7b\\xdb\\x78\\x8a\\xea\\xc4\\x32\\xfc\\x66\\x63\\x85\\xd2\\xf4\\x9f\\x70\\x81\\x19\\x98\\xde\\xaa\\xf0\\x4a\\xd9\\x42\\x20\\x38\\xf5\\x08\\x54\\xdc\\x3a\\x2c\\x0e\\xed\\x03\\x29\\x6f\\xc5\\x53\\x30\\x2a\\x69\\xa5\\xfb\\x5d\\xde\\xf2\\x8d\\xa7\\xa0\\x48\\x51\\xba\\x4a\\x3a\\x48\\x34\\x18\\x15\\x02\\xf8\\xf9\\x22\\x91\\xf7\\x04\\x09\\x41\\x22\\x25\\xbd\\x1b\\x5a\\x35\\xa3\\x16\\x4f\\x16\\x8d\\x69\\xbb\\xa2\\x85\\x96\\xe3\\xe3\\x9a\\xdd\\x85\\x76\\x0e\\xb0\\xb9\\xd0\\x6e\\x83\\x2b\\x85\\xcf\\xa3\\x5f\\x68\\x82\\x4c\\x1b\\x66\\x3c\\x9c\\x9d\\x25\\xca\\x10\\x63\\x9b\\xf5\\x3a\\xe4\\x5c\\x0a\\x9e\\x8f\\x28\\xb5\\x34\\xcf\\xf9\\x63\\xa8\\x7e\\xfd\\x42\\xa4\\x00\\x5e\\xd5\\x5c\\x9d\\x53\\xc4\\xd7\\x6a\\x8a\\x1d\\x18\\x23\\x27\\xd6\\xc8\\x59\\x28\\x47\\x81\\x37\\x2d\\xd8\\x2b\\xbf\\x41\\x2a\\xdf\\xf5\\x3c\\xe5\\x49\\x8f\\x97\\x33\\x44\\x85\\x51\\x31\\x31\\xd3\\x63\\xde\\x01\\x1c\\x4f\\x05\\xf4\\xa8\\x30\\x8c\\x43\\x29\\x63\\x55\\xa5\\x34\\xcc\\x46\\xe6\\x96\\x48\\xe9\\x92\\x58\\xe4\\x72\\xb3\\xb7\\x62\\xb1\\xf2\\xac\\x36\\xba\\x06\\x22\\x16\\xe4\\x6a\\x01\\x3f\\x56\\x0b\\x81\\x0a\\xc2\\x8b\\xc4\\x52\\x4c\\x48\\x3a\\xb4\\x4b\\x68\\x99\\x25\\xba\\xc7\\x9a\\xc9\\x0e\\x3b\\xd0\\xe9\\x21\\x78\\x27\\x4c\\xad\\xe8\\xcb\\x4b\\x1d\\x74\\x7a\\xb4\\x9e\\xee\\xdf\\x52\\xb8\\x83\\x44\\xc6\\x15\\xfa\\xa6\\x73\\xce\\x32\\x64\\x93\\x85\\x61\\x5c\\x7a\\xe3\\x7b\\x4e\\x35\\x1f\\xc3\\x9a\\xdd\\x9a\\x7b\\x42\\xa9\\xe0\\xcd\\xa1\\xc3\\x6e\\x3a\\x60\\x98\\xb8\\x12\\x72\\xbf\\x27\\xc3\\xef\\xbc\\xba\\x52\\x38\\xd1\\x4d\\xee\\x3b\\x39\\xb7\\x49\\x99\\xaf\\x97\\x48\\x7b\\x86\\x63\\xcc\\xd4\\x7e\\x72\\x1e\\x7e\\xc1\\x55\\xa4\\x42\\xa9\\xe2\\x8d\\x48\\x4d\\x71\\x5b\\xa7\\xa1\\x80\\x4d\\x88\\x5a\\xb8\\x5b\\xdd\\xc2\\x39\\x9e\\x79\\x93\\x9d\\x30\\x71\\x62\\x68\\xba\\x01\\x1d')>",
            "<SPARTN(SPARTN-1X-HPAC-GAL, msgType=1, nData=524, eaf=1, crcType=2, frameCrc=6, msgSubtype=2, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=17, authInd=1, embAuthLen=0, crc=3600973, payload=b'\\xb8\\x33\\x0e\\x1e\\x40\\xbe\\x88\\xd6\\xf5\\x06\\x0c\\xce\\x26\\xa0\\x2f\\xfb\\x3f\\x34\\x34\\xf3\\x85\\xef\\x04\\xca\\xb4\\x2e\\x6a\\xcd\\x2f\\xaf\\x60\\x4d\\x7f\\x17\\x48\\x2d\\x31\\x8f\\x7f\\xa6\\x5f\\x11\\xf4\\xf9\\x2e\\x46\\x98\\xab\\xc0\\xde\\x9c\\xf0\\x2a\\xae\\x42\\x20\\x24\\xe0\\x87\\x67\\x89\\x40\\xe6\\xb1\\xf9\\xb2\\xc5\\xf3\\xe9\\x40\\x03\\xa8\\xd2\\x82\\x04\\x33\\xba\\x25\\x78\\xbd\\x78\\x0c\\x83\\x4b\\xd9\\xdc\\x93\\x6c\\x71\\x92\\x24\\x18\\x6e\\x3d\\xd2\\x14\\x5b\\x18\\x3f\\x4a\\x78\\xf7\\x30\\xea\\x45\\x99\\x5f\\x34\\xc4\\x24\\x77\\x1a\\xbb\\x49\\x08\\x29\\x1b\\x48\\x54\\xf0\\x1a\\x9c\\xf7\\x1c\\x71\\x83\\xdb\\x07\\x8b\\x5a\\x09\\x0c\\xf2\\x74\\xb3\\x2a\\x44\\x4c\\x3e\\xbe\\xa1\\xff\\xb7\\x23\\xb8\\x91\\x6e\\x00\\xe6\\xda\\x55\\x47\\x04\\x40\\x9f\\xb3\\xb7\\xd4\\xce\\xf8\\x97\\x25\\x4b\\x10\\x4b\\xd3\\xb9\\xe6\\xc5\\xfc\\x54\\x3c\\x36\\x7d\\x36\\x83\\x7b\\xdf\\x92\\x9c\\x47\\xe7\\x27\\x4d\\xfb\\x31\\x08\\x15\\xc4\\xe9\\x28\\xe5\\x89\\x11\\xb2\\xee\\xa3\\xeb\\x84\\x81\\x8d\\x5f\\x5f\\xab\\xf2\\x93\\x3b\\xd5\\x59\\x6e\\x95\\x41\\x20\\x0a\\x5b\\xb7\\x1a\\x12\\xb1\\xc8\\x1e\\x78\\x79\\xc4\\x44\\x3f\\xc9\\xc4\\x50\\xd8\\xe7\\x38\\x4c\\x7e\\xee\\xc7\\x91\\x1e\\x0c\\x6f\\xfd\\xe7\\x70\\xdb\\xf0\\x3a\\x05\\x8e\\x6f\\xf5\\xa7\\xad\\xdf\\x38\\x99\\x7d\\xcb\\x75\\xd5\\xcc\\xdb\\xd3\\x48\\xfe\\x5e\\xcd\\x81\\x27\\x27\\x19\\xe3\\xc4\\xe7\\x67\\x4b\\xf6\\x98\\xd3\\x11\\x9a\\x5d\\x13\\x54\\x5b\\xb1\\xfe\\xf8\\x2f\\x39\\xdf\\x4d\\x23\\x97\\xad\\x0a\\xb1\\x49\\x6c\\xd5\\x00\\xbc\\xe0\\xb6\\x53\\xb0\\xca\\x18\\x35\\x7e\\x33\\x25\\x09\\xc7\\x51\\x35\\x61\\x82\\xea\\x76\\x44\\x2e\\xd3\\xdb\\x60\\xf7\\x1d\\x6b\\x8f\\x9f\\xc0\\x9f\\xcc\\xd4\\x62\\x78\\x03\\x5e\\xb0\\x36\\xf3\\x86\\xde\\x2f\\x5b\\x92\\x6e\\x88\\x8b\\x0e\\x44\\x7a\\x0c\\xb1\\x7f\\xff\\xb2\\x2c\\xac\\x50\\x7f\\x90\\x48\\xd6\\x8f\\xc4\\x26\\x3c\\x3f\\x9b\\x3e\\x0a\\x2e\\x70\\xc5\\x35\\x2a\\x15\\xcf\\x80\\xf0\\x2f\\x92\\xce\\x91\\x10\\xd7\\x80\\x8d\\x60\\xbf\\xd2\\xf5\\xec\\x6a\\x0d\\x14\\x99\\x6b\\x9e\\xc5\\xd2\\xa9\\x93\\x0c\\xa0\\x11\\xc7\\xff\\x75\\xdc\\x25\\x27\\x16\\x0c\\xb8\\xb2\\x45\\x92\\x4a\\x67\\xa0\\x45\\x04\\x2a\\x42\\x07\\x94\\xdb\\xf2\\x07\\xe7\\x86\\x2f\\x32\\x1b\\x68\\x54\\x71\\x7e\\xaf\\xa5\\x5c\\xe8\\x92\\x33\\xa3\\xf9\\xcf\\x65\\x46\\x76\\x18\\x8c\\xdf\\x55\\x1b\\x4b\\x82\\x2e\\x9c\\x2b\\xe8\\x68\\x47\\xd9\\x93\\xd9\\x82\\x0d\\x81\\xbb\\x66\\x58\\x30\\xae\\xcc\\x2a\\x83\\x50\\x6b\\x0f\\x01\\x70\\xee\\x2b\\x68\\xb9\\x95\\x6b\\x80\\x4e\\x78\\x75\\x60\\x6e\\xec\\x14\\xab\\x27\\x85\\x90\\x99\\x21\\x27\\x9d\\xea\\xac\\x9f\\x55\\xf4\\x6f\\xc1\\xb0\\x8e\\x80\\xce\\x84\\xe5\\x9f\\x96\\x59\\x51\\x83\\x62')>",
            "<SPARTN(SPARTN-1X-HPAC-GAL, msgType=1, nData=465, eaf=1, crcType=2, frameCrc=13, msgSubtype=2, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=18, authInd=1, embAuthLen=0, crc=11477640, payload=b'\\x46\\x42\\xc8\\x3c\\xa5\\x4b\\x30\\xa6\\xef\\xd5\\xf4\\x8a\\x8f\\x49\\x6a\\xd8\\x4c\\xaa\\x91\\x72\\x1f\\x5c\\xa7\\x13\\xc7\\x60\\xf5\\x65\\xf5\\x04\\xd6\\x07\\x5d\\xb5\\x8e\\x54\\x1a\\xe7\\x69\\x2e\\xab\\x44\\x3f\\xdb\\xdd\\xcd\\x05\\x83\\xdc\\x8f\\xe6\\x98\\xa9\\x56\\x2a\\x31\\xea\\x76\\x47\\x08\\xa9\\x7e\\xa7\\x2c\\xf6\\x07\\x8c\\x01\\x52\\x0c\\x55\\x1a\\x56\\xc9\\x33\\xd9\\xdf\\x06\\xad\\xf0\\x41\\x46\\xa0\\xcd\\x91\\x50\\x1b\\x88\\x9f\\x02\\x51\\x1f\\xa0\\x74\\x52\\x7e\\xa1\\xde\\x7e\\xb1\\x45\\xb4\\xa4\\x19\\xe7\\x45\\x81\\x66\\xec\\x67\\xe7\\x25\\x1e\\x87\\x6d\\xa4\\x53\\xd2\\x50\\x84\\x26\\x96\\xea\\xb9\\x89\\x1c\\x16\\xd3\\xb7\\xd9\\xc5\\x39\\x59\\x5a\\xfd\\x69\\xef\\x7b\\xef\\xc3\\xb3\\x87\\x7b\\x66\\xd6\\x10\\xfd\\x8e\\xc3\\x74\\x30\\x74\\x92\\xe0\\xd2\\x4b\\x47\\x13\\x54\\xca\\x2e\\xfe\\xdd\\x93\\x66\\xc7\\xc2\\xb2\\x22\\xb9\\xde\\x6f\\x78\\x7b\\xc4\\xa7\\x8d\\x26\\x25\\x61\\x93\\xcd\\x23\\x82\\xb4\\x5d\\x7e\\x58\\x2c\\xfc\\x55\\xf9\\x3f\\x2a\\x98\\x44\\xc1\\x21\\x36\\xcc\\x23\\x11\\xa8\\xd4\\x30\\xd7\\xde\\x1d\\x08\\x57\\x3e\\x9c\\x1a\\x81\\x91\\x16\\x08\\x69\\x5e\\x6e\\x39\\x18\\xc1\\x63\\xdb\\x55\\xc5\\x22\\x39\\xde\\xf8\\xe7\\x4a\\x9c\\x90\\xa6\\x7d\\x24\\xea\\x35\\xbc\\x3e\\xfa\\x1b\\x64\\x8b\\xdc\\xab\\xcf\\x9f\\xe1\\x61\\xa3\\x93\\x9a\\x80\\x77\\x3d\\xfe\\x24\\x21\\xfa\\xa2\\xc4\\xc1\\x9f\\x3a\\x46\\x41\\x1b\\x8c\\x54\\x01\\xdd\\x1a\\x58\\xcf\\x41\\xe1\\x88\\xa4\\x92\\xf8\\x03\\x54\\xe5\\x5c\\x0d\\xaf\\xea\\x36\\x90\\xa0\\xf1\\x67\\xbc\\xb4\\x4b\\x33\\xd8\\x73\\xaf\\x1a\\x04\\x1f\\x94\\x23\\xb9\\x90\\x36\\x56\\xcc\\x02\\x1a\\x1a\\x2d\\x43\\xfd\\x24\\x2b\\xec\\x7b\\x9e\\x81\\x03\\xdf\\x14\\x9a\\x0f\\x91\\x6b\\x9e\\x4e\\x47\\xb2\\xba\\x87\\x92\\x41\\xd9\\xa3\\x32\\x6f\\x2c\\x9f\\xad\\xba\\x6b\\x8a\\xb1\\x41\\x36\\x07\\x9b\\xe3\\xb5\\x9b\\xa1\\xd5\\x40\\xfe\\x21\\x39\\x68\\xf3\\xdc\\xf0\\x2b\\x8e\\x6f\\xaf\\x6d\\x8f\\x09\\xd7\\x36\\xdd\\xcc\\x84\\x14\\xaf\\x90\\xad\\x2e\\x5d\\x43\\x60\\x34\\x14\\xe6\\x02\\x39\\x46\\xb8\\x6e\\x08\\xd2\\x5d\\x40\\xe0\\x41\\xd5\\xbd\\xd3\\xd1\\xb8\\x05\\x2c\\xe6\\xf5\\xca\\xa1\\xc1\\xfa\\xbf\\xf9\\xb8\\x26\\x53\\x58\\x76\\x11\\xb7\\xa2\\x6f\\x2a\\x63\\x8b\\x47\\x36\\x11\\xac\\x9c\\xbb\\x18\\x29\\x93\\x96\\x59\\x2d\\x6c\\xa0\\x8c\\xa9\\xbf\\x14\\x7d\\xc5\\x45\\xc3\\x66\\x61\\x49\\x77\\xf9\\xb1\\xdd\\x23\\x6a\\x0a\\x83\\xbb\\x46\\xfe\\x9d')>",
            "<SPARTN(SPARTN-1X-OCB-GPS, msgType=0, nData=218, eaf=1, crcType=2, frameCrc=8, msgSubtype=0, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=7, authInd=1, embAuthLen=0, crc=4538711, payload=b'\\xa3\\x2e\\x91\\x00\\x64\\x49\\xd3\\x8e\\x9b\\x8e\\x3b\\x9a\\x41\\xb1\\x3c\\x08\\xec\\x54\\x99\\xeb\\xfd\\xed\\xf2\\xaa\\xd7\\xad\\x9b\\x94\\x5e\\x04\\xf1\\x18\\xfd\\x48\\x8d\\xed\\xd3\\xc7\\x4d\\x04\\x90\\xb7\\xe6\\xfc\\x81\\x78\\x9a\\x7f\\x9a\\x58\\xce\\xee\\x59\\x4e\\xd7\\x71\\x08\\x97\\xc5\\x27\\x77\\x0e\\x8c\\x97\\x70\\xe5\\x2e\\x02\\x74\\xc0\\xdf\\x13\\x5c\\x44\\xcb\\x38\\x16\\x1d\\x7c\\x01\\xc1\\x8b\\x00\\x04\\x3a\\xe8\\x8f\\xa3\\xb9\\x9e\\xf2\\x72\\x7d\\x15\\x3d\\xc7\\xd5\\xa8\\x58\\x50\\x0f\\x27\\x22\\x0c\\xfd\\xd8\\x0c\\x8f\\xe5\\x65\\xdb\\x08\\x7b\\x37\\x86\\x86\\xbb\\xb6\\x26\\x0c\\x4e\\xa4\\x50\\x31\\x32\\xaf\\xa1\\x85\\x8b\\xb8\\x65\\x34\\x99\\xe1\\x8b\\xf9\\x7d\\x7f\\x8c\\xd1\\x1d\\x21\\x91\\x1a\\x3b\\x93\\xe6\\xd5\\x6c\\x8f\\x1c\\x6b\\xcd\\x0e\\x83\\x2b\\xf7\\x55\\x4c\\x94\\xe7\\x9c\\x4f\\x8d\\x57\\x1e\\xfb\\x8d\\x81\\x32\\x90\\x7e\\xa9\\x69\\x42\\xf7\\xa5\\x8a\\x9e\\x0a\\x04\\xf5\\x80\\xd5\\x08\\xe1\\xed\\xdb\\xeb\\x55\\x9d\\xa5\\x0e\\xe6\\xfd\\xe8\\x81\\x4e\\x5e\\x67\\x70\\x2f\\x11\\x2f\\x0c\\x69\\x1f\\xe5\\x93\\x22\\xab\\x0a\\x21\\xbd\\x35\\xe3\\xd2\\xbd')>",
            "<SPARTN(SPARTN-1X-OCB-GLO, msgType=0, nData=152, eaf=1, crcType=2, frameCrc=2, msgSubtype=1, timeTagtype=1, gnssTimeTag=413913942, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=46, authInd=1, embAuthLen=0, crc=8221523, payload=b'\\x86\\x4b\\xe9\\x56\\xff\\x6d\\x91\\x1b\\xd9\\x03\\xfa\\x5b\\x29\\x82\\xe5\\x31\\x01\\x3c\\x3c\\x1e\\x39\\x7b\\x0f\\x74\\x89\\xa1\\x79\\xa1\\x80\\x2a\\x4c\\xe9\\xc6\\xb9\\x30\\xd2\\x72\\x90\\x2f\\xaa\\x0d\\x11\\x6c\\x10\\x9d\\xd2\\x7c\\x9f\\x87\\x27\\x03\\xbe\\x13\\x17\\xab\\xab\\x8e\\xd5\\xde\\x69\\xa4\\xd1\\xf1\\xa9\\xf8\\xc0\\x56\\x65\\x88\\x56\\xc2\\x22\\xda\\x5e\\x3e\\x0e\\xf7\\xca\\x44\\xa1\\xe2\\xd9\\x39\\x0a\\xe4\\xec\\xb4\\xd2\\xe5\\x67\\x15\\x2e\\xc0\\x7a\\xa7\\xcf\\xb1\\xb3\\x87\\x13\\xf9\\xbf\\x4c\\x4d\\x2b\\xb4\\xa0\\xee\\x10\\x3b\\x43\\x7a\\xa6\\x74\\x13\\x4d\\x2b\\x6a\\x9f\\xd7\\xd8\\xf0\\x5e\\x57\\xba\\x19\\xa2\\x7d\\xc1\\x3b\\x70\\x42\\x67\\xf7\\xdf\\x76\\xe8\\x47\\xcb\\x65\\x8c\\x86\\x3f\\x21\\x70\\x38\\x99\\xb1\\x29\\x8b\\x26\\x6e')>",
            "<SPARTN(SPARTN-1X-OCB-GAL, msgType=0, nData=191, eaf=1, crcType=2, frameCrc=11, msgSubtype=2, timeTagtype=1, gnssTimeTag=413903160, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=3, authInd=1, embAuthLen=0, crc=12340159, payload=b'\\x5f\\x01\\x7a\\x90\\x93\\x26\\xb6\\x88\\x1a\\x19\\x94\\x16\\x7b\\x7d\\x92\\xd4\\x63\\x7a\\x77\\xb8\\x12\\xb7\\xc8\\x05\\xce\\x87\\x30\\x4e\\x65\\x59\\x7d\\x27\\xb0\\xcc\\x8f\\xeb\\xb9\\xbb\\xeb\\xa3\\x22\\xa6\\x78\\x67\\xc7\\x7b\\x7f\\x1b\\xd9\\xe8\\x4b\\xe7\\x18\\xca\\x08\\x09\\xc2\\x85\\x7e\\x8a\\xa1\\xdd\\xd6\\xba\\x6f\\x33\\x24\\x4f\\x47\\x7f\\x88\\x55\\xd2\\x41\\xaf\\x42\\xad\\x5e\\xb7\\x31\\xf6\\xa3\\xba\\xb9\\xcb\\xc1\\xed\\x26\\x05\\xfb\\xc4\\x3c\\xa7\\xc9\\x4f\\x7c\\x6b\\x23\\xa1\\xc9\\x69\\xa8\\x47\\xc0\\x17\\xfe\\xfa\\x76\\x2a\\x66\\xb6\\xb8\\xd8\\x0c\\x61\\xaa\\x5f\\x34\\x32\\x79\\xd4\\x4f\\xfb\\x8f\\xe2\\xf2\\x16\\x2a\\x06\\x86\\x1c\\x66\\x72\\x31\\xe7\\xf2\\xb4\\x97\\x58\\x59\\x6f\\xb4\\x73\\xa9\\x7c\\x12\\x26\\x7c\\xa6\\x2f\\xdb\\x37\\xed\\xe7\\x4d\\x82\\x32\\xcf\\x6f\\x11\\x5e\\x64\\x98\\x70\\x2f\\xa1\\xde\\xd7\\x1f\\x93\\x7d\\xa3\\x71\\x43\\xac\\xa8\\x57\\xc3\\xcb\\xad\\x09\\xba\\x49\\xa9\\xeb\\x3f\\x6a\\xf9\\x49\\x4a\\xba')>",
            "<SPARTN(SPARTN-1X-OCB-GPS, msgType=0, nData=37, eaf=1, crcType=2, frameCrc=2, msgSubtype=0, timeTagtype=0, gnssTimeTag=3965, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=8, authInd=1, embAuthLen=0, crc=6970314, payload=b'\\x85\\xe3\\x3a\\x9b\\xe7\\x09\\x86\\xd4\\x26\\x6b\\x4c\\xbf\\xd2\\xac\\x51\\xdf\\x38\\x80\\xbc\\x66\\x72\\xcc\\x9d\\x8a\\x80\\xc8\\xbd\\xbc\\x0e\\x2d\\x4e\\x0a\\x6e\\x46\\x4a\\x77\\x13')>",
            "<SPARTN(SPARTN-1X-OCB-GLO, msgType=0, nData=33, eaf=1, crcType=2, frameCrc=3, msgSubtype=1, timeTagtype=0, gnssTimeTag=14747, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=47, authInd=1, embAuthLen=0, crc=12368174, payload=b'\\xd1\\x25\\xa6\\x7f\\x7b\\x7b\\x52\\x07\\xf7\\xc6\\xa9\\xe8\\x27\\x5d\\xd7\\x16\\x50\\x39\\xc7\\xd4\\x9d\\xcf\\xaf\\xb4\\x48\\x8e\\x21\\x85\\x3d\\x4e\\xc4\\xc8\\x6e')>",
            "<SPARTN(SPARTN-1X-OCB-GAL, msgType=0, nData=34, eaf=1, crcType=2, frameCrc=3, msgSubtype=2, timeTagtype=0, gnssTimeTag=3965, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=4, authInd=1, embAuthLen=0, crc=8851501, payload=b'\\x99\\xed\\x77\\x63\\x58\\x06\\xe6\\x3b\\x3c\\x1e\\xab\\x3c\\xd5\\x9c\\xc7\\xff\\xc7\\x40\\xd6\\xf3\\xa9\\x2c\\xb2\\x41\\x52\\x66\\xd6\\xbd\\xb6\\xd9\\x28\\x5e\\x0f\\xf8')>",
            "<SPARTN(SPARTN-1X-OCB-GPS, msgType=0, nData=37, eaf=1, crcType=2, frameCrc=2, msgSubtype=0, timeTagtype=0, gnssTimeTag=3970, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=9, authInd=1, embAuthLen=0, crc=7627181, payload=b'\\xf5\\x09\\xa0\\xb4\\x2b\\x99\\x02\\x15\\xe2\\x05\\x85\\xb7\\x83\\xc5\\xfd\\x0f\\xfe\\xdf\\x18\\xbe\\x7f\\x76\\x20\\xc3\\x60\\x82\\x98\\x10\\x07\\xdc\\xeb\\x82\\x7f\\xcf\\xf8\\x9e\\xa3')>",
            "<SPARTN(SPARTN-1X-OCB-GLO, msgType=0, nData=33, eaf=1, crcType=2, frameCrc=3, msgSubtype=1, timeTagtype=0, gnssTimeTag=14752, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=48, authInd=1, embAuthLen=0, crc=15490832, payload=b'\\x47\\xd4\\x08\\xc5\\x43\\xce\\x1a\\x84\\x3e\\x60\\x8f\\x75\\xd1\\x6a\\x9b\\xd4\\x24\\x01\\xeb\\x6c\\x38\\x2a\\x97\\xfb\\x71\\x80\\x28\\xc9\\x7c\\x4a\\x94\\xc6\\x2a')>",
            "<SPARTN(SPARTN-1X-OCB-GAL, msgType=0, nData=34, eaf=1, crcType=2, frameCrc=3, msgSubtype=2, timeTagtype=0, gnssTimeTag=3970, solutionId=5, solutionProcId=11, encryptionId=1, encryptionSeq=5, authInd=1, embAuthLen=0, crc=15632803, payload=b'\\xd0\\xa3\\xb9\\xd3\\x09\\x1f\\x2b\\x73\\x44\\x5d\\xb9\\x1f\\x7f\\x61\\x2e\\x6d\\xc3\\x1e\\xc7\\x10\\xa3\\x67\\xdf\\x27\\x33\\xbe\\x3e\\x5e\\x28\\xed\\xaf\\xed\\x0f\\xe1')>",
        )

        i = 0
        spr = SPARTNReader(self.streamSPARTN, quitonerror=ERRRAISE)
        # with open("spartn.txt", "w") as output:
        for raw, parsed in spr:
            if raw is not None:
                # output.write(f'"{parsed},"\n'.replace("\\", "\\\\"))
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1

    def testdatastream(self):  # test serialize()
        spr = SPARTNReader(self.streamSPARTN)
        res = str(spr.datastream)
        print(res[-17:])
        self.assertEqual(
            res[-17:],
            "spartn_mqtt.log'>",
        )


if __name__ == "__main__":
    # import sys;sys.argv = ['', 'Test.testName']
    unittest.main()
