import MDAnalysis
import numpy
import h5py
import re
from io import StringIO, FileIO
from MDAnalysis.lib.mdamath import triclinic_vectors, triclinic_box

__PropertiesFinder = re.compile('Properties="{0,1}(.*?)"{0,1}(?: |$)', flags=0)
__LatticeFinder = re.compile('Lattice="(.*?)"', flags=0)


def __frameConfrontation(
    lines: "list[str]",
    nat: int,
    coords: numpy.ndarray,
    box: numpy.ndarray,
    types: "list[str]",
    allFramesProperty,
    perFrameProperties,
    **passedValues,
):
    assert int(lines[0]) == nat
    Lattice = __LatticeFinder.search(lines[1]).group(1).split()
    Properties = __PropertiesFinder.search(lines[1]).group(1).split(":")
    WhereIsTheProperty = dict()
    for name in passedValues.keys():
        assert name in Properties
        mapPos = Properties.index(name)
        WhereIsTheProperty[name] = numpy.sum([int(k) for k in Properties[2:mapPos:3]])
    numberOfproperties = int(numpy.sum([int(k) for k in Properties[2::3]]))

    universeBox = triclinic_vectors(box).flatten()
    print(universeBox)
    print(Lattice)
    print(box)
    x, y, z = numpy.array([float(l) for l in Lattice]).reshape(3, 3)
    print(triclinic_box(x, y, z))
    for original, control in zip(universeBox, Lattice):
        print(original, control)
        assert (original - float(control)) < 1e-7
    if allFramesProperty is not None:
        assert allFramesProperty in lines[1]
    if perFrameProperties is not None:
        assert perFrameProperties in lines[1]
    for atomID in range(nat):
        thisline = lines[2 + atomID]
        assert thisline.split()[0] == types[atomID]
        assert len(thisline.split()) == numberOfproperties
        for name in passedValues.keys():
            if len(passedValues[name].shape) == 1:
                assert (
                    int((thisline.split()[WhereIsTheProperty[name]]))
                    == passedValues[name][atomID]
                )
            else:
                for i, d in enumerate(passedValues[name][atomID]):
                    assert int((thisline.split()[WhereIsTheProperty[name] + i])) == d

        for i in range(3):
            assert float(thisline.split()[i + 1]) == coords[atomID][i]


def checkStringDataFromUniverse(
    stringData: StringIO,
    myUniverse: "MDAnalysis.Universe | MDAnalysis.AtomGroup",
    frameSlice: slice,
    allFramesProperty: str = None,
    perFrameProperties: list = None,
    **passedValues,
):
    universe = myUniverse.universe
    atoms = myUniverse.atoms
    types = myUniverse.atoms.types
    nat = len(atoms)
    lines = stringData.getvalue().splitlines()

    # REMARKS: works only for static number of atoms during simulation
    for frameIndex, traj in enumerate(universe.trajectory[frameSlice]):
        frameID = frameIndex * (nat + 2)
        box = myUniverse.dimensions
        coords = atoms.positions
        print(f"in frame: {frameIndex}")
        __frameConfrontation(
            lines[frameID : frameID + nat + 2],
            nat,
            coords,
            box,
            types,
            allFramesProperty,
            perFrameProperties[frameIndex] if perFrameProperties is not None else None,
            **{k: passedValues[k][frameIndex] for k in passedValues},
        )


def checkStringDataFromHDF5(
    stringData: "StringIO or FileIO",
    myUniverse: "h5py.Group",
    frameSlice: slice,
    allFramesProperty: str = None,
    perFrameProperties: list = None,
    **passedValues,
):
    trajectory = myUniverse["Trajectory"]
    boxes = myUniverse["Box"]
    lines = (
        stringData.getvalue() if type(stringData) is StringIO else stringData.read()
    ).splitlines()
    types = myUniverse["Types"].asstr()
    nat = len(types)

    # REMARKS: works only for static number of atoms during simulation
    for frameIndex, (coords, box) in enumerate(
        zip(trajectory[frameSlice], boxes[frameSlice])
    ):
        frameID = frameIndex * (nat + 2)
        print(f"in frame: {frameIndex}")
        __frameConfrontation(
            lines[frameID : frameID + nat + 2],
            nat,
            coords,
            box,
            types,
            allFramesProperty,
            perFrameProperties[frameIndex] if perFrameProperties is not None else None,
            **{k: passedValues[k][frameIndex] for k in passedValues},
        )


def getUniverseWithWaterMolecules(n_residues=10):
    # following the tutorial at https://userguide.mdanalysis.org/stable/examples/constructing_universe.html
    n_atoms = n_residues * 3
    # create resindex list
    resindices = numpy.repeat(range(n_residues), 3)
    assert len(resindices) == n_atoms
    # all water molecules belong to 1 segment
    segindices = [0] * n_residues
    # create the Universe
    sol = MDAnalysis.Universe.empty(
        n_atoms,
        n_residues=n_residues,
        atom_resindex=resindices,
        residue_segindex=segindices,
        trajectory=True,
    )  # necessary for adding coordinates
    sol.add_TopologyAttr("name", ["O", "H1", "H2"] * n_residues)
    sol.add_TopologyAttr("type", ["O", "H", "H"] * n_residues)
    sol.add_TopologyAttr("resname", ["SOL"] * n_residues)
    sol.add_TopologyAttr("resid", list(range(1, n_residues + 1)))
    sol.add_TopologyAttr("segid", ["SOL"])

    # coordinates obtained by building a molecule in the program IQMol
    h2o = numpy.array(
        [
            [0, 0, 0],  # oxygen
            [0.95908, -0.02691, 0.03231],  # hydrogen
            [-0.28004, -0.58767, 0.70556],  # hydrogen
        ]
    )

    grid_size = numpy.ceil(numpy.cbrt(n_residues))
    spacing = 8
    coordinates = []

    # translating h2o coordinates around a grid
    for i in range(n_residues):
        x = spacing * (i % grid_size)
        y = spacing * ((i // grid_size) % grid_size)
        z = spacing * (i // (grid_size * grid_size))

        xyz = numpy.array([x, y, z])

        coordinates.extend(h2o + xyz.T)
    sol.dimensions = [
        grid_size * spacing,
        grid_size * spacing,
        grid_size * spacing,
        90,
        90,
        90,
    ]

    coord_array = numpy.array(coordinates)
    assert coord_array.shape == (n_atoms, 3)
    sol.atoms.positions = coord_array
    bonds = []
    for o in range(0, n_atoms, 3):
        bonds.extend([(o, o + 1), (o, o + 2)])

    sol.add_TopologyAttr("bonds", bonds)

    return sol


def giveUniverse(angles: set = (90.0, 90.0, 90.0)) -> MDAnalysis.Universe:
    traj = numpy.array(
        [
            [[0.0, 0.0, 0.0], [1.0, 1.0, 1.0], [2.0, 2.0, 2.0], [3.0, 3.0, 3.0]],
            [[0.1, 0.1, 0.1], [1.1, 1.1, 1.1], [2.1, 2.1, 2.1], [3.1, 3.1, 3.1]],
            [[0.2, 0.2, 0.2], [1.2, 1.2, 1.2], [2.2, 2.2, 2.2], [3.2, 3.2, 3.2]],
            [[0.3, 0.3, 0.3], [1.3, 1.3, 1.3], [2.3, 2.3, 2.3], [3.3, 3.3, 3.3]],
            [[0.4, 0.4, 0.4], [1.4, 1.4, 1.4], [2.4, 2.4, 2.4], [3.4, 3.4, 3.4]],
        ]
    )
    u = MDAnalysis.Universe.empty(
        4, trajectory=True, atom_resindex=[0, 0, 0, 0], residue_segindex=[0]
    )

    u.add_TopologyAttr("type", ["H"] * 4)
    u.atoms.positions = traj[0]
    u.trajectory = MDAnalysis.coordinates.memory.MemoryReader(
        traj,
        order="fac",
        # this tests the non orthogonality of the box
        dimensions=numpy.array(
            [[6.0, 6.0, 6.0, angles[0], angles[1], angles[2]]] * traj.shape[0]
        ),
    )
    # adding this for recognisign this univers during tests:
    u.myUsefulName = "FixedBox"
    return u


def give_ico923():
    traj = numpy.array(
        [
            [
                [-2.64e-14, -9.36e-16, -3.42e-14],
                [2.305576182, 1.424924444, -2.57e-15],
                [2.305576182, -1.424924444, -2.30e-15],
                [-2.305576182, 1.424924444, -2.59e-15],
                [-2.305576182, -1.424924444, -2.49e-15],
                [1.424924444, -9.55e-17, 2.305576182],
                [-1.424924444, 1.80e-17, 2.305576182],
                [1.424924444, 1.07e-16, -2.305576182],
                [-1.424924444, -3.23e-16, -2.305576182],
                [-2.18e-15, 2.305576182, 1.424924444],
                [-2.22e-15, 2.305576182, -1.424924444],
                [-2.44e-15, -2.305576182, 1.424924444],
                [-2.22e-15, -2.305576182, -1.424924444],
                [4.62127297, 2.856103768, -4.09e-16],
                [4.62127297, -2.856103768, -2.22e-18],
                [-4.62127297, 2.856103768, -5.43e-16],
                [-4.62127297, -2.856103768, -5.67e-17],
                [2.856103768, 4.19e-16, 4.62127297],
                [-2.856103768, -6.64e-17, 4.62127297],
                [2.856103768, 2.22e-17, -4.62127297],
                [-2.856103768, -1.15e-17, -4.62127297],
                [-6.51e-16, 4.62127297, 2.856103768],
                [-4.31e-16, 4.62127297, -2.856103768],
                [-3.05e-16, -4.62127297, 2.856103768],
                [-4.75e-16, -4.62127297, -2.856103768],
                [4.691111829, -1.67e-16, 3.10e-16],
                [3.795189193, 1.449633278, 2.345555915],
                [3.795189193, -1.449633278, 2.345555915],
                [3.795189193, 1.449633278, -2.345555915],
                [3.795189193, -1.449633278, -2.345555915],
                [2.345555915, 3.795189193, 1.449633278],
                [1.449633278, 2.345555915, 3.795189193],
                [2.345555915, 3.795189193, -1.449633278],
                [1.449633278, 2.345555915, -3.795189193],
                [-1.34e-16, 4.691111829, 1.56e-15],
                [2.345555915, -3.795189193, 1.449633278],
                [1.449633278, -2.345555915, 3.795189193],
                [2.345555915, -3.795189193, -1.449633278],
                [1.449633278, -2.345555915, -3.795189193],
                [-4.03e-16, -4.691111829, 1.36e-15],
                [-4.691111829, 1.47e-16, -2.71e-16],
                [-3.795189193, 1.449633278, 2.345555915],
                [-3.795189193, -1.449633278, 2.345555915],
                [-3.795189193, 1.449633278, -2.345555915],
                [-3.795189193, -1.449633278, -2.345555915],
                [-2.345555915, 3.795189193, 1.449633278],
                [-1.449633278, 2.345555915, 3.795189193],
                [-2.345555915, 3.795189193, -1.449633278],
                [-1.449633278, 2.345555915, -3.795189193],
                [-2.345555915, -3.795189193, 1.449633278],
                [-1.449633278, -2.345555915, 3.795189193],
                [-2.345555915, -3.795189193, -1.449633278],
                [-1.449633278, -2.345555915, -3.795189193],
                [1.01e-15, 4.42e-17, 4.691111829],
                [4.25e-16, -1.62e-16, -4.691111829],
                [6.951958178, 4.296546444, -7.70e-17],
                [6.951958178, -4.296546444, 2.60e-16],
                [-6.951958178, 4.296546444, -1.64e-16],
                [-6.951958178, -4.296546444, -6.18e-17],
                [4.296546444, 4.05e-16, 6.951958178],
                [-4.296546444, 1.84e-16, 6.951958178],
                [4.296546444, -1.32e-16, -6.951958178],
                [-4.296546444, 1.75e-16, -6.951958178],
                [-5.11e-16, 6.951958178, 4.296546444],
                [-2.95e-16, 6.951958178, -4.296546444],
                [-3.99e-16, -6.951958178, 4.296546444],
                [-4.97e-16, -6.951958178, -4.296546444],
                [7.053659059, 1.43719129, 2.03e-17],
                [7.053659059, -1.43719129, -8.63e-18],
                [6.150646585, 2.898296168, 2.364117352],
                [5.26241352, 1.461104877, 4.689541707],
                [6.150646585, -2.898296168, 2.364117352],
                [5.26241352, -1.461104877, 4.689541707],
                [6.226166464, -1.39e-16, 2.378183969],
                [6.150646585, 2.898296168, -2.364117352],
                [5.26241352, 1.461104877, -4.689541707],
                [6.150646585, -2.898296168, -2.364117352],
                [5.26241352, -1.461104877, -4.689541707],
                [6.226166464, -3.10e-17, -2.378183969],
                [4.689541707, 5.26241352, 1.461104877],
                [2.364117352, 6.150646585, 2.898296168],
                [2.898296168, 2.364117352, 6.150646585],
                [1.461104877, 4.689541707, 5.26241352],
                [3.847982494, 3.847982494, 3.847982494],
                [4.689541707, 5.26241352, -1.461104877],
                [2.364117352, 6.150646585, -2.898296168],
                [2.898296168, 2.364117352, -6.150646585],
                [1.461104877, 4.689541707, -5.26241352],
                [3.847982494, 3.847982494, -3.847982494],
                [-2.78e-16, 7.053659059, 1.43719129],
                [-1.26e-16, 7.053659059, -1.43719129],
                [2.378183969, 6.226166464, 1.26e-16],
                [4.689541707, -5.26241352, 1.461104877],
                [2.364117352, -6.150646585, 2.898296168],
                [2.898296168, -2.364117352, 6.150646585],
                [1.461104877, -4.689541707, 5.26241352],
                [3.847982494, -3.847982494, 3.847982494],
                [4.689541707, -5.26241352, -1.461104877],
                [2.364117352, -6.150646585, -2.898296168],
                [2.898296168, -2.364117352, -6.150646585],
                [1.461104877, -4.689541707, -5.26241352],
                [3.847982494, -3.847982494, -3.847982494],
                [-5.62e-16, -7.053659059, 1.43719129],
                [-5.91e-16, -7.053659059, -1.43719129],
                [2.378183969, -6.226166464, -3.07e-16],
                [-7.053659059, 1.43719129, 1.05e-16],
                [-7.053659059, -1.43719129, -1.50e-16],
                [-6.150646585, 2.898296168, 2.364117352],
                [-5.26241352, 1.461104877, 4.689541707],
                [-6.150646585, -2.898296168, 2.364117352],
                [-5.26241352, -1.461104877, 4.689541707],
                [-6.226166464, -6.58e-17, 2.378183969],
                [-6.150646585, 2.898296168, -2.364117352],
                [-5.26241352, 1.461104877, -4.689541707],
                [-6.150646585, -2.898296168, -2.364117352],
                [-5.26241352, -1.461104877, -4.689541707],
                [-6.226166464, 2.50e-17, -2.378183969],
                [-4.689541707, 5.26241352, 1.461104877],
                [-2.364117352, 6.150646585, 2.898296168],
                [-2.898296168, 2.364117352, 6.150646585],
                [-1.461104877, 4.689541707, 5.26241352],
                [-3.847982494, 3.847982494, 3.847982494],
                [-4.689541707, 5.26241352, -1.461104877],
                [-2.364117352, 6.150646585, -2.898296168],
                [-2.898296168, 2.364117352, -6.150646585],
                [-1.461104877, 4.689541707, -5.26241352],
                [-3.847982494, 3.847982494, -3.847982494],
                [-2.378183969, 6.226166464, 2.47e-17],
                [-4.689541707, -5.26241352, 1.461104877],
                [-2.364117352, -6.150646585, 2.898296168],
                [-2.898296168, -2.364117352, 6.150646585],
                [-1.461104877, -4.689541707, 5.26241352],
                [-3.847982494, -3.847982494, 3.847982494],
                [-4.689541707, -5.26241352, -1.461104877],
                [-2.364117352, -6.150646585, -2.898296168],
                [-2.898296168, -2.364117352, -6.150646585],
                [-1.461104877, -4.689541707, -5.26241352],
                [-3.847982494, -3.847982494, -3.847982494],
                [-2.378183969, -6.226166464, -3.90e-17],
                [1.43719129, -1.83e-18, 7.053659059],
                [-1.43719129, 2.63e-16, 7.053659059],
                [-2.83e-16, 2.378183969, 6.226166464],
                [-2.62e-16, -2.378183969, 6.226166464],
                [1.43719129, -9.85e-17, -7.053659059],
                [-1.43719129, 8.35e-17, -7.053659059],
                [-3.84e-16, 2.378183969, -6.226166464],
                [-2.65e-16, -2.378183969, -6.226166464],
                [9.285026724, 5.738462105, -1.25e-16],
                [9.285026724, -5.738462105, -7.73e-17],
                [-9.285026724, 5.738462105, 1.98e-16],
                [-9.285026724, -5.738462105, -5.26e-17],
                [5.738462105, -2.80e-17, 9.285026724],
                [-5.738462105, 2.40e-16, 9.285026724],
                [5.738462105, 5.10e-16, -9.285026724],
                [-5.738462105, 2.23e-16, -9.285026724],
                [-4.62e-16, 9.285026724, 5.738462105],
                [-6.36e-16, 9.285026724, -5.738462105],
                [-5.11e-16, -9.285026724, 5.738462105],
                [-3.68e-16, -9.285026724, -5.738462105],
                [9.408421292, 2.883041548, 1.32e-16],
                [9.441346435, -2.31e-16, 4.05e-17],
                [9.408421292, -2.883041548, -2.88e-18],
                [8.50248155, 4.348882845, 2.37178104],
                [7.638209717, 2.917536499, 4.720673218],
                [6.720663885, 1.465841297, 7.036640253],
                [8.50248155, -4.348882845, 2.37178104],
                [7.638209717, -2.917536499, 4.720673218],
                [6.720663885, -1.465841297, 7.036640253],
                [8.621171684, 1.452754264, 2.395143048],
                [8.621171684, -1.452754264, 2.395143048],
                [7.723320173, 2.86e-16, 4.745748824],
                [8.50248155, 4.348882845, -2.37178104],
                [7.638209717, 2.917536499, -4.720673218],
                [6.720663885, 1.465841297, -7.036640253],
                [8.50248155, -4.348882845, -2.37178104],
                [7.638209717, -2.917536499, -4.720673218],
                [6.720663885, -1.465841297, -7.036640253],
                [8.621171684, 1.452754264, -2.395143048],
                [8.621171684, -1.452754264, -2.395143048],
                [7.723320173, 9.76e-18, -4.745748824],
                [7.036640253, 6.720663885, 1.465841297],
                [4.720673218, 7.638209717, 2.917536499],
                [2.37178104, 8.50248155, 4.348882845],
                [4.348882845, 2.37178104, 8.50248155],
                [2.917536499, 4.720673218, 7.638209717],
                [1.465841297, 7.036640253, 6.720663885],
                [6.226028636, 5.328177125, 3.875422861],
                [5.328177125, 3.875422861, 6.226028636],
                [3.875422861, 6.226028636, 5.328177125],
                [7.036640253, 6.720663885, -1.465841297],
                [4.720673218, 7.638209717, -2.917536499],
                [2.37178104, 8.50248155, -4.348882845],
                [4.348882845, 2.37178104, -8.50248155],
                [2.917536499, 4.720673218, -7.638209717],
                [1.465841297, 7.036640253, -6.720663885],
                [6.226028636, 5.328177125, -3.875422861],
                [5.328177125, 3.875422861, -6.226028636],
                [3.875422861, 6.226028636, -5.328177125],
                [-6.63e-16, 9.408421292, 2.883041548],
                [-8.27e-16, 9.441346435, -2.08e-16],
                [-4.07e-16, 9.408421292, -2.883041548],
                [4.745748824, 7.723320173, -1.58e-16],
                [2.395143048, 8.621171684, 1.452754264],
                [2.395143048, 8.621171684, -1.452754264],
                [7.036640253, -6.720663885, 1.465841297],
                [4.720673218, -7.638209717, 2.917536499],
                [2.37178104, -8.50248155, 4.348882845],
                [4.348882845, -2.37178104, 8.50248155],
                [2.917536499, -4.720673218, 7.638209717],
                [1.465841297, -7.036640253, 6.720663885],
                [6.226028636, -5.328177125, 3.875422861],
                [5.328177125, -3.875422861, 6.226028636],
                [3.875422861, -6.226028636, 5.328177125],
                [7.036640253, -6.720663885, -1.465841297],
                [4.720673218, -7.638209717, -2.917536499],
                [2.37178104, -8.50248155, -4.348882845],
                [4.348882845, -2.37178104, -8.50248155],
                [2.917536499, -4.720673218, -7.638209717],
                [1.465841297, -7.036640253, -6.720663885],
                [6.226028636, -5.328177125, -3.875422861],
                [5.328177125, -3.875422861, -6.226028636],
                [3.875422861, -6.226028636, -5.328177125],
                [-2.74e-16, -9.408421292, 2.883041548],
                [-4.99e-16, -9.441346435, 2.04e-16],
                [-7.89e-16, -9.408421292, -2.883041548],
                [4.745748824, -7.723320173, -1.56e-16],
                [2.395143048, -8.621171684, 1.452754264],
                [2.395143048, -8.621171684, -1.452754264],
                [-9.408421292, 2.883041548, 4.03e-18],
                [-9.441346435, 6.51e-17, 1.48e-16],
                [-9.408421292, -2.883041548, -1.03e-16],
                [-8.50248155, 4.348882845, 2.37178104],
                [-7.638209717, 2.917536499, 4.720673218],
                [-6.720663885, 1.465841297, 7.036640253],
                [-8.50248155, -4.348882845, 2.37178104],
                [-7.638209717, -2.917536499, 4.720673218],
                [-6.720663885, -1.465841297, 7.036640253],
                [-8.621171684, 1.452754264, 2.395143048],
                [-8.621171684, -1.452754264, 2.395143048],
                [-7.723320173, 8.95e-17, 4.745748824],
                [-8.50248155, 4.348882845, -2.37178104],
                [-7.638209717, 2.917536499, -4.720673218],
                [-6.720663885, 1.465841297, -7.036640253],
                [-8.50248155, -4.348882845, -2.37178104],
                [-7.638209717, -2.917536499, -4.720673218],
                [-6.720663885, -1.465841297, -7.036640253],
                [-8.621171684, 1.452754264, -2.395143048],
                [-8.621171684, -1.452754264, -2.395143048],
                [-7.723320173, -3.64e-16, -4.745748824],
                [-7.036640253, 6.720663885, 1.465841297],
                [-4.720673218, 7.638209717, 2.917536499],
                [-2.37178104, 8.50248155, 4.348882845],
                [-4.348882845, 2.37178104, 8.50248155],
                [-2.917536499, 4.720673218, 7.638209717],
                [-1.465841297, 7.036640253, 6.720663885],
                [-6.226028636, 5.328177125, 3.875422861],
                [-5.328177125, 3.875422861, 6.226028636],
                [-3.875422861, 6.226028636, 5.328177125],
                [-7.036640253, 6.720663885, -1.465841297],
                [-4.720673218, 7.638209717, -2.917536499],
                [-2.37178104, 8.50248155, -4.348882845],
                [-4.348882845, 2.37178104, -8.50248155],
                [-2.917536499, 4.720673218, -7.638209717],
                [-1.465841297, 7.036640253, -6.720663885],
                [-6.226028636, 5.328177125, -3.875422861],
                [-5.328177125, 3.875422861, -6.226028636],
                [-3.875422861, 6.226028636, -5.328177125],
                [-4.745748824, 7.723320173, 2.52e-16],
                [-2.395143048, 8.621171684, 1.452754264],
                [-2.395143048, 8.621171684, -1.452754264],
                [-7.036640253, -6.720663885, 1.465841297],
                [-4.720673218, -7.638209717, 2.917536499],
                [-2.37178104, -8.50248155, 4.348882845],
                [-4.348882845, -2.37178104, 8.50248155],
                [-2.917536499, -4.720673218, 7.638209717],
                [-1.465841297, -7.036640253, 6.720663885],
                [-6.226028636, -5.328177125, 3.875422861],
                [-5.328177125, -3.875422861, 6.226028636],
                [-3.875422861, -6.226028636, 5.328177125],
                [-7.036640253, -6.720663885, -1.465841297],
                [-4.720673218, -7.638209717, -2.917536499],
                [-2.37178104, -8.50248155, -4.348882845],
                [-4.348882845, -2.37178104, -8.50248155],
                [-2.917536499, -4.720673218, -7.638209717],
                [-1.465841297, -7.036640253, -6.720663885],
                [-6.226028636, -5.328177125, -3.875422861],
                [-5.328177125, -3.875422861, -6.226028636],
                [-3.875422861, -6.226028636, -5.328177125],
                [-4.745748824, -7.723320173, -2.17e-16],
                [-2.395143048, -8.621171684, 1.452754264],
                [-2.395143048, -8.621171684, -1.452754264],
                [2.883041548, 3.21e-16, 9.408421292],
                [-3.56e-17, 7.54e-16, 9.441346435],
                [-2.883041548, 5.26e-16, 9.408421292],
                [1.452754264, 2.395143048, 8.621171684],
                [-1.452754264, 2.395143048, 8.621171684],
                [-2.77e-17, 4.745748824, 7.723320173],
                [1.452754264, -2.395143048, 8.621171684],
                [-1.452754264, -2.395143048, 8.621171684],
                [-2.23e-16, -4.745748824, 7.723320173],
                [2.883041548, 1.40e-16, -9.408421292],
                [-1.14e-16, 1.38e-17, -9.441346435],
                [-2.883041548, -1.50e-16, -9.408421292],
                [1.452754264, 2.395143048, -8.621171684],
                [-1.452754264, 2.395143048, -8.621171684],
                [-2.31e-16, 4.745748824, -7.723320173],
                [1.452754264, -2.395143048, -8.621171684],
                [-1.452754264, -2.395143048, -8.621171684],
                [-2.16e-16, -4.745748824, -7.723320173],
                [11.64095174, 7.194503839, 7.32e-16],
                [11.64095174, -7.194503839, -9.28e-17],
                [-11.64095174, 7.194503839, -5.96e-16],
                [-11.64095174, -7.194503839, -4.01e-17],
                [7.194503839, 5.97e-16, 11.64095174],
                [-7.194503839, -9.81e-17, 11.64095174],
                [7.194503839, 1.16e-16, -11.64095174],
                [-7.194503839, 2.17e-16, -11.64095174],
                [-7.42e-16, 11.64095174, 7.194503839],
                [-1.36e-15, 11.64095174, -7.194503839],
                [-3.21e-16, -11.64095174, 7.194503839],
                [-8.78e-16, -11.64095174, -7.194503839],
                [11.78470589, 4.334557453, -5.53e-17],
                [11.83544214, 1.450485732, 3.02e-16],
                [11.83544214, -1.450485732, 2.16e-17],
                [11.78470589, -4.334557453, -5.40e-17],
                [10.87347926, 5.808953123, 2.385622306],
                [10.02329857, 4.382595624, 4.744253462],
                [9.126849086, 2.932109892, 7.091188675],
                [8.194575429, 1.474395669, 9.399083587],
                [10.87347926, -5.808953123, 2.385622306],
                [10.02329857, -4.382595624, 4.744253462],
                [9.126849086, -2.932109892, 7.091188675],
                [8.194575429, -1.474395669, 9.399083587],
                [11.02492707, 2.908012411, 2.41389691],
                [11.06650717, 1.02e-17, 2.420713849],
                [10.16334929, 1.461340141, 4.785211864],
                [11.02492707, -2.908012411, 2.41389691],
                [10.16334929, -1.461340141, 4.785211864],
                [9.227676567, -8.01e-17, 7.119159827],
                [10.87347926, 5.808953123, -2.385622306],
                [10.02329857, 4.382595624, -4.744253462],
                [9.126849086, 2.932109892, -7.091188675],
                [8.194575429, 1.474395669, -9.399083587],
                [10.87347926, -5.808953123, -2.385622306],
                [10.02329857, -4.382595624, -4.744253462],
                [9.126849086, -2.932109892, -7.091188675],
                [8.194575429, -1.474395669, -9.399083587],
                [11.02492707, 2.908012411, -2.41389691],
                [11.06650717, -4.69e-16, -2.420713849],
                [10.16334929, 1.461340141, -4.785211864],
                [11.02492707, -2.908012411, -2.41389691],
                [10.16334929, -1.461340141, -4.785211864],
                [9.227676567, -1.44e-16, -7.119159827],
                [9.399083587, 8.194575429, 1.474395669],
                [7.091188675, 9.126849086, 2.932109892],
                [4.744253462, 10.02329857, 4.382595624],
                [2.385622306, 10.87347926, 5.808953123],
                [5.808953123, 2.385622306, 10.87347926],
                [4.382595624, 4.744253462, 10.02329857],
                [2.932109892, 7.091188675, 9.126849086],
                [1.474395669, 9.399083587, 8.194575429],
                [8.611030163, 6.813779657, 3.905767246],
                [7.742635442, 5.378137427, 6.281295302],
                [6.281295302, 7.742635442, 5.378137427],
                [6.813779657, 3.905767246, 8.611030163],
                [5.378137427, 6.281295302, 7.742635442],
                [3.905767246, 8.611030163, 6.813779657],
                [9.399083587, 8.194575429, -1.474395669],
                [7.091188675, 9.126849086, -2.932109892],
                [4.744253462, 10.02329857, -4.382595624],
                [2.385622306, 10.87347926, -5.808953123],
                [5.808953123, 2.385622306, -10.87347926],
                [4.382595624, 4.744253462, -10.02329857],
                [2.932109892, 7.091188675, -9.126849086],
                [1.474395669, 9.399083587, -8.194575429],
                [8.611030163, 6.813779657, -3.905767246],
                [7.742635442, 5.378137427, -6.281295302],
                [6.281295302, 7.742635442, -5.378137427],
                [6.813779657, 3.905767246, -8.611030163],
                [5.378137427, 6.281295302, -7.742635442],
                [3.905767246, 8.611030163, -6.813779657],
                [-7.42e-16, 11.78470589, 4.334557453],
                [-1.03e-15, 11.83544214, 1.450485732],
                [-1.09e-15, 11.83544214, -1.450485732],
                [-5.21e-16, 11.78470589, -4.334557453],
                [7.119159827, 9.227676567, -2.32e-16],
                [4.785211864, 10.16334929, 1.461340141],
                [4.785211864, 10.16334929, -1.461340141],
                [2.41389691, 11.02492707, 2.908012411],
                [2.420713849, 11.06650717, -4.49e-16],
                [2.41389691, 11.02492707, -2.908012411],
                [9.399083587, -8.194575429, 1.474395669],
                [7.091188675, -9.126849086, 2.932109892],
                [4.744253462, -10.02329857, 4.382595624],
                [2.385622306, -10.87347926, 5.808953123],
                [5.808953123, -2.385622306, 10.87347926],
                [4.382595624, -4.744253462, 10.02329857],
                [2.932109892, -7.091188675, 9.126849086],
                [1.474395669, -9.399083587, 8.194575429],
                [8.611030163, -6.813779657, 3.905767246],
                [7.742635442, -5.378137427, 6.281295302],
                [6.281295302, -7.742635442, 5.378137427],
                [6.813779657, -3.905767246, 8.611030163],
                [5.378137427, -6.281295302, 7.742635442],
                [3.905767246, -8.611030163, 6.813779657],
                [9.399083587, -8.194575429, -1.474395669],
                [7.091188675, -9.126849086, -2.932109892],
                [4.744253462, -10.02329857, -4.382595624],
                [2.385622306, -10.87347926, -5.808953123],
                [5.808953123, -2.385622306, -10.87347926],
                [4.382595624, -4.744253462, -10.02329857],
                [2.932109892, -7.091188675, -9.126849086],
                [1.474395669, -9.399083587, -8.194575429],
                [8.611030163, -6.813779657, -3.905767246],
                [7.742635442, -5.378137427, -6.281295302],
                [6.281295302, -7.742635442, -5.378137427],
                [6.813779657, -3.905767246, -8.611030163],
                [5.378137427, -6.281295302, -7.742635442],
                [3.905767246, -8.611030163, -6.813779657],
                [-8.98e-16, -11.78470589, 4.334557453],
                [-5.12e-16, -11.83544214, 1.450485732],
                [-1.11e-15, -11.83544214, -1.450485732],
                [-9.86e-16, -11.78470589, -4.334557453],
                [7.119159827, -9.227676567, -1.69e-16],
                [4.785211864, -10.16334929, 1.461340141],
                [4.785211864, -10.16334929, -1.461340141],
                [2.41389691, -11.02492707, 2.908012411],
                [2.420713849, -11.06650717, 7.01e-17],
                [2.41389691, -11.02492707, -2.908012411],
                [-11.78470589, 4.334557453, -1.61e-16],
                [-11.83544214, 1.450485732, 1.62e-16],
                [-11.83544214, -1.450485732, -7.19e-17],
                [-11.78470589, -4.334557453, -2.51e-16],
                [-10.87347926, 5.808953123, 2.385622306],
                [-10.02329857, 4.382595624, 4.744253462],
                [-9.126849086, 2.932109892, 7.091188675],
                [-8.194575429, 1.474395669, 9.399083587],
                [-10.87347926, -5.808953123, 2.385622306],
                [-10.02329857, -4.382595624, 4.744253462],
                [-9.126849086, -2.932109892, 7.091188675],
                [-8.194575429, -1.474395669, 9.399083587],
                [-11.02492707, 2.908012411, 2.41389691],
                [-11.06650717, 2.14e-16, 2.420713849],
                [-10.16334929, 1.461340141, 4.785211864],
                [-11.02492707, -2.908012411, 2.41389691],
                [-10.16334929, -1.461340141, 4.785211864],
                [-9.227676567, 2.63e-16, 7.119159827],
                [-10.87347926, 5.808953123, -2.385622306],
                [-10.02329857, 4.382595624, -4.744253462],
                [-9.126849086, 2.932109892, -7.091188675],
                [-8.194575429, 1.474395669, -9.399083587],
                [-10.87347926, -5.808953123, -2.385622306],
                [-10.02329857, -4.382595624, -4.744253462],
                [-9.126849086, -2.932109892, -7.091188675],
                [-8.194575429, -1.474395669, -9.399083587],
                [-11.02492707, 2.908012411, -2.41389691],
                [-11.06650717, 1.64e-18, -2.420713849],
                [-10.16334929, 1.461340141, -4.785211864],
                [-11.02492707, -2.908012411, -2.41389691],
                [-10.16334929, -1.461340141, -4.785211864],
                [-9.227676567, -3.13e-16, -7.119159827],
                [-9.399083587, 8.194575429, 1.474395669],
                [-7.091188675, 9.126849086, 2.932109892],
                [-4.744253462, 10.02329857, 4.382595624],
                [-2.385622306, 10.87347926, 5.808953123],
                [-5.808953123, 2.385622306, 10.87347926],
                [-4.382595624, 4.744253462, 10.02329857],
                [-2.932109892, 7.091188675, 9.126849086],
                [-1.474395669, 9.399083587, 8.194575429],
                [-8.611030163, 6.813779657, 3.905767246],
                [-7.742635442, 5.378137427, 6.281295302],
                [-6.281295302, 7.742635442, 5.378137427],
                [-6.813779657, 3.905767246, 8.611030163],
                [-5.378137427, 6.281295302, 7.742635442],
                [-3.905767246, 8.611030163, 6.813779657],
                [-9.399083587, 8.194575429, -1.474395669],
                [-7.091188675, 9.126849086, -2.932109892],
                [-4.744253462, 10.02329857, -4.382595624],
                [-2.385622306, 10.87347926, -5.808953123],
                [-5.808953123, 2.385622306, -10.87347926],
                [-4.382595624, 4.744253462, -10.02329857],
                [-2.932109892, 7.091188675, -9.126849086],
                [-1.474395669, 9.399083587, -8.194575429],
                [-8.611030163, 6.813779657, -3.905767246],
                [-7.742635442, 5.378137427, -6.281295302],
                [-6.281295302, 7.742635442, -5.378137427],
                [-6.813779657, 3.905767246, -8.611030163],
                [-5.378137427, 6.281295302, -7.742635442],
                [-3.905767246, 8.611030163, -6.813779657],
                [-7.119159827, 9.227676567, -3.06e-16],
                [-4.785211864, 10.16334929, 1.461340141],
                [-4.785211864, 10.16334929, -1.461340141],
                [-2.41389691, 11.02492707, 2.908012411],
                [-2.420713849, 11.06650717, -4.53e-16],
                [-2.41389691, 11.02492707, -2.908012411],
                [-9.399083587, -8.194575429, 1.474395669],
                [-7.091188675, -9.126849086, 2.932109892],
                [-4.744253462, -10.02329857, 4.382595624],
                [-2.385622306, -10.87347926, 5.808953123],
                [-5.808953123, -2.385622306, 10.87347926],
                [-4.382595624, -4.744253462, 10.02329857],
                [-2.932109892, -7.091188675, 9.126849086],
                [-1.474395669, -9.399083587, 8.194575429],
                [-8.611030163, -6.813779657, 3.905767246],
                [-7.742635442, -5.378137427, 6.281295302],
                [-6.281295302, -7.742635442, 5.378137427],
                [-6.813779657, -3.905767246, 8.611030163],
                [-5.378137427, -6.281295302, 7.742635442],
                [-3.905767246, -8.611030163, 6.813779657],
                [-9.399083587, -8.194575429, -1.474395669],
                [-7.091188675, -9.126849086, -2.932109892],
                [-4.744253462, -10.02329857, -4.382595624],
                [-2.385622306, -10.87347926, -5.808953123],
                [-5.808953123, -2.385622306, -10.87347926],
                [-4.382595624, -4.744253462, -10.02329857],
                [-2.932109892, -7.091188675, -9.126849086],
                [-1.474395669, -9.399083587, -8.194575429],
                [-8.611030163, -6.813779657, -3.905767246],
                [-7.742635442, -5.378137427, -6.281295302],
                [-6.281295302, -7.742635442, -5.378137427],
                [-6.813779657, -3.905767246, -8.611030163],
                [-5.378137427, -6.281295302, -7.742635442],
                [-3.905767246, -8.611030163, -6.813779657],
                [-7.119159827, -9.227676567, -2.44e-16],
                [-4.785211864, -10.16334929, 1.461340141],
                [-4.785211864, -10.16334929, -1.461340141],
                [-2.41389691, -11.02492707, 2.908012411],
                [-2.420713849, -11.06650717, 2.48e-16],
                [-2.41389691, -11.02492707, -2.908012411],
                [4.334557453, 6.13e-18, 11.78470589],
                [1.450485732, 4.78e-16, 11.83544214],
                [-1.450485732, 2.94e-16, 11.83544214],
                [-4.334557453, 1.56e-16, 11.78470589],
                [2.908012411, 2.41389691, 11.02492707],
                [-3.53e-16, 2.420713849, 11.06650717],
                [1.461340141, 4.785211864, 10.16334929],
                [-2.908012411, 2.41389691, 11.02492707],
                [-1.461340141, 4.785211864, 10.16334929],
                [1.18e-16, 7.119159827, 9.227676567],
                [2.908012411, -2.41389691, 11.02492707],
                [-1.17e-16, -2.420713849, 11.06650717],
                [1.461340141, -4.785211864, 10.16334929],
                [-2.908012411, -2.41389691, 11.02492707],
                [-1.461340141, -4.785211864, 10.16334929],
                [-5.64e-16, -7.119159827, 9.227676567],
                [4.334557453, 3.09e-17, -11.78470589],
                [1.450485732, 5.14e-17, -11.83544214],
                [-1.450485732, -8.18e-17, -11.83544214],
                [-4.334557453, -1.29e-16, -11.78470589],
                [2.908012411, 2.41389691, -11.02492707],
                [1.71e-16, 2.420713849, -11.06650717],
                [1.461340141, 4.785211864, -10.16334929],
                [-2.908012411, 2.41389691, -11.02492707],
                [-1.461340141, 4.785211864, -10.16334929],
                [-9.70e-17, 7.119159827, -9.227676567],
                [2.908012411, -2.41389691, -11.02492707],
                [5.62e-16, -2.420713849, -11.06650717],
                [1.461340141, -4.785211864, -10.16334929],
                [-2.908012411, -2.41389691, -11.02492707],
                [-1.461340141, -4.785211864, -10.16334929],
                [-2.66e-17, -7.119159827, -9.227676567],
                [13.89959619, 8.590422877, 3.08e-16],
                [13.89959619, -8.590422877, 2.71e-18],
                [-13.89959619, 8.590422877, -7.02e-16],
                [-13.89959619, -8.590422877, 1.44e-16],
                [8.590422877, 4.79e-17, 13.89959619],
                [-8.590422877, -1.38e-16, 13.89959619],
                [8.590422877, -3.02e-17, -13.89959619],
                [-8.590422877, 1.08e-16, -13.89959619],
                [-3.75e-16, 13.89959619, 8.590422877],
                [-7.18e-16, 13.89959619, -8.590422877],
                [-9.99e-16, -13.89959619, 8.590422877],
                [-9.32e-16, -13.89959619, -8.590422877],
                [14.05702801, 5.822346742, 2.73e-16],
                [14.10782477, 2.922613738, 2.34e-16],
                [14.12746471, -4.78e-16, -1.00e-16],
                [14.10782477, -2.922613738, 1.25e-19],
                [14.05702801, -5.822346742, 1.78e-16],
                [13.17157864, 7.255033917, 2.318136544],
                [12.3166073, 5.820864478, 4.689468203],
                [11.42935904, 4.365626685, 7.063732355],
                [10.51033268, 2.898250739, 9.418356564],
                [9.573170462, 1.432687175, 11.73889146],
                [13.17157864, -7.255033917, 2.318136544],
                [12.3166073, -5.820864478, 4.689468203],
                [11.42935904, -4.365626685, 7.063732355],
                [10.51033268, -2.898250739, 9.418356564],
                [9.573170462, -1.432687175, 11.73889146],
                [13.32858729, 4.400431898, 2.371450845],
                [13.39464295, 1.469266979, 2.362339304],
                [12.4716919, 2.962633152, 4.778656527],
                [13.39464295, -1.469266979, 2.362339304],
                [12.51183303, 6.93e-16, 4.779094954],
                [11.56363497, 1.493366173, 7.155980436],
                [13.32858729, -4.400431898, 2.371450845],
                [12.4716919, -2.962633152, 4.778656527],
                [11.56363497, -1.493366173, 7.155980436],
                [10.60897081, -2.15e-16, 9.491499216],
                [13.17157864, 7.255033917, -2.318136544],
                [12.3166073, 5.820864478, -4.689468203],
                [11.42935904, 4.365626685, -7.063732355],
                [10.51033268, 2.898250739, -9.418356564],
                [9.573170462, 1.432687175, -11.73889146],
                [13.17157864, -7.255033917, -2.318136544],
                [12.3166073, -5.820864478, -4.689468203],
                [11.42935904, -4.365626685, -7.063732355],
                [10.51033268, -2.898250739, -9.418356564],
                [9.573170462, -1.432687175, -11.73889146],
                [13.32858729, 4.400431898, -2.371450845],
                [13.39464295, 1.469266979, -2.362339304],
                [12.4716919, 2.962633152, -4.778656527],
                [13.39464295, -1.469266979, -2.362339304],
                [12.51183303, -3.05e-16, -4.779094954],
                [11.56363497, 1.493366173, -7.155980436],
                [13.32858729, -4.400431898, -2.371450845],
                [12.4716919, -2.962633152, -4.778656527],
                [11.56363497, -1.493366173, -7.155980436],
                [10.60897081, -2.14e-16, -9.491499216],
                [11.73889146, 9.573170462, 1.432687175],
                [9.418356564, 10.51033268, 2.898250739],
                [7.063732355, 11.42935904, 4.365626685],
                [4.689468203, 12.3166073, 5.820864478],
                [2.318136544, 13.17157864, 7.255033917],
                [7.255033917, 2.318136544, 13.17157864],
                [5.820864478, 4.689468203, 12.3166073],
                [4.365626685, 7.063732355, 11.42935904],
                [2.898250739, 9.418356564, 10.51033268],
                [1.432687175, 11.73889146, 9.573170462],
                [10.95713644, 8.237519969, 3.83708807],
                [10.10935259, 6.784978439, 6.238662511],
                [8.61598642, 9.201295663, 5.315711459],
                [9.201295663, 5.315711459, 8.61598642],
                [7.732738073, 7.732738073, 7.732738073],
                [6.238662511, 10.10935259, 6.784978439],
                [8.237519969, 3.83708807, 10.95713644],
                [6.784978439, 6.238662511, 10.10935259],
                [5.315711459, 8.61598642, 9.201295663],
                [3.83708807, 10.95713644, 8.237519969],
                [11.73889146, 9.573170462, -1.432687175],
                [9.418356564, 10.51033268, -2.898250739],
                [7.063732355, 11.42935904, -4.365626685],
                [4.689468203, 12.3166073, -5.820864478],
                [2.318136544, 13.17157864, -7.255033917],
                [7.255033917, 2.318136544, -13.17157864],
                [5.820864478, 4.689468203, -12.3166073],
                [4.365626685, 7.063732355, -11.42935904],
                [2.898250739, 9.418356564, -10.51033268],
                [1.432687175, 11.73889146, -9.573170462],
                [10.95713644, 8.237519969, -3.83708807],
                [10.10935259, 6.784978439, -6.238662511],
                [8.61598642, 9.201295663, -5.315711459],
                [9.201295663, 5.315711459, -8.61598642],
                [7.732738073, 7.732738073, -7.732738073],
                [6.238662511, 10.10935259, -6.784978439],
                [8.237519969, 3.83708807, -10.95713644],
                [6.784978439, 6.238662511, -10.10935259],
                [5.315711459, 8.61598642, -9.201295663],
                [3.83708807, 10.95713644, -8.237519969],
                [-9.19e-16, 14.05702801, 5.822346742],
                [-9.88e-16, 14.10782477, 2.922613738],
                [-1.15e-15, 14.12746471, -3.41e-16],
                [-9.25e-16, 14.10782477, -2.922613738],
                [-7.69e-16, 14.05702801, -5.822346742],
                [9.491499216, 10.60897081, -1.18e-16],
                [7.155980436, 11.56363497, 1.493366173],
                [7.155980436, 11.56363497, -1.493366173],
                [4.778656527, 12.4716919, 2.962633152],
                [4.779094954, 12.51183303, 2.86e-16],
                [4.778656527, 12.4716919, -2.962633152],
                [2.371450845, 13.32858729, 4.400431898],
                [2.362339304, 13.39464295, 1.469266979],
                [2.362339304, 13.39464295, -1.469266979],
                [2.371450845, 13.32858729, -4.400431898],
                [11.73889146, -9.573170462, 1.432687175],
                [9.418356564, -10.51033268, 2.898250739],
                [7.063732355, -11.42935904, 4.365626685],
                [4.689468203, -12.3166073, 5.820864478],
                [2.318136544, -13.17157864, 7.255033917],
                [7.255033917, -2.318136544, 13.17157864],
                [5.820864478, -4.689468203, 12.3166073],
                [4.365626685, -7.063732355, 11.42935904],
                [2.898250739, -9.418356564, 10.51033268],
                [1.432687175, -11.73889146, 9.573170462],
                [10.95713644, -8.237519969, 3.83708807],
                [10.10935259, -6.784978439, 6.238662511],
                [8.61598642, -9.201295663, 5.315711459],
                [9.201295663, -5.315711459, 8.61598642],
                [7.732738073, -7.732738073, 7.732738073],
                [6.238662511, -10.10935259, 6.784978439],
                [8.237519969, -3.83708807, 10.95713644],
                [6.784978439, -6.238662511, 10.10935259],
                [5.315711459, -8.61598642, 9.201295663],
                [3.83708807, -10.95713644, 8.237519969],
                [11.73889146, -9.573170462, -1.432687175],
                [9.418356564, -10.51033268, -2.898250739],
                [7.063732355, -11.42935904, -4.365626685],
                [4.689468203, -12.3166073, -5.820864478],
                [2.318136544, -13.17157864, -7.255033917],
                [7.255033917, -2.318136544, -13.17157864],
                [5.820864478, -4.689468203, -12.3166073],
                [4.365626685, -7.063732355, -11.42935904],
                [2.898250739, -9.418356564, -10.51033268],
                [1.432687175, -11.73889146, -9.573170462],
                [10.95713644, -8.237519969, -3.83708807],
                [10.10935259, -6.784978439, -6.238662511],
                [8.61598642, -9.201295663, -5.315711459],
                [9.201295663, -5.315711459, -8.61598642],
                [7.732738073, -7.732738073, -7.732738073],
                [6.238662511, -10.10935259, -6.784978439],
                [8.237519969, -3.83708807, -10.95713644],
                [6.784978439, -6.238662511, -10.10935259],
                [5.315711459, -8.61598642, -9.201295663],
                [3.83708807, -10.95713644, -8.237519969],
                [-8.16e-16, -14.05702801, 5.822346742],
                [-9.69e-16, -14.10782477, 2.922613738],
                [-1.01e-15, -14.12746471, -6.39e-17],
                [-1.47e-15, -14.10782477, -2.922613738],
                [-1.00e-15, -14.05702801, -5.822346742],
                [9.491499216, -10.60897081, 3.30e-16],
                [7.155980436, -11.56363497, 1.493366173],
                [7.155980436, -11.56363497, -1.493366173],
                [4.778656527, -12.4716919, 2.962633152],
                [4.779094954, -12.51183303, 1.12e-16],
                [4.778656527, -12.4716919, -2.962633152],
                [2.371450845, -13.32858729, 4.400431898],
                [2.362339304, -13.39464295, 1.469266979],
                [2.362339304, -13.39464295, -1.469266979],
                [2.371450845, -13.32858729, -4.400431898],
                [-14.05702801, 5.822346742, -4.74e-16],
                [-14.10782477, 2.922613738, -9.56e-17],
                [-14.12746471, 5.87e-16, 1.81e-16],
                [-14.10782477, -2.922613738, 7.56e-17],
                [-14.05702801, -5.822346742, -2.55e-16],
                [-13.17157864, 7.255033917, 2.318136544],
                [-12.3166073, 5.820864478, 4.689468203],
                [-11.42935904, 4.365626685, 7.063732355],
                [-10.51033268, 2.898250739, 9.418356564],
                [-9.573170462, 1.432687175, 11.73889146],
                [-13.17157864, -7.255033917, 2.318136544],
                [-12.3166073, -5.820864478, 4.689468203],
                [-11.42935904, -4.365626685, 7.063732355],
                [-10.51033268, -2.898250739, 9.418356564],
                [-9.573170462, -1.432687175, 11.73889146],
                [-13.32858729, 4.400431898, 2.371450845],
                [-13.39464295, 1.469266979, 2.362339304],
                [-12.4716919, 2.962633152, 4.778656527],
                [-13.39464295, -1.469266979, 2.362339304],
                [-12.51183303, 1.53e-16, 4.779094954],
                [-11.56363497, 1.493366173, 7.155980436],
                [-13.32858729, -4.400431898, 2.371450845],
                [-12.4716919, -2.962633152, 4.778656527],
                [-11.56363497, -1.493366173, 7.155980436],
                [-10.60897081, 1.19e-16, 9.491499216],
                [-13.17157864, 7.255033917, -2.318136544],
                [-12.3166073, 5.820864478, -4.689468203],
                [-11.42935904, 4.365626685, -7.063732355],
                [-10.51033268, 2.898250739, -9.418356564],
                [-9.573170462, 1.432687175, -11.73889146],
                [-13.17157864, -7.255033917, -2.318136544],
                [-12.3166073, -5.820864478, -4.689468203],
                [-11.42935904, -4.365626685, -7.063732355],
                [-10.51033268, -2.898250739, -9.418356564],
                [-9.573170462, -1.432687175, -11.73889146],
                [-13.32858729, 4.400431898, -2.371450845],
                [-13.39464295, 1.469266979, -2.362339304],
                [-12.4716919, 2.962633152, -4.778656527],
                [-13.39464295, -1.469266979, -2.362339304],
                [-12.51183303, 3.42e-16, -4.779094954],
                [-11.56363497, 1.493366173, -7.155980436],
                [-13.32858729, -4.400431898, -2.371450845],
                [-12.4716919, -2.962633152, -4.778656527],
                [-11.56363497, -1.493366173, -7.155980436],
                [-10.60897081, -7.90e-17, -9.491499216],
                [-11.73889146, 9.573170462, 1.432687175],
                [-9.418356564, 10.51033268, 2.898250739],
                [-7.063732355, 11.42935904, 4.365626685],
                [-4.689468203, 12.3166073, 5.820864478],
                [-2.318136544, 13.17157864, 7.255033917],
                [-7.255033917, 2.318136544, 13.17157864],
                [-5.820864478, 4.689468203, 12.3166073],
                [-4.365626685, 7.063732355, 11.42935904],
                [-2.898250739, 9.418356564, 10.51033268],
                [-1.432687175, 11.73889146, 9.573170462],
                [-10.95713644, 8.237519969, 3.83708807],
                [-10.10935259, 6.784978439, 6.238662511],
                [-8.61598642, 9.201295663, 5.315711459],
                [-9.201295663, 5.315711459, 8.61598642],
                [-7.732738073, 7.732738073, 7.732738073],
                [-6.238662511, 10.10935259, 6.784978439],
                [-8.237519969, 3.83708807, 10.95713644],
                [-6.784978439, 6.238662511, 10.10935259],
                [-5.315711459, 8.61598642, 9.201295663],
                [-3.83708807, 10.95713644, 8.237519969],
                [-11.73889146, 9.573170462, -1.432687175],
                [-9.418356564, 10.51033268, -2.898250739],
                [-7.063732355, 11.42935904, -4.365626685],
                [-4.689468203, 12.3166073, -5.820864478],
                [-2.318136544, 13.17157864, -7.255033917],
                [-7.255033917, 2.318136544, -13.17157864],
                [-5.820864478, 4.689468203, -12.3166073],
                [-4.365626685, 7.063732355, -11.42935904],
                [-2.898250739, 9.418356564, -10.51033268],
                [-1.432687175, 11.73889146, -9.573170462],
                [-10.95713644, 8.237519969, -3.83708807],
                [-10.10935259, 6.784978439, -6.238662511],
                [-8.61598642, 9.201295663, -5.315711459],
                [-9.201295663, 5.315711459, -8.61598642],
                [-7.732738073, 7.732738073, -7.732738073],
                [-6.238662511, 10.10935259, -6.784978439],
                [-8.237519969, 3.83708807, -10.95713644],
                [-6.784978439, 6.238662511, -10.10935259],
                [-5.315711459, 8.61598642, -9.201295663],
                [-3.83708807, 10.95713644, -8.237519969],
                [-9.491499216, 10.60897081, 9.13e-17],
                [-7.155980436, 11.56363497, 1.493366173],
                [-7.155980436, 11.56363497, -1.493366173],
                [-4.778656527, 12.4716919, 2.962633152],
                [-4.779094954, 12.51183303, -1.03e-16],
                [-4.778656527, 12.4716919, -2.962633152],
                [-2.371450845, 13.32858729, 4.400431898],
                [-2.362339304, 13.39464295, 1.469266979],
                [-2.362339304, 13.39464295, -1.469266979],
                [-2.371450845, 13.32858729, -4.400431898],
                [-11.73889146, -9.573170462, 1.432687175],
                [-9.418356564, -10.51033268, 2.898250739],
                [-7.063732355, -11.42935904, 4.365626685],
                [-4.689468203, -12.3166073, 5.820864478],
                [-2.318136544, -13.17157864, 7.255033917],
                [-7.255033917, -2.318136544, 13.17157864],
                [-5.820864478, -4.689468203, 12.3166073],
                [-4.365626685, -7.063732355, 11.42935904],
                [-2.898250739, -9.418356564, 10.51033268],
                [-1.432687175, -11.73889146, 9.573170462],
                [-10.95713644, -8.237519969, 3.83708807],
                [-10.10935259, -6.784978439, 6.238662511],
                [-8.61598642, -9.201295663, 5.315711459],
                [-9.201295663, -5.315711459, 8.61598642],
                [-7.732738073, -7.732738073, 7.732738073],
                [-6.238662511, -10.10935259, 6.784978439],
                [-8.237519969, -3.83708807, 10.95713644],
                [-6.784978439, -6.238662511, 10.10935259],
                [-5.315711459, -8.61598642, 9.201295663],
                [-3.83708807, -10.95713644, 8.237519969],
                [-11.73889146, -9.573170462, -1.432687175],
                [-9.418356564, -10.51033268, -2.898250739],
                [-7.063732355, -11.42935904, -4.365626685],
                [-4.689468203, -12.3166073, -5.820864478],
                [-2.318136544, -13.17157864, -7.255033917],
                [-7.255033917, -2.318136544, -13.17157864],
                [-5.820864478, -4.689468203, -12.3166073],
                [-4.365626685, -7.063732355, -11.42935904],
                [-2.898250739, -9.418356564, -10.51033268],
                [-1.432687175, -11.73889146, -9.573170462],
                [-10.95713644, -8.237519969, -3.83708807],
                [-10.10935259, -6.784978439, -6.238662511],
                [-8.61598642, -9.201295663, -5.315711459],
                [-9.201295663, -5.315711459, -8.61598642],
                [-7.732738073, -7.732738073, -7.732738073],
                [-6.238662511, -10.10935259, -6.784978439],
                [-8.237519969, -3.83708807, -10.95713644],
                [-6.784978439, -6.238662511, -10.10935259],
                [-5.315711459, -8.61598642, -9.201295663],
                [-3.83708807, -10.95713644, -8.237519969],
                [-9.491499216, -10.60897081, 6.63e-16],
                [-7.155980436, -11.56363497, 1.493366173],
                [-7.155980436, -11.56363497, -1.493366173],
                [-4.778656527, -12.4716919, 2.962633152],
                [-4.779094954, -12.51183303, 2.10e-16],
                [-4.778656527, -12.4716919, -2.962633152],
                [-2.371450845, -13.32858729, 4.400431898],
                [-2.362339304, -13.39464295, 1.469266979],
                [-2.362339304, -13.39464295, -1.469266979],
                [-2.371450845, -13.32858729, -4.400431898],
                [5.822346742, 3.64e-16, 14.05702801],
                [2.922613738, 2.27e-16, 14.10782477],
                [-2.44e-16, 8.67e-17, 14.12746471],
                [-2.922613738, 4.31e-17, 14.10782477],
                [-5.822346742, 2.98e-16, 14.05702801],
                [4.400431898, 2.371450845, 13.32858729],
                [1.469266979, 2.362339304, 13.39464295],
                [2.962633152, 4.778656527, 12.4716919],
                [-1.469266979, 2.362339304, 13.39464295],
                [-6.43e-16, 4.779094954, 12.51183303],
                [1.493366173, 7.155980436, 11.56363497],
                [-4.400431898, 2.371450845, 13.32858729],
                [-2.962633152, 4.778656527, 12.4716919],
                [-1.493366173, 7.155980436, 11.56363497],
                [-6.00e-16, 9.491499216, 10.60897081],
                [4.400431898, -2.371450845, 13.32858729],
                [1.469266979, -2.362339304, 13.39464295],
                [2.962633152, -4.778656527, 12.4716919],
                [-1.469266979, -2.362339304, 13.39464295],
                [-5.82e-17, -4.779094954, 12.51183303],
                [1.493366173, -7.155980436, 11.56363497],
                [-4.400431898, -2.371450845, 13.32858729],
                [-2.962633152, -4.778656527, 12.4716919],
                [-1.493366173, -7.155980436, 11.56363497],
                [-1.83e-16, -9.491499216, 10.60897081],
                [5.822346742, 3.56e-16, -14.05702801],
                [2.922613738, 1.28e-16, -14.10782477],
                [1.18e-16, 5.88e-17, -14.12746471],
                [-2.922613738, -1.83e-16, -14.10782477],
                [-5.822346742, 3.49e-16, -14.05702801],
                [4.400431898, 2.371450845, -13.32858729],
                [1.469266979, 2.362339304, -13.39464295],
                [2.962633152, 4.778656527, -12.4716919],
                [-1.469266979, 2.362339304, -13.39464295],
                [-4.83e-17, 4.779094954, -12.51183303],
                [1.493366173, 7.155980436, -11.56363497],
                [-4.400431898, 2.371450845, -13.32858729],
                [-2.962633152, 4.778656527, -12.4716919],
                [-1.493366173, 7.155980436, -11.56363497],
                [-4.81e-16, 9.491499216, -10.60897081],
                [4.400431898, -2.371450845, -13.32858729],
                [1.469266979, -2.362339304, -13.39464295],
                [2.962633152, -4.778656527, -12.4716919],
                [-1.469266979, -2.362339304, -13.39464295],
                [-2.22e-16, -4.779094954, -12.51183303],
                [1.493366173, -7.155980436, -11.56363497],
                [-4.400431898, -2.371450845, -13.32858729],
                [-2.962633152, -4.778656527, -12.4716919],
                [-1.493366173, -7.155980436, -11.56363497],
                [-7.00e-16, -9.491499216, -10.60897081],
            ]
        ]
    )
    nat = traj.shape[1]
    u = MDAnalysis.Universe.empty(
        nat, trajectory=True, atom_resindex=[0] * nat, residue_segindex=[0]
    )
    u.add_TopologyAttr("type", ["Au"] * nat)
    u.atoms.positions = traj[0]
    u.trajectory = MDAnalysis.coordinates.memory.MemoryReader(
        traj,
        order="fac",
        dimensions=numpy.array([[80.0, 80.0, 80.0, 90, 90, 90]]),
    )
    return u


def giveUniverse_ChangingBox(angles: set = (90.0, 90.0, 90.0)) -> MDAnalysis.Universe:
    traj = numpy.array(
        [
            [[0.0, 0.0, 0.0], [1.0, 1.0, 1.0], [2.0, 2.0, 2.0], [3.0, 3.0, 3.0]],
            [[0.1, 0.1, 0.1], [1.1, 1.1, 1.1], [2.1, 2.1, 2.1], [3.1, 3.1, 3.1]],
            [[0.2, 0.2, 0.2], [1.2, 1.2, 1.2], [2.2, 2.2, 2.2], [3.2, 3.2, 3.2]],
            [[0.3, 0.3, 0.3], [1.3, 1.3, 1.3], [2.3, 2.3, 2.3], [3.3, 3.3, 3.3]],
            [[0.4, 0.4, 0.4], [1.4, 1.4, 1.4], [2.4, 2.4, 2.4], [3.4, 3.4, 3.4]],
        ]
    )
    u = MDAnalysis.Universe.empty(
        4, trajectory=True, atom_resindex=[0, 0, 0, 0], residue_segindex=[0]
    )
    dimensions = numpy.array(
        [[6.0, 6.0, 6.0, angles[0], angles[1], angles[2]]] * traj.shape[0]
    )
    multiplier = numpy.array(
        [
            [1.5 - 0.5 * numpy.cos(k / traj.shape[0] * 2 * numpy.pi)]
            for k in range(traj.shape[0])
        ]
    )
    dimensions[:, :3] *= multiplier

    u.add_TopologyAttr("type", ["H"] * 4)
    u.atoms.positions = traj[0]
    u.trajectory = MDAnalysis.coordinates.memory.MemoryReader(
        traj,
        order="fac",
        # this tests the non orthogonality of the box
        dimensions=dimensions,
    )
    # adding this for recognisign this univers during tests:
    u.myUsefulName = "ChangingBox"
    return u


def giveUniverse_LongChangingBox(
    angles: set = (90.0, 90.0, 90.0)
) -> MDAnalysis.Universe:
    trajLen = 300
    traj = numpy.array(
        [[[0.0, 0.0, 0.0], [1.0, 1.0, 1.0], [2.0, 2.0, 2.0], [3.0, 3.0, 3.0]]] * trajLen
    )
    print(traj.shape)
    rng = numpy.random.default_rng(12345)
    traj[1:] += rng.random(size=traj[1:].shape) * 0.2
    u = MDAnalysis.Universe.empty(
        4, trajectory=True, atom_resindex=[0, 0, 0, 0], residue_segindex=[0]
    )
    dimensions = numpy.array(
        [[6.0, 6.0, 6.0, angles[0], angles[1], angles[2]]] * traj.shape[0]
    )
    multiplier = numpy.array(
        [[1.5 - 0.5 * numpy.cos(k / 20 * 2 * numpy.pi)] for k in range(traj.shape[0])]
    )
    dimensions[:, :3] *= multiplier

    u.add_TopologyAttr("type", ["H"] * 4)
    u.atoms.positions = traj[0]
    u.trajectory = MDAnalysis.coordinates.memory.MemoryReader(
        traj,
        order="fac",
        # this tests the non orthogonality of the box
        dimensions=dimensions,
    )
    # adding this for recognisign this univers during tests:
    u.myUsefulName = "ChangingBox"
    return u
