"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const chain_1 = require("../chain");
const fields_1 = require("../fields");
const util_1 = require("../util");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Task state in the state machine
 *
 * Reaching a Task state causes some work to be executed, represented by the
 * Task's resource property. Task constructs represent a generic Amazon
 * States Language Task.
 *
 * For some resource types, more specific subclasses of Task may be available
 * which are more convenient to use.
 */
class Task extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.timeout = props.timeout;
        const taskProps = props.task.bind(this);
        this.taskProps = {
            ...taskProps,
            parameters: util_1.noEmptyObject({ ...taskProps.parameters || {}, ...props.parameters || {} }),
        };
        this.endStates = [this];
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            ...this.renderNextEnd(),
            ...this.renderRetryCatch(),
            ...this.renderInputOutput(),
            Type: state_type_1.StateType.TASK,
            Comment: this.comment,
            Resource: this.taskProps.resourceArn,
            Parameters: this.taskProps.parameters && fields_1.FieldUtils.renderObject(this.taskProps.parameters),
            ResultPath: state_1.renderJsonPath(this.resultPath),
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
            HeartbeatSeconds: this.taskProps.heartbeat && this.taskProps.heartbeat.toSeconds(),
        };
    }
    /**
     * Return the given named metric for this Task
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: this.taskProps.metricDimensions,
            statistic: 'sum',
            ...props
        }).attachTo(this);
    }
    /**
     * The interval, in milliseconds, between the time the Task starts and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricRunTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'RunTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, for which the activity stays in the schedule state.
     *
     * @default average over 5 minutes
     */
    metricScheduleTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'ScheduleTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, between the time the activity is scheduled and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricTime(props) {
        return this.taskMetric(this.taskProps.metricPrefixSingular, 'Time', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the number of times this activity is scheduled
     *
     * @default sum over 5 minutes
     */
    metricScheduled(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Scheduled', props);
    }
    /**
     * Metric for the number of times this activity times out
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'TimedOut', props);
    }
    /**
     * Metric for the number of times this activity is started
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Started', props);
    }
    /**
     * Metric for the number of times this activity succeeds
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Succeeded', props);
    }
    /**
     * Metric for the number of times this activity fails
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'Failed', props);
    }
    /**
     * Metric for the number of times the heartbeat times out for this activity
     *
     * @default sum over 5 minutes
     */
    metricHeartbeatTimedOut(props) {
        return this.taskMetric(this.taskProps.metricPrefixPlural, 'HeartbeatTimedOut', props);
    }
    whenBoundToGraph(graph) {
        super.whenBoundToGraph(graph);
        for (const policyStatement of this.taskProps.policyStatements || []) {
            graph.registerPolicyStatement(policyStatement);
        }
    }
    taskMetric(prefix, suffix, props) {
        if (prefix === undefined) {
            throw new Error('This Task Resource does not expose metrics');
        }
        return this.metric(prefix + suffix, props);
    }
}
exports.Task = Task;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFzay5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRhc2sudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxzREFBc0Q7QUFFdEQsb0NBQWlDO0FBQ2pDLHNDQUF1QztBQUl2QyxrQ0FBd0M7QUFDeEMscURBQWlEO0FBQ2pELG1DQUFnRDtBQWtGaEQ7Ozs7Ozs7OztHQVNHO0FBQ0gsTUFBYSxJQUFLLFNBQVEsYUFBSztJQUszQixZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQWdCO1FBQzFELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXhCLElBQUksQ0FBQyxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM3QixNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV4QyxJQUFJLENBQUMsU0FBUyxHQUFHO1lBQ2IsR0FBRyxTQUFTO1lBQ1osVUFBVSxFQUFFLG9CQUFhLENBQUMsRUFBRSxHQUFHLFNBQVMsQ0FBQyxVQUFVLElBQUksRUFBRSxFQUFFLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxFQUFFLEVBQUUsQ0FBQztTQUMxRixDQUFDO1FBQ0YsSUFBSSxDQUFDLFNBQVMsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzVCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLFFBQVEsQ0FBQyxRQUFvQixFQUFFO1FBQ2xDLEtBQUssQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdkIsT0FBTyxJQUFJLENBQUM7SUFDaEIsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksUUFBUSxDQUFDLE9BQW1CLEVBQUUsUUFBb0IsRUFBRTtRQUN2RCxLQUFLLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDaEIsQ0FBQztJQUVEOztPQUVHO0lBQ0ksSUFBSSxDQUFDLElBQWdCO1FBQ3hCLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sYUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksV0FBVztRQUNkLE9BQU87WUFDSCxHQUFHLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdkIsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsR0FBRyxJQUFJLENBQUMsaUJBQWlCLEVBQUU7WUFDM0IsSUFBSSxFQUFFLHNCQUFTLENBQUMsSUFBSTtZQUNwQixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVztZQUNwQyxVQUFVLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLElBQUksbUJBQVUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUM7WUFDM0YsVUFBVSxFQUFFLHNCQUFjLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQztZQUMzQyxjQUFjLEVBQUUsSUFBSSxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRTtZQUN4RCxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxTQUFTLEVBQUU7U0FDckYsQ0FBQztJQUNOLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFVBQWtCLEVBQUUsS0FBZ0M7UUFDOUQsT0FBTyxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUM7WUFDekIsU0FBUyxFQUFFLFlBQVk7WUFDdkIsVUFBVTtZQUNWLFVBQVUsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLGdCQUFnQjtZQUMzQyxTQUFTLEVBQUUsS0FBSztZQUNoQixHQUFHLEtBQUs7U0FDWCxDQUFDLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksYUFBYSxDQUFDLEtBQWdDO1FBQ2pELE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLG9CQUFvQixFQUFFLFNBQVMsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQzNHLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksa0JBQWtCLENBQUMsS0FBZ0M7UUFDdEQsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsb0JBQW9CLEVBQUUsY0FBYyxFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUM7SUFDaEgsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxVQUFVLENBQUMsS0FBZ0M7UUFDOUMsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsb0JBQW9CLEVBQUUsTUFBTSxFQUFFLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxHQUFHLEtBQUssRUFBRSxDQUFDLENBQUM7SUFDeEcsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxlQUFlLENBQUMsS0FBZ0M7UUFDbkQsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsa0JBQWtCLEVBQUUsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2xGLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksY0FBYyxDQUFDLEtBQWdDO1FBQ2xELE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGFBQWEsQ0FBQyxLQUFnQztRQUNqRCxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxlQUFlLENBQUMsS0FBZ0M7UUFDbkQsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsa0JBQWtCLEVBQUUsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2xGLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksWUFBWSxDQUFDLEtBQWdDO1FBQ2hELE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixFQUFFLFFBQVEsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMvRSxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLHVCQUF1QixDQUFDLEtBQWdDO1FBQzNELE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixFQUFFLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzFGLENBQUM7SUFFUyxnQkFBZ0IsQ0FBQyxLQUFpQjtRQUN4QyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDOUIsS0FBSyxNQUFNLGVBQWUsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLGdCQUFnQixJQUFJLEVBQUUsRUFBRTtZQUNqRSxLQUFLLENBQUMsdUJBQXVCLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDbEQ7SUFDTCxDQUFDO0lBRU8sVUFBVSxDQUFDLE1BQTBCLEVBQUUsTUFBYyxFQUFFLEtBQWdDO1FBQzNGLElBQUksTUFBTSxLQUFLLFNBQVMsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLDRDQUE0QyxDQUFDLENBQUM7U0FDakU7UUFDRCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxHQUFHLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQztJQUMvQyxDQUFDO0NBQ0o7QUEvS0Qsb0JBK0tDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2xvdWR3YXRjaCBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDaGFpbiB9IGZyb20gJy4uL2NoYWluJztcbmltcG9ydCB7IEZpZWxkVXRpbHMgfSBmcm9tICcuLi9maWVsZHMnO1xuaW1wb3J0IHsgU3RhdGVHcmFwaCB9IGZyb20gJy4uL3N0YXRlLWdyYXBoJztcbmltcG9ydCB7IElTdGVwRnVuY3Rpb25zVGFzaywgU3RlcEZ1bmN0aW9uc1Rhc2tDb25maWcgfSBmcm9tICcuLi9zdGVwLWZ1bmN0aW9ucy10YXNrJztcbmltcG9ydCB7IENhdGNoUHJvcHMsIElDaGFpbmFibGUsIElOZXh0YWJsZSwgUmV0cnlQcm9wcyB9IGZyb20gJy4uL3R5cGVzJztcbmltcG9ydCB7IG5vRW1wdHlPYmplY3QgfSBmcm9tICcuLi91dGlsJztcbmltcG9ydCB7IFN0YXRlVHlwZSB9IGZyb20gJy4vcHJpdmF0ZS9zdGF0ZS10eXBlJztcbmltcG9ydCB7IHJlbmRlckpzb25QYXRoLCBTdGF0ZSB9IGZyb20gJy4vc3RhdGUnO1xuXG4vKipcbiAqIFByb3BzIHRoYXQgYXJlIGNvbW1vbiB0byBhbGwgdGFza3NcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBUYXNrUHJvcHMge1xuICAgIC8qKlxuICAgICAqIEFjdHVhbCB0YXNrIHRvIGJlIGludm9rZWQgaW4gdGhpcyB3b3JrZmxvd1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHRhc2s6IElTdGVwRnVuY3Rpb25zVGFzaztcblxuICAgIC8qKlxuICAgICAqIEFuIG9wdGlvbmFsIGRlc2NyaXB0aW9uIGZvciB0aGlzIHN0YXRlXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBjb21tZW50XG4gICAgICovXG4gICAgcmVhZG9ubHkgY29tbWVudD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIEpTT05QYXRoIGV4cHJlc3Npb24gdG8gc2VsZWN0IHBhcnQgb2YgdGhlIHN0YXRlIHRvIGJlIHRoZSBpbnB1dCB0byB0aGlzIHN0YXRlLlxuICAgICAqXG4gICAgICogTWF5IGFsc28gYmUgdGhlIHNwZWNpYWwgdmFsdWUgRElTQ0FSRCwgd2hpY2ggd2lsbCBjYXVzZSB0aGUgZWZmZWN0aXZlXG4gICAgICogaW5wdXQgdG8gYmUgdGhlIGVtcHR5IG9iamVjdCB7fS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0ICRcbiAgICAgKi9cbiAgICByZWFkb25seSBpbnB1dFBhdGg/OiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBKU09OUGF0aCBleHByZXNzaW9uIHRvIHNlbGVjdCBwYXJ0IG9mIHRoZSBzdGF0ZSB0byBiZSB0aGUgb3V0cHV0IHRvIHRoaXMgc3RhdGUuXG4gICAgICpcbiAgICAgKiBNYXkgYWxzbyBiZSB0aGUgc3BlY2lhbCB2YWx1ZSBESVNDQVJELCB3aGljaCB3aWxsIGNhdXNlIHRoZSBlZmZlY3RpdmVcbiAgICAgKiBvdXRwdXQgdG8gYmUgdGhlIGVtcHR5IG9iamVjdCB7fS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0ICRcbiAgICAgKi9cbiAgICByZWFkb25seSBvdXRwdXRQYXRoPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogSlNPTlBhdGggZXhwcmVzc2lvbiB0byBpbmRpY2F0ZSB3aGVyZSB0byBpbmplY3QgdGhlIHN0YXRlJ3Mgb3V0cHV0XG4gICAgICpcbiAgICAgKiBNYXkgYWxzbyBiZSB0aGUgc3BlY2lhbCB2YWx1ZSBESVNDQVJELCB3aGljaCB3aWxsIGNhdXNlIHRoZSBzdGF0ZSdzXG4gICAgICogaW5wdXQgdG8gYmVjb21lIGl0cyBvdXRwdXQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAkXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzdWx0UGF0aD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFBhcmFtZXRlcnMgdG8gaW52b2tlIHRoZSB0YXNrIHdpdGhcbiAgICAgKlxuICAgICAqIEl0IGlzIG5vdCByZWNvbW1lbmRlZCB0byB1c2UgdGhpcyBmaWVsZC4gVGhlIG9iamVjdCB0aGF0IGlzIHBhc3NlZCBpblxuICAgICAqIHRoZSBgdGFza2AgcHJvcGVydHkgd2lsbCB0YWtlIGNhcmUgb2YgcmV0dXJuaW5nIHRoZSByaWdodCB2YWx1ZXMgZm9yIHRoZVxuICAgICAqIGBQYXJhbWV0ZXJzYCBmaWVsZCBpbiB0aGUgU3RlcCBGdW5jdGlvbnMgZGVmaW5pdGlvbi5cbiAgICAgKlxuICAgICAqIFRoZSB2YXJpb3VzIGNsYXNzZXMgdGhhdCBpbXBsZW1lbnQgYElTdGVwRnVuY3Rpb25zVGFza2Agd2lsbCB0YWtlIGFcbiAgICAgKiBwcm9wZXJ0aWVzIHdoaWNoIG1ha2Ugc2Vuc2UgZm9yIHRoZSB0YXNrIHR5cGUuIEZvciBleGFtcGxlLCBmb3JcbiAgICAgKiBgSW52b2tlRnVuY3Rpb25gIHRoZSBmaWVsZCB0aGF0IHBvcHVsYXRlcyB0aGUgYHBhcmFtZXRlcnNgIGZpZWxkIHdpbGwgYmVcbiAgICAgKiBjYWxsZWQgYHBheWxvYWRgLCBhbmQgZm9yIHRoZSBgUHVibGlzaFRvVG9waWNgIHRoZSBgcGFyYW1ldGVyc2AgZmllbGRcbiAgICAgKiB3aWxsIGJlIHBvcHVsYXRlZCB2aWEgYSBjb21iaW5hdGlvbiBvZiB0aGUgcmVmZXJlbmNlZCB0b3BpYywgc3ViamVjdCBhbmRcbiAgICAgKiBtZXNzYWdlLlxuICAgICAqXG4gICAgICogSWYgcGFzc2VkIGFueXdheSwgdGhlIGtleXMgaW4gdGhpcyBtYXAgd2lsbCBvdmVycmlkZSB0aGUgcGFyYW1ldGVyc1xuICAgICAqIHJldHVybmVkIGJ5IHRoZSB0YXNrIG9iamVjdC5cbiAgICAgKlxuICAgICAqIEBzZWVcbiAgICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2lucHV0LW91dHB1dC1pbnB1dHBhdGgtcGFyYW1zLmh0bWwjaW5wdXQtb3V0cHV0LXBhcmFtZXRlcnNcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gVXNlIHRoZSBwYXJhbWV0ZXJzIGltcGxpZWQgYnkgdGhlIGB0YXNrYCBwcm9wZXJ0eVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBhcmFtZXRlcnM/OiB7IFtuYW1lOiBzdHJpbmddOiBhbnkgfTtcblxuICAgIC8qKlxuICAgICAqIE1heGltdW0gcnVuIHRpbWUgb2YgdGhpcyBzdGF0ZVxuICAgICAqXG4gICAgICogSWYgdGhlIHN0YXRlIHRha2VzIGxvbmdlciB0aGFuIHRoaXMgYW1vdW50IG9mIHRpbWUgdG8gY29tcGxldGUsIGEgJ1RpbWVvdXQnIGVycm9yIGlzIHJhaXNlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IDYwXG4gICAgICovXG4gICAgcmVhZG9ubHkgdGltZW91dD86IGNkay5EdXJhdGlvbjtcbn1cblxuLyoqXG4gKiBEZWZpbmUgYSBUYXNrIHN0YXRlIGluIHRoZSBzdGF0ZSBtYWNoaW5lXG4gKlxuICogUmVhY2hpbmcgYSBUYXNrIHN0YXRlIGNhdXNlcyBzb21lIHdvcmsgdG8gYmUgZXhlY3V0ZWQsIHJlcHJlc2VudGVkIGJ5IHRoZVxuICogVGFzaydzIHJlc291cmNlIHByb3BlcnR5LiBUYXNrIGNvbnN0cnVjdHMgcmVwcmVzZW50IGEgZ2VuZXJpYyBBbWF6b25cbiAqIFN0YXRlcyBMYW5ndWFnZSBUYXNrLlxuICpcbiAqIEZvciBzb21lIHJlc291cmNlIHR5cGVzLCBtb3JlIHNwZWNpZmljIHN1YmNsYXNzZXMgb2YgVGFzayBtYXkgYmUgYXZhaWxhYmxlXG4gKiB3aGljaCBhcmUgbW9yZSBjb252ZW5pZW50IHRvIHVzZS5cbiAqL1xuZXhwb3J0IGNsYXNzIFRhc2sgZXh0ZW5kcyBTdGF0ZSBpbXBsZW1lbnRzIElOZXh0YWJsZSB7XG4gICAgcHVibGljIHJlYWRvbmx5IGVuZFN0YXRlczogSU5leHRhYmxlW107XG4gICAgcHJpdmF0ZSByZWFkb25seSB0aW1lb3V0PzogY2RrLkR1cmF0aW9uO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgdGFza1Byb3BzOiBTdGVwRnVuY3Rpb25zVGFza0NvbmZpZztcblxuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVGFza1Byb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgICAgIHRoaXMudGltZW91dCA9IHByb3BzLnRpbWVvdXQ7XG4gICAgICAgIGNvbnN0IHRhc2tQcm9wcyA9IHByb3BzLnRhc2suYmluZCh0aGlzKTtcblxuICAgICAgICB0aGlzLnRhc2tQcm9wcyA9IHtcbiAgICAgICAgICAgIC4uLnRhc2tQcm9wcyxcbiAgICAgICAgICAgIHBhcmFtZXRlcnM6IG5vRW1wdHlPYmplY3QoeyAuLi50YXNrUHJvcHMucGFyYW1ldGVycyB8fCB7fSwgLi4ucHJvcHMucGFyYW1ldGVycyB8fCB7fSB9KSxcbiAgICAgICAgfTtcbiAgICAgICAgdGhpcy5lbmRTdGF0ZXMgPSBbdGhpc107XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogQWRkIHJldHJ5IGNvbmZpZ3VyYXRpb24gZm9yIHRoaXMgc3RhdGVcbiAgICAgKlxuICAgICAqIFRoaXMgY29udHJvbHMgaWYgYW5kIGhvdyB0aGUgZXhlY3V0aW9uIHdpbGwgYmUgcmV0cmllZCBpZiBhIHBhcnRpY3VsYXJcbiAgICAgKiBlcnJvciBvY2N1cnMuXG4gICAgICovXG4gICAgcHVibGljIGFkZFJldHJ5KHByb3BzOiBSZXRyeVByb3BzID0ge30pOiBUYXNrIHtcbiAgICAgICAgc3VwZXIuX2FkZFJldHJ5KHByb3BzKTtcbiAgICAgICAgcmV0dXJuIHRoaXM7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogQWRkIGEgcmVjb3ZlcnkgaGFuZGxlciBmb3IgdGhpcyBzdGF0ZVxuICAgICAqXG4gICAgICogV2hlbiBhIHBhcnRpY3VsYXIgZXJyb3Igb2NjdXJzLCBleGVjdXRpb24gd2lsbCBjb250aW51ZSBhdCB0aGUgZXJyb3JcbiAgICAgKiBoYW5kbGVyIGluc3RlYWQgb2YgZmFpbGluZyB0aGUgc3RhdGUgbWFjaGluZSBleGVjdXRpb24uXG4gICAgICovXG4gICAgcHVibGljIGFkZENhdGNoKGhhbmRsZXI6IElDaGFpbmFibGUsIHByb3BzOiBDYXRjaFByb3BzID0ge30pOiBUYXNrIHtcbiAgICAgICAgc3VwZXIuX2FkZENhdGNoKGhhbmRsZXIuc3RhcnRTdGF0ZSwgcHJvcHMpO1xuICAgICAgICByZXR1cm4gdGhpcztcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBDb250aW51ZSBub3JtYWwgZXhlY3V0aW9uIHdpdGggdGhlIGdpdmVuIHN0YXRlXG4gICAgICovXG4gICAgcHVibGljIG5leHQobmV4dDogSUNoYWluYWJsZSk6IENoYWluIHtcbiAgICAgICAgc3VwZXIubWFrZU5leHQobmV4dC5zdGFydFN0YXRlKTtcbiAgICAgICAgcmV0dXJuIENoYWluLnNlcXVlbmNlKHRoaXMsIG5leHQpO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIFJldHVybiB0aGUgQW1hem9uIFN0YXRlcyBMYW5ndWFnZSBvYmplY3QgZm9yIHRoaXMgc3RhdGVcbiAgICAgKi9cbiAgICBwdWJsaWMgdG9TdGF0ZUpzb24oKTogb2JqZWN0IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIC4uLnRoaXMucmVuZGVyTmV4dEVuZCgpLFxuICAgICAgICAgICAgLi4udGhpcy5yZW5kZXJSZXRyeUNhdGNoKCksXG4gICAgICAgICAgICAuLi50aGlzLnJlbmRlcklucHV0T3V0cHV0KCksXG4gICAgICAgICAgICBUeXBlOiBTdGF0ZVR5cGUuVEFTSyxcbiAgICAgICAgICAgIENvbW1lbnQ6IHRoaXMuY29tbWVudCxcbiAgICAgICAgICAgIFJlc291cmNlOiB0aGlzLnRhc2tQcm9wcy5yZXNvdXJjZUFybixcbiAgICAgICAgICAgIFBhcmFtZXRlcnM6IHRoaXMudGFza1Byb3BzLnBhcmFtZXRlcnMgJiYgRmllbGRVdGlscy5yZW5kZXJPYmplY3QodGhpcy50YXNrUHJvcHMucGFyYW1ldGVycyksXG4gICAgICAgICAgICBSZXN1bHRQYXRoOiByZW5kZXJKc29uUGF0aCh0aGlzLnJlc3VsdFBhdGgpLFxuICAgICAgICAgICAgVGltZW91dFNlY29uZHM6IHRoaXMudGltZW91dCAmJiB0aGlzLnRpbWVvdXQudG9TZWNvbmRzKCksXG4gICAgICAgICAgICBIZWFydGJlYXRTZWNvbmRzOiB0aGlzLnRhc2tQcm9wcy5oZWFydGJlYXQgJiYgdGhpcy50YXNrUHJvcHMuaGVhcnRiZWF0LnRvU2Vjb25kcygpLFxuICAgICAgICB9O1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIFJldHVybiB0aGUgZ2l2ZW4gbmFtZWQgbWV0cmljIGZvciB0aGlzIFRhc2tcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHN1bSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWMobWV0cmljTmFtZTogc3RyaW5nLCBwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIG5ldyBjbG91ZHdhdGNoLk1ldHJpYyh7XG4gICAgICAgICAgICBuYW1lc3BhY2U6ICdBV1MvU3RhdGVzJyxcbiAgICAgICAgICAgIG1ldHJpY05hbWUsXG4gICAgICAgICAgICBkaW1lbnNpb25zOiB0aGlzLnRhc2tQcm9wcy5tZXRyaWNEaW1lbnNpb25zLFxuICAgICAgICAgICAgc3RhdGlzdGljOiAnc3VtJyxcbiAgICAgICAgICAgIC4uLnByb3BzXG4gICAgICAgIH0pLmF0dGFjaFRvKHRoaXMpO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIFRoZSBpbnRlcnZhbCwgaW4gbWlsbGlzZWNvbmRzLCBiZXR3ZWVuIHRoZSB0aW1lIHRoZSBUYXNrIHN0YXJ0cyBhbmQgdGhlIHRpbWUgaXQgY2xvc2VzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgYXZlcmFnZSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNSdW5UaW1lKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICByZXR1cm4gdGhpcy50YXNrTWV0cmljKHRoaXMudGFza1Byb3BzLm1ldHJpY1ByZWZpeFNpbmd1bGFyLCAnUnVuVGltZScsIHsgc3RhdGlzdGljOiAnYXZnJywgLi4ucHJvcHMgfSk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogVGhlIGludGVydmFsLCBpbiBtaWxsaXNlY29uZHMsIGZvciB3aGljaCB0aGUgYWN0aXZpdHkgc3RheXMgaW4gdGhlIHNjaGVkdWxlIHN0YXRlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgYXZlcmFnZSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNTY2hlZHVsZVRpbWUocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLnRhc2tNZXRyaWModGhpcy50YXNrUHJvcHMubWV0cmljUHJlZml4U2luZ3VsYXIsICdTY2hlZHVsZVRpbWUnLCB7IHN0YXRpc3RpYzogJ2F2ZycsIC4uLnByb3BzIH0pO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIFRoZSBpbnRlcnZhbCwgaW4gbWlsbGlzZWNvbmRzLCBiZXR3ZWVuIHRoZSB0aW1lIHRoZSBhY3Rpdml0eSBpcyBzY2hlZHVsZWQgYW5kIHRoZSB0aW1lIGl0IGNsb3Nlcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGF2ZXJhZ2Ugb3ZlciA1IG1pbnV0ZXNcbiAgICAgKi9cbiAgICBwdWJsaWMgbWV0cmljVGltZShwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIHRoaXMudGFza01ldHJpYyh0aGlzLnRhc2tQcm9wcy5tZXRyaWNQcmVmaXhTaW5ndWxhciwgJ1RpbWUnLCB7IHN0YXRpc3RpYzogJ2F2ZycsIC4uLnByb3BzIH0pO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIE1ldHJpYyBmb3IgdGhlIG51bWJlciBvZiB0aW1lcyB0aGlzIGFjdGl2aXR5IGlzIHNjaGVkdWxlZFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgc3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY1NjaGVkdWxlZChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIHRoaXMudGFza01ldHJpYyh0aGlzLnRhc2tQcm9wcy5tZXRyaWNQcmVmaXhQbHVyYWwsICdTY2hlZHVsZWQnLCBwcm9wcyk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogTWV0cmljIGZvciB0aGUgbnVtYmVyIG9mIHRpbWVzIHRoaXMgYWN0aXZpdHkgdGltZXMgb3V0XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBzdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICAgKi9cbiAgICBwdWJsaWMgbWV0cmljVGltZWRPdXQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLnRhc2tNZXRyaWModGhpcy50YXNrUHJvcHMubWV0cmljUHJlZml4UGx1cmFsLCAnVGltZWRPdXQnLCBwcm9wcyk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogTWV0cmljIGZvciB0aGUgbnVtYmVyIG9mIHRpbWVzIHRoaXMgYWN0aXZpdHkgaXMgc3RhcnRlZFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgc3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY1N0YXJ0ZWQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLnRhc2tNZXRyaWModGhpcy50YXNrUHJvcHMubWV0cmljUHJlZml4UGx1cmFsLCAnU3RhcnRlZCcsIHByb3BzKTtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBNZXRyaWMgZm9yIHRoZSBudW1iZXIgb2YgdGltZXMgdGhpcyBhY3Rpdml0eSBzdWNjZWVkc1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgc3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY1N1Y2NlZWRlZChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgcmV0dXJuIHRoaXMudGFza01ldHJpYyh0aGlzLnRhc2tQcm9wcy5tZXRyaWNQcmVmaXhQbHVyYWwsICdTdWNjZWVkZWQnLCBwcm9wcyk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogTWV0cmljIGZvciB0aGUgbnVtYmVyIG9mIHRpbWVzIHRoaXMgYWN0aXZpdHkgZmFpbHNcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHN1bSBvdmVyIDUgbWludXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBtZXRyaWNGYWlsZWQocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIHJldHVybiB0aGlzLnRhc2tNZXRyaWModGhpcy50YXNrUHJvcHMubWV0cmljUHJlZml4UGx1cmFsLCAnRmFpbGVkJywgcHJvcHMpO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIE1ldHJpYyBmb3IgdGhlIG51bWJlciBvZiB0aW1lcyB0aGUgaGVhcnRiZWF0IHRpbWVzIG91dCBmb3IgdGhpcyBhY3Rpdml0eVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgc3VtIG92ZXIgNSBtaW51dGVzXG4gICAgICovXG4gICAgcHVibGljIG1ldHJpY0hlYXJ0YmVhdFRpbWVkT3V0KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgICAgICByZXR1cm4gdGhpcy50YXNrTWV0cmljKHRoaXMudGFza1Byb3BzLm1ldHJpY1ByZWZpeFBsdXJhbCwgJ0hlYXJ0YmVhdFRpbWVkT3V0JywgcHJvcHMpO1xuICAgIH1cblxuICAgIHByb3RlY3RlZCB3aGVuQm91bmRUb0dyYXBoKGdyYXBoOiBTdGF0ZUdyYXBoKSB7XG4gICAgICAgIHN1cGVyLndoZW5Cb3VuZFRvR3JhcGgoZ3JhcGgpO1xuICAgICAgICBmb3IgKGNvbnN0IHBvbGljeVN0YXRlbWVudCBvZiB0aGlzLnRhc2tQcm9wcy5wb2xpY3lTdGF0ZW1lbnRzIHx8IFtdKSB7XG4gICAgICAgICAgICBncmFwaC5yZWdpc3RlclBvbGljeVN0YXRlbWVudChwb2xpY3lTdGF0ZW1lbnQpO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgcHJpdmF0ZSB0YXNrTWV0cmljKHByZWZpeDogc3RyaW5nIHwgdW5kZWZpbmVkLCBzdWZmaXg6IHN0cmluZywgcHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgICAgIGlmIChwcmVmaXggPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGlzIFRhc2sgUmVzb3VyY2UgZG9lcyBub3QgZXhwb3NlIG1ldHJpY3MnKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdGhpcy5tZXRyaWMocHJlZml4ICsgc3VmZml4LCBwcm9wcyk7XG4gICAgfVxufVxuIl19