"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const types_1 = require("../types");
/**
 * Base class for all other state classes
 */
class State extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.branches = [];
        this.retries = [];
        this.catches = [];
        this.choices = [];
        this.prefixes = [];
        /**
         * States with references to this state.
         *
         * Used for finding complete connected graph that a state is part of.
         */
        this.incomingStates = [];
        this.startState = this;
        this.comment = props.comment;
        this.inputPath = props.inputPath;
        this.parameters = props.parameters;
        this.outputPath = props.outputPath;
        this.resultPath = props.resultPath;
    }
    /**
     * Add a prefix to the stateId of all States found in a construct tree
     */
    static prefixStates(root, prefix) {
        const queue = [root];
        while (queue.length > 0) {
            const el = queue.splice(0, 1)[0];
            if (isPrefixable(el)) {
                el.addPrefix(prefix);
            }
            queue.push(...el.node.children);
        }
    }
    /**
     * Find the set of end states states reachable through transitions from the given start state
     */
    static findReachableEndStates(start, options = {}) {
        const visited = new Set();
        const ret = new Set();
        const queue = [start];
        while (queue.length > 0) {
            const state = queue.splice(0, 1)[0];
            if (visited.has(state)) {
                continue;
            }
            visited.add(state);
            const outgoing = state.outgoingTransitions(options);
            if (outgoing.length > 0) {
                // We can continue
                queue.push(...outgoing);
            }
            else {
                // Terminal state
                ret.add(state);
            }
        }
        return Array.from(ret);
    }
    /**
     * Return only the states that allow chaining from an array of states
     */
    static filterNextables(states) {
        return states.filter(isNextable);
    }
    get id() {
        return this.node.id;
    }
    /**
     * Tokenized string that evaluates to the state's ID
     */
    get stateId() {
        return this.prefixes.concat(this.id).join('');
    }
    /**
     * Add a prefix to the stateId of this state
     */
    addPrefix(x) {
        if (x !== '') {
            this.prefixes.splice(0, 0, x);
        }
    }
    /**
     * Register this state as part of the given graph
     *
     * Don't call this. It will be called automatically when you work
     * with states normally.
     */
    bindToGraph(graph) {
        if (this.containingGraph === graph) {
            return;
        }
        if (this.containingGraph) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Trying to use state '${this.stateId}' in ${graph}, but is already in ${this.containingGraph}. Every state can only be used in one graph.`);
        }
        this.containingGraph = graph;
        this.whenBoundToGraph(graph);
        for (const incoming of this.incomingStates) {
            incoming.bindToGraph(graph);
        }
        for (const outgoing of this.outgoingTransitions({ includeErrorHandlers: true })) {
            outgoing.bindToGraph(graph);
        }
        for (const branch of this.branches) {
            branch.registerSuperGraph(this.containingGraph);
        }
        if (!!this.iteration) {
            this.iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a retrier to the retry list of this state
     * @internal
     */
    _addRetry(props = {}) {
        this.retries.push({
            ...props,
            errors: props.errors ? props.errors : [types_1.Errors.ALL],
        });
    }
    /**
     * Add an error handler to the catch list of this state
     * @internal
     */
    _addCatch(handler, props = {}) {
        this.catches.push({
            next: handler,
            props: {
                errors: props.errors ? props.errors : [types_1.Errors.ALL],
                resultPath: props.resultPath
            }
        });
        handler.addIncoming(this);
        if (this.containingGraph) {
            handler.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default transition of this state
     */
    makeNext(next) {
        // Can't be called 'setNext' because of JSII
        if (this._next) {
            throw new Error(`State '${this.id}' already has a next state`);
        }
        this._next = next;
        next.addIncoming(this);
        if (this.containingGraph) {
            next.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a choice branch to this state
     */
    addChoice(condition, next) {
        this.choices.push({ condition, next });
        next.startState.addIncoming(this);
        if (this.containingGraph) {
            next.startState.bindToGraph(this.containingGraph);
        }
    }
    /**
     * Add a paralle branch to this state
     */
    addBranch(branch) {
        this.branches.push(branch);
        if (this.containingGraph) {
            branch.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Add a map iterator to this state
     */
    addIterator(iteration) {
        this.iteration = iteration;
        if (this.containingGraph) {
            iteration.registerSuperGraph(this.containingGraph);
        }
    }
    /**
     * Make the indicated state the default choice transition of this state
     */
    makeDefault(def) {
        // Can't be called 'setDefault' because of JSII
        if (this.defaultChoice) {
            throw new Error(`Choice '${this.id}' already has a default next state`);
        }
        this.defaultChoice = def;
    }
    /**
     * Render the default next state in ASL JSON format
     */
    renderNextEnd() {
        if (this._next) {
            return { Next: this._next.stateId };
        }
        else {
            return { End: true };
        }
    }
    /**
     * Render the choices in ASL JSON format
     */
    renderChoices() {
        return {
            Choices: renderList(this.choices, renderChoice),
            Default: this.defaultChoice ? this.defaultChoice.stateId : undefined,
        };
    }
    /**
     * Render InputPath/Parameters/OutputPath in ASL JSON format
     */
    renderInputOutput() {
        return {
            InputPath: renderJsonPath(this.inputPath),
            Parameters: this.parameters,
            OutputPath: renderJsonPath(this.outputPath),
        };
    }
    /**
     * Render parallel branches in ASL JSON format
     */
    renderBranches() {
        return {
            Branches: this.branches.map(b => b.toGraphJson())
        };
    }
    /**
     * Render map iterator in ASL JSON format
     */
    renderIterator() {
        if (!this.iteration) {
            throw new Error(`Iterator must not be undefined !`);
        }
        return {
            Iterator: this.iteration.toGraphJson()
        };
    }
    /**
     * Render error recovery options in ASL JSON format
     */
    renderRetryCatch() {
        return {
            Retry: renderList(this.retries, renderRetry),
            Catch: renderList(this.catches, renderCatch),
        };
    }
    /**
     * Called whenever this state is bound to a graph
     *
     * Can be overridden by subclasses.
     */
    whenBoundToGraph(graph) {
        graph.registerState(this);
    }
    /**
     * Add a state to the incoming list
     */
    addIncoming(source) {
        this.incomingStates.push(source);
    }
    /**
     * Return all states this state can transition to
     */
    outgoingTransitions(options) {
        const ret = new Array();
        if (this._next) {
            ret.push(this._next);
        }
        if (this.defaultChoice) {
            ret.push(this.defaultChoice);
        }
        for (const c of this.choices) {
            ret.push(c.next);
        }
        if (options.includeErrorHandlers) {
            for (const c of this.catches) {
                ret.push(c.next);
            }
        }
        return ret;
    }
}
exports.State = State;
/**
 * Render a choice transition
 */
function renderChoice(c) {
    return {
        ...c.condition.renderCondition(),
        Next: c.next.stateId
    };
}
/**
 * Render a Retry object to ASL
 */
function renderRetry(retry) {
    return {
        ErrorEquals: retry.errors,
        IntervalSeconds: retry.interval && retry.interval.toSeconds(),
        MaxAttempts: retry.maxAttempts,
        BackoffRate: retry.backoffRate
    };
}
/**
 * Render a Catch object to ASL
 */
function renderCatch(c) {
    return {
        ErrorEquals: c.props.errors,
        ResultPath: renderJsonPath(c.props.resultPath),
        Next: c.next.stateId,
    };
}
/**
 * Render a list or return undefined for an empty list
 */
function renderList(xs, fn) {
    if (xs.length === 0) {
        return undefined;
    }
    return xs.map(fn);
}
exports.renderList = renderList;
/**
 * Render JSON path, respecting the special value DISCARD
 */
function renderJsonPath(jsonPath) {
    if (jsonPath === undefined) {
        return undefined;
    }
    if (jsonPath === types_1.DISCARD) {
        return null;
    }
    if (!jsonPath.startsWith('$')) {
        throw new Error(`Expected JSON path to start with '$', got: ${jsonPath}`);
    }
    return jsonPath;
}
exports.renderJsonPath = renderJsonPath;
/**
 * Whether an object is a Prefixable
 */
function isPrefixable(x) {
    return typeof (x) === 'object' && x.addPrefix;
}
/**
 * Whether an object is INextable
 */
function isNextable(x) {
    return typeof (x) === 'object' && x.next;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGF0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHFDQUFxQztBQUdyQyxvQ0FBMEY7QUFzRDFGOztHQUVHO0FBQ0gsTUFBc0IsS0FBTSxTQUFRLEdBQUcsQ0FBQyxTQUFTO0lBK0Y3QyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQWlCO1FBQzNELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUE3QkYsYUFBUSxHQUFpQixFQUFFLENBQUM7UUFTOUIsWUFBTyxHQUFpQixFQUFFLENBQUM7UUFDM0IsWUFBTyxHQUFzQixFQUFFLENBQUM7UUFDaEMsWUFBTyxHQUF1QixFQUFFLENBQUM7UUFDakMsYUFBUSxHQUFhLEVBQUUsQ0FBQztRQVN6Qzs7OztXQUlHO1FBQ2MsbUJBQWMsR0FBWSxFQUFFLENBQUM7UUFLMUMsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLENBQUM7UUFFdkIsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUNqQyxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7UUFDbkMsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1FBQ25DLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztJQUN2QyxDQUFDO0lBeEdEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxJQUFvQixFQUFFLE1BQWM7UUFDM0QsTUFBTSxLQUFLLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNyQixPQUFPLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3JCLE1BQU0sRUFBRSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBRSxDQUFDO1lBQ2xDLElBQUksWUFBWSxDQUFDLEVBQUUsQ0FBQyxFQUFFO2dCQUNsQixFQUFFLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2FBQ3hCO1lBQ0QsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDbkM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsc0JBQXNCLENBQUMsS0FBWSxFQUFFLFVBQTRCLEVBQUU7UUFDN0UsTUFBTSxPQUFPLEdBQUcsSUFBSSxHQUFHLEVBQVMsQ0FBQztRQUNqQyxNQUFNLEdBQUcsR0FBRyxJQUFJLEdBQUcsRUFBUyxDQUFDO1FBQzdCLE1BQU0sS0FBSyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdEIsT0FBTyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNyQixNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUUsQ0FBQztZQUNyQyxJQUFJLE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUU7Z0JBQUUsU0FBUzthQUFFO1lBQ3JDLE9BQU8sQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUM7WUFFbkIsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBRXBELElBQUksUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQ3JCLGtCQUFrQjtnQkFDbEIsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLFFBQVEsQ0FBQyxDQUFDO2FBQzNCO2lCQUFNO2dCQUNILGlCQUFpQjtnQkFDakIsR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQzthQUNsQjtTQUNKO1FBQ0QsT0FBTyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxlQUFlLENBQUMsTUFBZTtRQUN6QyxPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFRLENBQUM7SUFDNUMsQ0FBQztJQThERCxJQUFXLEVBQUU7UUFDVCxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO0lBQ3hCLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsT0FBTztRQUNkLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxTQUFTLENBQUMsQ0FBUztRQUN0QixJQUFJLENBQUMsS0FBSyxFQUFFLEVBQUU7WUFDVixJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1NBQ2pDO0lBQ0wsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksV0FBVyxDQUFDLEtBQWlCO1FBQ2hDLElBQUksSUFBSSxDQUFDLGVBQWUsS0FBSyxLQUFLLEVBQUU7WUFBRSxPQUFPO1NBQUU7UUFFL0MsSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3RCLDJDQUEyQztZQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLHdCQUF3QixJQUFJLENBQUMsT0FBTyxRQUFRLEtBQUssdUJBQXVCLElBQUksQ0FBQyxlQUFlLDhDQUE4QyxDQUFDLENBQUM7U0FDL0o7UUFFRCxJQUFJLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQztRQUM3QixJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFN0IsS0FBSyxNQUFNLFFBQVEsSUFBSSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ3hDLFFBQVEsQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDL0I7UUFDRCxLQUFLLE1BQU0sUUFBUSxJQUFJLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLG9CQUFvQixFQUFFLElBQUksRUFBRSxDQUFDLEVBQUU7WUFDN0UsUUFBUSxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUMvQjtRQUNELEtBQUssTUFBTSxNQUFNLElBQUksSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUNoQyxNQUFNLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ25EO1FBQ0QsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNsQixJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUMzRDtJQUNMLENBQUM7SUFPRDs7O09BR0c7SUFDTyxTQUFTLENBQUMsUUFBb0IsRUFBRTtRQUN0QyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQztZQUNkLEdBQUcsS0FBSztZQUNSLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQU0sQ0FBQyxHQUFHLENBQUM7U0FDckQsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUVEOzs7T0FHRztJQUNPLFNBQVMsQ0FBQyxPQUFjLEVBQUUsUUFBb0IsRUFBRTtRQUN0RCxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQztZQUNkLElBQUksRUFBRSxPQUFPO1lBQ2IsS0FBSyxFQUFFO2dCQUNILE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQU0sQ0FBQyxHQUFHLENBQUM7Z0JBQ2xELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTthQUMvQjtTQUNKLENBQUMsQ0FBQztRQUNILE9BQU8sQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDMUIsSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3RCLE9BQU8sQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQzdDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ08sUUFBUSxDQUFDLElBQVc7UUFDMUIsNENBQTRDO1FBQzVDLElBQUksSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLENBQUMsRUFBRSw0QkFBNEIsQ0FBQyxDQUFDO1NBQ2xFO1FBQ0QsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUM7UUFDbEIsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2QixJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDdEIsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUM7U0FDMUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDTyxTQUFTLENBQUMsU0FBb0IsRUFBRSxJQUFXO1FBQ2pELElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbEMsSUFBSSxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3RCLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUNyRDtJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNPLFNBQVMsQ0FBQyxNQUFrQjtRQUNsQyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUMzQixJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDdEIsTUFBTSxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUNuRDtJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNPLFdBQVcsQ0FBQyxTQUFxQjtRQUN2QyxJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztRQUMzQixJQUFJLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDdEIsU0FBUyxDQUFDLGtCQUFrQixDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUN0RDtJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNPLFdBQVcsQ0FBQyxHQUFVO1FBQzVCLCtDQUErQztRQUMvQyxJQUFJLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDcEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxXQUFXLElBQUksQ0FBQyxFQUFFLG9DQUFvQyxDQUFDLENBQUM7U0FDM0U7UUFDRCxJQUFJLENBQUMsYUFBYSxHQUFHLEdBQUcsQ0FBQztJQUM3QixDQUFDO0lBRUQ7O09BRUc7SUFDTyxhQUFhO1FBQ25CLElBQUksSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNaLE9BQU8sRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQztTQUN2QzthQUFNO1lBQ0gsT0FBTyxFQUFFLEdBQUcsRUFBRSxJQUFJLEVBQUUsQ0FBQztTQUN4QjtJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNPLGFBQWE7UUFDbkIsT0FBTztZQUNILE9BQU8sRUFBRSxVQUFVLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxZQUFZLENBQUM7WUFDL0MsT0FBTyxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQ3ZFLENBQUM7SUFDTixDQUFDO0lBRUQ7O09BRUc7SUFDTyxpQkFBaUI7UUFDdkIsT0FBTztZQUNILFNBQVMsRUFBRSxjQUFjLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUN6QyxVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDM0IsVUFBVSxFQUFFLGNBQWMsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDO1NBQzlDLENBQUM7SUFDTixDQUFDO0lBRUQ7O09BRUc7SUFDTyxjQUFjO1FBQ3BCLE9BQU87WUFDSCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7U0FDcEQsQ0FBQztJQUNOLENBQUM7SUFFRDs7T0FFRztJQUNPLGNBQWM7UUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDakIsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1NBQ3ZEO1FBQ0QsT0FBTztZQUNILFFBQVEsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRTtTQUN6QyxDQUFDO0lBQ04sQ0FBQztJQUVEOztPQUVHO0lBQ08sZ0JBQWdCO1FBQ3RCLE9BQU87WUFDSCxLQUFLLEVBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDO1lBQzVDLEtBQUssRUFBRSxVQUFVLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxXQUFXLENBQUM7U0FDL0MsQ0FBQztJQUNOLENBQUM7SUFFRDs7OztPQUlHO0lBQ08sZ0JBQWdCLENBQUMsS0FBaUI7UUFDeEMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRUQ7O09BRUc7SUFDSyxXQUFXLENBQUMsTUFBYTtRQUM3QixJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7O09BRUc7SUFDSyxtQkFBbUIsQ0FBQyxPQUF5QjtRQUNqRCxNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssRUFBUyxDQUFDO1FBQy9CLElBQUksSUFBSSxDQUFDLEtBQUssRUFBRTtZQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQUU7UUFDekMsSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7U0FBRTtRQUN6RCxLQUFLLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDMUIsR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDcEI7UUFDRCxJQUFJLE9BQU8sQ0FBQyxvQkFBb0IsRUFBRTtZQUM5QixLQUFLLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUU7Z0JBQzFCLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQ3BCO1NBQ0o7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7Q0FDSjtBQXpWRCxzQkF5VkM7QUE2QkQ7O0dBRUc7QUFDSCxTQUFTLFlBQVksQ0FBQyxDQUFtQjtJQUNyQyxPQUFPO1FBQ0gsR0FBRyxDQUFDLENBQUMsU0FBUyxDQUFDLGVBQWUsRUFBRTtRQUNoQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPO0tBQ3ZCLENBQUM7QUFDTixDQUFDO0FBaUJEOztHQUVHO0FBQ0gsU0FBUyxXQUFXLENBQUMsS0FBaUI7SUFDbEMsT0FBTztRQUNILFdBQVcsRUFBRSxLQUFLLENBQUMsTUFBTTtRQUN6QixlQUFlLEVBQUUsS0FBSyxDQUFDLFFBQVEsSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRTtRQUM3RCxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7UUFDOUIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO0tBQ2pDLENBQUM7QUFDTixDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLFdBQVcsQ0FBQyxDQUFrQjtJQUNuQyxPQUFPO1FBQ0gsV0FBVyxFQUFFLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTTtRQUMzQixVQUFVLEVBQUUsY0FBYyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDO1FBQzlDLElBQUksRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLE9BQU87S0FDdkIsQ0FBQztBQUNOLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLFVBQVUsQ0FBSSxFQUFPLEVBQUUsRUFBaUI7SUFDcEQsSUFBSSxFQUFFLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtRQUFFLE9BQU8sU0FBUyxDQUFDO0tBQUU7SUFDMUMsT0FBTyxFQUFFLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0FBQ3RCLENBQUM7QUFIRCxnQ0FHQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsY0FBYyxDQUFDLFFBQWlCO0lBQzVDLElBQUksUUFBUSxLQUFLLFNBQVMsRUFBRTtRQUFFLE9BQU8sU0FBUyxDQUFDO0tBQUU7SUFDakQsSUFBSSxRQUFRLEtBQUssZUFBTyxFQUFFO1FBQUUsT0FBTyxJQUFJLENBQUM7S0FBRTtJQUUxQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRTtRQUMzQixNQUFNLElBQUksS0FBSyxDQUFDLDhDQUE4QyxRQUFRLEVBQUUsQ0FBQyxDQUFDO0tBQzdFO0lBQ0QsT0FBTyxRQUFRLENBQUM7QUFDcEIsQ0FBQztBQVJELHdDQVFDO0FBU0Q7O0dBRUc7QUFDSCxTQUFTLFlBQVksQ0FBQyxDQUFNO0lBQ3hCLE9BQU8sT0FBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLFFBQVEsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO0FBQ2pELENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsVUFBVSxDQUFDLENBQU07SUFDdEIsT0FBTyxPQUFNLENBQUMsQ0FBQyxDQUFDLEtBQUssUUFBUSxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUM7QUFDNUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbmRpdGlvbiB9IGZyb20gJy4uL2NvbmRpdGlvbic7XG5pbXBvcnQgeyBTdGF0ZUdyYXBoIH0gZnJvbSAnLi4vc3RhdGUtZ3JhcGgnO1xuaW1wb3J0IHsgQ2F0Y2hQcm9wcywgRElTQ0FSRCwgRXJyb3JzLCBJQ2hhaW5hYmxlLCBJTmV4dGFibGUsIFJldHJ5UHJvcHMgfSBmcm9tICcuLi90eXBlcyc7XG5cbi8qKlxuICogUHJvcGVydGllcyBzaGFyZWQgYnkgYWxsIHN0YXRlc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFN0YXRlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIEEgY29tbWVudCBkZXNjcmliaW5nIHRoaXMgc3RhdGVcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGNvbW1lbnRcbiAgICAgKi9cbiAgICByZWFkb25seSBjb21tZW50Pzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogSlNPTlBhdGggZXhwcmVzc2lvbiB0byBzZWxlY3QgcGFydCBvZiB0aGUgc3RhdGUgdG8gYmUgdGhlIGlucHV0IHRvIHRoaXMgc3RhdGUuXG4gICAgICpcbiAgICAgKiBNYXkgYWxzbyBiZSB0aGUgc3BlY2lhbCB2YWx1ZSBESVNDQVJELCB3aGljaCB3aWxsIGNhdXNlIHRoZSBlZmZlY3RpdmVcbiAgICAgKiBpbnB1dCB0byBiZSB0aGUgZW1wdHkgb2JqZWN0IHt9LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgJFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGlucHV0UGF0aD86IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIFBhcmFtZXRlcnMgcGFzcyBhIGNvbGxlY3Rpb24gb2Yga2V5LXZhbHVlIHBhaXJzLCBlaXRoZXIgc3RhdGljIHZhbHVlcyBvciBKU09OUGF0aCBleHByZXNzaW9ucyB0aGF0IHNlbGVjdCBmcm9tIHRoZSBpbnB1dC5cbiAgICAgKlxuICAgICAqIEBzZWVcbiAgICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2lucHV0LW91dHB1dC1pbnB1dHBhdGgtcGFyYW1zLmh0bWwjaW5wdXQtb3V0cHV0LXBhcmFtZXRlcnNcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIHBhcmFtZXRlcnNcbiAgICAgKi9cbiAgICByZWFkb25seSBwYXJhbWV0ZXJzPzogeyBbbmFtZTogc3RyaW5nXTogYW55IH07XG5cbiAgICAvKipcbiAgICAgKiBKU09OUGF0aCBleHByZXNzaW9uIHRvIHNlbGVjdCBwYXJ0IG9mIHRoZSBzdGF0ZSB0byBiZSB0aGUgb3V0cHV0IHRvIHRoaXMgc3RhdGUuXG4gICAgICpcbiAgICAgKiBNYXkgYWxzbyBiZSB0aGUgc3BlY2lhbCB2YWx1ZSBESVNDQVJELCB3aGljaCB3aWxsIGNhdXNlIHRoZSBlZmZlY3RpdmVcbiAgICAgKiBvdXRwdXQgdG8gYmUgdGhlIGVtcHR5IG9iamVjdCB7fS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0ICRcbiAgICAgKi9cbiAgICByZWFkb25seSBvdXRwdXRQYXRoPzogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogSlNPTlBhdGggZXhwcmVzc2lvbiB0byBpbmRpY2F0ZSB3aGVyZSB0byBpbmplY3QgdGhlIHN0YXRlJ3Mgb3V0cHV0XG4gICAgICpcbiAgICAgKiBNYXkgYWxzbyBiZSB0aGUgc3BlY2lhbCB2YWx1ZSBESVNDQVJELCB3aGljaCB3aWxsIGNhdXNlIHRoZSBzdGF0ZSdzXG4gICAgICogaW5wdXQgdG8gYmVjb21lIGl0cyBvdXRwdXQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAkXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzdWx0UGF0aD86IHN0cmluZztcbn1cblxuLyoqXG4gKiBCYXNlIGNsYXNzIGZvciBhbGwgb3RoZXIgc3RhdGUgY2xhc3Nlc1xuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgU3RhdGUgZXh0ZW5kcyBjZGsuQ29uc3RydWN0IGltcGxlbWVudHMgSUNoYWluYWJsZSB7XG4gICAgLyoqXG4gICAgICogQWRkIGEgcHJlZml4IHRvIHRoZSBzdGF0ZUlkIG9mIGFsbCBTdGF0ZXMgZm91bmQgaW4gYSBjb25zdHJ1Y3QgdHJlZVxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgcHJlZml4U3RhdGVzKHJvb3Q6IGNkay5JQ29uc3RydWN0LCBwcmVmaXg6IHN0cmluZykge1xuICAgICAgICBjb25zdCBxdWV1ZSA9IFtyb290XTtcbiAgICAgICAgd2hpbGUgKHF1ZXVlLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgIGNvbnN0IGVsID0gcXVldWUuc3BsaWNlKDAsIDEpWzBdITtcbiAgICAgICAgICAgIGlmIChpc1ByZWZpeGFibGUoZWwpKSB7XG4gICAgICAgICAgICAgICAgZWwuYWRkUHJlZml4KHByZWZpeCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBxdWV1ZS5wdXNoKC4uLmVsLm5vZGUuY2hpbGRyZW4pO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogRmluZCB0aGUgc2V0IG9mIGVuZCBzdGF0ZXMgc3RhdGVzIHJlYWNoYWJsZSB0aHJvdWdoIHRyYW5zaXRpb25zIGZyb20gdGhlIGdpdmVuIHN0YXJ0IHN0YXRlXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmaW5kUmVhY2hhYmxlRW5kU3RhdGVzKHN0YXJ0OiBTdGF0ZSwgb3B0aW9uczogRmluZFN0YXRlT3B0aW9ucyA9IHt9KSB7XG4gICAgICAgIGNvbnN0IHZpc2l0ZWQgPSBuZXcgU2V0PFN0YXRlPigpO1xuICAgICAgICBjb25zdCByZXQgPSBuZXcgU2V0PFN0YXRlPigpO1xuICAgICAgICBjb25zdCBxdWV1ZSA9IFtzdGFydF07XG4gICAgICAgIHdoaWxlIChxdWV1ZS5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICBjb25zdCBzdGF0ZSA9IHF1ZXVlLnNwbGljZSgwLCAxKVswXSE7XG4gICAgICAgICAgICBpZiAodmlzaXRlZC5oYXMoc3RhdGUpKSB7IGNvbnRpbnVlOyB9XG4gICAgICAgICAgICB2aXNpdGVkLmFkZChzdGF0ZSk7XG5cbiAgICAgICAgICAgIGNvbnN0IG91dGdvaW5nID0gc3RhdGUub3V0Z29pbmdUcmFuc2l0aW9ucyhvcHRpb25zKTtcblxuICAgICAgICAgICAgaWYgKG91dGdvaW5nLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgICAgICAvLyBXZSBjYW4gY29udGludWVcbiAgICAgICAgICAgICAgICBxdWV1ZS5wdXNoKC4uLm91dGdvaW5nKTtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgICAgLy8gVGVybWluYWwgc3RhdGVcbiAgICAgICAgICAgICAgICByZXQuYWRkKHN0YXRlKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gQXJyYXkuZnJvbShyZXQpO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIFJldHVybiBvbmx5IHRoZSBzdGF0ZXMgdGhhdCBhbGxvdyBjaGFpbmluZyBmcm9tIGFuIGFycmF5IG9mIHN0YXRlc1xuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZmlsdGVyTmV4dGFibGVzKHN0YXRlczogU3RhdGVbXSk6IElOZXh0YWJsZVtdIHtcbiAgICAgICAgcmV0dXJuIHN0YXRlcy5maWx0ZXIoaXNOZXh0YWJsZSkgYXMgYW55O1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEZpcnN0IHN0YXRlIG9mIHRoaXMgQ2hhaW5hYmxlXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IHN0YXJ0U3RhdGU6IFN0YXRlO1xuXG4gICAgLyoqXG4gICAgICogQ29udGludWFibGUgc3RhdGVzIG9mIHRoaXMgQ2hhaW5hYmxlXG4gICAgICovXG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGVuZFN0YXRlczogSU5leHRhYmxlW107XG5cbiAgICAvLyBUaGlzIGNsYXNzIGhhcyBhIHN1cGVyc2V0IG9mIG1vc3Qgb2YgdGhlIGZlYXR1cmVzIG9mIHRoZSBvdGhlciBzdGF0ZXMsXG4gICAgLy8gYW5kIHRoZSBzdWJjbGFzc2VzIGRlY2lkZSB3aGljaCBwYXJ0IG9mIHRoZSBmZWF0dXJlcyB0byBleHBvc2UuIE1vc3RcbiAgICAvLyBmZWF0dXJlcyBhcmUgc2hhcmVkIGJ5IGEgY291cGxlIG9mIHN0YXRlcywgYW5kIGl0IGJlY29tZXMgY3VtYmVyc29tZSB0b1xuICAgIC8vIHNsaWNlIGl0IG91dCBhY3Jvc3MgYWxsIHN0YXRlcy4gVGhpcyBpcyBub3QgZ3JlYXQgZGVzaWduLCBidXQgaXQgaXNcbiAgICAvLyBwcmFnbWF0aWMhXG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IGNvbW1lbnQ/OiBzdHJpbmc7XG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IGlucHV0UGF0aD86IHN0cmluZztcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgcGFyYW1ldGVycz86IG9iamVjdDtcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgb3V0cHV0UGF0aD86IHN0cmluZztcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgcmVzdWx0UGF0aD86IHN0cmluZztcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgYnJhbmNoZXM6IFN0YXRlR3JhcGhbXSA9IFtdO1xuICAgIHByb3RlY3RlZCBpdGVyYXRpb24/OiBTdGF0ZUdyYXBoO1xuICAgIHByb3RlY3RlZCBkZWZhdWx0Q2hvaWNlPzogU3RhdGU7XG5cbiAgICAvKipcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwcm90ZWN0ZWQgX25leHQ/OiBTdGF0ZTtcblxuICAgIHByaXZhdGUgcmVhZG9ubHkgcmV0cmllczogUmV0cnlQcm9wc1tdID0gW107XG4gICAgcHJpdmF0ZSByZWFkb25seSBjYXRjaGVzOiBDYXRjaFRyYW5zaXRpb25bXSA9IFtdO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgY2hvaWNlczogQ2hvaWNlVHJhbnNpdGlvbltdID0gW107XG4gICAgcHJpdmF0ZSByZWFkb25seSBwcmVmaXhlczogc3RyaW5nW10gPSBbXTtcblxuICAgIC8qKlxuICAgICAqIFRoZSBncmFwaCB0aGF0IHRoaXMgc3RhdGUgaXMgcGFydCBvZi5cbiAgICAgKlxuICAgICAqIFVzZWQgZm9yIGd1YXJhbnRlZWluZyBjb25zaXN0ZW5jeSBiZXR3ZWVuIGdyYXBocyBhbmQgZ3JhcGggY29tcG9uZW50cy5cbiAgICAgKi9cbiAgICBwcml2YXRlIGNvbnRhaW5pbmdHcmFwaD86IFN0YXRlR3JhcGg7XG5cbiAgICAvKipcbiAgICAgKiBTdGF0ZXMgd2l0aCByZWZlcmVuY2VzIHRvIHRoaXMgc3RhdGUuXG4gICAgICpcbiAgICAgKiBVc2VkIGZvciBmaW5kaW5nIGNvbXBsZXRlIGNvbm5lY3RlZCBncmFwaCB0aGF0IGEgc3RhdGUgaXMgcGFydCBvZi5cbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IGluY29taW5nU3RhdGVzOiBTdGF0ZVtdID0gW107XG5cbiAgICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFN0YXRlUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgICAgICB0aGlzLnN0YXJ0U3RhdGUgPSB0aGlzO1xuXG4gICAgICAgIHRoaXMuY29tbWVudCA9IHByb3BzLmNvbW1lbnQ7XG4gICAgICAgIHRoaXMuaW5wdXRQYXRoID0gcHJvcHMuaW5wdXRQYXRoO1xuICAgICAgICB0aGlzLnBhcmFtZXRlcnMgPSBwcm9wcy5wYXJhbWV0ZXJzO1xuICAgICAgICB0aGlzLm91dHB1dFBhdGggPSBwcm9wcy5vdXRwdXRQYXRoO1xuICAgICAgICB0aGlzLnJlc3VsdFBhdGggPSBwcm9wcy5yZXN1bHRQYXRoO1xuICAgIH1cblxuICAgIHB1YmxpYyBnZXQgaWQoKSB7XG4gICAgICAgIHJldHVybiB0aGlzLm5vZGUuaWQ7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogVG9rZW5pemVkIHN0cmluZyB0aGF0IGV2YWx1YXRlcyB0byB0aGUgc3RhdGUncyBJRFxuICAgICAqL1xuICAgIHB1YmxpYyBnZXQgc3RhdGVJZCgpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gdGhpcy5wcmVmaXhlcy5jb25jYXQodGhpcy5pZCkuam9pbignJyk7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogQWRkIGEgcHJlZml4IHRvIHRoZSBzdGF0ZUlkIG9mIHRoaXMgc3RhdGVcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkUHJlZml4KHg6IHN0cmluZykge1xuICAgICAgICBpZiAoeCAhPT0gJycpIHtcbiAgICAgICAgICAgIHRoaXMucHJlZml4ZXMuc3BsaWNlKDAsIDAsIHgpO1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogUmVnaXN0ZXIgdGhpcyBzdGF0ZSBhcyBwYXJ0IG9mIHRoZSBnaXZlbiBncmFwaFxuICAgICAqXG4gICAgICogRG9uJ3QgY2FsbCB0aGlzLiBJdCB3aWxsIGJlIGNhbGxlZCBhdXRvbWF0aWNhbGx5IHdoZW4geW91IHdvcmtcbiAgICAgKiB3aXRoIHN0YXRlcyBub3JtYWxseS5cbiAgICAgKi9cbiAgICBwdWJsaWMgYmluZFRvR3JhcGgoZ3JhcGg6IFN0YXRlR3JhcGgpIHtcbiAgICAgICAgaWYgKHRoaXMuY29udGFpbmluZ0dyYXBoID09PSBncmFwaCkgeyByZXR1cm47IH1cblxuICAgICAgICBpZiAodGhpcy5jb250YWluaW5nR3JhcGgpIHtcbiAgICAgICAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVHJ5aW5nIHRvIHVzZSBzdGF0ZSAnJHt0aGlzLnN0YXRlSWR9JyBpbiAke2dyYXBofSwgYnV0IGlzIGFscmVhZHkgaW4gJHt0aGlzLmNvbnRhaW5pbmdHcmFwaH0uIEV2ZXJ5IHN0YXRlIGNhbiBvbmx5IGJlIHVzZWQgaW4gb25lIGdyYXBoLmApO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb250YWluaW5nR3JhcGggPSBncmFwaDtcbiAgICAgICAgdGhpcy53aGVuQm91bmRUb0dyYXBoKGdyYXBoKTtcblxuICAgICAgICBmb3IgKGNvbnN0IGluY29taW5nIG9mIHRoaXMuaW5jb21pbmdTdGF0ZXMpIHtcbiAgICAgICAgICAgIGluY29taW5nLmJpbmRUb0dyYXBoKGdyYXBoKTtcbiAgICAgICAgfVxuICAgICAgICBmb3IgKGNvbnN0IG91dGdvaW5nIG9mIHRoaXMub3V0Z29pbmdUcmFuc2l0aW9ucyh7IGluY2x1ZGVFcnJvckhhbmRsZXJzOiB0cnVlIH0pKSB7XG4gICAgICAgICAgICBvdXRnb2luZy5iaW5kVG9HcmFwaChncmFwaCk7XG4gICAgICAgIH1cbiAgICAgICAgZm9yIChjb25zdCBicmFuY2ggb2YgdGhpcy5icmFuY2hlcykge1xuICAgICAgICAgICAgYnJhbmNoLnJlZ2lzdGVyU3VwZXJHcmFwaCh0aGlzLmNvbnRhaW5pbmdHcmFwaCk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKCEhdGhpcy5pdGVyYXRpb24pIHtcbiAgICAgICAgICAgIHRoaXMuaXRlcmF0aW9uLnJlZ2lzdGVyU3VwZXJHcmFwaCh0aGlzLmNvbnRhaW5pbmdHcmFwaCk7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBSZW5kZXIgdGhlIHN0YXRlIGFzIEpTT05cbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgdG9TdGF0ZUpzb24oKTogb2JqZWN0O1xuXG4gICAgLyoqXG4gICAgICogQWRkIGEgcmV0cmllciB0byB0aGUgcmV0cnkgbGlzdCBvZiB0aGlzIHN0YXRlXG4gICAgICogQGludGVybmFsXG4gICAgICovXG4gICAgcHJvdGVjdGVkIF9hZGRSZXRyeShwcm9wczogUmV0cnlQcm9wcyA9IHt9KSB7XG4gICAgICAgIHRoaXMucmV0cmllcy5wdXNoKHtcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgZXJyb3JzOiBwcm9wcy5lcnJvcnMgPyBwcm9wcy5lcnJvcnMgOiBbRXJyb3JzLkFMTF0sXG4gICAgICAgIH0pO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEFkZCBhbiBlcnJvciBoYW5kbGVyIHRvIHRoZSBjYXRjaCBsaXN0IG9mIHRoaXMgc3RhdGVcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwcm90ZWN0ZWQgX2FkZENhdGNoKGhhbmRsZXI6IFN0YXRlLCBwcm9wczogQ2F0Y2hQcm9wcyA9IHt9KSB7XG4gICAgICAgIHRoaXMuY2F0Y2hlcy5wdXNoKHtcbiAgICAgICAgICAgIG5leHQ6IGhhbmRsZXIsXG4gICAgICAgICAgICBwcm9wczoge1xuICAgICAgICAgICAgICAgIGVycm9yczogcHJvcHMuZXJyb3JzID8gcHJvcHMuZXJyb3JzIDogW0Vycm9ycy5BTExdLFxuICAgICAgICAgICAgICAgIHJlc3VsdFBhdGg6IHByb3BzLnJlc3VsdFBhdGhcbiAgICAgICAgICAgIH1cbiAgICAgICAgfSk7XG4gICAgICAgIGhhbmRsZXIuYWRkSW5jb21pbmcodGhpcyk7XG4gICAgICAgIGlmICh0aGlzLmNvbnRhaW5pbmdHcmFwaCkge1xuICAgICAgICAgICAgaGFuZGxlci5iaW5kVG9HcmFwaCh0aGlzLmNvbnRhaW5pbmdHcmFwaCk7XG4gICAgICAgIH1cbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBNYWtlIHRoZSBpbmRpY2F0ZWQgc3RhdGUgdGhlIGRlZmF1bHQgdHJhbnNpdGlvbiBvZiB0aGlzIHN0YXRlXG4gICAgICovXG4gICAgcHJvdGVjdGVkIG1ha2VOZXh0KG5leHQ6IFN0YXRlKSB7XG4gICAgICAgIC8vIENhbid0IGJlIGNhbGxlZCAnc2V0TmV4dCcgYmVjYXVzZSBvZiBKU0lJXG4gICAgICAgIGlmICh0aGlzLl9uZXh0KSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFN0YXRlICcke3RoaXMuaWR9JyBhbHJlYWR5IGhhcyBhIG5leHQgc3RhdGVgKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLl9uZXh0ID0gbmV4dDtcbiAgICAgICAgbmV4dC5hZGRJbmNvbWluZyh0aGlzKTtcbiAgICAgICAgaWYgKHRoaXMuY29udGFpbmluZ0dyYXBoKSB7XG4gICAgICAgICAgICBuZXh0LmJpbmRUb0dyYXBoKHRoaXMuY29udGFpbmluZ0dyYXBoKTtcbiAgICAgICAgfVxuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEFkZCBhIGNob2ljZSBicmFuY2ggdG8gdGhpcyBzdGF0ZVxuICAgICAqL1xuICAgIHByb3RlY3RlZCBhZGRDaG9pY2UoY29uZGl0aW9uOiBDb25kaXRpb24sIG5leHQ6IFN0YXRlKSB7XG4gICAgICAgIHRoaXMuY2hvaWNlcy5wdXNoKHsgY29uZGl0aW9uLCBuZXh0IH0pO1xuICAgICAgICBuZXh0LnN0YXJ0U3RhdGUuYWRkSW5jb21pbmcodGhpcyk7XG4gICAgICAgIGlmICh0aGlzLmNvbnRhaW5pbmdHcmFwaCkge1xuICAgICAgICAgICAgbmV4dC5zdGFydFN0YXRlLmJpbmRUb0dyYXBoKHRoaXMuY29udGFpbmluZ0dyYXBoKTtcbiAgICAgICAgfVxuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEFkZCBhIHBhcmFsbGUgYnJhbmNoIHRvIHRoaXMgc3RhdGVcbiAgICAgKi9cbiAgICBwcm90ZWN0ZWQgYWRkQnJhbmNoKGJyYW5jaDogU3RhdGVHcmFwaCkge1xuICAgICAgICB0aGlzLmJyYW5jaGVzLnB1c2goYnJhbmNoKTtcbiAgICAgICAgaWYgKHRoaXMuY29udGFpbmluZ0dyYXBoKSB7XG4gICAgICAgICAgICBicmFuY2gucmVnaXN0ZXJTdXBlckdyYXBoKHRoaXMuY29udGFpbmluZ0dyYXBoKTtcbiAgICAgICAgfVxuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEFkZCBhIG1hcCBpdGVyYXRvciB0byB0aGlzIHN0YXRlXG4gICAgICovXG4gICAgcHJvdGVjdGVkIGFkZEl0ZXJhdG9yKGl0ZXJhdGlvbjogU3RhdGVHcmFwaCkge1xuICAgICAgICB0aGlzLml0ZXJhdGlvbiA9IGl0ZXJhdGlvbjtcbiAgICAgICAgaWYgKHRoaXMuY29udGFpbmluZ0dyYXBoKSB7XG4gICAgICAgICAgICBpdGVyYXRpb24ucmVnaXN0ZXJTdXBlckdyYXBoKHRoaXMuY29udGFpbmluZ0dyYXBoKTtcbiAgICAgICAgfVxuICAgIH1cblxuICAgIC8qKlxuICAgICAqIE1ha2UgdGhlIGluZGljYXRlZCBzdGF0ZSB0aGUgZGVmYXVsdCBjaG9pY2UgdHJhbnNpdGlvbiBvZiB0aGlzIHN0YXRlXG4gICAgICovXG4gICAgcHJvdGVjdGVkIG1ha2VEZWZhdWx0KGRlZjogU3RhdGUpIHtcbiAgICAgICAgLy8gQ2FuJ3QgYmUgY2FsbGVkICdzZXREZWZhdWx0JyBiZWNhdXNlIG9mIEpTSUlcbiAgICAgICAgaWYgKHRoaXMuZGVmYXVsdENob2ljZSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBDaG9pY2UgJyR7dGhpcy5pZH0nIGFscmVhZHkgaGFzIGEgZGVmYXVsdCBuZXh0IHN0YXRlYCk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5kZWZhdWx0Q2hvaWNlID0gZGVmO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIFJlbmRlciB0aGUgZGVmYXVsdCBuZXh0IHN0YXRlIGluIEFTTCBKU09OIGZvcm1hdFxuICAgICAqL1xuICAgIHByb3RlY3RlZCByZW5kZXJOZXh0RW5kKCk6IGFueSB7XG4gICAgICAgIGlmICh0aGlzLl9uZXh0KSB7XG4gICAgICAgICAgICByZXR1cm4geyBOZXh0OiB0aGlzLl9uZXh0LnN0YXRlSWQgfTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHJldHVybiB7IEVuZDogdHJ1ZSB9O1xuICAgICAgICB9XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogUmVuZGVyIHRoZSBjaG9pY2VzIGluIEFTTCBKU09OIGZvcm1hdFxuICAgICAqL1xuICAgIHByb3RlY3RlZCByZW5kZXJDaG9pY2VzKCk6IGFueSB7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBDaG9pY2VzOiByZW5kZXJMaXN0KHRoaXMuY2hvaWNlcywgcmVuZGVyQ2hvaWNlKSxcbiAgICAgICAgICAgIERlZmF1bHQ6IHRoaXMuZGVmYXVsdENob2ljZSA/IHRoaXMuZGVmYXVsdENob2ljZS5zdGF0ZUlkIDogdW5kZWZpbmVkLFxuICAgICAgICB9O1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIFJlbmRlciBJbnB1dFBhdGgvUGFyYW1ldGVycy9PdXRwdXRQYXRoIGluIEFTTCBKU09OIGZvcm1hdFxuICAgICAqL1xuICAgIHByb3RlY3RlZCByZW5kZXJJbnB1dE91dHB1dCgpOiBhbnkge1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgSW5wdXRQYXRoOiByZW5kZXJKc29uUGF0aCh0aGlzLmlucHV0UGF0aCksXG4gICAgICAgICAgICBQYXJhbWV0ZXJzOiB0aGlzLnBhcmFtZXRlcnMsXG4gICAgICAgICAgICBPdXRwdXRQYXRoOiByZW5kZXJKc29uUGF0aCh0aGlzLm91dHB1dFBhdGgpLFxuICAgICAgICB9O1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIFJlbmRlciBwYXJhbGxlbCBicmFuY2hlcyBpbiBBU0wgSlNPTiBmb3JtYXRcbiAgICAgKi9cbiAgICBwcm90ZWN0ZWQgcmVuZGVyQnJhbmNoZXMoKTogYW55IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIEJyYW5jaGVzOiB0aGlzLmJyYW5jaGVzLm1hcChiID0+IGIudG9HcmFwaEpzb24oKSlcbiAgICAgICAgfTtcbiAgICB9XG5cbiAgICAvKipcbiAgICAgKiBSZW5kZXIgbWFwIGl0ZXJhdG9yIGluIEFTTCBKU09OIGZvcm1hdFxuICAgICAqL1xuICAgIHByb3RlY3RlZCByZW5kZXJJdGVyYXRvcigpOiBhbnkge1xuICAgICAgICBpZiAoIXRoaXMuaXRlcmF0aW9uKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEl0ZXJhdG9yIG11c3Qgbm90IGJlIHVuZGVmaW5lZCAhYCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIEl0ZXJhdG9yOiB0aGlzLml0ZXJhdGlvbi50b0dyYXBoSnNvbigpXG4gICAgICAgIH07XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogUmVuZGVyIGVycm9yIHJlY292ZXJ5IG9wdGlvbnMgaW4gQVNMIEpTT04gZm9ybWF0XG4gICAgICovXG4gICAgcHJvdGVjdGVkIHJlbmRlclJldHJ5Q2F0Y2goKTogYW55IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIFJldHJ5OiByZW5kZXJMaXN0KHRoaXMucmV0cmllcywgcmVuZGVyUmV0cnkpLFxuICAgICAgICAgICAgQ2F0Y2g6IHJlbmRlckxpc3QodGhpcy5jYXRjaGVzLCByZW5kZXJDYXRjaCksXG4gICAgICAgIH07XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogQ2FsbGVkIHdoZW5ldmVyIHRoaXMgc3RhdGUgaXMgYm91bmQgdG8gYSBncmFwaFxuICAgICAqXG4gICAgICogQ2FuIGJlIG92ZXJyaWRkZW4gYnkgc3ViY2xhc3Nlcy5cbiAgICAgKi9cbiAgICBwcm90ZWN0ZWQgd2hlbkJvdW5kVG9HcmFwaChncmFwaDogU3RhdGVHcmFwaCkge1xuICAgICAgICBncmFwaC5yZWdpc3RlclN0YXRlKHRoaXMpO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIEFkZCBhIHN0YXRlIHRvIHRoZSBpbmNvbWluZyBsaXN0XG4gICAgICovXG4gICAgcHJpdmF0ZSBhZGRJbmNvbWluZyhzb3VyY2U6IFN0YXRlKSB7XG4gICAgICAgIHRoaXMuaW5jb21pbmdTdGF0ZXMucHVzaChzb3VyY2UpO1xuICAgIH1cblxuICAgIC8qKlxuICAgICAqIFJldHVybiBhbGwgc3RhdGVzIHRoaXMgc3RhdGUgY2FuIHRyYW5zaXRpb24gdG9cbiAgICAgKi9cbiAgICBwcml2YXRlIG91dGdvaW5nVHJhbnNpdGlvbnMob3B0aW9uczogRmluZFN0YXRlT3B0aW9ucyk6IFN0YXRlW10ge1xuICAgICAgICBjb25zdCByZXQgPSBuZXcgQXJyYXk8U3RhdGU+KCk7XG4gICAgICAgIGlmICh0aGlzLl9uZXh0KSB7IHJldC5wdXNoKHRoaXMuX25leHQpOyB9XG4gICAgICAgIGlmICh0aGlzLmRlZmF1bHRDaG9pY2UpIHsgcmV0LnB1c2godGhpcy5kZWZhdWx0Q2hvaWNlKTsgfVxuICAgICAgICBmb3IgKGNvbnN0IGMgb2YgdGhpcy5jaG9pY2VzKSB7XG4gICAgICAgICAgICByZXQucHVzaChjLm5leHQpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChvcHRpb25zLmluY2x1ZGVFcnJvckhhbmRsZXJzKSB7XG4gICAgICAgICAgICBmb3IgKGNvbnN0IGMgb2YgdGhpcy5jYXRjaGVzKSB7XG4gICAgICAgICAgICAgICAgcmV0LnB1c2goYy5uZXh0KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBmaW5kaW5nIHJlYWNoYWJsZSBzdGF0ZXNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBGaW5kU3RhdGVPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIG9yIG5vdCB0byBmb2xsb3cgZXJyb3ItaGFuZGxpbmcgdHJhbnNpdGlvbnNcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5jbHVkZUVycm9ySGFuZGxlcnM/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEEgQ2hvaWNlIFRyYW5zaXRpb25cbiAqL1xuaW50ZXJmYWNlIENob2ljZVRyYW5zaXRpb24ge1xuICAgIC8qKlxuICAgICAqIFN0YXRlIHRvIHRyYW5zaXRpb24gdG9cbiAgICAgKi9cbiAgICBuZXh0OiBTdGF0ZTtcblxuICAgIC8qKlxuICAgICAqIENvbmRpdGlvbiBmb3IgdGhpcyB0cmFuc2l0aW9uXG4gICAgICovXG4gICAgY29uZGl0aW9uOiBDb25kaXRpb247XG59XG5cbi8qKlxuICogUmVuZGVyIGEgY2hvaWNlIHRyYW5zaXRpb25cbiAqL1xuZnVuY3Rpb24gcmVuZGVyQ2hvaWNlKGM6IENob2ljZVRyYW5zaXRpb24pIHtcbiAgICByZXR1cm4ge1xuICAgICAgICAuLi5jLmNvbmRpdGlvbi5yZW5kZXJDb25kaXRpb24oKSxcbiAgICAgICAgTmV4dDogYy5uZXh0LnN0YXRlSWRcbiAgICB9O1xufVxuXG4vKipcbiAqIEEgQ2F0Y2ggVHJhbnNpdGlvblxuICovXG5pbnRlcmZhY2UgQ2F0Y2hUcmFuc2l0aW9uIHtcbiAgICAvKipcbiAgICAgKiBTdGF0ZSB0byB0cmFuc2l0aW9uIHRvXG4gICAgICovXG4gICAgbmV4dDogU3RhdGU7XG5cbiAgICAvKipcbiAgICAgKiBBZGRpdGlvbmFsIHByb3BlcnRpZXMgZm9yIHRoaXMgdHJhbnNpdGlvblxuICAgICAqL1xuICAgIHByb3BzOiBDYXRjaFByb3BzO1xufVxuXG4vKipcbiAqIFJlbmRlciBhIFJldHJ5IG9iamVjdCB0byBBU0xcbiAqL1xuZnVuY3Rpb24gcmVuZGVyUmV0cnkocmV0cnk6IFJldHJ5UHJvcHMpIHtcbiAgICByZXR1cm4ge1xuICAgICAgICBFcnJvckVxdWFsczogcmV0cnkuZXJyb3JzLFxuICAgICAgICBJbnRlcnZhbFNlY29uZHM6IHJldHJ5LmludGVydmFsICYmIHJldHJ5LmludGVydmFsLnRvU2Vjb25kcygpLFxuICAgICAgICBNYXhBdHRlbXB0czogcmV0cnkubWF4QXR0ZW1wdHMsXG4gICAgICAgIEJhY2tvZmZSYXRlOiByZXRyeS5iYWNrb2ZmUmF0ZVxuICAgIH07XG59XG5cbi8qKlxuICogUmVuZGVyIGEgQ2F0Y2ggb2JqZWN0IHRvIEFTTFxuICovXG5mdW5jdGlvbiByZW5kZXJDYXRjaChjOiBDYXRjaFRyYW5zaXRpb24pIHtcbiAgICByZXR1cm4ge1xuICAgICAgICBFcnJvckVxdWFsczogYy5wcm9wcy5lcnJvcnMsXG4gICAgICAgIFJlc3VsdFBhdGg6IHJlbmRlckpzb25QYXRoKGMucHJvcHMucmVzdWx0UGF0aCksXG4gICAgICAgIE5leHQ6IGMubmV4dC5zdGF0ZUlkLFxuICAgIH07XG59XG5cbi8qKlxuICogUmVuZGVyIGEgbGlzdCBvciByZXR1cm4gdW5kZWZpbmVkIGZvciBhbiBlbXB0eSBsaXN0XG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJMaXN0PFQ+KHhzOiBUW10sIGZuOiAoeDogVCkgPT4gYW55KTogYW55IHtcbiAgICBpZiAoeHMubGVuZ3RoID09PSAwKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cbiAgICByZXR1cm4geHMubWFwKGZuKTtcbn1cblxuLyoqXG4gKiBSZW5kZXIgSlNPTiBwYXRoLCByZXNwZWN0aW5nIHRoZSBzcGVjaWFsIHZhbHVlIERJU0NBUkRcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHJlbmRlckpzb25QYXRoKGpzb25QYXRoPzogc3RyaW5nKTogdW5kZWZpbmVkIHwgbnVsbCB8IHN0cmluZyB7XG4gICAgaWYgKGpzb25QYXRoID09PSB1bmRlZmluZWQpIHsgcmV0dXJuIHVuZGVmaW5lZDsgfVxuICAgIGlmIChqc29uUGF0aCA9PT0gRElTQ0FSRCkgeyByZXR1cm4gbnVsbDsgfVxuXG4gICAgaWYgKCFqc29uUGF0aC5zdGFydHNXaXRoKCckJykpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBFeHBlY3RlZCBKU09OIHBhdGggdG8gc3RhcnQgd2l0aCAnJCcsIGdvdDogJHtqc29uUGF0aH1gKTtcbiAgICB9XG4gICAgcmV0dXJuIGpzb25QYXRoO1xufVxuXG4vKipcbiAqIEludGVyZmFjZSBmb3Igc3RydWN0dXJhbCBmZWF0dXJlIHRlc3RpbmcgKHRvIG1ha2UgVHlwZVNjcmlwdCBoYXBweSlcbiAqL1xuaW50ZXJmYWNlIFByZWZpeGFibGUge1xuICAgIGFkZFByZWZpeCh4OiBzdHJpbmcpOiB2b2lkO1xufVxuXG4vKipcbiAqIFdoZXRoZXIgYW4gb2JqZWN0IGlzIGEgUHJlZml4YWJsZVxuICovXG5mdW5jdGlvbiBpc1ByZWZpeGFibGUoeDogYW55KTogeCBpcyBQcmVmaXhhYmxlIHtcbiAgICByZXR1cm4gdHlwZW9mKHgpID09PSAnb2JqZWN0JyAmJiB4LmFkZFByZWZpeDtcbn1cblxuLyoqXG4gKiBXaGV0aGVyIGFuIG9iamVjdCBpcyBJTmV4dGFibGVcbiAqL1xuZnVuY3Rpb24gaXNOZXh0YWJsZSh4OiBhbnkpOiB4IGlzIElOZXh0YWJsZSB7XG4gICAgcmV0dXJuIHR5cGVvZih4KSA9PT0gJ29iamVjdCcgJiYgeC5uZXh0O1xufVxuIl19