"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fields_1 = require("./fields");
/**
 * Type union for task classes that accept multiple types of payload
 */
class TaskInput {
    /**
     *
     * @param type type of task input
     * @param value payload for the corresponding input type.
     * It can be a JSON-encoded object, context, data, etc.
     */
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    /**
     * Use a literal string as task input
     *
     * This might be a JSON-encoded object, or just a text.
     */
    static fromText(text) {
        return new TaskInput(InputType.TEXT, text);
    }
    /**
     * Use an object as task input
     *
     * This object may contain Data and Context fields
     * as object values, if desired.
     */
    static fromObject(obj) {
        return new TaskInput(InputType.OBJECT, obj);
    }
    /**
     * Use a part of the execution data as task input
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution as complete payload
     * to a task.
     */
    static fromDataAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Data.stringAt(path));
    }
    /**
     * Use a part of the task context as task input
     *
     * Use this when you want to use a subobject or string from
     * the current task context as complete payload
     * to a task.
     */
    static fromContextAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Context.stringAt(path));
    }
}
exports.TaskInput = TaskInput;
/**
 * The type of task input
 */
var InputType;
(function (InputType) {
    /**
     * Use a literal string
     * This might be a JSON-encoded object, or just text.
     * valid JSON text: standalone, quote-delimited strings; objects; arrays; numbers; Boolean values; and null.
     *
     * example: `literal string`
     * example: {"json": "encoded"}
     */
    InputType[InputType["TEXT"] = 0] = "TEXT";
    /**
     * Use an object which may contain Data and Context fields
     * as object values, if desired.
     *
     * example:
     * {
     *  literal: 'literal',
     *  SomeInput: sfn.Data.stringAt('$.someField')
     * }
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-data.html
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/input-output-contextobject.html
     */
    InputType[InputType["OBJECT"] = 1] = "OBJECT";
})(InputType = exports.InputType || (exports.InputType = {}));
//# sourceMappingURL=data:application/json;base64,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