"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const json_path_1 = require("./json-path");
/**
 * Extract a field from the State Machine data that gets passed around between states
 */
class Data {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateDataPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateDataPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isJsonPathString(value) {
        return !!json_path_1.jsonPathString(value);
    }
    constructor() {
    }
}
exports.Data = Data;
/**
 * Extract a field from the State Machine Context data
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 */
class Context {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateContextPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateContextPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
    constructor() {
    }
}
exports.Context = Context;
/**
 * Helper functions to work with structures containing fields
 */
class FieldUtils {
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure
     */
    static renderObject(obj) {
        return json_path_1.renderObject(obj);
    }
    /**
     * Return all JSON paths used in the given structure
     */
    static findReferencedPaths(obj) {
        return Array.from(json_path_1.findReferencedPaths(obj)).sort();
    }
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     */
    static containsTaskToken(obj) {
        const paths = json_path_1.findReferencedPaths(obj);
        return paths.has('$$.Task.Token') || paths.has('$$.Task') || paths.has('$$');
    }
    constructor() {
    }
}
exports.FieldUtils = FieldUtils;
function validateDataPath(path) {
    if (path !== '$' && !path.startsWith('$.')) {
        throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    }
}
function validateContextPath(path) {
    if (path !== '$$' && !path.startsWith('$$.')) {
        throw new Error("Context JSON path values must either be exactly equal to '$$' or start with '$$.'");
    }
}
//# sourceMappingURL=data:application/json;base64,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