"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const stepfunctions_generated_1 = require("./stepfunctions.generated");
/**
 * Define a new Step Functions Activity
 */
class Activity extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.activityName ||
                core_1.Lazy.stringValue({ produce: () => this.generateName() }),
        });
        const resource = new stepfunctions_generated_1.CfnActivity(this, 'Resource', {
            name: this.physicalName // not null because of above call to `super`
        });
        this.activityArn = this.getResourceArnAttribute(resource.ref, {
            service: 'states',
            resource: 'activity',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.activityName = this.getResourceNameAttribute(resource.attrName);
    }
    /**
     * Construct an Activity from an existing Activity ARN
     */
    static fromActivityArn(scope, id, activityArn) {
        class Imported extends core_1.Resource {
            get activityArn() { return activityArn; }
            get activityName() {
                return core_1.Stack.of(this).parseArn(activityArn, ':').resourceName || '';
            }
        }
        return new Imported(scope, id);
    }
    /**
     * Construct an Activity from an existing Activity Name
     */
    static fromActivityName(scope, id, activityName) {
        return Activity.fromActivityArn(scope, id, core_1.Stack.of(scope).formatArn({
            service: 'states',
            resource: 'activity',
            resourceName: activityName,
            sep: ':',
        }));
    }
    /**
     * Return the given named metric for this Activity
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { ActivityArn: this.activityArn },
            statistic: 'sum',
            ...props
        }).attachTo(this);
    }
    /**
     * The interval, in milliseconds, between the time the activity starts and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricRunTime(props) {
        return this.metric('ActivityRunTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, for which the activity stays in the schedule state.
     *
     * @default average over 5 minutes
     */
    metricScheduleTime(props) {
        return this.metric('ActivityScheduleTime', { statistic: 'avg', ...props });
    }
    /**
     * The interval, in milliseconds, between the time the activity is scheduled and the time it closes.
     *
     * @default average over 5 minutes
     */
    metricTime(props) {
        return this.metric('ActivityTime', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the number of times this activity is scheduled
     *
     * @default sum over 5 minutes
     */
    metricScheduled(props) {
        return this.metric('ActivitiesScheduled', props);
    }
    /**
     * Metric for the number of times this activity times out
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.metric('ActivitiesTimedOut', props);
    }
    /**
     * Metric for the number of times this activity is started
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.metric('ActivitiesStarted', props);
    }
    /**
     * Metric for the number of times this activity succeeds
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.metric('ActivitiesSucceeded', props);
    }
    /**
     * Metric for the number of times this activity fails
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.metric('ActivitiesFailed', props);
    }
    /**
     * Metric for the number of times the heartbeat times out for this activity
     *
     * @default sum over 5 minutes
     */
    metricHeartbeatTimedOut(props) {
        return this.metric('ActivitiesHeartbeatTimedOut', props);
    }
    generateName() {
        const name = this.node.uniqueId;
        if (name.length > 80) {
            return name.substring(0, 40) + name.substring(name.length - 40);
        }
        return name;
    }
}
exports.Activity = Activity;
//# sourceMappingURL=data:application/json;base64,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