"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Severity = exports.ImageRepository = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const aws_ecr_1 = require("@aws-cdk/aws-ecr");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_iam_1 = require("@aws-cdk/aws-iam");
class ImageRepository extends aws_ecr_1.Repository {
    constructor(scope, id, props) {
        const { forceDelete = false, ...rest } = props !== null && props !== void 0 ? props : {};
        super(scope, id, {
            removalPolicy: forceDelete ? core_1.RemovalPolicy.DESTROY : undefined,
            ...rest,
        });
        if (forceDelete) {
            new custom_resources_1.AwsCustomResource(this, 'ForceImageRepositoryDeletion', {
                resourceType: 'Custom::ECRForceImageRepositoryDeletion',
                onDelete: {
                    service: 'ECR',
                    action: 'deleteRepository',
                    parameters: {
                        repositoryName: this.repositoryName,
                        force: true,
                    },
                    physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.repositoryArn),
                },
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                    resources: [this.repositoryArn],
                }),
            });
        }
    }
    onFinding(id, options) {
        const { alarmTopic, severity, imageTags } = options;
        const rule = this.onImageScanCompleted(id, { imageTags });
        const severityFilter = new aws_lambda_1.Function(this, 'SevierityFilter', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, 'lambdas', 'severity-filter')),
            handler: 'index.handler',
        });
        severityFilter.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['sns:Publish'],
            resources: [alarmTopic.topicArn],
        }));
        rule.addTarget(new aws_events_targets_1.LambdaFunction(severityFilter, {
            event: aws_events_1.RuleTargetInput.fromObject({
                account: aws_events_1.EventField.account,
                region: aws_events_1.EventField.region,
                time: aws_events_1.EventField.time,
                repositoryName: aws_events_1.EventField.fromPath('$.detail.repository-name'),
                imageDigest: aws_events_1.EventField.fromPath('$.detail.image-digest'),
                imageTags: aws_events_1.EventField.fromPath('$.detail.image-tags'),
                findingSeveriyCounts: aws_events_1.EventField.fromPath('$.detail.finding-severity-counts'),
                severity: severity,
                alarmTopicArn: alarmTopic.topicArn,
            }),
        }));
        return rule;
    }
}
exports.ImageRepository = ImageRepository;
var Severity;
(function (Severity) {
    Severity["CRITICAL"] = "CRITICAL";
    Severity["HIGH"] = "HIGH";
    Severity["MEDIUM"] = "MEDIUM";
    Severity["LOW"] = "LOW";
    Severity["INFORMATIONAL"] = "INFORMATIONAL";
    Severity["UNDEFINED"] = "UNDEFINED";
})(Severity = exports.Severity || (exports.Severity = {}));
//# sourceMappingURL=data:application/json;base64,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