"""
This module is present to handle everything related to input data that are not structural.

In the Grid2Op vocabulary a "GridValue" or "Chronics" is something that provides data to change the input parameter
of a power flow between 1 time step and the other.

It is a more generic terminology. Modification that can be performed by :class:`GridValue` object includes, but
are not limited to:

  - injections such as:

    - productions active production setpoint
    - generators voltage setpoint
    - loads active consumption
    - loads reactive consumption

  - structural informations such as:

    - planned outage: powerline disconnection anticipated in advance
    - hazards: powerline disconnection that cannot be anticipated, for example due to a windstorm.

All powergrid modification that can be performed using an :class:`grid2op.Action` can be implemented as form of a
:class:`GridValue`.

The same mechanism than for :class:`grid2op.Action` or :class:`grid2op.Observation` is pursued here. All states
modifications made by the :class:`grid2op.Environment` must derived from the :class:`GridValue`. It is not
recommended to instanciate them directly, but rather to use the :class:`ChronicsHandler` for such a purpose.

Note that the values returned by a :class:`GridValue` are **backend dependant**. A GridValue object should always
return the data in the order expected by the :class:`grid2op.Backend`, regardless of the order in which data are given
in the files or generated by the data generator process.

This implies that changing the backend will change the output of :class:`GridValue`. More information about this
is given in the description of the :func:`GridValue.initialize` method.

Finally, compared to other Reinforcement Learning problems, is the possibility to use "forecast". This optional feature
can be accessed via the :class:`grid2op.Observation` and mainly the :func:`grid2op.Observation.simulate` method. The
data that are used to generate this forecasts come from the :class:`grid2op.GridValue` and are detailed in the
:func:`GridValue.forecasts` method.

"""
import os
import copy

import numpy as np
import pandas as pd

import warnings

from datetime import datetime, timedelta

try:
    from .Exceptions import *
    from .Space import RandomObject
except (ModuleNotFoundError, ImportError):
    from Exceptions import *
    from Space import RandomObject

from abc import ABC, abstractmethod

import pdb


# TODO sous echantillonner ou sur echantilloner les scenario: need to modify everything that affect the number
# TODO of time steps there, for example "Space.gen_min_time_on" or "params.NB_TIMESTEP_POWERFLOW_ALLOWED" for
# TODO example. And more generally, it would be better to have all of this attributes exported / imported in
# TODO time interval, instead of time steps.

# TODO add a class to sample "online" the data.

# TODO add a method 'skip' that can skip a given number of timestep or a until a specific date.


class GridValue(ABC):
    """
    This is the base class for every kind of data for the _grid.

    It allows the :class:`grid2op.Environment` to perform powergrid modification that make the "game" time dependant.

    It is not recommended to directly create :class:`GridValue` object, but to use the
    :attr:`grid2op.Environment.chronics_handler" for such a purpose. This is made in an attempt to make sure the
    :func:`GridValue.initialize` is called. Before this initialization, it is not recommended to use any
    :class:`GridValue` object.

    The method :func:`GridValue.next_chronics` should be used between two epoch of the game. If there are no more
    data to be generated from this object, then :func:`GridValue.load_next` should raise a :class:`StopIteration`
    exception and a call to :func:`GridValue.done` should return True.

    Attributes
    ----------
    time_interval: :class:`.datetime.timedelta`
        Time interval between 2 consecutive timestamps. Default 5 minutes.

    start_datetime:  :class:`datetime.datetime`
        The datetime of the first timestamp of the scenario.

    current_datetime: :class:`datetime.datetime`
        The timestamp of the current scenario.

    n_gen: ``int``
        Number of generators in the powergrid

    n_load: ``int``
        Number of loads in the powergrid

    n_line: ``int``
        Number of powerline in the powergrid

    max_iter: ``int``
        Number maximum of data to generate for one episode.

    curr_iter: ``int``
        Duration of the current episode.

    maintenance_time: ``numpy.ndarray``, dtype:``int``
        Number of time steps the next maintenance will take place with the following convention:

            - -1 no maintenance are planned for the forseeable future
            - 0 a maintenance is taking place
            - 1, 2, 3 ... a maintenance will take place in 1, 2, 3, ... time step

        Some examples are given in :func:`GridValue.maintenance_time_1d`.

    maintenance_duration: ``numpy.ndarray``, dtype:``int``
        Duration of the next maintenance. 0 means no maintenance is happening. If a maintenance is planned for a
        given powerline, this number decreases each time step, up until arriving at 0 when the maintenance is over. Note
        that if a maintenance is planned (see :attr:`GridValue.maintenance_time`) this number indicates how long
        the maintenance will last, and does not suppose anything on the maintenance taking place or not (= there can be
        positive number here without a powerline being removed from the grid for maintenance reason). Some examples are
        given in :func:`GridValue.maintenance_duration_1d`.

    hazard_duration: ``numpy.ndarray``, dtype:``int``
        Duration of the next hzard. 0 means no maintenance is happening. If a hazard is taking place for a
        given powerline, this number decreases each time step, up until arriving at 0 when the maintenance is over. On
        the contrary to :attr:`GridValue.maintenance_duration`, if a component of this vector is higher than 1, it
        means that the powerline is out of service. Some examples are
        given in :func:`GridValue.get_hazard_duration_1d`.


    """
    def __init__(self, time_interval=timedelta(minutes=5), max_iter=-1,
                 start_datetime=datetime(year=2019, month=1, day=1),
                 chunk_size=None):
        self.time_interval = time_interval
        self.current_datetime = start_datetime
        self.start_datetime = start_datetime
        self.n_gen = None
        self.n_load = None
        self.n_line = None
        self.max_iter = max_iter
        self.curr_iter = 0

        self.maintenance_time = None
        self.maintenance_duration = None
        self.hazard_duration = None

    @abstractmethod
    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend):
        """
        This function is used to initialize the data generator.
        It can be use to load scenarios, or to initialize noise if scenarios are generated on the fly. It must also
        initialize :attr:`GridValue.maintenance_time`, :attr:`GridValue.maintenance_duration` and
        :attr:`GridValue.hazard_duration`.

        This function should also increment :attr:`GridValue.curr_iter` of 1 each time it is called.

        The :class:`GridValue` is what makes the connection between the data (generally in a shape of files on the
        hard drive) and the power grid. One of the main advantage of the Grid2Op package is its ability to change
        the tool that computes the load flows. Generally, such :class:`grid2op.Backend` expects data in a specific
        format that is given by the way their internal powergrid is represented, and in particular, the "same"
        objects can have different name and different position. To ensure that the same chronics would
        produce the same results on every backend (**ie** regardless of the order of which the Backend is expecting
        the data, the outcome of the powerflow is the same) we encourage the user to provide a file that maps the name
        of the object in the chronics to the name of the same object in the backend.

        This is done with the "names_chronics_to_backend" dictionnary that has the following keys:

          - "loads"
          - "prods"
          - "lines"

        The value associated to each of these keys is in turn a mapping dictionnary from the chronics to the backend.
        This means that each *keys* of these subdictionnary is a name of one column in the files, and each values
        is the corresponding name of this same object in the dictionnary. An example is provided bellow.

        Parameters
        ----------
        order_backend_loads: ``numpy.ndarray``, dtype:str
            Ordered name, in the Backend, of the loads. It is required that a :class:`grid2op.Backend` object always
            output the informations in the same order. This array gives the name of the loads following this order.
            See the documentation of :mod:`grid2op.Backend` for more information about this.

        order_backend_prods: ``numpy.ndarray``, dtype:str
            Same as order_backend_loads, but for generators.

        order_backend_lines: ``numpy.ndarray``, dtype:str
            Same as order_backend_loads, but for powerline.

        order_backend_subs: ``numpy.ndarray``, dtype:str
            Same as order_backend_loads, but for powerline.

        names_chronics_to_backend: ``dict``
            See in the description of the method for more information about its format.

        Returns
        -------
        ``None``

        Examples
        --------
        For example, suppose we have a :class:`grid2op.Backend` with:

          - substations ids strart from 0 to N-1 (N being the number of substations in the powergrid)
          - loads named "load_i" with "i" the subtations to which it is connected
          - generators units named "gen_i" (i still the substation id to which it is connected)
          - powerlnes are named "i_j" if it connected substations i to substation j

        And on the other side, we have some files with the following conventions:

          - substations are numbered from 1 to N
          - loads are named "i_C" with i being the substation to which it is connected
          - generators are named "i_G" with is being the id of the substations to which it is connected
          - powerlines are namesd "i_j_k" where i is the origin substation, j the extremity substations and "k"
            is a unique identifier of this powerline in the powergrid.

        In this case, instead of renaming the powergrid (in the backend) of the data files, it is advised to build the
        following elements and initialize the object gridval of type :class:`GridValue` with:

        .. code-block:: python

            gridval = GridValue()  # Note: this code won't execute because "GridValue" is an abstract class
            order_backend_loads = ['load_1', 'load_2', 'load_13', 'load_3', 'load_4', 'load_5', 'load_8', 'load_9',
                                     'load_10', 'load_11', 'load_12']
            order_backend_prods = ['gen_1', 'gen_2', 'gen_5', 'gen_7', 'gen_0']
            order_backend_lines = ['0_1', '0_4', '8_9', '8_13', '9_10', '11_12', '12_13', '1_2', '1_3', '1_4', '2_3',
                                       '3_4', '5_10', '5_11', '5_12', '3_6', '3_8', '4_5', '6_7', '6_8']
            order_backend_subs = ['sub_0', 'sub_1', 'sub_10', 'sub_11', 'sub_12', 'sub_13', 'sub_2', 'sub_3', 'sub_4',
                                      'sub_5', 'sub_6', 'sub_7', 'sub_8', 'sub_9']
            names_chronics_to_backend = {"loads": {"2_C": 'load_1', "3_C": 'load_2',
                                                       "14": 'load_13', "4_C": 'load_3', "5_C": 'load_4',
                                                       "6_C": 'load_5', "9_C": 'load_8', "10_C": 'load_9',
                                                       "11_C": 'load_10', "12_C": 'load_11',
                                                       "13_C": 'load_12'},
                                             "lines": {'1_2_1': '0_1', '1_5_2': '0_4', '9_10_16': '8_9', '9_14_17': '8_13',
                                                      '10_11_18': '9_10', '12_13_19': '11_12', '13_14_20': '12_13',
                                                       '2_3_3': '1_2', '2_4_4': '1_3', '2_5_5': '1_4', '3_4_6': '2_3',
                                                       '4_5_7': '3_4', '6_11_11': '5_10', '6_12_12': '5_11',
                                                       '6_13_13': '5_12', '4_7_8': '3_6', '4_9_9': '3_8', '5_6_10': '4_5',
                                                      '7_8_14': '6_7', '7_9_15': '6_8'},
                                             "prods": {"1_G": 'gen_0', "3_G": "gen_2", "6_G": "gen_5",
                                                       "2_G": "gen_1", "8_G": "gen_7"},
                                            }
            gridval.initialize(order_backend_loads, order_backend_prods, order_backend_lines, names_chronics_to_backend)

        """
        self.curr_iter += 1
        self.current_datetime += self.time_interval

    @staticmethod
    def get_maintenance_time_1d(maintenance):
        """
        This function allows to transform a 1d numpy aarray maintenance, where is specify:

            - 0 there is no maintenance at this time step
            - 1 there is a maintenance at this time step

        Into the representation in terms of "next maintenance time" as specified in
        :attr:`GridValue.maintenance_time` which is:

            - `-1` no foreseeable maintenance operation will be performed
            - `0` a maintenance operation is being performed
            - `1`, `2` etc. is the number of time step the next maintenance will be performed.

        Parameters
        ----------
        maintenance: ``numpy.ndarray``
            1 dimensional array representing the time series of the maintenance (0 there is no maintenance, 1 there
            is a maintenance at this time step)

        Returns
        -------
        maintenance_duration: ``numpy.ndarray``
            Array representing the time series of the duration of the next maintenance forseeable.

        Examples
        --------

        If no maintenance are planned:

        .. code-block:: python

            maintenance_time = GridValue.get_maintenance_time_1d(np.array([0 for _ in range(10)]))
            assert np.all(maintenance_time == np.array([-1  for _ in range(10)]))


        If a maintenance planned of 3 time steps starting at timestep 6 (index 5 - index starts at 0)

        .. code-block:: python

            maintenance = np.array([0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,0])
            maintenance_time = GridValue.get_maintenance_time_1d(maintenance)
            assert np.all(maintenance_time == np.array([5,4,3,2,1,0,0,0,-1,-1,-1,-1,-1,-1,-1,-1]))

        If a maintenance planned of 3 time steps starting at timestep 6
        (index 5 - index starts at 0), and a second one for 2 time steps at time step 13

        .. code-block:: python

            maintenance = np.array([0,0,0,0,0,1,1,1,0,0,0,0,1,1,0,0,0])
            maintenance_time = GridValue.get_maintenance_time_1d(maintenance)
            assert np.all(maintenance_time == np.array([5,4,3,2,1,0,0,0,4,3,2,1,0,0,-1,-1,-1]))

        """

        res = np.full(maintenance.shape, fill_value=np.NaN, dtype=np.int)
        maintenance = np.concatenate((maintenance, (0, 0)))
        a = np.diff(maintenance)
        # +1 is because numpy does the diff `t+1` - `t` so to get index of the initial array
        # I need to "+1"
        start = np.where(a == 1)[0] + 1  # start of maintenance
        end = np.where(a == -1)[0] + 1  # end of maintenance
        prev_ = 0
        # it's efficient here as i do a loop only on the number of time there is a maintenance
        # and maintenance are quite rare
        for beg_, end_ in zip(start, end):
            res[prev_:beg_] = list(range(beg_ - prev_, 0, -1))
            res[beg_:end_] = 0
            prev_ = end_

        # no maintenance are planned in the forseeable future
        res[prev_:] = -1
        return res

    @staticmethod
    def get_maintenance_duration_1d(maintenance):
        """
        This function allows to transform a 1d numpy aarray maintenance (or hazards), where is specify:

            - 0 there is no maintenance at this time step
            - 1 there is a maintenance at this time step

        Into the representation in terms of "next maintenance duration" as specified in
        :attr:`GridValue.maintenance_duration` which is:

            - `0` no forseeable maintenance operation will be performed
            - `1`, `2` etc. is the number of time step the next maintenance will last (it can be positive even in the
                case that no maintenance is currently being performed.

        Parameters
        ----------
        maintenance: ``numpy.ndarray``
            1 dimensional array representing the time series of the maintenance (0 there is no maintenance, 1 there
            is a maintenance at this time step)

        Returns
        -------
        maintenance_duration: ``numpy.ndarray``
            Array representing the time series of the duration of the next maintenance forseeable.

        Examples
        --------

        If no maintenance are planned:

        .. code-block:: python

            maintenance = np.array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0])
            maintenance_duration = GridValue.get_maintenance_duration_1d(maintenance)
            assert np.all(maintenance_duration == np.array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]))

        If a maintenance planned of 3 time steps starting at timestep 6 (index 5 - index starts at 0)

        .. code-block:: python

            maintenance = np.array([0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,0])
            maintenance_duration = GridValue.get_maintenance_duration_1d(maintenance)
            assert np.all(maintenance_duration == np.array([3,3,3,3,3,3,2,1,0,0,0,0,0,0,0,0]))

        If a maintenance planned of 3 time steps starting at timestep 6
        (index 5 - index starts at 0), and a second one for 2 time steps at time step 13

        .. code-block:: python

            maintenance = np.array([0,0,0,0,0,1,1,1,0,0,0,0,1,1,0,0,0])
            maintenance_duration = GridValue.get_maintenance_duration_1d(maintenance)
            assert np.all(maintenance_duration == np.array([3,3,3,3,3,3,2,1,2,2,2,2,2,1,0,0,0]))

        """

        res = np.full(maintenance.shape, fill_value=np.NaN, dtype=np.int)
        maintenance = np.concatenate((maintenance, (0,0)))
        a = np.diff(maintenance)
        # +1 is because numpy does the diff `t+1` - `t` so to get index of the initial array
        # I need to "+1"
        start = np.where(a == 1)[0] + 1  # start of maintenance
        end = np.where(a == -1)[0] + 1  # end of maintenance
        prev_ = 0
        # it's efficient here as i do a loop only on the number of time there is a maintenance
        # and maintenance are quite rare
        for beg_, end_ in zip(start, end):
            res[prev_:beg_] = end_ - beg_
            res[beg_:end_] = list(range(end_ - beg_, 0, -1))
            prev_ = end_

        # no maintenance are planned in the foreseeable future
        res[prev_:] = 0
        return res

    @staticmethod
    def get_hazard_duration_1d(hazard):
        """
        This function allows to transform a 1d numpy aarray maintenance (or hazards), where is specify:

            - 0 there is no maintenance at this time step
            - 1 there is a maintenance at this time step

        Into the representation in terms of "hzard duration" as specified in
        :attr:`GridValue.maintenance_duration` which is:

            - `0` no forseeable hazard operation will be performed
            - `1`, `2` etc. is the number of time step the next hzard will last (it is positive only when a hazard
                affect a given powerline)

        Compared to :func:`GridValue.get_maintenance_duration_1d` we only know when the hazard occurs how long it
        will last.

        Parameters
        ----------
        maintenance: ``numpy.ndarray``
            1 dimensional array representing the time series of the maintenance (0 there is no maintenance, 1 there
            is a maintenance at this time step)

        Returns
        -------
        maintenance_duration: ``numpy.ndarray``
            Array representing the time series of the duration of the next maintenance forseeable.

        Examples
        --------

        If no maintenance are planned:

        .. code-block:: python

            hazard = np.array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0])
            hazard_duration = GridValue.get_hazard_duration_1d(hazard)
            assert np.all(hazard_duration == np.array([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]))

        If a maintenance planned of 3 time steps starting at timestep 6 (index 5 - index starts at 0)

        .. code-block:: python

            hazard = np.array([0,0,0,0,0,1,1,1,0,0,0,0,0,0,0,0])
            hazard_duration = GridValue.get_hazard_duration_1d(hazard)
            assert np.all(hazard_duration == np.array([0,0,0,0,0,3,2,1,0,0,0,0,0,0,0,0]))

        If a maintenance planned of 3 time steps starting at timestep 6
        (index 5 - index starts at 0), and a second one for 2 time steps at time step 13

        .. code-block:: python

            hazard = np.array([0,0,0,0,0,1,1,1,0,0,0,0,1,1,0,0,0])
            hazard_duration = GridValue.get_hazard_duration_1d(hazard)
            assert np.all(hazard_duration == np.array([0,0,0,0,0,3,2,1,0,0,0,0,2,1,0,0,0]))

        """

        res = np.full(hazard.shape, fill_value=np.NaN, dtype=np.int)
        hazard = np.concatenate((hazard, (0, 0)))
        a = np.diff(hazard)
        # +1 is because numpy does the diff `t+1` - `t` so to get index of the initial array
        # I need to "+1"
        start = np.where(a == 1)[0] + 1  # start of maintenance
        end = np.where(a == -1)[0] + 1  # end of maintenance
        prev_ = 0
        # it's efficient here as i do a loop only on the number of time there is a maintenance
        # and maintenance are quite rare
        for beg_, end_ in zip(start, end):
            res[prev_:beg_] = 0
            res[(beg_):(end_)] = list(range(end_ - beg_, 0, -1))
            prev_ = end_

        # no maintenance are planned in the forseeable future
        res[prev_:] = 0
        return res

    @abstractmethod
    def load_next(self):
        """
        Generate the next values, either by reading from a file, or by generating on the fly and return a dictionnary
        compatible with the :class:`grid2op.Action` class allowed for the :class:`Environment`.

        More information about this dictionnary can be found at :func:`grid2op.Action.update`.

        As a (quick) reminder: this dictionnary has for keys:

          - "injection" (optional): a dictionnary with keys (optional) "load_p", "load_q", "prod_p", "prod_v"
          - "hazards" (optional) : the outage suffered from the _grid
          - "maintenance" (optional) : the maintenance operations planned on the grid for the current time step.

        Returns
        -------
        timestamp: ``datetime.datetime``
            The current timestamp for which the modifications have been generated.

        dict_: ``dict``
            A dictionnary understandable by the ::func:`grid2op.Action.update` method. **NB** this function should
            return the dictionnary that will be converted, is should not, in any case, return an action.

        maintenance_time: ``numpy.ndarray``, dtype:``int``
            Information about the next planned maintenance. See :attr:`GridValue.maintenance_time` for more information.

        maintenance_duration: ``numpy.ndarray``, dtype:``int``
            Information about the duration of next planned maintenance. See :attr:`GridValue.maintenance_duration`
            for more information.

        hazard_duration: ``numpy.ndarray``, dtype:``int``
            Information about the current hazard. See :attr:`GridValue.hazard_duration`
            for more information.

        Raises
        ------
        StopIteration
            if the chronics is over
        """
        self.current_datetime += self.time_interval
        return self.current_datetime, {}, self.maintenance_time, self.maintenance_duration, self.hazard_duration

    @abstractmethod
    def check_validity(self, backend):
        """
        To make sure that the data returned by this class are of the proper dimension, a call to this method
        must be performed before actually using the data generated by this class.

        In the grid2op framework, this is ensure because the :class:`grid2op.Environment` calls this method
        in its initialization.

        Parameters
        ----------
        backend: :class:`grid2op.Backend`
            The backend used by the :class;`Environment`.

        Returns
        -------

        """
        raise EnvError("check_validity not implemented")

    def done(self):
        """
        Whether the episode is over or not.

        Returns
        -------
        done: ``bool``
            ``True`` means the episode has arrived to the end (no more data to generate) ``False`` means that the episode
            is not over yet.

        """
        if self.max_iter >= 0:
            return self.curr_iter >= self.max_iter
        else:
            return False

    def forecasts(self):
        """
        This method is used to generate the forecasts that are made available to the :class:`grid2op.Agent`.
        This forecasts are behaving the same way than a list of tuple as the one returned by
        :func:`GridValue.load_next` method.

        The way they are generated depends on the GridValue class. If not forecasts are made available, then
        the empty list should be returned.

        Returns
        -------
        res: ``list``
            Each element of this list having the same type as what is returned by :func:`GridValue.load_next`.
        """
        return []

    @abstractmethod
    def next_chronics(self):
        """
        Load the next batch of chronics. This function is called after an episode has finished by the
        :class:`grid2op.Environment` or the :class:`grid2op.Runner`.

        A call to this function should also reset :attr:`GridValue.curr_iter` to 0.
        Returns
        -------
        ``None``
        """
        pass

    def tell_id(self, id_num):
        """
        Tell the backend to use one folder for the chronics in particular. This method is mainly use when the GridValue
        object can deal with many folder. In this case, this method is used by the :class:`grid2op.Runner` to indicate
        which chronics to load for the current simulated episode.

        This is important to ensure reproducibility, especially in parrallel computation settings.

        This should also be used in case of generation "on the fly" of the chronics to ensure the same property.

        By default it does nothing.

        Returns
        -------
        ``None``
        """
        warnings.warn("Class {} doesn't handle different input folder. \"tell_id\" method has no impact."
                      "".format(type(self).__name__))

    def get_id(self) -> str:
        """
        Utility to get the current name of the path of the data are looked at, if data are files.

        This could also be used to return a unique identifier to the generated chronics even in the case where they are
        generated on the fly, for example by return a hash of the seed.

        Returns
        -------
        res: ``str``
            A unique identifier of the chronics generated for this episode. For example, if the chronics comes from a
            specific folder, this could be the path to this folder.

        """
        warnings.warn("Class {} doesn't handle different input folder. \"get_id\" method will return \"\"."
                      "".format(type(self).__name__))
        return ""

    def max_timestep(self):
        """
        This method returned the maximum timestep that the current episode can last.
        Note that if the :class:`grid2op.Agent` performs a bad action that leads to a game over, then the episode
        can lasts less.

        Returns
        -------
        res: int
            -1 if possibly infinite length of a positive integer representing the maximum duration of this episode

        """
        # warnings.warn("Class {} has possibly and infinite duration.".format(type(self).__name__))
        return self.max_iter

    def shuffle(self, shuffler):
        """
        This method can be overiden if the data that are represented by this object need to be shuffle.

        By default it does nothing.
        Parameters
        ----------
        shuffler: ``object``
            Any function that can be used to shuffle the data.

        Returns
        -------

        """
        pass

    def set_chunk_size(self, new_chunk_size):
        """
        TODO
        Parameters
        ----------
        new_chunk_size

        Returns
        -------

        """
        pass


class ChangeNothing(GridValue):
    """
    This class is the most basic class to modify a powergrid values.
    It does nothing exceptie increasing :attr:`GridValue.max_iter` and the :attr:`GridValue.current_datetime`.
    """
    def __init__(self, time_interval=timedelta(minutes=5), max_iter=-1,
                 start_datetime=datetime(year=2019, month=1, day=1),
                 chunk_size=None):
        GridValue.__init__(self, time_interval=time_interval, max_iter=max_iter, start_datetime=start_datetime,
                           chunk_size=chunk_size)

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):
        self.n_gen = len(order_backend_prods)
        self.n_load = len(order_backend_loads)
        self.n_line = len(order_backend_lines)
        self.curr_iter = 0

        self.maintenance_time = np.zeros(shape=(self.n_line, ), dtype=np.int) - 1
        self.maintenance_duration = np.zeros(shape=(self.n_line, ), dtype=np.int)
        self.hazard_duration = np.zeros(shape=(self.n_line, ), dtype=np.int)

    def load_next(self):
        """
        This function does nothing but the two requirements of load_next ie:

          - increasing :attr:`GridValue.curr_iter` of 1
          - increasing :attr:`GridValue.current_datetime`

        Returns
        -------
        timestamp: ``datetime.datetime``
            The current timestamp for which the modifications have been generated.

        dict_: ``dict``
            Always empty, indicating i do nothing.

        maintenance_time: ``numpy.ndarray``, dtype:``int``
            Information about the next planned maintenance. See :attr:`GridValue.maintenance_time` for more information.

        maintenance_duration: ``numpy.ndarray``, dtype:``int``
            Information about the duration of next planned maintenance. See :attr:`GridValue.maintenance_duration`
            for more information.

        hazard_duration: ``numpy.ndarray``, dtype:``int``
            Information about the current hazard. See :attr:`GridValue.hazard_duration`
            for more information.

        """
        self.current_datetime += self.time_interval
        self.curr_iter += 1
        return self.current_datetime, {}, self.maintenance_time, self.maintenance_duration, self.hazard_duration, None

    def check_validity(self, backend):
        """

        Parameters
        ----------
        backend: :class:`grid2op.Backend`
            The backend, not used here.

        Returns
        -------
        res: ``bool``
            Always ``True``. As this doesn't change the powergird, there is no way to make invalid changed.
        """
        return True

    def next_chronics(self):
        """
        Restarts:

          - :attr:`GridValue.current_datetime` to its origin value ( 2019 / 01 / 01)
          - :attr:`GridValue.curr_iter` to 0

        Returns
        -------

        """
        self.current_datetime = self.start_datetime
        self.curr_iter = 0


class GridStateFromFile(GridValue):
    """
    Read the injections values from a file stored on hard drive. More detailed about the files is provided in the
    :func:`GridStateFromFile.initialize` method.

    This class reads only files stored as csv. The header of the csv is mandatory and should represent the name of
    the objects. This names should either be matched to the name of the same object in the backend using the
    `names_chronics_to_backend` argument pass into the :func:`GridStateFromFile.initialize` (see
    :func:`GridValue.initialize` for more information) or match the names of the object in the backend.

    When the grid value is initialized, all present csv are read, sorted in order compatible with the backend and
    extracted as numpy array.

    For now, the current date and times are not read from file. It is mandatory that the chronics starts at 00:00 and
    its first time stamps is corresponds to January, 1st 2019.

    Chronics read from this files don't implement the "forecast" value.

    In this values, only 1 episode is stored. If the end of the episode is reached and another one should start, then
    it will loop from the beginning.

    Attributes
    ----------
    path: ``str``
        The path of the folder where the data are stored. It is recommended to set absolute path, and not relative
        paths.

    load_p: ``numpy.ndarray``, dtype: ``float``
        All the values of the load active values

    load_q: ``numpy.ndarray``, dtype: ``float``
        All the values of the load reactive values

    prod_p: ``numpy.ndarray``, dtype: ``float``
        All the productions setpoint active values.

    prod_v: ``numpy.ndarray``, dtype: ``float``
        All the productions setpoint voltage magnitude values.

    hazards: ``numpy.ndarray``, dtype: ``bool``
        This vector represents the possible hazards. It is understood as: ``True`` there is a hazard
        for the given powerline, ``False`` there is not.

    maintenance: ``numpy.ndarray``, dtype: ``bool``
        This vector represents the possible maintenance. It is understood as: ``True`` there is a maintenance
        for the given powerline, ``False`` there is not.

    current_index: ``int``
        The index of the last observation sent to the :class:`grid2op.Environment`.

    sep: ``str``, optional
        The csv columns separator. By defaults it's ";"

    names_chronics_to_backend: ``dict``
        This directory matches the name of the objects (line extremity, generator or load) to the same object in the
        backed. See the help of :func:`GridValue.initialize` for more information).
    """
    def __init__(self,
                 path,
                 sep=";",
                 time_interval=timedelta(minutes=5),
                 max_iter=-1,
                 start_datetime=datetime(year=2019, month=1, day=1),
                 chunk_size=None):
        """
        Build an instance of GridStateFromFile. Such an instance should be built before an :class:`grid2op.Environment`
        is created.

        Parameters
        ----------
        path: ``str``
            Used to initialize :attr:`GridStateFromFile.path`

        sep: ``str``, optional
            Used to initialize :attr:`GridStateFromFile.sep`

        time_interval: ``datetime.timedelta``
            Used to initialize :attr:`GridValue.time_interval`

        max_iter: int, optional
            Used to initialize :attr:`GridValue.max_iter`

        """
        GridValue.__init__(self, time_interval=time_interval, max_iter=max_iter, start_datetime=start_datetime,
                           chunk_size=chunk_size)

        self.path = path
        self.n_ = None  # maximum number of rows of the array
        self.tmp_max_index = None  # size maximum of the current tables in memory
        self.load_p = None  # numpy array corresponding to the current active load values in the power _grid. It has the same size as the number of loads
        self.load_q = None  # numpy array corresponding to the current reactive load values in the power _grid. It has the same size as the number of loads
        self.prod_p = None  # numpy array corresponding to the current active production values in the power _grid. It has the same size as the number of generators
        self.prod_v = None   # numpy array corresponding to the current voltage production setpoint values in the power _grid. It has the same size as the number of generators

        # for the two following vector, the convention is the following: False(line is disconnected) / True(line is connected)
        self.hazards = None   # numpy array representing the outage (unplanned), same size as the number of powerlines on the _grid.
        self.maintenance = None  # numpy array representing the _maintenance (planned withdrawal of a powerline), same size as the number of powerlines on the _grid.

        self.current_index = -1
        self.sep = sep

        self.names_chronics_to_backend = None

        # added to provide an easier access to read data in chunk
        self.chunk_size = chunk_size
        self._data_chunk = {}
        self._order_load_p = None
        self._order_load_q = None
        self._order_prod_p = None
        self._order_prod_v = None
        self._order_hazards = None
        self._order_maintenance = None

    def _assert_correct(self, dict_convert, order_backend):
        len_backend = len(order_backend)
        len_dict_keys = len(dict_convert)
        vals = set(dict_convert.values())
        lend_dict_values = len(vals)

        if len_dict_keys != len_backend:
            err_msg = "Conversion mismatch between backend data {} elements and converter data {} (keys)"
            raise IncorrectNumberOfElements(err_msg.format(len_backend, len_dict_keys))
        if lend_dict_values != len_backend:
            err_msg = "Conversion mismatch between backend data {} elements and converter data {} (values)"
            raise IncorrectNumberOfElements(err_msg.format(len_backend, lend_dict_values))

        for el in order_backend:
            if not el in vals:
                raise ChronicsError("Impossible to find element \"{}\" in the original converter data".format(el))

    def _assert_correct_second_stage(self, pandas_name, dict_convert, key, extra=""):
        for i, el in enumerate(pandas_name):
            if not el in dict_convert[key]:
                raise ChronicsError("Element named {} is found in the data (column {}) but it is not found on the "
                                    "powergrid for data of type \"{}\".\nData in files  are: {}\n"
                                    "Converter data are: {}".format(el, i+1, key, sorted(list(pandas_name)),
                                                                    sorted(list(dict_convert[key].keys()))))

    def _init_date_time(self):
        if os.path.exists(os.path.join(self.path, "start_datetime.info")):
            with open(os.path.join(self.path, "start_datetime.info"), "r") as f:
                a = f.read().rstrip().lstrip()
            try:
                tmp = datetime.strptime(a, "%Y-%m-%d %H:%M")
            except ValueError:
                tmp = datetime.strptime(a, "%Y-%m-%d")
            except Exception:
                raise ChronicsNotFoundError("Impossible to understand the content of \"start_datetime.info\". Make sure "
                                            "it's composed of only one line with a datetime in the \"%Y-%m-%d %H:%M\""
                                            "format.")
            self.start_datetime = tmp

        if os.path.exists(os.path.join(self.path, "time_interval.info")):
            with open(os.path.join(self.path, "time_interval.info"), "r") as f:
                a = f.read().rstrip().lstrip()
            try:
                tmp = datetime.strptime(a, "%H:%M")
            except ValueError:
                tmp = datetime.strptime(a, "%M")
            except Exception:
                raise ChronicsNotFoundError("Impossible to understand the content of \"time_interval.info\". Make sure "
                                            "it's composed of only one line with a datetime in the \"%H:%M\""
                                            "format.")
            self.time_interval = timedelta(hours=tmp.hour, minutes=tmp.minute)

    def _get_fileext(self, data_name):
        read_compressed = ".csv"
        if not os.path.exists(os.path.join(self.path, "{}.csv".format(data_name))):
            # try to read compressed data
            if os.path.exists(os.path.join(self.path, "{}.csv.bz2".format(data_name))):
                read_compressed = ".csv.bz2"
            elif os.path.exists(os.path.join(self.path, "{}.zip".format(data_name))):
                read_compressed = ".zip"
            elif os.path.exists(os.path.join(self.path, "{}.csv.gzip".format(data_name))):
                read_compressed = ".csv.gzip"
            elif os.path.exists(os.path.join(self.path, "{}.csv.xz".format(data_name))):
                read_compressed = ".csv.xz"
            else:
                read_compressed = None
                # raise ChronicsNotFoundError(
                #     "GridStateFromFile: unable to locate the data files that should be at \"{}\"".format(self.path))
        return read_compressed

    def _get_data(self, data_name):
        file_ext = self._get_fileext(data_name)
        if file_ext is not None:
            res = pd.read_csv(os.path.join(self.path, "{}{}".format(data_name, file_ext)),
                        sep=self.sep, chunksize=self.chunk_size)
        else:
            res = None
        return res

    def _get_orders(self, load_p, load_q, prod_p, prod_v, hazards, maintenance,
                    order_backend_loads, order_backend_prods, order_backend_lines):

        order_chronics_load_p = None
        order_backend_load_q = None
        order_backend_prod_p = None
        order_backend_prod_v = None
        order_backend_hazards = None
        order_backend_maintenance = None

        if load_p is not None:
            self._assert_correct_second_stage(load_p.columns, self.names_chronics_to_backend, "loads", "active")
            order_chronics_load_p = np.array([order_backend_loads[self.names_chronics_to_backend["loads"][el]]
                                              for el in load_p.columns]).astype(np.int)
        if load_q is not None:
            self._assert_correct_second_stage(load_q.columns, self.names_chronics_to_backend, "loads", "reactive")
            order_backend_load_q = np.array([order_backend_loads[self.names_chronics_to_backend["loads"][el]]
                                             for el in load_q.columns]).astype(np.int)

        if prod_p is not None:
            self._assert_correct_second_stage(prod_p.columns, self.names_chronics_to_backend, "prods", "active")
            order_backend_prod_p = np.array([order_backend_prods[self.names_chronics_to_backend["prods"][el]]
                                             for el in prod_p.columns]).astype(np.int)

        if prod_v is not None:
            self._assert_correct_second_stage(prod_v.columns, self.names_chronics_to_backend, "prods", "voltage magnitude")
            order_backend_prod_v = np.array([order_backend_prods[self.names_chronics_to_backend["prods"][el]]
                                             for el in prod_v.columns]).astype(np.int)

        if hazards is not None:
            self._assert_correct_second_stage(hazards.columns, self.names_chronics_to_backend, "lines", "hazards")
            order_backend_hazards = np.array([order_backend_lines[self.names_chronics_to_backend["lines"][el]]
                                              for el in hazards.columns]).astype(np.int)

        if maintenance is not None:
            self._assert_correct_second_stage(maintenance.columns, self.names_chronics_to_backend, "lines", "maintenance")
            order_backend_maintenance = np.array([order_backend_lines[self.names_chronics_to_backend["lines"][el]]
                                                  for el in maintenance.columns]).astype(np.int)

        return order_chronics_load_p, order_backend_load_q, \
               order_backend_prod_p, order_backend_prod_v, \
               order_backend_hazards, order_backend_maintenance

    def _get_next_chunk(self):
        load_p = None
        load_q = None
        prod_p = None
        prod_v = None
        if self._data_chunk["load_p"] is not None:
            load_p = next(self._data_chunk["load_p"])
            self.tmp_max_index = load_p.shape[0]
        if self._data_chunk["load_q"] is not None:
            load_q = next(self._data_chunk["load_q"])
            self.tmp_max_index = load_q.shape[0]
        if self._data_chunk["prod_p"] is not None:
            prod_p = next(self._data_chunk["prod_p"])
            self.tmp_max_index = prod_p.shape[0]
        if self._data_chunk["prod_v"] is not None:
            prod_v = next(self._data_chunk["prod_v"])
            self.tmp_max_index = prod_v.shape[0]
        return load_p, load_q, prod_p, prod_v

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):
        """
        In this function, the numpy arrays are read from the csv using the panda.dataframe engine.

        In order to be valid, the folder located at :attr:`GridStateFromFile.path` should at least contain:

          - a file named "load_p.csv" used to initialize :attr:`GridStateFromFile.load_p`
          - a file named "load_q.csv" used to initialize :attr:`GridStateFromFile.load_q`
          - a file named "prod_p.csv" used to initialize :attr:`GridStateFromFile.prod_p`
          - a file named "prod_v.csv" used to initialize :attr:`GridStateFromFile.prod_v`
          - a file named "hazards.csv" used to initialize :attr:`GridStateFromFile.hazards`
          - a file named "maintenance.csv" used to initialize :attr:`GridStateFromFile.maintenance`

        All these csv must have the same separator specified by :attr:`GridStateFromFile.sep`.

        If a file named "start_datetime.info" is present, then it will be used to initialized
        :attr:`GridStateFromFile.start_datetime`. If this file exists, it should count only one row, with the
        initial datetime in the "%Y-%m-%d %H:%M" format.

        If a file named "time_interval.info" is present, then it will be used to initialized the
        :attr:`GridStateFromFile.time_interval` attribute.  If this file exists, it should count only one row, with the
        initial datetime in the "%H:%M" format. Only timedelta composed of hours and minutes are supported (time delta
        cannot go above 23 hours 55 minutes and cannot be smaller than 0 hour 1 minutes)

        The first row of these csv is understood as the name of the object concerned by the column. Either this name is
        present in the :class:`grid2op.Backend`, in this case no modification is performed, or in case the name
        is not found in the backend and in this case it must be specified in the "names_chronics_to_backend"
        parameters how to understand it. See the help of :func:`GridValue.initialize` for more information
        about this dictionnary.

        All files should have the same number of rows.

        Parameters
        ----------
        See help of :func:`GridValue.initialize` for a detailed help about the parameters.

        Returns
        -------
        ``None``

        """
        self.n_gen = len(order_backend_prods)
        self.n_load = len(order_backend_loads)
        self.n_line = len(order_backend_lines)

        self.names_chronics_to_backend = copy.deepcopy(names_chronics_to_backend)
        if self.names_chronics_to_backend is None:
            self.names_chronics_to_backend = {}
        if not "loads" in self.names_chronics_to_backend:
            self.names_chronics_to_backend["loads"] = {k: k for k in order_backend_loads}
        else:
            self._assert_correct(self.names_chronics_to_backend["loads"], order_backend_loads)
        if not "prods" in self.names_chronics_to_backend:
            self.names_chronics_to_backend["prods"] = {k: k for k in order_backend_prods}
        else:
            self._assert_correct(self.names_chronics_to_backend["prods"], order_backend_prods)
        if not "lines" in self.names_chronics_to_backend:
            self.names_chronics_to_backend["lines"] = {k: k for k in order_backend_lines}
        else:
            self._assert_correct(self.names_chronics_to_backend["lines"], order_backend_lines)
        if not "subs" in self.names_chronics_to_backend:
            self.names_chronics_to_backend["subs"] = {k: k for k in order_backend_subs}
        else:
            self._assert_correct(self.names_chronics_to_backend["subs"], order_backend_subs)

        self._init_date_time()

        # read the data
        load_p_iter = self._get_data("load_p")
        load_q_iter = self._get_data("load_q")
        prod_p_iter = self._get_data("prod_p")
        prod_v_iter = self._get_data("prod_v")
        read_compressed = self._get_fileext("hazards")
        if read_compressed is not None:
            hazards = pd.read_csv(os.path.join(self.path, "hazards{}".format(read_compressed)),
                                       sep=self.sep)
        else:
            hazards = None

        read_compressed = self._get_fileext("maintenance")
        if read_compressed is not None:
            maintenance = pd.read_csv(os.path.join(self.path, "maintenance{}".format(read_compressed)),
                                           sep=self.sep)
        else:
            maintenance = None

        # put the proper name in order
        order_backend_loads = {el: i for i, el in enumerate(order_backend_loads)}
        order_backend_prods = {el: i for i, el in enumerate(order_backend_prods)}
        order_backend_lines = {el: i for i, el in enumerate(order_backend_lines)}

        if self.chunk_size is None:
            load_p = load_p_iter
            load_q = load_q_iter
            prod_p = prod_p_iter
            prod_v = prod_v_iter
            if load_p is not None:
                self.tmp_max_index = load_p.shape[0]
            elif load_q is not None:
                self.tmp_max_index = load_q.shape[0]
            elif prod_p is not None:
                self.tmp_max_index = prod_p.shape[0]
            elif prod_v is not None:
                self.tmp_max_index = prod_v.shape[0]
            else:
                raise ChronicsError("No files are found in directory \"{}\". If you don't want to load any chronics,"
                                    " use  \"ChangeNothing\" and not \"{}\" to load chronics."
                                    "".format(self.path, type(self)))

        else:
            self._data_chunk = {"load_p": load_p_iter,
                                "load_q": load_q_iter,
                                "prod_p": prod_p_iter,
                                "prod_v": prod_v_iter}
            load_p, load_q, prod_p, prod_v = self._get_next_chunk()

        # get the chronics in order
        order_chronics_load_p, order_backend_load_q, \
        order_backend_prod_p, order_backend_prod_v, \
        order_backend_hazards, order_backend_maintenance \
            = self._get_orders(load_p, load_q, prod_p, prod_v, hazards, maintenance,
                               order_backend_loads, order_backend_prods, order_backend_lines)

        # now "sort" the columns of each chunk of data
        self._order_load_p = np.argsort(order_chronics_load_p)
        self._order_load_q = np.argsort(order_backend_load_q)
        self._order_prod_p = np.argsort(order_backend_prod_p)
        self._order_prod_v = np.argsort(order_backend_prod_v)
        self._order_hazards = np.argsort(order_backend_hazards)
        self._order_maintenance = np.argsort(order_backend_maintenance)

        self._init_attrs(load_p, load_q, prod_p, prod_v, hazards=hazards, maintenance=maintenance)

        self.curr_iter = 0
        if self.maintenance is not None:
            n_ = self.maintenance.shape[0]
        elif self.hazards is not None:
            n_ = self.hazards.shape[0]
        else:
            n_ = None
            for fn in ["prod_p", "load_p", "prod_v", "load_q"]:
                ext_ = self._get_fileext(fn)
                if ext_ is not None:
                    n_ = self._file_len(os.path.join(self.path, "{}{}".format(fn, ext_)), ext_)
                    break
            if n_ is None:
                raise ChronicsError("No files are found in directory \"{}\". If you don't want to load any chronics,"
                                    " use  \"ChangeNothing\" and not \"{}\" to load chronics."
                                    "".format(self.path, type(self)))
        self.n_ = n_  # the -1 is present because the initial grid state doesn't count as a "time step"

        if self.max_iter == -1:
            # if the number of maximum time step is not set yet, we set it to be the number of
            # data in the chronics (number of rows of the files) -1.
            # the -1 is present because the initial grid state doesn't count as a "time step" but is read
            # from these data.
            self.max_iter = self.n_ -1

    @staticmethod
    def _file_len(fname, ext_):
        # i = -1
        # reading_mode = "r"
        # if ext_ != ".csv":
        #     reading_mode += "b"
        # with open(fname, reading_mode) as f:
        #     for i, l in enumerate(f):
        #         pass
        res = pd.read_csv(fname, sep="@", dtype=str).shape[0]
        return res

    def _init_attrs(self, load_p, load_q, prod_p, prod_v, hazards=None, maintenance=None):
        self.load_p = None
        self.load_q = None
        self.prod_p = None
        self.prod_v = None
        self.hazards = None
        self.hazard_duration = None
        self.maintenance = None
        self.maintenance_time = None
        self.maintenance_duration = None

        if load_p is not None:
            self.load_p = copy.deepcopy(load_p.values[:, self._order_load_p])
        if load_q is not None:
            self.load_q = copy.deepcopy(load_q.values[:, self._order_load_q])
        if prod_p is not None:
            self.prod_p = copy.deepcopy(prod_p.values[:, self._order_prod_p])
        if prod_v is not None:
            self.prod_v = copy.deepcopy(prod_v.values[:, self._order_prod_v])

        # TODO optimize this piece of code, and the whole laoding process if hazards.csv and maintenance.csv are
        # provided in the proper format.
        if hazards is not None:
            # hazards and maintenance cannot be computed by chunk. So we need to differenciate their behaviour
            self.hazards = copy.deepcopy(hazards.values[:, self._order_hazards])
            self.hazard_duration = np.zeros(shape=(self.hazards.shape[0], self.n_line), dtype=np.int)
            for line_id in range(self.n_line):
                self.hazard_duration[:, line_id] = self.get_hazard_duration_1d(self.hazards[:, line_id])

            self.hazards = self.hazards != 0.

        if maintenance is not None:
            self.maintenance = copy.deepcopy(maintenance.values[:, self._order_maintenance])
            self.maintenance_time = np.zeros(shape=(self.maintenance.shape[0], self.n_line), dtype=np.int) - 1
            self.maintenance_duration = np.zeros(shape=(self.maintenance.shape[0], self.n_line), dtype=np.int)

            # test that with chunk size
            for line_id in range(self.n_line):
                self.maintenance_time[:, line_id] = self.get_maintenance_time_1d(self.maintenance[:, line_id])
                self.maintenance_duration[:, line_id] = self.get_maintenance_duration_1d(self.maintenance[:, line_id])

            # there are _maintenance and hazards only if the value in the file is not 0.
            self.maintenance = self.maintenance != 0.

    def done(self):
        """
        Compare to :func:`GridValue.done` an episode can be over for 2 main reasons:

          - :attr:`GridValue.max_iter` has been reached
          - There are no data in the csv.

        The episode is done if one of the above condition is met.

        Returns
        -------
        res: ``bool``
            Whether the episode has reached its end or not.

        """
        res = False
        # if self.current_index+1 >= self.tmp_max_index:
        if self.current_index > self.n_:
            res = True
        elif self.max_iter > 0:
            if self.curr_iter > self.max_iter:
                res = True
        return res

    def _data_in_memory(self):
        if self.chunk_size is None:
            # if i don't use chunk, all the data are in memory alreay
            return True
        if self.current_index == 0:
            # data are loaded the first iteration
            return True
        if self.current_index % self.chunk_size != 0:
            # data are already in ram
            return True
        return False

    def _load_next_chunk_in_memory(self):
        # print("I loaded another chunk")
        # i load the next chunk as dataframes
        load_p, load_q, prod_p, prod_v = self._get_next_chunk()
        # i put these dataframes in the right order (columns)
        self._init_attrs(load_p, load_q, prod_p, prod_v)
        # i don't forget to reset the reading index to 0
        self.current_index = 0

    def load_next(self):
        self.current_index += 1

        if not self._data_in_memory():
            try:
                self._load_next_chunk_in_memory()
            except StopIteration as e:
                raise e

        if self.current_index >= self.tmp_max_index:
            raise StopIteration
        if self.max_iter > 0:
            if self.curr_iter > self.max_iter:
                raise StopIteration

        res = {}
        dict_ = {}
        prod_v = None
        if self.load_p is not None:
            dict_["load_p"] =  1.0 * self.load_p[self.current_index, :]
        if self.load_q is not None:
            dict_["load_q"] = 1.0 * self.load_q[self.current_index, :]
        if self.prod_p is not None:
            dict_["prod_p"] = 1.0 * self.prod_p[self.current_index, :]
        if self.prod_v is not None:
            prod_v = 1.0 * self.prod_v[self.current_index, :]
            # dict_["prod_v"] = prod_v
        if dict_:
            res["injection"] = dict_

        if self.maintenance is not None:
            res["maintenance"] = self.maintenance[self.current_index, :]
        if self.hazards is not None:
            res["hazards"] = self.hazards[self.current_index, :]

        self.current_datetime += self.time_interval
        self.curr_iter += 1

        if self.maintenance_time is not None:
            maintenance_time = 1 * self.maintenance_time[self.current_index, :]
            maintenance_duration = 1 * self.maintenance_duration[self.current_index, :]
        else:
            maintenance_time = np.full(self.n_line, fill_value=-1, dtype=np.int)
            maintenance_duration = np.full(self.n_line, fill_value=0, dtype=np.int)

        if self.hazard_duration is not None:
            hazard_duration = 1 * self.hazard_duration[self.current_index, :]
        else:
            hazard_duration = np.full(self.n_line, fill_value=-1, dtype=np.int)

        return self.current_datetime, res, maintenance_time, maintenance_duration, hazard_duration, prod_v

    def check_validity(self, backend):
        """
        A call to this method ensure that the action that will be sent to the current :class:`grid2op.Environment`
        can be properly implemented by its :class:`grid2op.Backend`.
        This specific method check that the dimension of all vectors are consistent

        Parameters
        ----------
        backend: :class:`grid2op.Backend.Backend`
            The backend used by the :class:`grid2op.Environment.Environment`

        Returns
        -------
        ``None``
        """
        at_least_one = False
        if self.load_p is not None:
            if self.load_p.shape[1] != backend.n_load:
                msg_err = "for the active part. It should be {} but is in fact {}"
                raise IncorrectNumberOfLoads(msg_err.format(backend.n_load, self.load_p.shape[1]))
            at_least_one = True

        if self.load_q is not None:
            if self.load_q.shape[1] != backend.n_load:
                msg_err = "for the reactive part. It should be {} but is in fact {}"
                raise IncorrectNumberOfLoads(msg_err.format(backend.n_load, self.load_q.shape[1]))
            at_least_one = True
        if self.prod_p is not None:
            if self.prod_p.shape[1] != backend.n_gen:
                msg_err = "for the active part. It should be {} but is in fact {}"
                raise IncorrectNumberOfGenerators(msg_err.format(backend.n_gen, self.prod_p.shape[1]))
            at_least_one = True

        if self.prod_v is not None:
            if self.prod_v.shape[1] != backend.n_gen:
                msg_err = "for the voltage part. It should be {} but is in fact {}"
                raise IncorrectNumberOfGenerators(msg_err.format(backend.n_gen, self.prod_v.shape[1]))
            at_least_one = True

        if self.hazards is not None:
            if self.hazards.shape[1] != backend.n_line:
                msg_err = "for the outage. It should be {} but is in fact {}"
                raise IncorrectNumberOfLines(msg_err.format(backend.n_line, self.hazards.shape[1]))
            at_least_one = True

        if self.maintenance is not None:
            if self.maintenance.shape[1] != backend.n_line:
                msg_err = "for the maintenance. It should be {} but is in fact {}"
                raise IncorrectNumberOfLines(msg_err.format(backend.n_line, self.maintenance.shape[1]))
            at_least_one = True

        if self.maintenance_time is not None:
            if self.maintenance_time.shape[1] != backend.n_line:
                msg_err = "for the maintenance times. It should be {} but is in fact {}"
                raise IncorrectNumberOfLines(msg_err.format(backend.n_line, self.maintenance_time.shape[1]))
            at_least_one = True

        if self.maintenance_duration is not None:
            if self.maintenance_duration.shape[1] != backend.n_line:
                msg_err = "for the maintenance durations. It should be {} but is in fact {}"
                raise IncorrectNumberOfLines(msg_err.format(backend.n_line, self.maintenance_duration.shape[1]))
            at_least_one = True

        if self.hazard_duration is not None:
            if self.hazard_duration.shape[1] != backend.n_line:
                msg_err = "for the hazard durations. It should be {} but is in fact {}"
                raise IncorrectNumberOfLines(msg_err.format(backend.n_line, self.hazard_duration.shape[1]))
            at_least_one = True

        if not at_least_one:
            raise ChronicsError("No files are found in directory \"{}\". If you don't want to load any chronics, use "
                                "\"ChangeNothing\" and not \"{}\" to load chronics."
                                "".format(self.path, type(self)))

        for name_arr, arr in zip(["load_q", "load_p", "prod_v", "prod_p", "maintenance", "hazards",
                                  "maintenance time", "maintenance duration", "hazard duration"],
                                 [self.load_q, self.load_p, self.prod_v, self.prod_p, self.maintenance, self.hazards,
                                  self.maintenance_time, self.maintenance_duration, self.hazard_duration]):
            if arr is not None:
                if self.chunk_size is None:
                    if arr.shape[0] != self.n_:
                        msg_err = "Array {} has not the same number of rows of load_p. The chronics cannot be loaded properly."
                        raise EnvError(msg_err.format(name_arr))

        if self.max_iter > 0:
            if self.max_iter > self.n_:
                msg_err = "Files count {} rows and you ask this episode to last at {} timestep."
                raise InsufficientData(msg_err.format(self.n_, self.max_iter))

    def next_chronics(self):
        self.current_datetime = datetime(year=2019, month=1, day=1)
        self.current_index = -1
        self.curr_iter = 0

    def get_id(self) -> str:
        return self.path

    def set_chunk_size(self, new_chunk_size):
        self.chunk_size = new_chunk_size


class GridStateFromFileWithForecasts(GridStateFromFile):
    """
    An extension of :class:`GridStateFromFile` that implements the "forecast" functionality.

    Forecast are also read from a file. For this class, only 1 forecast per timestep is read. The "forecast"
    present in the file at row $i$ is the one available at the corresponding time step, so valid for the grid state
    at the next time step.

    To have more advanced forecasts, this class could be overridden.

    Attributes
    ----------
    load_p_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the load active values.

    load_q_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the load reactive values.

    prod_p_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the generator active production setpoint.

    prod_v_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the generator voltage magnitude setpoint.

    maintenance_forecast: ``numpy.ndarray``, dtype: ``float``
        Array used to store the forecasts of the _maintenance operations.

    """
    def __init__(self, path, sep=";", time_interval=timedelta(minutes=5), max_iter=-1, chunk_size=None):
        GridStateFromFile.__init__(self, path, sep=sep, time_interval=time_interval,
                                   max_iter=max_iter, chunk_size=chunk_size)

        self.load_p_forecast = None
        self.load_q_forecast = None
        self.prod_p_forecast = None
        self.prod_v_forecast = None
        self.maintenance_forecast = None

        # for when you read data in chunk
        self._order_load_p_forecasted = None
        self._order_load_q_forecasted = None
        self._order_prod_p_forecasted = None
        self._order_prod_v_forecasted = None
        self._order_maintenance_forecasted = None
        self._data_already_in_mem = False  # says if the "main" value from the base class had to be reloaded (used for chunk)

    def _get_next_chunk_forecasted(self):
        # TODO merge this class with GridStateFromFile
        load_p = None
        load_q = None
        prod_p = None
        prod_v = None
        if self._data_chunk["load_p_forecasted"] is not None:
            load_p = next(self._data_chunk["load_p_forecasted"])
        if self._data_chunk["load_q_forecasted"] is not None:
            load_q = next(self._data_chunk["load_q_forecasted"])
        if self._data_chunk["prod_p_forecasted"] is not None:
            prod_p = next(self._data_chunk["prod_p_forecasted"])
        if self._data_chunk["prod_v_forecasted"] is not None:
            prod_v = next(self._data_chunk["prod_v_forecasted"])
        return load_p, load_q, prod_p, prod_v

    def _data_in_memory(self):
        res = super()._data_in_memory()
        self._data_already_in_mem = res
        return res

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):
        """
        The same condition as :class:`GridStateFromFile.initialize` applies also for
        :attr:`GridStateFromFileWithForecasts.load_p_forecast`,  :attr:`GridStateFromFileWithForecasts.load_q_forecast`,
        :attr:`GridStateFromFileWithForecasts.prod_p_forecast`,
        :attr:`GridStateFromFileWithForecasts.prod_v_forecast` and
        :attr:`GridStateFromFileWithForecasts.maintenance_forecast`.

        Parameters
        ----------
        See help of :func:`GridValue.initialize` for a detailed help about the _parameters.

        Returns
        -------
        ``None``

        """
        super().initialize(order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                           names_chronics_to_backend)

        load_p_iter = self._get_data("load_p_forecasted")
        load_q_iter = self._get_data("load_q_forecasted")
        prod_p_iter = self._get_data("prod_p_forecasted")
        prod_v_iter = self._get_data("prod_v_forecasted")
        hazards = None  # no hazards in forecast

        read_compressed = self._get_fileext("maintenance_forecasted")
        if read_compressed is not None:
            maintenance = pd.read_csv(os.path.join(self.path, "maintenance_forecasted{}".format(read_compressed)),
                                      sep=self.sep)
        else:
            maintenance = None

        if self.chunk_size is None:
            load_p = load_p_iter
            load_q = load_q_iter
            prod_p = prod_p_iter
            prod_v = prod_v_iter
        else:
            self._data_chunk["load_p_forecasted"] = load_p_iter
            self._data_chunk["load_q_forecasted"] = load_q_iter
            self._data_chunk["prod_p_forecasted"] = prod_p_iter
            self._data_chunk["prod_v_forecasted"] = prod_v_iter
            load_p, load_q, prod_p, prod_v = self._get_next_chunk_forecasted()

        order_backend_loads = {el: i for i, el in enumerate(order_backend_loads)}
        order_backend_prods = {el: i for i, el in enumerate(order_backend_prods)}
        order_backend_lines = {el: i for i, el in enumerate(order_backend_lines)}

        order_chronics_load_p, order_backend_load_q, \
        order_backend_prod_p, order_backend_prod_v, \
        order_backend_hazards, order_backend_maintenance \
            = self._get_orders(load_p, load_q, prod_p, prod_v, hazards, maintenance,
                               order_backend_loads, order_backend_prods, order_backend_lines)

        self._order_load_p_forecasted = np.argsort(order_chronics_load_p)
        self._order_load_q_forecasted = np.argsort(order_backend_load_q)
        self._order_prod_p_forecasted = np.argsort(order_backend_prod_p)
        self._order_prod_v_forecasted = np.argsort(order_backend_prod_v)
        self._order_maintenance_forecasted = np.argsort(order_backend_maintenance)

        self._init_attrs_forecast(load_p, load_q, prod_p, prod_v, maintenance=maintenance)

    def _init_attrs_forecast(self, load_p, load_q, prod_p, prod_v, maintenance=None):
        # TODO refactor that with _init_attrs from super()
        self.maintenance_forecast = None
        self.load_p_forecast = None
        self.load_q_forecast = None
        self.prod_p_forecast = None
        self.prod_v_forecast = None

        if load_p is not None:
            self.load_p_forecast = copy.deepcopy(load_p.values[:, self._order_load_p_forecasted])
        if load_q is not None:
            self.load_q_forecast = copy.deepcopy(load_q.values[:, self._order_load_q_forecasted])
        if prod_p is not None:
            self.prod_p_forecast = copy.deepcopy(prod_p.values[:, self._order_prod_p_forecasted])
        if prod_v is not None:
            self.prod_v_forecast = copy.deepcopy(prod_v.values[:, self._order_prod_v_forecasted])

        if maintenance is not None:
            if maintenance is not None:
                self.maintenance_forecast = copy.deepcopy(maintenance.values[:, np.argsort(self._order_maintenance)])

            # there are _maintenance and hazards only if the value in the file is not 0.
            self.maintenance_forecast = self.maintenance != 0.

    def check_validity(self, backend):
        super(GridStateFromFileWithForecasts, self).check_validity(backend)
        at_least_one = False

        if self.load_p_forecast is not None:
            if self.load_p_forecast.shape[1] != backend.n_load:
                raise IncorrectNumberOfLoads("for the active part. It should be {} but is in fact {}"
                                             "".format(backend.n_load, len(self.load_p)))
            at_least_one = True

        if self.load_q_forecast is not None:
            if self.load_q_forecast.shape[1] != backend.n_load:
                raise IncorrectNumberOfLoads("for the reactive part. It should be {} but is in fact {}"
                                             "".format(backend.n_load, len(self.load_q)))
            at_least_one = True

        if self.prod_p_forecast is not None:
            if self.prod_p_forecast.shape[1] != backend.n_gen:
                raise IncorrectNumberOfGenerators("for the active part. It should be {} but is in fact {}"
                                                  "".format(backend.n_gen, len(self.prod_p)))
            at_least_one = True

        if self.prod_v_forecast is not None:
            if self.prod_v_forecast.shape[1] != backend.n_gen:
                raise IncorrectNumberOfGenerators("for the voltage part. It should be {} but is in fact {}"
                                                  "".format(backend.n_gen, len(self.prod_v)))
            at_least_one = True

        if self.maintenance_forecast is not None:
            if self.maintenance_forecast.shape[1] != backend.n_line:
                raise IncorrectNumberOfLines("for the _maintenance. It should be {} but is in fact {}"
                                             "".format(backend.n_line, len(self.maintenance)))
            at_least_one = True

        if not at_least_one:
            raise ChronicsError("You used a class that read forecasted data, yet there is no forecasted data in"
                                "\"{}\". Please fall back to using class \"GridStateFromFile\" instead of "
                                "\"{}\"".format(self.path, type(self)))

        for name_arr, arr in zip(["load_q", "load_p", "prod_v", "prod_p", "maintenance"],
                                 [self.load_q_forecast, self.load_p_forecast, self.prod_v_forecast,
                                  self.prod_p_forecast, self.maintenance_forecast]):
            if arr is not None:
                if self.chunk_size is None:
                    if arr.shape[0] < self.n_:
                        raise EnvError("Array for forecast {}_forecasted as not the same number of rows of load_p. "
                                       "The chronics cannot be loaded properly.".format(name_arr))

    def _load_next_chunk_in_memory_forecast(self):
        # i load the next chunk as dataframes
        load_p, load_q, prod_p, prod_v = self._get_next_chunk_forecasted()
        # i put these dataframes in the right order (columns)
        self._init_attrs_forecast(load_p, load_q, prod_p, prod_v)
        # resetting the index has been done in _load_next_chunk_in_memory, or at least it should have

    def forecasts(self):
        """
        This is the major difference between :class:`GridStateFromFileWithForecasts` and :class:`GridStateFromFile`.
        It returns non empty forecasts.

        As explained in the :func:`GridValue.forecasts`, forecasts are made of list of tuple. Each tuple having
        exactly 2 elements:

          1. Is the time stamp of the forecast
          2. An :class:`grid2op.Action` representing the modification of the powergrid after the forecast.

        For this class, only the forecast of the next time step is given, and only for the injections and maintenance.

        Returns
        -------
        See :func:`GridValue.forecasts` for more information.

        """
        if not self._data_already_in_mem:
            try:
                self._load_next_chunk_in_memory_forecast()
            except StopIteration as e:
                raise e

        res = {}
        dict_ = {}
        if self.load_p_forecast is not None:
            dict_["load_p"] = 1.0 * self.load_p_forecast[self.current_index, :]
        if self.load_q_forecast is not None:
            dict_["load_q"] = 1.0 * self.load_q_forecast[self.current_index, :]
        if self.prod_p_forecast is not None:
            dict_["prod_p"] = 1.0 * self.prod_p_forecast[self.current_index, :]
        if self.prod_v_forecast is not None:
            dict_["prod_v"] = 1.0 * self.prod_v_forecast[self.current_index, :]
        if dict_:
            res["injection"] = dict_

        if self.maintenance_forecast is not None:
            res["maintenance"] = self.maintenance_forecast[self.current_index, :]

        forecast_datetime = self.current_datetime + self.time_interval
        return [(forecast_datetime, res)]

    def get_id(self) -> str:
        return self.path


class Multifolder(GridValue):
    """
    The classes :class:`GridStateFromFile` and :class:`GridStateFromFileWithForecasts` implemented the reading of a
    single folder representing a single episode.

    This class is here to "loop" between different episode. Each one being stored in a folder readable by
    :class:`GridStateFromFile` or one of its derivate (eg. :class:`GridStateFromFileWithForecasts`).

    Chronics are always read in the alpha-numeric order for this class. This means that if the folder is not modified,
    the data are always loaded in the same order, regardless of the :class:`grid2op.Backend`, :class:`grid2op.Agent` or
    :class:`grid2op.Environment`.

    Attributes
    -----------
    gridvalueClass: ``type``, optional
        Type of class used to read the data from the disk. It defaults to :class:`GridStateFromFile`.

    data: :class:`GridStateFromFile`
        Data that will be loaded and used to produced grid state and forecasted values.


    path: ``str``
        Path where the folders of the episodes are stored.

    sep: ``str``
        Columns separtor, forwarded to :attr:`Multifolder.data` when it's built at the beginning of each episode.

    subpaths: ``list``
        List of all the episode that can be "played". It's a sorted list of all the directory in
        :attr:`Multifolder.path`. Each one should contain data in a format that is readable by
        :attr:`MultiFolder.gridvalueClass`.

    id_chron_folder_current: ``int``
        Id (in :attr:`MultiFolder.subpaths`) for which data are generated in the current episode.
    """
    def __init__(self, path,
                 time_interval=timedelta(minutes=5),
                 gridvalueClass=GridStateFromFile,
                 sep=";", max_iter=-1,
                 chunk_size=None):
        GridValue.__init__(self, time_interval=time_interval, max_iter=max_iter, chunk_size=chunk_size)
        self.gridvalueClass = gridvalueClass
        self.data = None
        self.path = os.path.abspath(path)
        self.sep = sep
        try:
            self.subpaths = [os.path.join(self.path, el) for el in os.listdir(self.path)
                             if os.path.isdir(os.path.join(self.path, el))]
            self.subpaths.sort()
            self.subpaths = np.array(self.subpaths)
        except FileNotFoundError:
            raise ChronicsError("Path \"{}\" doesn't exists.".format(self.path)) from None


        if len(self.subpaths) == 0:
            raise ChronicsNotFoundError("Not chronics are found in \"{}\". Make sure there are at least "
                                        "1 chronics folder there.".format(self.path))
        # np.random.shuffle(self.subpaths)
        self.id_chron_folder_current = 0
        self.chunk_size = chunk_size

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):

        self.n_gen = len(order_backend_prods)
        self.n_load = len(order_backend_loads)
        self.n_line = len(order_backend_lines)
        self.data = self.gridvalueClass(time_interval=self.time_interval,
                                        sep=self.sep,
                                        path=self.subpaths[self.id_chron_folder_current],
                                        max_iter=self.max_iter,
                                        chunk_size=self.chunk_size)
        self.data.initialize(order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                             names_chronics_to_backend=names_chronics_to_backend)

    def done(self):
        """
        Tells the :class:`grid2op.Environment` if the episode is over.

        Returns
        -------
        res: ``bool``
            Whether or not the episode, represented by :attr:`MultiFolder.data` is over.

        """
        return self.data.done()

    def load_next(self):
        """
        Load the next data from the current episode. It loads the next time step for the current episode.

        Returns
        -------
        See the return type of  :class:`GridStateFromFile.load_next` (or of :attr:`MultiFolder.gridvalueClass` if it
        has been changed) for more information.

        """
        return self.data.load_next()

    def check_validity(self, backend):
        """
        This method check that the data loaded can be properly read and understood by the :class:`grid2op.Backend`.

        Parameters
        ----------
        backend: :class:`grid2op.Backend`
            The backend used for the experiment.

        Returns
        -------
        See the return type of  :class:`GridStateFromFile.check_validity` (or of :attr:`MultiFolder.gridvalueClass` if it
        has been changed) for more information.
        """
        return self.data.check_validity(backend)

    def forecasts(self):
        """
        The representation of the forecasted grid state(s), if any.

        Returns
        -------
        See the return type of  :class:`GridStateFromFile.forecasts` (or of :attr:`MultiFolder.gridvalueClass` if it
        has been changed) for more information.
        """
        return self.data.forecasts()

    def next_chronics(self):
        """
        Load the next episode.

        Note that :func:`MultiFolder.initialize` must be called after a call to this method has been performed. This is
        either done by the :class:`grid2op.Environemnt` or by the :class:`grid2op.Runner`.

        Returns
        -------
        ``None``

        """
        self.id_chron_folder_current += 1
        self.id_chron_folder_current %= len(self.subpaths)

    def tell_id(self, id_num):
        """
        This tells this chronics to load for the next episode.
        By default, if id_num is greater than the number of episode, it is equivalent at restarting from the first
        one: episode are played indefinitely in the same order.

        Parameters
        ----------
        id_num: ``int``
            Id of the chronics to load.

        Returns
        -------

        """
        self.id_chron_folder_current = id_num
        self.id_chron_folder_current %= len(self.subpaths)
        # print("Chronics handler: going to chronics {}".format(self.id_chron_folder_current))

    def get_id(self) -> str:
        """
        Full absolute path of the current folder used for the current episode.

        Returns
        -------
        res: ``str``
            Path from which the data are generated for the current episode.
        """
        return self.subpaths[self.id_chron_folder_current]

    def max_timestep(self):
        return self.data.max_timestep()

    def shuffle(self, shuffler):
        """
        This method is used to have a better control on the order in which the subfolder containing the episode are
        processed.

        It can focus the evaluation on one specific folder, shuffle the folders, use only a subset of them etc. See the
        examples for more information.

        Parameters
        ----------
        shuffler: ``object``
            Shuffler should be a function that is called on :attr:`MultiFolder.subpaths` that will shuffle them.
            It can also be used to remove some path if needed (see example).

        Examples
        ---------
        If you want to simply shuffle the data you can do:

        .. code-block:: python

            import numpy as np
            data = Multifolder(path=".")
            data.shuffle(shuffler=lambda x: x[np.random.choice(len(x), size=len(x), replace=False)])

        If you want to use only a subset of the path, say for example the path with index 1, 5, and 6

        .. code-block:: python

            data = Multifolder(path=".")
            data.shuffle(shuffler=lambda x: x[1, 5, 6])

        """
        self.subpaths = shuffler(self.subpaths)

    def set_chunk_size(self, new_chunk_size):
        self.chunk_size = new_chunk_size


class ChronicsHandler(RandomObject):
    """
    Represents a Chronics handler that returns a grid state.

    As stated previously, it is not recommended to make an directly an object from the class :class:`GridValue`. This
    utility will ensure that the creation of such objects are properly made.

    The types of chronics used can be specified in the :attr:`ChronicsHandler.chronicsClass` attribute.

    Attributes
    ----------
    chronicsClass: ``type``, optional
        Type of chronics that will be loaded and generated. Default is :class:`ChangeNothing` (*NB* the class, and not
        an object / instance of the class should be send here.) This should be a derived class from :class:`GridValue`.

    kwargs: ``dict``, optional
        key word arguments that will be used to build new chronics.

    max_iter: ``int``, optional
        Maximum number of iterations per episode.

    real_data: :class:`GridValue`
        An instance of type given by :attr:`ChronicsHandler.chronicsClass`.

    path: ``str`` (or None)
        path where the data are located.

    """
    def __init__(self, chronicsClass=ChangeNothing, time_interval=timedelta(minutes=5), max_iter=-1,
                 **kwargs):
        RandomObject.__init__(self)
        if not isinstance(chronicsClass, type):
            raise Grid2OpException("Parameter \"chronicsClass\" used to build the ChronicsHandler should be a type "
                                   "(a class) and not an object (an instance of a class). It is currently "
                                   "\"{}\"".format(type(chronicsClass)))

        if not issubclass(chronicsClass, GridValue):
            raise ChronicsError("ChronicsHandler: the \"chronicsClass\" argument should be a derivative of the "
                                "\"Grid2Op.GridValue\" type and not {}.".format(type(chronicsClass)))

        self.chronicsClass = chronicsClass
        self.kwargs = kwargs
        self.max_iter = max_iter

        self.path = None
        if "path" in kwargs:
            self.path = kwargs["path"]

        self.real_data = None
        try:
            self.real_data = self.chronicsClass(time_interval=time_interval, max_iter=self.max_iter,
                                                **self.kwargs)
        except TypeError:
            raise ChronicsError("Impossible to build a chronics of type {} with arguments in "
                                "{}".format(chronicsClass, self.kwargs))

    def initialize(self, order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                   names_chronics_to_backend=None):
        """
        After being loaded, this method will initialize the data.

        See definition of :func:`GridValue.initialize` for more information about this method.

        Returns
        -------
        ``None``

        """
        self.real_data.initialize(order_backend_loads, order_backend_prods, order_backend_lines, order_backend_subs,
                                  names_chronics_to_backend)

    def check_validity(self, backend):
        """
        This method ensure the data are valid and compatible with the backend used.

        See definition of :func:`GridValue.check_validity` for more information about this method.

        Returns
        -------
        ``None``

        """
        self.real_data.check_validity(backend)

    def next_time_step(self):
        """
        This method returns the modification of the powergrid at the next time step for the same episode.

        See definition of :func:`GridValue.load_next` for more information about this method.

        """
        res = self.real_data.load_next()
        return res

    def done(self):
        """
        This method returns whether or not the episode is done.

        See definition of :func:`GridValue.done` for more information about this method.

        """
        return self.real_data.done()

    def forecasts(self):
        """
        This method returns the forecasts of the data.

        See definition of :func:`GridValue.forecasts` for more information about this method.

        """
        return self.real_data.forecasts()

    def next_chronics(self):
        """
        This method is called when changing the episode after game over or after it has reached the end.

        See definition of :func:`GridValue.next_chronics` for more information about this method.

        """
        self.real_data.next_chronics()

    def tell_id(self, id_num):
        """
        This method is called when setting a given episode after game over or after it has reached the end.

        See definition of :func:`GridValue.tell_id` for more information about this method.

        """
        self.real_data.tell_id(id_num=id_num)

    def max_timestep(self):
        """
        This method gives the maximum number of time step an episode can last.

        See definition of :func:`GridValue.max_timestep` for more information about this method.

        """
        return self.real_data.max_timestep()

    def get_id(self):
        """
        This method gives a unique identifier for the current episode.

        See definition of :func:`GridValue.get_id` for more information about this method.

        """
        return self.real_data.get_id()

    def get_name(self):
        """
        This method retrieve a unique name that is used to serialize episode data on
        disk. 
        
        See definition of :mod:`EpisodeData` for more information about this method.

        """
        return str(os.path.split(self.get_id())[-1])

    def shuffle(self, shuffler):
        """
        Will attempt to shuffle the underlying data.

        Note that a call to this function might not do anything is the :func:`GridValue.shuffle` is not implemented
        for :attr:`ChronicsHandler.real_data`.

        Parameters
        ----------
        shuffler: ``object``
            Anything that is used to shuffle the data.

        """
        self.real_data.shuffle(shuffler)

    def set_chunk_size(self, new_chunk_size):
        """
        This functions allows to adjust dynamically the chunk size when reading the data.

        **NB** this function is not supported by all data generating process.

        Parameters
        ----------
        new_chunk_size: ``int`` or ``None``
            The new chunk size

        """
        if new_chunk_size is None:
            pass
        elif not isinstance(new_chunk_size, int):
            raise Grid2OpException("Impossible to read data with an non integer chunk size.")
        self.real_data.set_chunk_size(new_chunk_size)

    def set_max_iter(self, max_iter):
        """
        This function is used to set the maximum number of iterations possible before the chronics ends.

        Parameters
        ----------
        max_iter: ``int``
            The maximum number of timestep in the chronics.

        Returns
        -------

        """

        if not isinstance(max_iter, int):
            raise Grid2OpException("The maximum number of iterations possible for this chronics, before it ends.")
        self.max_iter = max_iter
        self.real_data.max_iter = max_iter
