"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToStepFunction = void 0;
const defaults = require("@aws-solutions-constructs/core");
const aws_events_rule_step_function_1 = require("@aws-solutions-constructs/aws-events-rule-step-function");
const core_1 = require("@aws-cdk/core");
const cloudtrail = require("@aws-cdk/aws-cloudtrail");
class S3ToStepFunction extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the S3ToStepFunction class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {S3ToStepFunctionProps} props - user provided props for the construct
     * @since 0.9.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.s3Bucket = defaults.buildS3Bucket(this, {
            deployBucket: props.deployBucket,
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        this.addCfnNagSuppress(this.s3Bucket);
        if (!props.hasOwnProperty('deployCloudTrail') || props.deployCloudTrail === true) {
            const trailBucket = defaults.buildS3Bucket(this, {
                deployBucket: true
            }, 'CloudTrail');
            this.addCfnNagSuppress(trailBucket);
            this.cloudtrail = new cloudtrail.Trail(this, 'S3EventsTrail', {
                bucket: trailBucket
            });
            this.cloudtrail.addS3EventSelector([{
                    bucket: this.s3Bucket
                }], {
                readWriteType: cloudtrail.ReadWriteType.ALL,
                includeManagementEvents: false
            });
        }
        let _eventRuleProps = {};
        if (props.eventRuleProps) {
            _eventRuleProps = props.eventRuleProps;
        }
        else {
            // By default the CW Events Rule will filter any 's3:PutObject' events for the S3 Bucket
            _eventRuleProps = {
                eventPattern: {
                    source: ['aws.s3'],
                    detailType: ['AWS API Call via CloudTrail'],
                    detail: {
                        eventSource: [
                            "s3.amazonaws.com"
                        ],
                        eventName: [
                            "PutObject"
                        ],
                        requestParameters: {
                            bucketName: [
                                this.s3Bucket.bucketName
                            ]
                        }
                    }
                }
            };
        }
        const eventsRuleToStepFunction = new aws_events_rule_step_function_1.EventsRuleToStepFunction(this, 'test-events-rule-step-function-stack', {
            stateMachineProps: props.stateMachineProps,
            eventRuleProps: _eventRuleProps
        });
        this.stateMachine = eventsRuleToStepFunction.stateMachine;
        this.cloudwatchAlarms = eventsRuleToStepFunction.cloudwatchAlarms;
    }
    addCfnNagSuppress(bucket) {
        // Extract the CfnBucket from the s3Bucket
        const s3BucketResource = bucket.node.findChild('Resource');
        s3BucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W51',
                        reason: `This S3 bucket Bucket does not need a bucket policy`
                    }]
            }
        };
    }
}
exports.S3ToStepFunction = S3ToStepFunction;
//# sourceMappingURL=data:application/json;base64,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