"use strict";
// Copyright (c) wangsijie
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
const base_1 = require("@jupyter-widgets/base");
const version_1 = require("./version");
// Import the CSS
require("../css/widget.css");
class TurtleModel extends base_1.DOMWidgetModel {
    defaults() {
        return Object.assign(Object.assign({}, super.defaults()), { _model_name: TurtleModel.model_name, _model_module: TurtleModel.model_module, _model_module_version: TurtleModel.model_module_version, _view_name: TurtleModel.view_name, _view_module: TurtleModel.view_module, _view_module_version: TurtleModel.view_module_version });
    }
}
exports.TurtleModel = TurtleModel;
TurtleModel.serializers = Object.assign({}, base_1.DOMWidgetModel.serializers);
TurtleModel.model_name = 'TurtleModel';
TurtleModel.model_module = version_1.MODULE_NAME;
TurtleModel.model_module_version = version_1.MODULE_VERSION;
TurtleModel.view_name = 'TurtleView'; // Set to null if no view
TurtleModel.view_module = version_1.MODULE_NAME; // Set to null if no view
TurtleModel.view_module_version = version_1.MODULE_VERSION;
class TurtleView extends base_1.DOMWidgetView {
    constructor() {
        super(...arguments);
        this.isTurtleOn = true;
        this.lastCommandId = 0;
    }
    render() {
        this.el.classList.add('turtle-widget');
        this.div = document.createElement('div');
        const isCanvasFixed = this.model.get('is_canvas_fixed');
        const canvasWidth = this.model.get('canvas_width');
        const canvasHeight = this.model.get('canvas_height');
        if (isCanvasFixed) {
            this.div.style.position = 'fixed';
        }
        else {
            this.div.style.position = 'static';
        }
        this.div.style.right = '50px';
        this.div.style.top = '140px';
        this.div.style.width = canvasWidth + 'px';
        this.div.style.height = canvasHeight + 'px';
        this.div.style.border = 'thin solid #0000FF';
        this.div.style.background = '#efefef';
        this.div.className = 'turtle_div';
        this.canvas = document.createElement('canvas');
        this.canvas.setAttribute('width', canvasWidth.toString());
        this.canvas.setAttribute('height', canvasHeight.toString());
        this.canvas.style.width = canvasWidth + 'px';
        this.canvas.style.height = canvasHeight + 'px';
        this.div.appendChild(this.canvas);
        this.el.appendChild(this.div);
        this.canvas.className = 'turtle_canvas';
        const context = this.canvas.getContext('2d');
        this.clearImageData = context.getImageData(0, 0, this.canvas.width, this.canvas.height);
        this.drawTurtle();
        this.update(); // 恢复历史情况
    }
    drawTurtle() {
        if (!this.model.get('is_turtle_on')) {
            return;
        }
        const turtleLocationX = Math.round(this.model.get('turtle_location_x'));
        const turtleLocationY = Math.round(this.model.get('turtle_location_y'));
        const turtleHeadingX = Math.round(this.model.get('turtle_heading_x'));
        const turtleHeadingY = Math.round(this.model.get('turtle_heading_y'));
        const turtleWidth = this.model.get('turtle_width');
        const turtleHeight = this.model.get('turtle_height');
        const hX = 0.5 * turtleHeight * turtleHeadingX;
        const hY = 0.5 * turtleHeight * turtleHeadingY;
        const noseX = turtleLocationX + hX;
        const noseY = turtleLocationY + hY;
        const leftLegX = turtleLocationX - 0.5 * turtleWidth * turtleHeadingY - hX;
        const leftLegY = turtleLocationY + 0.5 * turtleWidth * turtleHeadingX - hY;
        const rightLegX = turtleLocationX + 0.5 * turtleWidth * turtleHeadingY - hX;
        const rightLegY = turtleLocationY - 0.5 * turtleWidth * turtleHeadingX - hY;
        const color = this.model.get('color');
        const context = this.canvas.getContext('2d');
        context.save();
        context.setTransform(1, 0, 0, -1, this.canvas.width / 2, this.canvas.height / 2);
        // 保存当前画面（不包含turtle三角形，用于恢复）
        this.imageData = context.getImageData(0, 0, this.canvas.width, this.canvas.height);
        context.strokeStyle = color;
        context.beginPath();
        context.moveTo(noseX, noseY);
        context.lineTo(rightLegX, rightLegY);
        context.lineTo(leftLegX, leftLegY);
        context.closePath();
        context.stroke();
        context.restore();
    }
    // 去掉turtle三角形
    clearTurtle() {
        if (!this.imageData)
            return;
        this.canvas.getContext('2d').putImageData(this.imageData, 0, 0);
        this.imageData = undefined;
    }
    clear() {
        this.canvas.getContext('2d').putImageData(this.clearImageData, 0, 0);
    }
    update() {
        const commands = (this.model.get('commands') || []).filter((c) => c.id > this.lastCommandId);
        if (!commands.length) {
            return;
        }
        this.clearTurtle();
        commands.forEach((command) => {
            const { type, x, y, dx, dy, color, id, lineWidth } = command;
            if (id <= this.lastCommandId) {
                return;
            }
            if (type === 'line') {
                this.drawLine(x, y, dx, dy, color, lineWidth);
            }
            else if (type === 'clear') {
                this.clear();
            }
            else if (type === 'updateTurtle') {
            }
            else if (type === 'write') {
                const { text, align, font, x, y, color } = command;
                this.writeText(text, align, font, x, y, color);
            }
            this.lastCommandId = id;
        });
        this.drawTurtle();
        return super.update();
    }
    drawLine(x, y, dx, dy, color, lineWidth) {
        const context = this.canvas.getContext('2d');
        context.save();
        context.lineWidth = lineWidth;
        context.setTransform(1, 0, 0, -1, this.canvas.width / 2, this.canvas.height / 2);
        context.beginPath();
        context.moveTo(x, y);
        context.lineTo(dx, dy);
        context.strokeStyle = color;
        context.closePath();
        context.stroke();
        context.restore();
    }
    writeText(text, align, font, x, y, color) {
        const context = this.canvas.getContext('2d');
        context.save();
        const [family, size, weight] = font;
        context.font = `${family} ${size}px ${weight}`;
        const textX = x + this.canvas.width / 2;
        const textY = -y + this.canvas.height / 2;
        context.textAlign = align;
        context.fillStyle = color;
        context.fillText(text, textX, textY);
        context.restore();
    }
}
exports.TurtleView = TurtleView;
//# sourceMappingURL=widget.js.map