"""Provide pipeline stub fragments and project files for Spark runtimes."""

from __future__ import annotations

from textwrap import dedent
from typing import Callable, Mapping

from .. import PipelineFile, PipelineProject, PipelineStub, register_pipeline_stub


def _coerce_hint(hints: Mapping[str, object], key: str) -> str:
    value = hints.get(key)
    if value is None:
        return ""
    return str(value).strip()


def _spark_readme(hints: Mapping[str, object]) -> str:
    runtime = _coerce_hint(hints, "spark_runtime")
    workspace = _coerce_hint(hints, "spark_workspace_url")
    profile = _coerce_hint(hints, "spark_workspace_profile")
    cluster = _coerce_hint(hints, "spark_cluster")

    wizard_hints = ["Wizard hints captured during export:"]
    if runtime:
        wizard_hints.append(f"- Runtime: {runtime}")
    if workspace:
        wizard_hints.append(f"- Workspace URL: {workspace}")
    if profile:
        wizard_hints.append(f"- CLI profile: {profile}")
    if cluster:
        wizard_hints.append(f"- Cluster reference: {cluster}")
    if len(wizard_hints) == 1:
        wizard_hints.append("- (no runtime-specific hints were recorded)")

    return (
        dedent(
            """
            # Spark pipeline example

            This directory contains a full PySpark pipeline that uses the dc43
            integration helpers exported by the setup wizard. The `main.py`
            entrypoint shows how to:

            - load service backends via ``bootstrap_pipeline.load_backends``
            - build a Spark session with ``build_spark_context``
            - read contract-bound datasets with automatic casting and
              validation
            - apply a simple transformation and quality evaluation
            - publish results through the data-product backend and governance
              store

            Update the placeholder constants in ``main.py`` before running the
            pipeline and remove or replace the demo transformation with your
            business logic.
            """
        ).strip()
        + "\n\n"
        + "\n".join(wizard_hints)
        + "\n"
    )


def _spark_main_script() -> str:
    return dedent(
        '''
        #!/usr/bin/env python3
        """Run the Spark pipeline example generated by the dc43 setup wizard."""

        from __future__ import annotations

        from bootstrap_pipeline import build_spark_context, load_backends
        from spark_pipeline import governance as spark_governance
        from spark_pipeline import io as spark_io
        from spark_pipeline import quality as spark_quality
        from spark_pipeline import transformations as spark_transformations


        CONTRACT_ID = "replace-with-contract-id"
        CONTRACT_VERSION = "replace-with-contract-version"
        DATA_PRODUCT_ID = "replace-with-data-product-id"
        DATA_PRODUCT_VERSION = "replace-with-dataset-version"
        OUTPUT_PORT = "replace-with-output-port"
        SOURCE_TABLE = "replace-with-source-table"
        TARGET_TABLE = "replace-with-target-table"


        def main() -> None:
            suite = load_backends()
            context = build_spark_context(app_name="dc43-spark-example")
            spark = context.get("spark")
            if spark is None:
                raise RuntimeError(
                    "build_spark_context did not return a Spark session. "
                    "Install pyspark and ensure the runtime is available."
                )

            print("[spark] Spark session initialised:", spark)

            contract_backend = context.get("contract_backend", suite.contract)
            dq_backend = context.get("data_quality_backend", suite.data_quality)
            governance_store = context.get(
                "governance_store", suite.governance_store
            )
            governance_service = context.get(
                "governance_service", suite.governance
            )

            source_df, status = spark_io.read_contract_dataset(
                spark=spark,
                governance_service=governance_service,
                contract_id=CONTRACT_ID,
                contract_version=CONTRACT_VERSION,
                table_hint=SOURCE_TABLE,
            )
            if status is not None:
                print(
                    "[data_quality] Input status:",
                    status.status,
                    status.reason or "",
                )

            enriched_df = spark_transformations.build_output_frame(source_df)

            validation = spark_quality.run_quality_evaluation(
                dataframe=enriched_df,
                contract_backend=contract_backend,
                data_quality_backend=dq_backend,
                contract_id=CONTRACT_ID,
                contract_version=CONTRACT_VERSION,
            )

            write_validation = spark_io.write_contract_outputs(
                dataframe=enriched_df,
                governance_service=governance_service,
                contract_id=CONTRACT_ID,
                contract_version=CONTRACT_VERSION,
                dataset_id=DATA_PRODUCT_ID,
                dataset_version=DATA_PRODUCT_VERSION,
                output_port=OUTPUT_PORT,
                output_table=TARGET_TABLE,
            )

            spark_governance.publish_governance_updates(
                governance_store=governance_store,
                contract_id=CONTRACT_ID,
                contract_version=CONTRACT_VERSION,
                dataset_id=DATA_PRODUCT_ID,
                dataset_version=DATA_PRODUCT_VERSION,
                validation=write_validation,
            )

            print(
                "[done] Example pipeline completed. Replace placeholder identifiers",
                "before running in production.",
            )


        if __name__ == "__main__":
            main()
        '''
    ).strip() + "\n"


def _spark_io_module() -> str:
    return dedent(
        '''
        """Read/write helpers used by the Spark pipeline example."""

        from __future__ import annotations

        from typing import Optional, Tuple

        from dc43_integrations.spark.io import (
            GovernanceSparkReadRequest,
            GovernanceSparkWriteRequest,
            read_with_governance,
            write_with_governance,
        )
        from dc43_integrations.spark.violation_strategy import SplitWriteViolationStrategy
        from dc43_service_clients.data_quality import ValidationResult
        from dc43_service_clients.governance.client.interface import GovernanceServiceClient
        from pyspark.sql import DataFrame, SparkSession


        def read_contract_dataset(
            *,
            spark: SparkSession,
            governance_service: GovernanceServiceClient,
            contract_id: str,
            contract_version: str,
            table_hint: str | None,
        ) -> Tuple[DataFrame, Optional[ValidationResult]]:
            """Load a dataset enforced by ``contract_id`` through governance."""

            request = GovernanceSparkReadRequest(
                context={
                    "contract": {
                        "contract_id": contract_id,
                        "contract_version": contract_version,
                    }
                },
                table=table_hint or None,
            )
            df, status = read_with_governance(
                spark,
                request,
                governance_service=governance_service,
                enforce=True,
                auto_cast=True,
                return_status=True,
            )
            return df, status


        def write_contract_outputs(
            *,
            dataframe: DataFrame,
            governance_service: GovernanceServiceClient,
            contract_id: str,
            contract_version: str,
            dataset_id: str,
            dataset_version: str,
            output_port: str,
            output_table: str | None,
        ) -> ValidationResult:
            """Persist pipeline results relying solely on governance."""

            request = GovernanceSparkWriteRequest(
                context={
                    "contract": {
                        "contract_id": contract_id,
                        "contract_version": contract_version,
                    },
                    "output_binding": {
                        "data_product": dataset_id,
                        "port_name": output_port,
                    },
                    "dataset_id": dataset_id,
                    "dataset_version": dataset_version,
                },
                table=output_table or None,
                mode="overwrite",
            )

            validation, reject_status = write_with_governance(
                df=dataframe,
                request=request,
                governance_service=governance_service,
                enforce=True,
                auto_cast=True,
                return_status=True,
                violation_strategy=SplitWriteViolationStrategy(
                    valid_suffix="valid",
                    reject_suffix="reject",
                ),
            )

            if reject_status is not None:
                print(
                    "[write] Reject status:",
                    reject_status.status,
                    reject_status.reason or "",
                )

            return validation
        '''
    ).strip() + "\n"


def _spark_quality_module() -> str:
    return dedent(
        '''
        """Quality helpers for the Spark pipeline example."""

        from __future__ import annotations

        from dc43_integrations.spark.data_quality import collect_observations
        from dc43_service_clients.data_quality import ObservationPayload, ValidationResult
        from dc43_service_clients.contracts.client.interface import ContractServiceClient
        from dc43_service_clients.data_quality.client.interface import DataQualityServiceClient
        from pyspark.sql import DataFrame


        def run_quality_evaluation(
            *,
            dataframe: DataFrame,
            contract_backend: ContractServiceClient,
            data_quality_backend: DataQualityServiceClient,
            contract_id: str,
            contract_version: str,
        ) -> ValidationResult:
            """Evaluate ``dataframe`` against the stored contract and metrics."""

            contract = contract_backend.get(contract_id, contract_version)
            schema, metrics = collect_observations(dataframe, contract)
            payload = ObservationPayload(schema=schema, metrics=metrics)
            result = data_quality_backend.evaluate(contract=contract, payload=payload)
            print("[data_quality] Output validation status:", result.status)
            return result
        '''
    ).strip() + "\n"


def _spark_transformations_module() -> str:
    return dedent(
        '''
        """Demo transformations applied by the Spark example pipeline."""

        from __future__ import annotations

        from pyspark.sql import DataFrame, functions as F


        def build_output_frame(source: DataFrame) -> DataFrame:
            """Return a demo dataset derived from ``source``.

            Replace this logic with your business-specific transformations once the
            scaffold is in place.
            """

            enriched = source.withColumn(
                "processed_at",
                F.current_timestamp(),
            )
            if "status" in source.columns:
                enriched = enriched.withColumn(
                    "is_valid",
                    F.when(F.col("status") == F.lit("valid"), F.lit(True)).otherwise(False),
                )
            return enriched
        '''
    ).strip() + "\n"


def _spark_governance_module() -> str:
    return dedent(
        '''
        """Governance helpers for the Spark pipeline example."""

        from __future__ import annotations

        from typing import Any


        def publish_governance_updates(
            *,
            governance_store,
            contract_id: str,
            contract_version: str,
            dataset_id: str,
            dataset_version: str,
            validation: Any,
        ) -> None:
            """Record lineage and validation metrics in the governance store."""

            status = getattr(validation, "status", "unknown")
            metrics = getattr(validation, "metrics", {}) or {}

            governance_store.link_dataset_contract(
                dataset_id=dataset_id,
                dataset_version=dataset_version,
                contract_id=contract_id,
                contract_version=contract_version,
            )
            governance_store.record_pipeline_event(
                dataset_id=dataset_id,
                dataset_version=dataset_version,
                contract_id=contract_id,
                contract_version=contract_version,
                event={"status": status, "metrics": metrics},
            )
        '''
    ).strip() + "\n"


def _spark_project(hints: Mapping[str, object]) -> PipelineProject:
    return PipelineProject(
        root="spark_pipeline",
        entrypoint="main.py",
        files=(
            PipelineFile(path="README.md", content=_spark_readme(hints)),
            PipelineFile(
                path="__init__.py",
                content='"""Spark pipeline example package."""\n',
            ),
            PipelineFile(path="main.py", content=_spark_main_script(), executable=True),
            PipelineFile(path="io.py", content=_spark_io_module()),
            PipelineFile(path="quality.py", content=_spark_quality_module()),
            PipelineFile(path="transformations.py", content=_spark_transformations_module()),
            PipelineFile(path="governance.py", content=_spark_governance_module()),
        ),
    )


def _spark_pipeline_stub(
    *,
    hints: Mapping[str, object],
    flags: Mapping[str, bool],
    json_literal: Callable[[object | None], str],
) -> PipelineStub:
    runtime_hint = json_literal(hints.get("spark_runtime"))
    workspace_hint = json_literal(hints.get("spark_workspace_url"))
    profile_hint = json_literal(hints.get("spark_workspace_profile"))
    cluster_hint = json_literal(hints.get("spark_cluster"))
    source_table = json_literal("replace-with-source-table")
    target_table = json_literal("replace-with-target-table")
    contract_id = json_literal("replace-with-contract-id")
    contract_version = json_literal("replace-with-contract-version")
    dataset_id = json_literal("replace-with-data-product-id")
    dataset_version = json_literal("replace-with-dataset-version")
    output_port = json_literal("replace-with-output-port")

    main_lines = (
        "    if integration == 'spark':",
        "        context = build_spark_context(app_name=\"dc43-pipeline-example\")",
        "        spark = context.get('spark')",
        "        if spark is None:",
        "            raise RuntimeError('Spark session was not initialised by build_spark_context')",
        "        print(\"[spark] Spark session initialised:\", spark)",
        f"        runtime_hint = {runtime_hint}",
        "        if runtime_hint:",
        "            print(\"[spark] Runtime configured in setup:\", runtime_hint)",
        f"        workspace_hint = {workspace_hint}",
        "        if workspace_hint:",
        "            print(\"[spark] Workspace URL:\", workspace_hint)",
        f"        profile_hint = {profile_hint}",
        "        if profile_hint:",
        "            print(\"[spark] CLI profile:\", profile_hint)",
        f"        cluster_hint = {cluster_hint}",
        "        if cluster_hint:",
        "            print(\"[spark] Cluster reference:\", cluster_hint)",
        "        contract_backend = context.get('contract_backend', contract_backend)",
        "        data_quality_backend = context.get('data_quality_backend', data_quality_backend)",
        "        governance_store = context.get('governance_store', governance_store)",
        "        governance_service = context.get('governance_service', governance_service)",
        f"        source_table = {source_table}",
        f"        target_table = {target_table}",
        f"        contract_id = {contract_id}",
        f"        contract_version = {contract_version}",
        f"        dataset_id = {dataset_id}",
        f"        dataset_version = {dataset_version}",
        f"        output_port = {output_port}",
        "        source_df, input_status = spark_io.read_contract_dataset(",
        "            spark=spark,",
        "            governance_service=governance_service,",
        "            contract_id=contract_id,",
        "            contract_version=contract_version,",
        "            table_hint=source_table,",
        "        )",
        "        if input_status:",
        "            print(\"[spark] Input status:\", input_status.status, input_status.reason or \"\")",
        "        enriched_df = spark_transformations.build_output_frame(source_df)",
        "        validation = spark_quality.run_quality_evaluation(",
        "            dataframe=enriched_df,",
        "            contract_backend=contract_backend,",
        "            data_quality_backend=data_quality_backend,",
        "            contract_id=contract_id,",
        "            contract_version=contract_version,",
        "        )",
        "        write_validation = spark_io.write_contract_outputs(",
        "            dataframe=enriched_df,",
        "            governance_service=governance_service,",
        "            contract_id=contract_id,",
        "            contract_version=contract_version,",
        "            dataset_id=dataset_id,",
        "            dataset_version=dataset_version,",
        "            output_port=output_port,",
        "            output_table=target_table,",
        "        )",
        "        spark_governance.publish_governance_updates(",
        "            governance_store=governance_store,",
        "            contract_id=contract_id,",
        "            contract_version=contract_version,",
        "            dataset_id=dataset_id,",
        "            dataset_version=dataset_version,",
        "            validation=write_validation,",
        "        )",
    )

    tail_lines = (
        "    if integration == 'spark':",
        "        print(\"[spark] Explore examples/spark_pipeline for modular helpers.\")",
    )

    return PipelineStub(
        bootstrap_imports=("build_spark_context",),
        additional_imports=(
            "from spark_pipeline import governance as spark_governance",
            "from spark_pipeline import io as spark_io",
            "from spark_pipeline import quality as spark_quality",
            "from spark_pipeline import transformations as spark_transformations",
        ),
        main_lines=main_lines,
        tail_lines=tail_lines,
        project=_spark_project(hints),
    )


register_pipeline_stub("spark", _spark_pipeline_stub)


__all__ = ["_spark_pipeline_stub"]

