import importlib
import re
from typing import Type, Tuple, Optional, TYPE_CHECKING

from pydantic import ValidationError, Extra, ConfigError, ExtraError, MissingError
from pydantic.error_wrappers import ErrorWrapper
from pydantic.typing import ForwardRef
from pydantic.utils import ROOT_KEY, GetterDict

from fox_orm.internal.const import EXCLUDE_KEYS

if TYPE_CHECKING:
    from fox_orm.model import OrmModel
    from pydantic.types import ModelOrDc  # pylint: disable=no-name-in-module,ungrouped-imports
    from pydantic.typing import DictStrAny, SetStr  # pylint: disable=no-name-in-module,ungrouped-imports


def full_import(name):
    mod, attr = name.rsplit('.', 1)
    mod = importlib.import_module(mod)
    mod = getattr(mod, attr)
    return mod


class OptionalAwaitable:
    def __init__(self, func, *args, **kwargs):
        self.func = func
        self.args = args
        self.kwargs = kwargs

    def __await__(self):
        return self.func(*self.args, **self.kwargs).__await__()


_missing = object()


# pylint: disable=fixme
# TODO: add tests
def validate_model(
        model: Type['OrmModel'], input_data: 'DictStrAny', cls: 'ModelOrDc' = None
) -> Tuple['DictStrAny', 'SetStr', Optional[ValidationError]]:
    # pylint: disable=R,E
    """
    validate data against a model.
    """
    values = {}
    errors = []
    # input_data names, possibly alias
    names_used = set()
    # field names, never aliases
    fields_set = set()
    config = model.__config__
    check_extra = config.extra is not Extra.ignore
    cls_ = cls or model

    for validator in model.__pre_root_validators__:
        try:
            input_data = validator(cls_, input_data)
        except (ValueError, TypeError, AssertionError) as exc:
            return {}, set(), ValidationError([ErrorWrapper(exc, loc=ROOT_KEY)], cls_)

    for name, field in model.__fields__.items():
        if name in EXCLUDE_KEYS and name != model.__pkey_name__:
            values[name] = field.default
            continue
        if field.type_.__class__ == ForwardRef:
            raise ConfigError(
                f'field "{field.name}" not yet prepared so type is still a ForwardRef, '
                f'you might need to call {cls_.__name__}.update_forward_refs().'
            )

        value = input_data.get(field.alias, _missing)
        using_name = False
        if value is _missing and config.allow_population_by_field_name and field.alt_alias:
            value = input_data.get(field.name, _missing)
            using_name = True

        if value is _missing:
            if field.required:
                errors.append(ErrorWrapper(MissingError(), loc=field.alias))
                continue

            value = field.get_default()

            if not config.validate_all and not field.validate_always:
                values[name] = value
                continue
        else:
            fields_set.add(name)
            if check_extra:
                names_used.add(field.name if using_name else field.alias)

        if not issubclass(field.type_, str) and value == 'null':
            value = None

        v_, errors_ = field.validate(value, values, loc=field.alias, cls=cls_)
        if isinstance(errors_, ErrorWrapper):
            errors.append(errors_)
        elif isinstance(errors_, list):
            errors.extend(errors_)
        else:
            values[name] = v_

    if check_extra:
        if isinstance(input_data, GetterDict):
            extra = input_data.extra_keys() - names_used
        else:
            extra = input_data.keys() - names_used
        if extra:
            fields_set |= extra
            if config.extra is Extra.allow:
                for f in extra:
                    values[f] = input_data[f]
            else:
                for f in sorted(extra):
                    errors.append(ErrorWrapper(ExtraError(), loc=f))

    for skip_on_failure, validator in model.__post_root_validators__:
        if skip_on_failure and errors:
            continue
        try:
            values = validator(cls_, values)
        except (ValueError, TypeError, AssertionError) as exc:
            errors.append(ErrorWrapper(exc, loc=ROOT_KEY))

    if errors:
        return values, fields_set, ValidationError(errors, cls_)
    else:
        return values, fields_set, None


# noinspection PyMethodOverriding,PyArgumentList,PyUnresolvedReferences,PyPep8Naming
# pylint: disable=invalid-name,no-member
class class_or_instancemethod(classmethod):
    def __get__(self, instance, type_):
        descr_get = super().__get__ if instance is None else self.__func__.__get__
        return descr_get(instance, type_)


CAMEL_TO_SNAKE_PAT1 = re.compile(r'(.)([A-Z][a-z]+)')
CAMEL_TO_SNAKE_PAT2 = re.compile(r'([a-z0-9])([A-Z])')


def camel_to_snake(name):
    name = CAMEL_TO_SNAKE_PAT1.sub(r'\1_\2', name)
    return CAMEL_TO_SNAKE_PAT2.sub(r'\1_\2', name).lower()


class NonInstantiable:
    def __new__(cls, *args, **kwargs):
        raise TypeError(f'Object of type {cls.__qualname__} is not instantiable')
