# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['motioneye_client']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.8.3,<4.0.0']

setup_kwargs = {
    'name': 'motioneye-client',
    'version': '0.3.13',
    'description': 'motionEye client library Python Package',
    'long_description': '<img src="https://github.com/dermotduffy/hass-motioneye/blob/main/images/motioneye.png?raw=true"\n     alt="motionEye icon"\n     width="15%"\n     align="right"\n     style="float: right; margin: 10px 0px 20px 20px;" />\n\n[![PyPi](https://img.shields.io/pypi/v/motioneye-client.svg?style=flat-square)](https://pypi.org/project/motioneye-client/)\n[![PyPi](https://img.shields.io/pypi/pyversions/motioneye-client.svg?style=flat-square)](https://pypi.org/project/motioneye-client/)\n[![Build Status](https://img.shields.io/github/actions/workflow/status/dermotduffy/motioneye-client/build.yaml?branch=main)](https://github.com/dermotduffy/motioneye-client/actions/workflows/build.yaml)\n[![Test Coverage](https://img.shields.io/codecov/c/gh/dermotduffy/motioneye-client?style=flat-square)](https://codecov.io/gh/dermotduffy/motioneye-client)\n[![License](https://img.shields.io/github/license/dermotduffy/hass-motioneye.svg?style=flat-square)](LICENSE)\n[![BuyMeCoffee](https://img.shields.io/badge/buy%20me%20a%20coffee-donate-yellow.svg?style=flat-square)](https://www.buymeacoffee.com/dermotdu)\n\n# motionEye Client\n\nA simple async API client for [motionEye](https://github.com/ccrisan/motioneye).\n\n## Constructor arguments\n\nThe following arguments may be passed to the `MotionEyeClient` constructor:\n\n| Argument              | Type                    | Default                     | Description                         |\n| --------------------- | ----------------------- | --------------------------- | ----------------------------------- |\n| url                   | `str`                   | URL of the motionEye server |\n| admin_username        | `str`                   | admin                       | The motionEye admin username        |\n| admin_password        | `str`                   | ""                          | The motionEye admin password        |\n| surveillance_username | `str`                   | user                        | The motionEye surveillance username |\n| surveillance_password | `str`                   | ""                          | The motionEye surveillance password |\n| session               | `aiohttp.ClientSession` | None                        | Optional aiohttp session to use     |\n\nThis client needs both `admin` and `surveillance` passwords in order to interact with\nthe API (which generally require the `admin` user), as well as prepare the URLs for\ndata streaming (which require the `surveillance` user).\n\n## Primary Client Methods\n\nAll async calls start with `async_`, and return the JSON response from the server (if any).\n\n### async_client_login\n\nLogin to the motionEye server. Not actually necessary, but useful for verifying credentials.\n### async_client_close\n\nClose the client session. Always returns True.\n\n### async_get_manifest\n\nGet the motionEye server manifest (e.g. server version number).\n\n### async_get_server_config\n\nGet the main motionEye server config.\n\n### async_get_cameras\n\nGet the listing of all cameras.\n\n### async_get_camera\n\nGet the configuration of a single camera. Takes an integer `camera_id` argument.\n\n### async_set_camera\n\nSet the configuration of a single camera. Takes an integer `camera_id` argument, and a\ndictionary of the same format as returned by `async_get_camera`.\n\n### async_action\n\nPerform a motionEye action on a camera. Takes an integer `camera_id` argument and an\naction string.\n\nCommon actions include `snapshot`, `record_start` and `record_stop`. motionEye also\nsupports other user configurable actions which may be called in this manner. See\n[Action Buttons](https://github.com/ccrisan/motioneye/wiki/Action-Buttons) for more details.\n\n### async_get_movies\n\nGet a list of recorded movies for a given `camera_id`. Accepts a `prefix` argument that\ngives a path prefix to list (does not recurse).\n\n### async_get_images\n\nGet a list of saved images for a given `camera_id`. Accepts a `prefix` argument that\ngives a path prefix to list (does not recurse).\n\n## Convenience Methods\n\n### is_camera_streaming\n\nConvenience method to take a camera dictionary (returned by `async_get_camera` or\n`async_get_cameras`) and return True if the camera has video stream enabled.\n\n### get_camera_stream_url\n\nConvenience method to take a camera dictionary (returned by `async_get_camera` or\n`async_get_cameras`) and return the string URL of the streamed content (which can be\nopened separately). This extracts the hostname out of the motionEye URL and attaches the\nstreaming port to it -- depending on the configuration this may not necessarily lead to\nan accessible URL (e.g. in the use of motionEye behind a reverse proxy).\n\nWill raise [MotionEyeClientURLParseError](#MotionEyeClientURLParseError) if the hostname\ncannot be extracted from the motionEye server URL.\n\n### get_camera_snapshot_url\n\nConvenience method to take a camera dictionary (returned by `async_get_camera` or\n`async_get_cameras`) and return the string URL of a single still frame.\n\n### get_movie_url\n\nConvenience method to take a camera id and the path to a saved movie, and return a link\nto playback the movie. Takes a `preview` argument that if `True` returns a URL to a thumbnail.\n\n### get_image_url\n\nConvenience method to take a camera id and the path to a saved image, and return a link\nto that image. Takes a `preview` argument that if `True` returns a URL to a thumbnail.\n\n### is_file_type_image / is_file_type_movie\n\nDetermine if a given file_type `int` (from a web hook callback) represents an image or a movie respectively.\n\n## Context Manager\n\nThe client may be used in as a context manager, which will automatically close the\nsession.\n\n```python\nasync with client.MotionEyeClient("http://localhost:8765", ) as mec:\n    if not mec:\n        return\n    ...\n````\n\n## Exceptions / Errors\n\n### MotionEyeClientError\n\nA generic base class -- all motionEye client exceptions inherit from this.\n\n### MotionEyeClientInvalidAuthError\n\nInvalid authentication detected during a request.\n\n### MotionEyeClientConnectionError\n\nConnected failed to given URL.\n\n<a name="MotionEyeClientURLParseError"></a>\n### MotionEyeClientURLParseError\n\nUnable to parse the required URL.\n\n\n### MotionEyeClientPathError\n\nUnable to parse a path.\n\n\n### MotionEyeClientRequestError\n\nA request failed in some other undefined way.\n\n## Simple Example\n\n```python\n#!/usr/bin/env python\n"""Client test for motionEye."""\nimport asyncio\n\nfrom motioneye_client.client import MotionEyeClient\n\n\nasync def query_motioneye_server() -> None:\n    """Test the motionEye client."""\n    async with MotionEyeClient("http://localhost:8765") as client:\n        if not client:\n            return\n\n        manifest = await client.async_get_manifest()\n        print(f"Manifest: {manifest}")\n\n        camera_list = await client.async_get_cameras()\n        print(f"Cameras: {camera_list}")\n\n\nasyncio.get_event_loop().run_until_complete(query_motioneye_server())\n```\n\n## Building / Testing\n\nThis library is built using [Poetry](https://python-poetry.org/).\n\nBuilding:\n\n```bash\n$ poetry build\n```\n\nTesting:\n```bash\n$ poetry run pytest\n```\n\n## Updating Dependencies\n\nUpdating dependencies:\n\n```bash\n$ poetry update\n```\n\nExporting to `requirements.txt` and `requirements_dev.txt`:\n\n```bash\n$ poetry export --without-hashes -o requirements.txt\n$ poetry export --without-hashes --dev -o requirements_dev.txt\n```\n',
    'author': 'Dermot Duffy',
    'author_email': 'dermot.duffy@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/dermotduffy/motioneye-client',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
