"""Configuration management for Backcraft CLI"""
import json
from pathlib import Path
from typing import Optional

CONFIG_DIR = Path.home() / ".backcraft"
CONFIG_FILE = CONFIG_DIR / "config.json"


class Config:
    """Manage Backcraft CLI configuration"""

    def __init__(self):
        self.server_url: Optional[str] = None
        self.api_key: Optional[str] = None
        self.load()

    def load(self):
        """Load configuration from file"""
        if CONFIG_FILE.exists():
            with open(CONFIG_FILE, "r") as f:
                data = json.load(f)
                self.server_url = data.get("server_url")
                self.api_key = data.get("api_key")

    def save(self):
        """Save configuration to file"""
        CONFIG_DIR.mkdir(parents=True, exist_ok=True)
        with open(CONFIG_FILE, "w") as f:
            json.dump({
                "server_url": self.server_url,
                "api_key": self.api_key
            }, f, indent=2)

    def is_configured(self) -> bool:
        """Check if CLI is properly configured"""
        return bool(self.server_url and self.api_key)

    def set_server(self, url: str):
        """Set server URL"""
        # Clean up URL
        self.server_url = url.rstrip("/")
        self.save()

    def set_api_key(self, key: str):
        """Set Anthropic API key"""
        self.api_key = key
        self.save()

    def clear(self):
        """Clear configuration"""
        self.server_url = None
        self.api_key = None
        if CONFIG_FILE.exists():
            CONFIG_FILE.unlink()


# Singleton instance
config = Config()
