"""API client for Backcraft server"""
import time
from typing import Dict, Any, Optional
import requests
from rich.console import Console
from rich.progress import Progress, SpinnerColumn, TextColumn, BarColumn

from .config import config

console = Console()


class BackcraftAPIError(Exception):
    """Backcraft API error"""
    pass


class BackcraftAPI:
    """Client for Backcraft API"""

    def __init__(self):
        if not config.is_configured():
            raise BackcraftAPIError(
                "CLI not configured. Run 'backcraft configure' first."
            )

        self.base_url = config.server_url
        self.api_key = config.api_key

    def _get_headers(self) -> Dict[str, str]:
        """Get request headers with API key"""
        return {
            "Content-Type": "application/json",
            "X-API-Key": self.api_key,  # Clé Anthropic de l'utilisateur
        }

    def _get(self, endpoint: str, **kwargs) -> Dict[str, Any]:
        """Generic GET request"""
        try:
            response = requests.get(
                f"{self.base_url}{endpoint}",
                headers=self._get_headers(),
                timeout=kwargs.get('timeout', 10),
                **{k: v for k, v in kwargs.items() if k != 'timeout'}
            )
            response.raise_for_status()
            return response.json()
        except requests.RequestException as e:
            raise BackcraftAPIError(f"GET {endpoint} failed: {e}")

    def _post(self, endpoint: str, data: Dict[str, Any], **kwargs) -> Dict[str, Any]:
        """Generic POST request"""
        try:
            response = requests.post(
                f"{self.base_url}{endpoint}",
                json=data,
                headers=self._get_headers(),
                timeout=kwargs.get('timeout', 30),
                **{k: v for k, v in kwargs.items() if k != 'timeout'}
            )
            response.raise_for_status()
            return response.json()
        except requests.RequestException as e:
            raise BackcraftAPIError(f"POST {endpoint} failed: {e}")

    def check_health(self) -> bool:
        """Check if server is healthy"""
        try:
            response = requests.get(
                f"{self.base_url}/health",
                timeout=5
            )
            return response.status_code == 200
        except requests.RequestException:
            return False

    def create_generation(self, name: str, idea: str, **kwargs) -> str:
        """Create a new generation"""
        payload = {
            "name": name,
            "idea": idea,
            "api_key": self.api_key,  # Envoyer la clé de l'utilisateur
            **kwargs
        }

        try:
            response = requests.post(
                f"{self.base_url}/api/generate",
                json=payload,
                headers=self._get_headers(),
                timeout=30
            )
            response.raise_for_status()
            data = response.json()
            return data.get("generation_id")

        except requests.RequestException as e:
            raise BackcraftAPIError(f"Failed to create generation: {e}")

    def get_generation_status(self, generation_id: str) -> Dict[str, Any]:
        """Get generation status"""
        try:
            response = requests.get(
                f"{self.base_url}/api/generations/{generation_id}",
                headers=self._get_headers(),
                timeout=10
            )
            response.raise_for_status()
            return response.json()

        except requests.RequestException as e:
            raise BackcraftAPIError(f"Failed to get status: {e}")

    def download_generation(self, generation_id: str, output_path: str) -> bool:
        """Download generated project"""
        try:
            response = requests.get(
                f"{self.base_url}/api/generations/{generation_id}/download",
                headers=self._get_headers(),
                stream=True,
                timeout=30
            )
            response.raise_for_status()

            with open(output_path, "wb") as f:
                for chunk in response.iter_content(chunk_size=8192):
                    f.write(chunk)

            return True

        except requests.RequestException as e:
            raise BackcraftAPIError(f"Failed to download: {e}")

    def wait_for_completion(self, generation_id: str, poll_interval: int = 5) -> Dict[str, Any]:
        """Wait for generation to complete with progress display"""

        with Progress(
            SpinnerColumn(),
            TextColumn("[progress.description]{task.description}"),
            BarColumn(),
            console=console
        ) as progress:

            task = progress.add_task("[cyan]Generating project...", total=100)

            while True:
                status = self.get_generation_status(generation_id)

                # Update progress
                percent = status.get("progress", 0)
                step = status.get("current_step", "Processing...")
                progress.update(task, completed=percent, description=f"[cyan]{step}")

                # Check if done
                if status.get("status") == "completed":
                    progress.update(task, completed=100)
                    return status

                if status.get("status") == "failed":
                    error = status.get("error", "Unknown error")
                    raise BackcraftAPIError(f"Generation failed: {error}")

                time.sleep(poll_interval)
