"""Backcraft CLI - AI-powered Python project generator"""
import click
from rich.console import Console
from rich.panel import Panel
from rich.prompt import Prompt

from .config import config
from .api import BackcraftAPI, BackcraftAPIError

console = Console()

__version__ = "2.0.0"


@click.group()
@click.version_option(__version__)
def cli():
    """🔥 Backcraft CLI - Generate Python projects with AI"""
    pass


@cli.command()
def configure():
    """Configure Backcraft CLI"""
    console.print("\n[bold cyan]🔧 Backcraft Configuration[/bold cyan]\n")

    # Server URL
    current_server = config.server_url or "http://localhost:8000"
    server_url = Prompt.ask(
        "[bold]Server URL[/bold]",
        default=current_server
    )
    config.set_server(server_url)
    console.print(f"[green]✓ Server set to: {server_url}[/green]")

    # API Key
    console.print("\n[cyan]Your Anthropic API key is needed for generations.[/cyan]")
    console.print("[dim]Get your key at: https://console.anthropic.com/settings/keys[/dim]\n")

    api_key = Prompt.ask(
        "[bold]Anthropic API Key[/bold]",
        password=True
    )
    config.set_api_key(api_key)
    console.print("[green]✓ API key saved[/green]")

    # Test connection
    console.print("\n[cyan]Testing connection...[/cyan]")
    try:
        api = BackcraftAPI()
        if api.check_health():
            console.print("[green]✓ Server is healthy![/green]")
        else:
            console.print("[yellow]⚠ Server unreachable. Check URL.[/yellow]")
    except Exception as e:
        console.print(f"[yellow]⚠ Could not connect: {e}[/yellow]")

    console.print("\n[bold green]✅ Configuration complete![/bold green]\n")


@cli.command()
@click.argument('name')
@click.option('--idea', required=True, help='Project description in natural language')
@click.option('--output', '-o', default='.', help='Output directory')
def create(name: str, idea: str, output: str):
    """Create a new project"""

    if not config.is_configured():
        console.print("[red]❌ Not configured. Run 'backcraft configure' first.[/red]")
        return

    console.print(f"\n[bold cyan]🚀 Creating project: {name}[/bold cyan]\n")
    console.print(f"[dim]Idea: {idea}[/dim]\n")

    try:
        api = BackcraftAPI()

        # Create generation
        console.print("[cyan]📤 Sending request to server...[/cyan]")
        generation_id = api.create_generation(name, idea)
        console.print(f"[green]✓ Generation started (ID: {generation_id})[/green]\n")

        # Wait for completion
        result = api.wait_for_completion(generation_id)

        # Download
        console.print("\n[cyan]📥 Downloading project...[/cyan]")
        output_file = f"{output}/{name}-installer.py"
        api.download_generation(generation_id, output_file)

        # Success
        console.print(Panel(
            f"[bold green]✅ Project generated successfully![/bold green]\n\n"
            f"Files: {result.get('files_count', 'N/A')}\n"
            f"Quality Score: {result.get('quality_score', 'N/A')}\n"
            f"Time: {result.get('generation_time', 'N/A')}s\n\n"
            f"[cyan]Installer saved to: {output_file}[/cyan]\n"
            f"[dim]Run: python {output_file}[/dim]",
            title="Success",
            border_style="green"
        ))

    except BackcraftAPIError as e:
        console.print(f"[red]❌ Error: {e}[/red]")
    except Exception as e:
        console.print(f"[red]❌ Unexpected error: {e}[/red]")


@cli.command()
@click.argument('generation_id')
def status(generation_id: str):
    """Check generation status"""

    if not config.is_configured():
        console.print("[red]❌ Not configured. Run 'backcraft configure' first.[/red]")
        return

    try:
        api = BackcraftAPI()
        result = api.get_generation_status(generation_id)

        console.print(Panel(
            f"Status: [bold]{result.get('status')}[/bold]\n"
            f"Progress: {result.get('progress', 0)}%\n"
            f"Step: {result.get('current_step', 'N/A')}",
            title=f"Generation {generation_id}",
            border_style="cyan"
        ))

    except BackcraftAPIError as e:
        console.print(f"[red]❌ Error: {e}[/red]")


@cli.command()
def info():
    """Show current configuration"""
    console.print("\n[bold cyan]Backcraft CLI Configuration[/bold cyan]\n")

    if config.is_configured():
        console.print(f"[green]✓ Configured[/green]")
        console.print(f"Server: {config.server_url}")
        console.print(f"API Key: {'*' * 20}{config.api_key[-8:]}")
    else:
        console.print("[yellow]Not configured. Run 'backcraft configure'[/yellow]")

    console.print()


@cli.command()
@click.argument('name')
@click.option('--output', '-o', default='.', help='Output directory')
def chat(name: str, output: str):
    """💬 Interactive mode - AI asks questions to understand your project better"""

    if not config.is_configured():
        console.print("[red]❌ Not configured. Run 'backcraft configure' first.[/red]")
        return

    from rich.panel import Panel
    from rich.markdown import Markdown
    from rich.live import Live
    from rich.spinner import Spinner
    from rich.prompt import Prompt as RichPrompt
    from rich.table import Table
    import time

    try:
        api = BackcraftAPI()

        # Step 1: Choose language FIRST (no welcome message yet)
        console.print("\n[bold cyan]🌍 Choose your language / Choisissez votre langue[/bold cyan]\n")
        console.print("  1. 🇫🇷 Français")
        console.print("  2. 🇬🇧 English")
        console.print("  3. 🇩🇪 Deutsch")
        console.print("  4. 🇪🇸 Español")
        console.print("  5. 🇮🇹 Italiano")
        console.print("  6. 🇵🇹 Português")
        console.print()

        language_choice = RichPrompt.ask(
            "[bold green]→[/bold green]",
            choices=["1", "2", "3", "4", "5", "6"],
            default="2",
            show_choices=False
        )

        language_map = {
            "1": "fr",
            "2": "en",
            "3": "de",
            "4": "es",
            "5": "it",
            "6": "pt"
        }
        language = language_map[language_choice]

        # Now translate everything based on language
        translations = {
            "fr": {
                "welcome_title": "🤖 Discutons",
                "welcome_text": "💬 Génération Interactive de Projet\n\nProjet: {name}\n\nJe vais vous poser quelques questions pour mieux comprendre vos besoins.\nCela m'aide à générer exactement ce que vous voulez !",
                "step_level": "🎯 Étape 2/3 : Choisissez votre niveau technique",
                "level_beginner": "👶 Débutant - Je ne suis pas technique, utilisez un langage simple",
                "level_pro": "🚀 Professionnel - Je connais la technique, vous pouvez utiliser des termes techniques",
                "level_prompt": "→",
                "selected_language": "✓ Langue",
                "selected_level": "✓ Niveau",
                "level_beginner_short": "👶 Débutant",
                "level_pro_short": "🚀 Professionnel",
                "step_idea": "🎯 Étape 3/3 : Parlez-moi de votre projet",
                "idea_question": "Que voulez-vous construire ?",
                "idea_required": "❌ L'idée du projet est requise",
                "analyzing": "🔍 Analyse de votre idée...",
                "understood": "✓ Voici ce que j'ai compris :",
                "questions_intro": "💭 Maintenant, laissez-moi vous poser quelques questions...",
                "question": "Question",
                "helps_determine": "💡 Cela permet de déterminer",
                "your_answer": "Votre réponse",
                "provide_answer": "Veuillez fournir une réponse",
                "done": "✅ Parfait ! J'ai toutes les informations dont j'ai besoin.",
                "building_understanding": "Construction de la compréhension du projet...",
                "final_understanding": "🎯 Compréhension Finale",
                "domain": "Domaine",
                "purpose": "Objectif",
                "target_users": "Utilisateurs Cibles",
                "key_entities": "Entités Clés",
                "similar_to": "Similaire à",
                "num_models": "Nombre de modèles",
                "num_features": "Nombre de features",
                "complexity": "Complexité",
                "stats_title": "📊 Statistiques",
                "detected_features": "Features détectées",
                "and_more": "... et {count} de plus",
                "features_added": "✓ {count} features ajoutées",
                "modify_features": "Voulez-vous ajouter/modifier des features ?",
                "add_features_prompt": "Entrez des features supplémentaires (séparées par des virgules, ou appuyez sur Entrée pour continuer)",
                "ready_generate": "Prêt à générer ?",
                "cancelled": "👋 Génération annulée",
                "starting": "🚀 Démarrage de la génération...",
                "started": "✓ Génération démarrée (ID: {id})",
                "features": "Fonctionnalités",
                "downloading": "📥 Téléchargement du projet...",
                "success_title": "🎉 Succès",
                "success_text": "✅ Projet généré avec succès !\n\nFichiers: {files}\nScore de Qualité: {score}\nTemps: {time}s\nFonctionnalités: {features}\n\nInstalleur sauvegardé : {output}\nExécutez : python {output}",
            },
            "en": {
                "welcome_title": "🤖 Let's Chat",
                "welcome_text": "💬 Interactive Project Generation\n\nProject: {name}\n\nI'll ask you a few questions to better understand your needs.\nThis helps me generate exactly what you want!",
                "step_level": "🎯 Step 2/3: Choose your technical level",
                "level_beginner": "👶 Beginner - I'm not technical, use simple language",
                "level_pro": "🚀 Professional - I know tech, you can use technical terms",
                "level_prompt": "→",
                "selected_language": "✓ Language",
                "selected_level": "✓ Level",
                "level_beginner_short": "👶 Beginner",
                "level_pro_short": "🚀 Professional",
                "step_idea": "🎯 Step 3/3: Tell me about your project",
                "idea_question": "What do you want to build?",
                "idea_required": "❌ Project idea is required",
                "analyzing": "🔍 Analyzing your idea...",
                "understood": "✓ Here's what I understood:",
                "questions_intro": "💭 Now, let me ask you a few questions...",
                "question": "Question",
                "helps_determine": "💡 This helps determine",
                "your_answer": "Your answer",
                "provide_answer": "Please provide an answer",
                "done": "✅ Perfect! I have all the information I need.",
                "building_understanding": "Building project understanding...",
                "final_understanding": "🎯 Final Understanding",
                "domain": "Domain",
                "purpose": "Purpose",
                "target_users": "Target Users",
                "key_entities": "Key Entities",
                "similar_to": "Similar To",
                "num_models": "Number of models",
                "num_features": "Number of features",
                "complexity": "Complexity",
                "stats_title": "📊 Statistics",
                "detected_features": "Detected features",
                "and_more": "... and {count} more",
                "features_added": "✓ {count} features added",
                "modify_features": "Would you like to add/modify features?",
                "add_features_prompt": "Enter additional features (comma-separated, or press Enter to continue)",
                "ready_generate": "Ready to generate?",
                "cancelled": "👋 Generation cancelled",
                "starting": "🚀 Starting generation...",
                "started": "✓ Generation started (ID: {id})",
                "features": "Features",
                "downloading": "📥 Downloading project...",
                "success_title": "🎉 Success",
                "success_text": "✅ Project generated successfully!\n\nFiles: {files}\nQuality Score: {score}\nTime: {time}s\nFeatures: {features}\n\nInstaller saved to: {output}\nRun: python {output}",
            },
            "de": {
                "welcome_title": "🤖 Lass uns reden",
                "welcome_text": "💬 Interaktive Projektgenerierung\n\nProjekt: {name}\n\nIch werde Ihnen ein paar Fragen stellen, um Ihre Bedürfnisse besser zu verstehen.\nDas hilft mir, genau das zu generieren, was Sie wollen!",
                "step_level": "🎯 Schritt 2/3: Wählen Sie Ihr technisches Niveau",
                "level_beginner": "👶 Anfänger - Ich bin nicht technisch, verwenden Sie einfache Sprache",
                "level_pro": "🚀 Profi - Ich kenne mich aus, Sie können Fachbegriffe verwenden",
                "level_prompt": "→",
                "selected_language": "✓ Sprache",
                "selected_level": "✓ Niveau",
                "level_beginner_short": "👶 Anfänger",
                "level_pro_short": "🚀 Profi",
                "step_idea": "🎯 Schritt 3/3: Erzählen Sie mir von Ihrem Projekt",
                "idea_question": "Was möchten Sie bauen?",
                "idea_required": "❌ Projektidee ist erforderlich",
                "analyzing": "🔍 Analysiere Ihre Idee...",
                "understood": "✓ Das habe ich verstanden:",
                "questions_intro": "💭 Jetzt lassen Sie mich ein paar Fragen stellen...",
                "question": "Frage",
                "helps_determine": "💡 Dies hilft zu bestimmen",
                "your_answer": "Ihre Antwort",
                "provide_answer": "Bitte geben Sie eine Antwort",
                "done": "✅ Perfekt! Ich habe alle Informationen, die ich brauche.",
                "building_understanding": "Projektverständnis aufbauen...",
                "final_understanding": "🎯 Endgültiges Verständnis",
                "domain": "Bereich",
                "purpose": "Zweck",
                "target_users": "Zielbenutzer",
                "key_entities": "Schlüsselentitäten",
                "similar_to": "Ähnlich wie",
                "num_models": "Anzahl der Modelle",
                "num_features": "Anzahl der Funktionen",
                "complexity": "Komplexität",
                "stats_title": "📊 Statistiken",
                "detected_features": "Erkannte Funktionen",
                "and_more": "... und {count} weitere",
                "features_added": "✓ {count} Funktionen hinzugefügt",
                "modify_features": "Möchten Sie Funktionen hinzufügen/ändern?",
                "add_features_prompt": "Geben Sie zusätzliche Funktionen ein (kommagetrennt, oder drücken Sie Enter zum Fortfahren)",
                "ready_generate": "Bereit zu generieren?",
                "cancelled": "👋 Generierung abgebrochen",
                "starting": "🚀 Generierung starten...",
                "started": "✓ Generierung gestartet (ID: {id})",
                "features": "Funktionen",
                "downloading": "📥 Projekt herunterladen...",
                "success_title": "🎉 Erfolg",
                "success_text": "✅ Projekt erfolgreich generiert!\n\nDateien: {files}\nQualitätsbewertung: {score}\nZeit: {time}s\nFunktionen: {features}\n\nInstaller gespeichert in: {output}\nAusführen: python {output}",
            },
            "es": {
                "welcome_title": "🤖 Hablemos",
                "welcome_text": "💬 Generación Interactiva de Proyectos\n\nProyecto: {name}\n\nTe haré algunas preguntas para entender mejor tus necesidades.\n¡Esto me ayuda a generar exactamente lo que quieres!",
                "step_level": "🎯 Paso 2/3: Elige tu nivel técnico",
                "level_beginner": "👶 Principiante - No soy técnico, usa lenguaje simple",
                "level_pro": "🚀 Profesional - Conozco la técnica, puedes usar términos técnicos",
                "level_prompt": "→",
                "selected_language": "✓ Idioma",
                "selected_level": "✓ Nivel",
                "level_beginner_short": "👶 Principiante",
                "level_pro_short": "🚀 Profesional",
                "step_idea": "🎯 Paso 3/3: Cuéntame sobre tu proyecto",
                "idea_question": "¿Qué quieres construir?",
                "idea_required": "❌ Se requiere la idea del proyecto",
                "analyzing": "🔍 Analizando tu idea...",
                "understood": "✓ Esto es lo que entendí:",
                "questions_intro": "💭 Ahora, déjame hacerte algunas preguntas...",
                "question": "Pregunta",
                "helps_determine": "💡 Esto ayuda a determinar",
                "your_answer": "Tu respuesta",
                "provide_answer": "Por favor proporciona una respuesta",
                "done": "✅ ¡Perfecto! Tengo toda la información que necesito.",
                "building_understanding": "Construyendo comprensión del proyecto...",
                "final_understanding": "🎯 Comprensión Final",
                "domain": "Dominio",
                "purpose": "Propósito",
                "target_users": "Usuarios Objetivo",
                "key_entities": "Entidades Clave",
                "similar_to": "Similar a",
                "num_models": "Número de modelos",
                "num_features": "Número de características",
                "complexity": "Complejidad",
                "stats_title": "📊 Estadísticas",
                "detected_features": "Características detectadas",
                "and_more": "... y {count} más",
                "features_added": "✓ {count} características añadidas",
                "modify_features": "¿Deseas agregar/modificar características?",
                "add_features_prompt": "Ingresa características adicionales (separadas por comas, o presiona Enter para continuar)",
                "ready_generate": "¿Listo para generar?",
                "cancelled": "👋 Generación cancelada",
                "starting": "🚀 Iniciando generación...",
                "started": "✓ Generación iniciada (ID: {id})",
                "features": "Características",
                "downloading": "📥 Descargando proyecto...",
                "success_title": "🎉 Éxito",
                "success_text": "✅ ¡Proyecto generado exitosamente!\n\nArchivos: {files}\nPuntuación de Calidad: {score}\nTiempo: {time}s\nCaracterísticas: {features}\n\nInstalador guardado en: {output}\nEjecutar: python {output}",
            },
            "it": {
                "welcome_title": "🤖 Parliamo",
                "welcome_text": "💬 Generazione Interattiva di Progetti\n\nProgetto: {name}\n\nTi farò alcune domande per capire meglio le tue esigenze.\nQuesto mi aiuta a generare esattamente quello che vuoi!",
                "step_level": "🎯 Passo 2/3: Scegli il tuo livello tecnico",
                "level_beginner": "👶 Principiante - Non sono tecnico, usa un linguaggio semplice",
                "level_pro": "🚀 Professionista - Conosco la tecnica, puoi usare termini tecnici",
                "level_prompt": "→",
                "selected_language": "✓ Lingua",
                "selected_level": "✓ Livello",
                "level_beginner_short": "👶 Principiante",
                "level_pro_short": "🚀 Professionista",
                "step_idea": "🎯 Passo 3/3: Parlami del tuo progetto",
                "idea_question": "Cosa vuoi costruire?",
                "idea_required": "❌ L'idea del progetto è richiesta",
                "analyzing": "🔍 Analizzando la tua idea...",
                "understood": "✓ Ecco cosa ho capito:",
                "questions_intro": "💭 Ora, lascia che ti faccia alcune domande...",
                "question": "Domanda",
                "helps_determine": "💡 Questo aiuta a determinare",
                "your_answer": "La tua risposta",
                "provide_answer": "Per favore fornisci una risposta",
                "done": "✅ Perfetto! Ho tutte le informazioni di cui ho bisogno.",
                "building_understanding": "Costruendo la comprensione del progetto...",
                "final_understanding": "🎯 Comprensione Finale",
                "domain": "Dominio",
                "purpose": "Scopo",
                "target_users": "Utenti Target",
                "key_entities": "Entità Chiave",
                "similar_to": "Simile a",
                "num_models": "Numero di modelli",
                "num_features": "Numero di funzionalità",
                "complexity": "Complessità",
                "stats_title": "📊 Statistiche",
                "detected_features": "Funzionalità rilevate",
                "and_more": "... e {count} in più",
                "features_added": "✓ {count} funzionalità aggiunte",
                "modify_features": "Vuoi aggiungere/modificare funzionalità?",
                "add_features_prompt": "Inserisci funzionalità aggiuntive (separate da virgole, o premi Enter per continuare)",
                "ready_generate": "Pronto a generare?",
                "cancelled": "👋 Generazione annullata",
                "starting": "🚀 Avvio generazione...",
                "started": "✓ Generazione avviata (ID: {id})",
                "features": "Funzionalità",
                "downloading": "📥 Download progetto...",
                "success_title": "🎉 Successo",
                "success_text": "✅ Progetto generato con successo!\n\nFile: {files}\nPunteggio di Qualità: {score}\nTempo: {time}s\nFunzionalità: {features}\n\nInstaller salvato in: {output}\nEsegui: python {output}",
            },
            "pt": {
                "welcome_title": "🤖 Vamos Conversar",
                "welcome_text": "💬 Geração Interativa de Projetos\n\nProjeto: {name}\n\nVou fazer algumas perguntas para entender melhor suas necessidades.\nIsso me ajuda a gerar exatamente o que você quer!",
                "step_level": "🎯 Passo 2/3: Escolha seu nível técnico",
                "level_beginner": "👶 Iniciante - Não sou técnico, use linguagem simples",
                "level_pro": "🚀 Profissional - Conheço técnica, pode usar termos técnicos",
                "level_prompt": "→",
                "selected_language": "✓ Idioma",
                "selected_level": "✓ Nível",
                "level_beginner_short": "👶 Iniciante",
                "level_pro_short": "🚀 Profissional",
                "step_idea": "🎯 Passo 3/3: Fale-me sobre seu projeto",
                "idea_question": "O que você quer construir?",
                "idea_required": "❌ A ideia do projeto é necessária",
                "analyzing": "🔍 Analisando sua ideia...",
                "understood": "✓ Aqui está o que eu entendi:",
                "questions_intro": "💭 Agora, deixe-me fazer algumas perguntas...",
                "question": "Pergunta",
                "helps_determine": "💡 Isto ajuda a determinar",
                "your_answer": "Sua resposta",
                "provide_answer": "Por favor forneça uma resposta",
                "done": "✅ Perfeito! Tenho todas as informações que preciso.",
                "building_understanding": "Construindo compreensão do projeto...",
                "final_understanding": "🎯 Compreensão Final",
                "domain": "Domínio",
                "purpose": "Propósito",
                "target_users": "Usuários Alvo",
                "key_entities": "Entidades Chave",
                "similar_to": "Similar a",
                "num_models": "Número de modelos",
                "num_features": "Número de funcionalidades",
                "complexity": "Complexidade",
                "stats_title": "📊 Estatísticas",
                "detected_features": "Funcionalidades detectadas",
                "and_more": "... e mais {count}",
                "features_added": "✓ {count} funcionalidades adicionadas",
                "modify_features": "Deseja adicionar/modificar funcionalidades?",
                "add_features_prompt": "Digite funcionalidades adicionais (separadas por vírgulas, ou pressione Enter para continuar)",
                "ready_generate": "Pronto para gerar?",
                "cancelled": "👋 Geração cancelada",
                "starting": "🚀 Iniciando geração...",
                "started": "✓ Geração iniciada (ID: {id})",
                "features": "Funcionalidades",
                "downloading": "📥 Baixando projeto...",
                "success_title": "🎉 Sucesso",
                "success_text": "✅ Projeto gerado com sucesso!\n\nArquivos: {files}\nPontuação de Qualidade: {score}\nTempo: {time}s\nFuncionalidades: {features}\n\nInstalador salvo em: {output}\nExecutar: python {output}",
            }
        }

        t = translations[language]

        # Welcome message (NOW translated)
        console.print()
        console.print(Panel(
            t["welcome_text"].format(name=name),
            border_style="cyan",
            title=t["welcome_title"]
        ))

        # Step 2: Choose technical level
        console.print(f"\n[bold cyan]{t['step_level']}[/bold cyan]\n")
        console.print(f"  1. {t['level_beginner']}")
        console.print(f"  2. {t['level_pro']}")
        console.print()

        level_choice = RichPrompt.ask(
            f"[bold green]{t['level_prompt']}[/bold green]",
            choices=["1", "2"],
            default="1",
            show_choices=False
        )

        technical_level = "beginner" if level_choice == "1" else "professional"

        # Show selection
        level_display = "👶 Beginner" if technical_level == "beginner" else "🚀 Professional"
        lang_names = {"fr": "🇫🇷 Français", "en": "🇬🇧 English", "de": "🇩🇪 Deutsch", "es": "🇪🇸 Español", "it": "🇮🇹 Italiano", "pt": "🇵🇹 Português"}
        console.print(f"\n[green]✓ Language: {lang_names[language]}[/green]")
        console.print(f"[green]✓ Level: {level_display}[/green]\n")

        # Step 3: Ask for project idea (in selected language)
        idea_prompts = {
            "fr": {
                "title": "📝 Parlez-moi de votre projet",
                "question": "Que voulez-vous construire ?"
            },
            "en": {
                "title": "📝 Tell me about your project",
                "question": "What do you want to build?"
            },
            "de": {
                "title": "📝 Erzählen Sie mir von Ihrem Projekt",
                "question": "Was möchten Sie bauen?"
            },
            "es": {
                "title": "📝 Cuéntame sobre tu proyecto",
                "question": "¿Qué quieres construir?"
            },
            "it": {
                "title": "📝 Parlami del tuo progetto",
                "question": "Cosa vuoi costruire?"
            },
            "pt": {
                "title": "📝 Fale-me sobre seu projeto",
                "question": "O que você quer construir?"
            }
        }

        idea_prompt = idea_prompts.get(language, idea_prompts["en"])
        console.print(f"\n[bold cyan]{t['step_idea']}[/bold cyan]\n")
        idea = RichPrompt.ask(f"[bold green]{t['idea_question']}[/bold green]")

        if not idea or not idea.strip():
            error_msgs = {
                "fr": "❌ L'idée du projet est requise",
                "en": "❌ Project idea is required",
                "de": "❌ Projektidee ist erforderlich",
                "es": "❌ Se requiere la idea del proyecto",
                "it": "❌ L'idea del progetto è richiesta",
                "pt": "❌ A ideia do projeto é necessária"
            }
            console.print(f"[red]{error_msgs.get(language, error_msgs['en'])}[/red]")
            return

        console.print()

        # Start conversation
        analyzing_msgs = {
            "fr": "🔍 Analyse de votre idée...",
            "en": "🔍 Analyzing your idea...",
            "de": "🔍 Analysiere Ihre Idee...",
            "es": "🔍 Analizando tu idea...",
            "it": "🔍 Analizzando la tua idea...",
            "pt": "🔍 Analisando sua ideia..."
        }
        console.print(f"[cyan]{analyzing_msgs.get(language, analyzing_msgs['en'])}[/cyan]")

        with console.status("[bold cyan]Thinking...", spinner="dots"):
            response = api._post("/api/chat/start", {
                "name": name,
                "idea": idea,
                "api_key": config.api_key,
                "language": language,
                "technical_level": technical_level
            })

        session_id = response["session_id"]

        # Show initial understanding
        if response.get("understanding"):
            u = response["understanding"]
            console.print(f"\n[bold green]{t['understood']}[/bold green]")
            table = Table(show_header=False, box=None, padding=(0, 2))
            table.add_row(f"📌 {t['domain']}:", f"[yellow]{u.get('domain', 'N/A')}[/yellow]")
            table.add_row(f"🎯 {t['purpose']}:", f"[white]{u.get('purpose', 'N/A')}[/white]")
            table.add_row(f"👥 {t['target_users']}:", f"[white]{u.get('target_users', 'N/A')}[/white]")
            console.print(table)

        console.print(f"\n[bold yellow]{t['questions_intro']}[/bold yellow]\n")
        time.sleep(0.5)

        # Question loop
        while not response.get("done"):
            question_num = response.get("question_num", 0)
            total = response.get("total_questions", 5)
            question = response.get("question")
            why = response.get("why")

            if not question:
                break

            # Display question with nice formatting
            console.print(f"[bold cyan]{t['question']} {question_num}/{total}:[/bold cyan]")
            console.print(f"  [bold white]{question}[/bold white]")
            console.print(f"  [dim]{t['helps_determine']}: {why}[/dim]\n")

            # Get answer
            answer = RichPrompt.ask(f"  [bold green]{t['your_answer']}[/bold green]")

            if not answer or not answer.strip():
                console.print(f"  [yellow]{t['provide_answer']}[/yellow]\n")
                continue

            console.print()

            # Send answer and get next question
            with console.status(f"[bold cyan]{t['building_understanding']}", spinner="dots"):
                response = api._post(f"/api/chat/{session_id}/answer", {
                    "answer": answer
                })

        # Done with questions
        console.print(f"[bold green]{t['done']}[/bold green]\n")

        # Show final understanding with stats and features
        with console.status(f"[bold cyan]{t['building_understanding']}", spinner="dots"):
            understanding = api._get(f"/api/chat/{session_id}/understanding")

        stats = understanding.get('stats', {})
        features = understanding.get('features', [])

        # Display final understanding with stats
        console.print(Panel(
            f"[bold]{t['domain']}:[/bold] {understanding.get('domain')}\n"
            f"[bold]{t['purpose']}:[/bold] {understanding.get('purpose')}\n"
            f"[bold]{t['target_users']}:[/bold] {understanding.get('target_users')}\n"
            f"[bold]{t['key_entities']}:[/bold] {', '.join(understanding.get('entities', {}).keys())}\n"
            f"[bold]{t['similar_to']}:[/bold] {', '.join(understanding.get('similar_to', []))}\n\n"
            f"[bold cyan]{t['stats_title']}:[/bold cyan]\n"
            f"[bold]{t['num_models']}:[/bold] {stats.get('num_models', 0)}\n"
            f"[bold]{t['num_features']}:[/bold] {stats.get('num_features', 0)}\n"
            f"[bold]{t['complexity']}:[/bold] {stats.get('complexity', 'medium').upper()}",
            title=t['final_understanding'],
            border_style="green"
        ))

        # Show detected features
        console.print(f"\n[bold cyan]{t['detected_features']}:[/bold cyan]")
        for i, feature in enumerate(features[:10], 1):
            console.print(f"  {i}. {feature}")
        if len(features) > 10:
            console.print(f"  [dim]{t['and_more'].format(count=len(features) - 10)}[/dim]")

        # Ask if user wants to modify features
        console.print()
        modify = RichPrompt.ask(
            f"[bold cyan]{t['modify_features']}[/bold cyan]",
            choices=["yes", "no"],
            default="no",
            show_choices=False
        )

        if modify.lower() == "yes":
            additional = RichPrompt.ask(f"[bold green]{t['add_features_prompt']}[/bold green]")
            if additional.strip():
                new_features = [f.strip() for f in additional.split(',') if f.strip()]
                features.extend(new_features)
                console.print(f"[green]{t['features_added'].format(count=len(new_features))}[/green]")

        # Ask to proceed
        console.print()
        proceed = RichPrompt.ask(
            f"[bold cyan]{t['ready_generate']}[/bold cyan]",
            choices=["yes", "no"],
            default="yes",
            show_choices=False
        )

        if proceed.lower() != "yes":
            console.print(f"[yellow]{t['cancelled']}[/yellow]")
            return

        # Start generation
        console.print("\n[bold cyan]🚀 Starting generation...[/bold cyan]")

        gen_response = api._post(f"/api/chat/{session_id}/generate", {
            "project_type": "api",  # Could make this configurable
            "custom_features": features if modify.lower() == "yes" else None  # Pass custom features if modified
        })

        generation_id = gen_response["generation_id"]
        features = gen_response.get("features", [])

        console.print(f"[green]✓ Generation started (ID: {generation_id})[/green]")
        console.print(f"[dim]Features: {', '.join(features[:5])}{'...' if len(features) > 5 else ''}[/dim]\n")

        # Wait for completion (reuse existing logic)
        result = api.wait_for_completion(generation_id)

        # Download
        console.print("\n[cyan]📥 Downloading project...[/cyan]")
        output_file = f"{output}/{name}-installer.py"
        api.download_generation(generation_id, output_file)

        # Success
        console.print(Panel(
            f"[bold green]✅ Project generated successfully![/bold green]\n\n"
            f"Files: {result.get('files_count', 'N/A')}\n"
            f"Quality Score: {result.get('quality_score', 'N/A')}\n"
            f"Time: {result.get('generation_time', 'N/A')}s\n"
            f"Features: {len(features)}\n\n"
            f"[cyan]Installer saved to: {output_file}[/cyan]\n"
            f"[dim]Run: python {output_file}[/dim]",
            title="🎉 Success",
            border_style="green"
        ))

    except BackcraftAPIError as e:
        console.print(f"[red]❌ Error: {e}[/red]")
    except Exception as e:
        console.print(f"[red]❌ Unexpected error: {e}[/red]")


def main():
    """Entry point"""
    cli()


if __name__ == "__main__":
    main()
