"""
-------------------------------------------------------------------------------
 Name:        OdfEdit
 Purpose:     Application helping to edit in plain text an organ definition file (ODF)
              made for the GrandOrgue application (see github.com/GrandOrgue/grandorgue)
              and allowing to convert a Hauptwerk ODF into GrandOrgue ODF
              Implemented with Python 3.1x
              Tested in Windows and Ubuntu
              It is contains following classes :
                   C_LOGS to manage the logs generated by the various fonctions of the application
                   C_AUDIO_PLAYER to play audio files (wave or wavpack format)
                   C_ODF_DATA_CHECK to check the syntax and the consistency of the edited ODF data
                   C_ODF_DATA to manage the edited ODF data
                   C_ODF_HW2GO to do the conversion from a Hauptwerk ODF to a GrandOrgue ODF
                   C_GUI_NOTEBOOK to manage the notebook widget and its tabs content in the GUI of the application
                   C_GUI to manage the graphical user interface of the application
                   ToolTip to display a tool tip on a GUI widget
                   AskUserChooseListItems to manage a dialog window asking the user to choose items in a list
                   AskUserAnswerQuestion to manage a dialog window asking the user to answer to a question by pressing buttons
                   AskUserEnterString to manage a dialog window asking the user to enter a string

 Author:      Eric Turpault (France, Châtellerault)
 Copyright:   open source
 Licence:     MIT licence, please share the modifications with the author

 The considered GrandOrgue ODF syntax is :
    [object_uid]  ; comment, section name (object_uid, object unique id) can contain only alphanumeric characters
    ; comment line, empty lines are ignored
    attribute1=value1  ; comment, attribute field can contain only alphanumeric or '_' characters
    attribute2=value2

 The new panel format is detected if the Panel000 section is present and contains the attribute NumberOfGUIElements

 Versions history :
   v1.0 - 15 April 2022 - initial version
   v1.1 - 16 April 2022 - minor changes to be Linux compatible, minor GUI fixes
   v1.2 - 27 April 2022 - some GUI behavior improvements, minor improvements in the help and the objects checks
   v1.3 - 19 May   2022 - data management improvement, change in the way to define the parent-child relations between the objects,
                          completed some attributes values maximum check, added a tab to search a string in the whole ODF
   v2.0 - 23 Dec.  2022 - fix made in the function check_object_Manual around the key_type checks
                          fix made in the function check_attribute_value to not change out of range integer value and better check HTML color code
                          use the PIL library instead of Tk to check the sample set images sizes
                          first implementation of the Hauptwerk to GrandOrgue ODF conversion feature
   v2.1 - 22 Dec.  2023 - HW2GO : fix for files path separator management in various OS
                          HW2GO : get the actual files path/name/extension case from the HW sample set instead of from the HW ODF
                          HW2GO : some design changes without functional impact
                          HW2GO : added general sound stops (blower, bells, ...) and stop action noise support
   v2.2 - 12 Jan. 2023  - HW2GO : improved and more robust way to detect Stop / Coupler / Switch objects to build
                          HW2GO : fix of issues observed with some newly tested HW sample sets
                          HW2GO : added support of HW continuous control and enclosure objects, converted into GO enclosure objects
                          GUI : graphical user interface rework with resizable width for lists and notebook areas
                          GUI : Organ object placed systematically at the top of the objects list
   v2.3 - 08 April 2023 - fix for help text not loaded with Windows OdfEdit.exe (fix in file Help.txt due to characters 0x81 not supported by unicode format)
                          add the support of comments at the end of the lines in the ODF
                          add the support of the separator / in the files path in the ODF
                          improved display and selection of parents/children objects of the selected object
                          improved several behaviors in the GUI management
                          ODF saving in a file places the objects by alphabetical UID order, but Header and Organ which are placed in first position
                          new buttons to add, link to parents/children or rename an object, with automatical update of the object referencing
                             and the total number in other objects, new panel format only is supported
                          HW2GO : visual elements of the main panel are now defined in Panel999Element999 objects
                          HW2GO : Switches have by default StoreInDivisional=Y, StoreInGeneral=Y, GCState=0
                          HW2GO : added a menu item checkbox to ask to convert to GO ranks the HW ranks not used by OdfEdit
   v2.4 - 30 May 2023   - objects having child(ren) cannot be deleted with the button Delete
                          added a menu item checkbox to select the file format to use when saving an ODF (ISO_8859_1 or UTF-8 BOM)
                          added a menu item checkbox to disable the automatic objects tree expand on object selection
                          added a menu item checkbox to enable the wave based tremulants convertion from HW to GO
                          added a menu item checkbox to enable the unused ranks convertion from HW to GO
                          configuration data of the application (last ODF folder, options of the menu) are saved in a file OdfEdit.cfg
                          improvements made in the objects list/tree behavior on object selection or change
                          the parents/children of the selected object can be selected and edited in the central list (above the text editor)
                          a double-click on a parent/child object makes it the selected object in the list/tree
                          ctrl+s keys permits to save changes in the edited object and to save changed data in the ODF
                          escape key permits to close the pop-up windows permitting to select parent/children objects
                          added the possiblity to search a text inside the loaded HW ODF
                          HW2GO : removed the attribute AcceptRetuning=N in ranks of pipes
                          HW2GO : pitch tuning attribute used for pipes which the sample has a different native frequency
                          HW2GO : convert the conditional switches (switches which the state depends on the state of other switches)
                          HW2GO : convert the synthetized and wave based tremulants
                          HW2GO : the stop and coupler objects are numeroted according to the manual number to which they belong to
                          several bugs fixing as at each release
   v2.5 - 26 Aug. 2023  - manage drag&drop of one object (type : Coupler, Enclosure, PanelElement, PanelImage, Rank, Stop, Tremulant)
                            between the objects lists or tree, in order to move it under another parent object
                          drag&drop with Control key pressed adds a copy of the dragged object as child of the object on which the drop has been done
                          Ctrl-a in object edition or logs text boxes permits to selected all the text (needed for Linux)
                          paste in object edition text box replaces the current selected text if any (needed for Linux)
                          mouse double click in object edition text box selects the text until the equal or brackets character
                          added the possibility to search in the ODF in a selected range (ODF, selected object, children of the selected object)
                          added the possibility to search in the ODF with a regular expression
                          added the possibility to replace the text found by the search
                          "object" renamed to "section" in the GUI
                          HW2GO : check that the mouse or text rectangle doesn't exceed the image size of a switch or label
                          HW2GO : unused HW noise ranks are converted to GO ranks if the convertion option is enabled in the menu
                          HW2GO : add in a switch or setter the text of a label which is overlapping it
                          HW2GO : rework of the way to calculate the rank related attributes in the Stop object (pipes stop)
                          HW2GO : update the organ / pipes pitch tuning calculation
                          HW2GO : manage the case where there are more than 99 stops in a manual
                          HW2GO : manage the case where one enclosure is controlled from several panel element objects
                          HW2GO : add the Loop/ReleaseCrossfadeLength attributes conversion in the ranks (clipped to GO max values waiting for GO 3.13.0)
                          HW2GO : use TextBreakWidth=0 instead of DispLabelText= to have no text displayed in a button or enclosure
                          several minor improvements and bugs fixing
   v2.6 - 13 Oct. 2023  - added a Viewer tab to view or play the selected file in the editor
                          HW2GO : Loop/ReleaseCrossfadeLength maximum value set at 3000 (needs GO 3.13.0-1)
   v2.7 - 07 Nov. 2023  - manage in the viewer the pipe borrowing format REF:xx:xx:xx + minor improvements
                          HW2GO : manage the keyboards noises conversion
                          HW2GO : add in manual sections the reference to the associated stop/coupler/tremulant switches
                          HW2GO : place the wave tremmed samples in the same rank as the not tremmed samples if selected in a new menu option
                          HW2GO : noise samples gains lower than -5 are set at -5
                          several minor improvements and bugs fixing
   v2.8 - 18 Nov. 2023  - improvement of the section drag&drop visual behavior
                          drag&drop permits to reorder between them sections of the same type
                          removal of the menu option asking to auto expand sections tree on section selection. Double clicking on an item of sections list makes expand the sections tree to show the selected section.
                          HW2GO : convert alternate screen layouts into GO panels if selected in a new dedicated menu option
                          some minor improvements and bugs fixing
   v2.9 - 01 Feb. 2024  - display the number of found occurrences in Search results
                          when a Panel section is deleted all his children Panelxxxx sections are deleted as well
                          any section can be deleted, a warning message is displayed if the section to delete has children sections
                          add the check of attributes Pipe999Attack999LoopCrossfadeLength, Pipe999Attack999ReleaseCrossfadeLength, Pipe999Release999ReleaseCrossfadeLength
                          fix made in the check of PanelElement section
                          add a menu item permitting to sort in the selected section the references to other sections
                          rework of the function doing references sorting in a section
                          sections referenced in General and Divisional sections are no more considered as their children
                          PanelElement sections are now child instead of parent of the section they are refering to
                          use sounddevice library instead of simpleaudio library, which reduces OdfEdit Linux binary by 21MB. WavPack files cannot be played anymore in the viewer
                          HW2GO : use the new attributes Pipe999Attack999LoopCrossfadeLength and Pipe999Release999ReleaseCrossfadeLength introduced in GO 3.14.0
                          HW2GO : set the attribute MIDIInputNumber in objects Manual and Enclosure (if real enclosure)
                          HW2GO : insert the conversion date in the generated ODF (header and organ comment)
                          HW2GO : improve the Tremulant to WindchestGroup linkage method
                          HW2GO : fix for negative pipe gain not being converted
                          HW2GO : fix an issue in enclosures linkage parsing
                          some minor improvements and bugs fixing
   v2.10 - 17 March 2024- add the check of attributes HasIndependentRelease and Pipe999HasIndependentRelease
                          when a section is deleted, rearrange the ID of other sections of the same type so that the IDs are increasing continuously without gap
                            this is applicable only for following sections types : Enclosure, General, Panel, PanelElement, PanelImage, Rank, Switch, Tremulant, WindchestGroup
                          improve the metadata presentation in the wave file viewer
                          HW2GO : added a menu option to not convert keys noises
                          HW2GO : added two menu options to make pipes pitch correction when needed, based on sample metadata or file name MIDI note
                          HW2GO : fix wrong manual/stop attributes when a manual does not start with MIDI note 36
                          HW2GO : manage unusual key to pipe MIDI note mapping
                          HW2GO : convert the pipe attack minimum velocity attribute
                          HW2GO : do not convert attack samples which the selectability is based on a continuous control state
                          HW2GO : manage the case where a windchest has more than one volume control slider
                          HW2GO : manage the case where keys noises are defined for several audio channels
                          HW2GO : disable temporarily the CrossfadeLength attributes conversion waiting for GO 3.14.0 to be released officially
                          several code and logic improvements
   v2.11 - 14 April 2024- integrate in the help the changes made in GO 3.14.0 help
                          add the check of attributes added/modified in GO 3.14.0
                          on application start restore the last window size and position, and sub-areas dimensions
                          HW2GO : improve the way to convert tremmed samples when they have to be placed in separate ranks
                          HW2GO : restore the usage of the attributes Pipe999Attack999LoopCrossfadeLength and Pipe999Release999ReleaseCrossfadeLength
                          HW2GO : manage the case where a file path contains // instead of /
                          HW2GO : set GCState=-1 in the Stop sections containing keys noises, to let them engaged after a General Cancel push
                          HW2GO : redesign of the way to identify the HW objects to convert in GO Stop / Coupler / Tremulant / Switch / Setter objects
                          HW2GO : usage of the new attribute HasIndependentRelease to group inside a unique Stop section attack and release samples of each noise and audio channel
                          some minor improvements and bugs fixing
   v2.12 - 31 May 2024  - new graphical interface colors, unified between main window and pop-up windows
                          clickable widgets have their background color changing on mouse cursor hovering
                          add a button to open an ODF from the recently opened ODFs list
                          add a menu item to clear the recently opened ODFs list
                          menu stays opened when clicking on a checkable option
                          add a check button to make case insensitive the search in the ODF
                          Collapse and Expand buttons act on the selected section of the sections tree and no longer on the entire tree
                          The viewer can render files (images, audio samples) selected in the HW sections
                          HW2GO : redesign of the way to build the switches network
                          HW2GO : optimization in the HW sections tree building and switches network, reducing significantly the conversion time
                          HW2GO : rework the way to map the HW Keyboard / Division to GO Manual sections
                          HW2GO : added in the header of the GO ODF information about the chosen conversion options
                          HW2GO : use a pitch specification method code in each sample to know if their pitch has to be read from the metadata of the sample file
                          HW2GO : in ranks, set an harmonic number value at rank level, and at pipe level set only if it is different from the one at rank level
                          HW2GO : conversion of HW Combination into GO General/Divisional and master capture (SET) and general cancel
                          HW2GO : set the Enclosure attribute AmpMinimumLevel at 0 instead of 1 as GO 3.14.3 does not mute anymore pipes after enclosure set to 0 and then reopened
   v2.13 - 23 June 2024 - The search is case insensitive by default (instead of case sensitive by default before)
                          Pressing the Delete key of the keyboard does the same action as clicking on the Delete button
                          Adding a button to clear the search&replace data
                          HW2GO : fixing an issue with the attribute DefaultToEngaged=N not set when required, preventing the generated ODF to be loaded in GO
                          HW2GO : improving the manual keys building to support any kind of first and last key note (sharp notes included) when the keys aspect is defined at octave level
                          HW2GO : converting the tremmed samples in the way they are defined in Piotr Grabowski sample sets (placed in second pipes layers and not in alternate ranks as done by Sonus Paradisi)
   v2.14 - 22 Sept 2024 - The Delete key pressing deletes the selected section only if the focus is in a sections list or tree
                          Implementation of a manual / stop / rank compass extension feature (accessible from a menu item)

TO DO :
    when a panel object is selected, display an overview of this panel in the viewer (display the position of the images defined in this panel)

-------------------------------------------------------------------------------
"""

APP_VERSION = 'v2.14'
RELEASE_DATE = 'September 22nd 2024'

DEV_MODE = False
LOG_HW2GO_drawstop = False
LOG_HW2GO_switch = False
LOG_HW2GO_keys_noise = False
LOG_HW2GO_windchest = False
LOG_HW2GO_manual = False
LOG_HW2GO_rank = False
LOG_HW2GO_perfo = False
LOG_wav_decode = False

import os
import re
import math
import sys
import time

from threading import Thread
from datetime  import date

import tkinter as tk                       # already installed with Python on Windows and macOS. Ubuntu/Debian Linux : sudo apt install python3-tk
import tkinter.filedialog as fdialog
import tkinter.font as tkf
from tkinter import ttk

import sounddevice as sd                   # install with : pip install sounddevice
from PIL import Image, ImageOps, ImageTk   # install with : pip install pillow or pip install -U Pillow (+ if needed : sudo apt-get install python3-pil python3-pil.imagetk)
from lxml import etree                     # install with : pip install lxml

MAIN_WINDOW_TITLE = 'OdfEdit - ' + APP_VERSION + (' - DEV MODE' if DEV_MODE else '')

# warning message displayed before to start a HW to GO ODF conversion
HW_CONV_MSG = """An ODF will be generated permitting to use in GrandOrgue the Hauptwerk sample set whose you have just chosen the ODF.
None file of the Hauptwerk sample set will be modified.

ATTENTION :
- Please do this operation only with a free Hauptwerk sample set or a not-free sample set that you have duly paid for, and if the editor of this sample set does not preclude its use outside Hauptwerk application.
- With the generated ODF, do not expect to necessarily get with GrandOrgue exactly the same sound rendering and control possibilities as this sample set can have with Hauptwerk."""

# possible ODF file encodings
ENCODING_ISO_8859_1 = 'ISO-8859-1'  # ISO-8859-1
ENCODING_UTF8_BOM   = 'utf_8_sig'   # UTF-8

# characters allowed in a attribute name of the ODF
ALLOWED_CHARS_4_FIELDS = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_'

# data types used in the attributes of the objects, used in the check object/attribute functions
ATTR_TYPE_INTEGER = 1
ATTR_TYPE_FLOAT = 2
ATTR_TYPE_BOOLEAN = 3
ATTR_TYPE_STRING = 4
ATTR_TYPE_COLOR = 5            # used in Button, Enclosure, Label, Panel
ATTR_TYPE_FONT_SIZE = 6        # used in Button, Enclosure, Label
ATTR_TYPE_PANEL_SIZE = 7       # used in Panel
ATTR_TYPE_COUPLER_TYPE = 8     # used in Coupler
ATTR_TYPE_ELEMENT_TYPE = 9     # used in Panel Element
ATTR_TYPE_TREMULANT_TYPE = 10  # used in Tremulant
ATTR_TYPE_PISTON_TYPE = 11     # used in Piston
ATTR_TYPE_DRAWSTOP_FCT = 12    # used in DrawStop
ATTR_TYPE_FILE_NAME = 13       # used in many objects for bitmap or wave files
ATTR_TYPE_OBJECT_REF = 14      # used in many objects to make a reference to another object ID
ATTR_TYPE_PIPE_WAVE = 15       # used in Rank

# notes and octaves constants
NOTES_NAMES = ['C', 'C#', 'D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B']
NOTES_NB_IN_OCTAVE = len(NOTES_NAMES)
OCTAVES_RANGE = list(range(-1,10))

# max MIDI note value allowed for the compass extension feature
COMPASS_EXTEND_MAX = 108  # note C8

# constants to identify the type of link between two objects
TO_PARENT = 1
TO_CHILD = 2

# constants to make the functions parameters usage more understandable
FIRST_ONE = True
MANDATORY = True

# GUI colors and fonts
COLOR_BACKGROUND0 = '#E7E7E7'  # background color of the application
COLOR_BACKGROUND1 = '#BFBFBF'  # background color of the selectable widgets when the mouse is not hovering them
COLOR_BACKGROUND2 = '#3396D1'  # background color of the selectable widgets when the mouse is hovering them
COLOR_BG_EDITOR   = 'gray95'
COLOR_BG_LOGS     = 'ivory2'
COLOR_BG_HELP     = 'azure'
COLOR_BG_SEARCH   = 'light yellow'
COLOR_BG_TEXT_SEL = 'snow3'
COLOR_BG_LIST     = 'snow'

TEXT_COLOR     = 'black'
TEXT_FONT      = 'Calibri 11'
TEXT_FONT_BOLD = 'Calibri 11 bold'

# tags and colors for highlightings
TAG_OBJ_UID  = 'tag_obj_uid'  # tag to color the object UID
TAG_FIELD    = 'tag_field'    # tag to color the object fields
TAG_COMMENT  = 'tag_comment'  # tag to color the comments
TAG_TITLE    = 'tag_title'    # tag to color the titles in the help
TAG_FOUND    = 'tag_found'    # tag to color the strings found by the search in the help
TAG_FOUND2   = 'tag_found2'   # tag to color the found string currently highlighted in the help
TAG_SAME_UID = 'tag_same_uid' # tag to color in the objects tree items having same UID as the selected one

COLOR_TAG_OBJ_UID = COLOR_BACKGROUND2
COLOR_TAG_FIELD   = '#AB221D'
COLOR_TAG_COMMENT = '#1E8C03'
COLOR_TAG_TITLE   = '#AB221D'
COLOR_TAG_FOUND   = '#F7ED67'
COLOR_TAG_FOUND2  = '#EDAF04'

COLOR_SELECTED_ITEM = COLOR_BACKGROUND2 # background color for the selected object UID in the lists or tree
COLOR_SAME_UID_ITEM = '#BCE6F2'         # background color for the objects of the lists or tree having the selected object UID but not being selected

#-------------------------------------------------------------------------------------------------
class C_LOGS:
    # class to manage logs

    logs_list = [] # list of logs strings (errors or messages resulting from file operation or syntax check or ODF conversion)

    #-------------------------------------------------------------------------------------------------
    def add(self, log_string):
        # add the given string to the events log list
        self.logs_list.append(log_string)

    #-------------------------------------------------------------------------------------------------
    def get(self):
        # recover the logs list
        return self.logs_list

    #-------------------------------------------------------------------------------------------------
    def nb_get(self):
        # recover the number of logs present in the list
        return len(self.logs_list)

    #-------------------------------------------------------------------------------------------------
    def clear(self):
        # clear the log list
        self.logs_list.clear()

# create a global instance of the C_LOGS class
logs = C_LOGS()

#-------------------------------------------------------------------------------------------------
class C_AUDIO_PLAYER:
    # class to extract metadata from wave or wavepack files, and manage audio playback of wav files

    playback_in_progress = False
    playback_paused = False
    playback_to_stop = False

    data_buffer = None
    data_buffer_size = 0
    data_buffer_pos = 0

    stream = None

    samples_nb_per_loop = 1024
    bytes_nb_per_loop = 0

    wavpack_sample_rates = (6000, 8000, 9600, 11025, 12000, 16000, 22050, 24000, 32000, 44100, 48000, 64000, 88200, 96000, 192000)
    wav_compression_codes = ('Unknown', 'Microsoft PCM', 'Microsoft ADPCM', 'Microsoft IEEE float')

    #-------------------------------------------------------------------------------------------------
    def start(self, file_name):
        # start the audio playback of the given audio file (wave or wavpack format)
        # return a dictionary with the metadata of the file and the sampled audio data buffer (see the description in wav_data_get function header)

        # if a playback is already in progress, stop it and wait until the stop is completed
        self.stop()
        while self.stream != None: sd.sleep(100)

        # recover the metadata and sampled audio data of the audio file
        data_dic = self.wav_data_get(file_name)

        if data_dic['error_msg'] == '' and len(data_dic['sampled_data']) > 0:
            # the given file has been decoded without error and the extracted buffer has audio samples inside

            # initialize the data of the playback
            self.playback_in_progress = True
            self.playback_paused = False
            self.playback_to_stop = False
            self.bytes_nb_per_loop = int(self.samples_nb_per_loop * data_dic['nb_of_channels'] * data_dic['bits_per_sample'] / 8)
            self.data_buffer = data_dic['sampled_data']
            self.data_buffer_size = data_dic['audio_data_size']
            self.data_buffer_pos = 0

            # get the size of the samples according to the compression code
            dtypes_int_list = (None, 'int8', 'int16', 'int24', 'int32')
            if data_dic['compression_code'] == 1:  # WAVE_FORMAT_PCM
                dtype = dtypes_int_list[data_dic['bits_per_sample'] // 8]
            elif data_dic['compression_code'] == 3:  # WAVE_FORMAT_IEEE_FLOAT
                dtype = 'float32'
            else:
                dtype = None

            if dtype != None:
                # create and start an audio output stream
                try:
                    self.stream = sd.RawOutputStream(samplerate=data_dic['sampling_rate'],
                                                     channels=data_dic['nb_of_channels'],
                                                     dtype=dtype,
                                                     blocksize=self.samples_nb_per_loop)
                except (sd.PortAudioError, OSError) as err:
                    # an error occurred while opening the PortAudio stream
                    data_dic['error_msg'] = format(err)
                else:
                    # start the PortAudio stream
                    self.stream.start()
                    # lauch the thread managing the audio samples playback in the loop function
                    Thread(target = self.loop).start()
            else:
                data_dic['error_msg'] = f"Unsupported Wave compression code {data_dic['compression_code']}"

        return data_dic

    #-------------------------------------------------------------------------------------------------
    def loop(self):
        # manage the playback by looping on the audio samples buffer to feed to the audio stream

        while self.data_buffer_pos < self.data_buffer_size and not self.playback_to_stop:

            if not self.playback_paused:
                self.stream.write(self.data_buffer[self.data_buffer_pos:self.data_buffer_pos + self.bytes_nb_per_loop])
                self.data_buffer_pos += self.bytes_nb_per_loop
            else:
                sd.sleep(100)

        # end of the buffer playback or playback has to be stopped
        self.playback_in_progress = False
        self.playback_paused = False
        self.playback_to_stop = False

        self.stream.close()
        self.stream = None

    #-------------------------------------------------------------------------------------------------
    def pause_resume(self):
        # if a playback is in progress, toggle its state between pause and resume

        if self.playback_in_progress:
            # ask to the loop fuction to pause/resume the playback
            self.playback_paused = not self.playback_paused

    #-------------------------------------------------------------------------------------------------
    def stop(self):
        # if a playback is in progress, stop it

        if self.playback_in_progress:
            # ask to the loop fuction to stop the playback
            self.playback_to_stop = True

    #-------------------------------------------------------------------------------------------------
    def wav_data_get(self, file_name, pitch_only=False):
        # return in a dictionary the metadata of the given .wav file (wave or wavpack format)
        # if pitch_only = True, recover only the MIDI note and pitch then exit immediately (for a fast processing)

        # the returned keys are :
        #   file_format : "wave" or "wavpack" or None if error
        #   error_msg   : message describing an error occured during the file processing, empty string if no error
        #   metadata_recovered : True if the metadata of the file have been recovered successfully
        #   compression_code : 1='Microsoft PCM', 3='Microsoft IEEE float', several other...
        #   nb_of_channels   : 1=mono, 2=stereo
        #   sampling_rate    : for example 44100 (Hz)
        #   bits_per_sample  : for example 16 (bits)
        #   nb_of_samples    : number of audio samples
        #   audio_data_size  : size in bytes of the sampled audio data
        #   audio_duration   : the duration in seconds (with two decimals) of the sampled data
        #   sampled_data     : bytes buffer containing the sampled audio data
        # keys present only if a sample chunk is defined in the file
        #   midi_note        : MIDI note of the sample, integer between 0 and 127
        #   midi_pitch_fract : pitch fraction to add to the MIDI note frequency to get the exact sample frequency, float between 0 and 100
        #   loops_nb                : number of defined loops
        #   loop<n>_id              : ID of the loop n
        #   loop<n>_type            : type of the loop n (0=normal forward looping, 1=alternating forward and backward, 2=backward looping)
        #   loop<n>_start_sample    : number of the starting sample of the loop n
        #   loop<n>_start_seconds   : seconds of the starting sample of the loop n (with two decimals)
        #   loop<n>_end_sample      : number of the ending sample of the loop n
        #   loop<n>_end_seconds     : seconds of the ending sample of the loop n (with two decimals)
        #   loop<n>_replay_times_nb : number of times the loop has to be played (0=infinite)
        # keys present only if a cue chunk is defined in the file
        #   cue_points_nb           : number of defined cue points
        #   cue<n>_id               : ID of the cue point n
        #   cue<n>_position         : 0 if there is no playlist chunk, else the sample at which the cue point should occur
        #   cue<n>_chunk_id         : chunk ID of the chunk containing the cue point (wave or slnt)
        #   cue<n>_chunk_start      : file position of the start of the chunk containing the cue point
        #                             (byte offset relative to the start of the data section of the wave list chunk 'wavl' if one is defined)
        #   cue<n>_block_start      : file position of the start of the block containing the position
        #                             (byte offset relative to the start of the data section of the wave list chunk 'wavl' if one is defined)
        #   cue<n>_sample_start     : sample offset of the cue point relative to the start of the block (bytes)
        # keys present only if a LIST chunk is defined in the file
        #   info                    : a dictionary containing as keys the info ID and as value the corresponding info text

        metadata_dic = {}
        metadata_dic['error_msg'] = ''
        metadata_dic['metadata_recovered'] = False
        metadata_dic['sampled_data'] = []

        if not os.path.isfile(file_name):
            # the provided file doesn't exist
            metadata_dic['error_msg'] = f'The file "{file_name}" does not exist.'
            return metadata_dic

        # open the given file
        file_obj = open(file_name, 'rb')

        file_format = None
        data_type = None        # can be : 'chunk' (Wav file), 'block' or 'sub_block' or 'sub_block_chunk' (WavPack file)
        chunk_data_size = 0     # Wav or WavPack file
        block_data_size = 0     # WavPack file
        sub_block_data_size = 0 # WavPack file

        while file_obj.read(1):
            # scan the file while the end of file is not reached and no error has occured
            file_obj.seek(-1, 1)   # move the file pointer one byte back to compensate the read(1) of the while instruction

            if file_format == None:
                # identification of the file format from the first 4 bytes of the file
                file_type_str = file_obj.read(4)
                file_obj.seek(-4, 1)  # rewind at the beginning of the file
                if file_type_str == b'wvpk':
                    file_format = 'wavpack'
                    data_type = 'block'
                    if LOG_wav_decode: print(f'WavPack file {file_name}')
                elif file_type_str == b'RIFF':
                    file_format = 'wave'
                    data_type = 'chunk'
                    if LOG_wav_decode: print(f'Wave file {file_name}')
                else:
                    metadata_dic['error_msg'] = f'Unsupported format in file {file_name}, Wave or WavPack format is expected'
                    if LOG_wav_decode: print(f'Unsupported format in file {file_name}')
                    file_obj.close()
                    return metadata_dic

                metadata_dic['file_format'] = file_format

            if data_type == 'block':
                # start of a WavPack block

                # read the data of the block header
                file_obj.read(4)      # skip the ID "wvpk"
                block_data_size = int.from_bytes(file_obj.read(4), 'little')   # size of the entire block minus 8 bytes
                version = int.from_bytes(file_obj.read(2), 'little')           # 0x402 to 0x410 are valid for decode
                block_index_u8 = int.from_bytes(file_obj.read(1), 'little')    # upper 8 bits  of 40-bit block index
                total_samples_u8 = int.from_bytes(file_obj.read(1), 'little')  # upper 8 bits  of 40-bit total samples
                total_samples_l32 = int.from_bytes(file_obj.read(4), 'little') # lower 32 bits of 40-bit total samples
                block_index_l32 = int.from_bytes(file_obj.read(4), 'little')   # lower 32 bits of 40-bit block index
                nb_of_samples_in_block = int.from_bytes(file_obj.read(4), 'little') # number of samples in this block, 0=non-audio block
                flags  = int.from_bytes(file_obj.read(4), 'little')            # flags for id and decoding
                file_obj.read(4)      # skip the CRC for actual decoded data
                block_data_size -= 24 # entire block size -8 bytes -24 bytes read in the block header

                block_index = (block_index_u8 << 32) + block_index_l32
                nb_of_samples = (total_samples_u8 << 32) + total_samples_l32
                if LOG_wav_decode: print(f"  block : version = 0x{version:03X}, size = {block_data_size+32}, data size = {block_data_size} bytes, first sample index = {block_index}, nb of samples in the block = {nb_of_samples_in_block}, total number of samples = {nb_of_samples}, flags = 0x{flags:08X}")

                # get metadata from flags of the first block
                if block_index == 0 and not pitch_only:
                    metadata_dic['nb_of_samples'] = nb_of_samples
                    metadata_dic['bits_per_sample'] = ((flags & 0b11) + 1) * 8

                    if ((flags >> 2) & 0b1) == 0:
                        metadata_dic['nb_of_channels'] = 2
                    else:
                        metadata_dic['nb_of_channels'] = 1

                    metadata_dic['sampling_rate'] = self.wavpack_sample_rates[(flags >> 23) & 0b1111]
                    metadata_dic['compression_code'] = (flags >> 3) & 0b1  # 0 = losslessaudio, 1 = hybrid mode
                    metadata_dic['audio_data_size'] = metadata_dic['nb_of_samples'] * metadata_dic['nb_of_channels'] * (metadata_dic['bits_per_sample'] // 8)

                    if LOG_wav_decode: print(f"  block : nb of channels = {metadata_dic['nb_of_channels']}, bits per sample = {metadata_dic['bits_per_sample']}, sampling rate = {metadata_dic['sampling_rate']}, compression code = {metadata_dic['compression_code']}")

                if block_data_size > 0:
                    # the block contains data, a sub-block is going to follow in the file
                    data_type = 'sub_block'
                else:
                    if LOG_wav_decode: print('  no data inside')

            if data_type == 'sub_block':
                # start of a WavPack metadata sub-block

                sub_block_id = int.from_bytes(file_obj.read(1), 'little')
                block_data_size -= 1
                sub_block_func_id = sub_block_id & 0x3f

                # read the size of the sub-block (in bytes)
                if sub_block_id & 0x80:
                    # it is a large block, its size is encoded on 3 bytes
                    sub_block_data_size = int.from_bytes(file_obj.read(3), 'little') * 2  # x2 to convert the size from words to bytes
                    block_data_size -= 3
                    large_block = True
                else:
                    sub_block_data_size = int.from_bytes(file_obj.read(1), 'little') * 2
                    block_data_size -= 1
                    large_block = False

                if LOG_wav_decode: print(f'    sub-block : ID = 0x{sub_block_id:X}, function ID = 0x{sub_block_func_id:X}, large block = {large_block}, data size = {sub_block_data_size} bytes')

                if sub_block_func_id in (0x21, 0x22): # 0x21 = ID_RIFF_HEADER, 0x22 = ID_RIFF_TRAILER
                    # RIFF chunks are present in this sub-block
                    data_type = 'sub_block_chunk'
                    if LOG_wav_decode: print('      RIFF chunks inside')

                if sub_block_data_size == 0:
                    if LOG_wav_decode: print('      no data inside')

            if data_type in ('chunk', 'sub_block_chunk'):
                # start of a Wave chunk (in a Wav or WavPack file)

                chunk_id = file_obj.read(4).decode('utf-8', 'ignore')  # string of 4 characters
                chunk_data_size = int.from_bytes(file_obj.read(4), 'little')

                chunk_read_data_size = 0

                if chunk_data_size % 2:
                    # the chunk size has an odd number of bytes, a word padding byte is present after it
                    # increment the chunk size by 1 to cover this padding byte
                    chunk_data_size += 1

                if chunk_id == 'RIFF':
                    if LOG_wav_decode: print(f'      chunk : [{chunk_id}], wave file size = {chunk_data_size + 8} bytes')
                else:
                    if LOG_wav_decode: print(f'      chunk : [{chunk_id}], size = {chunk_data_size} bytes')

                if chunk_id == 'RIFF':
                    # RIFF chunk descriptor
                    if file_obj.read(4) == b'WAVE':  # RIFF type ID
                        # the RIFF type ID is WAVE, it is a valid .wav file
                        if LOG_wav_decode: print('        RIFF type ID = "WAVE"')
                    else:
                        metadata_dic['error_msg'] = 'RIFF chunk has not the "WAVE" type ID, unsuported file format'
                        if LOG_wav_decode: print('        RIFF chunk has not the "WAVE" type ID, unsuported file format')
                        file_obj.close()
                        return metadata_dic

                    chunk_data_size = 4  # in the RIFF chunk the data size value is the size of the file - 8 bytes
                                         # there are only 4 bytes of data in this chunk (the RIFF type ID)
                    chunk_read_data_size += 4

                elif chunk_id == 'fmt ' and not pitch_only:
                    # format chunk
                    metadata_dic['compression_code'] = int.from_bytes(file_obj.read(2), 'little')
                    metadata_dic['nb_of_channels'] = int.from_bytes(file_obj.read(2), 'little')
                    metadata_dic['sampling_rate'] = int.from_bytes(file_obj.read(4), 'little')
                    file_obj.read(4) # skip the bytes per second
                    file_obj.read(2) # skip the block align
                    metadata_dic['bits_per_sample'] = int.from_bytes(file_obj.read(2), 'little')
                    if LOG_wav_decode: print(f"        compression code = {metadata_dic['compression_code']}, nb of channels = {metadata_dic['nb_of_channels']}, sampling rate = {metadata_dic['sampling_rate']}, bits per sample = {metadata_dic['bits_per_sample']}")

                    chunk_read_data_size += 16

                elif chunk_id == 'smpl':
                    # sample chunk
                    file_obj.read(4) # skip the manufacturer ID
                    file_obj.read(4) # skip the product ID
                    file_obj.read(4) # skip the sample period
                    metadata_dic['midi_note'] = int.from_bytes(file_obj.read(4), 'little')
                    metadata_dic['midi_pitch_fract'] = float(int.from_bytes(file_obj.read(4), 'little') * 100 / 0xFFFFFFFF)
                    if pitch_only:
                        # the MIDI note and pitch are recovered and only them are expected, exit the function
                        file_obj.close()
                        metadata_dic['metadata_recovered'] = True
                        return metadata_dic

                    file_obj.read(4) # skip the SMPTE format
                    file_obj.read(4) # skip the SMPTE offset
                    metadata_dic['loops_nb'] = loops_nb = int.from_bytes(file_obj.read(4), 'little')
                    file_obj.read(4) # skip the sampler data bytes number
                    chunk_read_data_size += 36


                    for l in range(1, loops_nb+1):
                        metadata_dic[f'loop{l}_id'] = int.from_bytes(file_obj.read(4), 'little')
                        metadata_dic[f'loop{l}_type'] = int.from_bytes(file_obj.read(4), 'little')
                        metadata_dic[f'loop{l}_start_sample'] = int.from_bytes(file_obj.read(4), 'little')
                        metadata_dic[f'loop{l}_start_seconds'] = int(metadata_dic[f'loop{l}_start_sample'] * 1000 / metadata_dic['sampling_rate']) / 1000
                        metadata_dic[f'loop{l}_end_sample'] = int.from_bytes(file_obj.read(4), 'little')
                        metadata_dic[f'loop{l}_end_seconds'] = int(metadata_dic[f'loop{l}_end_sample'] * 1000 / metadata_dic['sampling_rate']) / 1000
                        file_obj.read(4) # skip the loop fraction
                        metadata_dic[f'loop{l}_replay_times_nb'] = int.from_bytes(file_obj.read(4), 'little')
                        chunk_read_data_size += 24
                    if LOG_wav_decode: print(f"        midi note = {metadata_dic['midi_note']}, midi pitch = {metadata_dic['midi_pitch_fract']}, loops nb = {metadata_dic['loops_nb']}")

                elif chunk_id == 'cue ' and not pitch_only:
                    # cue chunk
                    metadata_dic['cue_points_nb'] = cue_points_nb = int.from_bytes(file_obj.read(4), 'little')
                    chunk_read_data_size += 4

                    for c in range(1, cue_points_nb+1):
                        metadata_dic[f'cue{c}_id'] = int.from_bytes(file_obj.read(4), 'little')
                        metadata_dic[f'cue{c}_position'] = int.from_bytes(file_obj.read(4), 'little')
                        metadata_dic[f'cue{c}_chunk_id'] = file_obj.read(4).decode('utf-8', 'ignore')  # string of 4 characters
                        metadata_dic[f'cue{c}_chunk_start'] = int.from_bytes(file_obj.read(4), 'little')
                        metadata_dic[f'cue{c}_block_start'] = int.from_bytes(file_obj.read(4), 'little')
                        metadata_dic[f'cue{c}_sample_start'] = int.from_bytes(file_obj.read(4), 'little')
                        chunk_read_data_size += 24
                    if LOG_wav_decode: print(f"        cue points nb = {metadata_dic['cue_points_nb']}")

                elif chunk_id == 'LIST' and not pitch_only:
                    # list chunk
                    list_type_id = file_obj.read(4).decode('utf-8', 'ignore')
                    chunk_read_data_size += 4

                    if list_type_id == 'INFO':
                        # get the data of the INFO type list
                        metadata_dic['info'] = {}
                        while chunk_read_data_size < chunk_data_size:  # loop until the end of the chunk
                            info_id = file_obj.read(4).decode('utf-8', 'ignore')
                            text_size = int.from_bytes(file_obj.read(4), 'little')
                            text_content = file_obj.read(text_size).decode('utf-8', 'ignore')
                            while text_content[-1:] == '\x00':  # remove the trailing 0x00 characters
                                text_content = text_content[:-1]
                            metadata_dic['info'][info_id] = text_content
                            chunk_read_data_size += 8 + text_size

                            if (text_size % 2) != 0:
                                # the text size is an odd value whereas text must be word aligned
                                # move the file reading position by 1 byte forward at the next word starting position
                                file_obj.read(1)
                                chunk_read_data_size += 1

                        if LOG_wav_decode: print(f"        INFO = {metadata_dic['info'].keys()}")
                    else:
                        if LOG_wav_decode: print('        none data recovered')

                elif chunk_id == 'data' and not pitch_only:
                    # data chunk
                    metadata_dic['audio_data_size'] = chunk_data_size    # size in bytes of the sampled audio data
                    metadata_dic['nb_of_samples'] = int(chunk_data_size / (metadata_dic['nb_of_channels'] * metadata_dic['bits_per_sample'] // 8))
                    if LOG_wav_decode: print(f"        nb of samples = {metadata_dic['nb_of_samples']}, audio data size = {metadata_dic['audio_data_size']}")
                    if data_type == 'chunk':
                        # get the buffer of raw sampled audio data in case of Wav file only
                        metadata_dic['sampled_data'] = file_obj.read(chunk_data_size)
                        chunk_read_data_size += chunk_data_size
                    else:  # data_type is 'sub_block_chunk'
                        # if WavPack file, there are no audio samples in the data chunk
                        chunk_data_size = 0

                else:
                    if LOG_wav_decode: print("        none data recovered")

                # move the file pointer at the end of the chunk if it is not already there
                if chunk_data_size - chunk_read_data_size > 0:
                    file_obj.seek(chunk_data_size - chunk_read_data_size, 1)
                    if LOG_wav_decode: print(f'        file pointer moved at the end of the chunk by {chunk_data_size - chunk_read_data_size} bytes')

                if data_type == 'sub_block_chunk':
                    # update the remaining size of data not read in the parents sub-block and block
                    block_data_size -= 8 + chunk_data_size  # 8 is the size of chunk ID and size fields at the start of the chunk
                    sub_block_data_size -= 8 + chunk_data_size

                    if sub_block_data_size <= 0:
                        # the end of the sub-block with RIFF chunks inside is reached
                        data_type = 'sub_block'

            if data_type == 'sub_block' and sub_block_data_size > 0:
                # sub-block with unread data inside
                # move the file pointer at the end of the sub-block
                file_obj.seek(sub_block_data_size, 1)
                if LOG_wav_decode: print(f'      {sub_block_data_size} bytes skipped')
                block_data_size -= sub_block_data_size
                sub_block_data_size = 0

            if data_type == 'sub_block' and block_data_size <= 0:
                # block end is reached
                data_type = 'block'

            if data_type == 'block' and block_data_size > 0:
                # block with unread data inside
                # move the file pointer at the end of the block
                file_obj.seek(block_data_size, 1)
                if LOG_wav_decode: print(f'  {block_data_size} bytes skipped')
                block_data_size = 0

        # close the given file
        file_obj.close()

        metadata_dic['audio_duration'] = 0
        if file_format != None and not pitch_only:
            if 'audio_data_size' not in metadata_dic.keys():
                # audio file without data chunk inside
                metadata_dic['error_msg'] = f'The file "{file_name}" has no audio samples inside.'
            else:
                # compute the duration of the audio samples in seconds (float with 2 decimals)
                metadata_dic['audio_duration'] = int(metadata_dic['audio_data_size'] * 1000 / (metadata_dic['sampling_rate'] * metadata_dic['nb_of_channels'] * metadata_dic['bits_per_sample'] / 8)) / 1000

        metadata_dic['metadata_recovered'] = True

        return metadata_dic

# create a global instance of the C_AUDIO_PLAYER class
audio_player = C_AUDIO_PLAYER()

#-------------------------------------------------------------------------------------------------
class C_ODF_MISC:
    # class containing miscellaneous functions to manage data of GO ODF objects

    def compass_get(self, object_uid, rank_compass_in_stop=False):
        # return in a tuple the first and last MIDI notes of the given object (must be Manual, Stop or Rank)
        # or return None in case a compass is not defined in this object or expected attributes are not defined correctly

        object_type = self.object_type_get(object_uid)
        object_dic = self.object_dic_get(object_uid)

        if object_type == 'Manual':

            manual_first_access_key_midi_note = myint(self.object_attr_value_get(object_dic, 'FirstAccessibleKeyMIDINoteNumber'))
            if manual_first_access_key_midi_note == None:
                logs.add(f'ERROR : The section {object_uid} has no FirstAccessibleKeyMIDINoteNumber value defined')
                return None

            manual_access_keys_nb = myint(self.object_attr_value_get(object_dic, 'NumberOfAccessibleKeys'))
            if manual_access_keys_nb == None:
                logs.add(f'ERROR : The section {object_uid} has no NumberOfAccessibleKeys value defined')
                return None

            if myint(self.object_attr_value_get(object_uid, 'FirstAccessibleKeyLogicalKeyNumber')) == None:
                logs.add(f'ERROR : The section {object_uid} has no NumberOfLogicalKeys value defined')
                return None

            if myint(self.object_attr_value_get(object_uid, 'NumberOfLogicalKeys')) == None:
                logs.add(f'ERROR : The section {object_uid} has no NumberOfLogicalKeys value defined')
                return None

            return (manual_first_access_key_midi_note, manual_first_access_key_midi_note + manual_access_keys_nb - 1)


        if object_type == 'Stop':

            manual_uid = self.object_parent_manual_get(object_uid)
            manual_first_access_key_midi_note = myint(self.object_attr_value_get(manual_uid, 'FirstAccessibleKeyMIDINoteNumber'))
            if manual_first_access_key_midi_note == None:
                logs.add(f'ERROR : The section {manual_uid} has no FirstAccessibleKeyMIDINoteNumber value defined')
                return None

            manual_first_access_key_logic_key_nb = myint(self.object_attr_value_get(manual_uid, 'FirstAccessibleKeyLogicalKeyNumber'))
            if manual_first_access_key_logic_key_nb == None:
                logs.add(f'ERROR : The section {manual_uid} has no FirstAccessibleKeyLogicalKeyNumber value defined')
                return None

            manual_first_logical_key_midi_note = manual_first_access_key_midi_note - manual_first_access_key_logic_key_nb + 1

            stop_ranks_nb = myint(self.object_attr_value_get(object_dic, 'NumberOfRanks'), 0)

            stop_first_access_pipe_logic_key_nb = myint(self.object_attr_value_get(object_dic, 'FirstAccessiblePipeLogicalKeyNumber'))
            if stop_first_access_pipe_logic_key_nb == None:
                logs.add(f'ERROR : The section {object_uid} has no attribute FirstAccessiblePipeLogicalKeyNumber defined')
                return None

            stop_first_access_pipe_midi_note = manual_first_logical_key_midi_note + stop_first_access_pipe_logic_key_nb - 1

            stop_access_pipes_nb = myint(self.object_attr_value_get(object_dic, 'NumberOfAccessiblePipes'))
            if stop_access_pipes_nb == None:
                logs.add(f'ERROR : The section {object_uid} has no attribute NumberOfAccessiblePipes defined')
                return None

            if stop_ranks_nb == 0:
                stop_first_access_pipe_logic_pipe_nb = myint(self.object_attr_value_get(object_uid, 'FirstAccessiblePipeLogicalPipeNumber'))
                if stop_first_access_pipe_logic_pipe_nb == None:
                    logs.add(f'ERROR : The section {object_uid} has no FirstAccessiblePipeLogicalPipeNumber value defined')
                    return None

                stop_logic_pipes_nb = myint(self.object_attr_value_get(object_dic, 'NumberOfLogicalPipes'))
                if stop_logic_pipes_nb == None:
                    logs.add(f'ERROR : The section {object_uid} has no attribute NumberOfLogicalPipes defined')
                    return None

            if rank_compass_in_stop and stop_ranks_nb == 0:
                # instead of returning the compass of the stop, do return the compass of the rank defined in the stop

                stop_first_logical_pipe_midi_note = stop_first_access_pipe_midi_note - stop_first_access_pipe_logic_pipe_nb + 1

                return (stop_first_logical_pipe_midi_note, stop_first_logical_pipe_midi_note + stop_logic_pipes_nb - 1)

            return (stop_first_access_pipe_midi_note, stop_first_access_pipe_midi_note + stop_access_pipes_nb - 1)


        if object_type == 'Rank':

            rank_midi_note_first = myint(self.object_attr_value_get(object_dic, 'FirstMidiNoteNumber'))
            if rank_midi_note_first == None:
                logs.add(f'ERROR : The section {object_uid} has no attribute FirstMidiNoteNumber defined')
                return None

            rank_pipes_nb = myint(self.object_attr_value_get(object_dic, 'NumberOfLogicalPipes'))
            if rank_pipes_nb == None:
                logs.add(f'ERROR : The section {object_uid} has no attribute NumberOfLogicalPipes defined')
                return None

            return (rank_midi_note_first, rank_midi_note_first + rank_pipes_nb - 1)

        logs.add(f"A section {object_type} has no compass attributes")
        return None

    #-------------------------------------------------------------------------------------------------
    def compass_extend(self, object_uid, midi_note_ext):
        # extends the compass of the given object (Manual or Stop or Rank) of the ODF up or down to the given MIDI note included
        # return in a tuple the new first and last MIDI notes of the manual
        # or return None if an error has occurred

        # check if the given object can be extended
        compass = self.compass_get(object_uid)
        if compass == None:
            # the object has not a compass which can be extended or there are errors in the object
            return None

        object_type = self.object_type_get(object_uid)
        midi_note_first, midi_note_last = compass
        logs.add(f'Trying to extend {object_uid} from MIDI notes compass {midi_note_first}-{midi_note_last} up to MIDI note {midi_note_ext}')
        logs.add('')

        if object_type == 'Rank':
            return self.compass_extend_rank(object_uid, midi_note_ext)

        if object_type == 'Stop':
            return self.compass_extend_stop(object_uid, midi_note_ext)

        if object_type == 'Manual':
            return self.compass_extend_manual(object_uid, midi_note_ext)

        return None

    #-------------------------------------------------------------------------------------------------
    def compass_extend_manual(self, object_uid, midi_note_ext):
        # extends the compass of the Stops objects which are children of the given Manual object UID, up to the given MIDI note (included)
        # return in a tuple the new first and last MIDI notes of the manual
        # or return None in case an issue has occurred

        object_type = self.object_type_get(object_uid)
        object_dic = self.object_dic_get(object_uid)

        if object_type != 'Manual':
            logs.add('INTERNAL ERROR : a Manual section is expected in compass_extend_manual()')
            return None

        # get the initial compass and data of the Manual
        first_midi_note_init, last_midi_note_init = self.compass_get(object_uid)
        first_midi_note_ext = first_midi_note_init
        last_midi_note_ext = last_midi_note_init

        nb_logical_keys = myint(self.object_attr_value_get(object_dic, 'NumberOfLogicalKeys'))
        nb_access_keys = myint(self.object_attr_value_get(object_dic, 'NumberOfAccessibleKeys'))
        first_access_key_logic_key_nb = myint(self.object_attr_value_get(object_dic, 'FirstAccessibleKeyLogicalKeyNumber'))
        first_access_key_midi_note = myint(self.object_attr_value_get(object_dic, 'FirstAccessibleKeyMIDINoteNumber'))

        stops_nb = myint(self.object_attr_value_get(object_dic, 'NumberOfStops'))
        if stops_nb == None:
            logs.add(f'ERROR : {object_uid} has no NumberOfStops value defined')
            return None
        if stops_nb == 0:
            logs.add(f'{object_uid} has no children Stop sections defined')
            return None

        pipes_stop_extended = False

        # extend each child stop of the given Manual object, if it is used by the last note of the manual
        for stop_idx in range(1, stops_nb + 1):

            stop_attr_id = 'Stop' + str(stop_idx).zfill(3)
            # get the ID and UID of the stop referenced in the current stop index
            stop_id = myint(self.object_attr_value_get(object_dic, stop_attr_id))
            if stop_id == None:
                logs.add(f'ERROR : the attribute {stop_attr_id} is not defined in {object_uid}')
                return None
            stop_uid = 'Stop' + str(stop_id).zfill(3)
            if self.object_dic_get(stop_uid) == None:
                logs.add(f'ERROR : the stop {stop_uid} referenced in {object_uid} as {stop_attr_id} does not exist')
                return None

            # get the initial compass of the current stop
            compass = self.compass_get(stop_uid)
            if compass == None:
                return None
            stop_first_midi_note_init, stop_last_midi_note_init = compass

            # extend the current stop
            if myint(self.object_attr_value_get(stop_uid, 'NumberOfAccessiblePipes')) == 1:
                compass = None
                logs.add(f'{stop_uid} is not extended as it has only one accessible pipe')
            elif stop_last_midi_note_init < last_midi_note_init:
                # the stop is not used by the last note of the manual
                compass = None
                logs.add(f'{stop_uid} does not have to be extended as it is not played by the last note of {object_uid}')
            elif mystr(self.object_attr_value_get(stop_uid, 'AcceptsRetuning')) == 'N' and pipes_stop_extended:
                # the current stop cannot be retuned, it should be a stop containing noises
                # extend it until the last extended note of the manual if pipes stop has been extended before, in case it is a stop with keys noises
                compass = self.compass_extend_stop(stop_uid, last_midi_note_ext)
            else:
                compass = self.compass_extend_stop(stop_uid, midi_note_ext)
                pipes_stop_extended = True

            if compass != None:
                # the stop has been extended
                stop_first_midi_note_ext, stop_last_midi_note_ext = compass
                stop_last_midi_note_ext = min(stop_last_midi_note_ext, midi_note_ext)  # if the stop goes beyond the extension MIDI note, ignore the beyong compass

                # update the last MIDI note of the manual based on the extension done in the current stop
                last_midi_note_ext = max(last_midi_note_ext, stop_last_midi_note_ext)

            logs.add('')  # add a blank line in the logs window between each stop

        # update the number of accessible keys
        nb_access_keys_ext = last_midi_note_ext - first_access_key_midi_note + 1
        self.object_attr_value_set(object_uid, 'NumberOfAccessibleKeys', nb_access_keys_ext)

        # update the number of logical keys
        if first_access_key_logic_key_nb + nb_access_keys_ext - 1 > nb_logical_keys:
            nb_logical_keys_ext = first_access_key_logic_key_nb + nb_access_keys_ext - 1
            self.object_attr_value_set(object_uid, 'NumberOfLogicalKeys', nb_logical_keys_ext)

        # add or update the DisplayKeys attribute, set at the initial compass, so that number of displayed keys is unchanged
        # to be placed in the child PanelElement (Type=Manual) object of the Manual if it is defined, else in the Manual object
        panel_elem_children_list = self.object_kinship_list_get(object_uid, TO_CHILD, 'PanelElement')
        if len(panel_elem_children_list) > 0:
            manual_panel_elem_uid = panel_elem_children_list[0]
            self.object_attr_value_set(manual_panel_elem_uid, 'DisplayKeys', last_midi_note_init - first_midi_note_init + 1)
        else:
            self.object_attr_value_set(object_uid, 'DisplayKeys', last_midi_note_init - first_midi_note_init + 1)

        if nb_access_keys_ext > nb_access_keys:
            logs.add(f'{object_uid} MIDI notes compass extended from {first_midi_note_init}-{last_midi_note_init} to {first_midi_note_ext}-{last_midi_note_ext}')
        else:
            logs.add(f'{object_uid} compass not extended')

        return first_midi_note_ext, last_midi_note_ext

    #-------------------------------------------------------------------------------------------------
    def compass_extend_stop(self, object_uid, midi_note_ext):
        # extends the compass of the Rank objects which are children of the given Stop object UID, up to the given MIDI note (included)
        # return in a tuple the new first and last MIDI notes of the stop
        # or return None in case an issue has occurred

        object_type = self.object_type_get(object_uid)
        object_dic = self.object_dic_get(object_uid)

        if object_type != 'Stop':
            logs.add('INTERNAL ERROR : a Stop section is expected in compass_extend_stop()')
            return None

        # get the initial compass of the Stop (accessible pipes compass)
        compass = self.compass_get(object_uid)
        if compass == None:
            return None

        first_midi_note_init, last_midi_note_init = compass

        if midi_note_ext <= last_midi_note_init:
            logs.add(f'{object_uid} has the MIDI notes compass {first_midi_note_init}-{last_midi_note_init} which covers the MIDI note {midi_note_ext}, it does not need to be extended')
            return first_midi_note_init, last_midi_note_init

        if myint(self.object_attr_value_get(object_uid, 'NumberOfAccessiblePipes')) == 1:
            logs.add(f'{object_uid} is not extended as it has only one accessible pipe')
            return first_midi_note_init, last_midi_note_init

        last_midi_note_ext = last_midi_note_init  # this value will increase based on the extension actually done in each rank used by the stop

        ranks_nb = myint(self.object_attr_value_get(object_uid, 'NumberOfRanks'), 0)

        if ranks_nb == 0:
            # one rank definition is included in the Stop object

            # extend the rank up to the given MIDI note extension
            compass = self.compass_extend_rank(object_uid, midi_note_ext)
            if compass == None:
                # an error has occured
                return None
            rank_first_midi_note_ext, rank_last_midi_note_ext = compass

            # update the number of accessible pipes of the stop
            rank_last_used_midi_note_ext = min(rank_last_midi_note_ext, midi_note_ext) # if the rank goes beyond the MIDI note extension, ignore the beyong compass
            nb_access_pipes = rank_last_used_midi_note_ext - first_midi_note_init + 1
            self.object_attr_value_set(object_uid, 'NumberOfAccessiblePipes', nb_access_pipes)

            last_midi_note_ext = rank_last_used_midi_note_ext

        else:
            # it is a stop having children ranks, extend each child rank if it is used by the last note of the stop

            for rank_idx in range(1, ranks_nb + 1):
                # scan the children ranks

                rank_attr_id = 'Rank' + str(rank_idx).zfill(3)
                # get the ID and UID of the rank referenced in the current rank index
                rank_id = myint(self.object_attr_value_get(object_dic, rank_attr_id))
                if rank_id == None:
                    logs.add(f'ERROR : the attribute {rank_attr_id} is not defined in {object_uid}')
                    return None
                rank_uid = 'Rank' + str(rank_id).zfill(3)
                if self.object_dic_get(rank_uid) == None:
                    logs.add(f'ERROR : the rank {rank_uid} referenced in {object_uid} as {rank_attr_id} does not exist')
                    return None

                # get the initial compass of the current rank
                compass = self.compass_get(rank_uid)
                if compass == None:
                    return None
                rank_first_midi_note_init, rank_last_midi_note_init = compass

                # recover the MIDI notes compass used by the Stop in the current rank
                rank_first_used_pipe_nb = myint(self.object_attr_value_get(object_dic, rank_attr_id + 'FirstPipeNumber'), 1)
                rank_used_pipes_count = myint(self.object_attr_value_get(object_dic, rank_attr_id + 'PipeCount'), rank_last_midi_note_init - rank_first_midi_note_init + 1 - (rank_first_used_pipe_nb - 1))
                rank_first_used_midi_note_init = rank_first_midi_note_init + (rank_first_used_pipe_nb - 1)
                rank_last_used_midi_note_init = rank_first_used_midi_note_init + rank_used_pipes_count - 1

                logs.add(f'{object_uid} uses in {rank_uid} the MIDI notes compass {rank_first_used_midi_note_init}-{rank_last_used_midi_note_init}')

                # recover the MIDI note shift from manual keys to rank pipes
                manual_first_access_key_nb = myint(self.object_attr_value_get(object_dic, rank_attr_id + 'FirstAccessibleKeyNumber'), 1)
                midi_note_manual_to_rank_shift = rank_first_used_midi_note_init - (first_midi_note_init + manual_first_access_key_nb - 1)
                if midi_note_manual_to_rank_shift != 0:
                    logs.add(f'                  from manual keys MIDI notes compass {rank_first_used_midi_note_init - midi_note_manual_to_rank_shift}-{rank_last_used_midi_note_init - midi_note_manual_to_rank_shift}')

                if last_midi_note_init + midi_note_manual_to_rank_shift <= rank_last_used_midi_note_init:
                    # the last note of the stop if using the current rank, so extend it
                    compass = self.compass_extend_rank(rank_uid, midi_note_ext + midi_note_manual_to_rank_shift)
                    if compass != None:
                        rank_first_midi_note_ext, rank_last_midi_note_ext = compass
                        rank_last_used_midi_note_ext = min(rank_last_midi_note_ext, midi_note_ext + midi_note_manual_to_rank_shift)  # if the rank goes beyond the extension MIDI note, ignore the beyong compass

                        # update the number of pipes used by the stop in the current rank
                        if self.object_attr_value_get(object_dic, rank_attr_id + 'PipeCount') != None:
                            # the attribute Rank999PipeCount is defined, update it
                            rank_pipe_count_ext = rank_last_used_midi_note_ext - rank_first_used_midi_note_init + 1
                            self.object_attr_value_set(object_dic, rank_attr_id + 'PipeCount', rank_pipe_count_ext)

                        if mystr(self.object_attr_value_get(rank_uid, 'AcceptsRetuning'), 'Y') == 'Y':
                            # update the last MIDI note of the stop based on the extension done in the current rank
                            # only if the rank can be retuned
                            # this is to avoid to take into the extension of the ranks which can be extended up to the required MIDI note without pitch tuning limitation
                            last_midi_note_ext = max(last_midi_note_ext, rank_last_used_midi_note_ext - midi_note_manual_to_rank_shift)

                else:
                    logs.add(f'{rank_uid} does not have to be extended as it is not played by the last note of {object_uid}')

            # update the number of pipes used in the stop, adding to it the number of added MIDI notes in the ranks
            nb_access_pipes = myint(self.object_attr_value_get(object_uid, 'NumberOfAccessiblePipes')) + last_midi_note_ext - last_midi_note_init
            self.object_attr_value_set(object_uid, 'NumberOfAccessiblePipes', nb_access_pipes)

            if last_midi_note_ext > last_midi_note_init:
                logs.add(f'{object_uid} MIDI notes compass extended from {first_midi_note_init}-{last_midi_note_init} to {first_midi_note_init}-{last_midi_note_ext}')
            else:
                logs.add(f'{object_uid} MIDI notes compass not extended')

        return first_midi_note_init, last_midi_note_ext

    #-------------------------------------------------------------------------------------------------
    def compass_extend_rank(self, object_uid, midi_note_ext):
        # extends the compass of the given object UID (rank or stop with rank data inside) of the GO ODF, up/down to the given MIDI note (included)
        # borrowing existing pipes of the rank with one octave of interval
        # the extension can be done below or above the existing compass
        # return in a tuple the new first and last MIDI notes of the rank
        # or return None in case an issue has occurred

        object_type = self.object_type_get(object_uid)
        object_dic = self.object_dic_get(object_uid)

        # get the initial MIDI notes compass of the given Rank or Stop
        if object_type not in ('Rank', 'Stop'):
            logs.add('INTERNAL ERROR : a Rank or Stop section is expected in compass_extend_rank()')
            return None

        # get the compass of the given Rank or the rank which is inside the given Stop
        compass = self.compass_get(object_uid, object_type == 'Stop')
        if compass == None:
            # there is an error in the objects
            return None

        first_midi_note_init, last_midi_note_init = compass

        if last_midi_note_init - first_midi_note_init < 12:
            logs.add(f'{object_uid} is not extended as it has less than 12 accessible pipes')
            return first_midi_note_init, last_midi_note_init

        rank_accepts_retuning = mystr(self.object_attr_value_get(object_dic, 'AcceptsRetuning'), 'Y')

        # define the extended MIDI notes compass of the given rank
        if midi_note_ext < first_midi_note_init:
            # extension below the initial rank compass
            first_midi_note_ext = midi_note_ext
            last_midi_note_ext = last_midi_note_init
        elif midi_note_ext > last_midi_note_init:
            # extension above the initial rank compass
            first_midi_note_ext = first_midi_note_init
            last_midi_note_ext = midi_note_ext
        else:
            # no extension to do, the given MIDI note is inside the initial rank compass
            logs.add(f'{object_uid} does not need to be extended, the MIDI note {midi_note_ext} is inside its MIDI notes compass {first_midi_note_init}-{last_midi_note_init}')
            return first_midi_note_init, last_midi_note_init

        # build a dictionary having as keys all MIDI notes of the extended rank
        # and for each MIDI note a tuple value with : the pipe number of the initial rank to use to play this MIDI note
        #                                             the pitch tuning in cents to apply to this pipe to play the MIDI note
        #                                             the gain to apply to the pipe (+5dB to apply if negative pitch tuning, -5dB if positive pitch tuning)
        # or None if none pipe can be mapped for the MIDI note
        midi_pipe_mapping_dic = {}
        for midi_note_nb in range(first_midi_note_ext, last_midi_note_ext + 1):
            # scan the MIDI notes of the extended compass

            # determine if an existing pipe has to be changed to play the current MIDI note
            if midi_note_nb < first_midi_note_init:
                # the current MIDI note is below the first MIDI note of the initial rank : need to apply a -1 octave pitch tuning
                change_factor = -1
            elif midi_note_nb > last_midi_note_init:
                # the current MIDI note is above the last MIDI note of the initial rank : need to apply a +1 octave pitch tuning
                change_factor = 1
            else:
                # the current MIDI note is within the initial MIDI notes range : no pitch tuning to apply
                change_factor = 0

            # determine which pipe of the rank has to be used to play the current MIDI note
            while 1:
                used_midi_note = midi_note_nb - 12 * change_factor

                if used_midi_note in range(first_midi_note_init, last_midi_note_init + 1):
                    # the note to use is inside the initial compass of the rank, take it
                    break

                if rank_accepts_retuning == 'Y' and abs(change_factor) == 1:
                    # the note to use is outside the initial compass of the rank with a pitch tuning of 1 x 1200 cents and the rank can be retuned
                    # the retuning cannot be more than 1 x 1200 cents, so none note can be used
                    used_midi_note = None
                    break

                if change_factor == 5:
                    # stop the factor increase loop at 5 x 1200 cents, none note can be used
                    used_midi_note = None
                    break

                # try to use a note with one octave of distance more
                if change_factor > 0:
                    change_factor += 1
                else:
                    change_factor -= 1

            # do the mapping between the current MIDI note and one existing pipe of the rank if possible
            if used_midi_note != None:
                # the note to use is inside the initial compass of the rank
                # determine the associated pipe
                used_pipe_nb = used_midi_note - first_midi_note_init + 1
                used_pipe_id = 'Pipe' + str(used_pipe_nb).zfill(3)

                # check if the pipe can be used, and if yes map it with the current MIDI note
                if mystr(self.object_attr_value_get(object_dic, used_pipe_id)).startswith('REF'):
                    # the pipe to use is borrowing another pipe using the REF:aa:bb:cc syntax
                    if change_factor != 0:
                        # the pipe to use must be retuned but it cannot because it is using the REF:aa:bb:cc syntax
                        midi_pipe_mapping_dic[midi_note_nb] = None
                        logs.add(f'{object_uid} : {used_pipe_id} (MIDI {used_midi_note}) cannot be used to play the MIDI note {midi_note_nb} as it is a borrowed pipe')
                    else:
                        midi_pipe_mapping_dic[midi_note_nb] = (used_pipe_nb, 0, 0)

                elif mystr(self.object_attr_value_get(object_dic, used_pipe_id + 'AcceptsRetuning'), rank_accepts_retuning) == 'Y':
                    # the pipe to use can be retuned, apply the change factor to its existing pitch tuning and gain values
                    pitch_tuning = 1200 * change_factor + myint(self.object_attr_value_get(object_dic, used_pipe_id + 'PitchTuning'), 0)
                    pipe_gain = max(min(-5.0 * change_factor + myfloat(self.object_attr_value_get(object_dic, used_pipe_id + 'Gain'), 0.0), 40.0), -120.0)

                    if abs(pitch_tuning) > 1800:
                        midi_pipe_mapping_dic[midi_note_nb] = None
                        logs.add(f'{object_uid} : {used_pipe_id} (MIDI {used_midi_note}) cannot be used to play the MIDI note {midi_note_nb}, a pitch tuning of {pitch_tuning} cents is necessary')
                    else:
                        midi_pipe_mapping_dic[midi_note_nb] = (used_pipe_nb, pitch_tuning, pipe_gain)

                else:
                    # the pipe to use cannot be retuned, let unchanged its existing pitch tuning and gain
                    pitch_tuning = myint(self.object_attr_value_get(object_dic, used_pipe_id + 'PitchTuning'), 0)
                    pipe_gain = myfloat(self.object_attr_value_get(object_dic, used_pipe_id + 'Gain'), 0)
                    midi_pipe_mapping_dic[midi_note_nb] = (used_pipe_nb, pitch_tuning, pipe_gain)

            else:
                # none pipe can be used
                midi_pipe_mapping_dic[midi_note_nb] = None
                logs.add(f'{object_uid} : none pipe of the rank can be used to play the MIDI note {midi_note_nb} with a pitch tuning of {1200 * change_factor} cents')

        # check if consecutive MIDI notes at the beginning or the end of the extended compass have no mapped pipe in order to remove them
        for midi_note_nb in list(midi_pipe_mapping_dic.keys()):
            if midi_pipe_mapping_dic[midi_note_nb] == None:
                midi_pipe_mapping_dic.pop(midi_note_nb)
            else:
                # stop the loop at the first mapped pipe
                break
        for midi_note_nb in reversed(list(midi_pipe_mapping_dic.keys())):
            if midi_pipe_mapping_dic[midi_note_nb] == None:
                midi_pipe_mapping_dic.pop(midi_note_nb)
            else:
                break
        # update the MIDI notes extended compass in case some dictionary entries have been removed above
        first_midi_note_ext = 999
        last_midi_note_ext = 0
        for midi_note_nb in midi_pipe_mapping_dic.keys():
            first_midi_note_ext = min(midi_note_nb, first_midi_note_ext)
            last_midi_note_ext  = max(midi_note_nb, last_midi_note_ext)

        if first_midi_note_ext == first_midi_note_init and last_midi_note_ext == last_midi_note_init:
            # no extension can be done finally
            logs.add(f'{object_uid} cannot be extended as none existing pipe can be reused')
            return first_midi_note_init, last_midi_note_init

        # create an object to put in it the extended rank/stop copied from the given rank/stop
        new_object_dic = self.object_new()
        new_object_dic['names'] = object_dic['names']
        new_object_dic['parents'] = object_dic['parents']
        new_object_dic['children'] = object_dic['children']
        # move in the extended rank/stop all the attributes of the current one except the pipe attributes
        for line in object_dic['lines']:
            if line[:4] != 'Pipe':
                new_object_dic['lines'].append(line)

        # build the pipe attributes of the extended rank/stop
        for (midi_note_nb, mapping_data) in midi_pipe_mapping_dic.items():
            # scan the MIDI notes of the mapping dictionary

            # define the pipe nb and ID that the current MIDI note must have in the new compass
            pipe_nb_new = midi_note_nb - first_midi_note_ext + 1
            pipe_id_new = 'Pipe' + str(pipe_nb_new).zfill(3)

            if mapping_data != None:
                # a pipe of the initial rank can be used to play the current MIDI note
                (pipe_nb_init, pitch_tuning, pipe_gain) = mapping_data

                pipe_id_init = 'Pipe' + str(pipe_nb_init).zfill(3)
                # recover all the attributes of this pipe
                pipe_lines_list = self.object_lines_search(object_uid, pipe_id_init)

                # add or update the Gain attribute in the pipe lines list
                if pipe_gain != 0:
                    gain_line_found = False
                    pipe_id_gain_init = pipe_id_init + 'Gain'
                    for i, line in enumerate(pipe_lines_list):
                        if line.startswith(pipe_id_gain_init):
                            # the pipe has the Gain attribute defined, update it
                            (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)
                            pipe_lines_list[i] = self.object_line_join(attr_name, pipe_gain, comment)
                            gain_line_found = True
                            break

                    if not gain_line_found:
                        # the pipe has not already the Gain attribute defined, add it in first position
                        pipe_lines_list.insert(0, pipe_id_gain_init + '=' + str(pipe_gain))

                # add or update the PitchTuning attribute in the pipe lines list if a pitch tuning has to be done
                if pitch_tuning != 0:
                    pitch_line_found = False
                    for i, line in enumerate(pipe_lines_list):
                        if line.startswith(pipe_id_init + 'PitchTuning='):
                            # the pipe has already the PitchTuning attribute defined, update it
                            (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)
                            pipe_lines_list[i] = self.object_line_join(attr_name, pitch_tuning, comment)
                            pitch_line_found = True
                            break

                    if not pitch_line_found:
                        # the pipe has not already the PitchTuning attribute defined, add it in first position
                        pipe_lines_list.insert(0, pipe_id_init + 'PitchTuning=' + str(pitch_tuning))

                # add all the attributes of the pipe to the new object, renaming the pipe ID if it is changed
                for line in pipe_lines_list:
                    if pipe_nb_new != pipe_nb_init:
                        line = line.replace(pipe_id_init, pipe_id_new)
                    new_object_dic['lines'].append(line)

            else:
                # a pipe of the initial rank cannot be used to play the current MIDI note, so set a dummy pipe
                new_object_dic['lines'].append(pipe_id_new + '=DUMMY')

        # delete in the ODF the current rank/stop
        self.odf_data_dic.pop(object_uid)

        # add in the ODF the extended rank/stop
        self.odf_data_dic[object_uid] = new_object_dic

        # update the compass related attributes of the Rank or Stop (with rank attributes inside)
        self.object_attr_value_set(new_object_dic, 'NumberOfLogicalPipes', last_midi_note_ext - first_midi_note_ext + 1)

        if object_type == 'Rank':
            self.object_attr_value_set(new_object_dic, 'FirstMidiNoteNumber', first_midi_note_ext)
        else:  # object_type == 'Stop'
            if myint(self.object_attr_value_get(new_object_dic, 'FirstMidiNoteNumber')) != None:
                # the attribute FirstMidiNoteNumber is defined in the Stop (it is not required in a Stop section), so update it
                self.object_attr_value_set(new_object_dic, 'FirstMidiNoteNumber', first_midi_note_ext)

        if last_midi_note_ext > last_midi_note_init:
            logs.add(f'{object_uid} MIDI notes compass extended from {first_midi_note_init}-{last_midi_note_init} to {first_midi_note_ext}-{last_midi_note_ext}')
        else:
            logs.add(f'{object_uid} compass not extended')

        return (first_midi_note_ext, last_midi_note_ext)


#-------------------------------------------------------------------------------------------------
class C_ODF_DATA_CHECK:
    # class to check the data (syntax, consistency) contained in the GO ODF data

    check_files_names = True    # flag indicating if files names have to be checked or not during the ODF data check
    checked_attr_nb = 0         # number of attributes checked during the checking operation

    #-------------------------------------------------------------------------------------------------
    def check_odf_data(self, progress_status_update_fct, files_names_to_check=True):
        # check the consistency of the data which are present in ODF data of the C_ODF_DATA class

        self.check_files_names = files_names_to_check
        self.checked_attr_nb = 0

        logs.add("ODF data check report :")

        # check the presence of the Organ object
        if 'Organ' not in self.odf_data_dic.keys():
            logs.add("ERROR the Organ section is not defined")

        for object_uid, object_dic in sorted(self.odf_data_dic.items()):
            # scan the objects of the ODF data

            # recover a copy of the lines of the current object
            object_lines_list = list(object_dic['lines'])

            # update in the GUI the name of the checked object
            progress_status_update_fct(f'Checking {object_uid}...')

            if len(object_lines_list) > 0:
                # lines have been recovered for the current object

                # sort the lines list to make faster the search which is done in check_attribute_value
                object_lines_list.sort()

                # remove the first line while it is empty (after the sorting the empty lines are all in first positions)
                while len(object_lines_list) > 0 and object_lines_list[0] == '':
                    object_lines_list.pop(0)

                # check if the attributes are all uniques in the object
                self.check_attributes_unicity(object_uid, object_lines_list)

                # check the attributes and values of the object by type
                object_type = self.object_type_get(object_uid)
                if object_type == 'Header':
                    pass
                elif object_type == 'Organ':
                    self.check_object_Organ(object_uid, object_lines_list)
                elif object_type == 'Coupler':
                    self.check_object_Coupler(object_uid, object_lines_list)
                elif object_type == 'Divisional':
                    self.check_object_Divisional(object_uid, object_lines_list)
                elif object_type == 'DivisionalCoupler':
                    self.check_object_DivisionalCoupler(object_uid, object_lines_list)
                elif object_type == 'Enclosure':
                    self.check_object_Enclosure(object_uid, object_lines_list)
                elif object_type == 'General':
                    self.check_object_General(object_uid, object_lines_list)
                elif object_type == 'Image':
                    self.check_object_Image(object_uid, object_lines_list)
                elif object_type == 'Label':
                    self.check_object_Label(object_uid, object_lines_list)
                elif object_type == 'Manual':
                    self.check_object_Manual(object_uid, object_lines_list)
                elif object_type == 'Panel':
                    self.check_object_Panel(object_uid, object_lines_list)
                elif object_type == 'PanelElement':
                    self.check_object_PanelElement(object_uid, object_lines_list)
                elif object_type[:5] == 'Panel': # Panel999Coupler999, Panel999Divisional999, Panel999Image999, ...
                    self.check_object_PanelOther(object_uid, object_lines_list)
                elif object_type == 'Rank':
                    self.check_object_Rank(object_uid, object_lines_list)
                elif object_type == 'ReversiblePiston':
                    self.check_object_ReversiblePiston(object_uid, object_lines_list)
                elif object_type == 'SetterElement':
                    self.check_object_SetterElement(object_uid, object_lines_list)
                elif object_type == 'Stop':
                    self.check_object_Stop(object_uid, object_lines_list)
                elif object_type == 'Switch':
                    self.check_object_Switch(object_uid, object_lines_list)
                elif object_type == 'Tremulant':
                    self.check_object_Tremulant(object_uid, object_lines_list)
                elif object_type == 'WindchestGroup':
                    self.check_object_WindchestGroup(object_uid, object_lines_list)
                else:
                    # the object UID has not been recognized
                    logs.add(f"WARNING the section {object_uid} has an unknown type")
                    # empty the lines list of the object which is not recognized, to not display in the log its attributes which have not been checked
                    object_lines_list = []

                # check the lines not checked by the function check_attribute_value() (that is which are still present in the lines list)
                for line in object_lines_list:
                    (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)
                    if error_msg != None:
                        logs.add(f'ERROR in {object_uid} section, line "{line}" : {error_msg}')

                    if attr_name not in (None, 'uid'):
                        # the current line is an attribute line
                        self.checked_attr_nb += 1
                        logs.add(f"WARNING in {object_uid} : the attribute {attr_name} is not expected in this section or is misspelled")

        # display in the log the number of checked attributes
        logs.add(f"{self.checked_attr_nb:,} attributes checked")

        # update the panel format flag
        self.check_panel_format()

        # display in the log if none error has been detected
        if logs.nb_get() <= 3:  # there are 3 log lines when no error : check start message + detected panel format + number of checked attributes
            logs.add("None error found, can be loaded in GrandOrgue for a final check")

    #-------------------------------------------------------------------------------------------------
    def check_panel_format(self):
        # check which is the panel format used in the ODF (new or old) and update the flag

        value = self.object_attr_value_get('Panel000', 'NumberOfGUIElements')
        self.new_panel_format_bool = (value != None and value.isdigit() and int(value) >= 0)

        if self.new_panel_format_bool:
            logs.add('New panel format')
        else:
            logs.add('Old panel format')

    #-------------------------------------------------------------------------------------------------
    def check_object_uid(self, object_uid):
        # return an error message if an issue has been detected in the given object UID, else None

        error_msg = None

        object_type = self.object_type_get(object_uid)

        if not object_uid.isalnum():
            error_msg = 'a section name can contain only alphanumeric characters'
        elif object_type not in self.go_objects_children_dic.keys():
            error_msg = f'{object_type} is an unknown section type'
        elif object_type not in ('Organ', 'Header'):
            if not object_uid[-3:].isdigit():
                error_msg = f'three digits are expected at the end of "{object_uid}"'
            elif int(object_uid[-3:]) == 0 and not object_type in ('Panel', 'Manual'):
                error_msg = f"{object_uid} cannot have the index 000"
            elif object_uid[:5] == 'Panel' and len(object_uid) > 8 and not object_uid[5:8].isdigit():
                # Panel999xxxxx999 object
                error_msg = f'three digits are expected after "Panel" in "{object_uid}"'

        return error_msg

    #-------------------------------------------------------------------------------------------------
    def check_object_line(self, line):
        # check the syntax of the given object line and extract from it the attribute name + attribute value + comment
        # return a tuple containing : (error message, attribute name, attribute value, comment)
        # attribute name = 'uid' if the given line contains an object UID between brackets, the UID is in the attribute value
        # error message = an error description message in case a syntax error has been detected in the given line, or None if no error found

        error_msg = attr_name = attr_value = comment = None

        if line != None and len(line) > 0: # not an empty line
            if line[0] == "[":
                # line with an object UID inside normally
                pos = line.find(']', 1)
                if pos == -1:  # object ID without closing bracket
                    error_msg = 'character "]" is missing to define an object ID'
                    comment = line
                elif pos == 1: # object ID with no string between the brackets
                    error_msg = 'no object identifier defined between the brackets'
                    comment = line
                else:
                    attr_name = 'uid'
                    attr_value = line[1:pos]
                    # check the coherency of the UID
                    error_msg = self.check_object_uid(attr_value)

                    if error_msg == None and len(line) > pos + 1:
                        # there are characters after the ]
                        comment = line[pos+1:]
                        if comment.lstrip()[0] != ';':
                            error_msg = 'only text beginning by ; is allowed after the ] character'

            elif line[0] != ";":  # not a comment line
                pos = line.find('=', 0)
                if pos == -1:  # no equal character in the line
                    error_msg = 'missing character ";" (comment) or "=" (attribute)'
                    comment = line
                elif pos == 0:  # the line starts by an equal character
                    error_msg = 'the character "=" cannot start a line'
                    comment = line
                elif line.find('=', pos+1) > pos:  # another equal character is present in the line
                    error_msg = 'more than one character "=" is defined'
                    comment = line
                else:
                    attr_name = line[0:pos]
                    for char in attr_name:
                        if char not in ALLOWED_CHARS_4_FIELDS:
                            # the attribute name has a forbiden character
                            error_msg = f'the attribute "{attr_name}" can contain only alphanumeric or "_" characters'
                            break

                    posc = line.find(';', pos + 1)
                    if posc != -1:
                        # there is a comment after the attribute value
                        while line[posc-1] == ' ': posc -= 1
                        attr_value = line[pos+1:posc]
                        comment = line[posc:]
                    else:
                        attr_value = line[pos+1:]
                    attr_value = attr_value.rstrip()

            else: # comment or empty line
                comment = line

        return (error_msg, attr_name, attr_value, comment)


    #-------------------------------------------------------------------------------------------------
    def check_object_Organ(self, object_uid, lines_list):
        # check the data of the Organ object which the lines are in the given lines list

        # required attributes
        self.check_attribute_value(object_uid, lines_list, 'ChurchName', ATTR_TYPE_STRING, True)
        self.check_attribute_value(object_uid, lines_list, 'ChurchAddress', ATTR_TYPE_STRING, True)

        value = self.check_attribute_value(object_uid, lines_list, 'HasPedals', ATTR_TYPE_BOOLEAN, True)
        if value == "Y" and not 'Manual000' in self.odf_data_dic:
            logs.add(f"ERROR in {object_uid} : HasPedals=Y but no Manual000 section is defined")
        elif value == "N" and 'Manual000' in self.odf_data_dic:
            logs.add(f"ERROR in {object_uid} : HasPedals=N whereas a Manual000 section is defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfDivisionalCouplers', ATTR_TYPE_INTEGER, True, 0, 8)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('DivisionalCoupler')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfDivisionalCouplers={value} whereas {count} DivisionalCoupler section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfEnclosures', ATTR_TYPE_INTEGER, True, 0, 999)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('Enclosure')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfEnclosures={value} whereas {count} Enclosure section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfGenerals', ATTR_TYPE_INTEGER, True, 0, 99)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('General')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfGenerals={value} whereas {count} General section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfManuals', ATTR_TYPE_INTEGER, True, 1, 16)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('Manual')
            if 'Manual000' in self.odf_data_dic.keys(): count -= 1
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfManuals={value} whereas {count} Manual section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfPanels', ATTR_TYPE_INTEGER, self.new_panel_format_bool, 0, 100)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('Panel')
            if 'Panel000' in self.odf_data_dic.keys(): count -= 1
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfPanels={value} whereas {count} Panel section(s) defined")

        if self.new_panel_format_bool and not 'Panel000' in self.odf_data_dic:
            logs.add("ERROR new panel format used but no Panel000 section is defined")
        elif not self.new_panel_format_bool and 'Panel000' in self.odf_data_dic:
            logs.add(f"ERROR in {object_uid} : old panel format used whereas a Panel000 is defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfReversiblePistons', ATTR_TYPE_INTEGER, True, 0, 32)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('ReversiblePiston')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfReversiblePistons={value} whereas {count} ReversiblePiston section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfTremulants', ATTR_TYPE_INTEGER, True, 0, 10)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('Tremulant')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfTremulants={value} whereas {count} Tremulant section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfWindchestGroups', ATTR_TYPE_INTEGER, True, 1, 999)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('WindchestGroup')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfWindchestGroups={value} whereas {count} WindchestGroup section(s) defined")

        self.check_attribute_value(object_uid, lines_list, 'DivisionalsStoreIntermanualCouplers', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'DivisionalsStoreIntramanualCouplers', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'DivisionalsStoreTremulants', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'GeneralsStoreDivisionalCouplers', ATTR_TYPE_BOOLEAN, True)

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'OrganBuilder', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'OrganBuildDate', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'OrganComments', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'RecordingDetails', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'InfoFilename', ATTR_TYPE_STRING, False)

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfImages', ATTR_TYPE_INTEGER, False, 0, 999) # old panel format
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('Image')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfImages={value} whereas {count} Image section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfLabels', ATTR_TYPE_INTEGER, False, 0, 999)  # old panel format
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('Label')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfLabels={value} whereas {count} Label section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfRanks', ATTR_TYPE_INTEGER, False, 0, 999)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('Rank')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfRanks={value} whereas {count} Rank section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfSetterElements', ATTR_TYPE_INTEGER, False, 0, 999)  # old panel format
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('SetterElement')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfSetterElements={value} whereas {count} SetterElement section(s) defined")

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfSwitches', ATTR_TYPE_INTEGER, False, 0, 999)
        if value != None and value.isdigit() and int(value) >= 0:
            count = self.objects_type_number_get('Switch')
            if count != int(value):
                logs.add(f"ERROR in {object_uid} : NumberOfSwitches={value} whereas {count} Switch section(s) defined")

        self.check_attribute_value(object_uid, lines_list, 'CombinationsStoreNonDisplayedDrawstops', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'AmplitudeLevel', ATTR_TYPE_FLOAT, False, 0, 1000)
        self.check_attribute_value(object_uid, lines_list, 'Gain', ATTR_TYPE_FLOAT, False, -120, 40)
        self.check_attribute_value(object_uid, lines_list, 'PitchTuning', ATTR_TYPE_FLOAT, False, -1800, 1800)
        self.check_attribute_value(object_uid, lines_list, 'PitchCorrection', ATTR_TYPE_FLOAT, False, -1800, 1800)
        self.check_attribute_value(object_uid, lines_list, 'TrackerDelay', ATTR_TYPE_FLOAT, False, 0, 10000)
        self.check_attribute_value(object_uid, lines_list, 'Percussive', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'HasIndependentRelease', ATTR_TYPE_BOOLEAN, False)


        if not self.new_panel_format_bool:
            # if old parnel format, the Organ object contains panel attributes
            self.check_object_Panel(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_Button(self, object_uid, lines_list):
        # check the data of a Button object section which the lines are in the given lines list

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'Name', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'ShortcutKey', ATTR_TYPE_INTEGER, False, 0, 255)
        self.check_attribute_value(object_uid, lines_list, 'StopControlMIDIKeyNumber', ATTR_TYPE_INTEGER, False, 0, 127)
        self.check_attribute_value(object_uid, lines_list, 'MIDIProgramChangeNumber', ATTR_TYPE_INTEGER, False, 1, 128)
        self.check_attribute_value(object_uid, lines_list, 'Displayed', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'DisplayInInvertedState', ATTR_TYPE_BOOLEAN, False)

        display_as_piston = self.check_attribute_value(object_uid, lines_list, 'DisplayAsPiston', ATTR_TYPE_BOOLEAN, False)
        if display_as_piston == '':
            # attribute not defined, set its default value
            if (object_uid.startwith(('Divisional', 'General')) or
                (object_uid[8:15] == 'Element' and self.object_attr_value_get(object_uid, 'Type') in ('Divisional', 'General'))):
                # the object is a Divisional or General button or a panel element of Divisional or General type, so it must be displayed as a piston by default
                display_as_piston = 'Y'
            else:
                display_as_piston = 'N'

        self.check_attribute_value(object_uid, lines_list, 'DispLabelColour', ATTR_TYPE_COLOR, False)
        self.check_attribute_value(object_uid, lines_list, 'DispLabelFontSize', ATTR_TYPE_FONT_SIZE, False)
        self.check_attribute_value(object_uid, lines_list, 'DispLabelFontName', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'DispLabelText', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'DispKeyLabelOnLeft', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'DispImageNum', ATTR_TYPE_INTEGER, False, 1, 5 if display_as_piston == 'Y' else 6)
        self.check_attribute_value(object_uid, lines_list, 'DispButtonRow', ATTR_TYPE_INTEGER, False, 0, 199)
        self.check_attribute_value(object_uid, lines_list, 'DispButtonCol', ATTR_TYPE_INTEGER, False, 1, 32)
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopRow', ATTR_TYPE_INTEGER, False, 1, 199)
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopCol', ATTR_TYPE_INTEGER, False, 1, 12)
        image_on = self.check_attribute_value(object_uid, lines_list, 'ImageOn', ATTR_TYPE_FILE_NAME, False)
        self.check_attribute_value(object_uid, lines_list, 'ImageOff', ATTR_TYPE_FILE_NAME, False)
        self.check_attribute_value(object_uid, lines_list, 'MaskOn', ATTR_TYPE_FILE_NAME, False)
        self.check_attribute_value(object_uid, lines_list, 'MaskOff', ATTR_TYPE_FILE_NAME, False)

        # get the dimensions of the parent panel
        panel_uid = self.object_parent_panel_get(object_uid)
        value = self.object_attr_value_get(panel_uid, 'DispScreenSizeHoriz')
        panel_width = int(value) if value != None and value.isdigit() else 3000
        value = self.object_attr_value_get(panel_uid, 'DispScreenSizeVert')
        panel_height = int(value) if value != None and value.isdigit() else 2000

        self.check_attribute_value(object_uid, lines_list, 'PositionX', ATTR_TYPE_INTEGER, False, 0, panel_width)
        self.check_attribute_value(object_uid, lines_list, 'PositionY', ATTR_TYPE_INTEGER, False, 0, panel_height)
        width = self.check_attribute_value(object_uid, lines_list, 'Width', ATTR_TYPE_INTEGER, False, 0, panel_width)
        height = self.check_attribute_value(object_uid, lines_list, 'Height', ATTR_TYPE_INTEGER, False, 0, panel_height)
        max_width = int(width) if width != None and width.isdigit() else panel_width
        max_height = int(height) if height != None and height.isdigit() else panel_height

        # get the dimensions of the button bitmap
        if image_on not in (None, ''):
            # an image is defined to display the button
            if self.check_files_names:
                # get the sizes of the image in the file which is existing
                im = Image.open(os.path.dirname(self.odf_file_name) + os.path.sep + path2ospath(image_on))
                bitmap_width = im.size[0]
                bitmap_height = im.size[1]
            else:
                bitmap_width = 500  # arbritrary default value
                bitmap_height = 200 # arbritrary default value
        else:
            # no image file defined, get the dimensions of the internal bitmap (piston or drawstop)
            if display_as_piston == 'Y':
                bitmap_width = bitmap_height = 32
            else:
                bitmap_width = bitmap_height = 62

        self.check_attribute_value(object_uid, lines_list, 'TileOffsetX', ATTR_TYPE_INTEGER, False, 0, bitmap_width)
        self.check_attribute_value(object_uid, lines_list, 'TileOffsetY', ATTR_TYPE_INTEGER, False, 0, bitmap_height)

        self.check_attribute_value(object_uid, lines_list, 'MouseRectLeft', ATTR_TYPE_INTEGER, False, 0, max_width)
        self.check_attribute_value(object_uid, lines_list, 'MouseRectTop', ATTR_TYPE_INTEGER, False, 0, max_height)
        mouse_rect_width = self.check_attribute_value(object_uid, lines_list, 'MouseRectWidth', ATTR_TYPE_INTEGER, False, 0, max_width)
        mouse_rect_height = self.check_attribute_value(object_uid, lines_list, 'MouseRectHeight', ATTR_TYPE_INTEGER, False, 0, max_height)

        if mouse_rect_width != None and mouse_rect_width.isdigit() and mouse_rect_height != None and  mouse_rect_height.isdigit():
            mouse_radius = max(int(mouse_rect_width), int(mouse_rect_height))
        else:
            mouse_radius = max(bitmap_width, bitmap_height)
        self.check_attribute_value(object_uid, lines_list, 'MouseRadius', ATTR_TYPE_INTEGER, False, 0, mouse_radius)

        self.check_attribute_value(object_uid, lines_list, 'TextRectLeft', ATTR_TYPE_INTEGER, False, 0, max_width)
        self.check_attribute_value(object_uid, lines_list, 'TextRectTop', ATTR_TYPE_INTEGER, False, 0, max_height)
        text_rect_width = self.check_attribute_value(object_uid, lines_list, 'TextRectWidth', ATTR_TYPE_INTEGER, False, 0, max_width)
        self.check_attribute_value(object_uid, lines_list, 'TextRectHeight', ATTR_TYPE_INTEGER, False, 0, max_height)

        if text_rect_width != None and text_rect_width.isdigit():
            text_break_width = int(text_rect_width)
        else:
            text_break_width = bitmap_width
        self.check_attribute_value(object_uid, lines_list, 'TextBreakWidth', ATTR_TYPE_INTEGER, False, 0, text_break_width)

    #-------------------------------------------------------------------------------------------------
    def check_object_Coupler(self, object_uid, lines_list):
        # check the data of a Coupler object section which the lines are in the given lines list

        is_coupler_obj = self.object_type_get(object_uid) == 'Coupler'

        # required attributes
        ret1 = self.check_attribute_value(object_uid, lines_list, 'UnisonOff', ATTR_TYPE_BOOLEAN, is_coupler_obj)
        ret2 = self.check_attribute_value(object_uid, lines_list, 'CouplerType', ATTR_TYPE_COUPLER_TYPE, False)  # optional but placed here to recover its value used after
        self.check_attribute_value(object_uid, lines_list, 'DestinationManual', ATTR_TYPE_INTEGER, ret1 == 'N', 0, 16) # conditional required/optional
        self.check_attribute_value(object_uid, lines_list, 'DestinationKeyshift', ATTR_TYPE_INTEGER, ret1 == 'N', -24, 24) # conditional required/optional

        is_required = (ret1 != None and ret2 != None and ret1 == 'N' and ret2.upper() not in ('MELODY', 'BASS'))
        self.check_attribute_value(object_uid, lines_list, 'CoupleToSubsequentUnisonIntermanualCouplers', ATTR_TYPE_BOOLEAN, is_required)
        self.check_attribute_value(object_uid, lines_list, 'CoupleToSubsequentUpwardIntermanualCouplers', ATTR_TYPE_BOOLEAN, is_required)
        self.check_attribute_value(object_uid, lines_list, 'CoupleToSubsequentDownwardIntermanualCouplers', ATTR_TYPE_BOOLEAN, is_required)
        self.check_attribute_value(object_uid, lines_list, 'CoupleToSubsequentUpwardIntramanualCouplers', ATTR_TYPE_BOOLEAN, is_required)
        self.check_attribute_value(object_uid, lines_list, 'CoupleToSubsequentDownwardIntramanualCouplers', ATTR_TYPE_BOOLEAN, is_required)

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'FirstMIDINoteNumber', ATTR_TYPE_INTEGER, False, 0, 127)
        self.check_attribute_value(object_uid, lines_list, 'NumberOfKeys', ATTR_TYPE_INTEGER, False, 0, 127)

        # a Coupler has in addition the attributes of a DrawStop
        self.check_object_DrawStop(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_Divisional(self, object_uid, lines_list):
        # check the data of a Divisional object section which the lines are in the given lines list

        is_divisional_obj = self.object_type_get(object_uid) == 'Divisional'

        # recover the ID of manual in which is referenced this Divisional
        parent_manual_uid = self.object_parent_manual_get(object_uid)

        # required attributes
        value = self.object_attr_value_get(parent_manual_uid, 'NumberOfCouplers')
        max_val = int(value) if value != None and value.isdigit() else 999
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfCouplers', ATTR_TYPE_INTEGER, is_divisional_obj, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Coupler{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        value = self.object_attr_value_get(parent_manual_uid, 'NumberOfStops')
        max_val = int(value) if value != None and value.isdigit() else 999
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfStops', ATTR_TYPE_INTEGER, is_divisional_obj, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Stop{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        value = self.object_attr_value_get(parent_manual_uid, 'NumberOfTremulants')
        max_val = int(value) if value != None and value.isdigit() else 10
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfTremulants', ATTR_TYPE_INTEGER, is_divisional_obj, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Tremulant{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'Protected', ATTR_TYPE_BOOLEAN, False)

        value = self.object_attr_value_get(parent_manual_uid, 'NumberOfSwitches')
        max_val = int(value) if value != None and value.isdigit() else 999
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfSwitches', ATTR_TYPE_INTEGER, False, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Switch{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        # a Divisional has in addition the attributes of a Push Button
        self.check_object_PushButton(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_DivisionalCoupler(self, object_uid, lines_list):
        # check the data of a Divisional Coupler object section which the lines are in the given lines list

        is_divisional_coupler_obj = self.object_type_get(object_uid) == 'DivisionalCoupler'

        # required attributes
        self.check_attribute_value(object_uid, lines_list, 'BiDirectionalCoupling', ATTR_TYPE_BOOLEAN, is_divisional_coupler_obj)

        value = self.object_attr_value_get('Organ', 'NumberOfManuals')
        max_val = int(value) if value != None and value.isdigit() else 16
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfManuals', ATTR_TYPE_INTEGER, is_divisional_coupler_obj, 1, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f"Manual{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)

        # a Divisional Coupler has in addition the attributes of a DrawStop
        self.check_object_DrawStop(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_DrawStop(self, object_uid, lines_list):
        # check the data of a DrawStop object section which the lines are in the given lines list

        # required attributes

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'DefaultToEngaged', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'Function', ATTR_TYPE_DRAWSTOP_FCT, False)

        max_val = myint(self.object_attr_value_get('Organ', 'NumberOfSwitches'), 999)
        switch_id = int(object_uid[-3:]) if (object_uid[-3:].isdigit() and object_uid[:-3] == 'Switch') else 999
        switch_nb = myint(self.check_attribute_value(object_uid, lines_list, 'SwitchCount', ATTR_TYPE_INTEGER, False, 1, max_val), 0)
        function = self.object_attr_value_get(object_uid, 'Function')
        if function == 'Not':
            if switch_nb > 1:
                logs.add(f'ERROR in {object_uid} section, a NOT switch cannot have more than one switch input')
            switch_nb = 1

        for idx in range(1, switch_nb + 1):
            attr_value = self.check_attribute_value(object_uid, lines_list, f"Switch{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
            if switch_id != 999 and int(attr_value) >= switch_id:
                # the given object is a Switch and it refers to another switch which has an higher ID than it
                logs.add(f'ERROR in {object_uid} section, cannot reference a switch having an equal or higher number')

        self.check_attribute_value(object_uid, lines_list, 'GCState', ATTR_TYPE_INTEGER, False, -1, 1)
        self.check_attribute_value(object_uid, lines_list, 'StoreInDivisional', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'StoreInGeneral', ATTR_TYPE_BOOLEAN, False)

        # a Drawstop has in addition the attributes of a Button
        self.check_object_Button(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_Enclosure(self, object_uid, lines_list):
        # check the data of an Enclosure object section which the lines are in the given lines list

        # required attributes
        # none required attribute

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'Name', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'AmpMinimumLevel', ATTR_TYPE_INTEGER, False, 0, 100)
        self.check_attribute_value(object_uid, lines_list, 'MIDIInputNumber', ATTR_TYPE_INTEGER, False, 0, 100)
        self.check_attribute_value(object_uid, lines_list, 'Displayed', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'DispLabelColour', ATTR_TYPE_COLOR, False)
        self.check_attribute_value(object_uid, lines_list, 'DispLabelFontSize', ATTR_TYPE_FONT_SIZE, False)
        self.check_attribute_value(object_uid, lines_list, 'DispLabelFontName', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'DispLabelText', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'EnclosureStyle', ATTR_TYPE_INTEGER, False, 1, 4)

        value = self.check_attribute_value(object_uid, lines_list, 'BitmapCount', ATTR_TYPE_INTEGER, False, 1, 128)
        if value != None and value.isdigit():
            image = None
            for idx in range(1, int(value)+1):
                image = self.check_attribute_value(object_uid, lines_list, f'Bitmap{str(idx).zfill(3)}', ATTR_TYPE_FILE_NAME, True)
                self.check_attribute_value(object_uid, lines_list, f'Mask{str(idx).zfill(3)}', ATTR_TYPE_FILE_NAME, False)
            # get the dimensions of the last enclosure bitmap
            if image != None and image != '' and self.check_files_names:
                # an image is defined to display the enclosure
                # get the sizes of the image in the file which is existing
                im = Image.open(os.path.dirname(self.odf_file_name) + os.path.sep + path2ospath(image))
                bitmap_width = im.size[0]
                bitmap_height = im.size[1]
            else:
                bitmap_width = 100  # arbitrary default value
                bitmap_height = 200 # arbitrary default value
        else:
            # no image file defined, get the dimensions of the internal bitmap
            bitmap_width = 46
            bitmap_height = 61

        # get the dimensions of the parent panel
        panel_uid = self.object_parent_panel_get(object_uid)
        value = self.object_attr_value_get(panel_uid, 'DispScreenSizeHoriz')
        panel_width = int(value) if value != None and value.isdigit() else 3000
        value = self.object_attr_value_get(panel_uid, 'DispScreenSizeVert')
        panel_height = int(value) if value != None and value.isdigit() else 2000

        self.check_attribute_value(object_uid, lines_list, 'PositionX', ATTR_TYPE_INTEGER, False, 0, panel_width)
        self.check_attribute_value(object_uid, lines_list, 'PositionY', ATTR_TYPE_INTEGER, False, 0, panel_height)
        width = self.check_attribute_value(object_uid, lines_list, 'Width', ATTR_TYPE_INTEGER, False, 0, panel_width)
        height = self.check_attribute_value(object_uid, lines_list, 'Height', ATTR_TYPE_INTEGER, False, 0, panel_height)
        max_width = int(width) if width != None and width.isdigit() else panel_width
        max_height = int(height) if height != None and height.isdigit() else panel_height

        self.check_attribute_value(object_uid, lines_list, 'TileOffsetX', ATTR_TYPE_INTEGER, False, 0, bitmap_width)
        self.check_attribute_value(object_uid, lines_list, 'TileOffsetY', ATTR_TYPE_INTEGER, False, 0, bitmap_height)

        self.check_attribute_value(object_uid, lines_list, 'MouseRectLeft', ATTR_TYPE_INTEGER, False, 0, max_width)
        self.check_attribute_value(object_uid, lines_list, 'MouseRectTop', ATTR_TYPE_INTEGER, False, 0, max_height)
        self.check_attribute_value(object_uid, lines_list, 'MouseRectWidth', ATTR_TYPE_INTEGER, False, 0, max_width)
        mouse_rect_height = self.check_attribute_value(object_uid, lines_list, 'MouseRectHeight', ATTR_TYPE_INTEGER, False, 0, max_height)

        if mouse_rect_height != None and mouse_rect_height.isdigit():
            max_start = int(mouse_rect_height)
        else:
            max_start = 200
        mouse_axis_start = self.check_attribute_value(object_uid, lines_list, 'MouseAxisStart', ATTR_TYPE_INTEGER, False, 0, max_start)

        if mouse_axis_start != None and mouse_axis_start.isdigit():
            min_end = int(mouse_axis_start)
        else:
            min_end = 200
        self.check_attribute_value(object_uid, lines_list, 'MouseAxisEnd', ATTR_TYPE_INTEGER, False, min_end, max_start)

        self.check_attribute_value(object_uid, lines_list, 'TextRectLeft', ATTR_TYPE_INTEGER, False, 0, max_width)
        self.check_attribute_value(object_uid, lines_list, 'TextRectTop', ATTR_TYPE_INTEGER, False, 0, max_height)
        text_rect_width = self.check_attribute_value(object_uid, lines_list, 'TextRectWidth', ATTR_TYPE_INTEGER, False, 0, max_width)
        self.check_attribute_value(object_uid, lines_list, 'TextRectHeight', ATTR_TYPE_INTEGER, False, 0, max_height)

        if text_rect_width != None and text_rect_width.isdigit():
            text_break_width = int(text_rect_width)
        else:
            text_break_width = bitmap_width
        self.check_attribute_value(object_uid, lines_list, 'TextBreakWidth', ATTR_TYPE_INTEGER, False, 0, text_break_width)

    #-------------------------------------------------------------------------------------------------
    def check_object_General(self, object_uid, lines_list):
        # check the data of a General object section which the lines are in the given lines list

        is_general_obj = self.object_type_get(object_uid) == 'General' # some mandatory attributes are not mandatory for objects which inherit the General attributes
        store_div_coupl_in_gen = self.object_attr_value_get(object_uid, 'GeneralsStoreDivisionalCouplers')

        # required attributes
        max_val = self.objects_type_number_get('Coupler')
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfCouplers', ATTR_TYPE_INTEGER, is_general_obj, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'CouplerNumber{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)
                self.check_attribute_value(object_uid, lines_list, f'CouplerManual{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        max_val = self.objects_type_number_get('DivisionalCoupler')
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfDivisionalCouplers', ATTR_TYPE_INTEGER, is_general_obj and store_div_coupl_in_gen == 'Y', 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'DivisionalCouplerNumber{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        max_val = self.objects_type_number_get('Stop')
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfStops', ATTR_TYPE_INTEGER, is_general_obj, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'StopNumber{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)
                self.check_attribute_value(object_uid, lines_list, f'StopManual{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        max_val = self.objects_type_number_get('Tremulant')
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfTremulants', ATTR_TYPE_INTEGER, is_general_obj, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'TremulantNumber{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        # optional attributes
        max_val = self.objects_type_number_get('Switch')
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfSwitches', ATTR_TYPE_INTEGER, False, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'SwitchNumber{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        self.check_attribute_value(object_uid, lines_list, 'Protected', ATTR_TYPE_BOOLEAN, False)

        # a General has in addition the attributes of a Push Button
        self.check_object_PushButton(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_Image(self, object_uid, lines_list):
        # check the data of an Image object section which the lines are in the given lines list

        # required attributes
        image = self.check_attribute_value(object_uid, lines_list, 'Image', ATTR_TYPE_FILE_NAME, True)

        # get the dimensions of the parent panel
        parent_panel_uid = self.object_parent_panel_get(object_uid)
        value = self.object_attr_value_get(parent_panel_uid, 'DispScreenSizeHoriz')
        panel_width = int(value) if value != None and value.isdigit() else 3000
        value = self.object_attr_value_get(parent_panel_uid, 'DispScreenSizeVert')
        panel_height = int(value) if value != None and value.isdigit() else 2000

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'Mask', ATTR_TYPE_FILE_NAME, False)
        self.check_attribute_value(object_uid, lines_list, 'PositionX', ATTR_TYPE_INTEGER, False, 0, panel_width)
        self.check_attribute_value(object_uid, lines_list, 'PositionY', ATTR_TYPE_INTEGER, False, 0, panel_height)
        self.check_attribute_value(object_uid, lines_list, 'Width', ATTR_TYPE_INTEGER, False, 0, panel_width)
        self.check_attribute_value(object_uid, lines_list, 'Height', ATTR_TYPE_INTEGER, False, 0, panel_height)

        # get the dimensions of the image bitmap
        if image not in (None, ''):
            # an image is defined
            if self.check_files_names:
                # get the sizes of the image in the file which is existing
                im = Image.open(os.path.dirname(self.odf_file_name) + os.path.sep + path2ospath(image))
                bitmap_width = im.size[0]
                bitmap_height = im.size[1]
            else:
                bitmap_width = panel_width
                bitmap_height = panel_height
        else:
            # no image file defined
            bitmap_width = panel_width
            bitmap_height = panel_height

        self.check_attribute_value(object_uid, lines_list, 'TileOffsetX', ATTR_TYPE_INTEGER, False, 0, bitmap_width)
        self.check_attribute_value(object_uid, lines_list, 'TileOffsetY', ATTR_TYPE_INTEGER, False, 0, bitmap_height)

    #-------------------------------------------------------------------------------------------------
    def check_object_Label(self, object_uid, lines_list):
        # check the data of a Label object section which the lines are in the given lines list

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'Name', ATTR_TYPE_STRING, False)
        ret1 = self.check_attribute_value(object_uid, lines_list, 'FreeXPlacement', ATTR_TYPE_BOOLEAN, False)
        ret2 = self.check_attribute_value(object_uid, lines_list, 'FreeYPlacement', ATTR_TYPE_BOOLEAN, False)

        # get the dimensions of the parent panel
        parent_panel_uid = self.object_parent_panel_get(object_uid)
        value = self.object_attr_value_get(parent_panel_uid, 'DispScreenSizeHoriz')
        panel_width = int(value) if value != None and value.isdigit() else 3000
        value = self.object_attr_value_get(parent_panel_uid, 'DispScreenSizeVert')
        panel_height = int(value) if value != None and value.isdigit() else 2000

        self.check_attribute_value(object_uid, lines_list, 'DispXpos', ATTR_TYPE_INTEGER, False, 0, panel_width)
        self.check_attribute_value(object_uid, lines_list, 'DispYpos', ATTR_TYPE_INTEGER, False, 0, panel_height)

        self.check_attribute_value(object_uid, lines_list, 'DispAtTopOfDrawstopCol', ATTR_TYPE_BOOLEAN, ret2 == 'N')

        # get the number of drawstop columns in the parent panel
        value = self.object_attr_value_get(parent_panel_uid, 'DispDrawstopCols')
        columns_nb = int(value) if value != None and value.isdigit() else 12
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopCol', ATTR_TYPE_INTEGER, ret1 == 'N', 1, columns_nb)

        self.check_attribute_value(object_uid, lines_list, 'DispSpanDrawstopColToRight', ATTR_TYPE_BOOLEAN, ret1 == 'N')
        self.check_attribute_value(object_uid, lines_list, 'DispLabelColour', ATTR_TYPE_COLOR, False)
        self.check_attribute_value(object_uid, lines_list, 'DispLabelFontSize', ATTR_TYPE_FONT_SIZE, False)
        self.check_attribute_value(object_uid, lines_list, 'DispLabelFontName', ATTR_TYPE_STRING, False)
        image_num = self.check_attribute_value(object_uid, lines_list, 'DispImageNum', ATTR_TYPE_INTEGER, False, 0, 12)
        image = self.check_attribute_value(object_uid, lines_list, 'Image', ATTR_TYPE_FILE_NAME, False)
        self.check_attribute_value(object_uid, lines_list, 'Mask', ATTR_TYPE_FILE_NAME, False)

        self.check_attribute_value(object_uid, lines_list, 'PositionX', ATTR_TYPE_INTEGER, False, 0, panel_width)
        self.check_attribute_value(object_uid, lines_list, 'PositionY', ATTR_TYPE_INTEGER, False, 0, panel_height)
        width = self.check_attribute_value(object_uid, lines_list, 'Width', ATTR_TYPE_INTEGER, False, 0, panel_width)
        height = self.check_attribute_value(object_uid, lines_list, 'Height', ATTR_TYPE_INTEGER, False, 0, panel_height)
        max_width = int(width) if width != None and width.isdigit() else panel_width
        max_height = int(height) if height != None and height.isdigit() else panel_height

        # get the dimensions of the label bitmap
        if image not in (None, ''):
            # an image is defined to display the label
            if self.check_files_names:
                # get the sizes of the image in the file which is existing
                im = Image.open(os.path.dirname(self.odf_file_name) + os.path.sep + path2ospath(image))
                bitmap_width = im.size[0]
                bitmap_height = im.size[1]
            else:
                bitmap_width = 400  # arbritrary default value
                bitmap_height = 100 # arbritrary default value
        else:
            if   image_num == '1':  bitmap_width = 80; bitmap_height = 25
            elif image_num == '2':  bitmap_width = 80; bitmap_height = 50
            elif image_num == '3':  bitmap_width = 80; bitmap_height = 25
            elif image_num == '4':  bitmap_width = 160; bitmap_height = 25
            elif image_num == '5':  bitmap_width = 200; bitmap_height = 50
            elif image_num == '6':  bitmap_width = 80; bitmap_height = 50
            elif image_num == '7':  bitmap_width = 80; bitmap_height = 25
            elif image_num == '8':  bitmap_width = 160; bitmap_height = 25
            elif image_num == '9':  bitmap_width = 80; bitmap_height = 50
            elif image_num == '10': bitmap_width = 80; bitmap_height = 25
            elif image_num == '11': bitmap_width = 160; bitmap_height = 25
            else:                   bitmap_width = 200; bitmap_height = 50


        self.check_attribute_value(object_uid, lines_list, 'TileOffsetX', ATTR_TYPE_INTEGER, False, 0, bitmap_width)
        self.check_attribute_value(object_uid, lines_list, 'TileOffsetY', ATTR_TYPE_INTEGER, False, 0, bitmap_height)

        self.check_attribute_value(object_uid, lines_list, 'TextRectLeft', ATTR_TYPE_INTEGER, False, 0, max_width)
        self.check_attribute_value(object_uid, lines_list, 'TextRectTop', ATTR_TYPE_INTEGER, False, 0, max_height)
        text_rect_width = self.check_attribute_value(object_uid, lines_list, 'TextRectWidth', ATTR_TYPE_INTEGER, False, 0, max_width)
        self.check_attribute_value(object_uid, lines_list, 'TextRectHeight', ATTR_TYPE_INTEGER, False, 0, max_height)

        if text_rect_width != None and text_rect_width.isdigit():
            text_break_width = int(text_rect_width)
        else:
            text_break_width = bitmap_width
        self.check_attribute_value(object_uid, lines_list, 'TextBreakWidth', ATTR_TYPE_INTEGER, False, 0, text_break_width)

    #-------------------------------------------------------------------------------------------------
    def check_object_Manual(self, object_uid, lines_list):
        # check the data of a Manual object section which the lines are in the given lines list

        is_manual_obj = self.object_type_get(object_uid) == 'Manual' # some mandatory attributes are not mandatory for objects which inherit the Manual attributes

        if not is_manual_obj:
            # object_uid is a PanelElement object with Type=Manual, get the UID of the linked Manual
            manual_uid = self.object_attr_value_get(object_uid, 'Manual')
            if manual_uid != None:
                manual_uid = 'Manual' + str(int(manual_uid)).zfill(3)
        else:
            manual_uid = None

        # required attributes
        self.check_attribute_value(object_uid, lines_list, 'Name', ATTR_TYPE_STRING, is_manual_obj)
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfLogicalKeys', ATTR_TYPE_INTEGER, is_manual_obj, 1, 192)
        if value == None and manual_uid != None:
            # recover the number of accessible keys in the linked Manual object
            value = self.object_attr_value_get(manual_uid, 'NumberOfAccessibleKeys')

        if value != None and value.isdigit() and int(value) > 0:
            for idx in range(1, int(value) + 1):
                # attributes Key999xxx
                image = self.check_attribute_value(object_uid, lines_list, f'Key{str(idx).zfill(3)}ImageOn', ATTR_TYPE_FILE_NAME, False)
                if image not in (None, ''):
                    # check the other attributes for this key only if an image on is defined
                    key_id = f'Key{str(idx).zfill(3)}'
                    self.check_attribute_value(object_uid, lines_list, key_id + 'ImageOff', ATTR_TYPE_FILE_NAME, False)
                    self.check_attribute_value(object_uid, lines_list, key_id + 'MaskOn', ATTR_TYPE_FILE_NAME, False)
                    self.check_attribute_value(object_uid, lines_list, key_id + 'MaskOff', ATTR_TYPE_FILE_NAME, False)
                    self.check_attribute_value(object_uid, lines_list, key_id + 'Width', ATTR_TYPE_INTEGER, False, 0, 500)
                    self.check_attribute_value(object_uid, lines_list, key_id + 'Offset', ATTR_TYPE_INTEGER, False, -500, 500)
                    self.check_attribute_value(object_uid, lines_list, key_id + 'YOffset', ATTR_TYPE_INTEGER, False, -500, 500)

                    # get the dimensions of the key bitmap
                    # an image is defined to display the key
                    if self.check_files_names:
                        # get the sizes of the image in the file which is existing
                        im = Image.open(os.path.dirname(self.odf_file_name) + os.path.sep + path2ospath(image))
                        bitmap_width = im.size[0]
                        bitmap_height = im.size[1]
                    else:
                        bitmap_width = 100  # arbritrary default value
                        bitmap_height = 300 # arbritrary default value

                    self.check_attribute_value(object_uid, lines_list, key_id + 'MouseRectLeft', ATTR_TYPE_INTEGER, False, 0, bitmap_width)
                    self.check_attribute_value(object_uid, lines_list, key_id + 'MouseRectTop', ATTR_TYPE_INTEGER, False, 0, bitmap_height)
                    self.check_attribute_value(object_uid, lines_list, key_id + 'MouseRectWidth', ATTR_TYPE_INTEGER, False, 0, bitmap_width)
                    self.check_attribute_value(object_uid, lines_list, key_id + 'MouseRectHeight', ATTR_TYPE_INTEGER, False, 0, bitmap_height)

        logical_keys_nb = int(value) if value != None and value.isdigit() else 192
        self.check_attribute_value(object_uid, lines_list, 'FirstAccessibleKeyLogicalKeyNumber', ATTR_TYPE_INTEGER, is_manual_obj, 1, logical_keys_nb)
        self.check_attribute_value(object_uid, lines_list, 'FirstAccessibleKeyMIDINoteNumber', ATTR_TYPE_INTEGER, is_manual_obj, 0, 127)

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfAccessibleKeys', ATTR_TYPE_INTEGER, is_manual_obj, 0, 85)
        accessible_keys_nb = int(value) if value != None and value.isdigit() else 85

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfCouplers', ATTR_TYPE_INTEGER, False, 0, 999)
        if value != None and value.isdigit() and int(value) > 0:
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Coupler{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfDivisionals', ATTR_TYPE_INTEGER, False, 0, 999)
        if value != None and value.isdigit() and int(value) > 0:
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Divisional{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfStops', ATTR_TYPE_INTEGER, False, 0, 999)
        if value != None and value.isdigit() and int(value) > 0:
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Stop{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        max_val = self.objects_type_number_get('Switch')
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfSwitches', ATTR_TYPE_INTEGER, False, 0, max_val)
        if value != None and value.isdigit() and int(value) > 0:
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Switch{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        max_val = self.objects_type_number_get('Tremulant')
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfTremulants', ATTR_TYPE_INTEGER, False, 0, max_val)
        if value != None and value.isdigit() and int(value) > 0:
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Tremulant{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        # optional attributes
        for idx in range(0, 128):
            self.check_attribute_value(object_uid, lines_list, f'MIDIKey{str(idx).zfill(3)}', ATTR_TYPE_INTEGER, False, 0, 127)

        self.check_attribute_value(object_uid, lines_list, 'MIDIInputNumber', ATTR_TYPE_INTEGER, False, 0, 200)
        self.check_attribute_value(object_uid, lines_list, 'Displayed', ATTR_TYPE_BOOLEAN, False)

        # get the dimensions of the parent panel
        parent_panel_uid = self.object_parent_panel_get(object_uid)
        value = self.object_attr_value_get(parent_panel_uid, 'DispScreenSizeHoriz')
        panel_width = int(value) if value != None and value.isdigit() else 3000
        value = self.object_attr_value_get(parent_panel_uid, 'DispScreenSizeVert')
        panel_height = int(value) if value != None and value.isdigit() else 2000

        self.check_attribute_value(object_uid, lines_list, 'PositionX', ATTR_TYPE_INTEGER, False, 0, panel_width)
        self.check_attribute_value(object_uid, lines_list, 'PositionY', ATTR_TYPE_INTEGER, False, 0, panel_height)

        self.check_attribute_value(object_uid, lines_list, 'DispKeyColourInverted', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'DispKeyColourWooden', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'DisplayFirstNote', ATTR_TYPE_INTEGER, False, 0, 127)

        value = self.check_attribute_value(object_uid, lines_list, 'DisplayKeys', ATTR_TYPE_INTEGER, False, 1, accessible_keys_nb)
        if value != None and value.isdigit() and int(value) > 0:
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'DisplayKey{str(idx).zfill(3)}', ATTR_TYPE_INTEGER, False, 0, 127)
                self.check_attribute_value(object_uid, lines_list, f'DisplayKey{str(idx).zfill(3)}Note', ATTR_TYPE_INTEGER, False, 0, 127)

        # optional attributes with the KEYTYPE format
        ImageOn_First_keytype = None # variable to store if the first attribute have been already checked for the ImageOn key type
        ImageOff_First_keytype = None
        MaskOn_First_keytype = None
        MaskOff_First_keytype = None
        Width_First_keytype = None
        Offset_First_keytype = None
        YOffset_First_keytype = None

        ImageOn_Last_keytype = None
        ImageOff_Last_keytype = None
        MaskOn_Last_keytype = None
        MaskOff_Last_keytype = None
        Width_Last_keytype = None
        Offset_Last_keytype = None
        YOffset_Last_keytype = None

        if object_uid == 'Panel000Element001':
            pass

        for keytype in ('C', 'Cis', 'D', 'Dis', 'E', 'F', 'Fis', 'G', 'Gis', 'A', 'Ais', 'B'):
            self.check_attribute_value(object_uid, lines_list, f'ImageOn_{keytype}', ATTR_TYPE_FILE_NAME, False)
            self.check_attribute_value(object_uid, lines_list, f'ImageOff_{keytype}', ATTR_TYPE_FILE_NAME, False)
            self.check_attribute_value(object_uid, lines_list, f'MaskOn_{keytype}', ATTR_TYPE_FILE_NAME, False)
            self.check_attribute_value(object_uid, lines_list, f'MaskOff_{keytype}', ATTR_TYPE_FILE_NAME, False)
            self.check_attribute_value(object_uid, lines_list, f'Width_{keytype}', ATTR_TYPE_INTEGER, False, 0, 500)
            self.check_attribute_value(object_uid, lines_list, f'Offset_{keytype}', ATTR_TYPE_INTEGER, False, -500, 500)
            self.check_attribute_value(object_uid, lines_list, f'YOffset_{keytype}', ATTR_TYPE_INTEGER, False, -500, 500)
            # the First and Last attributes are checked only once for each key property
            # so if there is more than one First or Last definition it will appear in the warning logs because it will not have been checked here
            if ImageOn_First_keytype == None : ImageOn_First_keytype = self.check_attribute_value(object_uid, lines_list, f'ImageOn_First{keytype}', ATTR_TYPE_FILE_NAME, False)
            if ImageOff_First_keytype == None : ImageOff_First_keytype = self.check_attribute_value(object_uid, lines_list, f'ImageOff_First{keytype}', ATTR_TYPE_FILE_NAME, False)
            if MaskOn_First_keytype == None : MaskOn_First_keytype = self.check_attribute_value(object_uid, lines_list, f'MaskOn_First{keytype}', ATTR_TYPE_FILE_NAME, False)
            if MaskOff_First_keytype == None : MaskOff_First_keytype = self.check_attribute_value(object_uid, lines_list, f'MaskOff_First{keytype}', ATTR_TYPE_FILE_NAME, False)
            if Width_First_keytype == None : Width_First_keytype = self.check_attribute_value(object_uid, lines_list, f'Width_First{keytype}', ATTR_TYPE_INTEGER, False, 0, 500)
            if Offset_First_keytype == None : Offset_First_keytype = self.check_attribute_value(object_uid, lines_list, f'Offset_First{keytype}', ATTR_TYPE_INTEGER, False, -500, 500)
            if YOffset_First_keytype == None : YOffset_First_keytype = self.check_attribute_value(object_uid, lines_list, f'YOffset_First{keytype}', ATTR_TYPE_INTEGER, False, 0, 500)

            if ImageOn_Last_keytype == None : ImageOn_Last_keytype = self.check_attribute_value(object_uid, lines_list, f'ImageOn_Last{keytype}', ATTR_TYPE_FILE_NAME, False)
            if ImageOff_Last_keytype == None : ImageOff_Last_keytype = self.check_attribute_value(object_uid, lines_list, f'ImageOff_Last{keytype}', ATTR_TYPE_FILE_NAME, False)
            if MaskOn_Last_keytype == None : MaskOn_Last_keytype = self.check_attribute_value(object_uid, lines_list, f'MaskOn_Last{keytype}', ATTR_TYPE_FILE_NAME, False)
            if MaskOff_Last_keytype == None : MaskOff_Last_keytype = self.check_attribute_value(object_uid, lines_list, f'MaskOff_Last{keytype}', ATTR_TYPE_FILE_NAME, False)
            if Width_Last_keytype == None : Width_Last_keytype = self.check_attribute_value(object_uid, lines_list, f'Width_Last{keytype}', ATTR_TYPE_INTEGER, False, 0, 500)
            if Offset_Last_keytype == None : Offset_Last_keytype = self.check_attribute_value(object_uid, lines_list, f'Offset_Last{keytype}', ATTR_TYPE_INTEGER, False, -500, 500)
            if YOffset_Last_keytype == None : YOffset_Last_keytype = self.check_attribute_value(object_uid, lines_list, f'YOffset_Last{keytype}', ATTR_TYPE_INTEGER, False, 0, 500)

    #-------------------------------------------------------------------------------------------------
    def check_object_Panel(self, object_uid, lines_list):
        # check the data of a Panel object section which the lines are in the given lines list

        is_additional_panel = object_uid not in ('Panel000', 'Organ')  # it is an additional panel, in addition to the Panel000 or Organ (old format) panel

        if self.new_panel_format_bool:

            # required attributes (new panel format)
            self.check_attribute_value(object_uid, lines_list, 'Name', ATTR_TYPE_STRING, is_additional_panel)
            self.check_attribute_value(object_uid, lines_list, 'HasPedals', ATTR_TYPE_BOOLEAN, True)

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfGUIElements', ATTR_TYPE_INTEGER, True, 0, 999)
            if value != None and value.isdigit() and int(value) >= 0:
                count = self.objects_type_number_get(f'{object_uid}Element')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfGUIElements={value} whereas {count} {object_uid}Element section(s) defined")

            # optional attributes (new panel format)
            self.check_attribute_value(object_uid, lines_list, 'Group', ATTR_TYPE_STRING, False)

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfImages', ATTR_TYPE_INTEGER, False, 0, 999)
            if value != None and value.isdigit() and int(value) >= 0:
                count = self.objects_type_number_get(f'{object_uid}Image')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfImages={value} whereas {count} {object_uid}Image section(s) defined")

        elif is_additional_panel:  # additional panel in the old panel format (for the main panel, the non display metrics attributes are defined in the Organ object)

            # required attributes (old panel format, additional panel)
            self.check_attribute_value(object_uid, lines_list, 'Name', ATTR_TYPE_STRING, True)
            self.check_attribute_value(object_uid, lines_list, 'HasPedals', ATTR_TYPE_BOOLEAN, True)

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfCouplers', ATTR_TYPE_INTEGER, True, 0, 999)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"Coupler{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
                    self.check_attribute_value(object_uid, lines_list, f"Coupler{str(idx).zfill(3)}Manual", ATTR_TYPE_OBJECT_REF, True)
                count = self.objects_type_number_get(f'{object_uid}Coupler')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfCouplers={value} whereas {count} {object_uid}Coupler section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfDivisionals', ATTR_TYPE_INTEGER, True, 0, 999)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"Divisional{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
                    self.check_attribute_value(object_uid, lines_list, f"Divisional{str(idx).zfill(3)}Manual", ATTR_TYPE_OBJECT_REF, True)
                count = self.objects_type_number_get(f'{object_uid}Divisional')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfDivisionals={value} whereas {count} {object_uid}Divisional section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfDivisionalCouplers', ATTR_TYPE_INTEGER, True, 0, 8)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"DivisionalCoupler{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
                count = self.objects_type_number_get(f'{object_uid}DivisionalCoupler')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfDivisionalCouplers={value} whereas {count} {object_uid}DivisionalCoupler section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfEnclosures', ATTR_TYPE_INTEGER, True, 0, 50)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"Enclosure{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
                count = self.objects_type_number_get(f'{object_uid}Enclosure')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfEnclosures={value} whereas {count} {object_uid}Enclosure section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfGenerals', ATTR_TYPE_INTEGER, True, 0, 99)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"General{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
                count = self.objects_type_number_get(f'{object_uid}General')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfGenerals={value} whereas {count} {object_uid}General section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfImages', ATTR_TYPE_INTEGER, True, 0, 999)
            if value != None and value.isdigit() and int(value) >= 0:
                count = self.objects_type_number_get(f'{object_uid}Image')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfImages={value} whereas {count} {object_uid}Image section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfLabels', ATTR_TYPE_INTEGER, True, 0, 999)
            if value != None and value.isdigit() and int(value) >= 0:
                count = self.objects_type_number_get(f'{object_uid}Label')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfLabels={value} whereas {count} {object_uid}Label section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfManuals', ATTR_TYPE_INTEGER, True, 0, 16)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"Manual{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfReversiblePistons', ATTR_TYPE_INTEGER, True, 0, 32)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"ReversiblePiston{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
                count = self.objects_type_number_get(f'{object_uid}ReversiblePiston')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfReversiblePistons={value} whereas {count} {object_uid}ReversiblePiston section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfStops', ATTR_TYPE_INTEGER, True, 0, 999)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"Stop{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
                    self.check_attribute_value(object_uid, lines_list, f"Stop{str(idx).zfill(3)}Manual", ATTR_TYPE_OBJECT_REF, True)
                count = self.objects_type_number_get(f'{object_uid}Stop')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfStops={value} whereas {count} {object_uid}Stop section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfTremulants', ATTR_TYPE_INTEGER, True, 0, 10)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"Tremulant{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
                count = self.objects_type_number_get(f'{object_uid}Tremulant')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfTremulants={value} whereas {count} {object_uid}Tremulant section(s) defined")

            # optional attributes (old panel format, additional panel)
            self.check_attribute_value(object_uid, lines_list, 'Group', ATTR_TYPE_STRING, False)

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfSetterElements', ATTR_TYPE_INTEGER, False, 0, 8)
            if value != None and value.isdigit() and int(value) >= 0:
                count = self.objects_type_number_get(f'{object_uid}SetterElement')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfSetterElements={value} whereas {count} {object_uid}SetterElement section(s) defined")

            value = self.check_attribute_value(object_uid, lines_list, 'NumberOfSwitches', ATTR_TYPE_INTEGER, False, 0, 999)
            if value != None and value.isdigit() and int(value) >= 0:
                for idx in range(1, int(value)+1):
                    self.check_attribute_value(object_uid, lines_list, f"Switch{str(idx).zfill(3)}", ATTR_TYPE_OBJECT_REF, True)
                count = self.objects_type_number_get(f'{object_uid}Switch')
                if count != int(value):
                    logs.add(f"ERROR in {object_uid} : NumberOfSwitches={value} whereas {count} {object_uid}Switch section(s) defined")


        # display metrics (common to old and new panel formats)

        # required attributes (panel display metrics)
        self.check_attribute_value(object_uid, lines_list, 'DispScreenSizeHoriz', ATTR_TYPE_PANEL_SIZE, True)
        self.check_attribute_value(object_uid, lines_list, 'DispScreenSizeVert', ATTR_TYPE_PANEL_SIZE, True)
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopBackgroundImageNum', ATTR_TYPE_INTEGER, True, 1, 64)
        self.check_attribute_value(object_uid, lines_list, 'DispConsoleBackgroundImageNum', ATTR_TYPE_INTEGER, True, 1, 64)
        self.check_attribute_value(object_uid, lines_list, 'DispKeyHorizBackgroundImageNum', ATTR_TYPE_INTEGER, True, 1, 64)
        self.check_attribute_value(object_uid, lines_list, 'DispKeyVertBackgroundImageNum', ATTR_TYPE_INTEGER, True, 1, 64)
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopInsetBackgroundImageNum', ATTR_TYPE_INTEGER, True, 1, 64)
        self.check_attribute_value(object_uid, lines_list, 'DispControlLabelFont', ATTR_TYPE_STRING, True)
        self.check_attribute_value(object_uid, lines_list, 'DispShortcutKeyLabelFont', ATTR_TYPE_STRING, True)
        self.check_attribute_value(object_uid, lines_list, 'DispShortcutKeyLabelColour', ATTR_TYPE_COLOR, True)
        self.check_attribute_value(object_uid, lines_list, 'DispGroupLabelFont', ATTR_TYPE_STRING, True)
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopCols', ATTR_TYPE_INTEGER, True, 2, 12)
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopRows', ATTR_TYPE_INTEGER, True, 1, 20)
        cols_offset = self.check_attribute_value(object_uid, lines_list, 'DispDrawstopColsOffset', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'DispPairDrawstopCols', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'DispExtraDrawstopRows', ATTR_TYPE_INTEGER, True, 0, 99)
        self.check_attribute_value(object_uid, lines_list, 'DispExtraDrawstopCols', ATTR_TYPE_INTEGER, True, 0, 40)
        self.check_attribute_value(object_uid, lines_list, 'DispButtonCols', ATTR_TYPE_INTEGER, True, 1, 32)
        self.check_attribute_value(object_uid, lines_list, 'DispExtraButtonRows', ATTR_TYPE_INTEGER, True, 0, 99)
        extra_pedal_buttons = self.check_attribute_value(object_uid, lines_list, 'DispExtraPedalButtonRow', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'DispButtonsAboveManuals', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'DispExtraDrawstopRowsAboveExtraButtonRows', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'DispTrimAboveManuals', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'DispTrimBelowManuals', ATTR_TYPE_BOOLEAN, True)
        self.check_attribute_value(object_uid, lines_list, 'DispTrimAboveExtraRows', ATTR_TYPE_BOOLEAN, True)

        # optional attributes (panel display metrics)
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopWidth', ATTR_TYPE_INTEGER, False, 1, 150)
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopHeight', ATTR_TYPE_INTEGER, False, 1, 150)
        self.check_attribute_value(object_uid, lines_list, 'DispDrawstopOuterColOffsetUp', ATTR_TYPE_BOOLEAN, cols_offset != None and cols_offset == 'Y')
        self.check_attribute_value(object_uid, lines_list, 'DispExtraPedalButtonRowOffset', ATTR_TYPE_BOOLEAN, extra_pedal_buttons != None and extra_pedal_buttons == 'Y')
        self.check_attribute_value(object_uid, lines_list, 'DispExtraPedalButtonRowOffsetRight', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'DispPistonWidth', ATTR_TYPE_INTEGER, False, 1, 150)
        self.check_attribute_value(object_uid, lines_list, 'DispPistonHeight', ATTR_TYPE_INTEGER, False, 1, 150)
        self.check_attribute_value(object_uid, lines_list, 'DispEnclosureWidth', ATTR_TYPE_INTEGER, False, 1, 150)
        self.check_attribute_value(object_uid, lines_list, 'DispEnclosureHeight', ATTR_TYPE_INTEGER, False, 1, 150)
        self.check_attribute_value(object_uid, lines_list, 'DispPedalHeight', ATTR_TYPE_INTEGER, False, 1, 500)
        self.check_attribute_value(object_uid, lines_list, 'DispPedalKeyWidth', ATTR_TYPE_INTEGER, False, 1, 500)
        self.check_attribute_value(object_uid, lines_list, 'DispManualHeight', ATTR_TYPE_INTEGER, False, 1, 500)
        self.check_attribute_value(object_uid, lines_list, 'DispManualKeyWidth', ATTR_TYPE_INTEGER, False, 1, 500)

    #-------------------------------------------------------------------------------------------------
    def check_object_PanelElement(self, object_uid, lines_list):
        # check the data of a Panel Element object section which the lines are in the given lines list

        # required attributes
        elem_type = self.check_attribute_value(object_uid, lines_list, 'Type', ATTR_TYPE_ELEMENT_TYPE, True)

        if elem_type == None:
            pass
        elif elem_type == 'Coupler':
            self.check_attribute_value(object_uid, lines_list, 'Manual', ATTR_TYPE_OBJECT_REF, True)
            self.check_attribute_value(object_uid, lines_list, 'Coupler', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_Coupler(object_uid, lines_list)
        elif elem_type == 'Divisional':
            self.check_attribute_value(object_uid, lines_list, 'Manual', ATTR_TYPE_OBJECT_REF, True)
            self.check_attribute_value(object_uid, lines_list, 'Divisional', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_Divisional(object_uid, lines_list)
        elif elem_type == 'DivisionalCoupler':
            self.check_attribute_value(object_uid, lines_list, 'DivisionalCoupler', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_DivisionalCoupler(object_uid, lines_list)
        elif elem_type == 'Enclosure':
            self.check_attribute_value(object_uid, lines_list, 'Enclosure', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_Enclosure(object_uid, lines_list)
        elif elem_type == 'General':
            self.check_attribute_value(object_uid, lines_list, 'General', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_General(object_uid, lines_list)
        elif elem_type == 'Label':
            self.check_object_Label(object_uid, lines_list)
        elif elem_type == 'Manual':
            self.check_attribute_value(object_uid, lines_list, 'Manual', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_Manual(object_uid, lines_list)
        elif elem_type == 'ReversiblePiston':
            self.check_attribute_value(object_uid, lines_list, 'ReversiblePiston', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_ReversiblePiston(object_uid, lines_list)
        elif elem_type == 'Stop':
            self.check_attribute_value(object_uid, lines_list, 'Manual', ATTR_TYPE_OBJECT_REF, True)
            self.check_attribute_value(object_uid, lines_list, 'Stop', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_Button(object_uid, lines_list)
        elif elem_type == 'Swell':
            self.check_object_Enclosure(object_uid, lines_list)
        elif elem_type == 'Switch':
            self.check_attribute_value(object_uid, lines_list, 'Switch', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_Switch(object_uid, lines_list)
        elif elem_type == 'Tremulant':
            self.check_attribute_value(object_uid, lines_list, 'Tremulant', ATTR_TYPE_OBJECT_REF, True)
            self.check_object_Tremulant(object_uid, lines_list)
        else:
            self.check_object_SetterElement(object_uid, lines_list, elem_type)

    #-------------------------------------------------------------------------------------------------
    def check_object_PanelOther(self, object_uid, lines_list):
        # check the data of an other kind of Panel object section (Panel999Coupler999, Panel999Divisional999, ...) which the lines are in the given lines list

        # get the panel elemnt type from the object UID (for example Coupler from Panel999Coupler999)
        panel_element_type = object_uid[8:-3]

        # check the attributes of the object depending on the object type
        if panel_element_type == 'Coupler':
            self.check_object_Coupler(object_uid, lines_list)
        elif panel_element_type == 'Divisional':
            self.check_object_Divisional(object_uid, lines_list)
        elif panel_element_type == 'DivisionalCoupler':
            self.check_object_DivisionalCoupler(object_uid, lines_list)
        elif panel_element_type == 'Enclosure':
            self.check_object_Enclosure(object_uid, lines_list)
        elif panel_element_type == 'General':
            self.check_object_General(object_uid, lines_list)
        elif panel_element_type == 'Image':
            self.check_object_Image(object_uid, lines_list)
        elif panel_element_type == 'Label':
            self.check_object_Label(object_uid, lines_list)
        elif panel_element_type == 'ReversiblePiston':
            self.check_object_ReversiblePiston(object_uid, lines_list)
        elif panel_element_type == 'SetterElement':
            self.check_object_SetterElement(object_uid, lines_list)
        elif panel_element_type == 'Stop':
            self.check_object_Stop(object_uid, lines_list)
        elif panel_element_type == 'Switch':
            self.check_object_Switch(object_uid, lines_list)
        elif panel_element_type == 'Tremulant':
            self.check_object_Tremulant(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_Piston(self, object_uid, lines_list):
        # check the data of a Piston object section which the lines are in the given lines list

        # required attributes
        value = self.check_attribute_value(object_uid, lines_list, 'ObjectType', ATTR_TYPE_PISTON_TYPE, True)
        self.check_attribute_value(object_uid, lines_list, 'ManualNumber', ATTR_TYPE_OBJECT_REF, value in ('STOP', 'COUPLER'))
        self.check_attribute_value(object_uid, lines_list, 'ObjectNumber', ATTR_TYPE_INTEGER, False, 1, 200)

        # a Piston has also the attributes of a Push Button
        self.check_object_PushButton(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_PushButton(self, object_uid, lines_list):
        # check the data of a Push Button object section which the lines are in the given lines list

        # a Push Button has only the attributes of a Button
        self.check_object_Button(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_Rank(self, object_uid, lines_list):
        # check the data of a Rank object section which the lines are in the given lines list

        is_rank_obj = self.object_type_get(object_uid) == 'Rank'  # some mandatory attributes are not mandatory for objects which inherit the Rank attributes (like Stop)

        # required attributes
        self.check_attribute_value(object_uid, lines_list, 'Name', ATTR_TYPE_STRING, True)
        self.check_attribute_value(object_uid, lines_list, 'FirstMidiNoteNumber', ATTR_TYPE_INTEGER, is_rank_obj, 0, 256)
        self.check_attribute_value(object_uid, lines_list, 'WindchestGroup', ATTR_TYPE_OBJECT_REF, True)

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'AmplitudeLevel', ATTR_TYPE_FLOAT, False, 0, 1000)
        self.check_attribute_value(object_uid, lines_list, 'Gain', ATTR_TYPE_FLOAT, False, -120, 40)
        self.check_attribute_value(object_uid, lines_list, 'Percussive', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'HasIndependentRelease', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'PitchTuning', ATTR_TYPE_FLOAT, False, -1800, 1800)
        self.check_attribute_value(object_uid, lines_list, 'TrackerDelay', ATTR_TYPE_INTEGER, False, 0, 10000)
        self.check_attribute_value(object_uid, lines_list, 'HarmonicNumber', ATTR_TYPE_FLOAT, False, 1, 1024)
        self.check_attribute_value(object_uid, lines_list, 'PitchCorrection', ATTR_TYPE_FLOAT, False, -1800, 1800)
        self.check_attribute_value(object_uid, lines_list, 'MinVelocityVolume', ATTR_TYPE_FLOAT, False, 0, 1000)
        self.check_attribute_value(object_uid, lines_list, 'MaxVelocityVolume', ATTR_TYPE_FLOAT, False, 0, 1000)
        self.check_attribute_value(object_uid, lines_list, 'AcceptsRetuning', ATTR_TYPE_BOOLEAN, False)

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfLogicalPipes', ATTR_TYPE_INTEGER, is_rank_obj, 1, 192)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):  # Pipe999xxx attributes
                pipe_id = f'Pipe{str(idx).zfill(3)}'
                self.check_attribute_value(object_uid, lines_list, pipe_id, ATTR_TYPE_PIPE_WAVE, True)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'Percussive', ATTR_TYPE_BOOLEAN, False)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'HasIndependentRelease', ATTR_TYPE_BOOLEAN, False)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'AmplitudeLevel', ATTR_TYPE_FLOAT, False, 0, 1000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'Gain', ATTR_TYPE_FLOAT, False, -120, 40)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'PitchTuning', ATTR_TYPE_FLOAT, False, -1800, 1800)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'TrackerDelay', ATTR_TYPE_FLOAT, False, 0, 10000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'LoadRelease', ATTR_TYPE_BOOLEAN, False)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'AttackVelocity', ATTR_TYPE_INTEGER, False, 0, 127)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'MaxTimeSinceLastRelease', ATTR_TYPE_INTEGER, False, -1, 100000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'IsTremulant', ATTR_TYPE_INTEGER, False, -1, 1)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'MaxKeyPressTime', ATTR_TYPE_INTEGER, False, -1, 100000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'AttackStart', ATTR_TYPE_INTEGER, False, 0, 158760000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'CuePoint', ATTR_TYPE_INTEGER, False, -1, 158760000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'ReleaseEnd', ATTR_TYPE_INTEGER, False, -1, 158760000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'HarmonicNumber', ATTR_TYPE_FLOAT, False, 1, 1024)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'MIDIKeyNumber', ATTR_TYPE_INTEGER, False, -1, 127)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'PitchCorrection', ATTR_TYPE_FLOAT, False, -1800, 1800)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'AcceptsRetuning', ATTR_TYPE_BOOLEAN, False)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'WindchestGroup', ATTR_TYPE_OBJECT_REF, False)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'MinVelocityVolume', ATTR_TYPE_FLOAT, False, 0, 1000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'MaxVelocityVolume', ATTR_TYPE_FLOAT, False, 0, 1000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'LoopCrossfadeLength', ATTR_TYPE_INTEGER, False, 0, 3000)
                self.check_attribute_value(object_uid, lines_list, pipe_id + 'ReleaseCrossfadeLength', ATTR_TYPE_INTEGER, False, 0, 3000)

                ret1 = self.check_attribute_value(object_uid, lines_list, pipe_id + 'LoopCount', ATTR_TYPE_INTEGER, False, 1, 100)
                if ret1 != None and ret1.isdigit():
                    for idx1 in range(1, int(ret1)+1):  # Pipe999Loop999xxx attributes
                        value = self.check_attribute_value(object_uid, lines_list, pipe_id + f'Loop{str(idx1).zfill(3)}Start', ATTR_TYPE_INTEGER, False, 0, 158760000)
                        loop_start = int(value) if value.isdigit() else 1
                        self.check_attribute_value(object_uid, lines_list, pipe_id + f'Loop{str(idx1).zfill(3)}End', ATTR_TYPE_INTEGER, False, loop_start + 1, 158760000)

                ret1 = self.check_attribute_value(object_uid, lines_list, pipe_id + 'AttackCount', ATTR_TYPE_INTEGER, False, 1, 100)
                if ret1 != None and ret1.isdigit():
                    for idx1 in range(1, int(ret1)+1):  # Pipe999Attack999xxx attributes
                        pipe_atk_id = pipe_id + f'Attack{str(idx1).zfill(3)}'
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id, ATTR_TYPE_FILE_NAME, True)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'LoadRelease', ATTR_TYPE_BOOLEAN, False)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'AttackVelocity', ATTR_TYPE_INTEGER, False, 0, 127)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'MaxTimeSinceLastRelease', ATTR_TYPE_INTEGER, False, -1, 100000)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'IsTremulant', ATTR_TYPE_INTEGER, False, -1, 1)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'MaxKeyPressTime', ATTR_TYPE_INTEGER, False, -1, 100000)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'AttackStart', ATTR_TYPE_INTEGER, False, 0, 158760000)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'CuePoint', ATTR_TYPE_INTEGER, False, -1, 158760000)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'ReleaseEnd', ATTR_TYPE_INTEGER, False, -1, 158760000)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'LoopCrossfadeLength', ATTR_TYPE_INTEGER, False, 0, 3000)
                        self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'ReleaseCrossfadeLength', ATTR_TYPE_INTEGER, False, 0, 3000)

                        ret2 = self.check_attribute_value(object_uid, lines_list, pipe_atk_id + 'LoopCount', ATTR_TYPE_INTEGER, False, 1, 100)
                        if ret2 != None and ret2.isdigit():
                            for idx2 in range(1, int(ret2)+1):  # Pipe999Attack999Loop999xxx attributes
                                value = self.check_attribute_value(object_uid, lines_list, pipe_atk_id + f'Loop{str(idx2).zfill(3)}Start', ATTR_TYPE_INTEGER, True, 0, 158760000)
                                loop_start = int(value) if value != None and value.isdigit() else 1
                                self.check_attribute_value(object_uid, lines_list, pipe_atk_id + f'Loop{str(idx2).zfill(3)}End', ATTR_TYPE_INTEGER, True, loop_start + 1, 158760000)

                ret1 = self.check_attribute_value(object_uid, lines_list, f'Pipe{str(idx).zfill(3)}ReleaseCount', ATTR_TYPE_INTEGER, False, 1, 100)
                if ret1 != None and ret1.isdigit():
                    for idx1 in range(1, int(ret1)+1):  # Pipe999Release999xxx attributes
                        pipe_rel_id = pipe_id + f'Release{str(idx1).zfill(3)}'
                        self.check_attribute_value(object_uid, lines_list, pipe_rel_id, ATTR_TYPE_FILE_NAME, True)
                        self.check_attribute_value(object_uid, lines_list, pipe_rel_id + 'IsTremulant', ATTR_TYPE_INTEGER, False, -1, 1)
                        self.check_attribute_value(object_uid, lines_list, pipe_rel_id + 'MaxKeyPressTime', ATTR_TYPE_INTEGER, False, -1, 100000)
                        self.check_attribute_value(object_uid, lines_list, pipe_rel_id + 'CuePoint', ATTR_TYPE_INTEGER, False, -1, 158760000)
                        self.check_attribute_value(object_uid, lines_list, pipe_rel_id + 'ReleaseEnd', ATTR_TYPE_INTEGER, False, -1, 158760000)
                        self.check_attribute_value(object_uid, lines_list, pipe_rel_id + 'ReleaseCrossfadeLength', ATTR_TYPE_INTEGER, False, 0, 3000)

    #-------------------------------------------------------------------------------------------------
    def check_object_ReversiblePiston(self, object_uid, lines_list):
        # check the data of a Reversible Piston object section which the lines are in the given lines list

        # unkown expected attributes...
        pass

     #-------------------------------------------------------------------------------------------------
    def check_object_SetterElement(self, object_uid, lines_list, elem_type = None):
        # check the data of a Setter Element object section which the lines are in the given lines list

        # required attributes
        if elem_type == None:
            # elem_type not provided by the caller, recover it from the object lines list
            elem_type = self.check_attribute_value(object_uid, lines_list, 'Type', ATTR_TYPE_ELEMENT_TYPE, True)

        if elem_type == None:
            pass
        elif elem_type == 'CrescendoLabel':
            self.check_object_Label(object_uid, lines_list)
        elif elem_type in ('CrescendoA', 'CrescendoB', 'CrescendoC', 'CrescendoD'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type in ('CrescendoPrev', 'CrescendoNext', 'CrescendoCurrent'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type in ('Current', 'Full', 'GC'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type[:7] == "General" and len(elem_type) == 9 and elem_type[7:9].isdigit() and int(elem_type[7:9]) in range(1, 51):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type == 'GeneralLabel':
            self.check_object_Label(object_uid, lines_list)
        elif elem_type in ('GeneralPrev', 'GeneralNext', 'Home', 'Insert', 'Delete'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type[:1] == "L" and len(elem_type) == 2 and elem_type[1:2].isdigit() and int(elem_type[1:2]) in range(0, 10):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type in ('M100', 'M10', 'M1', 'P1', 'P10', 'P100'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type == 'PitchLabel':
            self.check_object_Label(object_uid, lines_list)
        elif elem_type in ('PitchM100', 'PitchM10', 'PitchM1', 'PitchP1', 'PitchP10', 'PitchP100'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type in ('Prev', 'Next', 'Set'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type in ('Regular', 'Scope', 'Scoped', 'Save'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type in ('SequencerLabel', 'TemperamentLabel'):
            self.check_object_Label(object_uid, lines_list)
        elif elem_type in ('TemperamentPrev', 'TemperamentNext'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type in ('TransposeDown', 'TransposeUp'):
            self.check_object_Button(object_uid, lines_list)
        elif elem_type == 'TransposeLabel':
            self.check_object_Label(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_Stop(self, object_uid, lines_list):
        # check the data of a Stop object section which the lines are in the given lines list

        is_stop_obj = self.object_type_get(object_uid) == 'Stop' # some mandatory attributes are not mandatory for objects which inherit the Stop attributes

        # optional attribute
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfRanks', ATTR_TYPE_INTEGER, False, 0, 999)
        if value == None or (value != None and not value.isdigit()):
            # number of ranks not defined or not a number
            nb_ranks = 0
        else:
            nb_ranks = int(value)

        # required attributes
        self.check_attribute_value(object_uid, lines_list, 'FirstAccessiblePipeLogicalKeyNumber', ATTR_TYPE_INTEGER, is_stop_obj, 1, 128)
        self.check_attribute_value(object_uid, lines_list, 'FirstAccessiblePipeLogicalPipeNumber', ATTR_TYPE_INTEGER, is_stop_obj and nb_ranks == 0, 1, 192)

        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfAccessiblePipes', ATTR_TYPE_INTEGER, is_stop_obj, 1, 192)
        nb_pipes = int(value) if value != None and value.isdigit() else 192

        # optional attributes
        if nb_ranks > 0:
            for idx in range(1, nb_ranks+1):
                rank_id = f'Rank{str(idx).zfill(3)}'
                self.check_attribute_value(object_uid, lines_list, rank_id, ATTR_TYPE_OBJECT_REF, True)
                self.check_attribute_value(object_uid, lines_list, rank_id + 'FirstPipeNumber', ATTR_TYPE_INTEGER, False, 1, nb_pipes)
                self.check_attribute_value(object_uid, lines_list, rank_id + 'PipeCount', ATTR_TYPE_INTEGER, False, 0, nb_pipes)
                self.check_attribute_value(object_uid, lines_list, rank_id + 'FirstAccessibleKeyNumber', ATTR_TYPE_INTEGER, False, 1, nb_pipes)
        elif nb_ranks == 0:
            # number of ranks set at 0, the Stop must contain rank attributes
            self.check_object_Rank(object_uid, lines_list)

        # a Stop has also the attributes of a Drawstop
        self.check_object_DrawStop(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_Switch(self, object_uid, lines_list):
        # check the data of a Switch object section which the lines are in the given lines list

        # a Switch has only the attributes of a Drawstop
        self.check_object_DrawStop(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_Tremulant(self, object_uid, lines_list):
        # check the data of a Tremulant object section which the lines are in the given lines list

        # optional attributes
        value = self.check_attribute_value(object_uid, lines_list, 'TremulantType', ATTR_TYPE_TREMULANT_TYPE, False)
        is_synth = value == 'Synth'
        self.check_attribute_value(object_uid, lines_list, 'Period', ATTR_TYPE_INTEGER, is_synth, 32, 44100)
        self.check_attribute_value(object_uid, lines_list, 'StartRate', ATTR_TYPE_INTEGER, is_synth, 1, 100)
        self.check_attribute_value(object_uid, lines_list, 'StopRate', ATTR_TYPE_INTEGER, is_synth, 1, 100)
        self.check_attribute_value(object_uid, lines_list, 'AmpModDepth', ATTR_TYPE_INTEGER, is_synth, 1, 100)

        # a Tremulant has also the attributes of a Drawstop
        self.check_object_DrawStop(object_uid, lines_list)

    #-------------------------------------------------------------------------------------------------
    def check_object_WindchestGroup(self, object_uid, lines_list):
        # check the data of a WindChest Group object section which the lines are in the given lines list

        # required attributes
        max_val = self.objects_type_number_get('Enclosure')
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfEnclosures', ATTR_TYPE_INTEGER, True, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Enclosure{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        max_val = self.objects_type_number_get('Tremulant')
        value = self.check_attribute_value(object_uid, lines_list, 'NumberOfTremulants', ATTR_TYPE_INTEGER, True, 0, max_val)
        if value != None and value.isdigit():
            for idx in range(1, int(value)+1):
                self.check_attribute_value(object_uid, lines_list, f'Tremulant{str(idx).zfill(3)}', ATTR_TYPE_OBJECT_REF, True)

        # optional attributes
        self.check_attribute_value(object_uid, lines_list, 'Name', ATTR_TYPE_STRING, False)
        self.check_attribute_value(object_uid, lines_list, 'AmplitudeLevel', ATTR_TYPE_FLOAT, False, 0, 1000)
        self.check_attribute_value(object_uid, lines_list, 'Gain', ATTR_TYPE_FLOAT, False, -120, 40)
        self.check_attribute_value(object_uid, lines_list, 'PitchTuning', ATTR_TYPE_FLOAT, False, -1800, 1800)
        self.check_attribute_value(object_uid, lines_list, 'PitchCorrection', ATTR_TYPE_FLOAT, False, -1800, 1800)
        self.check_attribute_value(object_uid, lines_list, 'TrackerDelay', ATTR_TYPE_FLOAT, False, 0, 10000)
        self.check_attribute_value(object_uid, lines_list, 'Percussive', ATTR_TYPE_BOOLEAN, False)
        self.check_attribute_value(object_uid, lines_list, 'HasIndependentRelease', ATTR_TYPE_BOOLEAN, False)

    #-------------------------------------------------------------------------------------------------
    def check_attribute_value(self, object_uid, lines_list, attribute_name, attribute_value_type, required_attribute_bool, attribute_value_min=0, attribute_value_max=0):
        # check if the given attribute name is present in the given object lines list (sorted)
        # and if its value is correct for its value type and min/max values
        # the min and max values are ignored if max <= min. The given lines list is considered to be sorted
        # returns the value of the attribute if it has been found and without error, else return None

        # search in the given lines list the line with the attribute to check
        attr_value = None
        line = None
        for i, line in enumerate(lines_list):
            if line[:len(attribute_name)] > attribute_name:
                # the given lines list being sorted, exit the loop if the current line starts by a string higher than the one of the attribute name
                break
            if line.startswith(attribute_name + '='):
                # line found
                (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)
                if error_msg != None:
                    logs.add(f'ERROR in {object_uid} section, line "{line}" : {error_msg}')
                if attr_name != None:
                    # the current line contains the attribute to check
                    self.checked_attr_nb += 1
                     # remove the line of the found attribute, to know at the end of the object check which of its attributes have not been checked
                    lines_list.pop(i)
                    break

        if attr_value not in (None, ''):
            # the attribute has been found and it has a value

            # check that the given max value is higher or equal to the min value (this should never happen)
            if attribute_value_max < attribute_value_min:
                logs.add(f"ERROR in {object_uid} {line} : for attribute {attribute_name} the mininum value {attribute_value_min} is higher than the maximum value {attribute_value_max}")
                return None

            # check the attribute value according to the given type

            if attribute_value_type == ATTR_TYPE_INTEGER:
                if (not attr_value.lstrip("-+").isdigit() or
                    ((int(attr_value) < attribute_value_min or int(attr_value) > attribute_value_max))):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value must be an integer in the range [{attribute_value_min} - {attribute_value_max}]")

            elif attribute_value_type == ATTR_TYPE_FLOAT:
                if (not(attr_value.lstrip("-+").replace('.', '', 1).isdigit()) or
                    ((float(attr_value) < attribute_value_min or float(attr_value) > attribute_value_max))):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value must be an integer or decimal in the range [{attribute_value_min} - {attribute_value_max}]")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_BOOLEAN:
                if attr_value.upper() not in ('Y', 'N'):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value must be Y or N (boolean attribute)")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_STRING:
                pass # nothing to check in case of string value

            elif attribute_value_type == ATTR_TYPE_COLOR:
                if (not(attr_value.upper() in ('BLACK', 'BLUE', 'DARK BLUE', 'GREEN', 'DARK GREEN', 'CYAN', 'DARK CYAN', 'RED', 'DARK RED',
                                               'MAGENTA', 'DARK MAGENTA', 'YELLOW', 'DARK YELLOW', 'LIGHT GREY', 'DARK GREY', 'WHITE', 'BROWN')) and
                    not(len(attr_value) == 7 and attr_value[0] == '#' and attr_value[1:].isalnum())):  # check of the HTML format #RRGGBB
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value is not a valid color (look at the help)")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_FONT_SIZE:
                if (not(attr_value.upper() in ('SMALL', 'NORMAL', 'LARGE')) and
                    not(attr_value.isdigit() and int(attr_value) >= 1 and int(attr_value) <= 50)):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value is not a valid font size (look at the help)")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_PANEL_SIZE:
                if (not(attr_value.upper() in ('SMALL', 'MEDIUM', 'MEDIUM LARGE', 'LARGE')) and
                    not(attr_value.isdigit() and int(attr_value) >= 100 and int(attr_value) <= 32000)):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value is not a valid panel size (look at the help)")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_OBJECT_REF:  # for example Switch002=12 or ManualNumber=2 or Stop003Manual=2 or Pipe015WindchestGroup=1
                if attribute_name[-3:].isdigit():
                    attribute_name = attribute_name[:-3]   # remove the three digits at the end of the attribute name to get the object name

                if attribute_name[-6:] == 'Number':
                    attribute_name = attribute_name[:-6]   # remove the 'Number' string at the end, used in General and Piston objects
                elif attribute_name[-6:] == 'Manual':
                    attribute_name = 'Manual'         # keep only the 'Manual' string, used in General object
                elif attribute_name[-14:] == 'WindchestGroup':
                    attribute_name = 'WindchestGroup' # keep only the 'WindchestGroup' string, used in Rank object

                attr_value = attr_value.lstrip("+-") # remove possible + or - at the beginning of the value, used in General or Divisional objects

                if attribute_name + attr_value.zfill(3) not in self.odf_data_dic:
                    logs.add(f"ERROR in {object_uid} {line} : the object {attribute_name + attr_value.zfill(3)} does not exist")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_ELEMENT_TYPE:
                if (not(attr_value in ('Coupler', 'Divisional', 'DivisionalCoupler', 'Enclosure', 'General', 'Label', 'Manual', 'ReversiblePiston', 'Stop', 'Swell',
                                      'Switch', 'Tremulant', 'CrescendoA', 'CrescendoB', 'CrescendoC', 'CrescendoD', 'CrescendoPrev', 'CrescendoNext', 'CrescendoCurrent',
                                      'Current', 'Full', 'GC', 'GeneralLabel', 'GeneralPrev', 'GeneralNext', 'Home', 'Insert', 'Delete', 'M100', 'M10', 'M1', 'P1', 'P10', 'P100',
                                      'PitchLabel', 'PitchP1', 'PitchP10', 'PitchP100', 'PitchM1', 'PitchM10', 'PitchM100', 'Prev', 'Next', 'Set', 'Regular', 'Scope', 'Scoped',
                                      'Save', 'SequencerLabel', 'TemperamentLabel', 'TemperamentPrev', 'TemperamentNext', 'TransposeDown', 'TransposeUp', 'TransposeLabel')) and
                    not(attr_value[0] == 'L' and attr_value[1].isdigit() and int(attr_value[1]) in range(0, 10)) and
                    not(attr_value[:14] == 'CrescendoLabel' and attr_value[14:].isdigit() and int(attr_value[14:]) in range(1, 33)) and
                    not(attr_value[:7] == 'General' and attr_value[7:].isdigit() and int(attr_value[7:]) in range(1, 51))):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value is not a valid panel element type (look at the help)")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_COUPLER_TYPE:
                if attr_value.upper() not in ('NORMAL', 'BASS', 'MELODY'):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value is not a valid coupler type (look at the help)")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_TREMULANT_TYPE:
                if attr_value.upper() not in ('SYNTH', 'WAVE'):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value is not a valid tremulant type (look at the help)")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_PISTON_TYPE:
                if attr_value.upper() not in ('STOP', 'COUPLER', 'SWITCH', 'TREMULANT'):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value is not a valid piston type (look at the help)")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_DRAWSTOP_FCT:
                if attr_value not in ('Input', 'Not', 'And', 'Xor', 'Nand', 'Nor', 'Or'):
                    logs.add(f"ERROR in {object_uid} {line} : the assigned value is not a valid drawstop function (look at the help)")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_FILE_NAME:
                if self.check_files_names and not os.path.isfile(os.path.dirname(self.odf_file_name) + os.path.sep + path2ospath(attr_value)):
                    logs.add(f"ERROR in {object_uid} {line} : file does not exist")
                    attr_value = ''

            elif attribute_value_type == ATTR_TYPE_PIPE_WAVE and self.check_files_names :
                if attr_value.upper()[-4:] == '.WAV':
                    if self.check_files_names and not os.path.isfile(os.path.dirname(self.odf_file_name) + os.path.sep + path2ospath(attr_value)):
                        logs.add(f"ERROR in {object_uid} {line} : file not found")
                        attr_value = ''
                elif attr_value[:4] == 'REF:':  # for example REF:001:005:007
                    refs_list = attr_value[4:].split(':')
                    if len(refs_list) < 3 or not (refs_list[0].isdigit() and refs_list[1].isdigit() and refs_list[2].isdigit()):
                        logs.add(f"ERROR in {object_uid} {line} : wrong pipe referencing, expected REF:999:999:999")
                        attr_value = ''
                elif attr_value != 'DUMMY':
                    logs.add(f"ERROR in {object_uid} {line} : wrong pipe definition")
                    attr_value = ''

        elif attr_value == None and required_attribute_bool:
            # the attribute has not been found and it is required
            logs.add(f"ERROR in {object_uid} : the attribute {attribute_name} is expected, it is missing or misspelled")

        elif attr_value == '' and attribute_value_type != ATTR_TYPE_STRING:
            # the attribute has no value and it is not a string type
            logs.add(f"ERROR in {object_uid} : the attribute {attribute_name} has no value defined")

        return attr_value

    #-------------------------------------------------------------------------------------------------
    def check_attributes_unicity(self, object_uid, lines_list):
        # check in the given object lines list if each attribute is unique

        # copy the attributes names of the given lines list in an attributes list
        attributes_list = []
        for line in lines_list:
            pos = line.find('=', 1)
            if pos != -1 and line[0] != ';':
                # line with an attribute
                attributes_list.append(line[:pos])

        # sort the attributes list
        attributes_list.sort()

        # check if there are consecutive identical names in the sorted list
        for i in range(0, len(attributes_list) - 1):
            if attributes_list[i] == attributes_list[i+1]:
                logs.add(f"ERROR in {object_uid} : the attribute {attributes_list[i]} is defined more than once")


#-------------------------------------------------------------------------------------------------
class C_ODF_DATA(C_ODF_DATA_CHECK, C_ODF_MISC):
    # class to store and manage GO ODF data

    odf_file_name = ""      # name of the ODF which the data have been loaded
    odf_file_encoding = ""  # encoding type of the loaded ODF

    new_panel_format_bool = False  # flag indicating if the ODF data use the new panel format or not

    odf_data_dic = {}  # dictionary in which are stored the data of the loaded GrandOrgue ODF
                       # it has the following structure with two nested dictionaries :
                       #   {object UID: string (for example Organ, Panel001, Rank003)
                       #       {"names": list of strings (names to identify better the object)
                       #        "parents": list of strings (UID)
                       #        "children": list of strings (UID)
                       #        "lines": list of strings (data lines of the object), the first line contains the object UID between brackets
                       #       }
                       #    ...
                       #   }

    # dictionary containing the possible GO objects types and their possible child objects types
    go_objects_children_dic = {
        'Header': [],
        'Organ': ['General', 'Manual', 'Panel', 'Switch', 'WindchestGroup'],  # are not real children but are presented as children in the objects tree
        'Coupler': ['Manual', 'PanelElement', 'Switch'],
        'Divisional': ['PanelElement'],
        'DivisionalCoupler': ['Manual', 'PanelElement', 'Switch'],
        'Enclosure': ['PanelElement'],
        'General': ['PanelElement'],
        'Manual': ['Coupler', 'Divisional', 'PanelElement', 'Stop', 'Switch', 'Tremulant'],
        'Panel': ['PanelElement', 'PanelImage'],
        'PanelElement': [],
        'PanelImage': [],
        'Rank': [],
        'Stop': ['PanelElement', 'Rank', 'Switch'],
        'Switch': ['PanelElement', 'Switch'],
        'Tremulant': ['PanelElement', 'Switch'],
        'WindchestGroup': ['Enclosure', 'Stop', 'Rank', 'Tremulant'],
        # old panel format
        'Image': [],
        'Label': [],
        'SetterElement': [],
        'PanelCoupler': ['Coupler'],
        'PanelDivisional': ['Divisional'],
        'PanelDivisionalCoupler': ['DivisionalCoupler'],
        'PanelEnclosure': ['Enclosure'],
        'PanelGeneral': ['General'],
        'PanelLabel': ['Label'],
        'PanelSetterElement': ['SetterElement'],
        'PanelStop': ['Stop'],
        'PanelSwitch': ['Switch'],
        'PanelTremulant': ['Tremulant'],
        }

    # dictionary containing the possible GO objects types and their possible parent objects types
    # it is built automatically from go_objects_children_dic in the function objects_templates_load
    go_objects_parents_dic = {}

    # dictionary in which are stored the GO objects templates (loaded from the file GoObjectsTemplates.txt located in the resources sub-folder)
    go_templates_dic = {}

    #-------------------------------------------------------------------------------------------------
    def reset_all_data(self):
        # reset all the data of the class

        self.odf_file_name = ''
        self.odf_file_encoding = ENCODING_ISO_8859_1
        self.new_panel_format_bool = False
        self.odf_data_dic.clear()

    #-------------------------------------------------------------------------------------------------
    def objects_templates_load(self):
        # load the GO objects templates from the file GoObjectsTemplates.txt (if it is present and there is no error)
        # return True or False whether the operation has succeeded or not

        # initialize the go_objects_parents_dic dictionary from the keys of the go_objects_children_dic dictionary
        for object_type in self.go_objects_children_dic.keys():
            self.go_objects_parents_dic[object_type] = []
        # build the go_objects_parents_dic dictionary from the go_objects_children_dic dictionary content (create the opposite kinship)
        for parent_object_type, object_children_list in self.go_objects_children_dic.items():
            # scan the objects types of the object children dictionary
            for child_object_type in object_children_list:
                if parent_object_type not in ('Organ'): # do not include Organ in the parents
                    self.go_objects_parents_dic[child_object_type].append(parent_object_type)

        if len(self.go_templates_dic) == 0:
            # the dictionary has not been loaded yet

            file_name = os.path.dirname(__file__) + os.path.sep + 'resources' + os.path.sep + 'GoObjectsTemplates.txt'

            try:
                with open(file_name, 'r') as f:
                    self.go_templates_dic = eval(f.read())

            except OSError as err:
                # it has not be possible to open the file
                logs.add(f'ERROR Cannot open the file "{file_name}" : {err}')
            except SyntaxError as err:
                # syntax error in the dictionary structure which is in the file
                logs.add(f'ERROR Syntax error in the file "{file_name}" : {err}')
                logs.add( 'ERROR Fix the issue in the file then restart OdfEdit')
            except:
                # other error
                logs.add(f'ERROR while opening the file "{file_name}"')
            return False

        return True

    #-----------------------------------------------------------------------------------------------
    def load_from_file(self, file_name):
        # load the data of the given ODF
        # returns True/False whether the file has been loaded correctly or not

        file_name = path2ospath(file_name)

        # open the given ODF in read mode, and check its encoding format
        try:
            file = open(file_name, mode='r', encoding=ENCODING_ISO_8859_1)
        except OSError as err:
            # it has not be possible to open the file
            logs.add(f"Cannot open the file. {err}")
            valid_odf_file_bool = False
        else:
            if file.readline(3) == "ï»¿":  # UTF-8 BOM file encoding header
                # close the file
                file.close()
                # reopen the file with the proper encoding format
                file = open(file_name, mode='r', encoding=ENCODING_UTF8_BOM)
                self.odf_file_encoding = ENCODING_UTF8_BOM
            else:
                file.seek(0)  # reset the position of the cursor at the beginning of the file
                self.odf_file_encoding = ENCODING_ISO_8859_1
            # store the name of the ODF
            self.odf_file_name = file_name
            valid_odf_file_bool = True

            # clear data before to scan the file
            self.odf_data_dic.clear()

            object_types_list = []  # list containing the types of objects loaded from the ODF, to display statistics
            object_uid = 'Header'   # UID of the object currently recovered, header section by default
            object_dic = self.object_new()    # dictionary of the object currently recovered
            total_attr_nb = 0       # number of whole attributes in the loaded ODF
            file_lines_nb = 0       # number of lines in the loaded ODF
            max_new_id = 999

            for line in file:
                # scan the lines of the ODF to load
                file_lines_nb += 1

                # remove the ending \n character if present in the current line
                if line[-1:] == '\n': line = line[:-1]

                # recover and check the syntax of the data present in the current line
                (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)
                if error_msg != None:
                    logs.add(f'ERROR in {object_uid} section, line "{line}" : {error_msg}')
                    line = ';!!! ' + line
                    attr_name = None

                if attr_name == 'uid':
                    # line containing an object UID

                    # add in the ODF data the object in construction for the previous UID, if it has lines inside
                    # in case of Header the UID line is not considered, so 2 lines min are expected
                    if (object_uid == 'Header' and len(object_dic['lines']) > 1) or len(object_dic['lines']) > 0:
                        self.odf_data_dic[object_uid] = object_dic

                    object_uid = attr_value

                    while object_uid in self.odf_data_dic.keys() and max_new_id > 0:
                        object_uid = object_uid[:-3] + str(max_new_id).zfill(3)
                        max_new_id -= 1
                    if object_uid != attr_value:
                        logs.add(f"WARNING : another occurence of the object {attr_value} is present in the ODF, it has been renamed in {object_uid}")

                    # create the next object in construction
                    object_dic = self.object_new()

                    # store the object type for statistics
                    object_type = self.object_type_get(object_uid)
                    if object_type not in object_types_list:
                        object_types_list.append(object_type)

                elif attr_name != None and object_uid == 'Header':
                    # attribute line in the header : it is an error
                    logs.add('ERROR only comments are allowed in the header section')
                    line = ';!!! ' + line
                    attr_name = None

                elif attr_name != None :
                    total_attr_nb += 1

                # add the current line to the object in construction
                if len(line) > 0:
                    object_dic['lines'].append(line)

            # add in the ODF dictionary the dictionary of the last loaded object
            if len(object_dic) > 0:
                self.odf_data_dic[object_uid] = object_dic

            # close the ODF
            file.close()

            # update the kinship links between the objects
            self.objects_kinship_update()

            if self.odf_file_encoding == ENCODING_UTF8_BOM:
                file_encoding = 'UTF-8-BOM'
            else:
                file_encoding = 'ISO_8859_1'

            logs.add(f'GrandOrgue ODF loaded "{file_name}"')
            logs.add(f'{file_lines_nb:,} lines, file encoding {file_encoding}')

            logs.add(f'{total_attr_nb:,} attributes among {len(self.odf_data_dic)-1:,} sections among {len(object_types_list)} section types')

            # update the panel format flag from the ODF data content
            self.check_panel_format()

        return valid_odf_file_bool

    #-------------------------------------------------------------------------------------------------
    def save_to_file(self, file_name, file_encoding):
        # save the odf_data_dic in the given ODF and with the given encoding format (ENCODING_ISO_8859_1 or ENCODING_UTF8_BOM)
        # if no file name is given, the saving is done in the already loaded ODF file ('Save as' feature)
        # returns True/False whether the writting in file has been done correctly or not

        if file_encoding not in (ENCODING_ISO_8859_1, ENCODING_UTF8_BOM):
            logs.add(f"INTERNAL ERROR wrong encoding {file_encoding} given to save_to_file")
            return False

        file_saved_bool = False

        if len(self.odf_data_dic) == 0:
            # the ODF dictionary is empty, there are no data to save
            logs.add(f"None data to save in the file {file_name}")
        elif file_name == '' and self.odf_file_name == '':
            # no file name known, should not occur, so no possibility to make the save operation
            pass
        else:
            # open the given ODF in write mode
            if file_name == '':
                # no given file name, make the saving in the already loaded ODF
                file_name = self.odf_file_name

            # check if the file name has an extension, if not add the .organ extension
            if file_name[-6:] != '.organ':
                file_name += '.organ'

            try:
                file = open(file_name, mode='w', encoding=file_encoding)
            except OSError as err:
                logs.add(f"Cannot write in the file. {err}")
            else:
                # write the ODF data dictionary content in the ODF
                # eliminating the blank lines and ensuring a blanck line before each object section start
                # write the Header and Organ objects if defined
                for object_uid in ('Header', 'Organ'):
                    if object_uid in self.odf_data_dic.keys():
                        for line in self.odf_data_dic[object_uid]['lines']:
                            if line != '':
                                if line[0] == '[':
                                    # start of a section : add a blank line before
                                    file.write('\n\n' + line)
                                else:
                                    file.write('\n' + line)
                # write the other objects by UID ascending order
                for object_uid in sorted(self.odf_data_dic.keys()):
                    # scan the sorted objects of the ODF data
                    if object_uid not in ('Header', 'Organ'):
                        for line in self.odf_data_dic[object_uid]['lines']:
                            # scan the lines of the current object
                            if line != '':
                                if line[0] == '[':
                                    # start of a section : add a blank line before
                                    file.write('\n\n' + line)
                                else:
                                    file.write('\n' + line)

                file.close()
                file_saved_bool = True

                # store the name of the ODF file
                self.odf_file_name = file_name

                if file_encoding == ENCODING_UTF8_BOM:
                    file_enc = 'UTF-8-BOM'
                else:
                    file_enc = 'ISO_8859_1'

                logs.add(f'Data saved in file "{self.odf_file_name}" with encoding {file_enc}')

        return file_saved_bool

    #-------------------------------------------------------------------------------------------------
    def object_line_split(self, line):
        # check the syntax of the given object line and extract from it the attribute name + attribute value + comment
        # return a tuple containing : (error message, attribute name, attribute value, comment)
        # attribute name = 'uid' if the given line contains an object UID between brackets, the UID is in the attribute value
        # error message = an error description message in case a syntax error has been detected in the given line, or None if no error found

        return self.check_object_line(line)

    #-------------------------------------------------------------------------------------------------
    def object_line_join(self, attr_name, attr_value, comment=None):
        # join in a single string line in ODF format the provided data, return this line

        attr_value = str(attr_value)
        line = ''

        if attr_name == 'uid':
            # UID of the object, start of a section
            if comment != None:
                line = '[' + attr_value + ']' + comment
            else:
                line = '[' + attr_value + ']'

        elif attr_name != None:
            # attribute line
            if comment != None:
                line = attr_name + '=' + attr_value + comment
            else:
                line = attr_name + '=' + attr_value

        elif comment != None:
            # comment line
            line = comment

        return line

    #-------------------------------------------------------------------------------------------------
    def object_lines_read(self, object_uid):
        # return in a list the lines of the given object UID (the first line contains the object UID in brackets)
        # or an empty list if the object UID doesn't exist

        object_dic = self.object_dic_get(object_uid)
        if object_dic != None:
            return list(object_dic['lines'])

        return []

    #-------------------------------------------------------------------------------------------------
    def object_lines_search(self, object_uid, text_to_search):
        # return in a list the lines of the given object UID which contain the given text to search
        # or an empty list if the object UID doesn't exist or no line found

        object_dic = self.object_dic_get(object_uid)
        if object_dic != None:
            found_lines_list = []
            for line in object_dic['lines']:
                if text_to_search in line:
                    found_lines_list.append(line)
            return found_lines_list

        return []

    #-------------------------------------------------------------------------------------------------
    def object_lines_write(self, object_lines_list, expected_object_uid=None, parent_object_uid=None):
        # write the given object attributes lines in the ODF
        # the first line of the list must contain an object UID between brackets else the lines are considered as being the Header section
        # if expected_object_uid is given, it is the UID expected in the first line of the given lines list
        # if parent_object_uid is given, link the object defined in the lines (if is new in the ODF) to this parent
        # return the UID of the added/updated/renamed object or None if a syntax error has been detected

        # expected_object_uid | object_uid (defined in lines)    | action in the ODF
        # --------------------|----------------------------------|------------------------------------
        # None                | Object999 (not in ODF)           | Object999 added
        # None                | Object999 (in ODF)               | Object999 updated
        # Object999           | Object999 (not in ODF)           | Object999 added
        # Object999           | Object999 (in ODF)               | Object999 updated
        # Object999           | Object888 (different type)       | error, Object999 cannot be renamed in Object888 which has different object type
        # Object999           | Object888 (same type in ODF)     | error, Object999 cannot be renamed in Object888 which already exists
        # Object999           | Object888 (same type not in ODF) | Object999 renamed in Object888 and Object888 updated

        # set the initial object UID before to scan the given lines lise
        if expected_object_uid == None:
            object_uid = 'Header'   # Header object by default if no expected UID
        else:
            object_uid = expected_object_uid

        tmp_object_dic = self.object_new()  # object to store the provided lines and their UID

        for i, line in enumerate(object_lines_list):
            # scan the lines of the given list to check them, to recover the UID and store them in tmp_object_dic

            # recover and check the data present in the current line
            (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)

            if error_msg != None:
                logs.add(f'ERROR in line "{line}" : {error_msg}')
                return None
            if i == 0 and attr_name != 'uid' and object_uid != 'Header':
                logs.add('ERROR : an object identifier is expected in the first line')
                return None
            if i > 0 and attr_name == 'uid':
                logs.add('ERROR : an object identifier can be defined only in the first line')
                return None
            if object_uid == 'Header' and attr_name != None:
                logs.add('ERROR : only comments are allowed in the header section')
                return None

            if i == 0 and attr_name == 'uid' and attr_value != None:
                # there is a valid UID in first line
                object_uid = attr_value

            # add the current line to the temporary object
            tmp_object_dic['lines'].append(line)

        # recover the object corresponding to the UID defined in the first of the given lines if defined in the ODF
        object_dic = self.object_dic_get(object_uid)

        if expected_object_uid != None and self.object_type_get(expected_object_uid) != self.object_type_get(object_uid):
            logs.add(f'ERROR : {expected_object_uid} cannot be renamed in {object_uid} which is of different type')
            object_uid = None

        elif expected_object_uid != None and expected_object_uid != object_uid and object_dic != None:
            logs.add(f'ERROR : {expected_object_uid} cannot be renamed in {object_uid} which is already defined in the ODF')
            object_uid = None

        elif object_dic == None and expected_object_uid in (None, object_uid):
            # the object UID defined in the given lines is not present in the ODF
            # and there is no expected UID or it is equal to the UID defined in the given lines
            # add an object in the ODF with the given lines
            self.odf_data_dic[object_uid] = tmp_object_dic
            logs.add(f"{object_uid} : added")

            if parent_object_uid != None:
                # link the new object to its parent if provided
                self.object_parent2child_link(parent_object_uid, object_uid, 'link')
            elif self.object_type_get(object_uid) in ('PanelElement', 'PanelImage'):
                # link the new object to its parent panel if it is PanelElement or PanelImage object
                self.object_parent2child_link(object_uid[:8], object_uid, 'link')

        else:
            # other cases
            if expected_object_uid not in (None, object_uid):
                # the UID defined in the lines is not the expected one
                # rename the expected object UID to the UID defined in the lines
                object_uid = self.object_rename(expected_object_uid, object_uid)
                object_dic = self.object_dic_get(object_uid)

            if object_dic != None:
                # update the object having the UID defined in the lines
                object_dic['lines'] = tmp_object_dic['lines']
                logs.add(f"{object_uid} : updated")

        # update the number of object type in the Organ object if necessary
        self.object_organ_numbers_update(object_uid)

        # update the kinship links between the objects
        self.objects_kinship_update()

        return object_uid

    #-------------------------------------------------------------------------------------------------
    def object_new(self):
        # return the dictionary of a new and empty object

        object_dic = {}
        object_dic['names'] = []    # list with the various names of the object
        object_dic['parents'] = []  # list with the UID of parent objects
        object_dic['children'] = [] # list with the UID of children objects
        object_dic['lines'] = [] # list with the data lines (including the UID) of the object

        return object_dic

    #-------------------------------------------------------------------------------------------------
    def object_add(self, object_type, parent_uid, object_lines_list=[]):
        # add in the ODF data an object corresponding to the given object type
        # and having the attributes of the given lines list if provided or of the template if defined
        # return the UID of the created object, or None if an error occured

        if not object_type in self.go_objects_children_dic.keys():
            logs.add(f'INTERNAL ERROR in object_add : object type {object_type} is unknown')
            return None

        if len(object_lines_list) == 0:
            # no object lines list provided, recover the one from the template if defined
            if object_type in self.go_templates_dic.keys():
                # the given object type has a template defined
                # recover the lines of the template
                object_lines_list = list(self.go_templates_dic[object_type])

        # define the UID of the object to add by using a free UID for the given object type
        # and having the given parent UID
        object_uid = self.object_type_free_uid_get(object_type, parent_uid)

        if object_uid != None:
            # a new object can be added
            # add the object UID in first line of the attribute lines list if it is not the Header object
            if object_uid != 'Header':
                object_lines_list.insert(0, '[' + object_uid + ']')

            # add the new object and its attribute lines in the ODF, link it to the parent UID
            object_uid = self.object_lines_write(object_lines_list, object_uid, parent_uid)

        return object_uid

    #-------------------------------------------------------------------------------------------------
    def object_copy(self, object_uid, parent_uid):
        # make a copy of the object having given UID and place it as child of the given parent object UID (can be None)
        # return the UID of the copied object, or None if an error occured

        # recover the lines of the object to copy
        attribute_lines_list = self.object_lines_read(object_uid)
        if len(attribute_lines_list) > 0:
            # remove the first line which contains the original object UID (the new UID will be added in object_add)
            attribute_lines_list.pop(0)
            # add in the ODF a copy of the given object
            object_type = self.object_type_get(object_uid)
            new_object_uid = self.object_add(object_type, parent_uid, attribute_lines_list)
        else:
            new_object_uid = None

        if new_object_uid != None:
            logs.add(f'{new_object_uid} : copied from {object_uid}')
        else:
            logs.add(f'ERROR cannot make a copy of {object_uid}')

        return new_object_uid

    #-------------------------------------------------------------------------------------------------
    def object_link(self, object_uid, kinship_objects_list, relationship):
        # link the given object to the objects of the given kinship objects list with the given relationship (TO_PARENT or TO_CHILD)
        # return the object_uid (which can have be renamed in case of children of a Panel or Manual) or None if an issue occured

        if relationship not in (TO_PARENT, TO_CHILD):
            logs.add(f'INTERNAL ERROR in object_link : wrong given relationship {relationship}')

        object_dic = self.object_dic_get(object_uid)
        if object_dic == None:
            return None
        object_type = self.object_type_get(object_uid)

        # recover the list of the current kinship list (parents or children) of the given object
        current_kinship_list = sorted(self.object_kinship_list_get(object_uid, relationship))

        # build a dictionary with as keys the object types which are present in the given kinship objects list
        #                     and as values how many ther are in the list
        kinship_objects_types_dic = {}
        for obj_uid in kinship_objects_list:
            obj_type = self.object_type_get(obj_uid)
            if obj_type not in kinship_objects_types_dic.keys():
                kinship_objects_types_dic[obj_type] = 0
            kinship_objects_types_dic[obj_type] += 1

        # check if the given kinship list contains more parents/children type than allowed for the given object type
        if relationship == TO_PARENT:
            if object_type in ('PanelElement', 'PanelImage'):
                if 'Panel' in kinship_objects_types_dic.keys():
                    if kinship_objects_types_dic['Panel'] > 1:
                        # PanelElement or PanelImage with more than one Panel parent
                        logs.add(f'A {object_type} section cannot have more than one parent Panel.')
                        return None
                else:
                    # PanelElement or PanelImage with none Panel parent
                    logs.add(f'A {object_type} section must have one parent Panel.')
                    return None
            elif (object_type in ('Rank', 'Stop') and
                  'WindchestGroup' in kinship_objects_types_dic.keys() and kinship_objects_types_dic['WindchestGroup'] > 1):
                # Rank and Stop object with more than one parent WindchestGroup
                logs.add(f'A {object_type} section cannot have more than one parent WindchestGroup.')
                return None

        ret_object_uid = object_uid  # UID which will be returned by the function
        if relationship == TO_PARENT:
            # unlink the given object from its current parents which have no more to be a parent
            for parent_uid in current_kinship_list:
                # scan the current parent objects of the given object
                if parent_uid not in kinship_objects_list:
                    # the current parent has no more to be a parent : remove the link
                    self.object_parent2child_link(parent_uid, object_uid, 'unlink')

            # link the given object to the objects which must be now a parent
            for parent_uid in kinship_objects_list:
                # scan the given parent objects
                if parent_uid not in current_kinship_list:
                    # it is not yet a parent : add the link
                    ret_object_uid = self.object_parent2child_link(parent_uid, object_uid, 'link')
        else:
            # unlink the given object from its current children which have no more to be its child
            for child_uid in current_kinship_list:
                # scan the current child objects of the given object
                if child_uid not in kinship_objects_list:
                    # the current child has no more to be a child : remove the link
                    self.object_parent2child_link(object_uid, child_uid, 'unlink')

            # link the given object to the objects which must be now its child
            for child_uid in kinship_objects_list:
                # scan the given children objects
                if child_uid not in current_kinship_list:
                    # it is not yet a child : add the link
                    self.object_parent2child_link(object_uid, child_uid, 'link')

        # update the kinship links between the objects of the ODF
        self.objects_kinship_update()

        return ret_object_uid

    #-------------------------------------------------------------------------------------------------
    def object_rename(self, object_uid, new_object_uid, update_links_bool=True, same_uid_shift=0):
        # rename in the ODF data the object having the given UID to the given new UID (only the three ending digits of the UID can be changed)
        # if same_uid_shift is is not 0 and an object already has the same UID as the given new UID,
        #       its UID is shifted by the given shift value to permit the given UID to have the new UID
        # return the new UID of the object, or None if failure

        # check the provided object UIDs
        if object_uid in ('Header', 'Organ'):
            logs.add("INTERNAL ERROR in object_rename : Header and Organ objects cannot be renamed")
            return None

        if self.object_type_get(object_uid) != self.object_type_get(new_object_uid):
            # current and new objects are not of the same object type
            logs.add(f"INTERNAL ERROR in object_rename : objects {object_uid} and {new_object_uid} are not of the same type")
            return None

        if self.object_dic_get(object_uid) == None:
            # there is no object in the ODF having the given object UID
            logs.add(f"INTERNAL ERROR in object_rename : object {object_uid} does not exit")
            return None

        if self.object_dic_get(new_object_uid) != None:
            # there is already in the ODF an object having the same UID as the given new UID
            if same_uid_shift != 0:
                # the object having already the new UID has to be renamed to the UID + same_uid_shift value
                # apply recursively the UID renaming to next/previous consecutive objects
                shifted_object_uid = new_object_uid[:-3] + str(int(new_object_uid[-3:]) + same_uid_shift).zfill(3)
                self.object_rename(new_object_uid, shifted_object_uid, update_links_bool, same_uid_shift)
            else:
                logs.add(f"Warning : cannot rename {object_uid} in {new_object_uid} which is already used")
                return None

        # rename the reference to the object UID in its parents or children objects if necessary

        object_type = self.object_type_get(object_uid)
        object_dic = self.object_dic_get(object_uid)
        object_id = self.object_id_get(object_uid)
        new_object_id = self.object_id_get(new_object_uid)

        if object_type == 'Panel':
            # rename the UID of all the children (PanelElement and PanelImage objects) of the panel to rename
            # so that they contain the new UID of the parent panel
            for child_uid in object_dic['children']:
                # scan the children of the Panel to rename
                new_child_uid = new_object_uid + child_uid[8:]   # for example from Panel000Element001 to Panel001Element001
                # move the child object data under another key which has the new child UID
                self.odf_data_dic[new_child_uid] = self.odf_data_dic.pop(child_uid)
                # update the UID in the first line of the child object
                (error_msg, attr_name, attr_value, comment) = self.object_line_split(self.odf_data_dic[new_child_uid]['lines'][0])
                if attr_name == 'uid':
                    # it is the first line with the UID, rename it
                    self.odf_data_dic[new_child_uid]['lines'][0] = self.object_line_join(attr_name, new_child_uid, comment)
                logs.add(f"{child_uid} : renamed to {new_child_uid}")

        elif object_type == 'WindchestGroup':
            # rename the reference to the new WindchestGroup UID in its children objects Stop or Rank
            for child_uid in object_dic['children']:
                # scan the children of the WindchestGroup to rename
                if self.object_type_get(child_uid) in ('Stop', 'Rank'):
                    self.object_attr_value_set(child_uid, 'WindchestGroup', new_object_id)
                    logs.add(f"{child_uid} : attribute WindchestGroup changed from {object_id} to {new_object_id}")

        elif object_type not in ('PanelElement', 'PanelImage'):
            # other kinds of objects (but PanelElement and PanelImage which are not referenced in a parent or child)

            # rename the reference to the object to rename in its parents
            for parent_uid in sorted(object_dic['parents']):
                # scan the parents of the object to rename
                parent_type = self.object_type_get(parent_uid)
                parent_dic = self.object_dic_get(parent_uid)
                # define the name of the attribute in the parent which refers to the type of the object to rename
                if object_type == 'Manual' and parent_type == 'Coupler':
                    ref_attr_name = 'DestinationManual'
                else:
                    ref_attr_name = object_type

                for line in parent_dic['lines']:
                    # scan the lines of the parent object to find the attribute refering to the object to rename
                    if line.startswith(ref_attr_name):
                        # the current line contains a reference to the type of the object to rename
                        (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)
                        if (attr_name != None and attr_value != '' and ref_attr_name in (attr_name, attr_name[:-3]) and
                            abs(int(attr_value)) == object_id):
                            # the current line refers to the object to rename
                            # write in this line the new ID of the object to rename
                            self.object_attr_value_set(parent_dic, attr_name, str(new_object_id).zfill(3))
                            logs.add(f"{parent_uid} : attribute {attr_name} changed from {str(object_id).zfill(3)} to {str(new_object_id).zfill(3)}")
                            # sort in the parent object the references to the object type if necessary
                            self.object_children_ref_update(parent_uid, object_type)
                            break

            # rename the reference to the object to rename in its children if necessary
            for child_uid in sorted(object_dic['children']):
                # scan the children of the object to rename
                if self.object_type_get(child_uid) == 'PanelElement':
                    # rename in the PanelElement the reference to its parent to rename
                    self.object_attr_value_set(child_uid, object_type, str(new_object_id).zfill(3))
                    logs.add(f"{child_uid} : attribute {object_type} changed from {str(object_id).zfill(3)} to {str(new_object_id).zfill(3)}")

            if object_type == 'Manual':
                # a manual has to be renamed

                # rename the StopManual and CouplerManual attributes value in General objects
                for obj_uid in self.odf_data_dic.keys():
                    # scan the objects of the ODF data
                    if obj_uid.startswith('General'):
                        # the current object is a General
                        obj_dic = self.object_dic_get(obj_uid)
                        for line in obj_dic['lines']:
                            # scan the lines of the current General object
                            if line.startswith(('StopManual', 'CouplerManual')):
                                # the current line contains a reference to a manual
                                (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)
                                if attr_name != None and int(attr_value) == int(object_uid[-3:]):
                                    # the current line refers to the manual to rename
                                    # write in this line the new ID of the manual to rename
                                    self.object_attr_value_set(obj_dic, attr_name, new_object_uid[-3:])
                                    logs.add(f"{obj_uid} : attribute {attr_name} updated")

                # rename the ID of the children Coupler, Divisional, Stop objects to align its first digit on the new manual ID
                new_manual_id = int(new_object_uid[-1:])
                for child_uid in list(object_dic['children']):
                    # scan the children of the Manual to rename
                    if self.object_type_get(child_uid) in ('Coupler', 'Divisional', 'Stop'):
                        # rename the current child object UID by changing the first digit of its ID with the number of the parent manual
                        self.object_rename(child_uid, child_uid[:-3] + str(new_manual_id) + child_uid[-2:], False)

        # rename the UID of the object to rename
        # move the given object under another key in the ODF data dictionary which has the new UID
        object_dic = self.odf_data_dic[new_object_uid] = self.odf_data_dic.pop(object_uid)
        # update the UID in the first line with the new object UID
        (error_msg, attr_name, attr_value, comment) = self.object_line_split(object_dic['lines'][0])
        if attr_name == 'uid':
            # it is actually the first line with the UID which as been recovered, update it
            object_dic['lines'][0] = self.object_line_join(attr_name, new_object_uid, comment)

        logs.add(f"{object_uid} : renamed to {new_object_uid}")

        # update in the Organ object the number of objects type corresponding to the renamed object if applicable
        self.object_organ_numbers_update(new_object_uid)

        if update_links_bool:
            # update the kinship links between the objects
            self.objects_kinship_update()

        return new_object_uid

    #-------------------------------------------------------------------------------------------------
    def object_delete(self, object_uid):
        # delete in the ODF data the object having the given UID
        # return True or False whether the deletion has been done or not

        object_dic = self.object_dic_get(object_uid)
        if object_dic != None:
            # unlink the given object with its parents
            for parent_uid in object_dic['parents']:
                self.object_parent2child_link(parent_uid, object_uid, 'unlink')
            # unlink the given object with its children
            for child_uid in object_dic['children']:
                self.object_parent2child_link(object_uid, child_uid, 'unlink')

            # if the object is a Panel, delete all his children in the ODF data (PanelElement or PanelImage or Panelxxxx)
            if self.object_type_get(object_uid) == 'Panel':
                for child_uid in object_dic['children']:
                    del self.odf_data_dic[child_uid]
                    logs.add(f"{child_uid} : deleted")

            # delete the given object in the ODF data
            del self.odf_data_dic[object_uid]
            logs.add(f"{object_uid} : deleted")

            # update the NumberOf attribute of the parent Panel if a PanelElement or PanelImage has been deleted
            self.object_panel_numbers_update(object_uid)

            # update the number of object type in the Organ object if necessary
            self.object_organ_numbers_update(object_uid)

            # rename the ID of the objects of the same type to fill the gap created by the deleted object
            object_type = self.object_type_get(object_uid)
            if object_type in ('Enclosure', 'General', 'Panel', 'PanelElement', 'PanelImage', 'Rank', 'Switch', 'Tremulant', 'WindchestGroup'):
                object_type = self.object_type_get(object_uid, False)
                objects_uid_list = sorted(self.objects_type_list_get(object_type))
                if len(objects_uid_list) > 0:
                    expected_object_id = 0 if object_type == 'Panel' else 1
                    for obj_uid in objects_uid_list:
                        # scan the list of the existing objects UID of the same type
                        obj_id = self.object_id_get(obj_uid)
                        if obj_id > expected_object_id:
                            shifted_obj_uid = object_type + str(expected_object_id).zfill(3)
                            self.object_rename(obj_uid, shifted_obj_uid, False)
                        expected_object_id += 1

            # update the kinship links between the objects
            self.objects_kinship_update()

            return True

        logs.add(f"ERROR {object_uid} cannot be deleted because not found")
        return False

    #-------------------------------------------------------------------------------------------------
    def object_id_get(self, object_uid):
        # return the ID (integer) of the given object UID
        # or None if no UID is given (empty or None) or the last 3 characters of the UID are not digits
        # for example will return 12 if the given UID is from Stop012
        # return None if the ID is not valid : there are not 3 digits at the end of the UID

        if object_uid in (None, '') or not object_uid[-3:].isdigit():
            # the last 3 characters are not all digits
            return None

        return int(object_uid[-3:])

    #-------------------------------------------------------------------------------------------------
    def object_type_get(self, object_uid, none_digit=True):
        # return the type (string) of the given object UID or None if no UID is given (empty or None)
        # for example will return Stop if the given UID is Stop102
        # if none_digit = True,  for example will return PanelElement    if the given UID is Panel999Element999
        # if none_digit = False, for example will return Panel999Element if the given UID is Panel999Element999 (remove only the last 3 digits)

        if object_uid not in (None, ''):
            if none_digit:
                # remove all digits of the UID string
                object_type = ''
                for c in object_uid:
                    if not c.isdigit():
                        object_type += c
            else:
                # remove the last three characters of the UID string
                object_type = object_uid[:-3]
        else:
            object_type = None

        return object_type

    #-------------------------------------------------------------------------------------------------
    def object_type_free_uid_get(self, object_type, parent_uid=None, starting_id=0):
        # return the first unused object UID for the given object type and being child of the given parent UID if provided (required for Manual and Panel parents)
        # the given object type must be with none internal digit when applicable (for example PanelElement and not Panel999Element)

        if object_type in ('Header', 'Organ'):
            if object_type in self.odf_data_dic.keys():
                # there is already a Header or Organ object in the ODF
                return None

            return object_type

        parent_type = self.object_type_get(parent_uid)

        if object_type in ('Manual', 'Panel'):
            # Manual and Panel ID can start from 0
            free_id = 0
        elif parent_type == 'Panel':
            # if the parent is a Panel object, the UID to return has to include the panel UID
            free_id = 1
            object_type = parent_uid + object_type[5:]   # for example Panel001Element or Panel002Image
        elif parent_type == 'Manual' and object_type in ('Coupler', 'Divisional', 'Stop'):
            # if the parent is a Manual object and the object is a Coupler/Divisional/Stop
            # the first digit of the ID to return has to be the manual ID
            free_id = int(parent_uid[-1]) * 100 + 1
        else:
            free_id = 1
        free_id = max(free_id, starting_id)

        while free_id <= 998 :  # let 999 free to permit objects swap in the GUI with drag&drop
            # scan the ID values until finding a unused one in the given object type
            free_uid = object_type + str(free_id).zfill(3)
            if free_uid not in self.odf_data_dic.keys():
                # the current UID is not present in the ODF
                return free_uid

            free_id += 1

        return None

    #-------------------------------------------------------------------------------------------------
    def object_dic_get(self, object_uid):
        # return the dictionary of the object of the ODF dictionary having the given UID

        try:
            return self.odf_data_dic[object_uid]
        except:
            # object not existing
            return None

    #-------------------------------------------------------------------------------------------------
    def object_names_get(self, object_uid):
        # return the UID followed by the names between parenthesis (if defined) of the given object UID

        object_dic = self.object_dic_get(object_uid)
        if object_dic != None and len(object_dic['names']) > 0:
            return object_uid + ' (' + ' | '.join(object_dic['names']) + ')'

        return object_uid

    #-------------------------------------------------------------------------------------------------
    def object_attr_value_get(self, object_dic_or_uid, attribute_name):
        # return the value of the given attribute defined in the given object dictionary or UID
        # return None if the given object or the attribute doesn't exist

        if object_dic_or_uid == None:
            return None

        if isinstance(object_dic_or_uid, str):
            # the given parameter is a string, so an UID
            object_dic = self.object_dic_get(object_dic_or_uid)
        else:
            # the given parameter is not a string, so an object dictionary
            object_dic = object_dic_or_uid

        if object_dic != None:
            for line in object_dic['lines']:
                # scan the attribute lines of the object
                if line.startswith(attribute_name):
                    (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)
                    if attr_name == attribute_name:
                        # given attribute found : return its value
                        return attr_value

        return None

    #-------------------------------------------------------------------------------------------------
    def object_attr_value_set(self, object_dic_or_uid, attribute_name, attribute_value, attribute_comment=None):
        # write the given value in the given attribute defined in the given object dictionary or UID
        # return True or False whether it has been possible or not to the set the value

        if isinstance(object_dic_or_uid, str):
            # the given parameter is a string, so an UID
            object_dic = self.object_dic_get(object_dic_or_uid)
        else:
            # the given parameter is not a string, so an object dictionary
            object_dic = object_dic_or_uid

        if object_dic != None:
            for i, line in enumerate(object_dic['lines']):
                # scan the attribute lines of the object
                if line.startswith(attribute_name):
                    (error_msg, attr_name, attr_value, attr_comment) = self.object_line_split(line)
                    if attr_name == attribute_name:
                        # the given attribute is found : update its value and comment if given
                        if attribute_comment == None:
                            object_dic['lines'][i] = self.object_line_join(attribute_name, attribute_value, attr_comment)
                        else:
                            object_dic['lines'][i] = self.object_line_join(attribute_name, attribute_value, attribute_comment)
                        return True

            # attribute not present in the object lines, add a line for it
            object_dic['lines'].append(self.object_line_join(attribute_name, attribute_value, attribute_comment))
            return True

        return False

    #-------------------------------------------------------------------------------------------------
    def object_parent2child_link(self, parent_object_uid, child_object_uid, operation):
        # link/unlink the given child object to the given parent object (both must already exist in the ODF data)
        # operation parameter can be 'link' or 'unlink'
        # return the UID of the child object (can be renamed in case of child of a Panel) or None if an error occured

        # types of referencing parent to child :
        #   WindchestGroup -> Stop (rank inside) or Rank : WindchestGroup ID is set in the attribute WindchestGroup of the Stop or Rank object
        #   Panel -> PanelElement or PanelImage : PanelElement or PanelImage UID starts with the parent Panel UID
        #   PanelElement -> Coupler/General/Manual/... : only one child can be defined
        #   Stop -> Switch (example) : the value of the Stop attribute Switchxxx is the Switch ID

        if (parent_object_uid == None or child_object_uid == None):
            return None

        if operation not in ('link', 'unlink'):
            logs.add(f'INTERNAL ERROR in object_parent2child_link : wrong requested operation name "{operation}"')
            return None

        # recover the dictionary, type and ID of the given parent
        parent_object_dic = self.object_dic_get(parent_object_uid)
        if parent_object_dic == None:
            logs.add(f"INTERNAL ERROR in object_parent2child_link : parent object {parent_object_uid} does not exit")
            return None
        parent_object_type = self.object_type_get(parent_object_uid)
        parent_object_id = self.object_id_get(parent_object_uid)

        # recover the dictionary, type and ID of the given child
        child_object_dic = self.object_dic_get(child_object_uid)
        if child_object_dic == None:
            logs.add(f"INTERNAL ERROR in object_parent2child_link : child object {child_object_uid} does not exit")
            return None
        child_object_type = self.object_type_get(child_object_uid)
        child_object_id = self.object_id_get(child_object_uid)

        # check that the given parent object can have as a child the given child object type
        if child_object_type not in self.go_objects_children_dic[parent_object_type] or parent_object_type == 'Organ':
            logs.add(f"INTERNAL ERROR in object_parent2child_link : the object type {parent_object_type} cannot have the object type {child_object_type} as a child")
            return None

        if parent_object_type == 'Coupler' and child_object_type == 'Manual':
            # set or erase the ID of the child Manual in the attribute DestinationManual of the parent Coupler object
            if operation == 'link':
                self.object_attr_value_set(parent_object_dic, 'DestinationManual', child_object_id)
                logs.add(f"{parent_object_uid} : reference to {child_object_uid} added")
            else:
                self.object_attr_value_set(parent_object_dic, 'DestinationManual', '')
                logs.add(f"{parent_object_uid} : reference to {child_object_uid} removed")
            return child_object_uid

        if parent_object_type == 'WindchestGroup' and child_object_type in ('Rank', 'Stop'):
            # set or erase the ID of the parent WindchestGroup in the attribute WindchestGroup of the child Rank or Stop object
            if operation == 'link':
                self.object_attr_value_set(child_object_dic, 'WindchestGroup', parent_object_id)
                logs.add(f"{child_object_uid} : reference to {parent_object_uid} added")
            else:
                self.object_attr_value_set(child_object_dic, 'WindchestGroup', '')
                logs.add(f"{child_object_uid} : reference to {parent_object_uid} removed")
            return child_object_uid

        if parent_object_type == 'Panel':
            # link between parent Panel and child PanelElement or PanelImage object
            # store the parent panel UID present in the child UID before to rename it

            if operation == 'link':
                prev_parent_panel_uid = child_object_uid[:8]
                if parent_object_uid != prev_parent_panel_uid:
                    # the child is moved under another parent panel
                    # get a free new UID for the child
                    new_child_object_uid = self.object_type_free_uid_get(child_object_type, parent_object_uid)

                    # move the object data of the child object under another key which has the new child UID
                    new_object_dic = self.odf_data_dic[new_child_object_uid] = self.odf_data_dic.pop(child_object_uid)

                    # write the new UID in the first line of the new object
                    (error_msg, attr_name, attr_value, comment) = self.object_line_split(new_object_dic['lines'][0])
                    if attr_name == 'uid':
                        new_object_dic['lines'][0] = self.object_line_join(attr_name, new_child_object_uid, comment)

                    logs.add(f"{child_object_uid} : renamed in {new_child_object_uid}")

                    child_object_uid = new_child_object_uid

                    # update the NumberOf attribute of the previous parent Panel object which doesn't contain anymore the child object
                    self.object_panel_numbers_update(prev_parent_panel_uid)

                # update the NumberOf attribute of the parent Panel object to take into account the added child
                self.object_panel_numbers_update(parent_object_uid)

            elif operation == 'unlink':
                # unlink a PanelElement or PanelImage to its parent panel is not possible as its UID contain necessarily a parent UID
                # the object can be only renamed, which is done by the Link operation, or deleted
                child_object_uid = None

            return child_object_uid

        if child_object_type == 'PanelElement':  # and parent_object_type != 'Panel'
            # link between a child PanelElement and a parent which is not a Panel and which has to be unique)
            # remove in the PanelElement object all the lines with Type or referencing attributes
            line_nb = 0
            child_lines_list = child_object_dic['lines']
            while line_nb < len(child_lines_list):
                # scan the lines of the child object
                (error_msg, attr_name, attr_value, comment) = self.object_line_split(child_lines_list[line_nb])
                if attr_name != None and (attr_name in self.go_objects_parents_dic['PanelElement'] or attr_name == 'Type'):
                    # the current line contains an attribute which the name is an allowed parent name : remove this line
                    del child_object_dic['lines'][line_nb]
                    line_nb -= 1
                line_nb += 1
            if operation == 'link':
                # add the type and reference to the parent object to link to the child PanelElement
                child_object_dic['lines'].insert(1, f'Type={parent_object_type}')
                child_object_dic['lines'].insert(2, f'{parent_object_type}={str(parent_object_id).zfill(3)}')
                logs.add(f"{child_object_uid} : link to {parent_object_uid} added")
            else:
                # add an empty "Type=" attribute
                child_object_dic['lines'].insert(1, 'Type=')
                logs.add(f"{child_object_uid} : link to {parent_object_uid} removed")

            return child_object_uid

        # management of the other types of objects links where the parent has attributes to refer to the children, like Switch001=025

        if operation == 'link':
            if parent_object_type == 'Manual' and child_object_type in ('Coupler', 'Divisional', 'Stop'):
                # if a parent Manual has to be linked with a child Coupler/Divisional/Stop, rename the UID of the child to have the parent Manual ID has first digit if its ID
                manual_id = int(parent_object_uid[-3:])
                if int(child_object_uid[-3]) != manual_id:
                    # the first digit of the child object ID is not equal to the manual ID
                    new_child_object_uid = self.object_type_free_uid_get(child_object_type, parent_object_uid)
                    # rename the child object UID
                    if self.object_rename(child_object_uid, new_child_object_uid) != None:
                        # the renaming has succeeded
                        child_object_uid = new_child_object_uid
                        child_object_id = self.object_id_get(child_object_uid)

            # add in the parent object the attribute which refers to the child object and update the NumberOf attribute of the parent for the child type
            logs.add(f"{parent_object_uid} : reference to {child_object_uid} added")
            self.object_children_ref_update(parent_object_uid, child_object_type, 'add', child_object_uid)

        elif operation == 'unlink':
            # remove in the parent object the attribute which refers to the child object and update the NumberOf attribute of the parent for the child type
            logs.add(f"{parent_object_uid} : reference to {child_object_uid} removed")
            self.object_children_ref_update(parent_object_uid, child_object_type, 'remove', child_object_uid)

        return child_object_uid

    #-------------------------------------------------------------------------------------------------
    def object_children_ref_update(self, object_uid, ref_object_type, operation=None, ref_object_uid=None):
        # update and sort in the given object the references to the given object type
        # if operation = "add", a reference to the given ref object UID is added to the object (must be of same ref object type)
        # if operation = "remove", the reference to the given ref object UID is removed from the object (must be of same ref object type)
        # update the NumberOfxxxx attribute value if needed
        # return True or False whether a change has been done or not in the given object

        object_type = self.object_type_get(object_uid)
        object_dic = self.object_dic_get(object_uid)

        ref_object_type_len = len(ref_object_type)
        ref_object_id = self.object_id_get(ref_object_uid)

        # define the name of the attribute in the given object which gives the number of references to the given object type
        if ref_object_type == 'Switch':
            if object_type in ('Coupler', 'DivisionalCoupler', 'Stop', 'Switch', 'Tremulant'):
                nb_of_attr_name = 'SwitchCount'
            else:
                nb_of_attr_name = 'NumberOfSwitches'
        else:
            nb_of_attr_name = 'NumberOf' + ref_object_type + 's'

        # define the attributes names which are referencing the given referenced object type
        ref_attributes_list = []
        if object_type == 'General':
            if ref_object_type in ('Coupler', 'Stop'):
                ref_attributes_list.append(ref_object_type + 'Manual999')
            ref_attributes_list.append(ref_object_type + 'Number999')
        elif object_type == 'Stop' and ref_object_type == 'Rank':
            ref_attributes_list.append(ref_object_type + '999')
            ref_attributes_list.append(ref_object_type + '999FirstPipeNumber')
            ref_attributes_list.append(ref_object_type + '999FirstAccessibleKeyNumber')
            ref_attributes_list.append(ref_object_type + '999PipeCount')
        else:
            ref_attributes_list.append(ref_object_type + '999')
            ref_attributes_list.append('')

        # store in a dictionary the references to objects of the given type which are defined in the given object
        references_dic = {}        # dictionary with as keys the ID of the referencing attributes and as value a dictionary with the attribute name and value
        nb_of_attr_line_nb = None  # line number in the given object where is located the NumberOf attribute
        nb_of_attr_value = None    # value of the NumberOf attribute
        object_lines_list = object_dic['lines'] # lines of the given object
        object_lines_list_copy = list(object_lines_list)
        line_nb = 0
        while line_nb < len(object_lines_list):
            # scan the lines of the given object
            curr_line = object_lines_list[line_nb]
            equ_pos = curr_line.find('=', ref_object_type_len+3)
            if equ_pos != -1:
                # there is an equal character in the current line
                # recover the attribute name before the equal character
                attr_name = curr_line[:equ_pos]
                # recover the ID of the referencing attribute and replace the digits by 9 in its name
                attr_name_999 = ''
                attr_name_id = ''
                for c in attr_name:
                    # scan the characters of the attribute name
                    if not c.isdigit():
                        attr_name_999 += c
                    else:
                        attr_name_999 += '9'
                        attr_name_id += c

                if attr_name_999 in ref_attributes_list:
                    # the attribute is an expected referencing attribute
                    attr_name_id = int(attr_name_id)
                    # recover the attribute value
                    attr_value = curr_line[equ_pos+1:]
                    # recover the ID which is referenced in the attribute value (removing eventual heading + - chars and leading comment)
                    if len(attr_value) > 0:
                        # if there is no value the attribute is ignored and will be deleted
                        attr_value_id = attr_value.split(';')[0].strip(' +-')
                        if len(attr_value_id) > 0 and attr_value_id.isdigit():
                            attr_value_id = int(attr_value_id)
                        else:
                            attr_value_id = 999

                        if not(operation == 'remove' and attr_value_id == ref_object_id):
                            # the current referenced ID has not to be removed
                            if not attr_name_id in references_dic.keys():
                                # create a new entry in the references dictionary with all expected attributes names
                                references_dic[attr_name_id] = {}
                                references_dic[attr_name_id]['key'] = 0
                                for attr in ref_attributes_list:
                                    references_dic[attr_name_id][attr] = None
                            # fill the entry
                            references_dic[attr_name_id][attr_name_999] = attr_value
                            # define the sorting key of the entry based on two firsts expected attributes
                            if   attr_name_999 == ref_attributes_list[0]: references_dic[attr_name_id]['key'] += attr_value_id * 1000
                            elif attr_name_999 == ref_attributes_list[1]: references_dic[attr_name_id]['key'] += attr_value_id
                    # erase the content of the line with a marker used later
                    object_lines_list[line_nb] = '#'

            if curr_line.startswith(nb_of_attr_name):
                # line containing the NumberOfxxxx or SwitchCount attribute, store the line number and the number value
                nb_of_attr_line_nb = line_nb
                nb_of_attr_value = int(curr_line[len(nb_of_attr_name + '='):])
            line_nb += 1

        if operation == 'add':
            # add in the references dictionary an entry 999 with the given reference to add
            references_dic[999] = {}
            references_dic[999]['key'] = ref_object_id * 1000
            references_dic[999][ref_attributes_list[0]] = str(ref_object_id).zfill(3)

        # place in a dictionary the correspondence between attributes sorting key and ID of the referencing attributes
        keys_dic = {}
        for ref_id, element in references_dic.items():
            keys_dic[element['key']] = ref_id
        # place in a list the referencing lines sorted by their sorting key, and for the same key sorted as defined in ref_attributes_list
        ref_attr_lines_list = []
        ref_nb = 1
        for attr_key in sorted(keys_dic.keys()):
            ref_id = keys_dic[attr_key]
            for attr_name, attr_value in references_dic[ref_id].items():
                if attr_name != 'key' and attr_value != None:
                    attr_name = attr_name.replace('999', str(ref_nb).zfill(3))
                    ref_attr_lines_list.append(f'{attr_name}={attr_value}')
            ref_nb += 1

        # add if needed the NumberOfxxxx attribute
        if nb_of_attr_line_nb == None and len(keys_dic) > 0:
            # attribute NumberOfxxxx or SwitchCount not found and there are references : add it in the last line of the object
            object_lines_list.append(nb_of_attr_name + '=0')
            nb_of_attr_line_nb = len(object_lines_list) - 1

        # update the NumberOfxxxx attribute if it exists
        if nb_of_attr_line_nb != None:
            nb_of_value = len(keys_dic)
            self.object_attr_value_set(object_dic, nb_of_attr_name, nb_of_value)
            if nb_of_attr_value not in (None, nb_of_value):
                logs.add(f"{object_uid} : attribute {nb_of_attr_name} changed from {nb_of_attr_value} to {nb_of_value}")
            elif nb_of_attr_value == None:
                logs.add(f"{object_uid} : attribute {nb_of_attr_name} set at {nb_of_value}")

        # restore in the object lines list the sorted referencing attributes
        line_nb = 0
        ref_line_nb = nb_of_attr_line_nb # line number where to write additional referencing attributes
        ref_list_idx = 0
        while line_nb < len(object_lines_list):
            # scan the lines of the given object
            if object_lines_list[line_nb] == '#':
                # it is a line which was containing a referencing attribute to sort
                if ref_list_idx < len(ref_attr_lines_list):
                    # it remains attributes to restore
                    object_lines_list[line_nb] = ref_attr_lines_list[ref_list_idx]
                    ref_list_idx += 1
                    ref_line_nb = line_nb
                else:
                    del object_lines_list[line_nb]
                    line_nb -= 1
            line_nb += 1
        while ref_list_idx < len(ref_attr_lines_list):
            # the referencing attributes have not been all copied in the object lines list (case of reference adding)
            ref_line_nb += 1
            object_lines_list.insert(ref_line_nb, ref_attr_lines_list[ref_list_idx])
            ref_list_idx += 1

        if object_lines_list_copy != object_lines_list:
            logs.add(f'{object_uid} : references to {ref_object_type} sections have been rearranged')
            return True

        return False

    #-------------------------------------------------------------------------------------------------
    def object_children_ref_all_sort(self, object_uid):
        # sort in the given object the references to all other objects

        object_type = self.object_type_get(object_uid)

        if object_type == 'Manual':
            ref_types_list = ['Coupler', 'Divisional', 'Stop', 'Switch', 'Tremulant']

        elif object_type in ['Coupler', 'Switch', 'Tremulant']:
            ref_types_list = ['Switch']

        elif object_type == 'Stop':
            ref_types_list = ['Switch', 'Rank']

        elif object_type in ['General', 'Divisional']:
            ref_types_list = ['Coupler', 'DivisionalCoupler', 'Stop', 'Switch', 'Tremulant']

        elif object_type == 'WindchestGroup':
            ref_types_list = ['Enclosure', 'Tremulant']

        else:
            logs.add(f'No references to rearrange in a section of type {object_type}')
            return False

        changes_made = False
        for ref_type in ref_types_list:
            changes_made = self.object_children_ref_update(object_uid, ref_type) or changes_made

        if not changes_made:
            logs.add(f'No change has been done in section {object_uid}')

        return changes_made

    #-------------------------------------------------------------------------------------------------
    def object_parent_panel_get(self, object_uid):
        # returns the UID of the panel (Panel999 or Organ if old panel format) to which belongs the given object UID
        # returns None if it has no parent panel

        if object_uid[:5] == 'Panel':
            if len(object_uid) == 8:
                # Panel999 : it has no parent panel
                parent_panel_uid = None
            else:
                # Panel999NNNNN999
                parent_panel_uid = object_uid[:8]
        else:
            # the object UID is not Panel999 or Panel999Element999, so it is necessarily displayed in the main panel
            if self.new_panel_format_bool:
                parent_panel_uid = 'Panel000'
            else:
                parent_panel_uid = 'Organ'

        return parent_panel_uid

    #-------------------------------------------------------------------------------------------------
    def object_parent_manual_get(self, object_uid):
        # returns the UID of the manual (Manual999) to which belongs the given object UID
        # returns None if it has no parent manuel

        for parent_uid in self.object_kinship_list_get(object_uid, TO_PARENT):
            if parent_uid[:6] == 'Manual':
                return parent_uid

        return None

    #-------------------------------------------------------------------------------------------------
    def object_kinship_list_add(self, object_uid, ref_object_uid, relationship):
        # add in the parents/children list of the given object UID the given referenced object UID
        # the given relationship must be TO_PARENT or TO_CHILD

        object_dic = self.object_dic_get(object_uid)
        if object_dic != None:
            ref_object_dic = self.object_dic_get(ref_object_uid)
            if ref_object_dic != None:
                # both the given object and referenced object are defined
                if relationship == TO_CHILD:
                    # the target object is child of the given object
                    if ref_object_uid not in object_dic['children']: object_dic['children'].append(ref_object_uid)
                    if object_uid     not in ref_object_dic['parents']:  ref_object_dic['parents'].append(object_uid)
                elif relationship == TO_PARENT:
                    # the target object is parent of the given object
                    if ref_object_uid not in object_dic['parents']:  object_dic['parents'].append(ref_object_uid)
                    if object_uid     not in ref_object_dic['children']: ref_object_dic['children'].append(object_uid)
                else:
                    logs.add('INTERNAL ERROR undefined link type given to object_kinship_list_add')
            else:
                if relationship == TO_CHILD:
                    logs.add(f'WARNING cannot link {object_uid} to child {ref_object_uid} which does not exist')
                else:
                    logs.add(f'WARNING cannot link {object_uid} to parent {ref_object_uid} which does not exist')

    #-------------------------------------------------------------------------------------------------
    def object_kinship_list_get(self, object_uid, relationship, object_type=None):
        # get the parents/children list of the given object UID
        # if an object type is given only the objects of this type are returned
        # the given relationship must be TO_PARENT or TO_CHILD

        object_dic = self.object_dic_get(object_uid)
        if object_dic != None:
            if relationship == TO_PARENT:
                kinship_list = list(object_dic['parents'])
            elif relationship == TO_CHILD:
                kinship_list = list(object_dic['children'])
            else:
                logs.add('INTERNAL ERROR undefined link type given to object_kinship_link_get')
                kinship_list = []
        else:
            kinship_list = []

        if object_type != None:
            for obj_uid in list(kinship_list):  # scan a copy of the list in which elements may be removed
                if self.object_type_get(obj_uid) != object_type:
                    kinship_list.remove(obj_uid)

        return kinship_list

    #-------------------------------------------------------------------------------------------------
    def objects_kinship_update(self):
        # update the kinship links (parent - child) between the objects of the ODF dictionary

        for object_uid, object_dic in self.odf_data_dic.items():
            # scan the objects of the ODF dictionary to reset their internal attributes
            object_dic['names'] = []
            object_dic['parents'] = []
            object_dic['children'] = []

        for object_uid, object_dic in self.odf_data_dic.items():
            # scan the objects of the ODF dictionary to define their parent/child kinship links and their name to display in the objects list/tree
            object_type = self.object_type_get(object_uid)

            for line in object_dic['lines']:
                # scan the lines of the current object to detect links toward other objects or names to display
                (error_msg, attr_name, attr_value, comment) = self.object_line_split(line)
                if attr_name != None and attr_value not in (None, ''):
                    # the attribute has a name and a value

                    # get possible names to display in the objects list/tree
                    if attr_name in ['Name', 'ChurchName', 'Comment'] or attr_name[-4:] == 'Text':
                        object_name = attr_value
                    elif object_type == 'PanelImage' and attr_name == 'Image':
                        # image attribute, the value is the path of the image : keep from the image path only the file name (string after the last separator character)
                        object_name = path2ospath(attr_value)
                        path_elements_list = object_name.split(os.path.sep)
                        object_name = path_elements_list[len(path_elements_list)-1]
                    else:
                        object_name = None
                    # add the found name in the names list of the current object
                    if object_name != None:
                        object_dic['names'].append(object_name)

                    if (attr_name[-3:].isdigit() and (attr_value.isdigit() or (len(attr_value) > 0 and attr_value[0] == '-' and attr_value[1:].isdigit())) and
                        object_type not in ('General', 'Divisional')):
                        # attribute which ends with 3 digits (like Coupler999) : it contains in its value the reference to another object
                        # link the current object to the referenced object as child
                        ref_object_type = attr_name[:-3]
                        if ref_object_type not in (None, 'MIDIKey', 'DisplayKey'):
                            ref_object_uid = ref_object_type + str(abs(int(attr_value))).zfill(3)
                            self.object_kinship_list_add(object_uid, ref_object_uid, TO_CHILD)

                    elif attr_name == 'WindchestGroup' and attr_value.isdigit():
                        # attribute WindchestGroup : it contains in its value the reference to a WindchestGroup
                        # link the current object to the referenced WindchestGroup as parent
                        ref_object_uid = attr_name + str(int(attr_value)).zfill(3)
                        self.object_kinship_list_add(object_uid, ref_object_uid, TO_PARENT)

                    elif object_type == 'PanelElement' and attr_name in self.go_objects_parents_dic['PanelElement']:
                        # Panel999Element999 object with a reference to a parent object
                        element_type = self.object_attr_value_get(object_dic, 'Type')
                        # link the current object to the parent object
                        if attr_name in ('Coupler', 'Divisional', 'Stop'):
                            # the reference is the number of the referenced object on the manual given in the Manual attribute
                            ref_object_nb = int(attr_value)
                            manual_id = myint(self.object_attr_value_get(object_dic, 'Manual'))
                            if manual_id != None:
                                manual_uid = 'Manual' + str(manual_id).zfill(3)
                                manual_children_list = sorted(self.object_kinship_list_get(manual_uid, TO_CHILD, attr_name))
                                if 0 < ref_object_nb <= len(manual_children_list):
                                    ref_object_uid = manual_children_list[ref_object_nb - 1]
                                else:
                                    ref_object_uid = attr_name  # to display in the error message in the logs
                                self.object_kinship_list_add(object_uid, ref_object_uid, TO_PARENT)
                        elif not (attr_name == 'Manual' and element_type != 'Manual'):
                            ref_object_uid = attr_name + str(int(attr_value)).zfill(3)
                            self.object_kinship_list_add(object_uid, ref_object_uid, TO_PARENT)

                    elif object_type == 'Coupler' and attr_name == 'DestinationManual':
                        # link the coupler to the destination manual as child
                        ref_object_uid = 'Manual' + str(int(attr_value)).zfill(3)
                        self.object_kinship_list_add(object_uid, ref_object_uid, TO_CHILD)

            if object_type.startswith('Panel') and len(object_type) > 5:
                # Panel999xxxx object type
                # link the current object to its Panel parent
                panel_uid = object_uid[:8]
                self.object_kinship_list_add(object_uid, panel_uid, TO_PARENT)

                # add to the object names the name of its parent panel
                panel_dic = self.object_dic_get(panel_uid)
                if panel_dic != None:
                    name = self.object_attr_value_get(panel_dic, 'Name')
                    if name not in (None, ''):
                        object_dic['names'].append(name)

                # add to the object names the UID and name of the referenced child object element if any or the type of the element
                if object_type == 'PanelElement':
                    ref_object_type = self.object_attr_value_get(object_dic, 'Type')
                    ref_object_id = self.object_attr_value_get(object_dic, ref_object_type)
                    if ref_object_id not in (None, ''):
                        # there is a referenced object ID, recover the UID and name of this object
                        ref_object_uid = ref_object_type + str(int(ref_object_id)).zfill(3)
                        ref_object_dic = self.object_dic_get(ref_object_uid)
                        ref_object_name = self.object_attr_value_get(ref_object_dic, 'Name')
                        if ref_object_name != None:
                            # add the referenced object name to the PanelElement names list
                            object_dic['names'].insert(0, ref_object_name)
                        # add the referenced object UID to the PanelElement names list
                        object_dic['names'].insert(0, ref_object_uid)
                    else:
                        object_dic['names'].insert(0, ref_object_type)

    #-------------------------------------------------------------------------------------------------
    def object_organ_numbers_update(self, object_uid):
        # update in the Organ object the NumberOf attribute corresponding to the given object

        object_type = self.object_type_get(object_uid)

        if object_type == 'Switch':
            nb_of_attr_name = 'NumberOfSwitches'
        elif object_type in ('Enclosure', 'General', 'Manual', 'Panel', 'Rank', 'Tremulant', 'WindchestGroup'):
            nb_of_attr_name = 'NumberOf' + object_type + 's'
        else:
            nb_of_attr_name = None

        if nb_of_attr_name != None:
            # recover the current NumberOf value
            curr_nb = myint(self.object_attr_value_get('Organ', nb_of_attr_name))
            # set the new NumberOf value according to the actual number of objects of the given type
            new_nb = self.objects_type_number_get(object_type)
            if ((object_type == 'Manual' and ('Manual000' in self.odf_data_dic.keys() or 'Manual999' in self.odf_data_dic.keys())) or
                (object_type == 'Panel' and ('Panel000' in self.odf_data_dic.keys() or 'Panel999' in self.odf_data_dic.keys()))):
                # Manual000 and Panel000 are not counted
                new_nb -= 1
            if self.object_attr_value_set('Organ', nb_of_attr_name, new_nb):
                if curr_nb not in (None, new_nb):
                    logs.add(f'Organ : attribute {nb_of_attr_name} changed from {curr_nb} to {new_nb}')
                elif curr_nb == None:
                    logs.add(f'Organ : attribute {nb_of_attr_name} set to {new_nb}')

    #-------------------------------------------------------------------------------------------------
    def object_panel_numbers_update(self, object_uid):
        # update in the given object (Panel, or in parent of given PanelElement / PanelImage) the NumberOfGUIElements and NumberOfImages attributes

        object_type = self.object_type_get(object_uid)
        if object_type in ('PanelElement', 'PanelImage'):
            panel_uid = object_uid[:8]
        elif object_type == 'Panel':
            panel_uid = object_uid
        else:
            return

        # recover the current NumberOfGUIElements value in the panel object
        curr_nb = myint(self.object_attr_value_get(panel_uid, 'NumberOfGUIElements'))
        # set the new NumberOfGUIElements value according to the actual number of objects in the panel
        new_nb = self.objects_type_number_get(panel_uid + 'Element')
        if self.object_attr_value_set(panel_uid, 'NumberOfGUIElements', new_nb):
            if curr_nb not in (None, new_nb):
                logs.add(f"{panel_uid} : attribute NumberOfGUIElements changed from {curr_nb} to {new_nb}")
            elif curr_nb == None:
                logs.add(f"{panel_uid} : attribute NumberOfGUIElements set at {new_nb}")

        # recover the current NumberOfImages value in the panel object
        curr_nb = myint(self.object_attr_value_get(panel_uid, 'NumberOfImages'))
        # set the new NumberOfImages value according to the actual number of objects in the panel
        new_nb = self.objects_type_number_get(panel_uid + 'Image')
        if curr_nb != None or new_nb > 0:
            # NumberOfImages is optional, set it only if it already exists or there is a not null new number to set
            if self.object_attr_value_set(panel_uid, 'NumberOfImages', new_nb):
                if curr_nb not in (None, new_nb):
                    logs.add(f"{panel_uid} : attribute NumberOfImages changed from {curr_nb} to {new_nb}")
                elif curr_nb == None:
                    # the attribute NumberOfImages is optional if null
                    logs.add(f"{panel_uid} : attribute NumberOfImages set at {new_nb}")

    #-------------------------------------------------------------------------------------------------
    def objects_number_get(self):
        # return the total number of objects defined in the ODF data

        return len(self.odf_data_dic)

    #-------------------------------------------------------------------------------------------------
    def objects_type_number_get(self, object_type):
        # return the number of objects defined in the ODF data having the given type (Manual, Enclosure, Panel999Element, Panel999Image, ...)
        # the given object type must be with internal digits when applicable (for example Panel999Element and not PanelElement)

        number = 0
        for object_uid in self.odf_data_dic.keys():
            # scan the objects of the ODF data
            if object_uid[:-3] == object_type:
                number += 1

        return number

    #-------------------------------------------------------------------------------------------------
    def objects_type_list_get(self, object_type):
        # return the list of all the objects UID of the ODF data having the given object type (Manual, Enclosure, Panel999Element, Panel999Image, ...)
        # the given object type must be with internal digits when applicable (for example Panel999Element and not PanelElement)

        objects_uid_list = []
        for object_uid in self.odf_data_dic.keys():
            # scan the objects of the ODF data
            if object_uid[:-3] == object_type:
                objects_uid_list.append(object_uid)

        return objects_uid_list

    #-------------------------------------------------------------------------------------------------
    def objects_list_get(self):
        # return the list of all the objects UID of the ODF data

        return list(self.odf_data_dic.keys())

    #-------------------------------------------------------------------------------------------------
    def object_poss_kinship_list_get(self, object_uid):
        # return two lists in a tuple : the one with the objects UID of the ODF data which can be possibly parent of the given object UID
        #                               the one with the objects UID of the ODF data which can be possibly children of the given object UID

        object_dic = self.object_dic_get(object_uid)
        if object_dic == None: return ([], [])

        object_type = self.object_type_get(object_uid)
        parents_uid_list = []
        children_uid_list = []

        if object_type in ('Header', 'Organ'):
            # no parent and no child for these object types
            return ([], [])

        # recover the lists of possible parents/children object types for the given object type
        parent_types_list = list(self.go_objects_parents_dic[object_type])
        child_types_list = list(self.go_objects_children_dic[object_type])

        # if the given object is a Stop not refering to a WindchestGroup (it has not a rank defined inside), remove the WindchestGroup parent
        if object_type == 'Stop' and self.object_attr_value_get(object_dic, 'WindchestGroup') == None:
            parent_types_list.remove('WindchestGroup')

        # recover from the ODF data the objects UID which have a type which can be parent/child of the given object
        for obj_uid in self.odf_data_dic.keys():
            # scan the objects of the ODF dictionary
            obj_type = self.object_type_get(obj_uid)
            if obj_type in parent_types_list and obj_uid != object_uid:
                # the current object has a type which can be parent of the given object
                if object_type == 'PanelElement':
                    if self.object_attr_value_get(object_uid, 'Type') == obj_type:
                        parents_uid_list.append(obj_uid)
                else:
                    parents_uid_list.append(obj_uid)
            if obj_type in child_types_list and obj_uid != object_uid:
                # the current object has a type which can be child of the given object
                children_uid_list.append(obj_uid)

        return (parents_uid_list, children_uid_list)

    #-------------------------------------------------------------------------------------------------
    def object_poss_children_type_list_get(self, object_uid):
        # return the list of the object types which can be possibly children of the given object or which doesn't need a parent (Header and Organ if not already existing)

        object_dic = self.object_dic_get(object_uid)
        object_type = self.object_type_get(object_uid)

        # define the objects types which can be created as child of the given object type
        if object_type in self.go_objects_children_dic.keys():
            child_types_list = list(self.go_objects_children_dic[object_type])
        else:
            child_types_list = []

        # if the given object is a Stop refering to a WindchestGroup (it has a rank defined inside), remove the Rank child
        if object_type == 'Stop' and self.object_attr_value_get(object_dic, 'WindchestGroup') != None:
            child_types_list.remove('Rank')

        # it is not possible to add a PanelElement or PanelImage if the parent is not a Panel, remove them from the object types list
        if object_type != 'Panel':
            if 'PanelElement' in child_types_list: child_types_list.remove('PanelElement')
            if 'PanelImage'   in child_types_list: child_types_list.remove('PanelImage')

        return child_types_list

#-------------------------------------------------------------------------------------------------
class C_ODF_HW2GO():
    # class to manage the conversion of a Hauptwerk ODF in a GrandOrgue ODF

    HW_sample_set_path = ''     # path of the folder containing the loaded Hauptwerk sample set (which contains the sub-folders OrganDefinitions and OrganInstallationPackages)
    HW_sample_set_odf_path = '' # path of the folder containing the ODF of the loaded Hauptwerk sample set (folder OrganDefinitions)
    HW_odf_file_name = ''       # path of the loaded Hauptwerk ODF (which is inside the sub-folder OrganDefinitions)

    HW_odf_dic = {}  # dictionary in which are stored the data of the loaded Hauptwerk ODF file (XML file)
                     # it has the following structure with three nested dictionaries :
                     #   {ObjectType:                      -> string, for example _General, KeyImageSet, DisplayPage
                     #       {ObjectID:                    -> integer, from 1 to 999999, recovered from the HW ODF objects ID when possible, else set by an incremented counter
                     #           {Attribute: Value, ...},  -> string: string
                     #        ...
                     #       },
                     #       ...
                     #    ...
                     #   }
                     # the ObjectUID (unique ID) is a string made by the concatenation of the ObjectType and the ObjectID on 6 digits, for example DisplayPage000006
                     # exception : the ObjectType _General has the ObjectUID _General

    GO_odf_dic = {}  # dictionary in which are stored the data of the GrandOrgue ODF built from the Hauptwerk ODF dictionary
                     # it has the following structure with two nested dictionaries :
                     #   {ObjectUID:                   -> string, for example Organ, Panel001, Rank003
                     #       {Attribute: Value, ...}   -> string: string or integer if number / dimension / code
                     #    ...
                     #   }

    HW_odf_attr_dic = {} # dictionary which contains the definition of the various HW object types and their attributes (loaded from the file HwObjectsAttributesDict.txt)
                         # it has the following structure with two nested dictionaries :
                         #   {ObjectType:                                  -> string, for example _General, KeyImageSet, DisplayPage
                         #       {AttributeLetter: AttributeFullName, ...} -> string: string
                         #    ...
                         #   }

    keys_disp_attr_dic = {}  # dictionary containing the display attributes of HW and GO keyboard keys when they are defined at octave level

    available_HW_packages_id_list = []  # list storing the ID of the installation packages which are actually accessible in the sample set package

    HW_default_display_page_dic = None # dictionary of the HW default display page (which is displayed by default on organ loading and will be the GO Panel000)
    HW_console_display_page_dic = None # dictionary of the HW console display page (which contains the displayed keyboards, can be different from the default display page)

    HW_general_dic = None  # dictionary of the HW _General object
    GO_organ_dic = None    # dictionary of the GO Organ object

    organ_base_pitch_hz = 440  # base pitch of the organ in Hz

    # value indicating how to manage the tremmed samples : 'integrated' in non tremmed ranks, 'separated' in dedicated ranks or None conversion
    trem_samples_mode = None

    max_screen_layout_id = 1  # maximum screen layout ID to convert from HW to GO

    last_manual_uid = 'Manual001'  # UID of the last build GO Manual object

    progress_status_show_function = None # address of a callback function to call to show a progression message during the ODF building

    GO_object_ext_ID = 700  # ID value used to define extended object UID, when a manual already contains 99 objects of the same type (Stop objecs)

    #-------------------------------------------------------------------------------------------------
    def reset_all_data(self):
        # reset all the data of the class, except the HW_odf_attr_dic dictionary

        self.HW_odf_dic.clear()
        self.GO_odf_dic.clear()
        self.available_HW_packages_id_list = []
        self.HW_odf_file_name = ''
        self.HW_sample_set_path = ''

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_load_from_file(self, file_name):
        # fill the Hauptwerk ODF dictionary from the data of the given Hauptwerk ODF XML file
        # return True or False whether the loading has succeeded or not

        """
        the considered Hauptwerk ODF XML syntax is :

        <Hauptwerk FileFormat="Organ" FileFormatVersion="xxxxxx">
            <ObjectList ObjectType="ObjectTypeName">
                <"ObjectTypeName">     --> not compressed format
                    <"Attribute1">Value</"Attribute1">
                    <"Attribute2">Value</"Attribute2">
                    ...
                </"ObjectTypeName">
                ...
                <o>                    --> compressed format
                    <a>Value</a>
                    <b>Value</b>
                    ...
                </o>
                ...
            </ObjectList>
               ...
        </Hauptwerk>

        the attributes letters of the compressed format are converted to attributes full name thanks to the dictionary HW_odf_attr_dic
        """

        # convert the path separators to the one of the host OS
        file_name = path2ospath(file_name)

        # check the extension of the given file name
        if os.path.splitext(file_name)[1] not in ('.Organ_Hauptwerk_xml', '.xml'):
            # the file extension is not expected
            logs.add(f'ERROR : The file "{file_name}" does not have the expected extension .xml or .Organ_Hauptwerk_xml')
            return False

        # check the existence of the given file name
        if not os.path.isfile(file_name):
            logs.add(f'ERROR : The file "{file_name}" does not exist')
            return False

        # load the dictionary HwObjectsAttributesDict if not already loaded
        if not self.HW_ODF_attr_dic_file_load():
            # error occurred while loading the dictionary
            return False

        # load the content of the HW XML file as an elements tree
        HW_ODF_xml_tree = etree.parse(file_name, etree.XMLParser(remove_comments=True))

        # check that it is actually an Hauptwerk ODF and recover the file format version
        HW_xml_id_tag = HW_ODF_xml_tree.xpath("/Hauptwerk")
        HW_file_format = HW_xml_id_tag[0].get("FileFormat")
        HW_file_format_version = HW_xml_id_tag[0].get("FileFormatVersion")
        if HW_file_format != 'Organ':
            # it is not an XML containing ODF data
            logs.add(f'ERROR : The file "{file_name}" is not a Hauptwerk organ definition file')
            return False

        object_types_nb = 0       # total number of object types found
        objects_nb = 0            # total number of objects found
        object_attributes_nb = 0  # total number of attributes found in the objects
        for xml_object_type in HW_ODF_xml_tree.xpath("/Hauptwerk/ObjectList"):
            # scan the object types defined in the XML file (in the tags <ObjectList ObjectType="xxxx">)
            object_types_nb += 1

            # recover the name of the current object type
            HW_object_type = xml_object_type.get("ObjectType")

            if HW_object_type not in self.HW_odf_attr_dic.keys():
                # the recovered HW object type is not known in the HW ODF types/attributes dictionary
                # it can be due to a problem of characters case in the XML, tries to recover the correct object name characters case from the dictionary
                for HW_obj_type in self.HW_odf_attr_dic.keys():
                    if HW_object_type.upper() == HW_obj_type.upper():
                        HW_object_type = HW_obj_type
                        break

            if HW_object_type in self.HW_odf_attr_dic.keys():
                # the current object type is defined in the HW attributes dictionary

                # create an entry in the HW dictionary for the current object type
                object_type_dic = self.HW_odf_dic[HW_object_type] = {}

                # get the dictionary defining the attributes of the current object type
                object_type_attr_dic = self.HW_odf_attr_dic[HW_object_type]

                # recover the name of the attribute of the object attribute of the current object type which defines the ID of each object, if it exists
                object_id_attr_name = object_type_attr_dic['IDattr']

                objects_in_type_nb = 0  # number of objects defined in the current object type
                                        # can be used to assign an ID to the current object if it has not an ID defined in its attributes (object_id_attr_name = '')
                for xml_object in xml_object_type:
                    # scan the objects defined in the current XML object type
                    objects_nb += 1
                    objects_in_type_nb += 1
                    object_id = None  # is defined later

                    # create a new object dictionary
                    object_dic = {}

                    # add at the beginning of the current object dictionary some custom attributes used for the GO ODF building
                    object_dic['_type']   = ''    # type of the HW object
                    object_dic['_uid'] = ''       # unique ID of the HW object (composed by its type and a number of 6 digits)
                    object_dic['_GO_uid'] = ''    # unique ID of the corresponding built GO object if any
                    object_dic['_parents'] = []   # list of the parent HW objects dictionaries
                    object_dic['_children'] = []  # list of the children HW objects dictionaries

                    for xml_object_attribute in xml_object:
                        # scan the attributes defined in the current XML object
                        object_attributes_nb += 1
                        attribute_name = xml_object_attribute.tag
                        attribute_value = xml_object_attribute.text

                        if attribute_value not in ('', None):
                            # the attributes with an empty or undefined value are ignored
                            if len(attribute_name) <= 2:
                                # the attribute name is defined by a tag of one or two characters (this is the Hauptwerk XML compressed format)
                                # recover the attribute long name corresponding to this tag
                                try:
                                    attribute_name = object_type_attr_dic[attribute_name]
                                except:
                                    # no attribute long name known
                                    attribute_name = attribute_name + '???'

                            # add the current attribute name and value to the current object
                            object_dic[attribute_name] = attribute_value

                            if object_id == None and attribute_name == object_id_attr_name:
                                # the current attribute is the attribute which contains the ID of the object
                                if not attribute_value.isnumeric():
                                    logs.add(f'ERROR : attribute {attribute_name}={attribute_value} has not a numeric value in the object {HW_object_type} #{objects_in_type_nb}')
                                else:
                                    object_id = int(attribute_value)

                    if object_id == None:
                        # an object ID has not been recovered from the current object attributes
                        if object_id_attr_name != '':
                            # the object should have had an defined ID attribute
                            logs.add(f'ERROR : attribute {object_id_attr_name} not found in the object {HW_object_type} #{objects_in_type_nb}')
                        # use as object ID the objects counter
                        object_id = objects_in_type_nb

                    # store in the object its UID (unique ID composed by the object type followed by the object ID in 6 digits)
                    if HW_object_type == '_General':
                        object_dic['_type'] = '_General'
                        object_dic['_uid'] = '_General'
                    else:
                        object_dic['_type'] = HW_object_type
                        object_dic['_uid'] = HW_object_type + str(object_id).zfill(6)

                    # add the object dictionary to the current object type dictionary
                    object_type_dic[object_id] = object_dic

            else:
                logs.add(f'INTERNAL ERROR : object type {HW_object_type} unknown in the HW attributes dictionary')

        logs.add(f'Hauptwerk ODF loaded "{file_name}"')
        logs.add(f'Hauptwerk organ file format version {HW_file_format_version}')
        logs.add(f'{object_attributes_nb:,} attributes among {objects_nb:,} sections among {object_types_nb} section types')

        self.HW_odf_file_name = path2ospath(file_name)
        self.HW_sample_set_path = path2ospath(os.path.dirname(os.path.dirname(file_name)))
        self.HW_sample_set_odf_path = self.HW_sample_set_path + os.path.sep + 'OrganDefinitions'

        self.HW_odf_dic['_General'][1]['_sample_set_path'] = self.HW_sample_set_path

        return True

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_attr_dic_file_load(self):
        # load the Hauptwerk attributes dictionary from the file HwObjectsAttributesDict.txt (if it is present and there is no error)
        # return True or False whether the operation has succeeded or not

        if len(self.HW_odf_attr_dic) == 0:
            # the dictionary has not been loaded yet

            file_name = os.path.dirname(__file__) + os.path.sep + 'resources' + os.path.sep + 'HwObjectsAttributesDict.txt'

            try:
                with open(file_name, 'r') as f:
                    self.HW_odf_attr_dic = eval(f.read())
                    return True
            except OSError as err:
                # it has not be possible to open the file
                logs.add(f'ERROR Cannot open the file "{file_name}" : {err}')
            except SyntaxError as err:
                # syntax error in the dictionary structure which is in the file
                logs.add(f'ERROR Syntax error in the file "{file_name}" : {err}')
            except:
                # other error
                logs.add(f'ERROR while opening the file "{file_name}"')

            return False

        return True

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_do_links_between_objects(self):
        # set in the Hauptwerk ODF dictionary the relationships (parent, children) between the various objects
        # add in the objects of the HW_odf_dic the attributes "_parents" and "_children" with as value the list of the respective parent or child objects

        self.HW_general_dic = self.HW_ODF_get_object_dic_from_uid('_General')
        self.HW_ODF_do_link_between_obj_by_id(self.HW_general_dic, 'SpecialObjects_DefaultDisplayPageID', 'DisplayPage', TO_CHILD)
        self.HW_ODF_do_link_between_obj_by_id(self.HW_general_dic, 'SpecialObjects_MasterCaptureSwitchID', 'Switch', TO_CHILD)
        self.HW_general_dic['Name'] = self.HW_general_dic['Identification_Name']

        HW_object_type = 'RequiredInstallationPackage'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj(HW_object_dic, self.HW_general_dic, TO_PARENT)

        HW_object_type = 'DivisionInput'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'DivisionID', 'Division', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SwitchID', 'Switch', TO_PARENT)

        HW_object_type = 'Keyboard'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'KeyGen_DisplayPageID', 'DisplayPage', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'Hint_PrimaryAssociatedDivisionID', 'Division', TO_CHILD)
                for layout_id in range(0, 4):
                    # scan the screen layouts to make link with defined ImageSets
                    if layout_id == 0:
                        attr_name = 'KeyGen_KeyImageSetID'
                    else:
                        attr_name = f'KeyGen_AlternateScreenLayout{layout_id}_KeyImageSetID'
                    self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, attr_name, 'KeyImageSet',  TO_CHILD)

        HW_object_type = 'KeyAction'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SourceKeyboardID', 'Keyboard', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'DestKeyboardID', 'Keyboard', TO_CHILD)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'DestDivisionID', 'Division', TO_CHILD)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ConditionSwitchID', 'Switch', TO_PARENT)

        HW_object_type = 'KeyboardKey'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'KeyboardID', 'Keyboard', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SwitchID', 'Switch', TO_PARENT)

        HW_object_type = 'KeyImageSet'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                for obj_attr_name in list(HW_object_dic.keys()):
                    if obj_attr_name.startswith('KeyShapeImageSetID'):
                        self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, obj_attr_name, 'ImageSet', TO_CHILD)

        HW_object_type = 'ImageSetElement'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ImageSetID', 'ImageSet', TO_PARENT)

        HW_object_type = 'TextInstance'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'DisplayPageID', 'DisplayPage', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'TextStyleID', 'TextStyle', TO_CHILD)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'AttachedToImageSetInstanceID', 'ImageSetInstance', TO_CHILD)

        HW_object_type = 'Switch'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'Disp_ImageSetInstanceID', 'ImageSetInstance', TO_CHILD)
                # if the Switch is linked to an ImageSetInstance object, link it to the DisplayPage in which it is displayed
                HW_image_set_inst_dic = self.HW_ODF_get_object_dic_by_ref_id('ImageSetInstance', HW_object_dic, 'Disp_ImageSetInstanceID')
                if HW_image_set_inst_dic != None:
                    HW_display_page_dic = self.HW_ODF_get_object_dic_by_ref_id('DisplayPage', HW_image_set_inst_dic, 'DisplayPageID')
                    self.HW_ODF_do_link_between_obj(HW_object_dic, HW_display_page_dic, TO_PARENT)

                switch_asgn_code = myint(self.HW_ODF_get_attribute_value(HW_object_dic, 'DefaultInputOutputSwitchAsgnCode'), 0)
                if switch_asgn_code in range(12, 900):
                    # the current Switch is controlling a setter
                    # look for the Combination object having the same code in CombinationTypeCode to link it to this Switch as child
                    for HW_comb_dic in self.HW_odf_dic['Combination'].values():
                        comb_type_code = myint(self.HW_ODF_get_attribute_value(HW_comb_dic, 'CombinationTypeCode'), 0)
                        if comb_type_code == switch_asgn_code:
                            self.HW_ODF_do_link_between_obj(HW_object_dic, HW_comb_dic, TO_CHILD)

        HW_object_type = 'SwitchLinkage'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SourceSwitchID', 'Switch', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'DestSwitchID', 'Switch', TO_CHILD)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ConditionSwitchID', 'Switch', TO_PARENT)
                if DEV_MODE:
                    # only in development mode to speed up the links creation in application mode, this parent/child association is not used to convert the HW to GO ODF
                    # make direct link between source and destination switches
                    HW_source_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_object_dic, 'SourceSwitchID')
                    HW_dest_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_object_dic, 'DestSwitchID')
                    HW_cond_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_object_dic, 'ConditionSwitchID')
                    if HW_source_switch_dic != None and HW_dest_switch_dic != None :
                        self.HW_ODF_do_link_between_obj(HW_source_switch_dic, HW_dest_switch_dic, TO_CHILD)
                        if HW_cond_switch_dic != None:
                            self.HW_ODF_do_link_between_obj(HW_cond_switch_dic, HW_dest_switch_dic, TO_CHILD)

        HW_object_type = 'SwitchExclusiveSelectGroupElement'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SwitchID', 'Switch', TO_CHILD)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'GroupID', 'SwitchExclusiveSelectGroup', TO_PARENT)

        HW_object_type = 'WindCompartment'
        if DEV_MODE and HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'PressureOutputContinuousControlID', 'ContinuousControl', TO_PARENT)

        HW_object_type = 'WindCompartmentLinkage'
        if DEV_MODE and HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'FirstWindCompartmentID', 'WindCompartment', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SecondWindCompartmentID', 'WindCompartment', TO_CHILD)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ValveControllingContinuousControlID', 'ContinuousControl', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ValveControllingSwitchID', 'Switch', TO_PARENT)
                if DEV_MODE:
                    # make direct link between source and destination wind compartments
                    HW_first_wind_comp_dic = self.HW_ODF_get_object_dic_by_ref_id('WindCompartment', HW_object_dic, 'FirstWindCompartmentID')
                    HW_second_wind_comp_dic = self.HW_ODF_get_object_dic_by_ref_id('WindCompartment', HW_object_dic, 'SecondWindCompartmentID')
                    if HW_first_wind_comp_dic != None and HW_second_wind_comp_dic != None :
                        self.HW_ODF_do_link_between_obj(HW_first_wind_comp_dic, HW_second_wind_comp_dic, TO_CHILD)

        HW_object_type = 'Stop'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'DivisionID', 'Division', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ControllingSwitchID', 'Switch', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'Hint_PrimaryAssociatedRankID', 'Rank', TO_CHILD)

        HW_object_type = 'StopRank'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'StopID', 'Stop', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'RankID', 'Rank', TO_CHILD)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SwitchIDToSwitchToAlternateRank', 'Switch', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'AlternateRankID', 'Rank', TO_CHILD)

        HW_object_type = 'Combination'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ActivatingSwitchID', 'Switch', TO_PARENT)
                if self.HW_ODF_get_attribute_value(HW_object_dic, 'CombinationTypeCode') == '1':
                    # master capture combination, link to it the master capture switch defined in the _General object
                    HW_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', self.HW_general_dic, 'SpecialObjects_MasterCaptureSwitchID')
                    self.HW_ODF_do_link_between_obj(HW_object_dic, HW_switch_dic, TO_PARENT)

        HW_object_type = 'CombinationElement'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'CombinationID', 'Combination', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ControlledSwitchID', 'Switch', TO_CHILD)

        HW_object_type = 'Pipe_SoundEngine01'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'RankID', 'Rank', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'WindSupply_SourceWindCompartmentID', 'WindCompartment', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ControllingPalletSwitchID', 'Switch', TO_PARENT)
                if DEV_MODE:
                    # only in development mode to speed up the links creation in application mode, these parent/child association are not used to convert the HW to GO ODF
                    self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'WindSupply_OutputWindCompartmentID', 'WindCompartment', TO_CHILD)

        HW_object_type = 'Pipe_SoundEngine01_Layer'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'PipeID', 'Pipe_SoundEngine01', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'AmpLvl_ScalingContinuousControlID', 'ContinuousControl', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'Main_AttackSelCriteria_ContinuousControlID', 'ContinuousControl', TO_PARENT)
                if DEV_MODE:
                    # only in development mode to speed up the links creation in application mode, these parent/child association are not used to convert the HW to GO ODF
                    self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'Main_ReleaseSelCriteria_ContinuousControlID', 'ContinuousControl', TO_PARENT)
                    self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'PitchLvl_ScalingContinuousControlID', 'ContinuousControl', TO_PARENT)
                    self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'PitchLvl_IncrementingContinuousControlID', 'ContinuousControl', TO_PARENT)
                    self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'HarmonicShaping_IncrementingContinuousControlID', 'ContinuousControl', TO_PARENT)
                    self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'PhaseAngleOutputContinuousControlID', 'ContinuousControl', TO_PARENT)

        HW_object_type = 'Pipe_SoundEngine01_AttackSample'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'LayerID', 'Pipe_SoundEngine01_Layer', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SampleID', 'Sample', TO_CHILD)

        HW_object_type = 'Pipe_SoundEngine01_ReleaseSample'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'LayerID', 'Pipe_SoundEngine01_Layer', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SampleID', 'Sample', TO_CHILD)

        HW_object_type = 'ContinuousControlStageSwitch'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ContinuousControlID', 'ContinuousControl', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ControlledSwitchID', 'Switch', TO_CHILD)

        HW_object_type = 'ContinuousControlLinkage'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SourceControlID', 'ContinuousControl', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'DestControlID', 'ContinuousControl', TO_CHILD)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ConditionSwitchID', 'Switch', TO_PARENT)

        HW_object_type = 'ContinuousControl'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ImageSetInstanceID', 'ImageSetInstance', TO_CHILD)
                # if the ContinuousControl is linked to an ImageSetInstance object, link it to the DisplayPage in which it is displayed
                HW_image_set_inst_dic = self.HW_ODF_get_object_dic_by_ref_id('ImageSetInstance', HW_object_dic, 'ImageSetInstanceID')
                if HW_image_set_inst_dic != None:
                    HW_display_page_dic = self.HW_ODF_get_object_dic_by_ref_id('DisplayPage', HW_image_set_inst_dic, 'DisplayPageID')
                    self.HW_ODF_do_link_between_obj(HW_object_dic, HW_display_page_dic, TO_PARENT)

        HW_object_type = 'ContinuousControlImageSetStage'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ImageSetID', 'ImageSet', TO_PARENT)

        HW_object_type = 'ContinuousControlDoubleLinkage'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'FirstSourceControl_ID', 'ContinuousControl', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'SecondSourceControl_ID', 'ContinuousControl', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'DestControl_ID', 'ContinuousControl', TO_CHILD)

        HW_object_type = 'Enclosure'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ShutterPositionContinuousControlID', 'ContinuousControl', TO_PARENT)

        HW_object_type = 'EnclosurePipe'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'EnclosureID', 'Enclosure', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'PipeID', 'Pipe_SoundEngine01', TO_CHILD)

        HW_object_type = 'TremulantWaveformPipe'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'PipeID', 'Pipe_SoundEngine01', TO_CHILD)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'TremulantWaveformID', 'TremulantWaveform', TO_PARENT)

        HW_object_type = 'Tremulant'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'ControllingSwitchID', 'Switch', TO_PARENT)

        HW_object_type = 'TremulantWaveform'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'TremulantID', 'Tremulant', TO_PARENT)
                self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'PitchAndFundamentalWaveformSampleID', 'Sample', TO_CHILD)

        HW_object_type = 'ImageSetInstance'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                if len(HW_object_dic['_parents']) == 0:
                    # this ImageSetInstance object has none parent, link it with its DisplayPage
                    self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, 'DisplayPageID', 'DisplayPage', TO_PARENT)
                for layout_id in range(0, 4):
                    # scan the screen layouts to make link with a possibly defined ImageSet
                    if layout_id == 0:
                        attr_name = 'ImageSetID'
                    else:
                        attr_name = f'AlternateScreenLayout{layout_id}_ImageSetID'
                    self.HW_ODF_do_link_between_obj_by_id(HW_object_dic, attr_name, 'ImageSet', TO_CHILD)

        # link to _General all the Division objects
        HW_object_type = 'Division'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj(HW_object_dic, self.HW_general_dic, TO_PARENT)

        # link to _General the Keyboard objects which the attribute DefaultInputOutputKeyboardAsgnCode is defined (it is the visible position of the keyboard on the console)
        HW_object_type = 'Keyboard'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                if myint(self.HW_ODF_get_attribute_value(HW_object_dic, 'DefaultInputOutputKeyboardAsgnCode'), 0) > 0:
                    self.HW_ODF_do_link_between_obj(HW_object_dic, self.HW_general_dic, TO_PARENT)

        # link to _General the DisplayPage objects (which are not already linked by SpecialObjects_DefaultDisplayPageID)
        HW_object_type = 'DisplayPage'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                if HW_object_dic not in self.HW_general_dic['_children']:
                    self.HW_ODF_do_link_between_obj(HW_object_dic, self.HW_general_dic, TO_PARENT)

        # link to _General all the Tremulant objects (to find them more easily in the objects tree)
        HW_object_type = 'Tremulant'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                self.HW_ODF_do_link_between_obj(HW_object_dic, self.HW_general_dic, TO_PARENT)

        # link to _General all the WindCompartment objects which have no parent
        HW_object_type = 'WindCompartment'
        if HW_object_type in self.HW_odf_dic.keys():
            for HW_object_dic in self.HW_odf_dic[HW_object_type].values():
                if len(HW_object_dic['_parents']) == 0:
                    self.HW_ODF_do_link_between_obj(HW_object_dic, self.HW_general_dic, TO_PARENT)

        return True

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_do_link_between_obj_by_id(self, HW_object_dic, HW_attr_id_name_str, linked_object_type_str, link_type):
        # do a link between the given HW object dict and the given linked HW object type dict based on an ID
        # the given link_type must be TO_PARENT or TO_CHILD

        # recover the value of the ID permitting to establish a linkage between the two objects
        linkage_id_value_int = myint(self.HW_ODF_get_attribute_value(HW_object_dic, HW_attr_id_name_str), 0)

        if linkage_id_value_int != 0:
            try:
                linked_object_dic = self.HW_odf_dic[linked_object_type_str][linkage_id_value_int]
            except:
                logs.add(f'INTERNAL ERROR : {HW_object_dic["_uid"]} - not found reference to object type {linked_object_type_str} with ID {linkage_id_value_int}')
                return False

            return self.HW_ODF_do_link_between_obj(HW_object_dic, linked_object_dic, link_type)

        return False

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_do_link_between_obj(self, HW_object_dic, linked_HW_object_dic, link_type):
        # do a link between the given HW object dict and the given linked HW object dict
        # the given link_type must be TO_PARENT or TO_CHILD

        if link_type == TO_CHILD:
            self.HW_ODF_add_attribute_value(HW_object_dic, '_children', linked_HW_object_dic)
            self.HW_ODF_add_attribute_value(linked_HW_object_dic, '_parents', HW_object_dic)
        elif link_type == TO_PARENT:
            self.HW_ODF_add_attribute_value(HW_object_dic, '_parents', linked_HW_object_dic)
            self.HW_ODF_add_attribute_value(linked_HW_object_dic, '_children', HW_object_dic)
        else:
            logs.add('INTERNAL ERROR : undefined link type given to HW_ODF_do_link_between_obj')
            return False

        return True

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_add_attribute_value(self, HW_object_dic, attr_name, attr_value):
        # add the given attribute value to the list of the given attribute name in the given HW object dictionary (for _xxx attributes which contain a list)
        # if the given value already exists in the list, it is not added to avoid doubles

        try:
            HW_object_dic[attr_name].append(attr_value)
        except:
            # the attr_name doesn't exist, create it and add the value
            HW_object_dic[attr_name] = []
            HW_object_dic[attr_name].append(attr_value)

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_attribute_value(self, HW_object_dic, attr_name, mandatory_bool=False):
        # return the string value of the given attribute defined in the given object sub-dictionary of the Hauptwerk ODF dictionary
        # generate a log in case of attribute not found and if mandatory_bool=MANDATORY (True), mandatory_bool=False permits to get silently an attribute which the presence is optional
        # return None if the attribute name is not defined in the given dictionary

        if HW_object_dic == None:
            return None

        try:
            attr_value = HW_object_dic[attr_name]
        except:
            attr_value = None
            if mandatory_bool:
                logs.add(f'ERROR : unable to read the attribute "{attr_name}" in the sample set object {HW_object_dic["_uid"]}')

        return attr_value

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_object_dic_from_id(self, HW_object_type, HW_object_id):
        # return the HW object dictionary having the given object type and ID
        # return None if the object has not been found with the given data

        try:
            # recover the dictionary of the object having the given type and ID
            return self.HW_odf_dic[HW_object_type][HW_object_id]
        except:
            # object dictionary not existing for the given type and/or ID
            return None

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_object_dic_from_uid(self, HW_object_uid):
        # return the HW object dictionary having the given object UID (unique ID)
        # return None if there is none object having the given UID

        if HW_object_uid == None:
            return None

        if HW_object_uid == '_General':
            return self.HW_ODF_get_object_dic_from_id('_General', 1)

        # get the first 6 digits of the UID to get the object type
        # get the last  6 digits of the UID to get the object ID
        return self.HW_ODF_get_object_dic_from_id(HW_object_uid[:-6], int(HW_object_uid[-6:]))

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_object_dic_by_ref_id(self, HW_object_type, ref_HW_object_dic, ref_HW_attr_id_name):
        # return the HW object dictionary having the given object type and which the ID is referenced in the given object dictionary and attribute name

        # get the ID of the referenced object
        HW_object_id = myint(self.HW_ODF_get_attribute_value(ref_HW_object_dic, ref_HW_attr_id_name))

        if HW_object_id != None:
            return self.HW_ODF_get_object_dic_from_id(HW_object_type, HW_object_id)

        return None

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_linked_objects_dic_by_type(self, HW_object_dic, object_type, link_type=TO_CHILD, first_occurence=False, sorted_by=None):
        # return a list containing the dictionary of the HW objects which are parent/child (according to link_type) of the given object and which have the given object type
        # link_type must be equal to TO_PARENT or TO_CHILD if object_type is not 'root'
        # if first_occurence = FIRST_ONE (True), only the first occurence of the linked object is returned as a dictionary, not as a list, and the sorting parameter is ignored
        # if sorted_by = 'ID', the returned objects list is sorted by object ID order
        # if sorted_by = another string, it must be an attribute name of the given object type and the returned list is sorted according to this attribute
        # if HW_object_dic = 'root', return all the objects of the given object type on which a sorting criteria can be applied
        # return an empty list or None (if first_occurence=True) if there is no parent/child found

        HW_linked_objects_dic_list = []

        if HW_object_dic == 'root':
            # recover in a list the dictionaries of all the objects of the given type
            HW_linked_objects_dic_list = list(self.HW_odf_dic[object_type].values())

        elif HW_object_dic != None:
            if link_type == TO_PARENT:
                HW_kinship_objects_dic_list = HW_object_dic['_parents']
            else:
                HW_kinship_objects_dic_list = HW_object_dic['_children']

            for HW_obj_dic in HW_kinship_objects_dic_list:
                # scan the list of linked objects (parents or children) to recover the one having the given type
                if HW_obj_dic['_type'] == object_type:
                    # the current object has the expected type
                    HW_linked_objects_dic_list.append(HW_obj_dic)
                    if first_occurence:
                        # stop at the first occurrence
                        break

        if len(HW_linked_objects_dic_list) > 1 and sorted_by != None:
            # the built list has more than 1 element and it has to be sorted
            obj_id_list = []    # list permitting to sort the objects by their ID order
            attr_id_list = []   # list permitting to sort the objects by one of their attribute value order
            for HW_obj_dic in HW_linked_objects_dic_list:
                # scan the list of linked object dictionaries to build one list with the objects ID and one list with the attribute name + object ID
                if sorted_by == 'ID':
                    obj_id_list.append(int(HW_obj_dic['_uid'][-6:]))
                elif sorted_by in HW_obj_dic.keys():
                    # the list has to be sorted by an other attribute than the ID and this attribute is actually defined in the object
                    attr_id_list.append(HW_obj_dic[sorted_by] + '|' + HW_obj_dic['_uid'][-6:])

            HW_linked_objects_dic_list.clear()
            if sorted_by == 'ID':
                # rebuild the linked objects list by their ID order
                for obj_id in sorted(obj_id_list):
                    HW_linked_objects_dic_list.append(self.HW_ODF_get_object_dic_from_id(object_type, obj_id))
            else:
                # rebuild the linked objects list by the given attribute value order
                for attr_id_key in sorted(attr_id_list):
                    attr_value, obj_id = attr_id_key.split('|')
                    HW_linked_objects_dic_list.append(self.HW_ODF_get_object_dic_from_id(object_type, int(obj_id)))

        if first_occurence:
            if len(HW_linked_objects_dic_list) > 0:
                return HW_linked_objects_dic_list[0]

            return None

        return HW_linked_objects_dic_list

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_object_attr_list(self, HW_object_uid):
        # return a list containing the object attributes name/value of the given HW object UID (for display purpose in the GUI)
        # or None if the given HW object doesn't exist

        data_list = []

        HW_object_dic = self.HW_ODF_get_object_dic_from_uid(HW_object_uid)

        if HW_object_dic != None:
            for obj_attr_name, obj_attr_value in HW_object_dic.items():
                if obj_attr_name in ('_parents', '_children'):
                    # this attribute value contains a list of parents/children HW objects dictionaries
                    obj_attr_value = sorted(self.HW_DIC2UID(obj_attr_value))
                    if len(obj_attr_value) > 50:
                        obj_attr_value = obj_attr_value[:50]
                        obj_attr_value.append(' ...')

                elif obj_attr_name == '_GO_windchests_uid_list':
                    # this attribute value contains a list of GO objects UID strings
                    obj_attr_value = sorted(obj_attr_value)

                elif isinstance(obj_attr_value, dict):
                    # this attribute value contains the dictionary of a HW object
                    obj_attr_value = obj_attr_value['_uid']

                data_list.append(f'{obj_attr_name}={obj_attr_value}')

        return data_list

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_image_attributes(self, HW_object_dic, HW_image_attr_dic, HW_image_index_in_set = None, layout_id=0):
        # fill the given HW_image_attr_dic dictionary with the following HW attributes recovered from
        # the given object (can be ImageSetInstance or ImageSet) and its linked ImageSet / ImageSetElement objects
        # and defined for the given layout ID if an ImageSetInstance is given
        #    Name (string)
        #    LeftXPosPixels (integer, default 0)
        #    TopYPosPixels (integer, default 0)
        #    ImageWidthPixels (integer, default None)
        #    ImageHeightPixels (integer, default None)
        #    ImageWidthPixelsTiling (integer, default None)
        #    ImageHeightPixelsTiling (integer, default None)
        #    ClickableAreaLeftRelativeXPosPixels (integer, default None)
        #    ClickableAreaRightRelativeXPosPixels (integer, default None)
        #    ClickableAreaTopRelativeYPosPixels (integer, default None)
        #    ClickableAreaBottomRelativeYPosPixels (integer, default None)
        #    InstallationPackageID (integer)
        #    BitmapFilename (string, default None)
        #    TransparencyMaskBitmapFilename (string, default None)
        # if HW_image_index_in_set = None, use the ImageSetInstance attribute DefaultImageIndexWithinSet if available, else use the index 1 by default
        # return True or False whether the operation has succeeded or not

        if HW_object_dic['_type'] == 'ImageSetInstance':
            # ImageSetInstance object provided

            HW_image_set_inst_dic = HW_object_dic

            # recover the ImageSet object associated to the given ImageSetInstance object
            if layout_id == 0:
                HW_image_set_dic = self.HW_ODF_get_object_dic_by_ref_id('ImageSet', HW_image_set_inst_dic, 'ImageSetID')
            else:
                HW_image_set_dic = self.HW_ODF_get_object_dic_by_ref_id('ImageSet', HW_image_set_inst_dic, f'AlternateScreenLayout{layout_id}_ImageSetID')
            if HW_image_set_dic == None:
                return False

            HW_image_attr_dic['Name'] = self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'Name')

            if layout_id == 0:
                HW_image_attr_dic['LeftXPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'LeftXPosPixels'), 0)
                HW_image_attr_dic['TopYPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'TopYPosPixels'), 0)
            else:
                HW_image_attr_dic['LeftXPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, f'AlternateScreenLayout{layout_id}_LeftXPosPixels'), 0)
                HW_image_attr_dic['TopYPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, f'AlternateScreenLayout{layout_id}_TopYPosPixels'), 0)

            if layout_id == 0:
                HW_image_attr_dic['ImageWidthPixelsTiling'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'RightXPosPixelsIfTiling'))
            else:
                HW_image_attr_dic['ImageWidthPixelsTiling'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'AlternateScreenLayout{layout_id}_RightXPosPixelsIfTiling'))
            if HW_image_attr_dic['ImageWidthPixelsTiling'] == 0:  # some sample sets define 0 to mean None
                HW_image_attr_dic['ImageWidthPixelsTiling'] = None

            if layout_id == 0:
                HW_image_attr_dic['ImageHeightPixelsTiling'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'BottomYPosPixelsIfTiling'))
            else:
                HW_image_attr_dic['ImageHeightPixelsTiling'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'AlternateScreenLayout{layout_id}_BottomYPosPixelsIfTiling'))
            if HW_image_attr_dic['ImageHeightPixelsTiling'] == 0:  # some sample sets define 0 to mean None
                HW_image_attr_dic['ImageHeightPixelsTiling'] = None

        elif HW_object_dic['_type'] == 'ImageSet':
            # ImageSet object provided

            HW_image_set_inst_dic = None
            HW_image_set_dic = HW_object_dic

            HW_image_attr_dic['Name'] = self.HW_ODF_get_attribute_value(HW_image_set_dic, 'Name')

            # set the default values of the ImageSetInstance attributes
            HW_image_attr_dic['LeftXPosPixels'] = 0
            HW_image_attr_dic['TopYPosPixels'] = 0
            HW_image_attr_dic['ImageWidthPixelsTiling'] = None
            HW_image_attr_dic['ImageHeightPixelsTiling'] = None

        else:
            return False

        if HW_image_index_in_set == None:
            # image index not provided in parameter of the function : use the default index from the ImageSetInstance object if known, else use index 1
            HW_image_index_in_set = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'DefaultImageIndexWithinSet'), 1)

        # recover the image dimensions
        if HW_image_attr_dic['ImageWidthPixelsTiling'] != None:
            HW_image_attr_dic['ImageWidthPixels'] = myint(HW_image_attr_dic['ImageWidthPixelsTiling'])
        else:
            HW_image_attr_dic['ImageWidthPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_dic, 'ImageWidthPixels'))

        if HW_image_attr_dic['ImageHeightPixelsTiling'] != None:
            HW_image_attr_dic['ImageHeightPixels'] = myint(HW_image_attr_dic['ImageHeightPixelsTiling'])
        else:
            HW_image_attr_dic['ImageHeightPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_dic, 'ImageHeightPixels'))

        # recover the clickable area dimensions
        HW_image_attr_dic['ClickableAreaLeftRelativeXPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_dic, 'ClickableAreaLeftRelativeXPosPixels'))
        HW_image_attr_dic['ClickableAreaRightRelativeXPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_dic, 'ClickableAreaRightRelativeXPosPixels'))
        HW_image_attr_dic['ClickableAreaTopRelativeYPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_dic, 'ClickableAreaTopRelativeYPosPixels'))
        HW_image_attr_dic['ClickableAreaBottomRelativeYPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_dic, 'ClickableAreaBottomRelativeYPosPixels'))

        # correct the clickable area width if greater than the image width
        if (HW_image_attr_dic['ImageWidthPixels'] != None and HW_image_attr_dic['ClickableAreaRightRelativeXPosPixels'] != None and
            HW_image_attr_dic['ClickableAreaRightRelativeXPosPixels'] > HW_image_attr_dic['ImageWidthPixels'] - 1):
            HW_image_attr_dic['ClickableAreaRightRelativeXPosPixels'] = HW_image_attr_dic['ImageWidthPixels'] - 1
        # correct the clickable area height if greater than the image height
        if (HW_image_attr_dic['ImageHeightPixels'] != None and HW_image_attr_dic['ClickableAreaBottomRelativeYPosPixels'] != None and
            HW_image_attr_dic['ClickableAreaBottomRelativeYPosPixels'] > HW_image_attr_dic['ImageHeightPixels'] - 1):
            HW_image_attr_dic['ClickableAreaBottomRelativeYPosPixels'] = HW_image_attr_dic['ImageHeightPixels'] - 1

        # recover the image installation package ID
        HW_image_attr_dic['InstallationPackageID'] = myint(self.HW_ODF_get_attribute_value(HW_image_set_dic, 'InstallationPackageID', MANDATORY))

        # recover the bitmap file of the transparency image if any
        file_name = self.HW_ODF_get_attribute_value(HW_image_set_dic, 'TransparencyMaskBitmapFilename')
        if file_name != None:
            HW_image_attr_dic['TransparencyMaskBitmapFilename'] = self.convert_HW2GO_file_name(file_name, HW_image_attr_dic['InstallationPackageID'])
        else:
            HW_image_attr_dic['TransparencyMaskBitmapFilename'] = None

        # recover the bitmap file corresponding to the given or default image index
        HW_image_attr_dic['BitmapFilename'] = None
        for image_set_elem_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_image_set_dic, 'ImageSetElement', TO_CHILD):
            # scan the ImageSetElement objects which are children of the ImageSet object to find the one having the given or default image index
            image_index = myint(self.HW_ODF_get_attribute_value(image_set_elem_dic, 'ImageIndexWithinSet'), 1)
            if image_index == HW_image_index_in_set:
                # it is the expected index of ImageSetElement object
                file_name = self.HW_ODF_get_attribute_value(image_set_elem_dic, 'BitmapFilename')
                if file_name != None:
                    HW_image_attr_dic['BitmapFilename'] = self.convert_HW2GO_file_name(file_name, HW_image_attr_dic['InstallationPackageID'])
                else:
                    HW_image_attr_dic['BitmapFilename'] = None
                break

        return True

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_text_attributes(self, HW_text_inst_dic, HW_text_attr_dic):
        # fill the given HW_text_attr_dic dictionary with the following HW attributes recovered from
        # the given TextInstance object and its linked TextStyle object, and from the linked ImageSetInstance object if any
        # the not defined attributes are set at None
        #    Text (string, default ?)
        #    XPosPixels (integer, default 0)
        #    YPosPixels (integer, default 0)
        #    PosRelativeToTopLeftOfImage : Y or N (string, default N)
        #    WordWrapWithinABoundingBox : Y or N (string, default Y)
        #    BoundingBoxWidthPixelsIfWordWrap (integer, default 0)
        #    BoundingBoxHeightPixelsIfWordWrap (integer, default 0)
        #    Face_WindowsName (string, default Arial)
        #    Font_SizePixels (integer, default 10)
        #    Font_WeightCode : 1 = light, 2 = normal, 3 = bold (integer, default 2)
        #    Colour_Red (integer, default 0)
        #    Colour_Green (integer, default 0)
        #    Colour_Blue (integer, default 0)
        #    HorizontalAlignmentCode : 0 or 3 = center, 1 = left, 2 = right  (integer, default 0)
        #    VerticalAlignmentCode   : 0 = center, 1 = top,  2 = bottom (integer, default 1)
        #    ImageSetInstanceDic : dictionary of the linked ImageSetInstance object if any, else None
        #    + the attributes returned by HW_ODF_get_image_attributes if an ImageSetInstance object is linked

        # recover the data from the given TextInstance object
        HW_text_attr_dic['Text'] = mystr(self.HW_ODF_get_attribute_value(HW_text_inst_dic, 'Text'), '?')
        HW_text_attr_dic['XPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_text_inst_dic, 'XPosPixels'), 0)
        HW_text_attr_dic['YPosPixels'] = myint(self.HW_ODF_get_attribute_value(HW_text_inst_dic, 'YPosPixels'), 0)
        HW_text_attr_dic['PosRelativeToTopLeftOfImage'] = mystr(self.HW_ODF_get_attribute_value(HW_text_inst_dic, 'PosRelativeToTopLeftOfImageSetInstance'), 'N')
        HW_text_attr_dic['WordWrapWithinABoundingBox'] = mystr(self.HW_ODF_get_attribute_value(HW_text_inst_dic, 'WordWrapWithinABoundingBox'), 'Y')
        HW_text_attr_dic['BoundingBoxWidthPixelsIfWordWrap'] = myint(self.HW_ODF_get_attribute_value(HW_text_inst_dic, 'BoundingBoxWidthPixelsIfWordWrap'), 0)
        HW_text_attr_dic['BoundingBoxHeightPixelsIfWordWrap'] = myint(self.HW_ODF_get_attribute_value(HW_text_inst_dic, 'BoundingBoxHeightPixelsIfWordWrap'), 0)

        # recover the data from the TextStyle object associated to the given TextInstance object
        HW_text_style_dic = self.HW_ODF_get_object_dic_by_ref_id('TextStyle', HW_text_inst_dic, 'TextStyleID')
        HW_text_attr_dic['Face_WindowsName'] = mystr(self.HW_ODF_get_attribute_value(HW_text_style_dic, 'Face_WindowsName'), 'Arial')
        HW_text_attr_dic['Font_SizePixels'] = myint(self.HW_ODF_get_attribute_value(HW_text_style_dic, 'Font_SizePixels'), 10)
        HW_text_attr_dic['Font_WeightCode'] = myint(self.HW_ODF_get_attribute_value(HW_text_style_dic, 'Font_WeightCode'), 2)
        HW_text_attr_dic['Colour_Red'] = myint(self.HW_ODF_get_attribute_value(HW_text_style_dic, 'Colour_Red'), 0)
        HW_text_attr_dic['Colour_Green'] = myint(self.HW_ODF_get_attribute_value(HW_text_style_dic, 'Colour_Green'), 0)
        HW_text_attr_dic['Colour_Blue'] = myint(self.HW_ODF_get_attribute_value(HW_text_style_dic, 'Colour_Blue'), 0)
        HW_text_attr_dic['HorizontalAlignmentCode'] = myint(self.HW_ODF_get_attribute_value(HW_text_style_dic, 'HorizontalAlignmentCode'), 0)
        HW_text_attr_dic['VerticalAlignmentCode'] = myint(self.HW_ODF_get_attribute_value(HW_text_style_dic, 'VerticalAlignmentCode'), 1)

        # add in the HW_text_attr_dic the attributes of the associated ImageSetInstance object if one is defined
        HW_image_set_inst_dic = self.HW_ODF_get_object_dic_by_ref_id('ImageSetInstance', HW_text_inst_dic, 'AttachedToImageSetInstanceID')
        HW_text_attr_dic['ImageSetInstanceDic'] = HW_image_set_inst_dic
        if HW_image_set_inst_dic != None:
            self.HW_ODF_get_image_attributes(HW_image_set_inst_dic, HW_text_attr_dic)

        return True

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_switch_controlled_objects(self, HW_switch_dic, controlled_HW_objects_dic_list, is_linkage_inverted=False, can_control_keys=False):
        # recursive fonction which fills the given controlled_HW_objects_dic_list with the list of HW objects controlled by the given HW Switch (itself included in the list)
        # the parameter controlled_HW_objects_dic_list must be given as an empty list
        # the parameter is_linkage_inverted is for internal function usage, it indicates if the current control branch has an inverted effect on the controlled objects
        # if a HW Pipe_SoundEngine01 is controlled in inverted way, the key '_hint' = 'inverted' is added in this HW object
        # if a HW SwitchLinkage      is controlled as a condition,  the key '_hint' = 'condition' is added in this HW object

        #   Switch C> any object which can have a Switch as parent

        #   pipes ranks stop :
        #     Switch C> Stop C> StopRank(s) (ActionTypeCode = 1, ActionEffectCode = 1) C> Rank C> Pipe_SoundEngine01 ... (main or alternate rank)
        #     Switch C> Stop (Hint_PrimaryAssociatedRankID) C> Rank C> Pipe_SoundEngine01 ... (for some demo sample sets where there is no StopRank object defined)
        #   engage noise :
        #     Switch C> Stop C> StopRank (ActionTypeCode = 21, ActionEffectCode = 2) C> Rank C> Pipe_SoundEngine01 ...
        #     Switch C> SwitchLinkage (EngageLinkActionCode=1, DisengageLinkActionCode=2) C> Switch C> Pipe_SoundEngine01 ...
        #     Switch C> SwitchLinkage (EngageLinkActionCode=1, DisengageLinkActionCode=7) C> Switch C> Pipe_SoundEngine01 ...
        #     Switch C> SwitchLinkage (EngageLinkActionCode=4, DisengageLinkActionCode=7) C> Switch C> Pipe_SoundEngine01 ...
        #   engage noise or sustaining noise (i.e. blower) :
        #     Switch C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample (no ReleaseSample) ...
        #   disengage noise :
        #     Switch C> Stop C> StopRank (ActionTypeCode = 21, ActionEffectCode = 3) C> Rank C> Pipe_SoundEngine01 ...
        #     Switch C> SwitchLinkage (EngageLinkActionCode=1, DisengageLinkActionCode=2, SourceSwitchLinkIfEngaged=N) C> Switch C> Pipe_SoundEngine01 ...
        #     Switch C> SwitchLinkage (EngageLinkActionCode=7, DisengageLinkActionCode=4) C> Switch C> Pipe_SoundEngine01 ...
        #     Switch C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_ReleaseSample (AttackSample ignored) ...
        #   sustaining noise (i.e. blower) :
        #     Switch C> Stop C> StopRank (ActionTypeCode = 21, ActionEffectCode = 1) C> Rank C> Pipe_SoundEngine01 ...

        if HW_switch_dic == None:
            return

        if HW_switch_dic in controlled_HW_objects_dic_list:
            # the given switch has been already checked (it is closing a switches loop)
            if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} is closing a switches LOOP")
            return

        if (not can_control_keys and
            (self.HW_ODF_get_linked_objects_dic_by_type(HW_switch_dic, 'KeyboardKey', TO_CHILD, FIRST_ONE) != None or
             self.HW_ODF_get_linked_objects_dic_by_type(HW_switch_dic, 'DivisionInput', TO_CHILD, FIRST_ONE) != None)):
            # the given HW Switch is controlling a keyboard key or a division input, it is ignored
            return

        # add the given HW Switch in the list
        controlled_HW_objects_dic_list.append(HW_switch_dic)

        for HW_child_obj_dic in HW_switch_dic['_children']:
            # scan the objects controlled by the given HW Switch (which are its children)

            HW_child_obj_type = HW_child_obj_dic['_type']

            if HW_child_obj_type not in ('Switch', 'SwitchLinkage'):
                # SwitchLinkage has a special processing later in this function
                if is_linkage_inverted:
                    # the current child is controlled in an inverted way
                    HW_child_obj_dic['_hint'] = 'inverted'
                    if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} inverted")
                else:
                    if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']}")
                # add the controlled object in the list
                controlled_HW_objects_dic_list.append(HW_child_obj_dic)

            if HW_child_obj_type == 'Stop':
                # the current HW switch is controlling a Stop, find the noise Pipes controlled by this stop if any
                for HW_stop_rank_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_child_obj_dic, 'StopRank', TO_CHILD):
                    # scan the HW StopRank objects which are children of the HW Stop object
                    HW_action_type_code = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'ActionTypeCode'), 1)
                    HW_action_effect_code = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'ActionEffectCode'), 1)
                    if HW_action_type_code == 1 and HW_action_effect_code == 1:
                        # the current HW StopRank controls pipes rank
                        if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} > {HW_stop_rank_dic['_uid']} (pipes)")

                    elif HW_action_type_code == 21 and HW_action_effect_code in (1, 2, 3):
                        # the current HW StopRank controls noise samples
                        HW_rank_dic = self.HW_ODF_get_object_dic_by_ref_id('Rank', HW_stop_rank_dic, 'RankID')
                        HW_pipe_dic = None
                        # take into account a MIDI note increment if defined to use the proper Pipe_SoundEngine01 object
                        HW_div_midi_note_increment_to_rank = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'MIDINoteNumIncrementFromDivisionToRank'), 0)
                        if HW_div_midi_note_increment_to_rank != 0:
                            # search for the Pipe_SoundEngine01 object having the given MIDI note number
                            for HW_pipe_check_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_rank_dic, 'Pipe_SoundEngine01', TO_CHILD):
                                midi_note_nb = myint(self.HW_ODF_get_attribute_value(HW_pipe_check_dic, 'NormalMIDINoteNumber'))
                                if midi_note_nb == None: midi_note_nb = 60
                                if midi_note_nb == HW_div_midi_note_increment_to_rank:
                                    HW_pipe_dic = HW_pipe_check_dic
                                    break
                        if HW_pipe_dic == None:
                            # Pipe_SoundEngine01 object not found, take by default the first Pipe_SoundEngine01 child of the Rank
                            HW_pipe_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_rank_dic, 'Pipe_SoundEngine01', TO_CHILD, FIRST_ONE)
                        if HW_pipe_dic != None:
                            # a Pipe_SoundEngine01 is defined
                            controlled_HW_objects_dic_list.append(HW_pipe_dic)
                            if HW_action_effect_code in (1, 2):  # sustaining or engaging noise
                                if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} > {HW_stop_rank_dic['_uid']} > {HW_pipe_dic['_uid']} direct")
                            else: # HW_action_effect_code == 3:  # disengaging noise
                                HW_pipe_dic['_hint'] = 'inverted'
                                if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} > {HW_stop_rank_dic['_uid']} > {HW_pipe_dic['_uid']} inverted")

                HW_rank_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_child_obj_dic, 'Rank', TO_CHILD, FIRST_ONE)
                if HW_rank_dic != None:
                    # the Stop has a child HW Rank through the Hint_PrimaryAssociatedRankID attribute
                    if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} > {HW_rank_dic['_uid']} (pipes)")

            elif HW_child_obj_type == 'SwitchLinkage':
                controlled_HW_objects_dic_list.append(HW_child_obj_dic)
                HW_source_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_child_obj_dic, 'SourceSwitchID')
                if HW_switch_dic == HW_source_switch_dic:
                    # the given HW Switch is the source of the current SwitchLinkage (and not its condition)
                    EngageLinkActionCode = myint(self.HW_ODF_get_attribute_value(HW_child_obj_dic, 'EngageLinkActionCode'), 1)
                    DisengageLinkActionCode = myint(self.HW_ODF_get_attribute_value(HW_child_obj_dic, 'DisengageLinkActionCode'), 2)
                    SourceSwitchLinkIfEngaged = self.HW_ODF_get_attribute_value(HW_child_obj_dic, 'SourceSwitchLinkIfEngaged')
                    HW_dest_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_child_obj_dic, 'DestSwitchID')
                    HW_cond_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_child_obj_dic, 'ConditionSwitchID')

                    if ((EngageLinkActionCode == 1 and DisengageLinkActionCode == 2 and SourceSwitchLinkIfEngaged == 'N') or
                        (EngageLinkActionCode == 7 and DisengageLinkActionCode == 4)):
                        # inverting link
                        if HW_cond_switch_dic != None:
                            if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} towards {HW_dest_switch_dic['_uid']} by INVERTING linkage, with condition {HW_cond_switch_dic['_uid']}")
                        else:
                            if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} towards {HW_dest_switch_dic['_uid']} by INVERTING linkage")
                        self.HW_ODF_get_switch_controlled_objects(HW_dest_switch_dic, controlled_HW_objects_dic_list, not is_linkage_inverted, can_control_keys)

                    elif ((EngageLinkActionCode == 1 and DisengageLinkActionCode == 2) or
                          (EngageLinkActionCode == 4 and DisengageLinkActionCode == 7)):
                        # non inverting link
                        if HW_cond_switch_dic != None:
                            if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} towards {HW_dest_switch_dic['_uid']}, with condition {HW_cond_switch_dic['_uid']}")
                        else:
                            if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} towards {HW_dest_switch_dic['_uid']}")
                        self.HW_ODF_get_switch_controlled_objects(HW_dest_switch_dic, controlled_HW_objects_dic_list, is_linkage_inverted, can_control_keys)
                    else:
                        HW_child_obj_dic['_hint'] = 'not_supported_linkage'
                        if HW_cond_switch_dic != None:
                            if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} towards {HW_dest_switch_dic['_uid']} with unsupported engaged action code {EngageLinkActionCode} / disengage action code {DisengageLinkActionCode}, with condition {HW_cond_switch_dic['_uid']}")
                        else:
                            if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} towards {HW_dest_switch_dic['_uid']} with unsupported engaged action code {EngageLinkActionCode} / disengage action code {DisengageLinkActionCode}")

                else:
                    # the given HW Switch controls the current SwitchLinkage as a condition input switch
                    HW_child_obj_dic['_hint'] = 'condition'
                    if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} controls {HW_child_obj_dic['_uid']} as its condition switch")

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_keyboard_controlled_objects(self, HW_keyboard_dic, data_dic):
        # fill the given dictionary data_dic with the elements controlled by the given HW Keyboard directly or through non condition KeyAction
        # data_dic can be returned with the following keys
        #   'Division'     : dictionary of the first controlled HW Division
        #   'KeyboardKeys' : dictionary of the first controlled HW Keyboard having HW KeyboardKey children objects
        #   'KeyboardImg'  : dictionary of the first controlled HW Keyboard having a HW KeyImageSet children

        for HW_object_dic in HW_keyboard_dic['_children']:
            # scan the children of the given HW Keyboard
            HW_object_type = HW_object_dic['_type']

            if HW_object_type == 'Division' and 'Division' not in data_dic.keys():
                data_dic['Division'] = HW_object_dic

            elif HW_object_type == 'KeyboardKey' and 'KeyboardKeys' not in data_dic.keys():
                data_dic['KeyboardKeys'] = HW_keyboard_dic

            elif HW_object_type == 'KeyImageSet' and 'KeyboardImg' not in data_dic.keys():
                data_dic['KeyboardImg'] = HW_keyboard_dic

            elif HW_object_type == 'KeyAction':
                if self.HW_ODF_get_attribute_value(HW_object_dic, 'ConditionSwitchID') == None:
                    # the KeyAction is not controlled by a condition switch, it is always active
                    HW_dest_division_dic = self.HW_ODF_get_object_dic_by_ref_id('Division', HW_object_dic, 'DestDivisionID')
                    if HW_dest_division_dic != None and 'Division' not in data_dic.keys():
                        data_dic['Division'] = HW_dest_division_dic

                    HW_dest_keyboard_dic = self.HW_ODF_get_object_dic_by_ref_id('Keyboard', HW_object_dic, 'DestKeyboardID')
                    if HW_dest_keyboard_dic != None:
                        self.HW_ODF_get_keyboard_controlled_objects(HW_dest_keyboard_dic, data_dic)

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_controlling_continuous_controls(self, HW_cont_ctrl_dic, cont_ctrl_dic_lists, branch_nb=0):
        # recursive fonction which fills the given list cont_ctrl_dic_lists with the visible HW Continuous controls or the latching HW Switches
        # which are controlling the given HW Continuous Control
        # the HW ContinuousControlLinkage having a condition switch are ignored, but this switch is added to the given list if it latching

        if len(cont_ctrl_dic_lists) <= branch_nb:
            cont_ctrl_dic_lists.append([])

        if HW_cont_ctrl_dic not in cont_ctrl_dic_lists[branch_nb]:
            # the given HW continuous control has not been already checked (to avoid loops between controls)
            if self.HW_ODF_get_object_dic_by_ref_id('ImageSetInstance', HW_cont_ctrl_dic, 'ImageSetInstanceID') != None:
                # it has a graphical interface
                cont_ctrl_dic_lists[branch_nb].append(HW_cont_ctrl_dic)

            # check the HW continuous control objects controlling the given HW continuous control via a ContinuousControlLinkage
            for HW_cont_ctrl_link_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_cont_ctrl_dic, 'ContinuousControlLinkage', TO_PARENT):
                # scan the parent HW ContinuousControlLinkage objects
                HW_cond_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_cont_ctrl_link_dic, 'ConditionSwitchID')
                if HW_cond_switch_dic == None:
                    # it is a linkage without condition switch
                    HW_source_cont_ctrl_dic = self.HW_ODF_get_object_dic_by_ref_id('ContinuousControl', HW_cont_ctrl_link_dic, 'SourceControlID')
                    if HW_source_cont_ctrl_dic != None:
                        self.HW_ODF_get_controlling_continuous_controls(HW_source_cont_ctrl_dic, cont_ctrl_dic_lists, branch_nb)
                elif self.HW_ODF_get_attribute_value(HW_cond_switch_dic, 'Latching') != 'N':
                    # it is a linkage with a condition switch which is latching : add the condition switch in the given list
                    cont_ctrl_dic_lists[branch_nb].append(HW_cond_switch_dic)

            # check the HW continuous control objects controlling the given HW continuous control via a ContinuousControlDoubleLinkage
            for HW_cont_ctrl_dbl_link_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_cont_ctrl_dic, 'ContinuousControlDoubleLinkage', TO_PARENT):
                # scan the parent HW ContinuousControlDoubleLinkage objects
                HW_source_cont_ctrl_dic = self.HW_ODF_get_object_dic_by_ref_id('ContinuousControl', HW_cont_ctrl_dbl_link_dic, 'FirstSourceControl_ID')
                if HW_source_cont_ctrl_dic != None:
                    self.HW_ODF_get_controlling_continuous_controls(HW_source_cont_ctrl_dic, cont_ctrl_dic_lists, branch_nb)
                HW_source_cont_ctrl_dic = self.HW_ODF_get_object_dic_by_ref_id('ContinuousControl', HW_cont_ctrl_dbl_link_dic, 'SecondSourceControl_ID')
                if HW_source_cont_ctrl_dic != None:
                    self.HW_ODF_get_controlling_continuous_controls(HW_source_cont_ctrl_dic, cont_ctrl_dic_lists, branch_nb + 1)

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_pipe_noise_kind_check(self, HW_pipe_dic, known_noise_kind):
        # returns the verified kind of noise of the given HW Pipe : 'attack' or 'release', using the given known noise kind ('attack' or 'release')
        # returns 'release' when known_noise_kind is 'attack' and the attack sample of the HW Pipe is a file containing 'Blank'
        # else retunr known_noise_kind

        if known_noise_kind == 'attack':
            # check if the first attack sample of the given pipe is a silent sample (case of sample sets of Piotr Grabowski to manage release noises)

            # get the first attack sample linked to the given HW Pipe_SoundEngine01
            HW_pipe_layer_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD, FIRST_ONE)
            HW_pipe_attack_sample_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_layer_dic, 'Pipe_SoundEngine01_AttackSample', TO_CHILD, FIRST_ONE)
            HW_sample_dic = self.HW_ODF_get_object_dic_by_ref_id('Sample', HW_pipe_attack_sample_dic, 'SampleID')
            if HW_sample_dic != None and 'Blank' in self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename'):
                return 'release'

        return known_noise_kind

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_get_pipe_audio_channel_id(self, HW_pipe_dic):
        # returns an audio channel ID (based on windchest ID and continuous control ID if defined) for the given HW Pipe

        if HW_pipe_dic == None:
            return 0

        HW_windchest_dic = self.HW_ODF_get_object_dic_by_ref_id('WindCompartment', HW_pipe_dic, 'WindSupply_SourceWindCompartmentID')
        HW_pipe_layer_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD, FIRST_ONE)
        HW_cont_ctrl_dic = self.HW_ODF_get_object_dic_by_ref_id('ContinuousControl', HW_pipe_layer_dic, 'AmpLvl_ScalingContinuousControlID')

        if HW_cont_ctrl_dic != None:
            audio_channel_id = int(HW_windchest_dic['_uid'][-6:]) + int(HW_cont_ctrl_dic['_uid'][-6:]) << 8
        else:
            audio_channel_id = int(HW_windchest_dic['_uid'][-6:])

        return audio_channel_id

    #-------------------------------------------------------------------------------------------------
    def HW_ODF_save2textfile(self, file_name):
        # save the Hauptwerk ODF objects dictionary into the given text file path/name in a GrandOrgue ODF format (for development/debug purpose)

        with open(file_name, 'w', encoding=ENCODING_UTF8_BOM) as f:
            f.write(';Hauptwerk ODF XML formatted in a GrandOrgue ODF manner\n')
            f.write('\n')
            for object_type_dic in self.HW_odf_dic.values():
                for HW_object_dic in object_type_dic.values():
                    f.write(f'[{HW_object_dic["_uid"]}]\n')
                    for obj_attr_name, obj_attr_value in HW_object_dic.items():
                        if obj_attr_name in ('_parents', '_children'):
                            # this attribute contains a list of objects dictionaries
                            relations = ''
                            for HW_object_dic2 in obj_attr_value:
                                relations += (HW_object_dic2['_uid'] + '  ')
                            obj_attr_value = relations
                        f.write(f'{obj_attr_name}={obj_attr_value}\n')
                    f.write('\n')

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_save2organfile(self, file_name, file_encoding):
        # save the GrandOrgue ODF objects dictionary into the given .organ ODF file and in the given file encoding (ISO_8859_1 or UTF-8)
        # return True or False whether the saving has succeeded or not

        # check the extension of the given file name
        filename_str, file_extension_str = os.path.splitext(file_name)
        if file_extension_str != '.organ':
            logs.add(f'The file "{file_name}" does not have the expected extension .organ')
            return False

        with open(file_name, 'w', encoding=file_encoding) as f:
            # set the list of objects UID to save : Organ in first, then the others by alphabetical order

            # sort the objects UID
            uid_list = sorted(self.GO_odf_dic.keys())

            # move the Header and Organ objects in first and second positions in the UID list
            uid_list.remove('Header')
            uid_list.remove('Organ')
            uid_list.insert(0, 'Header')
            uid_list.insert(1, 'Organ')

            # write the objects in the file
            for object_uid in uid_list:
                # scan the defined UID of the GO ODF
                if object_uid != 'Header':
                    # there is no UID to write for the header of the ODF
                    f.write(f'[{object_uid}]\n')

                for obj_attr_name, obj_attr_value in self.GO_odf_dic[object_uid].items():
                    # scan the defined attributes of the current object UID
                    if obj_attr_name[0] == ';':
                        # it is a comment line, the comment text is placed in the value of the attribute
                        line = obj_attr_value + '\n'
                    elif obj_attr_name[0] != '_':
                        # it is not a temporary attribute created for HW to GO conversion
                        line = obj_attr_name + '=' + str(obj_attr_value) + '\n'
                    else:
                        line = None

                    if line != None:
                        if file_encoding == ENCODING_ISO_8859_1:
                            # convert the line from UTF-8 to ISO_8859_1 format
                            line = line.encode('utf-8', 'ignore').decode('ISO-8859-1', 'ignore')
                        f.write(line)

                f.write('\n')  # insert an empty line between each object section

        return True

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_from_HW_ODF(self, HW_odf_file_name, GO_odf_file_name, progress_status_update_fct,
                                 conv_trem_samples_bool,
                                 trem_samples_in_sep_ranks_bool,
                                 pitch_tuning_metadata_bool,
                                 pitch_tuning_filename_bool,
                                 build_alt_scr_layouts_bool,
                                 do_not_build_keys_noise_bool,
                                 build_unused_ranks_bool,
                                 GO_odf_encoding):
        # build and save a GrandOrgue ODF from the given Hauptwerk ODF and its associated sample set (which is not touched)
        # taking into the various given conversion options status
        # use the given function callback to display a progression status in the GUI
        # return False if an issue has occured, else return True

        self.reset_all_data()

        self.progress_status_update = progress_status_update_fct

        if conv_trem_samples_bool:
            # the tremmed samples have to be converted from HW to GO
            if trem_samples_in_sep_ranks_bool:
                # the tremmed samples have to be placed in separated ranks from the non tremmed ranks
                self.trem_samples_mode = 'separated'
            else:
                # the tremmed samples have to be integrated in the non tremmed ranks
                self.trem_samples_mode = 'integrated'
        else:
            self.trem_samples_mode = None

        self.tune_pitch_from_sample_metadata = pitch_tuning_metadata_bool
        self.tune_pitch_from_sample_filename = pitch_tuning_filename_bool

        # if alternate screens layouts have to be converted, the maximum layout ID is 3
        # an HW ODF can define up to 3 alternate screens layouts, in addition to the default one which has the ID 0
        self.max_screen_layout_id = 3 if build_alt_scr_layouts_bool else 0

        # load the HW ODF in the HW ODF dictionary
        self.progress_status_update('Loading the Hauptwerk ODF data...')
        if self.HW_ODF_load_from_file(HW_odf_file_name):
            # the loading has been done with success

            # link the HW objects together
            self.progress_status_update('Building the Hauptwerk ODF sections tree...')
            self.HW_ODF_do_links_between_objects()

            # build the list of the divisions dictionaries sorted by ascending ID order
            HW_sorted_divisions_dic_list = []
            for HW_division_id in sorted(self.HW_odf_dic['Division'].keys()):
                HW_sorted_divisions_dic_list.append(self.HW_odf_dic['Division'][HW_division_id])

            # build the list of the display pages dictionaries sorted by ascending ID order
            # excluding the Crescendo named pages which the contained features are not converted to GO ODF
            HW_sorted_display_pages_dic_list = []
            for HW_display_page_id in sorted(self.HW_odf_dic['DisplayPage'].keys()):
                HW_display_page_dic = self.HW_odf_dic['DisplayPage'][HW_display_page_id]
                page_name = HW_display_page_dic['Name'].upper()
                if not any(x in page_name for x in ['CRESC']):
                    HW_sorted_display_pages_dic_list.append(HW_display_page_dic)

            # build the various GO objects in the GO ODF dictionary from the HW ODF
            # the order of calling the below functions is important, there are dependencies between some of them

            # build a GO Header object with information about the conversion
            GO_header_dic = self.GO_odf_dic['Header'] = {}
            GO_header_dic[';1'] = f'; GrandOrgue ODF generated from Hauptwerk ODF "{os.path.basename(self.HW_odf_file_name)}"'
            GO_header_dic[';2'] = f'; on {date.today()} by OdfEdit {APP_VERSION} (see github.com/GrandOrgue/OdfEdit)'
            GO_header_dic[';3'] =  '; Conversion options :'
            GO_header_dic[';4'] = f';    Tremmed samples are {"converted." if conv_trem_samples_bool else "not converted."}'
            if conv_trem_samples_bool:
                GO_header_dic[';5'] = f';    Tremmed samples are placed {"in dedicated ranks." if trem_samples_in_sep_ranks_bool else "in same ranks as non tremmed samples."}'
            GO_header_dic[';6'] = f';    Alternate panels layouts are {"converted." if build_alt_scr_layouts_bool else "not converted."}'
            GO_header_dic[';7'] = f';    Manuals keys noises are {"not converted." if do_not_build_keys_noise_bool else "converted."}'
            GO_header_dic[';8'] = f';    Hauptwerk ranks unused in the generated GO ODF are {"converted." if build_unused_ranks_bool else "not converted."}'
            if pitch_tuning_metadata_bool:
                GO_header_dic[';9'] = '; Some pipes pitch tuning can have been set using MIDI note present in their audio sample metadata.'
            if pitch_tuning_filename_bool:
                GO_header_dic[';10'] = '; Some pipes pitch tuning can have been set using MIDI note present in their audio sample file name.'

            # build the GO Organ object
            self.progress_status_update('Building the GrandOrgue Organ section...')
            if self.GO_ODF_build_Organ_object() == None:
                logs.add('ERROR : issue occured while building the GO Organ section')
                return False

            # build the GO Panel objects by sorted HW DisplayPage ID order
            self.progress_status_update('Building GrandOrgue Panels...')
            for HW_display_page_dic in HW_sorted_display_pages_dic_list:
                self.GO_ODF_build_Panel_object(HW_display_page_dic)

            # build GO Manual objects from HW Keyboard objects
            self.progress_status_update('Building GrandOrgue Manuals...')
            for HW_keyboard_dic in self.HW_ODF_get_linked_objects_dic_by_type('root', 'Keyboard', sorted_by='ID'):
                self.GO_ODF_build_Manual_object(HW_keyboard_dic)

            # build GO Manual objects from HW Division objects not yet converted to a GO Manual by the previous loop
            # (case of a Division not directly controlled by a Keyboard)
            for HW_division_dic in self.HW_ODF_get_linked_objects_dic_by_type('root', 'Division', sorted_by='ID'):
                if HW_division_dic['_GO_uid'] == '':
                    # current HW Division is not yet converted to a GO Manual
                    self.GO_ODF_build_Manual_object(HW_division_dic)

            # build GO combination objects (General, Divisional)
            self.progress_status_update('Building GrandOrgue Combinations...')
            for HW_combination_dic in self.HW_odf_dic['Combination'].values():
                # scan the HW Combination objects
                self.GO_ODF_build_Drawstop_controlled_object(HW_combination_dic)
            # build the combination master capture button which is given in the _General object
            self.GO_ODF_build_Drawstop_controlled_object(self.HW_general_dic)

            # build the GO Coupler objects
            self.progress_status_update('Building GrandOrgue Couplers...')
            for HW_key_action_dic in self.HW_ODF_get_linked_objects_dic_by_type('root', 'KeyAction', sorted_by='SourceKeyboardID'):
                self.GO_ODF_build_Drawstop_controlled_object(HW_key_action_dic)

            # build the GO pipes Stop objects by sorted HW Division ID
            for HW_division_dic in HW_sorted_divisions_dic_list:
                for HW_stop_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_division_dic, 'Stop', TO_CHILD, sorted_by='ID'):
                    # scan the HW Stop objects belonging to the current HW Division for the stop pipes or noise objects
                    self.progress_status_update(f'Building the GrandOrgue Stop "{HW_stop_dic["Name"]}"...')
                    self.GO_ODF_build_Drawstop_controlled_object(HW_stop_dic)

            # build the GO Tremulant objects
            if 'Tremulant' in self.HW_odf_dic.keys():
                self.progress_status_update('Building GrandOrgue tremulants...')
                for HW_tremulant_dic in self.HW_odf_dic['Tremulant'].values():
                    self.GO_ODF_build_Drawstop_controlled_object(HW_tremulant_dic)

            # build GO Stop or Switch objects controlled by switches and which can have a function in GO (noises or switched images)
            self.progress_status_update('Building other switch controlled GrandOrgue features...')
            for HW_switch_dic in self.HW_ODF_get_linked_objects_dic_by_type('root', 'Switch', sorted_by='ID'):
                # scan the HW Switch objects
                self.GO_ODF_build_Switch_effects_object(HW_switch_dic)

            # build the GO Stop objects for keyboard keys noises by sorted HW Division ID
            if not do_not_build_keys_noise_bool:
                for HW_keyboard_dic in self.HW_ODF_get_linked_objects_dic_by_type('root', 'Keyboard', sorted_by='ID'):
                    self.progress_status_update(f'Building the GrandOrgue Stop for keyboard "{HW_keyboard_dic["Name"]}" keys noise...')
                    self.GO_ODF_build_Manual_noise_object(HW_keyboard_dic)

            # build the labels
            self.progress_status_update('Building the GrandOrgue Labels...')
            for HW_display_page_dic in HW_sorted_display_pages_dic_list:
                if HW_display_page_dic['_GO_uid'] != '':
                    # the current HW display page has been converted in a GO panel
                    # recover the corresponding GO panel
                    GO_panel_uid = HW_display_page_dic['_GO_uid']
                    for HW_text_inst_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_display_page_dic, 'TextInstance', TO_CHILD):
                        # scan the HW TextInstance objects of the current display page
                        self.GO_ODF_build_Label_object(HW_text_inst_dic, GO_panel_uid)

            if build_unused_ranks_bool:
                # build the GO Ranks objects corresponding to HW Ranks of pipes not converted in GO Ranks previously
                for HW_rank_dic in self.HW_ODF_get_linked_objects_dic_by_type('root', 'Rank', TO_CHILD, sorted_by='Name'):
                    if HW_rank_dic['_GO_uid'] == '':
                        # HW Rank not converted in a GO Rank or Stop
                        self.progress_status_update(f'Building unused GrandOrgue Rank "{HW_rank_dic["Name"]}"...')
                        GO_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic)
                        if GO_rank_uid != None:
                            self.GO_odf_dic[GO_rank_uid]['Name'] += ' UNUSED'

            self.progress_status_update('Completing the building operation...')

            # apply in the GO objects the references to children by object types alphabetical order
            for object_uid in self.GO_odf_dic.keys():
                self.GO_ODF_apply_children_ref(object_uid)

            # in the Divisional object, replace the references to Switch object by reference to their number in their manual
            self.GO_ODF_convert_divisionals_ref()

            # set the MIDIInputNumber attribute of enclosures objects
            self.GO_ODF_enclosures_midi_input_number_set()

            # if the file SilentLoop.wav is present at the root of the sample set, remove it
            # it is no more needed from OdfEdit v2.11 thanks to the HasIndependentRelease attribute introduced in GO 3.14.0
            if os.path.exists(self.HW_sample_set_path + os.path.sep + 'SilentLoop.wav'):
                os.remove(self.HW_sample_set_path + os.path.sep + 'SilentLoop.wav')

            # save the HW ODF data in a GO ODF text format (for development/debug purpose, more easy to read than a xml file)
##            self.HW_ODF_save2textfile(HW_odf_file_name + '.txt')

            # save the built GO ODF data in a .organ file
            if self.GO_ODF_save2organfile(GO_odf_file_name, GO_odf_encoding):
                logs.add(f'GrandOrgue ODF built and saved in "{GO_odf_file_name}"')

        # clear the last progression message
        self.progress_status_update('')

        return True

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Organ_object(self):
        # build the GO Organ object from the HW ODF
        # return None if an issue has occured, else return the UID of the created GO Organ

        # used HW objects :
        #   _General, RequiredInstallationPackage

        # check if the folders of the required installation packages are present in the folder OrganInstallationPackages
        self.available_HW_packages_id_list = []
        for HW_install_pack_dic in self.HW_odf_dic['RequiredInstallationPackage'].values():
            # scan and check the defined HW RequiredInstallationPackage objects
            HW_package_id = myint(self.HW_ODF_get_attribute_value(HW_install_pack_dic, 'InstallationPackageID', MANDATORY))
            if HW_package_id == None:
                logs.add('ERROR : no installation package ID defined in the HW ODF')
                return None

            HW_package_name = self.HW_ODF_get_attribute_value(HW_install_pack_dic, 'Name', MANDATORY)
            HW_package_supplier = self.HW_ODF_get_attribute_value(HW_install_pack_dic, 'SupplierName', MANDATORY)
            if HW_package_name == None or HW_package_supplier == None:
                logs.add('ERROR : no installation package name or supplier defined in the HW ODF')
                return None

            folder_name = os.path.join(self.HW_sample_set_path, 'OrganInstallationPackages', str(HW_package_id).zfill(6))
            if not os.path.isdir(folder_name):
                # the folder doesn't exist in the sample set package
                logs.add(f'WARNING : the package ID {HW_package_id} named "{HW_package_name}" provided by "{HW_package_supplier}"')
                logs.add(f'          is not present in the folder {path2ospath(folder_name)}')
                logs.add( '          some graphical or sound elements of this organ may be not rendered in GrandOrgue')
            else:
                self.available_HW_packages_id_list.append(HW_package_id)

        # recover the main installation package ID
        HW_install_package_id = myint(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'OrganInfo_InstallationPackageID', MANDATORY))
        if HW_install_package_id == None:
            logs.add('ERROR : no main installation package ID defined in the HW ODF')
            return None

        # add an entry in the GO ODF dictionary for the Organ object
        GO_organ_uid = 'Organ'
        GO_organ_dic = self.GO_odf_dic[GO_organ_uid] = {}

        GO_organ_dic['ChurchName'] = mystr(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'Identification_Name'))
        GO_organ_dic['ChurchAddress'] = mystr(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'OrganInfo_Location'))
        GO_organ_dic['OrganBuilder'] = mystr(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'OrganInfo_Builder'))
        GO_organ_dic['OrganBuildDate'] = mystr(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'OrganInfo_BuildDate'))
        GO_organ_dic['OrganComments'] = f'Sample set made for Hauptwerk converted for GrandOrgue on {date.today()} by OdfEdit {APP_VERSION} (see github.com/GrandOrgue/OdfEdit)'
        GO_organ_dic['RecordingDetails'] = mystr(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'Control_OrganDefinitionSupplierName', MANDATORY))

        GO_organ_dic['HasPedals'] = 'N'  # will be set later in GO_ODF_build_Manual_objects
        GO_organ_dic['NumberOfManuals'] = 0
        GO_organ_dic['NumberOfPanels'] = 0
        GO_organ_dic['NumberOfWindchestGroups'] = 0
        GO_organ_dic['NumberOfRanks'] = 0
        GO_organ_dic['NumberOfSwitches'] = 0
        GO_organ_dic['NumberOfEnclosures'] = 0
        GO_organ_dic['NumberOfTremulants'] = 0
        GO_organ_dic['NumberOfGenerals'] = 0
        GO_organ_dic['NumberOfDivisionalCouplers'] = 0
        GO_organ_dic['NumberOfReversiblePistons'] = 0

        GO_organ_dic['GeneralsStoreDivisionalCouplers'] = 'Y'
        GO_organ_dic['DivisionalsStoreTremulants'] = 'Y'
        GO_organ_dic['DivisionalsStoreIntermanualCouplers'] = 'Y'
        GO_organ_dic['DivisionalsStoreIntramanualCouplers'] = 'Y'
        GO_organ_dic['CombinationsStoreNonDisplayedDrawstops'] = 'N'

        # recover the ID of the HW default display page (used in GO_ODF_build_Panel_object function)
        self.HW_default_display_page_dic = self.HW_ODF_get_object_dic_by_ref_id('DisplayPage', self.HW_general_dic, 'SpecialObjects_DefaultDisplayPageID')
        if self.HW_default_display_page_dic == None:
            # cannot continue the convertion if there is no default display page defined
            logs.add('ERROR : no default display page defined in the HW ODF')
            return None

        # define the organ pitch tuning if the organ base pitch is defined
        self.organ_base_pitch_hz = myfloat(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'AudioEngine_BasePitchHz'), 440.0)
        if self.organ_base_pitch_hz == 0: self.organ_base_pitch_hz = 440.0
        organ_pitch_tuning = freq_diff_to_cents(440.0, self.organ_base_pitch_hz)
        if organ_pitch_tuning != 0:
            GO_organ_dic['PitchTuning'] = organ_pitch_tuning

        gain = myfloat(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'AudioOut_AmplitudeLevelAdjustDecibels'), 0)
        if gain != 0:
            GO_organ_dic['Gain'] = gain

        # add in the HW _General object the ID of the corresponding GO object
        self.HW_general_dic['_GO_uid'] = 'Organ'

        self.GO_organ_dic = GO_organ_dic

        return GO_organ_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Panel_object(self, HW_display_page_dic):
        # build a GO Panel999 corresponding to each screen layer defined in the given HW DisplayPage
        # build also the GO static images objects which are defined in this panel
        # return the UID of the created GO Panel object of the default screen layout or None in case of error

        # used HW objects :
        #   _General
        #   DisplayPage
        #   DisplayPage C> Keyboard
        #   DisplayPage C> ImageSetInstance C> ImageSet C> ImageSetElement (in function GO_ODF_build_Image_object)

        page_name = self.HW_ODF_get_attribute_value(HW_display_page_dic, 'Name')
        ret_GO_panel_uid = None  # Panel UID returned by the function

        if self.HW_ODF_get_linked_objects_dic_by_type(HW_display_page_dic, 'Keyboard', TO_CHILD, FIRST_ONE) != None:
            # the given HW DisplayPage object contains at least one Keyboard object in his children, so it is the HW console page
            self.HW_console_display_page_dic = HW_display_page_dic

        # group by layer number the static images defined in the given HW display page
        # to build their corresponding GO panels in the same order so that they are visible in GO as they are visible in HW
        HW_images_list_per_layer_dict = {}
        for HW_image_set_inst_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_display_page_dic, 'ImageSetInstance', TO_CHILD):
            # scan the children HW ImageSetInstance objects of the given HW DisplayPage
            if len(HW_image_set_inst_dic['_parents']) == 1:
                # the current HW ImageSetInstance object has a single parent (a DisplayPage) : it is a static image
                HW_layer_nb_int = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'ScreenLayerNumber'), 1)  # default to 1
                if HW_layer_nb_int not in HW_images_list_per_layer_dict.keys():
                    # there is not yet an entry in the dictionary for the layer number of the current HW ImageSetInstance
                    # add one entry initialized with an empty list
                    HW_images_list_per_layer_dict[HW_layer_nb_int] = []
                # add the current HW ImageSetInstance to the list of the layer numbers
                HW_images_list_per_layer_dict[HW_layer_nb_int].append(HW_image_set_inst_dic)

        # count how many screens layouts (including the default one) are defined for this display page and considering the maximum layout ID allowed
        layouts_nb = 0
        for layout_id in range(0, self.max_screen_layout_id + 1):
            # scan the screens layouts IDs
            if layout_id == 0 or self.HW_ODF_get_attribute_value(HW_display_page_dic, f'AlternateConsoleScreenLayout{layout_id}_Include') == 'Y':
                layouts_nb += 1

        # create panels for default (layout_id = 0) and alternate (layout_id > 0) screens layouts
        layout_nb = 0
        for layout_id in range(0, self.max_screen_layout_id + 1):
            # scan the screen layouts IDs
            if layout_id == 0 or self.HW_ODF_get_attribute_value(HW_display_page_dic, f'AlternateConsoleScreenLayout{layout_id}_Include') == 'Y':
                # the current layout ID is used to display a page (default or alternate)
                layout_nb += 1
                if layout_id == 0 and HW_display_page_dic == self.HW_default_display_page_dic:
                    # this is the HW default display page of the default layout, so assigned to the GO Panel000
                    GO_panel_uid = 'Panel000'
                else:
                    self.GO_organ_dic['NumberOfPanels'] += 1  # Panel000 is not counted
                    GO_panel_uid = 'Panel' + str(self.GO_organ_dic['NumberOfPanels']).zfill(3)
                if layout_id == 0: ret_GO_panel_uid = GO_panel_uid

                # add a GO Panel object in the GO ODF dictionary
                GO_panel_dic = self.GO_odf_dic[GO_panel_uid] = {}
                GO_panel_dic['_GO_uid'] = GO_panel_uid
                GO_panel_dic['Name'] = page_name if layouts_nb == 1 else f'{page_name} #{layout_nb}'
                GO_panel_dic['HasPedals'] = 'N'  # will be set later in GO_ODF_build_Manual_objects
                GO_panel_dic['NumberOfGUIElements'] = 0
                GO_panel_dic['NumberOfImages'] = 0

                # add in the HW DisplayPage object the UID of the corresponding GO Panel object
                if layout_id == 0: HW_display_page_dic['_GO_uid'] = GO_panel_uid
                HW_display_page_dic[f'_GO_uid_layout{layout_id}'] = GO_panel_uid

                if HW_display_page_dic == self.HW_console_display_page_dic:
                    # the current HW DisplayPage object is the HW console page, get the dimensions of the console page defined in the HW _General object
                    if layout_id == 0:
                        GO_panel_dic['DispScreenSizeHoriz'] = myint(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'Display_ConsoleScreenWidthPixels'), 0)
                        GO_panel_dic['DispScreenSizeVert'] = myint(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'Display_ConsoleScreenHeightPixels'), 0)
                    else:
                        GO_panel_dic['DispScreenSizeHoriz'] = myint(self.HW_ODF_get_attribute_value(self.HW_general_dic, f'Display_AlternateConsoleScreenLayout{layout_id}_WidthPixels'), 0)
                        GO_panel_dic['DispScreenSizeVert'] = myint(self.HW_ODF_get_attribute_value(self.HW_general_dic, f'Display_AlternateConsoleScreenLayout{layout_id}_HeightPixels'), 0)
                else:
                    GO_panel_dic['DispScreenSizeHoriz'] = 0  # will be set later when GO_ODF_build_Panel_size_update will be called
                    GO_panel_dic['DispScreenSizeVert'] = 0

                # set the other mandatory attributes of a GO panel at a default value
                GO_panel_dic['DispDrawstopBackgroundImageNum'] = '1'
                GO_panel_dic['DispDrawstopInsetBackgroundImageNum'] = '1'
                GO_panel_dic['DispConsoleBackgroundImageNum'] = '1'
                GO_panel_dic['DispKeyHorizBackgroundImageNum'] = '1'
                GO_panel_dic['DispKeyVertBackgroundImageNum'] = '1'
                GO_panel_dic['DispControlLabelFont'] = 'Arial'
                GO_panel_dic['DispShortcutKeyLabelFont'] = 'Arial'
                GO_panel_dic['DispShortcutKeyLabelColour'] = 'Black'
                GO_panel_dic['DispGroupLabelFont'] = 'Arial'
                GO_panel_dic['DispDrawstopCols'] = '2'
                GO_panel_dic['DispDrawstopRows'] = '1'
                GO_panel_dic['DispDrawstopColsOffset'] = 'N'
                GO_panel_dic['DispPairDrawstopCols'] = 'N'
                GO_panel_dic['DispExtraDrawstopRows'] = '0'
                GO_panel_dic['DispExtraDrawstopCols'] = '0'
                GO_panel_dic['DispButtonCols'] = '1'
                GO_panel_dic['DispExtraButtonRows'] = '0'
                GO_panel_dic['DispExtraPedalButtonRow'] = 'N'
                GO_panel_dic['DispButtonsAboveManuals'] = 'N'
                GO_panel_dic['DispExtraDrawstopRowsAboveExtraButtonRows'] = 'N'
                GO_panel_dic['DispTrimAboveManuals'] = 'N'
                GO_panel_dic['DispTrimBelowManuals'] = 'N'
                GO_panel_dic['DispTrimAboveExtraRows'] = 'N'

                # build by layer order the static images defined in the current screen layout of the display page
                for HW_layer_nb_int in sorted(HW_images_list_per_layer_dict.keys()):
                    # scan the HW display layers of the page by ascending order in order to build the images in this same order in the GO panel
                    for HW_image_set_inst_dic in HW_images_list_per_layer_dict[HW_layer_nb_int]:
                        # scan the HW ImageSetInstance objects of the current display layer
                        self.GO_ODF_build_Image_object(HW_image_set_inst_dic, layout_id, GO_panel_dic)

        return ret_GO_panel_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Panel_size_update(self, GO_panel_dic, x, y):
        # increase the width or height of the given GO panel so that the given coordinates are visible in this panel

        if x != None and x > GO_panel_dic['DispScreenSizeHoriz']: GO_panel_dic['DispScreenSizeHoriz'] = x
        if y != None and y > GO_panel_dic['DispScreenSizeVert']: GO_panel_dic['DispScreenSizeVert'] = y

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Image_object(self, HW_image_set_inst_dic, layout_id, GO_panel_dic):
        # build a GO Panel999Image999 object in the given GO Panel corresponding to the given HW ImageSetInstance + layout ID
        # return the UID of the created GO PanelImage object or None if no panel image created

        # used HW objects :
        #   ImageSetInstance C> ImageSet C> ImageSetElement

        image_attr_dic = {}
        if self.HW_ODF_get_image_attributes(HW_image_set_inst_dic, image_attr_dic, None, layout_id) and image_attr_dic['BitmapFilename'] != None:
            # the data about the current HW ImageSetInstance object have been recovered successfully and an image file name is defined

            if image_attr_dic['ImageWidthPixels'] == None or image_attr_dic['ImageHeightPixels'] == None:
                # if one dimension of the image is not defined, get the dimensions of the image in the bitmap file
                image_filename = os.path.dirname(self.HW_odf_file_name) + os.path.sep + path2ospath(image_attr_dic['BitmapFilename'])
                if os.path.isfile(image_filename):
                    im = Image.open(image_filename)
                    image_attr_dic['ImageWidthPixels'] = im.size[0]
                    image_attr_dic['ImageHeightPixels'] = im.size[1]
                else:
                    image_attr_dic['ImageWidthPixels'] = None
                    image_attr_dic['ImageHeightPixels'] = None

            # create a GO Panel999Image999 object for the given GO Panel
            GO_panel_uid = GO_panel_dic['_GO_uid']
            self.GO_odf_dic[GO_panel_uid]['NumberOfImages'] += 1
            GO_panel_image_uid = GO_panel_uid + 'Image' + str(self.GO_odf_dic[GO_panel_uid]['NumberOfImages']).zfill(3)
            GO_panel_image_dic = self.GO_odf_dic[GO_panel_image_uid] = {}

            # set the position and sizes of the image
            image_max_x = image_max_y = 0
            if image_attr_dic['LeftXPosPixels'] > 0:
                GO_panel_image_dic['PositionX'] = image_attr_dic['LeftXPosPixels']
            if image_attr_dic['TopYPosPixels'] > 0:
                GO_panel_image_dic['PositionY'] = image_attr_dic['TopYPosPixels']
            if image_attr_dic['ImageWidthPixels'] != None:
                GO_panel_image_dic['Width'] = image_attr_dic['ImageWidthPixels']
                image_max_x = image_attr_dic['LeftXPosPixels'] + image_attr_dic['ImageWidthPixels']
            if image_attr_dic['ImageHeightPixels'] != None:
                GO_panel_image_dic['Height'] = image_attr_dic['ImageHeightPixels']
                image_max_y = image_attr_dic['TopYPosPixels'] + image_attr_dic['ImageHeightPixels']

            # set the image and its mask if any
            GO_panel_image_dic['Image'] = image_attr_dic['BitmapFilename']
            if image_attr_dic['TransparencyMaskBitmapFilename'] != None:
                GO_panel_image_dic['Mask'] = image_attr_dic['TransparencyMaskBitmapFilename']

            # increase if necessary the GO panel dimensions to display entirely the image
            self.GO_ODF_build_Panel_size_update(GO_panel_dic, image_max_x, image_max_y)

            # add in the HW ImageSetInstance object the ID of the corresponding GO object
            HW_image_set_inst_dic['_GO_uid'] = GO_panel_image_uid

            return GO_panel_image_uid

        return None

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Label_object(self, HW_text_inst_dic, GO_panel_uid):
        # build a GO Panel999Element999 object with type=Label corresponding to the given HW TextInstance and in the given GO Panel
        # sub-function of GO_ODF_build_Panel_object
        # return the UID of the created GO Panel Element object, or None if no panel element created

        # used HW objects :
            #   TextInstance C> TextStyle
            #   TextInstance C> ImageSetInstance C> ImageSet C> ImageSetElement

        # recover the attributes of the given HW TextInstance object and his linked HW ImageSetInstance object if any
        text_attr_dic = {}
        if not self.HW_ODF_get_text_attributes(HW_text_inst_dic, text_attr_dic):
            return False

        # create a GO Panel999Element999 object with label type in the given GO panel
        self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements'] += 1
        GO_panel_element_uid = GO_panel_uid + 'Element' + str(self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements']).zfill(3)
        GO_panel_element_dic = self.GO_odf_dic[GO_panel_element_uid] = {}

        GO_panel_element_dic['Type'] = 'Label'
        GO_panel_element_dic['Name'] = text_attr_dic['Text'].replace('\n', ' ')  # replace eventual carriage returns by black space

        # set the font size/name and text colour
        GO_panel_element_dic['DispLabelFontSize'] = text_attr_dic['Font_SizePixels']
        GO_panel_element_dic['DispLabelFontName'] = text_attr_dic['Face_WindowsName']
        GO_panel_element_dic['DispLabelColour'] = f"#{text_attr_dic['Colour_Red']:02X}{text_attr_dic['Colour_Green']:02X}{text_attr_dic['Colour_Blue']:02X}"

        # recover the display dimensions of the label text according to the font name/size/weight
        text_font = tkf.Font(family=text_attr_dic['Face_WindowsName'], size=1 * text_attr_dic['Font_SizePixels'],
                             weight='bold' if text_attr_dic['Font_WeightCode'] == 3 else 'normal')
        text_width = text_font.measure(text_attr_dic['Text'])
        text_height = text_font.metrics('ascent') + text_font.metrics('descent')

        # move the XPosPixels value at the left of the text according to the text horizontal alignment attribute
        if text_attr_dic['HorizontalAlignmentCode'] in (0, 3): # centered : move by the half of the text width
            text_attr_dic['XPosPixels'] -= int(text_width / 2)
        elif text_attr_dic['HorizontalAlignmentCode'] == 2: # right aligned : move by the text width
            text_attr_dic['XPosPixels'] -= text_width
        if text_attr_dic['XPosPixels'] < 0:
            text_attr_dic['XPosPixels'] = 0

        # move the YPosPixels value at the top of the text according to the text vertical alignment attribute
        if text_attr_dic['VerticalAlignmentCode'] == 0: # centered : move by the half of the text height
            text_attr_dic['YPosPixels'] -= int(text_height / 2)
        elif text_attr_dic['VerticalAlignmentCode'] == 2: # bottom aligned : move by the text height
            text_attr_dic['YPosPixels'] -= text_height
        if text_attr_dic['YPosPixels'] < 0:
            text_attr_dic['YPosPixels'] = 0

        # get the absolute position of the top left corner of the text
        if text_attr_dic['ImageSetInstanceDic'] == None or text_attr_dic['PosRelativeToTopLeftOfImage'] == 'N':
            # no image attached to the text or text not placed relatively to an eventual attached image : XPosPixels / YPosPixels are already absolute values
            text_abs_x = text_attr_dic['XPosPixels']
            text_abs_y = text_attr_dic['YPosPixels']
        else:
            # an image is attached to the text and the text is placed relatively to the position of its attached image
            text_abs_x = text_attr_dic['LeftXPosPixels'] + text_attr_dic['XPosPixels']
            text_abs_y = text_attr_dic['TopYPosPixels'] + text_attr_dic['YPosPixels']


        if text_attr_dic['ImageSetInstanceDic'] == None or text_attr_dic['BitmapFilename'] == None:
            # no image attached to the text or the attached image is not found (image from Hauptwerk default images bank for example)
            # build a GO label without background image

            GO_panel_element_dic['DispImageNum'] = 0  # no background image

            # set the text position
            GO_panel_element_dic['PositionX'] = text_abs_x
            GO_panel_element_dic['PositionY'] = text_abs_y

            # set the width and height attributes with the text width and height to be sure that the text is displayed entirely and in a single ligne by GO
            GO_panel_element_dic['Width'] = text_width
            GO_panel_element_dic['Height'] = text_height

        else:
            # the text has a background image

            # set the image file name
            GO_panel_element_dic['Image'] = text_attr_dic['BitmapFilename']
            # set the image mask file name
            if text_attr_dic['TransparencyMaskBitmapFilename'] != None:
                GO_panel_element_dic['Mask'] = text_attr_dic['TransparencyMaskBitmapFilename']

            # get the image dimensions if they are not defined in the HW ODF
            if text_attr_dic['ImageWidthPixels'] == None or text_attr_dic['ImageHeightPixels'] == None:
                image_path = self.HW_sample_set_odf_path + os.path.sep + path2ospath(text_attr_dic['BitmapFilename'])
                if os.path.isfile(image_path):
                    im = Image.open(image_path)
                    text_attr_dic['ImageWidthPixels'] = im.size[0]
                    text_attr_dic['ImageHeightPixels'] = im.size[1]

            # set the position of the image
            GO_panel_element_dic['PositionX'] = text_attr_dic['LeftXPosPixels']
            GO_panel_element_dic['PositionY'] = text_attr_dic['TopYPosPixels']

            # set the image dimensions
            if text_attr_dic['ImageWidthPixels'] != None:
                GO_panel_element_dic['Width'] = text_attr_dic['ImageWidthPixels']
            if text_attr_dic['ImageHeightPixels'] != None:
                GO_panel_element_dic['Height'] = text_attr_dic['ImageHeightPixels']

            # set the text position relatively to the image position
            GO_panel_element_dic['TextRectLeft'] = text_abs_x - GO_panel_element_dic['PositionX']
            GO_panel_element_dic['TextRectTop'] = text_abs_y - GO_panel_element_dic['PositionY']

            # set the text dimensions
            if text_attr_dic['BoundingBoxWidthPixelsIfWordWrap'] > 0:
                # a text boundary is defined in the HW ODF
                GO_panel_element_dic['TextRectWidth'] = text_attr_dic['BoundingBoxWidthPixelsIfWordWrap']
            else:
                # use text width for the text rectangle width
                GO_panel_element_dic['TextRectWidth'] = text_width

            if text_attr_dic['BoundingBoxHeightPixelsIfWordWrap'] > 0:
                # a text boundary is defined in the HW ODF
                GO_panel_element_dic['TextRectHeight'] = text_attr_dic['BoundingBoxHeightPixelsIfWordWrap']
            else:
                # use text height for the text rectangle width
                GO_panel_element_dic['TextRectHeight'] = text_height

            if ('Width' in GO_panel_element_dic.keys() and
                GO_panel_element_dic['TextRectLeft'] + GO_panel_element_dic['TextRectWidth'] > GO_panel_element_dic['Width']):
                # the text width is larger than the image width (can occur in case of carriage returns removed in the text)
                # use the full rectangle of the image to display the text (which is centered inside by GO)
                GO_panel_element_dic['TextRectLeft'] = 0
                GO_panel_element_dic['TextRectTop'] = 0
                GO_panel_element_dic['TextRectWidth'] = GO_panel_element_dic['Width']
                GO_panel_element_dic['TextRectHeight'] = GO_panel_element_dic['Height']


        # check if the label is overlapping a GO panel element (but Manual or Label) of the same panel
        # if yes set the DispLabelText attribute of the GO panel element with the name of the label and delete the GO label element just build before
        for object_uid, object_dic in self.GO_odf_dic.items():
            # scan the objects of the GO ODF
            if len(object_uid) == 18 and object_uid[:8] == GO_panel_uid:
                # Panel999Element999 object which the UID starts with the given panel UID
                GO_panelem_dic = object_dic
                if GO_panelem_dic['Type'] not in ('Manual', 'Label'):
                    # define on which width/height dimensions the overlapping has to be considered
                    if 'Width' in GO_panelem_dic.keys():
                        check_width = GO_panelem_dic['Width']
                    else:
                        # the width of the overlapped element is unknown
                        if GO_panelem_dic['Type'] in ('General', 'Divisional', 'GC', 'Set'):
                            check_width = 32  # default piston image size in GO
                        else:
                            check_width = 65  # default drawstop image size in GO
                    if 'Height' in GO_panelem_dic.keys():
                        check_height = GO_panelem_dic['Height']
                    else:
                        # the width of the overlapped element is unknown
                        if GO_panelem_dic['Type'] in ('General', 'Divisional', 'GC', 'Set'):
                            check_height = 32  # default piston image size in GO
                        else:
                            check_height = 65  # default drawstop image size in GO

                    if (GO_panel_element_dic['PositionX'] in range(GO_panelem_dic['PositionX'], GO_panelem_dic['PositionX'] + check_width) and
                        GO_panel_element_dic['PositionY'] in range(GO_panelem_dic['PositionY'], GO_panelem_dic['PositionY'] + check_height)):
                        # the top left corner of the label is inside the panel element area, so it is overlapping it
                        if mydickey(GO_panelem_dic, '_label_text') == None:
                            GO_panelem_dic['DispLabelText'] = GO_panel_element_dic['Name']
                            GO_panelem_dic['_label_text'] = ''
                        else:
                            GO_panelem_dic['DispLabelText'] += ' ' + GO_panel_element_dic['Name']
                        GO_panelem_dic['DispLabelFontSize'] = 11
                        GO_panelem_dic['DispLabelColour'] = 'Black'
                        # remove the attribute TextBreakWidth to permit the DispLabelText to be displayed
                        if 'TextBreakWidth' in GO_panelem_dic.keys(): del GO_panelem_dic['TextBreakWidth']

                        if GO_panelem_dic['Type'] == 'Switch':
                            # remove the image attributes of the switch else the image will hide the text
                            if 'ImageOn' in GO_panelem_dic.keys(): del GO_panelem_dic['ImageOn']
                            if 'ImageOff' in GO_panelem_dic.keys(): del GO_panelem_dic['ImageOff']
                            if 'MaskOn' in GO_panelem_dic.keys(): del GO_panelem_dic['MaskOn']
                            if 'MaskOff' in GO_panelem_dic.keys(): del GO_panelem_dic['MaskOff']
                            # set the dimensions of the default drawstop image of GrangOrgue
                            if 'Width' in GO_panelem_dic.keys(): GO_panelem_dic['Width'] = 65
                            if 'Height' in GO_panelem_dic.keys(): GO_panelem_dic['Height'] = 65
                            if 'MouseRectWidth' in GO_panelem_dic.keys(): GO_panelem_dic['MouseRectWidth'] = 65
                            if 'MouseRectHeight' in GO_panelem_dic.keys(): GO_panelem_dic['MouseRectHeight'] = 65

                        # get the absolute position of the label
                        if 'TextRectLeft' in GO_panel_element_dic.keys():
                            label_abs_x = GO_panel_element_dic['PositionX'] + GO_panel_element_dic['TextRectLeft']
                        else:
                            label_abs_x = GO_panel_element_dic['PositionX']

                        if 'TextRectTop' in GO_panel_element_dic.keys():
                            label_abs_y = GO_panel_element_dic['PositionY'] + GO_panel_element_dic['TextRectTop']
                        else:
                            label_abs_y = GO_panel_element_dic['PositionY']

                        # get the position of the label relatively to the position of the hosting panel element
                        label_rel_x = label_abs_x - GO_panelem_dic['PositionX']
                        label_rel_y = label_abs_y - GO_panelem_dic['PositionY']

                        # place the label inside the hosting panel element if it has a defined image
                        if 'ImageOn' in GO_panelem_dic.keys():
                            if 'TextRectLeft' not in GO_panelem_dic.keys() or GO_panelem_dic['TextRectLeft'] > label_rel_x:
                                GO_panelem_dic['TextRectLeft'] = label_rel_x
                            if 'TextRectTop' not in GO_panelem_dic.keys() or GO_panelem_dic['TextRectTop'] > label_rel_y:
                                GO_panelem_dic['TextRectTop'] = label_rel_y
                        else:
                            # delete the TextRectLeft and TextRectTop attributes if there is no defined image in the PanelElement
                            if 'TextRectLeft' in GO_panelem_dic.keys(): del GO_panelem_dic['TextRectLeft']
                            if 'TextRectTop' in GO_panelem_dic.keys(): del GO_panelem_dic['TextRectTop']

                        # delete the label built before in this function
                        self.GO_odf_dic.pop(GO_panel_element_uid)
                        self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements'] -= 1

                        # add in the HW TextInstance and ImageSetInstance objects the ID of the corresponding GO object
                        HW_text_inst_dic['_GO_uid'] = object_uid

                        return None

        # add in the HW TextInstance and ImageSetInstance objects the ID of the corresponding GO object
        HW_text_inst_dic['_GO_uid'] = GO_panel_element_uid
        if text_attr_dic['ImageSetInstanceDic'] != None:
            text_attr_dic['ImageSetInstanceDic']['_GO_uid'] = GO_panel_element_uid

        return GO_panel_element_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Manual_object(self, HW_object_dic):
        # build a GO Manual999 object based on the given HW Keyboard or Division object
        # return the UID of the built GO Manuel

        # used HW objects :
        #   Keyboard C> Division
        #            C> KeyAction (no condition) C> Division
        #            C> KeyboardKey P> Switch C> SwitchLinkage C> Switch C> DivisionInput P> Division
        #   Division C> DivisionInput (MIDI note)

        # keyboard_disp_mode = 1
        #   Keyboard C> KeyboardKey P> Switch C> ImageSetInstance C> ImageSet C> ImageSetElement
        #                                     C> SwitchLinkage (engage_action_code is 1 & disengage_action_code is 2) C> Switch C> DivisionInput
        # keyboard_disp_mode = 2
        #   Keyboard C> KeyImageSet C> ImageSet C> ImageSetElement
        #            C> KeyAction (no condition) C> Keyboard C> KeyImageSet C> ImageSet C> ImageSetElement

        HW_object_type = HW_object_dic['_type']
        if HW_object_type == 'Keyboard':
            HW_keyboard_dic = HW_object_dic
            HW_division_dic = None
        elif HW_object_type == 'Division':
            HW_keyboard_dic = None
            HW_division_dic = HW_object_dic
        else:
            return None

        if HW_object_type == 'Keyboard':
            # if the attribute DefaultInputOutputKeyboardAsgnCode is defined, it means that the keyboard is visible at the position corresponding to this code
            #   1 : Pedal
            #   2 : Manual 1
            #   3 : Manual 2
            #   4 : Manual 3
            #   5 : Manual 4
            #   6 : Manual 5
            #   7 : Manual 6 miscellaneous/noises
            HW_keyboard_assignment_code = myint(self.HW_ODF_get_attribute_value(HW_keyboard_dic, 'DefaultInputOutputKeyboardAsgnCode'), 0)

            if LOG_HW2GO_manual: print(f"Building GO manual from HW {HW_keyboard_dic['_uid']} '{HW_keyboard_dic['Name']}', DefaultInputOutputKeyboardAsgnCode={HW_keyboard_assignment_code}")

            if (HW_keyboard_assignment_code == 0 and
                (self.HW_ODF_get_linked_objects_dic_by_type(HW_keyboard_dic, 'KeyAction', TO_PARENT, FIRST_ONE) == None or
                 self.HW_ODF_get_linked_objects_dic_by_type(HW_keyboard_dic, 'KeyAction', TO_CHILD, FIRST_ONE) == None)):
                # the given HW Keyboard is not visible and it is not forwarding action from parent or to child KeyAction, it is ignored
                if LOG_HW2GO_manual: print("    ====> SKIPPED : not visible and not forwarding KeyAction effect")
                return None

            # find what the HW Keyboard is controlling (Division, Keyboard with KeyboardKey children, KeyImageSet)
            data_dic = {}
            self.HW_ODF_get_keyboard_controlled_objects(HW_keyboard_dic, data_dic)
            HW_division_dic      = mydickey(data_dic, 'Division')
            HW_keyboard_keys_dic = mydickey(data_dic, 'KeyboardKeys')
            HW_keyboard_img_dic  = mydickey(data_dic, 'KeyboardImg')

            if HW_division_dic == None:
                # there is no HW Division controlled by the given HW Keyboard through KeyAction or defined in Hint_PrimaryAssociatedDivisionID attribute
                # try to find the controlled division through the switch action of a KeyboardKey
                #     Keyboard C> KeyboardKey P> Switch C> >>> DivisionInput P> Division
                for HW_keyboard_key_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_keyboard_dic, 'KeyboardKey', TO_CHILD):
                    HW_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_keyboard_key_dic, 'SwitchID')
                    controlled_HW_objects_dic_list = []
                    self.HW_ODF_get_switch_controlled_objects(HW_switch_dic, controlled_HW_objects_dic_list, can_control_keys=True)
                    for HW_ctrl_object_dic in controlled_HW_objects_dic_list:
                        # scan the controlled objects to find a DivisionInput
                        if HW_ctrl_object_dic['_type'] == 'DivisionInput':
                            HW_division_dic = self.HW_ODF_get_object_dic_by_ref_id('Division', HW_ctrl_object_dic, 'DivisionID')
                    if HW_division_dic != None:
                        break

        else:  # HW_object_type == 'Division'
            # division without associated keyboard
            HW_keyboard_assignment_code = 0
            HW_keyboard_keys_dic = None
            HW_keyboard_img_dic = None
            if LOG_HW2GO_manual: print(f"Building GO manual from HW {HW_division_dic['_uid']} '{HW_division_dic['Name']}'")

        if HW_division_dic != None:
            # get in priority the name of the HW Division rather than one of the HW Keyboard
            manual_name = HW_division_dic['Name']
        else:
            manual_name = HW_keyboard_dic['Name']

        # define the keyboard display mode
        if HW_keyboard_img_dic != None:
            keyboard_disp_mode = 2  # keyboard graphical elements are defined for one octave
        elif HW_keyboard_keys_dic != None:
            keyboard_disp_mode = 1  # keyboard graphical elements are defined for each key
        else:
            keyboard_disp_mode = 0  # keyboard is not visible

        # define the GO Manual UID to associate to the given HW Keyboard or Division
        if HW_keyboard_assignment_code == 1:
            # Pedal keyboard
            GO_manual_id = 0
        else:
            # other visible keyboards
            self.GO_organ_dic['NumberOfManuals'] += 1
            GO_manual_id = self.GO_organ_dic['NumberOfManuals']
        GO_manual_uid = 'Manual' + str(GO_manual_id).zfill(3)
        self.last_manual_uid = GO_manual_uid

        # create the GO Manual object
        GO_manual_dic = self.GO_odf_dic[GO_manual_uid] = {}
        GO_manual_dic['_GO_uid'] = GO_manual_uid

        if LOG_HW2GO_manual:
            if HW_division_dic == None:
                print(f"    GO {GO_manual_uid} '{manual_name}' created, with no associated HW division, keyboard_disp_mode={keyboard_disp_mode}")
            else:
                print(f"    GO {GO_manual_uid} '{manual_name}' created, acting on HW {HW_division_dic['_uid']}, keyboard_disp_mode={keyboard_disp_mode}")

        # add in the HW Keyboard and Division objects the UID of the corresponding HW and GO objects
        if HW_keyboard_dic != None:
            HW_keyboard_dic['_GO_uid'] = GO_manual_uid
            if HW_division_dic != None:
                # add in the HW keyboard its associated HW division
                HW_keyboard_dic['_HW_division_dic'] = HW_division_dic

        if HW_division_dic != None:
            if HW_division_dic['_GO_uid'] != '': print(f"!!!!! {HW_division_dic['_uid']} is already associated to {HW_division_dic['_GO_uid']}")
            HW_division_dic['_GO_uid'] = GO_manual_uid

        # update in the GO Organ and in the console GO Panel the HasPedal attribute value
        if GO_manual_uid == 'Manual000':
            self.GO_organ_dic['HasPedals'] = 'Y'
            if self.HW_console_display_page_dic != None:
                self.GO_odf_dic[self.HW_console_display_page_dic['_GO_uid']]['HasPedals'] = 'Y'

        # get the compass of the logical and accessible keys of the keyboard

        key_div_mapping_dic = {} # dictionary containing the mapping between keyboard keys MIDI notes and division MIDI notes if defined in the HW ODF
        keys_switch_dic = {}     # dictionary containing the HW Switch object associated to each MIDI note if defined in the HW ODF

        logical_key_midi_first = 999
        logical_key_midi_last = 0
        access_key_midi_first = 999
        access_key_midi_last = 0

        if HW_keyboard_keys_dic != None:
            # the given HW Keyboard has children KeyboardKey objects or is controlling another Keyboard having KeyboardKey children
            for HW_keyboard_key_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_keyboard_keys_dic, 'KeyboardKey', TO_CHILD):
                # scan the children HW KeyboardKey objects to recover the keys MIDI notes compass (logical and accessible keys)

                # recover the Switch controlling the current HW KeyboardKey
                HW_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_keyboard_key_dic, 'SwitchID')
                if self.HW_ODF_get_attribute_value(HW_switch_dic, 'Disp_ImageSetInstanceID') != None:
                    # the key Switch is visible

                    # recover the MIDI note number of the current HW KeyboardKey object (it is an accessible key)
                    key_midi_note_nb = myint(self.HW_ODF_get_attribute_value(HW_keyboard_key_dic, 'NormalMIDINoteNumber'), 60)  # MIDI note = 60 if not defined
                    access_key_midi_first = min(access_key_midi_first, key_midi_note_nb)
                    access_key_midi_last = max(access_key_midi_last, key_midi_note_nb)

                    keys_switch_dic[key_midi_note_nb] = HW_switch_dic

                    # add in the HW KeyboardKey object the UID of the corresponding GO object
                    HW_keyboard_key_dic['_GO_uid'] = GO_manual_uid

                    # recover the MIDI note of the HW DivisionInput linked to the current HW KeyboardKey
                    for HW_switch_linkage_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_switch_dic, 'SwitchLinkage', TO_CHILD):
                        # scran the HW SwitchLinkage objects controlled by the HW Switch, one can make link to a DivisionInput
                        HW_engage_action_code = myint(self.HW_ODF_get_attribute_value(HW_switch_linkage_dic, 'EngageLinkActionCode'))
                        HW_disengage_action_code = myint(self.HW_ODF_get_attribute_value(HW_switch_linkage_dic, 'DisengageLinkActionCode'))
                        if HW_engage_action_code == 1 and HW_disengage_action_code == 2:
                            # standard action codes
                            HW_dest_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_switch_linkage_dic, 'DestSwitchID')
                            HW_division_input_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_dest_switch_dic, 'DivisionInput', TO_CHILD, FIRST_ONE)
                            if HW_division_input_dic != None:
                                # linked DivisionInput is found
                                # recover the MIDI note number of the HW DivisionInput object (it is a logical key)
                                div_midi_note_nb = myint(self.HW_ODF_get_attribute_value(HW_division_input_dic, 'NormalMIDINoteNumber'), 60)
                                logical_key_midi_first = min(logical_key_midi_first, div_midi_note_nb)
                                logical_key_midi_last = max(logical_key_midi_last, div_midi_note_nb)
                                key_div_mapping_dic[key_midi_note_nb] = div_midi_note_nb
                            break  # no need to check other SwitchLinkage objects

            # ensure that the logical keys compass covers the accessible keys entire compass
            logical_key_midi_first = min(logical_key_midi_first, access_key_midi_first)
            logical_key_midi_last = max(logical_key_midi_last, access_key_midi_last)

        # if no HW Keyboard is defined and a HW Division is defined, scan the DivisionInput objects to find the logical keys compass
        if HW_keyboard_dic == None and HW_division_dic != None:
            for HW_division_input_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_division_dic, 'DivisionInput', TO_CHILD):
                # recover the MIDI note number of the HW DivisionInput object (it is a logical key)
                div_midi_note_nb = myint(self.HW_ODF_get_attribute_value(HW_division_input_dic, 'NormalMIDINoteNumber'), 60)
                logical_key_midi_first = min(logical_key_midi_first, div_midi_note_nb)
                logical_key_midi_last = max(logical_key_midi_last, div_midi_note_nb)
            # set the accessible compass equal to the logical compass
            access_key_midi_first = logical_key_midi_first
            access_key_midi_last = logical_key_midi_last

        if access_key_midi_last == 0:
            # no accessible compass found yet
            # recover the accessible compass from the Keyboard object attributes
            access_key_midi_first = myint(self.HW_ODF_get_attribute_value(HW_keyboard_dic, 'KeyGen_MIDINoteNumberOfFirstKey'), 999)
            if access_key_midi_first < 999:
                # a first key MIDI note number is defined
                access_key_midi_last = access_key_midi_first + myint(self.HW_ODF_get_attribute_value(HW_keyboard_dic, 'KeyGen_NumberOfKeys')) - 1

        if logical_key_midi_last == 0:
            # no logical compass found yet
            # set the logical compass equal to the accessible compass
            logical_key_midi_first = access_key_midi_first
            logical_key_midi_last = access_key_midi_last

        if LOG_HW2GO_manual:
            print(f'    logic keys {logical_key_midi_first}-{logical_key_midi_last}, accessible keys {access_key_midi_first}-{access_key_midi_last}')
            for key_midi_note_nb in sorted(key_div_mapping_dic.keys()):
                print(f'    key {key_midi_note_nb} -> div {key_div_mapping_dic[key_midi_note_nb]}')

        # set GO manual attributes
        GO_manual_dic['Name'] = manual_name
        if access_key_midi_first == 999:
            # there are no accessible keys, set default values
            GO_manual_dic['NumberOfLogicalKeys'] = 1
            GO_manual_dic['NumberOfAccessibleKeys'] = 0
            GO_manual_dic['FirstAccessibleKeyLogicalKeyNumber'] = 1
            GO_manual_dic['FirstAccessibleKeyMIDINoteNumber'] = 0
        else:
            GO_manual_dic['NumberOfLogicalKeys'] = logical_key_midi_last - logical_key_midi_first + 1
            GO_manual_dic['NumberOfAccessibleKeys'] = access_key_midi_last - logical_key_midi_first + 1
                          # DisplayFirstNote is set in the manual panel element to hide notes between logical first and accessible first
            if access_key_midi_first in key_div_mapping_dic.keys():
                # a mapping is defined between key MIDI note and division MIDI note
                # apply it for the first accessible key
                GO_manual_dic['FirstAccessibleKeyLogicalKeyNumber'] = key_div_mapping_dic[access_key_midi_first] - logical_key_midi_first + 1
                GO_manual_dic['FirstAccessibleKeyMIDINoteNumber'] = key_div_mapping_dic[access_key_midi_first]
            else:
                GO_manual_dic['FirstAccessibleKeyLogicalKeyNumber'] = access_key_midi_first - logical_key_midi_first + 1
                GO_manual_dic['FirstAccessibleKeyMIDINoteNumber'] = access_key_midi_first

            # set the MIDI input number (position of the keyboard on the console)
            if HW_keyboard_assignment_code != 0:
                GO_manual_dic['MIDIInputNumber'] = HW_keyboard_assignment_code

        GO_manual_dic['NumberOfCouplers'] = 0
        GO_manual_dic['NumberOfDivisionals'] = 0
        GO_manual_dic['NumberOfStops'] = 0
        GO_manual_dic['NumberOfSwitches'] = 0
        GO_manual_dic['NumberOfTremulants'] = 0

        if keyboard_disp_mode == 0 or HW_keyboard_assignment_code == 0:
            # the keyboard is not visible, stop here the building of the GO Manual
            return GO_manual_uid

        if access_key_midi_first > logical_key_midi_first:
            # the first keys of the logical keyboard are not accessible : map them to no MIDI note to make them inactive
            for key_midi_note_nb in range(logical_key_midi_first, access_key_midi_first):
                GO_manual_dic['MIDIKey' + str(key_midi_note_nb).zfill(3)] = 0

        for key_midi_note_nb in sorted(key_div_mapping_dic.keys()):
            # scan the key to pipe mapping dictionary
            if key_midi_note_nb != key_div_mapping_dic[key_midi_note_nb]:
                # the current key is mapped to a pipe having another MIDI note than its normal key MIDI note : map it to this other MIDI note
                GO_manual_dic['MIDIKey' + str(key_midi_note_nb).zfill(3)] = key_div_mapping_dic[key_midi_note_nb]

        # define the manual graphical attributes in Panel999Element999 objects with Type = Manual

        # get the HW DisplayPage ID in which is displayed the keyboard
        if keyboard_disp_mode == 1:
            # recover the display page ID from the HW ImageSetInstance of the first key of the keyboard
            HW_image_set_inst_id = myint(self.HW_ODF_get_attribute_value(keys_switch_dic[access_key_midi_first], 'Disp_ImageSetInstanceID', MANDATORY))
            HW_image_set_inst_dic = self.HW_ODF_get_object_dic_from_id('ImageSetInstance', HW_image_set_inst_id)
            keyboard_disp_page_id = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'DisplayPageID'))
        elif keyboard_disp_mode == 2:
            # recover the display page ID from the HW Keyboard object
            keyboard_disp_page_id = myint(self.HW_ODF_get_attribute_value(HW_keyboard_img_dic, 'KeyGen_DisplayPageID'))
        HW_display_page_dic = self.HW_ODF_get_object_dic_from_id('DisplayPage', keyboard_disp_page_id)

        if HW_display_page_dic == None:
            # the HW display page is not found, stop here the building of the GO Manual
            return GO_manual_uid

        for layout_id in range(0, self.max_screen_layout_id + 1):
            # scan the various screen layouts
            if (layout_id == 0 or
                (keyboard_disp_mode == 1 and self.HW_ODF_get_object_dic_by_ref_id('ImageSet', HW_image_set_inst_dic, f'AlternateScreenLayout{layout_id}_ImageSetID') != None) or
                (keyboard_disp_mode == 2 and self.HW_ODF_get_object_dic_by_ref_id('ImageSet', HW_keyboard_img_dic, f'KeyGen_AlternateScreenLayout{layout_id}_KeyImageSetID') != None)):
                # the keyboard is displayed in the current screen layout

                # recover the GO panel UID corresponding to the HW display page and screen layout ID
                GO_panel_uid = HW_display_page_dic[f'_GO_uid_layout{layout_id}']

                # create the GO Panel999Element999 object to display the keyboard
                self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements'] += 1
                GO_panel_element_uid = GO_panel_uid + 'Element' + str(self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements']).zfill(3)
                GO_panel_element_dic = self.GO_odf_dic[GO_panel_element_uid] = {}
                GO_panel_element_dic['Type'] = 'Manual'
                GO_panel_element_dic['Manual'] = str(int(GO_manual_uid[-3:])).zfill(3)

                GO_panel_element_dic['_GO_uid'] = GO_manual_uid
                if LOG_HW2GO_manual: print(f"    Panel element {GO_panel_element_uid} built in layout ID {layout_id}")

                if access_key_midi_first > logical_key_midi_first:
                    # the first keys of the logical keyboard are not accessible
                    # the number of keys to display is the number of accessible keys
                    GO_panel_element_dic['DisplayKeys'] = access_key_midi_last - access_key_midi_first + 1
                    # display keys from the first accessible key
                    GO_panel_element_dic['DisplayFirstNote'] = access_key_midi_first

                # apply keys to MIDI note remapping if defined
                for key_midi_note_nb in sorted(key_div_mapping_dic.keys()):
                    # scan the key to pipe mapping dictionary
                    if key_midi_note_nb != key_div_mapping_dic[key_midi_note_nb]:
                        # the current key is mapped to a pipe having another MIDI note than its normal key MIDI note : map it to this other MIDI note
                        GO_panel_element_dic['DisplayKey' + str(key_midi_note_nb - access_key_midi_first + 1).zfill(3)] = key_div_mapping_dic[key_midi_note_nb]

                # define the graphical properties of the GO Manual
                if keyboard_disp_mode == 1:
                    # keys graphical aspect is defined for each key
                    for midi_note_nb in range(access_key_midi_first, access_key_midi_last + 1):
                        # scan the switches of the HW Keyboard by increasing MIDI note number
                        GO_key_nb = midi_note_nb - access_key_midi_first + 1

                        if midi_note_nb < access_key_midi_last:
                            # it is not the latest key of the keyboard
                            self.GO_ODF_build_Manual_keyimage_by_switch(keys_switch_dic[midi_note_nb], keys_switch_dic[midi_note_nb + 1], GO_panel_element_dic, GO_key_nb, layout_id)
                        else:
                            self.GO_ODF_build_Manual_keyimage_by_switch(keys_switch_dic[midi_note_nb], None, GO_panel_element_dic, GO_key_nb, layout_id)

                else:
                    # keys graphical aspect is defined for one octave + the first and last keys

                    # get the HW KeyImageSet and keyboard position associated to the HW Keyboard for the current screen layout
                    if layout_id == 0:
                        HW_key_img_set_dic = self.HW_ODF_get_object_dic_by_ref_id('KeyImageSet', HW_keyboard_img_dic, 'KeyGen_KeyImageSetID')
                        GO_panel_element_dic['PositionX'] = myint(self.HW_ODF_get_attribute_value(HW_keyboard_img_dic, 'KeyGen_DispKeyboardLeftXPos'))
                        GO_panel_element_dic['PositionY'] = myint(self.HW_ODF_get_attribute_value(HW_keyboard_img_dic, 'KeyGen_DispKeyboardTopYPos'))
                    else:
                        HW_key_img_set_dic = self.HW_ODF_get_object_dic_by_ref_id('KeyImageSet', HW_keyboard_img_dic, 'KeyGen_AlternateScreenLayout{layout_id}_KeyImageSetID')
                        GO_panel_element_dic['PositionX'] = myint(self.HW_ODF_get_attribute_value(HW_keyboard_img_dic, 'KeyGen_AlternateScreenLayout{layout_id}_DispKeyboardLeftXPos'))
                        GO_panel_element_dic['PositionY'] = myint(self.HW_ODF_get_attribute_value(HW_keyboard_img_dic, 'KeyGen_AlternateScreenLayout{layout_id}_DispKeyboardTopYPos'))

                    # get the key up (not pressed) and key down (pressed) images index within image set if defined, else use the default index
                    key_up_img_index = myint(self.HW_ODF_get_attribute_value(HW_key_img_set_dic, 'ImageIndexWithinImageSets_Disengaged'))
                    if key_up_img_index == None: key_up_img_index = 1
                    HW_key_img_set_dic['_key_up_img_index'] = key_up_img_index

                    key_down_img_index = myint(self.HW_ODF_get_attribute_value(HW_key_img_set_dic, 'ImageIndexWithinImageSets_Engaged'))
                    if key_down_img_index == None: key_down_img_index = 2
                    HW_key_img_set_dic['_key_down_img_index'] = key_down_img_index

                    if len(self.keys_disp_attr_dic) == 0:
                        # build the dictionary containing the HW/GO display attributes names of the keyboard keys for when they are defined at octave level
                        self.build_keyboard_octave_disp_attr_dic()

                    # ---------------------------
                    # define the visual attributes of the first key of the keyboard
                    first_note_name, octave = midi_nb_to_note(access_key_midi_first)
                    key_disp_attr_dic = self.keys_disp_attr_dic[first_note_name]

                    # set the first key on/off image and mask
                    self.GO_ODF_build_Manual_keyimage_by_keytype(HW_key_img_set_dic, key_disp_attr_dic['HW_type_first'],
                                                                 GO_panel_element_dic, key_disp_attr_dic['GO_type_first'], layout_id)

                    # set the first key width
                    GO_panel_element_dic['Width_' + key_disp_attr_dic['GO_type_first']] = self.HW_ODF_get_attribute_value(HW_key_img_set_dic,
                                                                                                                          'HorizSpacingPixels_' + key_disp_attr_dic['HW_hspacing'])

                    # set the key offset
                    GO_panel_element_dic['Offset_' + key_disp_attr_dic['GO_type_first']]   = '0'

                    # ---------------------------
                    # define the visual attributes for each of the standard 12 keys of one octave
                    for note_name in NOTES_NAMES:  # NOTES_NAMES = ('C', 'C#', 'D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B')

                        key_disp_attr_dic = self.keys_disp_attr_dic[note_name]

                        # set the keys on/off image and mask
                        self.GO_ODF_build_Manual_keyimage_by_keytype(HW_key_img_set_dic, key_disp_attr_dic['HW_type'],
                                                                     GO_panel_element_dic, key_disp_attr_dic['GO_type'], layout_id)

                        # set the keys width
                        GO_panel_element_dic['Width_' + key_disp_attr_dic['GO_type']]   = self.HW_ODF_get_attribute_value(HW_key_img_set_dic,
                                                                                                                          'HorizSpacingPixels_' + key_disp_attr_dic['HW_hspacing'])

                        # set the keys offset
                        GO_panel_element_dic['Offset_' + key_disp_attr_dic['GO_type']]   = '0'

                    # ---------------------------
                    # define the visual attributes of the last key of the keyboard
                    last_note_name,  octave = midi_nb_to_note(access_key_midi_last)
                    key_disp_attr_dic = self.keys_disp_attr_dic[last_note_name]

                    # set the first key on/off image and mask
                    self.GO_ODF_build_Manual_keyimage_by_keytype(HW_key_img_set_dic, key_disp_attr_dic['HW_type_last'],
                                                                 GO_panel_element_dic, key_disp_attr_dic['GO_type_last'], layout_id)

                    # the last key width is not needed as there is no key to display at its right

                    # set the key offset
                    GO_panel_element_dic['Offset_' + key_disp_attr_dic['GO_type_last']]   = '0'

        return GO_manual_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Manual_keyimage_by_keytype(self, HW_key_img_set_dic, HW_key_type, GO_panel_element_dic, GO_key_type, layout_id):
        # add in the given GO panel element the key images attributes of the given HW key type
        # sub-function of GO_ODF_build_Manual_object

        HW_image_set_id = myint(self.HW_ODF_get_attribute_value(HW_key_img_set_dic, 'KeyShapeImageSetID_' + HW_key_type))
        if HW_image_set_id != None:
            HW_image_set_dic = self.HW_ODF_get_object_dic_from_id('ImageSet', HW_image_set_id)

            # image for key up (not pressed)
            image_attr_dic = {}
            self.HW_ODF_get_image_attributes(HW_image_set_dic, image_attr_dic, HW_key_img_set_dic['_key_up_img_index'], layout_id)
            if image_attr_dic['BitmapFilename'] != None:
                GO_panel_element_dic['ImageOff_' + GO_key_type] = image_attr_dic['BitmapFilename']
            if image_attr_dic['TransparencyMaskBitmapFilename'] != None:
                GO_panel_element_dic['MaskOff_' + GO_key_type] = image_attr_dic['TransparencyMaskBitmapFilename']

            # image for key down (pressed)
            image_attr_dic = {}
            self.HW_ODF_get_image_attributes(HW_image_set_dic, image_attr_dic, HW_key_img_set_dic['_key_down_img_index'], layout_id)
            if image_attr_dic['BitmapFilename'] != None:
                GO_panel_element_dic['ImageOn_' + GO_key_type] = image_attr_dic['BitmapFilename']
            if image_attr_dic['TransparencyMaskBitmapFilename'] != None:
                GO_panel_element_dic['MaskOn_' + GO_key_type] = image_attr_dic['TransparencyMaskBitmapFilename']

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Manual_keyimage_by_switch(self, HW_switch_dic, HW_next_switch_dic, GO_panel_element_dic, GO_key_nb, layout_id):
        # add in the given GO panel element the key images attributes of the given HW Switch
        # sub-function of GO_ODF_build_Manual_object

        if HW_switch_dic == None: return

        HW_image_set_inst_id = myint(self.HW_ODF_get_attribute_value(HW_switch_dic, 'Disp_ImageSetInstanceID', MANDATORY))
        HW_image_set_inst_dic = self.HW_ODF_get_object_dic_from_id('ImageSetInstance', HW_image_set_inst_id)

        # get the key engaged and disengaged images indexes
        key_up_img_index = myint(self.HW_ODF_get_attribute_value(HW_switch_dic, 'Disp_ImageSetIndexDisengaged', MANDATORY))
        key_down_img_index = myint(self.HW_ODF_get_attribute_value(HW_switch_dic, 'Disp_ImageSetIndexEngaged', MANDATORY))

        # add in the HW Switch and ImageSetInstance objects the UID of the corresponding GO object
        HW_switch_dic['_GO_uid'] = GO_panel_element_dic['_GO_uid']
        HW_image_set_inst_dic['_GO_uid'] = GO_panel_element_dic['_GO_uid']

        key_nb_3digit_str = str(GO_key_nb).zfill(3)

        if GO_key_nb == 1:
            # set the GO keyboard position which is the position of the first key
            image_attr_dic = {}
            self.HW_ODF_get_image_attributes(HW_image_set_inst_dic, image_attr_dic, key_up_img_index, layout_id)
            GO_panel_element_dic['PositionX'] = image_attr_dic['LeftXPosPixels']
            GO_panel_element_dic['PositionY'] = image_attr_dic['TopYPosPixels']

        # image for key up (not pressed)
        image_attr_dic = {}
        self.HW_ODF_get_image_attributes(HW_image_set_inst_dic, image_attr_dic, key_up_img_index, layout_id)
        if image_attr_dic['BitmapFilename'] != None:
            GO_panel_element_dic['Key' + key_nb_3digit_str + 'ImageOff'] = image_attr_dic['BitmapFilename']
        if image_attr_dic['TransparencyMaskBitmapFilename'] != None:
            GO_panel_element_dic['Key' + key_nb_3digit_str + 'MaskOff'] = image_attr_dic['TransparencyMaskBitmapFilename']

        # image for key down (pressed)
        image_attr_dic = {}
        self.HW_ODF_get_image_attributes(HW_image_set_inst_dic, image_attr_dic, key_down_img_index, layout_id)
        if image_attr_dic['BitmapFilename'] != None:
            GO_panel_element_dic['Key' + key_nb_3digit_str + 'ImageOn'] = image_attr_dic['BitmapFilename']
        if image_attr_dic['TransparencyMaskBitmapFilename'] != None:
            GO_panel_element_dic['Key' + key_nb_3digit_str + 'MaskOn'] = image_attr_dic['TransparencyMaskBitmapFilename']

        # width of the key, width calculated by the diff of XPos of the key and its next one
        if HW_next_switch_dic != None:
            HW_next_img_set_instance_id = myint(self.HW_ODF_get_attribute_value(HW_next_switch_dic, 'Disp_ImageSetInstanceID', MANDATORY))
            HW_next_img_set_instance_dic = self.HW_ODF_get_object_dic_from_id('ImageSetInstance', HW_next_img_set_instance_id)

            next_image_dic = {}
            self.HW_ODF_get_image_attributes(HW_next_img_set_instance_dic, next_image_dic, key_up_img_index, layout_id)
            key_width = int(next_image_dic['LeftXPosPixels']) - int(image_attr_dic['LeftXPosPixels'])
            if key_width <= 0:
                # key width is not correct, recover it from the image width
                image_filename = os.path.dirname(self.HW_odf_file_name) + os.path.sep + path2ospath(image_attr_dic['BitmapFilename'])
                if os.path.isfile(image_filename):
                    im = Image.open(image_filename)
                    key_width = im.size[0]
            else:  # key_width > 0:
                GO_panel_element_dic['Key' + key_nb_3digit_str + 'Width'] = str(key_width)

        # offset of the key
        GO_panel_element_dic['Key' + key_nb_3digit_str + 'Offset'] = '0'
        if image_attr_dic['TopYPosPixels'] - GO_panel_element_dic['PositionY'] != 0:
            GO_panel_element_dic['Key' + key_nb_3digit_str + 'YOffset'] = image_attr_dic['TopYPosPixels'] - GO_panel_element_dic['PositionY']

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Manual_noise_object(self, HW_keyboard_dic):
        # build GO Stop data with rank data inside for keyboard keys action noise rendering from the given HW Keyboard

        # used HW objects :
        #   attack noise (key press) :
        #    Division C> Stop C> StopRank(1 per key) (ActionTypeCode = 1, ActionEffectCode = 2) C> Rank C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample
        #    Division C> Stop C> StopRank (ActionTypeCode = 1, ActionEffectCode = 2) C> Rank C> Pipe_SoundEngine01 (1 per key) C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample
        #    Division C> DivisionInput (1 per key) P> Switch C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample
        #    Keyboard C> KeyboardKey (1 per key) P> Switch C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample
        #    Keyboard C> KeyboardKey (1 per key) P> Switch C> SwitchLinkage (EngageLinkActionCode=4, DisengageLinkActionCode=7) C> Switch C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample
        #   release noise (key release) :
        #    Division C> Stop C> StopRank(1 per key) (ActionTypeCode = 1, ActionEffectCode = 3) C> Rank C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample
        #    Division C> Stop C> StopRank (ActionTypeCode = 1, ActionEffectCode = 3) C> Rank C> Pipe_SoundEngine01 (1 per key) C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample
        #    Division C> DivisionInput (1 per key) P> Switch C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample (blank) + Pipe_SoundEngine01_ReleaseSample
        #    Division C> DivisionInput (1 per key) P> Switch C> SwitchLinkage (SourceSwitchLinkIfEngaged=N) C> Switch C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample (blank) + Pipe_SoundEngine01_ReleaseSample
        #    Keyboard C> KeyboardKey (1 per key) P> Switch C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample (blank) + Pipe_SoundEngine01_ReleaseSample
        #    Keyboard C> KeyboardKey (1 per key) P> Switch C> SwitchLinkage (EngageLinkActionCode=7, DisengageLinkActionCode=4) C> Switch C> Pipe_SoundEngine01 C> Pipe_SoundEngine01Layer C> Pipe_SoundEngine01_AttackSample

        # in case of configuration KeyboardKey > Switch1 > SwitchLinkage > Switch2 > Pipe_SoundEngine01, one key can trigger several switch2 and noise pipes (for several audio channels for example)

        # recover the GO manual corresponding to the given HW Keyboard, and its number of accessible keys
        GO_manual_uid = HW_keyboard_dic['_GO_uid']
        if GO_manual_uid == '':
            # the given HW Keyboard has not been converted to a GO Manual
            return

        GO_manual_dic = self.GO_odf_dic[GO_manual_uid]
        GO_manual_keys_nb = GO_manual_dic['NumberOfAccessibleKeys']
        GO_manual_first_midi_note_nb = GO_manual_dic['FirstAccessibleKeyMIDINoteNumber']
        if LOG_HW2GO_keys_noise: print(f'Building keys noises Stop for {GO_manual_uid} from MIDI note {GO_manual_first_midi_note_nb} and {GO_manual_keys_nb} keys')

        # recover the HW Division linked to the given HW Keyboard
        HW_division_dic = mydickey(HW_keyboard_dic, '_HW_division_dic')

        HW_pipes_descr_dic = {}  # dictionary in which are assembled the attack/release keys noise pipes for each audio channel ID and for each key MIDI note number
                                 # each item of the dictionary has the format : {channel_id:{midi_note:[attack_pipe_dic, release_pipe_dic]}}

        if HW_division_dic != None:
            # recover key noises if they are defined under HW Division / Stop / StopRank objects
            for HW_stop_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_division_dic, 'Stop', TO_CHILD):
                # scan the children Stops of the given HW Division
                for HW_stop_rank_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_stop_dic, 'StopRank', TO_CHILD):
                    # scan the children StopRanks of the current Stop
                    HW_action_type_code = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'ActionTypeCode'), 1)
                    HW_action_effect_code = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'ActionEffectCode'), 1)
                    if HW_action_type_code == 1 and HW_action_effect_code in (2, 3):
                        # it is a StopRank acting on a key noise samples
                        # get the properties of the current StopRank
                        nb_of_mapped_div_inputs = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'NumberOfMappedDivisionInputNodes'), GO_manual_keys_nb)
                        first_mapped_div_midi_note = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'MIDINoteNumOfFirstMappedDivisionInputNode'), 36)
                        midi_note_increment = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'MIDINoteNumIncrementFromDivisionToRank'), 0)
                        # get the Rank linked to the current StopRank
                        HW_rank_dic = self.HW_ODF_get_object_dic_by_ref_id('Rank', HW_stop_rank_dic, 'RankID')
                        if HW_rank_dic != None :
                            # apply the MIDI note increment to the first mapped MIDI note
                            first_mapped_pipe_midi_note = first_mapped_div_midi_note + midi_note_increment
                            found_pipes = 0
                            for HW_pipe_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_rank_dic, 'Pipe_SoundEngine01', TO_CHILD):
                                # scan the children Pipe_SoundEngine01 of the Rank
                                pipe_midi_note =  myint(self.HW_ODF_get_attribute_value(HW_pipe_dic, 'NormalMIDINoteNumber'), 60)
                                div_midi_note = pipe_midi_note - midi_note_increment

                                if first_mapped_pipe_midi_note <= pipe_midi_note < first_mapped_pipe_midi_note + nb_of_mapped_div_inputs:
                                    # the current pipe MIDI note is inside the mapped pipes MIDI notes range with increment
                                    if HW_action_effect_code == 2:
                                        noise_kind = 'attack'
                                    else:
                                        noise_kind = 'release'
                                    self.GO_ODF_build_Manual_noise_add_pipe(HW_pipes_descr_dic, HW_pipe_dic, div_midi_note, noise_kind)

                                    found_pipes += 1
                                    if found_pipes == nb_of_mapped_div_inputs:
                                        # all mapped pipes have been found, exit the rank pipes scan
                                        break

            # recover key noises if they are defined under HW Division / DivisionInput / Switch objects
            for HW_div_input_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_division_dic, 'DivisionInput', TO_CHILD):
                # scan the children DivisionInput of the given HW Division
                midi_note = myint(self.HW_ODF_get_attribute_value(HW_div_input_dic, 'NormalMIDINoteNumber'), 60)
                HW_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_div_input_dic, 'SwitchID')
                if HW_switch_dic != None:
                    for HW_pipe_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_switch_dic, 'Pipe_SoundEngine01', TO_CHILD):
                        # scan the children Pipe_SoundEngine01 of the current Switch
                        for HW_pipe_layer_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD):
                            # scan the children Pipe_SoundEngine01_Layer of the current Pipe_SoundEngine01
                            if self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_layer_dic, 'Pipe_SoundEngine01_ReleaseSample', TO_CHILD, FIRST_ONE) != None:
                                # the current pipe layer contains a release sample, it is a key release noise
                                self.GO_ODF_build_Manual_noise_add_pipe(HW_pipes_descr_dic, HW_pipe_dic, midi_note, 'release')
                            else:
                                # key press noise
                                self.GO_ODF_build_Manual_noise_add_pipe(HW_pipes_descr_dic, HW_pipe_dic, midi_note, 'attack')

                    # check if the current Switch has an inverting SwitchLinkage toward a Pipe_SoundEngine01
                    for HW_switch_link_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_switch_dic, 'SwitchLinkage', TO_CHILD):
                        # scan the children SwitchLinkage of the current Switch
                        if self.HW_ODF_get_attribute_value(HW_switch_link_dic, 'SourceSwitchLinkIfEngaged') == 'N':
                            # it is an inverting switch linkage, to react to the key release by an attack sample in Hauptwerk
                            HW_switch2_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_switch_link_dic, 'DestSwitchID')
                            if HW_switch2_dic != None:
                                HW_pipe_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_switch2_dic, 'Pipe_SoundEngine01', TO_CHILD, FIRST_ONE)
                                if HW_pipe_dic != None:
                                    # key release noise
                                    self.GO_ODF_build_Manual_noise_add_pipe(HW_pipes_descr_dic, HW_pipe_dic, midi_note, 'release')

        # recover key noises if they are defined under HW Keyboard / KeyboardKey / Switch objects
        for HW_key_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_keyboard_dic, 'KeyboardKey', TO_CHILD):
            # scan the children KeyboardKey of the current HW Keyboard
            midi_note = myint(self.HW_ODF_get_attribute_value(HW_key_dic, 'NormalMIDINoteNumber'), 60)
            HW_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_key_dic, 'SwitchID')
            if HW_switch_dic != None:
                for HW_pipe_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_switch_dic, 'Pipe_SoundEngine01', TO_CHILD):
                    # scan the children Pipe_SoundEngine01 of the current Switch
                    for HW_pipe_layer_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD):
                        # scan the children Pipe_SoundEngine01_Layer of the current Pipe_SoundEngine01
                        if self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_layer_dic, 'Pipe_SoundEngine01_ReleaseSample', TO_CHILD, FIRST_ONE) != None:
                            # the current pipe layer contains a release sample, it is a key release noise
                            self.GO_ODF_build_Manual_noise_add_pipe(HW_pipes_descr_dic, HW_pipe_dic, midi_note, 'release')
                        else:
                            # key press noise
                            self.GO_ODF_build_Manual_noise_add_pipe(HW_pipes_descr_dic, HW_pipe_dic, midi_note, 'attack')

                # check if the current Switch has a SwitchLinkage toward a Pipe_SoundEngine01
                for HW_switch_link_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_switch_dic, 'SwitchLinkage', TO_CHILD, sorted_by='ID'):
                    # scan the children SwitchLinkage of the current Switch by ID order to have always the same audio channels order in the recovered pipes
                    HW_engage_action_code = myint(self.HW_ODF_get_attribute_value(HW_switch_link_dic, 'EngageLinkActionCode'))
                    HW_disengage_action_code = myint(self.HW_ODF_get_attribute_value(HW_switch_link_dic, 'DisengageLinkActionCode'))
                    if ((HW_engage_action_code == 4 and HW_disengage_action_code == 7) or
                        (HW_engage_action_code == 7 and HW_disengage_action_code == 4)):
                        # it is a key press/release noise switch linkage
                        HW_switch2_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_switch_link_dic, 'DestSwitchID')
                        HW_pipe_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_switch2_dic, 'Pipe_SoundEngine01', TO_CHILD, FIRST_ONE)
                        if HW_pipe_dic != None:
                            if HW_engage_action_code == 4:
                                # key press noise
                                self.GO_ODF_build_Manual_noise_add_pipe(HW_pipes_descr_dic, HW_pipe_dic, midi_note, 'attack')
                            else:
                                # key release noise
                                self.GO_ODF_build_Manual_noise_add_pipe(HW_pipes_descr_dic, HW_pipe_dic, midi_note, 'release')

        # build GO Stops for keys noises, one Stop for each audio channel
        for audio_channel_id, audio_channel_dic in HW_pipes_descr_dic.items():

            HW_attack_pipes_dic_list = []
            HW_release_pipes_dic_list = []

            for key_midi_note_nb in range(GO_manual_first_midi_note_nb, GO_manual_first_midi_note_nb + GO_manual_keys_nb):
                # scan the accessible MIDI notes of the manual to build the attack and release pipes lists
                if key_midi_note_nb in audio_channel_dic.keys():
                    HW_attack_pipes_dic_list.append(audio_channel_dic[key_midi_note_nb][0])
                    HW_release_pipes_dic_list.append(audio_channel_dic[key_midi_note_nb][1])
                else:
                    HW_attack_pipes_dic_list.append(None)
                    HW_release_pipes_dic_list.append(None)

            if LOG_HW2GO_keys_noise: print(f"Audio channel {audio_channel_id} with {len(HW_attack_pipes_dic_list)} attacks and {len(HW_release_pipes_dic_list)} releases")

            GO_attr_dic = self.GO_ODF_build_Stop_noises_attributes(HW_attack_pipes_dic_list, HW_release_pipes_dic_list)

            if GO_attr_dic != None:
                # create a Stop object in the GO ODF
                GO_stop_uid = self.GO_ODF_get_free_uid_in_manual(GO_manual_uid, 'Stop')
                GO_stop_dic = self.GO_odf_dic[GO_stop_uid] = {}

                GO_stop_dic['Name'] = f'{GO_manual_dic["Name"]} keys noise'

                # set it engaged by default and not reset by the general cancel button
                GO_stop_dic['DefaultToEngaged'] = 'Y'
                GO_stop_dic['GCState'] = '-1'

                # copy in new GO object the GO object attributes built before
                for attr, value in GO_attr_dic.items():
                    GO_stop_dic[attr] = value

                # add the GO stop to the GO Manual to which it belongs
                self.GO_ODF_child_add(GO_manual_uid, GO_stop_uid)

                if LOG_HW2GO_keys_noise: print(f"Keys noises {GO_stop_uid} built ")

                # indicate in the HW rank of the first defined attack and release pipes the UID of the GO stop
                for HW_pipe_dic in HW_attack_pipes_dic_list:
                    if HW_pipe_dic != None:
                        HW_rank_dic = self.HW_ODF_get_object_dic_by_ref_id('Rank', HW_pipe_dic, 'RankID')
                        HW_rank_dic['_GO_uid'] = GO_stop_uid
                        break
                for HW_pipe_dic in HW_release_pipes_dic_list:
                    if HW_pipe_dic != None:
                        HW_rank_dic = self.HW_ODF_get_object_dic_by_ref_id('Rank', HW_pipe_dic, 'RankID')
                        HW_rank_dic['_GO_uid'] = GO_stop_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Manual_noise_add_pipe(self, HW_pipes_descr_dic, HW_pipe_dic, midi_note, noise_kind):
        # add the given HW pipe in the given noises pipes description dictionary, for the given MIDI note and noise kind ('attack' or 'release')

        # double check the noise kind to take into consideration particular use cases
        noise_kind = self.HW_ODF_pipe_noise_kind_check(HW_pipe_dic, noise_kind)

        audio_channel_id = self.HW_ODF_get_pipe_audio_channel_id(HW_pipe_dic)
        if audio_channel_id not in HW_pipes_descr_dic.keys():
            # create an entry in the dictionary for the audio channel ID, with empty dictionary inside
            HW_pipes_descr_dic[audio_channel_id] = {}
        if midi_note not in HW_pipes_descr_dic[audio_channel_id].keys():
            # create an entry in the dictionary for the audio channel ID and MIDI note, with empty attack and release pipes placeholders
            HW_pipes_descr_dic[audio_channel_id][midi_note] = [None, None]


        # store the pipe at the correct placeholder in the pipes description list
        if noise_kind == 'attack':
            HW_pipes_descr_dic[audio_channel_id][midi_note][0] = HW_pipe_dic
            if LOG_HW2GO_keys_noise: print(f"   audio channel {audio_channel_id} key press noise {HW_pipe_dic['_uid']} MIDI note {midi_note}")
        else:
            HW_pipes_descr_dic[audio_channel_id][midi_note][1] = HW_pipe_dic
            if LOG_HW2GO_keys_noise: print(f"   audio channel {audio_channel_id} key release noise {HW_pipe_dic['_uid']} MIDI note {midi_note}")

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Drawstop_controlled_object(self, HW_object_dic):
        # build the GO drawstop object (pipes Stop, Coupler, Tremulant, General/Division/SET) and the GO switches which are controlling it
        # from the given HW object (respectively Stop, KeyAction, Tremulant, Combination/_General) and in the GO Manual corresponding to this object
        # return the UID of the built object

        # used HW objects :
        #   coupler :
        #     KeyAction (see GO_ODF_build_Coupler_attributes)
        #   tremulant :
        #     Tremulant (see GO_ODF_build_Tremulant_attributes)
        #   pipes ranks stop :
        #     Stop (see GO_ODF_build_Stop_noises_attributes)
        #   alternate rank control switch :
        #     Stop C> StopRank(s) P> Switch
        #   combination setter :
        #     Combination C> CombinationElement C> Switch C> Stop/Coupler/Tremulant
        #     _General C> Switch (master capture switch)

        if HW_object_dic == None:
            return None

        HW_object_type = HW_object_dic['_type']

        if HW_object_type not in ('Stop', 'KeyAction', 'Tremulant', 'Combination', '_General'):
            print(f"ERROR wrong HW object type {HW_object_dic['_uid']} given to GO_ODF_build_Drawstop_controlled_object")
            return None

        if LOG_HW2GO_drawstop: print(f"{HW_object_dic['_uid']} '{HW_object_dic['Name']}'")

        if HW_object_dic['_GO_uid'] != '' and HW_object_dic != self.HW_general_dic:
            # the given HW object has been already converted to a GO object and it is not the _General object, return its UID
            if LOG_HW2GO_drawstop: print(f"     already converted to GO {HW_object_dic['_GO_uid']}")
            return HW_object_dic['_GO_uid']

        #-----------------------------------------------------------------------
        # build the attributes of the GO object corresponding to the given HW object
        GO_attr_dic = None
        trem_HW_switch_dic = trem_GO_switch_uid = None
        trem_def_method = None  # tremmed samples definition method : None (no tremmed samples) or 'alt_rank'  (tremmed samples are in an alternate rank)
                                #                                                               or 'sec_layer' (tremmed samples are in the second pipe layer)

        if HW_object_type == 'Stop':
            if self.trem_samples_mode != None:
                # tremmed samples have to be converted

                # look if the given HW Stop has tremmed samples defined in an alternate rank
                # using StopRank attribute AlternateRankID
                # (this tremmed samples definition method is used by Sonus Paradisi)
                for HW_stop_rank_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_object_dic, 'StopRank', TO_CHILD):
                    # scan the HW StopRank objects child of the HW Stop
                    if myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'AlternateRankID'), 0) != 0:
                        # the current StopRank has an alternate rank defined
                        # recover the HW Switch permitting to switch to the alternate rank
                        trem_HW_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_stop_rank_dic, 'SwitchIDToSwitchToAlternateRank')
                        trem_def_method = 'alt_rank'
                        if LOG_HW2GO_drawstop: print(f"     {HW_object_dic['_uid']} has alternate ranks controlled by {trem_HW_switch_dic['_uid']}")
                        break

                # look if the given HW Stop has tremmed samples defined in an alternate pipes layer (the second layer normally)
                # using Stop attribute Hint_PrimaryAssociatedRankID and Rank attribute SoundEngine01_Layer2Desc
                # (this tremmed samples definition method is used by Piotr Grabowski)
                if trem_HW_switch_dic == None:
                    # the HW Stop has no tremmed samples defined in alternate ranks
                    HW_rank_dic = self.HW_ODF_get_object_dic_by_ref_id('Rank', HW_object_dic, 'Hint_PrimaryAssociatedRankID')
                    if mystr(self.HW_ODF_get_attribute_value(HW_rank_dic, 'SoundEngine01_Layer2Desc'), '') != '':
                        # a second pipes layer is defined in the primary associated rank
                        # recover the HW Switch permitting to switch to the second pipe layer


                        for HW_pipe_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_rank_dic, 'Pipe_SoundEngine01', TO_CHILD):
                            # scan the pipes of the HW Rank to find the first one having a second layer
                            HW_pipe_layers_dic_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD, sorted_by='ID')
                            if len(HW_pipe_layers_dic_list) > 1:
                                # get the second pipe layer of the current pipe
                                HW_pipe_layer_dic = HW_pipe_layers_dic_list[1]
                                HW_scaling_cc_dic = self.HW_ODF_get_object_dic_by_ref_id('ContinuousControl', HW_pipe_layer_dic, 'AmpLvl_ScalingContinuousControlID')
                                # search for the first latching HW Switch defined in the parents of the second HW Pipe layer
                                ctrl_dic_lists = []
                                self.HW_ODF_get_controlling_continuous_controls(HW_scaling_cc_dic, ctrl_dic_lists)
                                for branch_nb, ctrl_dic_list in enumerate(ctrl_dic_lists):
                                    # scan the various branches of controlling objects to get the first latching HW Switch
                                    for HW_obj_dic in ctrl_dic_list:
                                        if HW_obj_dic['_type'] == 'Switch':
                                            trem_HW_switch_dic = HW_obj_dic
                                            trem_def_method = 'sec_layer'
                                            break
                                    if trem_HW_switch_dic != None:
                                        break
                                if trem_HW_switch_dic != None:
                                    break

            GO_object_type = 'Stop'
            GO_attr_dic = self.GO_ODF_build_Stop_pipes_attributes(HW_object_dic, self.trem_samples_mode, trem_def_method)

        elif HW_object_type == 'KeyAction':
            GO_object_type = 'Coupler'
            GO_attr_dic = self.GO_ODF_build_Coupler_attributes(HW_object_dic)

        elif HW_object_type == 'Tremulant':
            GO_object_type = 'Tremulant'
            GO_attr_dic = self.GO_ODF_build_Tremulant_attributes(HW_object_dic)

        elif HW_object_type in ('Combination', '_General'):
            GO_attr_dic = self.GO_ODF_build_Combination_attributes(HW_object_dic)
            GO_object_type = 'Combination'

        if GO_attr_dic == None:
            if LOG_HW2GO_drawstop: print(f"     ====> SKIPPED : GO {GO_object_type} attributes NOT built for HW {HW_object_dic['_uid']}")
            return None

        # recover the GO manual UID found in the function called above in case it has been defined
        GO_manual_uid = mydickey(GO_attr_dic, '_GO_manual_uid')
        if GO_manual_uid == None and (GO_object_type in ('Stop', 'Coupler') or (GO_object_type == 'Combination' and GO_attr_dic['_comb_type'] == 'divisional')):
            if LOG_HW2GO_drawstop: print(f"     ====> SKIPPED : no GO manual found to match with HW {HW_object_dic['_uid']}")
            return None

        #-----------------------------------------------------------------------
        # define the UID of the GO object to build

        GO_object_uid = None

        if GO_object_type == 'Combination':
            if GO_attr_dic['_comb_type'] == 'general':
                # define the UID of the GO General to build
                GO_object_uid = 'General' + str(self.GO_organ_dic['NumberOfGenerals'] + 1).zfill(3)
            elif GO_attr_dic['_comb_type'] == 'divisional':
                # define the UID of the GO Divisional to build in the GO manual UID
                GO_object_uid = self.GO_ODF_get_free_uid_in_manual(GO_manual_uid, 'Divisional')
            # else in case of Set and GC, no GO object has to be created, it is the PanelElement which manages it

        elif GO_object_type == 'Tremulant':
            # define the UID of the GO Tremulant to build
            GO_object_uid = 'Tremulant' + str(self.GO_organ_dic['NumberOfTremulants'] + 1).zfill(3)

        else:
            # define the UID of the GO object (Coupler or Stop) to build
            GO_object_uid = self.GO_ODF_get_free_uid_in_manual(GO_manual_uid, GO_object_type)

        if LOG_HW2GO_drawstop: print(f"     GO {GO_object_type} attributes built with success for HW {HW_object_dic['_uid']}, to be placed in GO object {GO_object_uid} and manual {GO_manual_uid}")

        #-----------------------------------------------------------------------
        # build the GO objects (Switches, Panel Elements) controlling the given HW object

        if GO_object_type == 'Stop' and trem_HW_switch_dic != None:
            # build the GO switch permitting to activate the tremmed samples if not already built
            if LOG_HW2GO_switch: print(f"Building objects controlling {trem_HW_switch_dic['_uid']} '{trem_HW_switch_dic['Name']}' for tremmed samples activation")
            settings_dic = {'Type': 'Switch', 'PanElemObject': None, 'GCState': -1, 'Manual': None, 'NbControlling': 0}
            # gc_state=-1 to not disengage the GO switch when general cancel is pressed, no GO manual UID to provide
            trem_GO_switch_uid = self.GO_ODF_build_Switch_controlling_objects(trem_HW_switch_dic, settings_dic)
            if LOG_HW2GO_drawstop: print(f"       Tremmed samples activated by GO {trem_GO_switch_uid}")


        if GO_object_type != 'Combination': # type is Stop, Coupler or Tremulant
            settings_dic = {'Type': 'Switch', 'PanElemObject': None, 'GCState': 0, 'Manual': GO_manual_uid, 'CtrlObject': GO_object_uid, 'NbControlling': 0}
            # gc_state=0 to disengage the GO switch when general cancel is pressed

            if mydickey(GO_attr_dic, 'DefaultToEngaged') == 'Y':
                # if the controlled object is engaged by default (like a Coupler not controlled by a switch), consider that it is a controlling object
                settings_dic['NbControlling'] += 1

        else:  # GO_object_type == 'Combination'
            if GO_attr_dic['_comb_type'] == 'general' and GO_object_uid != None:
                settings_dic = {'Type': 'General', 'PanElemObject': GO_object_uid, 'GCState': 0, 'Manual': None, 'NbControlling': 0}

            elif GO_attr_dic['_comb_type'] == 'divisional' and GO_object_uid != None and GO_manual_uid != None:
                settings_dic = {'Type': 'Divisional', 'PanElemObject': GO_object_uid, 'GCState': 0, 'Manual': GO_manual_uid, 'NbControlling': 0}

            elif GO_attr_dic['_comb_type'] == 'set':
                settings_dic = {'Type': 'Set', 'PanElemObject': 'Set', 'GCState': 0, 'Manual': None, 'NbControlling': 0}

            elif GO_attr_dic['_comb_type'] == 'gc':
                settings_dic = {'Type': 'GC', 'PanElemObject': 'GC', 'GCState': 0, 'Manual': None, 'NbControlling': 0}

            else:
                settings_dic = None  # dictionary where are given the settings to build the controlling PanelElement or Switch objects

        if settings_dic != None:
            GO_switch_uid = self.GO_ODF_build_Switch_controlling_objects(HW_object_dic, settings_dic)
            # GO_switch_uid is the switch to link to the GO object corresponding to the given HW object
        else:
            GO_switch_uid = None

        if settings_dic == None or settings_dic['NbControlling'] == 0:
            # there is none controlling object, abort the GO object building
            # (a PanelElement can have been created all the same in GO_ODF_build_Switch_controlling_objects in case of Set or GC object)
            if LOG_HW2GO_drawstop: print(f"     ====> ABORTED : HW {HW_object_dic['_uid']} is controlled by none object")
            return None

        #-----------------------------------------------------------------------
        # build the GO object corresponding to the given HW object and which the UID has been defined earlier

        if GO_object_uid != None:
            # build the object in the GO ODF dictionary
            GO_object_dic = self.GO_odf_dic[GO_object_uid] = {}

            # copy in new GO object the GO object attributes built before
            for attr, value in GO_attr_dic.items():
                GO_object_dic[attr] = value

            # write in the HW object the corresponding GO object
            HW_object_dic['_GO_uid'] = GO_object_uid

            # write in the GO object the UID of the manual to which it is attached
            if GO_manual_uid != None:
                GO_object_dic['_GO_manual_uid'] = GO_manual_uid

            if GO_object_type != 'Combination':
                # it is not a combination object (which is not controlled by a switch)
                if GO_switch_uid != None:
                    # add in the GO object its controlling GO switch
                    GO_object_dic['Function'] = 'And'
                    GO_object_dic['SwitchCount'] = 1
                    GO_object_dic['Switch001'] = GO_switch_uid[-3:]
                else:
                    # the GO object is not controlled by a switch, engage it by default with GCState=-1 (not disengaged by GC push)
                    GO_object_dic['DefaultToEngaged'] = 'Y'
                    GO_object_dic['GCState'] = -1

                if GO_object_type == 'Tremulant':
                    # increase the NumberOfTremulants counter in the Organ object
                    self.GO_organ_dic['NumberOfTremulants'] += 1
                    # add the GO Tremulant to the GO WindchestGroups on which it has an effect
                    for GO_windchest_uid in GO_attr_dic['_GO_windchests_uid_list']:
                        self.GO_ODF_child_add(GO_windchest_uid, GO_object_uid)
                    # add the GO Tremulant to the GO Manuals to which it belongs
                    for GO_man_uid in GO_attr_dic['_GO_manuals_uid_list']:
                        self.GO_ODF_child_add(GO_man_uid, GO_object_uid)
                else:
                    # add a reference to the GO object (Stop or Coupler) in the GO Manual to which it belongs
                    self.GO_ODF_child_add(GO_manual_uid, GO_object_uid)

            elif GO_attr_dic['_comb_type'] == 'divisional':
                # add a reference to the GO Divisional in the GO Manual to which it belongs
                self.GO_ODF_child_add(GO_manual_uid, GO_object_uid)

            elif GO_attr_dic['_comb_type'] == 'general':
                # increase the NumberOfGenerals counter in the Organ object
                self.GO_organ_dic['NumberOfGenerals'] += 1

            # add in the created object the reference of its controlling GO Switch UID if any
            GO_object_dic['_switch'] = GO_switch_uid

            if GO_switch_uid != None:
                if LOG_HW2GO_drawstop: print(f"     GO {GO_object_uid} '{GO_object_dic['Name']}' built, controlled by {GO_switch_uid}")
            else:
                if LOG_HW2GO_drawstop: print(f"     GO {GO_object_uid} '{GO_object_dic['Name']}' built, controlled by PanelElement only (none switch)")
        else:
            if LOG_HW2GO_drawstop: print(f"     GO {GO_object_type} object built without GO functional object")

        #-----------------------------------------------------------------------
        # build if needed a GO wave based Tremulant controlled by the alternate rank switch
        if GO_object_type == 'Stop' and trem_GO_switch_uid != None:
            # the given object is a stop having an alternate rank switch to enable tremmed wav samples

            if '_GO_wav_trem_uid' not in trem_HW_switch_dic.keys():
                # a GO wav tremulant is not already created, create a new GO wave Tremulant object
                self.GO_organ_dic['NumberOfTremulants'] += 1
                GO_tremulant_uid = 'Tremulant' + str(self.GO_organ_dic['NumberOfTremulants']).zfill(3)
                GO_object_dic = self.GO_odf_dic[GO_tremulant_uid] = {}

                GO_object_dic['Name'] = trem_HW_switch_dic['Name']
                GO_object_dic['TremulantType'] = 'Wave'

                # add in the tremulant its controlling switch
                GO_object_dic['Function'] = 'And'
                GO_object_dic['SwitchCount'] = 1
                GO_object_dic['Switch001'] = trem_GO_switch_uid[-3:]

                trem_HW_switch_dic['_GO_wav_trem_uid'] = GO_tremulant_uid
                trem_HW_switch_dic['_GO_manual_uid'] = GO_manual_uid

                if LOG_HW2GO_drawstop: print(f"     GO Wave Tremulant {GO_tremulant_uid} created, controlled by {trem_HW_switch_dic['_GO_uid']}")
            else:
                GO_tremulant_uid = trem_HW_switch_dic['_GO_wav_trem_uid']
                GO_object_dic = self.GO_odf_dic[GO_tremulant_uid]
                if LOG_HW2GO_drawstop: print(f"     GO Wave Tremulant {GO_tremulant_uid} reused, controlled by {trem_HW_switch_dic['_GO_uid']}")

            # add the GO Tremulant to the GO WindchestGroups on which it has an effect (which are listed in the associated HW Stop)
            for GO_windchest_uid in HW_object_dic['_GO_windchests_uid_list']:
                if self.GO_ODF_child_add(GO_windchest_uid, GO_tremulant_uid):
                    if LOG_HW2GO_drawstop: print(f'       {GO_tremulant_uid} added to {GO_windchest_uid}')

            # add the GO Tremulant to the GO Manual to which it belongs
            if self.GO_ODF_child_add(GO_manual_uid, GO_tremulant_uid):
                if LOG_HW2GO_drawstop: print(f'       {GO_tremulant_uid} added to {GO_manual_uid}')

            if LOG_HW2GO_drawstop: print(f"Wave based tremulant {GO_tremulant_uid} '{GO_object_dic['Name']}' for {HW_object_dic['_uid']} acting on {sorted(HW_object_dic['_GO_windchests_uid_list'])} and {GO_manual_uid}")

        return GO_object_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Switch_controlling_objects(self, HW_object_dic, settings_dic, rec_level=0):
        # recursive function to build the GO Switches and/or PanelElements controlling the given HW object (Switch, SwitchLinkage, Stop, Tremulant, KeyAction, Combination, _General)
        # the following keys must be given in settings_dic :
        #   'Type' : GO object type that the PanelElement will have to control : 'Switch', 'General', 'Divisional', 'Set', 'GC'
        #   'PanElemObject' : UID of the GO object controlled by the PanelElement in case of a combination. In case of a switch type it is defined in this function
        #   'Manual' : UID of the GO Manual associated to the controlled object if any
        #   'GCState' : value to give to the GCState attribute of the controlling GO Switch (needed only if Type is Switch)
        #   'NbControlling' : value to set at 0, will be returned with the number of found controlling objects

        # the given GCState value is set only in the visible switches having no input switch
        # the visible switches are added to the GO Manual if given
        # rec_level is for function logging, it gives the recursivity level of the function
        # return the UID of the GO Switch corresponding to the given HW object or controlling it directly, or None if no GO Switch has been built

        if LOG_HW2GO_switch: rlspace = '     ' + '  ' * rec_level

        if HW_object_dic == None:
            return None

        HW_object_type = HW_object_dic['_type']

        GO_switch_uid = None     # returned at the end of the function

        if HW_object_type == 'Switch':
            # read if the given HW switch is clickable (i.e. is visible and one can click on it)
            is_clickable = mystr(self.HW_ODF_get_attribute_value(HW_object_dic, 'Clickable'), 'N') == 'Y'
            is_visible = myint(self.HW_ODF_get_attribute_value(HW_object_dic, 'Disp_ImageSetInstanceID'), 0) != 0

            if HW_object_dic['_GO_uid'] != '':
                # the given HW Switch has been already converted to a GO object, recover its UID
                GO_switch_uid = HW_object_dic['_GO_uid']
                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} already converted to {HW_object_dic['_GO_uid']}")

                # it has been converted to GO, so it is a controlling switch
                settings_dic['NbControlling'] += 1

                if is_clickable and settings_dic['Type'] == 'Switch' and settings_dic['Manual'] != None:
                    # add the clickable GO switch to the given GO manual for which it is acting
                    self.GO_ODF_child_add(settings_dic['Manual'], GO_switch_uid)
                    HW_object_dic['_GO_manual_uid'] = settings_dic['Manual']

            else:
                # get properties of the given HW switch
                is_default_engaged = mystr(self.HW_ODF_get_attribute_value(HW_object_dic, 'DefaultToEngaged'), 'N') == 'Y'
                is_latching = mystr(self.HW_ODF_get_attribute_value(HW_object_dic, 'Latching'), 'N') == 'Y'
                switch_name = HW_object_dic['Name']
                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} '{switch_name}' (is clickable {is_clickable}, is default engaged {is_default_engaged}, is latching {is_latching})")

                GO_input_switches_uid_list = []  # list to collect the GO input switches controlling the given switch
                GO_comb_switch_uid = None        # UID of the GO Switch created if necessary to apply the action of a combination element on a GO Switch having already other imputs

                if '_swbuilt' not in HW_object_dic.keys():
                    # the given HW Switch has not been already managed by this function, add in it a key to keep a trace
                    HW_object_dic['_swbuilt'] = ''

                    for HW_switchlinkage_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_object_dic, 'SwitchLinkage', TO_PARENT):
                        # scan the parent SwitchLinkage objects of the given HW switch to recover the GO Switch that each one is returning
                        GO_input_switch_uid = self.GO_ODF_build_Switch_controlling_objects(HW_switchlinkage_dic, settings_dic, rec_level+1)
                        if GO_input_switch_uid != None and GO_input_switch_uid not in GO_input_switches_uid_list:
                            GO_input_switches_uid_list.append(GO_input_switch_uid)

                    if len(GO_input_switches_uid_list) > 0:
                        # the given switch is controlled by at least one input switch

                        if len(self.HW_ODF_get_linked_objects_dic_by_type(HW_object_dic, 'CombinationElement', TO_PARENT)):
                            # the given switch is controlled as well by at least one combination element
                            # create a GO switch which will be controlled by the combination elements and will be an input of the GO Switch in addition to its the other inputs

                            # recover the combination type of the HW Combination to which is linked the first CombinationElement controlling the given HW Switch
                            HW_comb_elem_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_object_dic, 'CombinationElement', TO_PARENT, FIRST_ONE)
                            HW_combination_dic = self.HW_ODF_get_object_dic_by_ref_id('Combination', HW_comb_elem_dic, 'CombinationID')
                            HW_comb_type_code = myint(self.HW_ODF_get_attribute_value(HW_combination_dic, 'CombinationTypeCode'), 0)
                            if HW_comb_type_code not in (0, 1) :  # 1 = master capture combination has to be ignored
                                # create the GO Switch permitting to apply the effect of the combination elements in addition to the other inputs of the GO Switch (with OR function)
                                GO_comb_switch_uid = self.GO_ODF_build_Switch_object('Combination action', store_in_comb=False)
                                if LOG_HW2GO_switch: print(f"{rlspace}{GO_comb_switch_uid} controlled by CombinationElements in addition to {GO_input_switches_uid_list}")
                                GO_input_switches_uid_list.append(GO_comb_switch_uid)
                                if settings_dic['Manual'] != None:
                                    # add the GO combination switch to the given GO manual for which it is acting
                                    self.GO_ODF_child_add(settings_dic['Manual'], GO_comb_switch_uid)

                        if is_clickable and mydickey(settings_dic, 'sw_loop_dic') == None:
                            # the given switch is controlled as well by a clickable button and it is not in a switches loop
                            # create a GO switch which will be controlled by the clickable button (PanelElement) and will be an input of the GO Switch in addition to its the other inputs
                            GO_click_switch_uid = self.GO_ODF_build_Switch_object(switch_name, gc_state=0, store_in_comb=False)
                            if LOG_HW2GO_switch: print(f"{rlspace}{GO_click_switch_uid} built to be controlled by a clickable button in addition to {GO_input_switches_uid_list}")
                            GO_input_switches_uid_list.append(GO_click_switch_uid)

                            if settings_dic['Manual'] != None:
                                # add the GO switch to the given GO manual for which it is acting
                                self.GO_ODF_child_add(settings_dic['Manual'], GO_click_switch_uid)
                                HW_object_dic['_GO_manual_uid'] = settings_dic['Manual']

                            HW_object_dic['_GO_uid'] = GO_click_switch_uid

                            # build the clickable button (a PanelElement)
                            self.GO_ODF_build_PanelElement_button_object(HW_object_dic, 'Switch', GO_click_switch_uid)
                            settings_dic['NbControlling'] += 1

                            # the GO Switch to create then is a OR gate (with GO_input_switches_uid_list inputs) and not clickable
                            is_clickable = False
                            switch_name = 'OR gate'

                    if HW_object_dic == mydickey(settings_dic, 'sw_loop_dic'):
                        settings_dic['sw_loop_dic'] = None
                        if LOG_HW2GO_switch: print(f"{rlspace}exiting switches LOOP with {HW_object_dic['_uid']}")

                else:
                    # the given HW Switch has been already managed, it is closing a switches loop
                    # ignore its controlling SwitchLinkage and consider it is at the top of a controlling branch
                    settings_dic['sw_loop_dic'] = HW_object_dic
                    if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} already managed, it is closing a switches LOOP")

                if settings_dic['Type'] == 'Switch':
                    # the given controlled object is controlled by switches, build the GO switch corresponding to the current HW Switch if needed
                    if len(GO_input_switches_uid_list) == 0:
                        # the given HW Switch has no switch controlling it, it is as the top of a controlling branch
                        if is_default_engaged or is_clickable or mydickey(settings_dic, 'sw_loop_dic') != None:
                            # the given HW switch is engaged by default or is clickable or is closing a switches loop, convert it in a GO Switch without inputs
                            # a non clickable switch engaged by default must not be disengaged by the GC button, so set GC_state=-1 in this case
                            GO_switch_uid = self.GO_ODF_build_Switch_object(switch_name, is_default_engaged=is_default_engaged,
                                                                            gc_state=-1 if (is_default_engaged and not is_clickable) else settings_dic['GCState'])
                            if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} converted to {GO_switch_uid} without inputs, default engaged = {is_default_engaged}, clickable = {is_clickable}, looping switch={self.HW_DIC2UID(mydickey(settings_dic, 'sw_loop_dic'))}")
                        else:
                            # the given HW Switch has not to be converted in GO
                            GO_switch_uid = None
                            if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} not converted to a GO Switch, none inputs and not default engaged or clickable or closing a loop")

                    elif len(GO_input_switches_uid_list) == 1:
                        # the given HW Switch has only one input, return the UID of its controlling switch
                        GO_switch_uid = GO_input_switches_uid_list[0]
                        if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} forwarding input {GO_switch_uid}")

                    else:
                        # the given HW Switch has several inputs, create a GO Switch applying a logical OR between the input switches
                        GO_switch_uid = self.GO_ODF_build_Switch_object(switch_name, 'Or', GO_input_switches_uid_list, gc_state=settings_dic['GCState'])
                        if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} converted to {GO_switch_uid}, OR function between inputs {GO_input_switches_uid_list}")

                    # indicate which GO Switch UID the PanelElement will control
                    settings_dic['PanElemObject'] = GO_switch_uid

                    if is_clickable and settings_dic['Manual'] != None:
                        # add the clickable GO switch to the given GO manual for which it is acting
                        self.GO_ODF_child_add(settings_dic['Manual'], GO_switch_uid)
                        HW_object_dic['_GO_manual_uid'] = settings_dic['Manual']

                    if HW_object_dic['_GO_uid'] == '':
                        HW_object_dic['_GO_uid'] = GO_switch_uid

                else:
                    if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} not converted to a GO switch as not needed for a {settings_dic['Type']}")
                    # mark the object as not converted to GO
                    HW_object_dic['_GO_uid'] = None

                # create a controlling GO PanelElement button if the current HW switch is clickable (it means it is has a graphical interface)
                if is_visible:  #is_clickable:
                    self.GO_ODF_build_PanelElement_button_object(HW_object_dic, settings_dic['Type'], settings_dic['PanElemObject'], settings_dic['Manual'])
                    settings_dic['NbControlling'] += 1
                elif is_default_engaged:
                    # if not clickable but engaged by default, consider it as a controlling object
                    settings_dic['NbControlling'] += 1

                if mydickey(settings_dic, 'sw_loop_dic') == None:
                    # manage the case where the current HW Switch is controlled by HW CombinationElement objects
                    # by adding the current GO Switch to the GO General or Division corresponding to the HW CombinationElement
                    if GO_comb_switch_uid == None:
                        GO_comb_switch_uid = GO_switch_uid
                    for HW_comb_elem_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_object_dic, 'CombinationElement', TO_PARENT):
                        # scan the parent HW CombinationElement objects of the current HW Switch if some are defined
                        HW_combination_dic = self.HW_ODF_get_object_dic_by_ref_id('Combination', HW_comb_elem_dic, 'CombinationID')
                        if HW_combination_dic != None:
                            # recover the UID of the GO combination object (General or Divisional)
                            GO_combination_uid = HW_combination_dic['_GO_uid']
                            if GO_combination_uid != '':
                                # the HW Combination has been converted in a GO combination
                                if GO_combination_uid.startswith('General'):
                                    attr_name = 'SwitchNumber'
                                else:
                                    attr_name = 'Switch'

                                GO_combination_dic = self.GO_odf_dic[GO_combination_uid]
                                GO_combination_dic['NumberOfSwitches'] += 1
                                if mystr(self.HW_ODF_get_attribute_value(HW_comb_elem_dic, 'InitialStoredStateIsEngaged'), 'N') == 'Y':
                                    # if 'Y' the controlled object is engaged by the combination element, if 'N' it is disengaged
                                    GO_combination_dic[attr_name + str(GO_combination_dic['NumberOfSwitches']).zfill(3)] = GO_comb_switch_uid[-3:]
                                    if LOG_HW2GO_switch: print(f"{rlspace}{GO_comb_switch_uid} added in {GO_combination_uid} as engaged, {HW_combination_dic['_uid']} {HW_comb_elem_dic['_uid']} {HW_object_dic['_uid']}")
                                else:
                                    GO_combination_dic[attr_name + str(GO_combination_dic['NumberOfSwitches']).zfill(3)] = '-' + GO_comb_switch_uid[-3:]
                                    if LOG_HW2GO_switch: print(f"{rlspace}{GO_comb_switch_uid} added in {GO_combination_uid} as disengaged, {HW_combination_dic['_uid']} {HW_comb_elem_dic['_uid']} {HW_object_dic['_uid']}")

                                if settings_dic['Manual'] != None:
                                    # add the GO switch to the given GO manual for which it is acting
                                    self.GO_ODF_child_add(settings_dic['Manual'], GO_comb_switch_uid)
                                    HW_object_dic['_GO_manual_uid'] = settings_dic['Manual']


        elif HW_object_type == 'SwitchLinkage':
            if HW_object_dic['_GO_uid'] != '':
                # the given HW SwitchLinkage has been already converted to a GO object, recover its UID
                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} already converted to {HW_object_dic['_GO_uid']}")
                GO_switch_uid = HW_object_dic['_GO_uid']
            else:
                # get properties about the given HW SwitchLinkage
                HW_source_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_object_dic, 'SourceSwitchID')
                HW_dest_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_object_dic, 'DestSwitchID')
                HW_cond_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_object_dic, 'ConditionSwitchID')
                HW_engage_action_code = myint(self.HW_ODF_get_attribute_value(HW_object_dic, 'EngageLinkActionCode'), 1)
                HW_disengage_action_code = myint(self.HW_ODF_get_attribute_value(HW_object_dic, 'DisengageLinkActionCode'), 2)

                if HW_cond_switch_dic != None:
                    if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} (engage action code {HW_engage_action_code}, disengage action code {HW_disengage_action_code}, source {HW_source_switch_dic['_uid']}, destination {HW_dest_switch_dic['_uid']}, condition {HW_cond_switch_dic['_uid']})")
                else:
                    if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} (engage action code {HW_engage_action_code}, disengage action code {HW_disengage_action_code}, source {HW_source_switch_dic['_uid']}, destination {HW_dest_switch_dic['_uid']}, condition None)")

                if HW_engage_action_code == 1 and HW_disengage_action_code == 2:
                    # action code 1 = to engage    the destination switch when the source switch is engaged
                    # action code 2 = to disengage the destination switch when the source switch is disengageds
                    # this is the standard switch linkage action which can be converted in GO

                    # get the GO switch UID coming from the HW source switch
                    if HW_source_switch_dic != None:
                        GO_source_switch_uid = self.GO_ODF_build_Switch_controlling_objects(HW_source_switch_dic, settings_dic, rec_level+1)
                        if GO_source_switch_uid != None and self.HW_ODF_get_attribute_value(HW_object_dic, 'SourceSwitchLinkIfEngaged') == 'N':
                            # the state of the source switch has to be inverted by a logical NOT
                            GO_inv_source_switch_uid = mydickey(HW_source_switch_dic, '_inv_GO_uid')
                            if GO_inv_source_switch_uid != None:
                                # the source switch has been already inverted in a GO switch
                                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} source switch {GO_source_switch_uid} inverted by {GO_inv_source_switch_uid} NOT function REUSED")
                            else:
                                # build a GO Switch to invert the source switch
                                GO_inv_source_switch_uid = self.GO_ODF_build_Switch_object(GO_source_switch_uid + ' invertor', 'Not', [GO_source_switch_uid])
                                HW_source_switch_dic['_inv_GO_uid'] = GO_inv_source_switch_uid
                                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} source switch {GO_source_switch_uid} inverted by {GO_inv_source_switch_uid} NOT function")
                            # the source switch is the inverted source switch
                            GO_source_switch_uid = GO_inv_source_switch_uid
                    else:
                        GO_source_switch_uid = None

                    # get the GO switch UID coming from the HW condition switch if the HW source switch has been converted to a GO switch
                    if HW_cond_switch_dic != None and GO_source_switch_uid != None:
                        GO_cond_switch_uid = self.GO_ODF_build_Switch_controlling_objects(HW_cond_switch_dic, settings_dic, rec_level+1)
                        if GO_cond_switch_uid != None and self.HW_ODF_get_attribute_value(HW_object_dic, 'ConditionSwitchLinkIfEngaged') == 'N':
                            # the state of the condition switch has to be inverted by a logical NOT
                            GO_inv_cond_switch_uid = mydickey(HW_cond_switch_dic, '_inv_GO_uid')
                            if GO_inv_cond_switch_uid != None:
                                # the condition switch has been already inverted in a GO switch
                                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} condition switch {GO_cond_switch_uid} inverted by {GO_inv_cond_switch_uid} NOT function REUSED")
                            else:
                                # build a GO Switch to invert the condition switch
                                GO_inv_cond_switch_uid = self.GO_ODF_build_Switch_object(GO_cond_switch_uid + ' invertor', 'Not', [GO_cond_switch_uid])
                                HW_cond_switch_dic['_inv_GO_uid'] = GO_inv_cond_switch_uid
                                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} condition switch {GO_cond_switch_uid} inverted by {GO_inv_cond_switch_uid} NOT function")
                            # the condition switch is the inverted condition switch
                            GO_cond_switch_uid = GO_inv_cond_switch_uid
                    else:
                        GO_cond_switch_uid = None

                    if settings_dic['Type'] == 'Switch':
                        # the controlled object is controlled by switches, built the needed switches corresponding to the current SwitchLinkage
                        if GO_cond_switch_uid == None:
                            # there is no condition input switch
                            if GO_source_switch_uid == None:
                                # there is neither source input nor condition input switch
                                # there is no switch UID to return
                                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} has no source and no condition input switch, ignored")
                            else:
                                # there is a source input but no condition input switch
                                # return the source switch UID
                                GO_switch_uid = GO_source_switch_uid
                                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} has no condition input switch, forwarding input source {GO_source_switch_uid}")
                        else:
                            # there is both source and condition input switches, a logical AND is applied between source and condition switches
                            # build a GO Switch for that
                            GO_switch_uid = self.GO_ODF_build_Switch_object('AND gate', 'And', [GO_source_switch_uid, GO_cond_switch_uid])
                            if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} converted to {GO_switch_uid}, AND function between source {GO_source_switch_uid} and condition {GO_cond_switch_uid}")
                    else:
                        if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} not converted to a GO switch as not needed for a {settings_dic['Type']}")

                    HW_object_dic['_GO_uid'] = GO_switch_uid

                else:
                    # the HW linkage has not supported engage or disengage action codes
                    if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} has unsupported engage action code {HW_engage_action_code} and disengage action code {HW_disengage_action_code} ====> SKIPPED")

        elif HW_object_type in ('Stop', 'Tremulant'):
            # recover the HW switch controlling the given HW Stop or Tremulant
            HW_cntrl_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_object_dic, 'ControllingSwitchID')
            if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} is controlled by {HW_cntrl_switch_dic['_uid']} : --------------------------------")
            GO_switch_uid = self.GO_ODF_build_Switch_controlling_objects(HW_cntrl_switch_dic, settings_dic, 0)

        elif HW_object_type == 'KeyAction':
            # recover the HW switch controlling the given HW KeyAction
            HW_cntrl_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_object_dic, 'ConditionSwitchID')
            if HW_cntrl_switch_dic != None:  # it can exist HW KeyAction object not controlled by a switch
                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} is controlled by {HW_cntrl_switch_dic['_uid']} : --------------------------------")
                GO_switch_uid = self.GO_ODF_build_Switch_controlling_objects(HW_cntrl_switch_dic, settings_dic, 0)
            else:
                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} is controlled by none switch")

        elif HW_object_type == 'Combination':
            # recover the HW switch controlling the given HW Combination
            HW_cntrl_switch_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_object_dic, 'Switch', TO_PARENT, FIRST_ONE)
            if HW_cntrl_switch_dic != None:  # it can exist HW Combination object not controlled by a switch
                if LOG_HW2GO_switch: print(f"{rlspace}{HW_object_dic['_uid']} is controlled by {HW_cntrl_switch_dic['_uid']} : --------------------------------")
                GO_switch_uid = self.GO_ODF_build_Switch_controlling_objects(HW_cntrl_switch_dic, settings_dic, 0)

        elif HW_object_type == '_General':
            # recover the master capture switch
            HW_cntrl_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_object_dic, 'SpecialObjects_MasterCaptureSwitchID')
            if HW_cntrl_switch_dic != None:
                if LOG_HW2GO_switch: print(f"{rlspace}General master capture is controlled by {HW_cntrl_switch_dic['_GO_uid']} : --------------------------------")
                GO_switch_uid = self.GO_ODF_build_Switch_controlling_objects(HW_cntrl_switch_dic, settings_dic, 0)

        if GO_switch_uid == None and HW_object_type == 'Switch':
            if LOG_HW2GO_switch: print(f"{rlspace}None GO switch built for {HW_object_dic['_uid']}")

        return GO_switch_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Switch_effects_object(self, HW_switch_dic):
        # build the GO objects (noise Stop, or image Switch) on which the given HW Switch as an effect
        # if it can be converted in GO ODF
        # calls the function HW_ODF_get_switch_controlled_objects to know the object controlled by the given HW Switch
        # return the UID of the built switch

        if (self.HW_ODF_get_linked_objects_dic_by_type(HW_switch_dic, 'KeyboardKey', TO_CHILD, FIRST_ONE) != None or
            self.HW_ODF_get_linked_objects_dic_by_type(HW_switch_dic, 'DivisionInput', TO_CHILD, FIRST_ONE) != None):
            # the given HW Switch is controlling a keyboard key or a division input, it is not managed in this function
            return None

        if LOG_HW2GO_drawstop: print(f"{HW_switch_dic['_uid']} '{HW_switch_dic['Name']}'")

        if (self.HW_ODF_get_attribute_value(HW_switch_dic, 'Clickable') != 'Y' and
            self.HW_ODF_get_attribute_value(HW_switch_dic, 'DefaultToEngaged') != 'Y'):
            # the given HW Switch is neither clickable nor engaged by default, it is ignored
            if LOG_HW2GO_drawstop: print("     ====> SKIPPED : neither clickable nor engaged by default")
            return None

        if '_ctrlchck' in HW_switch_dic.keys():
            # the given HW Switch has been already managed by this function (due to a switches loop), it is skipped
            if LOG_HW2GO_drawstop: print("     ====> SKIPPED : already managed previously")
            return None

        if 'MATRIX' in HW_switch_dic['Name'].upper():
            # the given HW Switch is related to matrix feature, not converted in GO, it is skipped
            if LOG_HW2GO_drawstop: print("     ====> SKIPPED : containing MATRIX in its name")
            return None

        # by default the switch is attached to the last Manual object
        # it can be changed later in this function if the switch is controlling an object linked to a manual
        GO_manual_uid = self.last_manual_uid

        #-----------------------------------------------------------------------
        # recover the list of HW objects controlled by the given HW Switch (which will include this HW Switch)
        if LOG_HW2GO_drawstop: print(f"     --- Looking for objects controlled by {HW_switch_dic['_uid']} ---------------------")
        controlled_HW_objects_dic_list = []
        self.HW_ODF_get_switch_controlled_objects(HW_switch_dic, controlled_HW_objects_dic_list)

        if LOG_HW2GO_drawstop: print(f"     --- Analysing objects controlled by {HW_switch_dic['_uid']} ------------------------")
        GO_switch_uid = None
        switch_name = None
        is_default_engaged_switch = False
        is_latching_switch = False
        is_controlling_cc_linkage = False
        is_controlling_wc_linkage = False
        has_condition_effect = False
        has_graphical_interface = False
        has_supported_linkage = False
        has_unsupported_linkage = False
        nb_controlled_GO_objects = 0
        nb_controlled_noise_pipes = 0
        HW_switch_asgn_code = None
        HW_pipes_descr_dic = {}  # dictionary in which are assembled the attack/release noise pipes couples for each audio channel ID
                                 # each item of the dictionary has the format : {channel_id:[attack_pipe_dic, release_pipe_dic]}

        for HW_ctrl_object_dic in controlled_HW_objects_dic_list:
            # scan the controlled HW objects

            HW_ctrl_obj_type = HW_ctrl_object_dic['_type']

            # get the GO object in which is converted the current HW controlled object if any
            GO_ctrl_object_uid = HW_ctrl_object_dic['_GO_uid']
            if GO_ctrl_object_uid not in ('', None):
                nb_controlled_GO_objects += 1
                if LOG_HW2GO_drawstop: print(f"     controlling {HW_ctrl_object_dic['_uid']} CONVERTED IN GO {GO_ctrl_object_uid}")

                # get the GO manual UID to which is linked the GO object if any
                GO_man_uid = mydickey(HW_ctrl_object_dic, '_GO_manual_uid')
                if GO_man_uid != None:
                    GO_manual_uid = GO_man_uid
                    if LOG_HW2GO_drawstop: print(f"     {HW_ctrl_object_dic['_uid']} LINKED TO {GO_man_uid}")

                # get the first seen GO switch UID, it is considered as the controlling switch, get its name also
                if HW_ctrl_obj_type == 'Switch' and GO_switch_uid == None:
                    GO_switch_uid = GO_ctrl_object_uid
                    if switch_name == None:
                        switch_name = self.GO_odf_dic[GO_ctrl_object_uid]['Name']

            else:
                if LOG_HW2GO_drawstop: print(f"     controlling {HW_ctrl_object_dic['_uid']} NOT CONVERTED IN GO")

            if HW_ctrl_obj_type == 'Pipe_SoundEngine01':
                nb_controlled_noise_pipes += 1

                # get an ID for the audio channel of the HW pipe
                audio_channel_id = self.HW_ODF_get_pipe_audio_channel_id(HW_ctrl_object_dic)
                if audio_channel_id not in HW_pipes_descr_dic.keys():
                    # create an entry in the dictionary for the audio channel ID and with empty attack and release pipes placeholders
                    HW_pipes_descr_dic[audio_channel_id] = [None, None]

                # identify the kind of noise (attack or release)
                if mydickey(HW_ctrl_object_dic, '_hint') == 'inverted':
                    # the current HW pipe is controlled by an inverted way
                    noise_kind = 'release'
                else:
                    noise_kind = 'attack'
                # double check the noise kind to take into consideration particular definitions
                noise_kind = self.HW_ODF_pipe_noise_kind_check(HW_ctrl_object_dic, noise_kind)

                # store the pipe at the correct placeholder in the pipes description list
                if noise_kind == 'attack':
                    HW_pipes_descr_dic[audio_channel_id][0] = HW_ctrl_object_dic
                    if LOG_HW2GO_drawstop: print(f"       {HW_ctrl_object_dic['_uid']} attack noise in channel ID {audio_channel_id}")
                else:  # noise_kind == 'release'
                    HW_pipes_descr_dic[audio_channel_id][1] = HW_ctrl_object_dic
                    if LOG_HW2GO_drawstop: print(f"       {HW_ctrl_object_dic['_uid']} release noise in channel ID {audio_channel_id}")

            elif HW_ctrl_obj_type == 'ContinuousControlLinkage':
                is_controlling_cc_linkage = True

            elif HW_ctrl_obj_type == 'WindCompartmentLinkage':
                is_controlling_wc_linkage = True

            elif HW_ctrl_obj_type == 'SwitchLinkage':
                if mydickey(HW_ctrl_object_dic, '_hint') == 'condition':
                    has_condition_effect = True
                elif mydickey(HW_ctrl_object_dic, '_hint') == 'not_supported_linkage':
                    has_unsupported_linkage = True
                else:
                    has_supported_linkage = True

            elif HW_ctrl_obj_type == 'Switch':
                # add a tag in the HW Switch to indicate that it has been checked by this function, to not check it again later
                HW_ctrl_object_dic['_ctrlchck'] = ''

                if self.HW_ODF_get_attribute_value(HW_ctrl_object_dic, 'DefaultToEngaged') == 'Y':
                    is_default_engaged_switch = True
                    if LOG_HW2GO_drawstop: print(f"     {HW_ctrl_object_dic['_uid']} is default engaged")

                if self.HW_ODF_get_attribute_value(HW_ctrl_object_dic, 'Latching') != 'N':
                    is_latching_switch = True
                    if LOG_HW2GO_drawstop: print(f"     {HW_switch_dic['_uid']} is latching")

                # get the HW Switch assignment code if it is defined
                asgn_code = myint(self.HW_ODF_get_attribute_value(HW_ctrl_object_dic, 'DefaultInputOutputSwitchAsgnCode'))
                if asgn_code != None:
                    HW_switch_asgn_code = asgn_code
                    if LOG_HW2GO_drawstop: print(f"     {HW_ctrl_object_dic['_uid']} has assignment code {asgn_code}")

                    # take the name of the HW Switch which has an assignment code, it will be relevant to name the GO switch if a name is not already defined
                    switch_name = HW_ctrl_object_dic['Name']

                # check if the HW Switch has a graphical interface (i.e. it has a defined HW ImageSetInstance object)
                HW_image_set_inst_dic = self.HW_ODF_get_object_dic_by_ref_id('ImageSetInstance', HW_ctrl_object_dic, 'Disp_ImageSetInstanceID')
                HW_display_page_dic = self.HW_ODF_get_object_dic_by_ref_id('DisplayPage', HW_image_set_inst_dic, 'DisplayPageID')
                if HW_display_page_dic != None:
                    page_name = HW_display_page_dic['Name'].upper()
                    if 'CRESC' in page_name:
                        # the current switch is located in a Crescendo panel, ignore it by considering it has not a graphical interface
                        if LOG_HW2GO_drawstop: print(f"     {HW_ctrl_object_dic['_uid']} has a graphical interface in a Crescendo panel")
                    else:
                        has_graphical_interface = True
                        if LOG_HW2GO_drawstop: print(f"     {HW_ctrl_object_dic['_uid']} has a graphical interface")

        # set the switch name if not already set before
        if switch_name == None:
            switch_name = HW_switch_dic['Name']
        if '(' in switch_name:
            # the HW switch has an opening parenthesis : get the text at the left of this parenthesis (seen in Sonus Paradisi sample sets)
            switch_name = switch_name.split('(')[0].rstrip()

        # check if the given HW Switch is a master capture switch defined in the _General object
        if int(HW_switch_dic['_uid'][-6:]) == myint(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'SpecialObjects_MasterCaptureSwitchID')):
            HW_switch_asgn_code = 12

        # if one assignment code has been recovered, define its corresponding functional category
        if HW_switch_asgn_code == None:
            HW_switch_func_category = None
        elif HW_switch_asgn_code == 1:
            HW_switch_func_category = 'blower'
        elif HW_switch_asgn_code == 12 or HW_switch_asgn_code in range (100, 1000):
            # master capture or general or divisional combination
            HW_switch_func_category = 'combination'
        elif 1000 <= HW_switch_asgn_code < 1690:
            HW_switch_func_category = 'coupler'
        elif 1690 <= HW_switch_asgn_code < 2000:
            HW_switch_func_category = 'tremulant'
        elif 2000 <= HW_switch_asgn_code < 3000:
            HW_switch_func_category = 'stop'
        elif HW_switch_asgn_code == 900369: # sequencer previous
            HW_switch_func_category = 'sequencer'
        elif HW_switch_asgn_code == 900366: # sequencer next
            HW_switch_func_category = 'sequencer'
        else:
            HW_switch_func_category = None

        # display a log with the synthesis of the controlled objects
        if LOG_HW2GO_drawstop:
            message = '     SYNTHESIS:'
            message += f' name "{switch_name}"'
            message += f', controlling {nb_controlled_GO_objects} GO object(s)'
            message += f', controlling {nb_controlled_noise_pipes} noise pipe(s)'
            if HW_switch_asgn_code != None: message += f', assignment code {HW_switch_asgn_code} ({HW_switch_func_category})'
            if not is_latching_switch: message += ', is not latching switch'
            if is_default_engaged_switch: message += ', is default engaged'
            if has_condition_effect: message += ', has condition action'
            message += ', has a GUI' if has_graphical_interface else ', has no GUI'
            if is_controlling_cc_linkage: message += ', is controlling a ContinuousControlLinkage'
            if is_controlling_wc_linkage: message += ', is controlling a WindCompartmentLinkage'
            print(message)

        #-----------------------------------------------------------------------
        # based on the data retrieved previously, decide if the given HW Switch has to be converted to a GO Switch
        # in order to avoid having a switch without functional or sound effect in GrandOrgue

        if not is_latching_switch:
            if LOG_HW2GO_drawstop: print('     ====> SKIPPED : having none latching switch')
            return None

        if HW_switch_func_category == 'combination':
            # switches controlling combinations are managed in GO_ODF_build_Drawstop_controlled_object
            if LOG_HW2GO_drawstop: print('     ====> SKIPPED : controlling a combination feature')
            return None

        if has_unsupported_linkage and not has_supported_linkage:
            if LOG_HW2GO_drawstop: print('     ====> SKIPPED : controlling through unsupported switch linkage and none supported switch linkage')
            return None

        if nb_controlled_GO_objects == 0:
            # the given HW Switch is not controlling an object converted to GO in GO_ODF_build_Drawstop_controlled_object
            # see if it has to be converted to GO

            if has_condition_effect:
                # switches having condition action on converted objects have been already converted in GO_ODF_build_Draswtop_from_controlled_object
                if LOG_HW2GO_drawstop: print('     ====> SKIPPED : controlling none GO feature and having a conditional effect')
                return None

            if nb_controlled_noise_pipes == 0:
                # the given HW Switch is not controlling noises (blower, stop noises)
                if not has_graphical_interface:
                    if LOG_HW2GO_drawstop: print('     ====> SKIPPED : controlling neither a GO feature nor noise pipes and having no graphical interface')
                    return None

                if HW_switch_func_category in ('stop', 'coupler', 'tremulant', 'blower'):
                    if LOG_HW2GO_drawstop: print('     ====> SKIPPED : should control a stop / coupler / tremulant / blower but controls neither a GO feature nor noises')
                    return None

                if is_controlling_cc_linkage:
                    if LOG_HW2GO_drawstop: print('     ====> SKIPPED : controlling neither a GO feature nor noise pipes and controlling ContinuousControlLinkage')
                    return None

                if is_controlling_wc_linkage:
                    if LOG_HW2GO_drawstop: print('     ====> SKIPPED : controlling neither a GO feature nor noise pipes and controlling WindCompartmentLinkage')
                    return None

            elif nb_controlled_noise_pipes % 2 == 0 and not is_default_engaged_switch:
                # if there is a not null and even number of pipe noises (only engage and release noises) and the switch is not engaged by default
                # it is probably a switch controlling a stop not converted to GO because it has no pipes (case of demo sample set)
                if LOG_HW2GO_drawstop: print('     ====> SKIPPED : controlling no GO feature, controlling an even number of noise pipes')
                return None

        #-----------------------------------------------------------------------
        # build visible GO controlling switches (PanelElement objects and one Switch), including the given controlling HW Switch

        if GO_switch_uid != None:
            # a GO switch UID has been recovered from the controlled GO objects
            if LOG_HW2GO_drawstop: print(f'     reusing {GO_switch_uid} with name "{switch_name}"')
        else:
            # create a new GO Switch object, using the name of the given HW Switch if a switch name has not been already set earlier in this function
            GO_switch_uid = self.GO_ODF_build_Switch_object(switch_name, is_default_engaged=is_default_engaged_switch, gc_state=-1, store_in_comb= nb_controlled_GO_objects > 0)
            if LOG_HW2GO_drawstop: print(f'     {GO_switch_uid} built from {HW_switch_dic["_uid"]} with name "{switch_name}"')

            # create GO PanelElement objects for each visible controlled HW switch and linked to the GO switch UID
            for HW_obj_dic in controlled_HW_objects_dic_list:
                # scan the controlled HW objects list (which includes the given controlling HW Switch)
                if HW_obj_dic['_type'] == 'Switch':
                    self.GO_ODF_build_PanelElement_button_object(HW_obj_dic, 'Switch', GO_switch_uid)

            # store the created GO Switch UID in the HW Switch
            HW_switch_dic['_GO_uid'] = GO_switch_uid

        if GO_switch_uid == None:
            # this should never occur
            if LOG_HW2GO_drawstop: print(f"       None switch built for {HW_switch_dic['_uid']}")
            return None

        #----------------------------------
        # build GO noise stops corresponding to the found HW noise attack/release pipes controlled by the given HW switch

        if LOG_HW2GO_drawstop: print(f"     --- Building {len(HW_pipes_descr_dic)} noise stop(s) ------------------------")

        for audio_channel_id, (HW_attack_pipe_dic, HW_release_pipe_dic) in HW_pipes_descr_dic.items():

            if HW_attack_pipe_dic != None:
                # not possible to build a noise Stop without at least an attack noise

                # build the attributes of the current attack/release pipes couple
                if HW_release_pipe_dic != None:
                    GO_attr_dic = self.GO_ODF_build_Stop_noises_attributes([HW_attack_pipe_dic], [HW_release_pipe_dic])
                else:
                    GO_attr_dic = self.GO_ODF_build_Stop_noises_attributes([HW_attack_pipe_dic], [])

                if GO_attr_dic == None:
                    # an error occurred to build the attributes of the object
                    if HW_release_pipe_dic != None:
                        if LOG_HW2GO_drawstop: print(f"     channel {audio_channel_id} : GO noise stop attributes NOT built from attack {HW_attack_pipe_dic['_uid']} and release {HW_release_pipe_dic['_uid']}")
                    else:
                        if LOG_HW2GO_drawstop: print(f"     channel {audio_channel_id} : GO noise stop attributes NOT built from attack {HW_attack_pipe_dic['_uid']} and no release pipe")
                else:
                    if HW_release_pipe_dic != None:
                        if LOG_HW2GO_drawstop: print(f"     channel {audio_channel_id} : GO noise stop attributes built from attack {HW_attack_pipe_dic['_uid']} and release {HW_release_pipe_dic['_uid']}")
                    else:
                        if LOG_HW2GO_drawstop: print(f"     channel {audio_channel_id} : GO noise stop attributes built from attack {HW_attack_pipe_dic['_uid']} and no release pipe")

                    # define the UID of the GO Stop to create in the GO ODF dictionary
                    GO_object_uid = self.GO_ODF_get_free_uid_in_manual(GO_manual_uid, 'Stop')

                    # create the Stop object in the GO dictionary
                    GO_object_dic = self.GO_odf_dic[GO_object_uid] = {}

                    GO_object_dic['Name'] = switch_name

                    # copy in new GO object the GO object attributes built before
                    for attr, value in GO_attr_dic.items():
                        GO_object_dic[attr] = value

                    if GO_object_dic['Percussive'] == 'Y' and not 'NOISE' in switch_name.upper():
                        GO_object_dic['Name'] += ' noise'

                    # add the controlling GO switch in the GO object
                    if GO_switch_uid != None:
                        GO_object_dic['Function'] = 'And'
                        GO_object_dic['SwitchCount'] = 1
                        GO_object_dic['Switch001'] = GO_switch_uid[-3:]
                    else:
                        # there is no controlling switch converted in GO, set a default state for the current object
                        if is_default_engaged_switch:
                            # the HW controlling switch is engaged by default
                            GO_object_dic['DefaultToEngaged'] = 'Y'
                        else:
                            GO_object_dic['DefaultToEngaged'] = 'N'

                    # add a reference to the GO object in the GO Manual to which it belongs
                    self.GO_ODF_child_add(GO_manual_uid, GO_object_uid)

                    # write in the HW Rank corresponding to the current HW Pipe the UID of the corresponding GO object
                    HW_rank_dic = self.HW_ODF_get_object_dic_by_ref_id('Rank', HW_switch_dic, 'RankID')  # HW_switch_dic is a HW Pipe_SoundEngine01 object
                    if HW_rank_dic != None:
                        HW_rank_dic['_GO_uid'] = GO_object_uid

                    HW_attack_pipe_dic['_GO_uid'] = GO_object_uid
                    if HW_release_pipe_dic != None:
                        HW_release_pipe_dic['_GO_uid'] = GO_object_uid

                    if LOG_HW2GO_drawstop: print(f"     GO {GO_object_uid} '{GO_object_dic['Name']}' built, controlled by {GO_switch_uid}")
            else:
                if LOG_HW2GO_drawstop: print("     NO attack pipe object defined")

        return GO_switch_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Switch_object(self, switch_name, switch_function=None, input_switches_uid_list=[], is_default_engaged=False, gc_state=-1, store_in_comb=True):
        # build a GO Switch from the given parameters : switch name, switch function ('Input'/None, 'And', 'Or', 'Not'), input switches UID list if 'And' or 'Or' or 'Not' function
        # if the given Function is None or 'Input', the following parameters are used :
        #   is_default_engaged flag to set the DefaultToEngaged attribute
        #   gc_state value to set the GCState attribute
        #   store_in_comb flag to set (Y or N) the StoreInGeneral and StoreInDivisional attributes
        # return the UID of the built GO Switch

        # add a Switch object in the GO ODF
        self.GO_organ_dic['NumberOfSwitches'] += 1
        GO_switch_uid = 'Switch' + str(self.GO_organ_dic['NumberOfSwitches']).zfill(3)
        GO_switch_dic = self.GO_odf_dic[GO_switch_uid] = {}

        if '(' in switch_name:
            # the HW switch has an opening parenthesis : get the text at the left of this parenthesis (seen in Sonus Paradisi sample sets)
            switch_name = switch_name.split('(')[0].rstrip()

        GO_switch_dic['Name'] = switch_name

        if switch_function not in (None, 'Input'):
            # a switch function is defined
            GO_switch_dic['Function'] = switch_function

            if switch_function != 'Not':
                GO_switch_dic['SwitchCount'] = len(input_switches_uid_list)

            for switch_id, GO_input_switch_uid in enumerate(input_switches_uid_list):
                GO_switch_dic['Switch' + str(switch_id+1).zfill(3)] = GO_input_switch_uid[-3:]

        else:
            # if no function is defined (so no input switches), define needed attributes to set the state of the switch
            if is_default_engaged:
                GO_switch_dic['DefaultToEngaged'] = 'Y'
            else:
                GO_switch_dic['DefaultToEngaged'] = 'N'

            GO_switch_dic['GCState'] = gc_state

            if store_in_comb == True:
                # the switch state has to be stored in combinations (General and Divisional)
                GO_switch_dic['StoreInGeneral'] = 'Y'
                GO_switch_dic['StoreInDivisional'] = 'Y'
            else:
                GO_switch_dic['StoreInGeneral'] = 'N'
                GO_switch_dic['StoreInDivisional'] = 'N'

        return GO_switch_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_PanelElement_button_object(self, HW_switch_dic, button_type, GO_controlled_object_uid=None, GO_manual_uid=None):
        # build a GO PanelElement corresponding to the given HW Switch and the other parameters :
        #    button_type   : value to set in the Type attribute of the PanelElement ('Switch', 'General', 'Divisional', 'Set', 'GC')
        #    GO_controlled_object_uid : UID of the GO object controlled by the PanelElement (if it is not a setter)
        #    GO_manual_uid : UID of the manual where is the controlled GO object in case of Divisional button type

        # used HW objects :
        #   Switch C> ImageSetInstance

        # get the HW ImageSetInstance object associated to the given HW Switch object
        HW_image_set_inst_dic = self.HW_ODF_get_object_dic_by_ref_id('ImageSetInstance', HW_switch_dic, 'Disp_ImageSetInstanceID')
        # get the ID of the HW display page in which the switch is displayed
        HW_switch_disp_page_id = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'DisplayPageID', MANDATORY))
        if HW_switch_disp_page_id == None:
            # the given HW Switch has no graphical interface, a PanelElement cannot be built
            return

        # recover the HW display page in which the switch is displayed
        HW_display_page_dic = self.HW_ODF_get_object_dic_from_id('DisplayPage', HW_switch_disp_page_id)

        if HW_display_page_dic['_GO_uid'] != '':
            # the display page has been converted to a GO Panel
            for layout_id in range(0, self.max_screen_layout_id + 1):
                # scan the screen layouts
                if layout_id == 0 or self.HW_ODF_get_object_dic_by_ref_id('ImageSet', HW_image_set_inst_dic, f'AlternateScreenLayout{layout_id}_ImageSetID') != None:
                    # the switch is displayed in the current screen layout
                    if f'_GO_panelem_uid_layout{layout_id}' not in HW_switch_dic.keys():
                        # the switch is not already converted in a GO panel element for this layout ID
                        # build a new GO Panel999Element999 object with Type=Switch to display the switch

                        # recover the GO panel UID corresponding to the HW display page and screen layout ID
                        GO_panel_uid = HW_display_page_dic[f'_GO_uid_layout{layout_id}']

                        # create the GO Panel999Element999 switch object to display the switch in it
                        self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements'] += 1
                        GO_panel_element_uid = GO_panel_uid + 'Element' + str(self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements']).zfill(3)
                        GO_panel_element_dic = self.GO_odf_dic[GO_panel_element_uid] = {}
                        GO_panel_element_dic['Type'] = button_type

                        if button_type == 'Switch':
                            GO_panel_element_dic['Switch'] = str(int(GO_controlled_object_uid[-3:])).zfill(3)
                        elif button_type == 'General':
                            GO_panel_element_dic['General'] = str(int(GO_controlled_object_uid[-3:])).zfill(3)
                        elif button_type == 'Divisional':
                            # take the two last digits of the Divisional UID to have the Divisional number in the manual
                            GO_panel_element_dic['Divisional'] = str(int(GO_controlled_object_uid[-2:])).zfill(3)
                            GO_panel_element_dic['Manual'] = str(int(GO_manual_uid[-3:])).zfill(3)

                        # add in the HW Switch object the ID of the corresponding GO object
                        HW_image_set_inst_dic[f'_GO_panelem_uid_layout{layout_id}'] = GO_panel_element_uid
                        HW_switch_dic[f'_GO_panelem_uid_layout{layout_id}'] = GO_panel_element_uid

                        # get the index of the switch image for OFF and ON positions
                        switch_off_img_index = myint(self.HW_ODF_get_attribute_value(HW_switch_dic, 'Disp_ImageSetIndexDisengaged'))
                        switch_on_img_index = myint(self.HW_ODF_get_attribute_value(HW_switch_dic, 'Disp_ImageSetIndexEngaged'))
                        if switch_off_img_index == None: switch_off_img_index = '1'
                        if switch_on_img_index == None: switch_on_img_index = '1'

                        # get the attributes of the ImageSetInstance objects and its children
                        image_attr_dic = {}
                        self.HW_ODF_get_image_attributes(HW_image_set_inst_dic, image_attr_dic, switch_off_img_index, layout_id)

                        # set the position of the switch image
                        GO_panel_element_dic['PositionX'] = image_attr_dic['LeftXPosPixels']
                        GO_panel_element_dic['PositionY'] = image_attr_dic['TopYPosPixels']

                        # get the image dimensions
                        img_w = image_attr_dic['ImageWidthPixels']
                        img_h = image_attr_dic['ImageHeightPixels']
                        if (img_w == None or img_h == None) and image_attr_dic['BitmapFilename'] != None:
                            # an image exists but its dimensions are unknown : get it from the image itself
                            image_path = self.HW_sample_set_odf_path + os.path.sep + path2ospath(image_attr_dic['BitmapFilename'])
                            if os.path.isfile(image_path):
                                im = Image.open(image_path)
                                img_w = im.size[0]
                                img_h = im.size[1]

                        if img_w != None:
                            GO_panel_element_dic['Width'] = img_w

                            # set the mouse clickable rectangle width
                            if image_attr_dic['ClickableAreaLeftRelativeXPosPixels'] not in (None, 0):
                                GO_panel_element_dic['MouseRectLeft'] = int(image_attr_dic['ClickableAreaLeftRelativeXPosPixels'])

                            if image_attr_dic['ClickableAreaRightRelativeXPosPixels'] not in (None, 0):
                                if image_attr_dic['ClickableAreaLeftRelativeXPosPixels'] != None:
                                    GO_panel_element_dic['MouseRectWidth'] = image_attr_dic['ClickableAreaRightRelativeXPosPixels'] - image_attr_dic['ClickableAreaLeftRelativeXPosPixels']
                                else:
                                    GO_panel_element_dic['MouseRectWidth'] = image_attr_dic['ClickableAreaRightRelativeXPosPixels']
                                # check if the mouse rectangle width is exceeding the image size
                                if 'MouseRectLeft' in GO_panel_element_dic.keys():
                                    if GO_panel_element_dic['MouseRectLeft'] + GO_panel_element_dic['MouseRectWidth'] > img_w:
                                        GO_panel_element_dic['MouseRectWidth'] = img_w - GO_panel_element_dic['MouseRectLeft']
                                else:
                                    if GO_panel_element_dic['MouseRectWidth'] > img_w:
                                        GO_panel_element_dic['MouseRectWidth'] = img_w

                        if img_h != None:
                            GO_panel_element_dic['Height'] = img_h

                            # set the mouse clickable rectangle height
                            if image_attr_dic['ClickableAreaTopRelativeYPosPixels'] not in (None, 0):
                                GO_panel_element_dic['MouseRectTop'] = int(image_attr_dic['ClickableAreaTopRelativeYPosPixels'])

                            if image_attr_dic['ClickableAreaBottomRelativeYPosPixels'] not in (None, 0):
                                if image_attr_dic['ClickableAreaTopRelativeYPosPixels'] != None:
                                    GO_panel_element_dic['MouseRectHeight'] = image_attr_dic['ClickableAreaBottomRelativeYPosPixels'] - image_attr_dic['ClickableAreaTopRelativeYPosPixels']
                                else:
                                    GO_panel_element_dic['MouseRectHeight'] = image_attr_dic['ClickableAreaBottomRelativeYPosPixels']
                                # check if the mouse rectangle height is exceeding the image size
                                if 'MouseRectTop' in GO_panel_element_dic.keys():
                                    if GO_panel_element_dic['MouseRectTop'] + GO_panel_element_dic['MouseRectHeight'] > img_h:
                                        GO_panel_element_dic['MouseRectHeight'] = img_h - GO_panel_element_dic['MouseRectTop']
                                else:
                                    if GO_panel_element_dic['MouseRectHeight'] > img_h:
                                        GO_panel_element_dic['MouseRectHeight'] = img_h

                                GO_panel_element_dic['MouseRadius'] = 0

                        # set the attributes of the switch OFF image
                        if image_attr_dic['BitmapFilename'] != None:
                            GO_panel_element_dic['ImageOff'] = image_attr_dic['BitmapFilename']
                            if image_attr_dic['TransparencyMaskBitmapFilename'] != None:
                                GO_panel_element_dic['MaskOff'] = image_attr_dic['TransparencyMaskBitmapFilename']

                        # set the attributes of the switch ON image
                        image_attr_dic = {}
                        self.HW_ODF_get_image_attributes(HW_image_set_inst_dic, image_attr_dic, switch_on_img_index, layout_id)
                        if image_attr_dic['BitmapFilename'] != None:
                            # an image is defined
                            GO_panel_element_dic['ImageOn'] = image_attr_dic['BitmapFilename']
                            if image_attr_dic['TransparencyMaskBitmapFilename'] != None:
                                GO_panel_element_dic['MaskOn'] = image_attr_dic['TransparencyMaskBitmapFilename']

                            # attribute set to have no text displayed
                            GO_panel_element_dic['TextBreakWidth'] = 0
                        else:
                            # no image is defined or found, display the GrandOrgue embedded default button image
                            if button_type == 'Switch':
                                # display the name of the controlling switch
                                GO_panel_element_dic['DispLabelText'] = HW_switch_dic['Name']
                            elif button_type in ('Set', 'GC'):
                                # display the name of the button type
                                GO_panel_element_dic['DispLabelText'] = button_type
                                GO_panel_element_dic['DisplayAsPiston'] = 'Y'
                            elif button_type in ('General', 'Divisional'):
                                # display the number of the combination (last two digits of the GO UID)
                                GO_panel_element_dic['DispLabelText'] = str(int(GO_controlled_object_uid[-2:]))

                        # update the host panel size if needed to see the switch images
                        if img_w != None and img_h != None:
                            self.GO_ODF_build_Panel_size_update(self.GO_odf_dic[GO_panel_uid], GO_panel_element_dic['PositionX'] + img_w,
                                                                                               GO_panel_element_dic['PositionY'] + img_h)

                        if LOG_HW2GO_drawstop: print(f"             {GO_panel_element_uid} button with type {button_type} built in screen layer {layout_id} for {GO_controlled_object_uid} from {HW_switch_dic['_uid']}")
                    else:
                        if LOG_HW2GO_drawstop: print(f"             {HW_switch_dic[f'_GO_panelem_uid_layout{layout_id}']} already built from {HW_switch_dic['_uid']}")

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Coupler_attributes(self, HW_key_action_dic):
        # build the GO Coupler attributes corresponding to the given HW KeyAction object
        # if the KeyAction is between two different divisions/keyboards or the same but with keys shift
        # return the dictionary of the GO Coupler attributes, or None in case of building issue
        # in the returned dictionary, add the following key :
        #    '_GO_manual_uid' : GO manual UID to which belongs the GO coupler

        # used HW objects :
        #   KeyAction P> Keyboard  (source keyboard)
        #   KeyAction C> Keyboard  (destination keyboard)
        #   KeyAction C> Division  (destination division)

        # recover the data of the given HW KeyAction
        HW_source_keyboard_dic = self.HW_ODF_get_object_dic_by_ref_id('Keyboard', HW_key_action_dic, 'SourceKeyboardID')
        HW_dest_keyboard_dic = self.HW_ODF_get_object_dic_by_ref_id('Keyboard', HW_key_action_dic, 'DestKeyboardID')
        HW_dest_division_dic = self.HW_ODF_get_object_dic_by_ref_id('Division', HW_key_action_dic, 'DestDivisionID')
        keys_shift = myint(self.HW_ODF_get_attribute_value(HW_key_action_dic, 'MIDINoteNumberIncrement'), 0)

        # get the source GO manual UID of the KeyAction
        GO_source_manual_uid = HW_source_keyboard_dic['_GO_uid']
        if GO_source_manual_uid == '':
            if LOG_HW2GO_drawstop: print(f"    {HW_key_action_dic['_uid']} has no source GO manual built from {HW_source_keyboard_dic['_uid']}")
            return None

        # get the destination GO manual UID of the KeyAction
        if HW_dest_keyboard_dic != None:
            # the destination is a HW Keyboard
            GO_dest_manual_uid = HW_dest_keyboard_dic['_GO_uid']
            if GO_dest_manual_uid == '':
                if LOG_HW2GO_drawstop: print(f"    {HW_key_action_dic['_uid']} has no destination GO manual built from {HW_dest_keyboard_dic['_uid']}")
                return None
            # get if the destination manual is visible or not (if it is not visible, coupler actions on it must go through it)
            is_destination_manuel_visible = myint(self.HW_ODF_get_attribute_value(HW_dest_keyboard_dic, 'DefaultInputOutputKeyboardAsgnCode'), 0) > 0
        else:
            # the destination is a HW Division
            GO_dest_manual_uid = HW_dest_division_dic['_GO_uid']
            if GO_dest_manual_uid == '':
                if LOG_HW2GO_drawstop: print(f"    {HW_key_action_dic['_uid']} has no destination GO manual built from {HW_dest_division_dic['_uid']}")
                return None
            is_destination_manuel_visible = True

        # ignore the given KeyAction if it has an effect on the same manual without keys shift
        if GO_source_manual_uid == GO_dest_manual_uid and keys_shift == 0:
            if LOG_HW2GO_drawstop: print(f"    {HW_key_action_dic['_uid']} has same source and destination GO manual and no keys shift")
            return None

        # create a dictionary to store in it the GO Coupler attributes
        GO_attr_dic = {}

        GO_attr_dic['Name'] = self.HW_ODF_get_attribute_value(HW_key_action_dic, 'Name')
        GO_attr_dic['UnisonOff'] = 'N'
        GO_attr_dic['DestinationManual'] = GO_dest_manual_uid[-3:]
        GO_attr_dic['DestinationKeyshift'] = keys_shift

        first_key_midi_note = myint(self.HW_ODF_get_attribute_value(HW_key_action_dic, 'MIDINoteNumOfFirstSourceKey'), 0)
        if first_key_midi_note != 0:
            GO_attr_dic['FirstMIDINoteNumber'] = first_key_midi_note

        keys_number = myint(self.HW_ODF_get_attribute_value(HW_key_action_dic, 'NumberOfKeys'), 0)
        if keys_number != 0:
            GO_attr_dic['NumberOfKeys'] = keys_number

        if myint(self.HW_ODF_get_attribute_value(HW_key_action_dic, 'ConditionSwitchID'), 0) == 0:
            # the HW KeyAction has no conditional switch to activate it, set the GO Coupler as engaged by default
            GO_attr_dic['DefaultToEngaged'] = 'Y'
            if LOG_HW2GO_drawstop: print("    is engaged by default")

        if is_destination_manuel_visible:
            GO_attr_dic['CoupleToSubsequentUnisonIntermanualCouplers'] = 'N'
        else:
            GO_attr_dic['CoupleToSubsequentUnisonIntermanualCouplers'] = 'Y'
        GO_attr_dic['CoupleToSubsequentUpwardIntermanualCouplers'] = 'N'
        GO_attr_dic['CoupleToSubsequentDownwardIntermanualCouplers'] = 'N'
        GO_attr_dic['CoupleToSubsequentUpwardIntramanualCouplers'] = 'N'
        GO_attr_dic['CoupleToSubsequentDownwardIntramanualCouplers'] = 'N'

        GO_attr_dic['_GO_manual_uid'] = GO_source_manual_uid

        if LOG_HW2GO_drawstop: print(f"    Coupler '{GO_attr_dic['Name']}' attributes built from {HW_key_action_dic['_uid']} : {GO_source_manual_uid} to {GO_dest_manual_uid}, key shift {keys_shift}, first key MIDI note {first_key_midi_note}, {keys_number} keys")

        return GO_attr_dic

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Tremulant_attributes(self, HW_tremulant_dic):
        # build the GO synthesized type Tremulant attributes corresponding to the given HW Tremulant object
        # place in the GO tremulant object attributes with the list of affected GO WindchestGroup and Manual UID
        # return the dictionary of the GO Tremulant attributes, or None in case of building issue
        # in the returned dictionary, add the following keys :
        #    '_GO_windchests_uid_list' : list of GO WindchestGroup UID on which is acting the Tremulant
        #    '_GO_manuals_uid_list'    : list of GO Manual UID on which is acting the Tremulant
        #    '_GO_ranks_uid_list'      : list of GO Rank sUID on which is acting the Tremulant

        # used HW objects :
        #   Tremulant C> TremulantWaveform C> TremulantWaveformPipe C> Pipe_SoundEngine01 P> Rank > GO WindchestGroup
        #                                                                                         > GO Manual

        # create a dictionary to store in it the GO Tremulant attributes
        GO_attr_dic = {}

        GO_attr_dic['Name'] = self.HW_ODF_get_attribute_value(HW_tremulant_dic, 'Name')

        GO_attr_dic['TremulantType'] = 'Synth'

        freq_hz = myfloat(self.HW_ODF_get_attribute_value(HW_tremulant_dic, 'FrequencyWhenEngagedHz'), 5.0)
        if freq_hz == 0: freq_hz = 5.0
        GO_attr_dic['Period'] = int(1000.0 / freq_hz)  # in milliseconds

        GO_attr_dic['StartRate'] = myint(myfloat(self.HW_ODF_get_attribute_value(HW_tremulant_dic, 'StartRatePercent'), 1))
        GO_attr_dic['StopRate'] = myint(myfloat(self.HW_ODF_get_attribute_value(HW_tremulant_dic, 'StopRatePercent'), 100))

        GO_attr_dic['AmpModDepth'] = 15
        # from kerkovits in GitHub GrandOrgue discussion : AmpModDepth is stored in wav files for each pipe.
        # One could multiple the minimum value found in the sample and multiply it by -100 to get AmpModDepth
        # but it is very tedious (an average of all pipes should be calculated),
        # so a default reasonable value (e.g. 15) could be set, which is still better than nothing

        # scan the HW pipes linked to the given HW Tremulant to identify the GO Rank / WindchestGroup / Manual impacted by this tremulant
        GO_attr_dic['_GO_windchests_uid_list'] = []
        GO_attr_dic['_GO_manuals_uid_list'] = []
        GO_attr_dic['_GO_ranks_uid_list'] = []
        for HW_trem_wave_form_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_tremulant_dic, 'TremulantWaveform', TO_CHILD):
            # scan the HW TremulantWaveForm objects which are children of the given HW Tremulant object
            for HW_trem_wave_form_pipe_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_trem_wave_form_dic, 'TremulantWaveformPipe', TO_CHILD):
                # scan the HW TremulantWaveFormPipe objects which are children of the HW TremulantWaveform object
                HW_pipe_dic = self.HW_ODF_get_object_dic_by_ref_id('Pipe_SoundEngine01', HW_trem_wave_form_pipe_dic, 'PipeID')
                if HW_pipe_dic != None:
                    # get the HW Rank to which belongs to the current HW Pipe
                    HW_rank_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_dic, 'Rank', TO_PARENT, FIRST_ONE)
                    if HW_rank_dic != None and HW_rank_dic['_GO_uid'] != '' and '_GO_manual_uid' in self.GO_odf_dic[HW_rank_dic['_GO_uid']]:
                        # get the GO Rank associated to the HW Rank
                        if HW_rank_dic['_GO_uid'] not in GO_attr_dic['_GO_ranks_uid_list']:
                            GO_attr_dic['_GO_ranks_uid_list'].append(HW_rank_dic['_GO_uid'])
                        # get the GO WindchestGroup associated to the GO Rank
                        if HW_rank_dic['_GO_windchest_uid'] not in GO_attr_dic['_GO_windchests_uid_list']:
                            GO_attr_dic['_GO_windchests_uid_list'].append(HW_rank_dic['_GO_windchest_uid'])
                        # get the GO Manual to which belongs the GO Rank
                        GO_manual_uid = self.GO_odf_dic[HW_rank_dic['_GO_uid']]['_GO_manual_uid']
                        if GO_manual_uid not in GO_attr_dic['_GO_manuals_uid_list']:
                            GO_attr_dic['_GO_manuals_uid_list'].append(GO_manual_uid)

        if len(GO_attr_dic['_GO_ranks_uid_list']) == 0:
            # the tremulant is acting on none pipe, ignore it
            if LOG_HW2GO_drawstop: print(f'     HW synthetized tremulant {HW_tremulant_dic["_uid"]} "{HW_tremulant_dic["Name"]}" is acting on none pipe, it is ignored')
            return None

        if len(GO_attr_dic['_GO_manuals_uid_list']) == 1:
            # the tremulant is acting on a single manual, indicate this manual UID in the returned attributes dictionary
            GO_attr_dic['_GO_manual_uid'] = GO_attr_dic['_GO_manuals_uid_list'][0]

        if LOG_HW2GO_drawstop: print(f"     Synthetized tremulant '{GO_attr_dic['Name']}' built from HW {HW_tremulant_dic['_uid']} acting on {sorted(GO_attr_dic['_GO_windchests_uid_list'])} and {sorted(GO_attr_dic['_GO_ranks_uid_list'])} and {sorted(GO_attr_dic['_GO_manuals_uid_list'])}")

        return GO_attr_dic

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Combination_attributes(self, HW_combination_dic):
        # build the attributes for a GO General or Divisional corresponding to the given HW Combination (or _General) object
        # return the dictionary of the GO attributes, or None in case of building issue
        # the switches controlled by the GO General or Divisional are added in it by the function GO_ODF_build_Switch_controlling_objects
        # in the returned dictionary, add the following keys :
        #    '_comb_type' : type of Combination 'set' / 'general' / 'divisional'

        # HW Combination attributes CanEngageControlledSwitches and CanDisengageControlledSwitches are supposed to be always at Y

        # create a dictionary to store in it the GO combination attributes
        GO_attr_dic = {}
        HW_comb_type_code = 0

        if HW_combination_dic != self.HW_general_dic:
            # CombinationTypeCode values :
            #    1 : Master capture
            #    2 : General
            #    3 : Divisional
            #    4 : Crescendo
            #    6 : General cancel
            #    7 : Divisional cancel
            #  100 : Comb. Gen. Cancel
            #  1xx : Comb. Gen. xx
            #  200 : Comb. Div. Pedal Cancel
            #  2xx : Comb. Div. Pedal xx
            #  300 : Comb. Div. Man. 1 Cancel
            #  3xx : Comb. Div. Man. 1 xx
            #  400 : Comb. Div. Man. 2 Cancel
            #  4xx : Comb. Div. Man. 2 xx
            #  500 : Comb. Div. Man. 3 Cancel
            #  5xx : Comb. Div. Man. 3 xx
            #  600 : Comb. Div. Man. 4 Cancel
            #  6xx : Comb. Div. Man. 4 xx
            #  700 : Comb. Div. Man. 5 Cancel
            #  7xx : Comb. Div. Man. 5 xx
            #  800 : Comb. Noise Cancel
            #  8xx : Comb. Noise xx
            HW_comb_type_code = myint(self.HW_ODF_get_attribute_value(HW_combination_dic, 'CombinationTypeCode'), 0)
            GO_attr_dic['Name'] = self.HW_ODF_get_attribute_value(HW_combination_dic, 'Name')

        elif myint(self.HW_ODF_get_attribute_value(self.HW_general_dic, 'SpecialObjects_MasterCaptureSwitchID'), 0) != 0:
            # _General object, containing the reference to the master combination capture switch (SET)
            HW_comb_type_code = 1
            GO_attr_dic['Name'] = 'Master capture'


        if HW_comb_type_code == 0:
            # attribute not defined
            if LOG_HW2GO_drawstop: print(f"     HW {HW_combination_dic['_uid']} '{GO_attr_dic['Name']}' has undefined type code")
            return None

        if HW_comb_type_code == 1:
            # master capture
            GO_attr_dic['_comb_type'] = 'set'
            if LOG_HW2GO_drawstop: print(f"     HW {HW_combination_dic['_uid']} '{GO_attr_dic['Name']}' has type code {HW_comb_type_code} : SET (master capture)")
            return GO_attr_dic

        if HW_comb_type_code == 100:
            # general cancel
            GO_attr_dic['_comb_type'] = 'gc'
            if LOG_HW2GO_drawstop: print(f"     HW {HW_combination_dic['_uid']} '{GO_attr_dic['Name']}' has type code {HW_comb_type_code} : GC (general cancel)")
            return GO_attr_dic

        if HW_comb_type_code in (2, 6) or HW_comb_type_code in range(100, 200):
            # General combination
            GO_attr_dic['_comb_type'] = 'general'
            if LOG_HW2GO_drawstop: print(f"     HW {HW_combination_dic['_uid']} '{GO_attr_dic['Name']}' has type code {HW_comb_type_code} : GENERAL")

        elif HW_comb_type_code in (3, 7) or HW_comb_type_code in range(200, 900):
            # Divisional combination
            GO_attr_dic['_comb_type'] = 'divisional'
            if LOG_HW2GO_drawstop: print(f"     HW {HW_combination_dic['_uid']} '{GO_attr_dic['Name']}' has type code {HW_comb_type_code} : DIVISIONAL")

        else:
            # unsupported type of combination
            if LOG_HW2GO_drawstop: print(f"     HW {HW_combination_dic['_uid']} '{GO_attr_dic['Name']}' has unsupported type code {HW_comb_type_code}")
            return None

        # AllowsCapture values :
        #  '1' : Template
        #  '2' : General
        #  '3' : Divisional
        #  '6' : General cancel
        #  '7' : Divisional cancel
        #  'Y' : Master capture
        #  'N' : capture not allowed
        HW_allows_capture = mystr(self.HW_ODF_get_attribute_value(HW_combination_dic, 'AllowsCapture'), 'N')
        if HW_allows_capture in ('N', '6', '7'):
            # capture is not allowed or it is a General/Divisional cancel button : the combination is protected
            GO_attr_dic['Protected'] = 'Y'
        else:
            GO_attr_dic['Protected'] = 'N'

        if LOG_HW2GO_drawstop: print(f"     AllowsCapture={HW_allows_capture}")

        GO_manual_uid = None
        GO_attr_dic['NumberOfCouplers'] = 0
        if GO_attr_dic['_comb_type'] == 'general':
            GO_attr_dic['NumberOfDivisionalCouplers'] = 0
        GO_attr_dic['NumberOfStops'] = 0
        GO_attr_dic['NumberOfTremulants'] = 0
        GO_attr_dic['NumberOfSwitches'] = 0

        if GO_attr_dic['_comb_type'] == 'divisional':
            # in case of divisional, search for the first controlled stop to retrieve the GO manual UID of this parent HW Division
            GO_attr_dic['_GO_manual_uid'] = None
            for HW_comb_elem_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_combination_dic, 'CombinationElement', TO_CHILD):
                # scan the CompinationElements objects which are children of the given HW Combination object
                HW_contr_switch_dic = self.HW_ODF_get_object_dic_by_ref_id('Switch', HW_comb_elem_dic, 'ControlledSwitchID')
                # recover the HW objects controlled by the combination element controlled switch
                controlled_HW_objects_dic_list = []
                self.HW_ODF_get_switch_controlled_objects(HW_contr_switch_dic, controlled_HW_objects_dic_list)

                for HW_ctrl_object_dic in controlled_HW_objects_dic_list:
                    # scan the controlled objects to find a Stop
                    if HW_ctrl_object_dic['_type'] == 'Stop':
                        # recover the HW Division to which belongs the Stop
                        HW_division_dic = self.HW_ODF_get_object_dic_by_ref_id('Division', HW_ctrl_object_dic, 'DivisionID')
                        if HW_division_dic != None:
                            GO_manual_uid = HW_division_dic['_GO_uid']
                            GO_attr_dic['_GO_manual_uid'] = GO_manual_uid
                            if LOG_HW2GO_drawstop: print(f"     {HW_comb_elem_dic['_uid']} DIVISIONAL is acting on {GO_manual_uid} ")
                            break
                if GO_attr_dic['_GO_manual_uid'] != None:
                    break

        return GO_attr_dic

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Stop_noises_attributes(self, HW_attack_pipes_dic_list, HW_release_pipes_dic_list):
        # build the GO Stop attributes corresponding to the given lists of HW attack/release pipes (noises)
        # return the dictionary of the GO Stop attributes, or None if error

        # used HW objects :
        #   Pipe_SoundEngine01 C> Pipe_SoundEngine01_Layer C> Pipe_SoundEngine01_AttackSample C> Sample
        #   Pipe_SoundEngine01 C> Pipe_SoundEngine01_Layer C> Pipe_SoundEngine01_ReleaseSample C> Sample

        if len(HW_attack_pipes_dic_list) == 0:
            return None

        # build a GO WindchestGroup for the GO Stop
        # get the first defined HW pipe of the given attack pipes list
        HW_pipe_dic = None
        for HW_pipe_dic in HW_attack_pipes_dic_list:
            if HW_pipe_dic != None:
                break
        # get the source HW WindCompartment of the first pipe to use is at GO WindchestGroup of all the pipes
        HW_wind_comp_dic = self.HW_ODF_get_object_dic_by_ref_id('WindCompartment', HW_pipe_dic, 'WindSupply_SourceWindCompartmentID')
        # get the HW ScalingContinuousControlID of the first layer of the first pipe to use it as GO Enclosure of all the pipes
        HW_pipe_layer_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD, FIRST_ONE)
        HW_cont_ctrl_dic = self.HW_ODF_get_object_dic_by_ref_id('ContinuousControl', HW_pipe_layer_dic, 'AmpLvl_ScalingContinuousControlID')
        # build a GO WindchestGroup for this pipe if not already built
        GO_windchest_uid = self.GO_ODF_build_WindchestGroup_object(HW_wind_comp_dic, HW_cont_ctrl_dic, None)

        # create a dictionary to store in it the GO Stop attributes
        GO_attr_dic = {}

        GO_attr_dic['FirstAccessiblePipeLogicalKeyNumber'] = 1
        GO_attr_dic['FirstAccessiblePipeLogicalPipeNumber'] = 1
        GO_attr_dic['NumberOfAccessiblePipes'] = len(HW_attack_pipes_dic_list)
        GO_attr_dic['NumberOfLogicalPipes'] = len(HW_attack_pipes_dic_list)
        GO_attr_dic['WindchestGroup'] = GO_windchest_uid[-3:]
        GO_attr_dic['AcceptsRetuning'] = 'N'

        if len(HW_release_pipes_dic_list) == 0:
            # there are no HW release pipes defined, attack pipes loop/release have to be handled
            GO_attr_dic['Percussive'] = 'N'
        else:
            # there are HW release pipes defined, release samples will be independant from the attack samples
            GO_attr_dic['Percussive'] = 'Y'
            GO_attr_dic['HasIndependentRelease'] = 'Y'

        release_pipes_nb = len(HW_release_pipes_dic_list)

        for pipe_nb, HW_attack_pipe_dic in enumerate(HW_attack_pipes_dic_list):
            # scan the given HW attack pipes, considering that there can be a HW release pipe present at the same index in the release pipes list
            pipe_id = 'Pipe' + str(pipe_nb + 1).zfill(3)

            # get the first HW Pipe_SoundEngine01_Layer linked to the current HW attack pipe
            HW_attack_pipe_layer_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_attack_pipe_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD, FIRST_ONE)
            if HW_attack_pipe_layer_dic != None:

                if pipe_nb < release_pipes_nb:
                    # get the first HW Pipe_SoundEngine01_Layer linked to the current HW release pipe if it exists
                    HW_release_pipe_dic = HW_release_pipes_dic_list[pipe_nb]
                    HW_release_pipe_layer_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_release_pipe_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD, FIRST_ONE)
                else:
                    HW_release_pipe_layer_dic = None

                # recover the list of attack and release samples of the attack pipe
                HW_attack_pipe_attack_samples_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_attack_pipe_layer_dic, 'Pipe_SoundEngine01_AttackSample', TO_CHILD)
                HW_attack_pipe_release_samples_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_attack_pipe_layer_dic, 'Pipe_SoundEngine01_ReleaseSample', TO_CHILD)

                # recover the list of attack and release samples of the release pipe
                HW_release_pipe_attack_samples_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_release_pipe_layer_dic, 'Pipe_SoundEngine01_AttackSample', TO_CHILD)
                HW_release_pipe_release_samples_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_release_pipe_layer_dic, 'Pipe_SoundEngine01_ReleaseSample', TO_CHILD)

                # the attack samples of the current GO pipe ID will come from the attack samples of the HW attack pipe
                HW_pipe_attack_samples_list = HW_attack_pipe_attack_samples_list
                # define which samples to use for the release samples of the current GO pipe ID
                if len(HW_release_pipe_release_samples_list) > 0:
                    HW_pipe_release_samples_list = HW_release_pipe_release_samples_list
                elif len(HW_release_pipe_attack_samples_list) > 0:
                    HW_pipe_release_samples_list = HW_release_pipe_attack_samples_list
                else:
                    HW_pipe_release_samples_list = HW_attack_pipe_release_samples_list

                # get the attack pipe gain if any
                pipe_gain = myfloat(self.HW_ODF_get_attribute_value(HW_attack_pipe_layer_dic, 'AmpLvl_LevelAdjustDecibels'), 0)
                if pipe_gain != 0:
                    GO_attr_dic[pipe_id + 'Gain'] = max(pipe_gain, -5)  # noise gain set at minimum at -5, some sample sets have a gain at -18 making the noise inaudible

                # define the attack pipes
                attacks_count = 0
                for HW_pipe_attack_sample_dic in HW_pipe_attack_samples_list:
                    # scan the HW Pipe_SoundEngine01_AttackSample child objects of the Pipe_SoundEngine01_Layer object
                    HW_sample_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_attack_sample_dic, 'Sample', TO_CHILD, FIRST_ONE)
                    if HW_sample_dic != None:
                        attacks_count += 1
                        HW_install_package_id = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'InstallationPackageID', MANDATORY))
                        sample_file_name = self.convert_HW2GO_file_name(self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename', MANDATORY), HW_install_package_id)
                        if sample_file_name != None:
                            if attacks_count == 1:
                                GO_attr_dic[pipe_id] = sample_file_name
                            else:
                                GO_attr_dic[pipe_id + 'AttackCount'] = attacks_count - 1
                                GO_attr_dic[pipe_id + 'Attack' + str(attacks_count - 1).zfill(3)] = sample_file_name

                # define the release pipes
                if len(HW_pipe_release_samples_list) > 0:
                    GO_attr_dic[pipe_id + 'LoadRelease'] = 'N'
                    GO_attr_dic[pipe_id + 'ReleaseCount'] = releases_count = 0
                    for HW_pipe_release_sample_dic in HW_pipe_release_samples_list:
                        HW_sample_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_release_sample_dic, 'Sample', TO_CHILD, FIRST_ONE)
                        if HW_sample_dic != None:
                            HW_install_package_id = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'InstallationPackageID', MANDATORY))
                            sample_file_name = self.convert_HW2GO_file_name(self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename', MANDATORY), HW_install_package_id)
                            if sample_file_name != None:
                                releases_count += 1
                                GO_attr_dic[pipe_id + 'ReleaseCount'] = releases_count
                                GO_attr_dic[pipe_id + 'Release' + str(releases_count).zfill(3)] = sample_file_name

            else:  # HW_pipe_dic is None
                GO_attr_dic[pipe_id] = 'DUMMY'

        return GO_attr_dic

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Stop_pipes_attributes(self, HW_stop_dic, trem_samples_mode=None, trem_def_method=None):
        # build the GO Stop attributes corresponding to the given HW Stop (pipes stop) and linked to the given GO manual
        # build the GO Rank(s) linked to the given HW Stop
        # trem_samples_mode can be : None, 'integrated', 'separated' (tremmed samples not to convert or to integrate in non tremmed samples rank or to place in separated rank)
        # trem_def_method can be : None, 'alt_rank', 'sec_layer'     (tremmed samples are not defined or are defined in an alternate rank or are defined in a second pipe layer)
        # return the dictionary of the GO Stop attributes, or None in case of stop without pipes
        # in the returned dictionary, add the following key :
        #    '_GO_manual_uid' : GO manual UID to which belongs the GO Stop

        # used HW objects :
        #   Stop C> StopRank(s) (ActionTypeCode = 1, ActionEffectCode = 1) C> Rank (see GO_ODF_build_Rank_object)
        #   Stop (Hint_PrimaryAssociatedRankID) C> Rank (for some demo sample sets where there is no StopRank object defined)

        # create a dictionary to store in it the GO Stop attributes
        GO_attr_dic = {}

        GO_attr_dic['Name'] = self.HW_ODF_get_attribute_value(HW_stop_dic, 'Name')

        # place in the GO Stop attributes which will be set later in this function
        GO_attr_dic['FirstAccessiblePipeLogicalKeyNumber'] = 0
        GO_attr_dic['NumberOfAccessiblePipes'] = 0
        GO_attr_dic['NumberOfRanks'] = 0
        GO_stop_nb_ranks = 0

        # recover the GO manual in which is acting the GO Stop
        HW_division_dic = self.HW_ODF_get_object_dic_by_ref_id('Division', HW_stop_dic, 'DivisionID')
        if HW_division_dic != None:
            GO_manual_uid = HW_division_dic['_GO_uid']
            if GO_manual_uid != '':
                GO_attr_dic['_GO_manual_uid'] = GO_manual_uid

                # get the MIDI notes range of the GO manual to which is attached the given GO Stop
                GO_manual_dic = self.GO_odf_dic[GO_manual_uid]
                manual_first_midi_note = GO_manual_dic['FirstAccessibleKeyMIDINoteNumber']

                manual_first_access_key_nb = 999
                manual_last_access_key_nb = 0
            else:
                if LOG_HW2GO_drawstop: print(f"     No GO manual found for {HW_stop_dic['_uid']}")
                return None

        if LOG_HW2GO_drawstop: print(f"     {HW_stop_dic['_uid']} in {GO_manual_uid} '{GO_manual_dic['Name']}' having {GO_manual_dic['NumberOfAccessibleKeys']} keys from MIDI note {manual_first_midi_note}")

        HW_stop_dic['_GO_windchests_uid_list'] = []

        for HW_stop_rank_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_stop_dic, 'StopRank', TO_CHILD, sorted_by='ID'):
            # scan the HW StopRank objects which are children of the given HW Stop object

            # get the HW Rank and alternate HW Rank if defined, linked to the current HW StopRank
            HW_rank_dic = self.HW_ODF_get_object_dic_by_ref_id('Rank', HW_stop_rank_dic, 'RankID')
            HW_alt_rank_dic = self.HW_ODF_get_object_dic_by_ref_id('Rank', HW_stop_rank_dic, 'AlternateRankID')

            # get the properties of the link between the division and the rank
            HW_div_midi_note_first_mapped_input = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'MIDINoteNumOfFirstMappedDivisionInputNode'), manual_first_midi_note)
            HW_div_midi_note_increment_to_rank = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'MIDINoteNumIncrementFromDivisionToRank'), 0)
            HW_div_nb_mapped_inputs = myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'NumberOfMappedDivisionInputNodes'), 0)

            # correct the MIDI note of the division first mapped input if it is lower that the manual first MIDI note
            if HW_div_midi_note_first_mapped_input < manual_first_midi_note:
                HW_div_nb_mapped_inputs -= manual_first_midi_note - HW_div_midi_note_first_mapped_input
                HW_div_midi_note_first_mapped_input = manual_first_midi_note

            # get the list of the HW Pipe_SoundEngine01 objects which are children of the HW Rank object
            HW_pipes_dic_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_rank_dic, 'Pipe_SoundEngine01', TO_CHILD, sorted_by='ID')
            if (HW_pipes_dic_list != None and len(HW_pipes_dic_list) > 0 and
                myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'ActionTypeCode'), 1) == 1 and
                myint(self.HW_ODF_get_attribute_value(HW_stop_rank_dic, 'ActionEffectCode'), 1) == 1):
                # the current StopRank has normal action codes and is linked to a Rank having at least one pipe inside

                # get the number of pipes layers defined for the first pipe of the current HW rank
                # we consider that all the pipes of the HW rank have the same number of layers as the first pipe has
                HW_pipe1_layers_dic_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipes_dic_list[0], 'Pipe_SoundEngine01_Layer', TO_CHILD, sorted_by='ID')

                # get the UID of the GO Rank associated to the HW Pipe_SoundEngine01_Layer of the first layer of the first pipe if it is already existing
                GO_rank_uid = HW_pipe1_layers_dic_list[0]['_GO_uid']
                GO_trem_rank_uid = None

                if GO_rank_uid == '':
                    # there is not yet a GO Rank built for the current HW rank layer : build it
                    if trem_samples_mode == None or trem_def_method == None:
                        # tremmed samples have not to be converted, or the given HW Stop has no tremmed samples defined inside it
                        GO_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic)

                    elif trem_samples_mode == 'integrated':
                        # tremmed samples have to be integrated in the rank of the non-tremmed samples
                        if trem_def_method == 'sec_layer':
                            # tremmed samples are defined in the second layer of the HW Pipes of the rank
                            GO_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic, 1, None, 10)
                        else:  # trem_def_method == 'alt_rank'
                            # tremmed samples are defined in an alternate HW Rank of the rank
                            GO_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic, None, HW_alt_rank_dic, 10)

                    elif trem_samples_mode == 'separated':
                        # tremmed samples have to be placed in a separated rank and not in the rank of the non-tremmed samples
                        if trem_def_method == 'sec_layer':
                            GO_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic, None, None, 0)
                            GO_trem_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic, 1, None, 1)
                        else:  # trem_def_method == 'alt_rank'
                            GO_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic, None, None, 0)
                            GO_trem_rank_uid = self.GO_ODF_build_Rank_object(None, None, HW_alt_rank_dic, 1)

                        if GO_trem_rank_uid != None:
                            self.GO_odf_dic[GO_rank_uid]['_GO_trem_rank_uid'] = GO_trem_rank_uid
                            self.GO_odf_dic[GO_trem_rank_uid]['_GO_manual_uid'] = GO_manual_uid

                    self.GO_odf_dic[GO_rank_uid]['_GO_manual_uid'] = GO_manual_uid

                else:
                    # a GO rank is already existing, recover the corresponding GO tremmed rank UID if any
                    if trem_samples_mode == 'separated':
                        GO_trem_rank_uid = mydickey(self.GO_odf_dic[GO_rank_uid], '_GO_trem_rank_uid')

                # store in the current HW StopRank the GO WindchestGroup UID in which is the rank
                HW_stop_rank_dic['_GO_windchest_uid'] = HW_rank_dic['_GO_windchest_uid']
                # add in the HW Stop the GO WindchestGroup UID in which is the rank
                HW_stop_dic['_GO_windchests_uid_list'].append(HW_rank_dic['_GO_windchest_uid'])

                GO_rank_dic = self.GO_odf_dic[GO_rank_uid]

                # add the GO Rank to the GO Stop
                GO_stop_nb_ranks += 1
                GO_attr_dic['Rank' + str(GO_stop_nb_ranks).zfill(3)] = GO_rank_uid[-3:]

                # get the rank MIDI notes range
                rank_first_midi_note = GO_rank_dic['FirstMidiNoteNumber']
                rank_last_midi_note = GO_rank_dic['FirstMidiNoteNumber'] + GO_rank_dic['NumberOfLogicalPipes'] - 1

                # if the number of division mapped inputs number is null, set it by default at the rank MIDI notes number
                if HW_div_nb_mapped_inputs == 0:
                    HW_div_nb_mapped_inputs = rank_last_midi_note - rank_first_midi_note + 1

                # define the MIDI notes range used in the rank by the division, taking into account the MIDI note increment from division to rank
                rank_first_used_midi_note = HW_div_midi_note_first_mapped_input + HW_div_midi_note_increment_to_rank
                if rank_first_used_midi_note < rank_first_midi_note:
                    # the first used MIDI note is before the rank first MIDI note
                    HW_div_midi_note_first_mapped_input += rank_first_midi_note - rank_first_used_midi_note
                    HW_div_nb_mapped_inputs -= rank_first_midi_note - rank_first_used_midi_note
                    rank_first_used_midi_note = rank_first_midi_note

                rank_last_used_midi_note = HW_div_midi_note_first_mapped_input + HW_div_nb_mapped_inputs - 1 + HW_div_midi_note_increment_to_rank
                if rank_last_used_midi_note > rank_last_midi_note:
                    # the last used MIDI note is after the rank last MIDI notes
                    HW_div_nb_mapped_inputs -= rank_last_used_midi_note - rank_last_midi_note
                    rank_last_used_midi_note = rank_last_midi_note

                # store the absolute number of the first key of the manual accessing to the rank
                # the value relative to FirstAccessiblePipeLogicalKeyNumber (as expected) will be set later in this function once all the ranks have been built
                rank_first_access_manual_key_nb = HW_div_midi_note_first_mapped_input - manual_first_midi_note + 1
                GO_attr_dic[f'Rank{str(GO_stop_nb_ranks).zfill(3)}FirstAccessibleKeyNumber'] = rank_first_access_manual_key_nb

                # define the number of the first pipe of the rank which is used by the manual
                GO_attr_dic[f'Rank{str(GO_stop_nb_ranks).zfill(3)}FirstPipeNumber'] = rank_first_used_midi_note - GO_rank_dic['FirstMidiNoteNumber'] + 1

                # define the number of pipes of the rank which are used by the manual
                GO_attr_dic[f'Rank{str(GO_stop_nb_ranks).zfill(3)}PipeCount'] = HW_div_nb_mapped_inputs

                # update the manual first and last accessible keys number
                manual_first_access_key_nb = min(manual_first_access_key_nb, rank_first_access_manual_key_nb)
                manual_last_access_key_nb = max(manual_last_access_key_nb, rank_first_access_manual_key_nb + HW_div_nb_mapped_inputs - 1)

                # add in the HW StopRank object the UID of the corresponding GO object
                HW_stop_rank_dic['_GO_uid'] = GO_rank_uid

                # add to the Stop the tremmed rank if it exists, with same attributes as the non tremmed rank
                if GO_trem_rank_uid != None:
                    GO_stop_nb_ranks += 1
                    GO_attr_dic['Rank' + str(GO_stop_nb_ranks).zfill(3)] = GO_trem_rank_uid[-3:]
                    GO_attr_dic[f'Rank{str(GO_stop_nb_ranks).zfill(3)}FirstAccessibleKeyNumber'] = GO_attr_dic[f'Rank{str(GO_stop_nb_ranks - 1).zfill(3)}FirstAccessibleKeyNumber']
                    GO_attr_dic[f'Rank{str(GO_stop_nb_ranks).zfill(3)}FirstPipeNumber'] = GO_attr_dic[f'Rank{str(GO_stop_nb_ranks - 1).zfill(3)}FirstPipeNumber']
                    GO_attr_dic[f'Rank{str(GO_stop_nb_ranks).zfill(3)}PipeCount'] = GO_attr_dic[f'Rank{str(GO_stop_nb_ranks - 1).zfill(3)}PipeCount']

                if LOG_HW2GO_drawstop: print(f"     {HW_stop_rank_dic['_uid']} {HW_rank_dic['_uid']} has pipes converted in GO {GO_rank_uid}")

            else:
                if not(HW_pipes_dic_list != None and len(HW_pipes_dic_list) > 0):
                    if LOG_HW2GO_drawstop: print(f"     {HW_stop_rank_dic['_uid']} {HW_rank_dic['_uid']} has none pipe inside")
                else:
                    if LOG_HW2GO_drawstop: print(f"     {HW_stop_rank_dic['_uid']} has not pipes action codes")

        if GO_stop_nb_ranks > 0:
            # based on the Rank999xxx attributes created just before in the GO Stop for each HW StopRank, compute remaining attributes of the GO Stop
            GO_attr_dic['FirstAccessiblePipeLogicalKeyNumber'] = manual_first_access_key_nb
            GO_attr_dic['NumberOfAccessiblePipes'] = manual_last_access_key_nb - manual_first_access_key_nb + 1
            GO_attr_dic['NumberOfRanks'] = GO_stop_nb_ranks

            # make final adjustments in the Rank999xxx attributes
            for r in range(1, GO_stop_nb_ranks + 1):
                rank_id = 'Rank' + str(r).zfill(3)

                # adjust the Rank999FirstAccessibleKeyNumber attributes so that it is an offset value compated to FirstAccessiblePipeLogicalKeyNumber and no more an absolute key number
                GO_attr_dic[rank_id + 'FirstAccessibleKeyNumber'] -= GO_attr_dic['FirstAccessiblePipeLogicalKeyNumber'] - 1

                # remove the attributes which have a default value
                if GO_attr_dic[rank_id + 'FirstPipeNumber'] == 1:
                    GO_attr_dic.pop(rank_id + 'FirstPipeNumber')

                    if GO_attr_dic[rank_id + 'PipeCount'] == GO_attr_dic['NumberOfAccessiblePipes']:
                        GO_attr_dic.pop(rank_id + 'PipeCount')

                if GO_attr_dic[rank_id + 'FirstAccessibleKeyNumber'] == 1:
                    GO_attr_dic.pop(rank_id + 'FirstAccessibleKeyNumber')

        else:
            # none rank build thanks to StopRank objects : try using the attribute Hint_PrimaryAssociatedRankID if defined
            HW_rank_id = myint(self.HW_ODF_get_attribute_value(HW_stop_dic, 'Hint_PrimaryAssociatedRankID'))
            HW_rank_dic = self.HW_ODF_get_object_dic_from_id('Rank', HW_rank_id)
            if HW_rank_dic != None:
                # the given HW Stop is linked to one Rank through the Hint attribute

                # get the list of the HW Pipe_SoundEngine01 objects which are children of the HW Rank
                HW_pipes_dic_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_rank_dic, 'Pipe_SoundEngine01', TO_CHILD, sorted_by='ID')
                if len(HW_pipes_dic_list) > 0:
                    # the Rank has at least one pipe inside

                    # get the number of pipes layers defined inside the first pipe of the HW rank
                    # we consider that all the pipes of the HW rank have the same number of layers as the first pipe has
                    HW_pipe1_layers_dic_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipes_dic_list[0], 'Pipe_SoundEngine01_Layer', TO_CHILD, sorted_by='ID')

                    # get the UID of the GO Rank associated to the HW Pipe_SoundEngine01_Layer of the first layer of the first pipe if it is already existing
                    GO_rank_uid = HW_pipe1_layers_dic_list[0]['_GO_uid']
                    GO_trem_rank_uid = None

                    if GO_rank_uid == '':
                        # there is not yet a GO Rank built for the current HW rank layer : built it
                        if trem_samples_mode == None or trem_def_method == None:
                            # tremmed samples have not to be converted, or the given HW Stop has no tremmed samples defined inside it
                            GO_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic)

                        elif trem_def_method == 'sec_layer':
                            # only the second layer definition method can be used for the tremmed samples as there it no StopRank defined
                            if trem_samples_mode == 'integrated':
                                # tremmed samples have to be integrated in the rank of the non-tremmed samples
                                GO_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic, 1, None, 10)

                            elif trem_samples_mode == 'separated':
                                # tremmed samples have to be placed in a separated rank and not in the rank of the non-tremmed samples
                                GO_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic, None, None, 0)
                                GO_trem_rank_uid = self.GO_ODF_build_Rank_object(HW_rank_dic, 1, None, 1)

                                if GO_trem_rank_uid != None:
                                    self.GO_odf_dic[GO_rank_uid]['_GO_trem_rank_uid'] = GO_trem_rank_uid
                                    self.GO_odf_dic[GO_trem_rank_uid]['_GO_manual_uid'] = GO_manual_uid

                        self.GO_odf_dic[GO_rank_uid]['_GO_manual_uid'] = GO_manual_uid

                    else:
                        # a GO rank is already existing, recover the corresponding GO tremmed rank UID if any
                        if  trem_samples_mode == 'separated':
                            GO_trem_rank_uid = mydickey(self.GO_odf_dic[GO_rank_uid], '_GO_trem_rank_uid')

                    # add the GO Rank to the GO Stop
                    GO_stop_nb_ranks += 1
                    GO_attr_dic['Rank' + str(GO_stop_nb_ranks).zfill(3)] = GO_rank_uid[-3:]

                    GO_attr_dic['FirstAccessiblePipeLogicalKeyNumber'] = 1
                    GO_attr_dic['NumberOfAccessiblePipes'] = len(HW_pipes_dic_list)
                    GO_attr_dic['NumberOfRanks'] = GO_stop_nb_ranks

                    # add to the Stop the tremmed rank if it exists
                    if GO_trem_rank_uid != None:
                        GO_stop_nb_ranks += 1
                        GO_attr_dic['Rank' + str(GO_stop_nb_ranks).zfill(3)] = GO_trem_rank_uid[-3:]

                    if LOG_HW2GO_drawstop: print(f"     {HW_rank_dic['_uid']} found by hint has pipes converted in GO {GO_rank_uid}")

                else:
                    if LOG_HW2GO_drawstop: print(f"     {HW_rank_dic['_uid']} found by hint has none pipe inside")
            else:
                if LOG_HW2GO_drawstop: print("     none pipes rank found for this Stop")

        if GO_attr_dic['NumberOfAccessiblePipes'] == 0:
            # no pipe accessible for the built stop
            return None

        return GO_attr_dic

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Rank_object(self, HW_rank_dic, HW_alt_layer_nb=None, HW_alt_rank_dic=None, is_trem_mode=None):
        # build the GO Rank from the given HW Rank and alternate HW Rank or pipe layer number if provided, applying the given IsTremulant mode (see below)
        # for ranks of pipes (not percussive), not for ranks of noises
        # HW_alt_rank_dic is ignored if HW_alt_layer_nb is different of None

        # if is_trem_mode =  None  IsTremulant attribute is not set, samples of HW_rank_dic are converted in the GO Rank
        # if is_trem_mode =  0     IsTremulant=0 only is set, samples of HW_rank_dic are converted
        # if is_trem_mode =  1     IsTremulant=1 only is set, samples of HW_rank_dic + HW_alt_layer_nb or HW_alt_rank_dic (if HW_alt_layer_nb=None) are converted
        # if is_trem_mode =  10    IsTremulant=0 and =1 are set, samples of HW_rank_dic for IsTrem=0, samples of HW_rank_dic + HW_alt_layer_nb or HW_alt_rank_dic for IsTrem=1 are converted

        # used HW objects :
        #   Rank C> Pipe_SoundEngine01 P> WindCompartment
        #   Rank C> Pipe_SoundEngine01 P> EnclosurePipe P> Enclosure
        #   Rank C> Pipe_SoundEngine01 C> Pipe_SoundEngine01_Layer P> ContinuousControl
        #   Rank C> Pipe_SoundEngine01 C> Pipe_SoundEngine01_Layer C> Pipe_SoundEngine01_AttackSample C> Sample
        #   Rank C> Pipe_SoundEngine01 C> Pipe_SoundEngine01_Layer C> Pipe_SoundEngine01_ReleaseSample C> Sample

        if is_trem_mode not in (None, 0, 1, 10):
            print(f'INTERNAL ERROR Wrong value is_trem_mode={is_trem_mode} given to function GO_ODF_build_Rank_object')
            return None

        GO_rank_uid = None
        HW_rank_harmonic_nb = 0  # 0 means undefined, will be defined using the first pipe harmonic number
        message_shown = False
        starting_time = time.time()

        # set which HW ranks and pipes layer to use for main pipes (rank1) and for secondary pipes (rank2)
        if is_trem_mode in (None, 0):
            # rank1 with no IsTremulant or IsTremulant=0 (non tremmed samples)
            HW_rank1_dic = HW_rank_dic
            HW_rank1_layer_nb = 0
            # rank2 not needed
            HW_rank2_dic = None

        elif is_trem_mode == 1:
            # rank1 with IsTremulant=1 (tremmed samples)
            if HW_alt_layer_nb != None:
                # using samples of an alternate pipes layer
                HW_rank1_dic = HW_rank_dic
                HW_rank1_layer_nb = HW_alt_layer_nb
            else:
                # using samples of an alternate rank
                HW_rank1_dic = HW_alt_rank_dic
                HW_rank1_layer_nb = 0
            # rank2 not needed
            HW_rank2_dic = None

        elif is_trem_mode == 10:
            # rank1 with IsTremulant=0 (non tremmed samples)
            HW_rank1_dic = HW_rank_dic
            HW_rank1_layer_nb = 0
            # rank2 with IsTremulant=1 (tremmed samples)
            if HW_alt_layer_nb != None:
                # using samples of an alternate pipes layer
                HW_rank2_dic = HW_rank_dic
                HW_rank2_layer_nb = HW_alt_layer_nb
            else:
                # using samples of an alternate rank
                HW_rank2_dic = HW_alt_rank_dic
                HW_rank2_layer_nb = 0

        if LOG_HW2GO_rank: print(f"Building GO Rank from {HW_rank1_dic['_uid']} '{HW_rank1_dic['Name']}' layer {HW_rank1_layer_nb} with is_trem_mode = {is_trem_mode}")
        if HW_rank2_dic != None:
            if LOG_HW2GO_rank: print(f"             and from {HW_rank2_dic['_uid']} '{HW_rank2_dic['Name']}' layer {HW_rank2_layer_nb} ")

        # get the list of the HW Pipe_SoundEngine01 objects which are children of the HW Rank1
        HW_pipes1_dic_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_rank1_dic, 'Pipe_SoundEngine01', TO_CHILD, sorted_by='ID')
        if len(HW_pipes1_dic_list) > 0:
            # the current HW rank has pipes defined inside

            # store in a dictionary the HW Pipe_SoundEngine01 objects of the HW Rank 1 with as key their MIDI note number
            # get the first and last MIDI note numbers of these pipes
            first_midi_note_nb = 999
            last_midi_note_nb = 0
            HW_pipes1_dic = {}
            for HW_pipe_dic in HW_pipes1_dic_list:
                # scan the Pipe_SoundEngine01 objects of the HW Rank 1

                # get the MIDI note number of the current HW Pipe_SoundEngine01
                # observed with Sound Paradisi sample sets, the MIDI note 60 is not defined, so it is the default value
                midi_note_nb = myint(self.HW_ODF_get_attribute_value(HW_pipe_dic, 'NormalMIDINoteNumber'), 60)
                # update the first/last MIDI note numbers
                first_midi_note_nb = min(midi_note_nb, first_midi_note_nb)
                last_midi_note_nb = max(midi_note_nb, last_midi_note_nb)

                # associate the dictionary of the current pipe to its MIDI note number
                HW_pipes1_dic[midi_note_nb] = HW_pipe_dic

            HW_pipes2_dic = {}
            if HW_rank2_dic != None:
                # a second HW rank is defined
                if HW_rank2_dic == HW_rank1_dic:
                    # Rank 2 is same as Rank 1 (but different pipes layer is used normally)
                    HW_pipes2_dic = HW_pipes1_dic
                else:
                    # store in a dictionary the HW Pipe_SoundEngine01 objects of the HW Rank 2 with as key their MIDI note number
                    for HW_pipe_dic in self.HW_ODF_get_linked_objects_dic_by_type(HW_rank2_dic, 'Pipe_SoundEngine01', TO_CHILD, sorted_by='ID'):
                        # scan the Pipe_SoundEngine01 objects of the HW Rank 2

                        # get the MIDI note number of the current HW Pipe_SoundEngine01
                        midi_note_nb = myint(self.HW_ODF_get_attribute_value(HW_pipe_dic, 'NormalMIDINoteNumber'), 60)

                        # associate the dictionary of the current pipe to its MIDI note number
                        HW_pipes2_dic[midi_note_nb] = HW_pipe_dic

            # create a GO Rank999 object
            self.GO_organ_dic['NumberOfRanks'] += 1
            GO_rank_uid = 'Rank' + str(self.GO_organ_dic['NumberOfRanks']).zfill(3)
            GO_rank_dic = self.GO_odf_dic[GO_rank_uid] = {}

            GO_rank_dic['Name'] = self.HW_ODF_get_attribute_value(HW_rank1_dic, 'Name')
            if is_trem_mode == 1:
                # it is a rank with only tremmed samples inside, mark it in its name
                GO_rank_dic['Name'] += ' tremmed'

            # set the rank compass
            GO_rank_dic['FirstMidiNoteNumber'] = first_midi_note_nb
            GO_rank_dic['NumberOfLogicalPipes'] = last_midi_note_nb - first_midi_note_nb + 1

            if LOG_HW2GO_rank: print(f"   building GO {GO_rank_uid} with {GO_rank_dic['NumberOfLogicalPipes']} pipes from MIDI note {first_midi_note_nb} to {last_midi_note_nb}")

            # get the source HW WindCompartment of the first pipe of the HW Rank1 to use it as GO WindchestGroup of the whole GO Rank
            HW_wind_comp_dic = self.HW_ODF_get_object_dic_by_ref_id('WindCompartment', HW_pipes1_dic_list[0], 'WindSupply_SourceWindCompartmentID')
            # get the HW scaling ContinuousControl of the first layer of the first pipe of the HW Rank1 to use it as GO Enclosure of the whole GO Rank if any
            HW_pipe1_layer_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipes1_dic_list[0], 'Pipe_SoundEngine01_Layer', TO_CHILD, sorted_by='ID')[0]
            HW_cont_ctrl_dic = self.HW_ODF_get_object_dic_by_ref_id('ContinuousControl', HW_pipe1_layer_dic, 'AmpLvl_ScalingContinuousControlID')
            # get the HW Enclosure of the first pipe to use it as GO Enclosure of the whole GO Rank if any
            HW_enclosure_pipe_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipes1_dic_list[0], 'EnclosurePipe', TO_PARENT, FIRST_ONE)
            HW_enclosure_dic = self.HW_ODF_get_object_dic_by_ref_id('Enclosure', HW_enclosure_pipe_dic, 'EnclosureID')
            # create the GO WindchestGroup corresponding to the current HW WindCompartment + ContinuousControl + Enclosure
            # if not already existing, else recover the UID of the existing associated GO WindchestGroup
            GO_windchest_uid = self.GO_ODF_build_WindchestGroup_object(HW_wind_comp_dic, HW_cont_ctrl_dic, HW_enclosure_dic)
            GO_rank_dic['WindchestGroup'] = GO_windchest_uid[-3:]
            HW_rank1_dic['_GO_windchest_uid'] = GO_windchest_uid

            GO_rank_dic['Percussive'] = 'N'
            GO_rank_dic['HarmonicNumber'] = 0  # is set later with the first pipe

            for pipe_midi_note_nb in range(first_midi_note_nb, last_midi_note_nb + 1):
                # scan the MIDI notes range of the HW rank by increasing MIDI note number to build the GO pipe attributes of this MIDI note

                # set the GO pipe ID corresponding to the current MIDI note (for example Pipe001, Pipe002, ...)
                pipe_id = 'Pipe' + str(pipe_midi_note_nb - first_midi_note_nb + 1).zfill(3)

                if pipe_midi_note_nb in HW_pipes1_dic.keys():
                    # there is a HW Pipe_SoundEngine01 object defined for the current MIDI note in the Rank 1
                    HW_pipe1_dic = HW_pipes1_dic[pipe_midi_note_nb]

                    # get the HW Pipe_SoundEngine01_Layer of the current HW Pipe_SoundEngine01_Layer in the Rank 1
                    # corresponding to the defined pipe layer number
                    HW_pipe1_layers_dic_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe1_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD, sorted_by='ID')
                    if HW_rank1_layer_nb < len(HW_pipe1_layers_dic_list):
                        HW_pipe1_layer_dic = HW_pipe1_layers_dic_list[HW_rank1_layer_nb]

                        # add in the HW Pipe_SoundEngine01_Layer the ID of the corresponding GO rank object
                        HW_pipe1_layer_dic['_GO_uid'] = GO_rank_uid
                    else:
                        HW_pipe1_layer_dic = None

                    # do the same with the HW Pipe_SoundEngine01 in the Rank 2 if existing
                    if pipe_midi_note_nb in HW_pipes2_dic.keys():
                        # there is a HW Pipe_SoundEngine01 object defined for the current MIDI note in the Rank 2
                        HW_pipe2_dic = HW_pipes2_dic[pipe_midi_note_nb]

                        HW_pipe2_layers_dic_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe2_dic, 'Pipe_SoundEngine01_Layer', TO_CHILD, sorted_by='ID')
                        if HW_rank2_layer_nb < len(HW_pipe2_layers_dic_list):
                            HW_pipe2_layer_dic = HW_pipe2_layers_dic_list[HW_rank2_layer_nb]

                            # add in the HW Pipe_SoundEngine01_Layer the ID of the corresponding GO rank object
                            HW_pipe2_layer_dic['_GO_uid'] = GO_rank_uid
                        else:
                            HW_pipe2_layer_dic = None
                    else:
                        HW_pipe2_layer_dic = None
                        HW_pipe2_dic = None

                    # recover the expected pitch in Hertz of the current Rank 1 pipe if defined
                    HW_pipe1_pitch_hz = myfloat(self.HW_ODF_get_attribute_value(HW_pipe1_dic, 'Pitch_OriginalOrgan_PitchHz'), 0)

                    # get the list of the attack samples of the current Rank 1 pipe layer
                    if HW_pipe1_layer_dic != None:
                        HW_pipe1_attack_samples_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe1_layer_dic, 'Pipe_SoundEngine01_AttackSample', TO_CHILD, sorted_by='ID')
                    else:
                        HW_pipe1_attack_samples_list = []
                    attacks_count = len(HW_pipe1_attack_samples_list)

                    # get the list of the attack samples of the current Rank 2 pipe layer
                    if HW_pipe2_layer_dic != None and attacks_count > 0:
                        # ignore the Rank 2 pipe layer if there is no attack defined for the current Rank 1 pipe layer
                        HW_pipe2_attack_samples_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe2_layer_dic, 'Pipe_SoundEngine01_AttackSample', TO_CHILD, sorted_by='ID')
                    else:
                        HW_pipe2_attack_samples_list = []
                    # add the Rank 2 pipe layer attacks number to the total attacks number
                    attacks_count += len(HW_pipe2_attack_samples_list)

                    # get the pipe harmonic number, 0 means undefined
                    HW_pipe_harmonic_nb = myint(self.HW_ODF_get_attribute_value(HW_pipe1_dic, 'Pitch_Tempered_RankBasePitch64ftHarmonicNum'), 0)

                    if attacks_count > 0:

                        # get the first attack sample of the current Rank 1 pipe layer to recover its properties, assuming that other attacks and releases will have the same
                        HW_pipe_attack_sample_dic = HW_pipe1_attack_samples_list[0]
                        HW_sample_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_attack_sample_dic, 'Sample', TO_CHILD, FIRST_ONE)

                        # get the sample pitch specification method code if defined
                        #   0 = no pitch tuning to apply
                        #   1 = pitch tuning to apply using the MIDI note defined in the metadata of the sample file
                        #   2 = for tremulant waveform, ignored
                        #   3 = pitch tuning to apply using the MIDI note defined in the attributes Pitch_NormalMIDINoteNumber and Pitch_RankBasePitch64ftHarmonicNum
                        #   4 = pitch tuning to apply using the exact sample frequency defined in the attribute Pitch_ExactSamplePitch
                        #       in case of noise sample, Pitch_ExactSamplePitch can be 100 or the value of AudioEngine_BasePitchHz
                        #   5 = for tremulant waveform, ignored
                        HW_pitch_spec_method = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'Pitch_SpecificationMethodCode'), 0)

                        # get the MIDI note of the sample, 0 means undefined (when Pitch_SpecificationMethodCode = 3)
                        HW_sample_midi_note = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'Pitch_NormalMIDINoteNumber'), 0)

                        # get the harmonic number of the sample, 0 means undefined (when Pitch_SpecificationMethodCode = 3)
                        HW_sample_harmonic_nb = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'Pitch_RankBasePitch64ftHarmonicNum'), 0)

                        # get the exact pitch of the sample, 0 means undefined (in Hertz, when Pitch_SpecificationMethodCode = 4)
                        HW_sample_pitch_hz = myfloat(self.HW_ODF_get_attribute_value(HW_sample_dic, 'Pitch_ExactSamplePitch'), 0)

                        # get the MIDI note of the sample from the first 3 digits of its file name if requested by the user
                        if self.tune_pitch_from_sample_filename:
                            file_name = self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename')
                            if file_name != None:
                                file_name_midi_note = file_name.split('/')[-1][:3]  # get first 3 digits of the string after the last '/' character
                                if file_name_midi_note.isdigit():
                                    file_name_midi_note = int(file_name_midi_note)
                                    if file_name_midi_note in range(1, 129):
                                        # it is a MIDI note value
                                        HW_sample_midi_note = file_name_midi_note

                        # get the MIDI note of the sample from its metadata if requested by the user or if the specifiction method code is 1
                        metadata_midi_note = None
                        if self.tune_pitch_from_sample_metadata or HW_pitch_spec_method == 1:
                            # the metadata recovery can increase a lot the rank conversion time, so inform the user if the rank conversion started more than 1 second ago
                            if not message_shown and time.time() - starting_time > 0.5:
                                self.progress_status_update('+ reading samples pitch in metadata...')
                                message_shown = True

                            HW_install_package_id = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'InstallationPackageID', MANDATORY))
                            file_name = self.convert_HW2GO_file_name(self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename', MANDATORY), HW_install_package_id)
                            full_file_name = self.HW_sample_set_odf_path + os.path.sep + file_name
                            metadata_dic = audio_player.wav_data_get(full_file_name, pitch_only=True)
                            if metadata_dic['error_msg'] == '' and 'midi_note' in metadata_dic.keys():
                                metadata_midi_note = metadata_dic['midi_note']
                                if metadata_dic['midi_pitch_fract'] > 50:
                                    # if the pitch fraction is higher than 50, increase the MIDI note by 1
                                    metadata_midi_note += 1

                                # shift the MIDI note of the sample according to the sample or pipe harmonic number to have a MIDI note at pipe level
                                if HW_pipe_harmonic_nb not in (0, 8):
                                    # shift needed only if the pipe harmonic number if not 8 and 0 (not defined)
                                    f1 = midi_nb_to_freq(metadata_midi_note, self.organ_base_pitch_hz)
                                    f2 = f1 * 8 / HW_pipe_harmonic_nb
                                    metadata_midi_note = freq_to_midi_nb(f2, self.organ_base_pitch_hz)
                                HW_sample_midi_note = metadata_midi_note

                        # apply if necessary a pitch tuning to the current pipe
                        pitch_tuning = 0
                        if HW_sample_pitch_hz != 0 and HW_pipe1_pitch_hz != 0 and HW_sample_pitch_hz != HW_pipe1_pitch_hz:
                            # the sample has a pitch in Hz different from the one of the current pipe
                            delta_cents = freq_diff_to_cents(HW_sample_pitch_hz, HW_pipe1_pitch_hz)
                            if abs(delta_cents) > 10:
                                # apply a pitch tuning to the samples of the current pipe if it is higher than 10
                                pitch_tuning = delta_cents
                                if LOG_HW2GO_rank: print(f"      {GO_rank_uid} {pipe_id} MIDI note {pipe_midi_note_nb} : pitch tuning {pitch_tuning} applied (pitch frequency diff.)")

                        elif HW_sample_midi_note not in (0, pipe_midi_note_nb):
                            # the sample has a MIDI note different from the one of the current pipe
                            pitch_tuning = (pipe_midi_note_nb - HW_sample_midi_note) * 100
                            if self.tune_pitch_from_sample_filename or self.tune_pitch_from_sample_metadata:
                                # if the user has enabled a pitch correction option based on MIDI note, show in the logs the done correction
                                logs.add(f"{GO_rank_uid} {pipe_id} MIDI note {pipe_midi_note_nb} : pitch tuning {pitch_tuning} applied (from MIDI note {HW_sample_midi_note})")
                            if LOG_HW2GO_rank: print(f"      {GO_rank_uid} {pipe_id} MIDI note {pipe_midi_note_nb} : pitch tuning {pitch_tuning} applied (from MIDI note {HW_sample_midi_note})")


                        if abs(pitch_tuning) <= 1800:
                            # the pitch tuning to apply, if any, is in the allowed range

                            # get/set the pipe gain if not null
                            pipe_gain = myfloat(self.HW_ODF_get_attribute_value(HW_pipe1_layer_dic, 'AmpLvl_LevelAdjustDecibels'), 0)
                            if pipe_gain != 0:  # 0 is the default value in GO ODF, so need to define it
                                GO_rank_dic[pipe_id + 'Gain'] = pipe_gain

                            if HW_rank_harmonic_nb == 0:
                                # the rank harmonic number is not yet known, take the one of the current pipe
                                HW_rank_harmonic_nb = HW_pipe_harmonic_nb

                            # set the pipe harmonic number if defined and different from the rank harmonic number
                            if HW_pipe_harmonic_nb not in (0, HW_rank_harmonic_nb):
                                GO_rank_dic[pipe_id + 'HarmonicNumber'] = HW_pipe_harmonic_nb

                            # set the pipe pitch tuning if not null
                            if pitch_tuning != 0:
                                GO_rank_dic[pipe_id + 'PitchTuning'] = pitch_tuning

                            # build the attack samples attributes of the current Rank 1 pipe layer
                            attack_nb = 0
                            for HW_pipe_attack_sample_dic in HW_pipe1_attack_samples_list:
                                # scan the HW Pipe_SoundEngine01_AttackSample objects of the current Rank 1 pipe layer

                                # get the dictionary of the first Sample child of the current Pipe_SoundEngine01_AttackSample object
                                HW_sample_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_attack_sample_dic, 'Sample', TO_CHILD, FIRST_ONE)
                                attack_sel_highest_cont_contrl_val = myint(self.HW_ODF_get_attribute_value(HW_pipe_attack_sample_dic, 'AttackSelCriteria_HighestCtsCtrlValue'), 127)
                                if HW_sample_dic != None and attack_sel_highest_cont_contrl_val == 127:
                                    # a sample is defined and its selection is not conditioned to a continuous control value
                                    # recover the file name of the current sample
                                    HW_install_package_id = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'InstallationPackageID', MANDATORY))
                                    sample_file_name = self.convert_HW2GO_file_name(self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename', MANDATORY), HW_install_package_id)
                                    if sample_file_name != None:
                                        attack_nb += 1
                                        if attack_nb == 2:
                                            # if there are more than one attack sample, set the additional attacks count attribute before the second attack sample
                                            GO_rank_dic[pipe_id + 'AttackCount'] = attacks_count - 1
                                        # define the string starting the current pipe attack attribute name
                                        if attack_nb == 1:
                                            pipe_atk_id = pipe_id
                                        else:
                                            pipe_atk_id = pipe_id + 'Attack' + str(attack_nb - 1).zfill(3)
                                        # write the current attack sample file and properties
                                        GO_rank_dic[pipe_atk_id] = sample_file_name
                                        GO_rank_dic[pipe_atk_id + 'LoadRelease'] = 'N'

                                        # set the IsTremulant attribute is needed
                                        if is_trem_mode in (0, 10):
                                            GO_rank_dic[pipe_atk_id + 'IsTremulant'] = 0
                                        elif is_trem_mode == 1:
                                            GO_rank_dic[pipe_atk_id + 'IsTremulant'] = 1

                                        # write the minimum velocity to use this attack sample if defined
                                        attack_sel_highest_velocity = myint(self.HW_ODF_get_attribute_value(HW_pipe_attack_sample_dic, 'AttackSelCriteria_HighestVelocity'), 127)
                                        if attack_sel_highest_velocity < 127:
                                            GO_rank_dic[pipe_atk_id + 'AttackVelocity'] = 127 - attack_sel_highest_velocity

                                        # write the attack loop cross fade length if defined
                                        attack_loop_cross_fade_length = myint(self.HW_ODF_get_attribute_value(HW_pipe_attack_sample_dic, 'LoopCrossfadeLengthInSrcSampleMs'), 0)
                                        if attack_loop_cross_fade_length != 0:
                                            GO_rank_dic[pipe_atk_id + 'LoopCrossfadeLength'] = min(attack_loop_cross_fade_length, 3000)
                                    else:
                                        # sample file not found
                                        GO_rank_dic[pipe_id] = f"DUMMY  ; in package ID {HW_install_package_id}, file not found : {self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename')}"

                            # build the attack samples attributes of the current Rank 2 pipe layer (there is necessarily at least one attack sample defined before for Rank 1)
                            for HW_pipe_attack_sample_dic in HW_pipe2_attack_samples_list:
                                # scan the HW Pipe_SoundEngine01_AttackSample objects of the current Rank 2 pipe layer

                                # get the dictionary of the first alternate Sample child object of the current Pipe_SoundEngine01_AttackSample object
                                HW_sample_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_attack_sample_dic, 'Sample', TO_CHILD, FIRST_ONE)
                                attack_sel_highest_cont_contrl_val = myint(self.HW_ODF_get_attribute_value(HW_pipe_attack_sample_dic, 'AttackSelCriteria_HighestCtsCtrlValue'), 127)
                                if HW_sample_dic != None and attack_sel_highest_cont_contrl_val == 127:
                                    # a sample is defined and its selection is not conditioned to a continuous control value
                                    # recover the file name of the current alternate sample
                                    HW_install_package_id = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'InstallationPackageID', MANDATORY))
                                    sample_file_name = self.convert_HW2GO_file_name(self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename', MANDATORY), HW_install_package_id)
                                    if sample_file_name != None:
                                        attack_nb += 1
                                        if attack_nb == 2:
                                            # if there are more than one attack sample, set the additional attacks count attribute before the second attack sample
                                            GO_rank_dic[pipe_id + 'AttackCount'] = attacks_count - 1
                                        pipe_atk_id = pipe_id + 'Attack' + str(attack_nb - 1).zfill(3)
                                        # write the current attack sample file and properties
                                        GO_rank_dic[pipe_atk_id] = sample_file_name
                                        GO_rank_dic[pipe_atk_id + 'LoadRelease'] = 'N'

                                        # set the IsTremulant attribute if needed
                                        if is_trem_mode == 10:
                                            GO_rank_dic[pipe_atk_id + 'IsTremulant'] = 1

                                        # write the minimum velocity to use this attack sample if defined
                                        pipe_attack_highest_velocity = myint(self.HW_ODF_get_attribute_value(HW_pipe_attack_sample_dic, 'AttackSelCriteria_HighestVelocity'), 127)
                                        if pipe_attack_highest_velocity < 127:
                                            GO_rank_dic[pipe_atk_id + 'AttackVelocity'] = 127 - pipe_attack_highest_velocity

                                        # write the attack loop crossfade length if defined
                                        attack_loop_cross_fade_length = myint(self.HW_ODF_get_attribute_value(HW_pipe_attack_sample_dic, 'LoopCrossfadeLengthInSrcSampleMs'), 0)
                                        if attack_loop_cross_fade_length != 0:
                                            GO_rank_dic[pipe_atk_id + 'LoopCrossfadeLength'] = min(attack_loop_cross_fade_length, 3000)
                                    else:
                                        # sample file not found
                                        GO_rank_dic[pipe_id] = f"DUMMY  ; in package ID {HW_install_package_id}, file not found : {self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename')}"

                            # update the attack count attribute in case some samples have not been actually written (missing sample file)
                            if 1 < attack_nb < attacks_count:
                                GO_rank_dic[pipe_id + 'AttackCount'] = attack_nb - 1

                            # --------------------------------------------
                            # build the release samples attributes

                            # get the list of the release samples of the current Rank 1 pipe layer
                            if HW_pipe1_layer_dic != None:
                                HW_pipe1_release_samples_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe1_layer_dic, 'Pipe_SoundEngine01_ReleaseSample', TO_CHILD, sorted_by='ID')
                            else:
                                HW_pipe1_release_samples_list = []
                            releases_count = len(HW_pipe1_release_samples_list)

                            # get the list of the release samples of the current Rank 2 pipe layer
                            if HW_pipe2_layer_dic != None and releases_count > 0:
                                # ignore the Rank 2 pipe layer if there is no release defined for the current Rank 1 pipe layer
                                HW_pipe2_release_samples_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe2_layer_dic, 'Pipe_SoundEngine01_ReleaseSample', TO_CHILD, sorted_by='ID')
                            else:
                                HW_pipe2_release_samples_list = []
                            # add the Rank 2 pipe layer releases number to the total releases number
                            releases_count += len(HW_pipe2_release_samples_list)

                            if releases_count > 0:
                                # there are release samples
                                release_nb = 0

                                GO_rank_dic[pipe_id + 'ReleaseCount'] = releases_count

                                # build the release samples attributes of the current Rank 1 pipe layer
                                for HW_pipe_release_sample_dic in HW_pipe1_release_samples_list:
                                    # scan the HW Pipe_SoundEngine01_ReleaseSample objects of the current Rank 1 pipe layer

                                    # get the dictionary of the first Sample child object of the current Pipe_SoundEngine01_ReleaseSample object
                                    HW_sample_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_release_sample_dic, 'Sample', TO_CHILD, FIRST_ONE)
                                    if HW_sample_dic != None:
                                        # recover the file name of the current sample
                                        HW_install_package_id = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'InstallationPackageID', MANDATORY))
                                        sample_file_name = self.convert_HW2GO_file_name(self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename', MANDATORY), HW_install_package_id)
                                        if sample_file_name != None:
                                            release_nb += 1
                                            pipe_rel_id = pipe_id + 'Release' + str(release_nb).zfill(3)
                                            # write the current release sample file
                                            GO_rank_dic[pipe_rel_id] = sample_file_name

                                            # set the IsTremulant attribute is needed
                                            if is_trem_mode in (0, 10):
                                                GO_rank_dic[pipe_rel_id + 'IsTremulant'] = 0
                                            elif is_trem_mode == 1:
                                                GO_rank_dic[pipe_rel_id + 'IsTremulant'] = 1

                                            # get the max key release time for the current release sample (-1 by default)
                                            HW_max_key_release_time_int = myint(self.HW_ODF_get_attribute_value(HW_pipe_release_sample_dic, 'ReleaseSelCriteria_LatestKeyReleaseTimeMs'), -1)
                                            if HW_max_key_release_time_int not in (-1, 99999):
                                                # not the default or infinite time which does not need to be indicated in the rank
                                                GO_rank_dic[pipe_rel_id + 'MaxKeyPressTime'] = HW_max_key_release_time_int

                                            # write the release crossfade length if defined
                                            release_cross_fade_length = myint(self.HW_ODF_get_attribute_value(HW_pipe_release_sample_dic, 'ReleaseCrossfadeLengthMs'), 0)
                                            if release_cross_fade_length != 0:
                                                GO_rank_dic[pipe_rel_id + 'ReleaseCrossfadeLength'] = min(release_cross_fade_length, 3000)
                                        else:
                                            # sample file not found
                                            GO_rank_dic[pipe_id] = f"DUMMY  ; in package ID {HW_install_package_id}, file not found : {self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename')}"

                                # build the release samples attributes of the current Rank 2 pipe layer
                                for HW_pipe_release_sample_dic in HW_pipe2_release_samples_list:
                                    # scan the HW Pipe_SoundEngine01_ReleaseSample objects of the current Rank 2 pipe layer

                                    # get the dictionary of the first Sample child object of the current Pipe_SoundEngine01_ReleaseSample object
                                    HW_sample_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_pipe_release_sample_dic, 'Sample', TO_CHILD, FIRST_ONE)
                                    if HW_sample_dic != None:
                                        # recover the file name of the current sample
                                        HW_install_package_id = myint(self.HW_ODF_get_attribute_value(HW_sample_dic, 'InstallationPackageID', MANDATORY))
                                        sample_file_name = self.convert_HW2GO_file_name(self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename', MANDATORY), HW_install_package_id)
                                        if sample_file_name != None:
                                            release_nb += 1
                                            pipe_rel_id = pipe_id + 'Release' + str(release_nb).zfill(3)
                                            # write the current release sample file and properties
                                            GO_rank_dic[pipe_rel_id] = sample_file_name

                                            # set the IsTremulant attribute if needed
                                            if is_trem_mode == 10:
                                                GO_rank_dic[pipe_rel_id + 'IsTremulant'] = 1

                                            # get the max key release time for the current release sample (-1 by default)
                                            HW_max_key_release_time_int = myint(self.HW_ODF_get_attribute_value(HW_pipe_release_sample_dic, 'ReleaseSelCriteria_LatestKeyReleaseTimeMs'), -1)
                                            if HW_max_key_release_time_int not in (-1, 99999):
                                                # not the default or infinite time which does not need to be indicated in the rank
                                                GO_rank_dic[pipe_rel_id + 'MaxKeyPressTime'] = HW_max_key_release_time_int

                                            # write the release crossfade length if defined
                                            release_cross_fade_length = myint(self.HW_ODF_get_attribute_value(HW_pipe_release_sample_dic, 'ReleaseCrossfadeLengthMs'), 0)
                                            if release_cross_fade_length != 0:
                                                GO_rank_dic[pipe_rel_id + 'ReleaseCrossfadeLength'] = min(release_cross_fade_length, 3000)
                                        else:
                                            # sample file not found
                                            GO_rank_dic[pipe_id] = f"DUMMY  ; in package ID {HW_install_package_id}, file not found : {self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename')}"

                                # update the release count attribute in case some samples have not been actually written (missing sample file)
                                if release_nb < releases_count:
                                    GO_rank_dic[pipe_id + 'ReleaseCount'] = release_nb

                        else:
                            # there is a pitch tuning to apply which is outside the allowed range
                            GO_rank_dic[pipe_id] = f"DUMMY  ; MIDI note {pipe_midi_note_nb}, not possible to apply a pitch tuning of {pitch_tuning} for the sample : {self.HW_ODF_get_attribute_value(HW_sample_dic, 'SampleFilename')}"
                    else:
                        # there is none Pipe_SoundEngine01_AttackSample object defined
                        GO_rank_dic[pipe_id] = 'DUMMY  ; MIDI note {pipe_midi_note_nb}, there is none defined attack sample'
                else:
                    # there is none HW Pipe_SoundEngine01 object defined for the current MIDI note
                    GO_rank_dic[pipe_id] = f'DUMMY  ; MIDI note {pipe_midi_note_nb}, there is none defined pipe'

            if HW_rank_harmonic_nb in (0, 8):
                # the rank harmonic number is unknow or at the default value 8, remove the attribute in the rank object
                GO_rank_dic.pop('HarmonicNumber')
            else:
                # set the attribute with the rank harmonic number
                GO_rank_dic['HarmonicNumber'] = HW_rank_harmonic_nb

            # add in the HW Rank object the ID of the corresponding GO object
            HW_rank1_dic['_GO_uid'] = GO_rank_uid
            if HW_rank2_dic != None:
                HW_rank2_dic['_GO_uid'] = GO_rank_uid

        return GO_rank_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_WindchestGroup_object(self, HW_wind_comp_dic, HW_cont_ctrl_dic, HW_enclosure_dic):
        # build the GO WindchestGroup and linked Enclosure objects corresponding to the given HW WindCompartment + ContinuousControl + Enclosure (two last parameters can be at None)
        # a GO WindchestGroup corresponds to a HW WindCompartment + HW ContinuousControl (if not None) + HW Enclosure (if not None)
        # return the UID of the GO WindchestGroup built or already existing

        # used HW objects :
        #   WindCompartment

        if LOG_HW2GO_windchest:
            msg = f"GO WindchestGroup to build from HW {HW_wind_comp_dic['_uid']} ({HW_wind_comp_dic['Name']})"
            if HW_cont_ctrl_dic != None:
                msg += f" + HW {HW_cont_ctrl_dic['_uid']} ({HW_cont_ctrl_dic['Name']})"
            if HW_enclosure_dic != None:
                msg += f"  + HW {HW_enclosure_dic['_uid']} ({HW_enclosure_dic['Name']})"
            print(msg)

        if HW_cont_ctrl_dic != None:
            # build new GO Enclosures or reuse existing ones controlling the given HW continuous control
            GO_enclosure_cc_uid_list = self.GO_ODF_build_Enclosure_object(HW_cont_ctrl_dic)
        else:
            GO_enclosure_cc_uid_list = []

        if HW_enclosure_dic != None:
            # build new GO Enclosures or reuse existing ones controlling the given HW enclosure
            GO_enclosure_enc_uid_list = self.GO_ODF_build_Enclosure_object(HW_enclosure_dic)
        else:
            GO_enclosure_enc_uid_list = []

        # search if there is already a GO WindchestGroup matching with the given HW WindCompartment + Continuouscontrol + Enclosure together
        for GO_object_uid, GO_object_dic in self.GO_odf_dic.items():
            # scan the defined GO objects
            if GO_object_uid[0] == 'W':
                # it is a WindchestGroup object
                if (GO_object_dic['_uid'] == HW_wind_comp_dic['_uid'] and
                    GO_object_dic['_GO_cc_uid_list'] == GO_enclosure_cc_uid_list and
                    GO_object_dic['_GO_enc_uid_list'] == GO_enclosure_enc_uid_list):
                    # the current GO WindchesGroup matches with the given parameters : no need to create a new GO WindchestGroup
                    if LOG_HW2GO_windchest: print(f"     GO {GO_object_uid} already matches with this need")
                    return GO_object_uid

        # there is no matching GO WindchestGroup, create a new one
        self.GO_organ_dic['NumberOfWindchestGroups'] += 1
        GO_windchest_uid = 'WindchestGroup' + str(self.GO_organ_dic['NumberOfWindchestGroups']).zfill(3)
        GO_windchest_dic = self.GO_odf_dic[GO_windchest_uid] = {}
        GO_windchest_dic['_GO_uid'] = GO_windchest_uid
        GO_windchest_dic['_GO_cc_uid_list'] = []
        GO_windchest_dic['_GO_enc_uid_list'] = []
        GO_windchest_dic['Name'] = self.HW_ODF_get_attribute_value(HW_wind_comp_dic, 'Name')
        GO_windchest_dic['NumberOfEnclosures'] = 0

        # add in the HW WindCompartment the UID of the corresponding GO object (several GO WindchestGroup can be linked to the same HW WindCompartment)
        if '_GO_uid_list' not in HW_wind_comp_dic.keys(): HW_wind_comp_dic['_GO_uid_list'] = []
        HW_wind_comp_dic['_GO_uid_list'].append(GO_windchest_uid)
        # add in the GO WindchestGroup the UID of the corresponding HW WindCompartment
        GO_windchest_dic['_uid'] =  HW_wind_comp_dic['_uid']

        # add in the GO WindchestGroup the reference to the GO Enclosure for the HW continuous control if any
        for GO_enclosure_cc_uid in GO_enclosure_cc_uid_list:
            GO_windchest_dic['_GO_cc_uid_list'].append(GO_enclosure_cc_uid)
            GO_windchest_dic['NumberOfEnclosures'] += 1
            GO_windchest_dic['Enclosure' + str(GO_windchest_dic['NumberOfEnclosures']).zfill(3)] = GO_enclosure_cc_uid[-3:]
            GO_windchest_dic['Name'] += '+' + self.GO_odf_dic[GO_enclosure_cc_uid]['Name']

        # add in the GO WindchestGroup the reference to the GO Enclosure for the HW enclosure if any
        for GO_enclosure_enc_uid in GO_enclosure_enc_uid_list:
            GO_windchest_dic['_GO_enc_uid_list'].append(GO_enclosure_enc_uid)
            GO_windchest_dic['NumberOfEnclosures'] += 1
            GO_windchest_dic['Enclosure' + str(GO_windchest_dic['NumberOfEnclosures']).zfill(3)] = GO_enclosure_enc_uid[-3:]
            GO_windchest_dic['Name'] += '+' + self.GO_odf_dic[GO_enclosure_enc_uid]['Name']

        GO_windchest_dic['NumberOfTremulants'] = 0  # will be managed later

        if LOG_HW2GO_windchest: print(f"     GO {GO_windchest_uid} created with {GO_windchest_dic['NumberOfEnclosures']} enclosures")

        return GO_windchest_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_build_Enclosure_object(self, HW_object_dic):
        # build GO Enclosure objects controlling the given HW object (ContinuousControl or Enclosure)
        # build GO Panel999Element999 object(s) with type=Enclosure corresponding to the HW ContinuousControls which controls the given object
        # return the list of UID of the built GO enclosures, or an empty list if none enclosure has been built

        # used HW objects :
        #   Enclosure P> ContinuousControl C> ImageSetInstance C> ImageSet C> ImageSetElement
        #                                                                  C> ContinuousControlImageSetStage

        HW_object_type = HW_object_dic['_type']
        if HW_object_type == 'ContinuousControl':
            HW_cont_ctrl_dic = HW_object_dic
        elif HW_object_type == 'Enclosure':
            # get the parent ContinuousControl of the given Enclosure
            HW_cont_ctrl_dic = self.HW_ODF_get_linked_objects_dic_by_type(HW_object_dic, 'ContinuousControl', TO_PARENT, FIRST_ONE)
        else:
            return []

        # recover the HW ContinuousControls which have a graphical interface and are controlling the given HW ContinuousControl (this one included possibly)
        cont_ctrl_dic_lists = []
        self.HW_ODF_get_controlling_continuous_controls(HW_cont_ctrl_dic, cont_ctrl_dic_lists)
        if len(cont_ctrl_dic_lists) == 0 or len(cont_ctrl_dic_lists[0]) == 0:
            # there is none graphical control for the given object, nothing to do
            if LOG_HW2GO_windchest: print(f"     HW {HW_object_dic['_uid']} is controlled by none visible object")
            return []

        GO_enclosures_uid_list = []

        for branch_nb, cont_ctrl_dic_list in enumerate(cont_ctrl_dic_lists):
            # scan the various branches of continuous controls lists to build one GO enclosure per branch

            GO_enclosure_uid = ''
            cont_ctrl_nb = 0
            for HW_cc_dic in cont_ctrl_dic_list:
                # scan the controlling ContinuousControl objects of the current branch
                if LOG_HW2GO_windchest: print(f"     HW {HW_object_dic['_uid']} is controlled by {HW_cc_dic['_uid']} ({HW_cc_dic['Name']}) in branch {branch_nb}")
                if HW_cc_dic['_type'] == 'ContinuousControl':
                    cont_ctrl_nb += 1
                    if HW_cc_dic['_GO_uid'] != '':
                        # the current ContinuousControl is already converted into a GO Enclosure
                        GO_enclosure_uid = HW_cc_dic['_GO_uid']
                        # set its UID to all the HW ContinuousControls of the current branch
                        for HW_cc_dic2 in cont_ctrl_dic_list:
                            if HW_cc_dic2['_type'] == 'ContinuousControl':
                                HW_cc_dic2['_GO_uid'] = GO_enclosure_uid
                        HW_object_dic['_GO_uid'] = GO_enclosure_uid
                        HW_cont_ctrl_dic['_GO_uid'] = GO_enclosure_uid
                        GO_enclosures_uid_list.append(GO_enclosure_uid)
                        if LOG_HW2GO_windchest: print(f"     HW {HW_object_dic['_uid']} is already converted to GO {GO_enclosure_uid}")
                        break

            if GO_enclosure_uid == '' and cont_ctrl_nb > 0:
                # create a GO Enclosure if none has been yet converted from the given HW object
                self.GO_organ_dic['NumberOfEnclosures'] += 1
                GO_enclosure_uid = 'Enclosure' + str(self.GO_organ_dic['NumberOfEnclosures']).zfill(3)
                GO_enclosure_dic = self.GO_odf_dic[GO_enclosure_uid] = {}
                GO_enclosures_uid_list.append(GO_enclosure_uid)

                if HW_object_type == 'Enclosure':
                    GO_enclosure_dic['Name'] = self.HW_ODF_get_attribute_value(HW_object_dic, 'Name')
                else:
                    # get the name of the first controlling continuous control of the current branch
                    GO_enclosure_dic['Name'] = self.HW_ODF_get_attribute_value(cont_ctrl_dic_list[0], 'Name')

                GO_enclosure_dic['AmpMinimumLevel'] = 0

                if HW_object_type == 'Enclosure':
                    # it is a real enclosure
                    GO_enclosure_dic['MIDIInputNumber'] = 0  # the value will be set later once the ODF conversion is completed

                # create the panel elements controlling the GO enclosure
                for HW_cc_dic in cont_ctrl_dic_list:
                    # scan the HW continuous controls which are controlling the given object and have a graphical interface

                    # get the HW ImageSetInstance object associated to the given HW ContinuousControl
                    HW_image_set_inst_dic = self.HW_ODF_get_object_dic_by_ref_id('ImageSetInstance', HW_cc_dic, 'ImageSetInstanceID')
                    if HW_cc_dic['_type'] == 'ContinuousControl' and HW_image_set_inst_dic != None:
                        HW_display_page_id = myint(self.HW_ODF_get_attribute_value(HW_image_set_inst_dic, 'DisplayPageID'))
                        HW_display_page_dic = self.HW_ODF_get_object_dic_from_id('DisplayPage', HW_display_page_id)

                        for layout_id in range(0, self.max_screen_layout_id + 1):
                            # scan the screen layouts
                            if (f'_GO_uid_layout{layout_id}' in HW_display_page_dic.keys() and
                                (layout_id == 0 or self.HW_ODF_get_object_dic_by_ref_id('ImageSet', HW_image_set_inst_dic, f'AlternateScreenLayout{layout_id}_ImageSetID') != None)):
                                # the current HW DisplayPage layout has been converted to a GO Panel and the enclosure is displayed in the current screen layout

                                # recover the GO panel UID corresponding to the HW display page and screen layout ID
                                GO_panel_uid = HW_display_page_dic[f'_GO_uid_layout{layout_id}']

                                # create a GO Panel999Element999 enclosure object to control the GO enclosure
                                self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements'] += 1
                                GO_panel_element_uid = GO_panel_uid + 'Element' + str(self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements']).zfill(3)
                                GO_panel_element_dic = self.GO_odf_dic[GO_panel_element_uid] = {}
                                GO_panel_element_dic['Type'] = 'Enclosure'
                                GO_panel_element_dic['Enclosure'] = str(int(GO_enclosure_uid[-3:])).zfill(3)

                                # get the image attributes of the first image index to set the attributes of the GO enclosure
                                image_attr_dic = {}
                                self.HW_ODF_get_image_attributes(HW_image_set_inst_dic, image_attr_dic, 1, layout_id)

                                GO_panel_element_dic['PositionX'] = image_attr_dic['LeftXPosPixels']
                                GO_panel_element_dic['PositionY'] = image_attr_dic['TopYPosPixels']

                                if image_attr_dic['ImageWidthPixels'] != None:
                                    GO_panel_element_dic['Width'] = image_attr_dic['ImageWidthPixels']
                                else:
                                    GO_panel_element_dic['Width'] = 0   # will be set later from the first bitmap dimensions

                                if image_attr_dic['ImageHeightPixels'] != None:
                                    GO_panel_element_dic['Height'] = image_attr_dic['ImageHeightPixels']
                                else:
                                    GO_panel_element_dic['Height'] = 0  # will be set later from the first bitmap dimensions

                                # let the mouse clickable area at the default location (i.e. the image dimensions)

                                # get the number of bitmaps of the HW ContinuousControl
                                HW_img_set_dic = self.HW_ODF_get_object_dic_by_ref_id('ImageSet', HW_image_set_inst_dic, 'ImageSetID')
                                HW_img_elems_list = self.HW_ODF_get_linked_objects_dic_by_type(HW_img_set_dic, 'ImageSetElement', TO_CHILD)
                                bitmap_count = min(128, len(HW_img_elems_list))  # GO supports up to 128 bitmaps
                                GO_panel_element_dic['BitmapCount'] = bitmap_count

                                # add the enclosure bitmaps
                                for img_idx in range(1, bitmap_count + 1):
                                    # scan the set of images of the enclosure
                                    image_attr_dic = {}
                                    self.HW_ODF_get_image_attributes(HW_image_set_inst_dic, image_attr_dic, img_idx, layout_id)
                                    if image_attr_dic['BitmapFilename'] != None:
                                        GO_panel_element_dic['Bitmap' + str(img_idx).zfill(3)] = image_attr_dic['BitmapFilename']

                                        if GO_panel_element_dic['Width'] == 0 or GO_panel_element_dic['Height'] == 0:
                                            # get the dimentions of the current image
                                            img_w = img_h = 0
                                            if (image_attr_dic['ImageWidthPixels'] == None or image_attr_dic['ImageHeightPixels'] == None) and image_attr_dic['BitmapFilename'] != None:
                                                image_path = self.HW_sample_set_odf_path + os.path.sep + path2ospath(image_attr_dic['BitmapFilename'])
                                                if os.path.isfile(image_path):
                                                    im = Image.open(image_path)
                                                    img_w = im.size[0]
                                                    img_h = im.size[1]
                                            GO_panel_element_dic['Width'] = img_w
                                            GO_panel_element_dic['Height'] = img_h

                                    if image_attr_dic['TransparencyMaskBitmapFilename'] != None:
                                        GO_panel_element_dic['Mask' + str(img_idx).zfill(3)] = image_attr_dic['TransparencyMaskBitmapFilename']

                                # attribute set to have no text displayed
                                GO_panel_element_dic['TextBreakWidth'] = 0

                                if GO_panel_element_dic['Width'] == 0 or GO_panel_element_dic['Height'] == 0:
                                    # in case there is no image present in the sample set
                                    del self.GO_odf_dic[GO_panel_element_uid]
                                    self.GO_odf_dic[GO_panel_uid]['NumberOfGUIElements'] -= 1
                                    if LOG_HW2GO_windchest: print(f"     GO enclosure panel element {GO_panel_element_uid} not created in panel {GO_panel_uid} because no image found")
                                else:
                                    if LOG_HW2GO_windchest: print(f"     GO enclosure panel element {GO_panel_element_uid} created from {HW_cc_dic['_uid']}")

                        HW_cc_dic['_GO_uid'] = GO_enclosure_uid

                if LOG_HW2GO_windchest: print(f"     GO {GO_enclosure_uid} created for HW {HW_object_dic['_uid']}")

        return GO_enclosures_uid_list

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_child_add(self, parent_uid, child_uid):
        # store in the given GO parent object children list a reference to the given GO child object
        # return True if the operation has been done, False else (reference already present)

        parent_dic = self.GO_odf_dic[parent_uid]

        if not '_children_list' in parent_dic.keys():
            parent_dic['_children_list'] = []

        if not child_uid in parent_dic['_children_list']:
            # the child UID is not already in the children list of the parent
            parent_dic['_children_list'].append(child_uid)
            return True

        return False

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_child_type_nb_get(self, parent_uid, child_type):
        # return the number of child objects of the given type referenced in the given parent object by GO_ODF_child_add

        parent_dic = self.GO_odf_dic[parent_uid]

        child_type_nb = 0
        if '_children_list' in parent_dic.keys():
            for child_uid in parent_dic['_children_list']:
                if child_uid[:-3] == child_type:
                    child_type_nb += 1

        return child_type_nb

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_apply_children_ref(self, parent_uid):
        # write in the givent GO parent object the sorted reference attributes to GO child objects previously stored in it by GO_ODF_child_add

        parent_dic = self.GO_odf_dic[parent_uid]
        if '_children_list' in parent_dic.keys():
            for child_uid in sorted(parent_dic['_children_list']):
                child_type = child_uid[:-3]
                child_id = child_uid[-3:]

                if child_type == 'Switch':
                    nb_of_attr = 'NumberOfSwitches'
                else:
                    nb_of_attr = 'NumberOf' + child_type + 's'

                if nb_of_attr not in parent_dic.keys():
                    parent_dic[nb_of_attr] = 0
                parent_dic[nb_of_attr] += 1
                parent_dic[child_type + str(parent_dic[nb_of_attr]).zfill(3)] = child_id

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_convert_divisionals_ref(self):
        # convert in the Divisional objects the references to switchs from absolute ID to their number in the parent Manual

        for object_uid, object_dic in self.GO_odf_dic.items():
            # scan all the objects of the GO ODF
            if object_uid.startswith('Divisional'):
                # recover the Manual UID from the first digit of the current Divisional UID
                GO_manual_uid = 'Manual00' + object_uid[10]
                GO_manual_dic = self.GO_odf_dic[GO_manual_uid]
                for attr_name, attr_value in object_dic.items():
                    # scan the attributes of the current GO Division
                    if attr_name.startswith('Switch'):
                        if attr_value[0] == '-':
                            GO_switch_id = attr_value[1:]
                            is_netagive = True
                        else:
                            GO_switch_id = attr_value
                            is_netagive = False

                        for man_attr_name, man_attr_value in GO_manual_dic.items():
                            # search in the Manual the number of the current Switch
                            if man_attr_name.startswith('Switch'):
                                if man_attr_value == GO_switch_id:
                                    if is_netagive:
                                        object_dic[attr_name] = '-' + man_attr_name[-3:]
                                    else:
                                        object_dic[attr_name] = man_attr_name[-3:]
                                    break

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_get_free_uid_in_manual(self, manual_uid, object_type):
        # return the next free UID of the given type in the given manual

        manual_id = int(manual_uid[-3:])

        # by default use the ID equal to the number of objects type in the manual + 1
        object_id = self.GO_ODF_child_type_nb_get(manual_uid, object_type) + 1

        if object_id <= 98:  # let value 99 free to permit objects swap in the GUI with drag&drop
            # the first digit of object UID is the manual ID, the two other digits are for the object ID in the parent manual
            object_uid = object_type + str(manual_id).zfill(1) + str(object_id).zfill(2)
        else:
            # if there are more than 98 objects of the object type in the manual, create an UID available over self.GO_object_ext_ID
            object_uid = object_type + str(self.GO_object_ext_ID).zfill(3)
            while object_uid in self.GO_odf_dic.keys() and self.GO_object_ext_ID < 998:
                self.GO_object_ext_ID += 1
                object_uid = object_type + str(self.GO_object_ext_ID).zfill(3)

        return object_uid

    #-------------------------------------------------------------------------------------------------
    def GO_ODF_enclosures_midi_input_number_set(self):
        # set in the Enclosure objects the MIDIInputNumber value when this attribute is already present in it

        enclosures_dic = {} # dictionary with as keys the enclosure UID and as value the total of enclosure element X + Y
        for object_uid, object_dic in self.GO_odf_dic.items():
            # scan all the objects of the GO ODF
            if (len(object_uid) == 18 and object_uid[:5] == 'Panel' and object_uid[8:15] == 'Element' and
                'Type' in object_dic.keys() and object_dic['Type'] == 'Enclosure'):
                # it is an enclosure PanelElement
                # get the associated enclosure UID
                enclosure_uid = 'Enclosure' + object_dic['Enclosure']
                if 'MIDIInputNumber' in self.GO_odf_dic[enclosure_uid].keys():
                    # it is a real enclosure (i.e. not a voicing slider, attribute MIDIInputNumber added in the function GO_ODF_build_Enclosure_object)
                    if enclosure_uid not in enclosures_dic.keys():
                        enclosures_dic[enclosure_uid] = 0
                    # add to the enclosure entry its X+Y positions in the panel
                    enclosures_dic[enclosure_uid] += object_dic['PositionX']
                    enclosures_dic[enclosure_uid] += object_dic['PositionY']

        # transpose the dictionary
        trans_enclosures_dic = {}
        for enclosure_uid, enclosure_dic in enclosures_dic.items():
            trans_enclosures_dic[enclosure_dic] = enclosure_uid

        # set the MIDIInputNumber attributes of the enclosures by order of increasing X+Y values
        for i, order_val in enumerate(sorted(trans_enclosures_dic.keys())):
            enclosure_uid = trans_enclosures_dic[order_val]
            self.GO_odf_dic[enclosure_uid]['MIDIInputNumber'] = i + 1

    #-------------------------------------------------------------------------------------------------
    def convert_HW2GO_file_name(self, HW_file_name, HW_install_package_id):
        # return the given file path/name (for images or sounds or info files) converted from HW to GO format with path relative to ODF folder
        # in HW format the files path starts from the root package folder (named with 6 digits) and the folders separator is either / or \
        # in GO format the files path starts from the ODF location (in the HW folder OrganDefinitions) and the folders separator is \ (it can be / as well)

        os_file_name = path2ospath(HW_file_name)

        if os_file_name[0] == os.path.sep:
            # the HW file name must not start by a path separator (seen on some sample sets), remove it
            os_file_name = os_file_name[1:]

        os_file_name = os.path.join(self.HW_sample_set_path, 'OrganInstallationPackages', str(HW_install_package_id).zfill(6), os_file_name)
        if DEV_MODE:
            actual_file_name_str = os_file_name
        else:
            actual_file_name_str = get_actual_file_name(os_file_name)

        # return the GO file path/name relative to the folder where is located the ODF and with the \ folders separator
        if actual_file_name_str != None:
            return '..' + actual_file_name_str[len(self.HW_sample_set_path):].replace(os.path.sep,'\\')

        # file not found in the sample set files
        if DEV_MODE:
            # return the given file name which comes from the HW ODF
            # permits to test HW ODF conversion without having all the files of the sample set on the computer
            return '..' + os_file_name[len(self.HW_sample_set_path):].replace(os.path.sep,'\\')

        if HW_install_package_id > 10:  #in self.available_HW_packages_id_list:
            # it may be a non standard package of Hauptwerk
            logs.add(f'WARNING : in package ID {HW_install_package_id}, file not found : {HW_file_name}')
        return None

    #-------------------------------------------------------------------------------------------------
    def build_keyboard_octave_disp_attr_dic(self):
        # build the dictionary self.keys_disp_attr_dic with the HW / GO display attributes names of the keyboard keys defined at one octave level
        # the structure of this dictionary is :
        # { note_name:                   -> 'C', 'C#', 'D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B'
        #       {'HW_type':       string,   -> trailing  part of the HW attribute 'KeyShapeImageSetID_'
        #        'HW_type_first': string,   -> trailing  part of the HW attribute 'KeyShapeImageSetID_' for first keyboard key
        #        'HW_type_last':  string,   -> trailing  part of the HW attribute 'KeyShapeImageSetID_' for last keyboard key
        #        'HW_hspacing':    string,  -> trailing  part of the HW attribute 'HorizSpacingPixels_'
        #        'GO_type':        string,  -> GO KEYTYPE value for attributes 'ImageOn_KEYTYPE', 'ImageOff_KEYTYPE', 'Width_KEYTYPE', ...
        #        'GO_type_first':  string,  -> GO KEYTYPE value for first keyboard key and same attributes as above
        #        'GO_type_last':   string}, -> GO KEYTYPE value for last keyboard key and same attributes as above
        # }

        self.keys_disp_attr_dic = {}

        for note_name in NOTES_NAMES:  # NOTES_NAMES = ('C', 'C#', 'D', 'D#', 'E', 'F', 'F#', 'G', 'G#', 'A', 'A#', 'B')
            # parse the 12 notes of one octave

            # initialize a dictionary for the current note name
            key_disp_attr_dic = self.keys_disp_attr_dic[note_name] = {}

            # set the HW key shape types (used in KeyImageSet object for attributes KeyShapeImageSetID_<type> and HorizSpacingPixels_<hspacing>)
            if note_name.endswith('#'):
                key_disp_attr_dic['HW_type'] = 'Sharp'
                key_disp_attr_dic['HW_type_first'] = 'Sharp'
                key_disp_attr_dic['HW_type_last']  = 'Sharp'

                sub_note_name = note_name[0]
                if sub_note_name in ('C', 'F'):
                    key_disp_attr_dic['HW_hspacing']  = 'LeftOfDGFromLeftOfCFSharp'
                elif sub_note_name in ('A', 'D'):
                    key_disp_attr_dic['HW_hspacing']  = 'LeftOfEBFromLeftOfDASharp'
                elif sub_note_name == 'G':
                    key_disp_attr_dic['HW_hspacing']  = 'LeftOfAFromLeftOfGSharp'

            elif note_name in ('C', 'F'):
                key_disp_attr_dic['HW_type'] = 'CF'
                key_disp_attr_dic['HW_type_first'] = 'CF'
                key_disp_attr_dic['HW_type_last']  = 'WholeNatural'
                key_disp_attr_dic['HW_hspacing']  = 'LeftOfCFSharpFromLeftOfCF'
            elif note_name == 'D':
                key_disp_attr_dic['HW_type'] = 'D'
                key_disp_attr_dic['HW_type_first'] = 'FirstKeyDA'
                key_disp_attr_dic['HW_type_last']  = 'LastKeyDG'
                key_disp_attr_dic['HW_hspacing']  = 'LeftOfDASharpFromLeftOfDA'
            elif note_name in ('E', 'B'):
                key_disp_attr_dic['HW_type'] = 'EB'
                key_disp_attr_dic['HW_type_first'] = 'WholeNatural'
                key_disp_attr_dic['HW_type_last']  = 'EB'
                key_disp_attr_dic['HW_hspacing']  = 'LeftOfNaturalFromLeftOfNatural'
            elif note_name == 'G':
                key_disp_attr_dic['HW_type'] = 'G'
                key_disp_attr_dic['HW_type_first'] = 'FirstKeyG'
                key_disp_attr_dic['HW_type_last']  = 'LastKeyDG'
                key_disp_attr_dic['HW_hspacing']  = 'LeftOfGSharpFromLeftOfG'
            elif note_name == 'A':
                key_disp_attr_dic['HW_type'] = 'A'
                key_disp_attr_dic['HW_type_first'] = 'FirstKeyDA'
                key_disp_attr_dic['HW_type_last']  = 'LastKeyA'
                key_disp_attr_dic['HW_hspacing']  = 'LeftOfDASharpFromLeftOfDA'

            # set the GO key types (used in Manual object for attributes ImageOn_KEYTYPE, ImageOff_KEYTYPE, Width_KEYTYPE, ...)
            key_disp_attr_dic['GO_type'] = note_name[0]
            if note_name.endswith('#'):
                key_disp_attr_dic['GO_type'] += 'is'

            key_disp_attr_dic['GO_type_first'] = 'First' + key_disp_attr_dic['GO_type']
            key_disp_attr_dic['GO_type_last']  = 'Last'  + key_disp_attr_dic['GO_type']

    #-------------------------------------------------------------------------------------------------
    def HW_DIC2UID(self, HW_DIC):
        # for logging purpose, return the same type as the given data with all the present HW object dictionary converted to HW object UID string

        if isinstance(HW_DIC, dict):
            if '_uid' in HW_DIC.keys():
                # it is the dictionary of a HW object
                return HW_DIC['_uid']

            # it is another kind of dictionary content, scan the values of each key
            uid_dic = {}
            for key, value in HW_DIC.items():
                uid_dic[key] = self.HW_DIC2UID(value)
            return uid_dic

        if isinstance(HW_DIC, list):
            uid_list = []
            for hw_dic in HW_DIC:
                uid_list.append(self.HW_DIC2UID(hw_dic))
            return uid_list

        # HW_DIC should be a string or None
        return HW_DIC


#-------------------------------------------------------------------------------------------------
class C_GUI_NOTEBOOK():
    # class to manage the graphical user interface of the application for the notebook area


    # variables used for the file viewer tab
    viewer_file_type = None    # type of the displayed file : None, image or sample
    viewer_file_name = None    # name of the file currently displayed in the viewer, or None if no file is displayed
    viewer_orig_image = None   # original image which has to be displayed in case of image file
    viewer_scaled_image = None # scaled image which is displayed (placed as member of the class to keep the image buffer in memory)
    viewer_zoom_factor = 1     # zoom factor at which is currently displayed the image
    viewer_text = ''           # text which is currently displayed at the top of the viewer
    viewer_samples_l = []
    viewer_samples_r = []

    text_to_search = ''

    #-----------------------------------------------------------------------------------------------
    def wnd_notebook_build(self, wnd_parent):
        # build the notebook and its internal GUI widgets inside the given parent widget

        # notebook to display the events logs or the help
        self.notebook = ttk.Notebook(wnd_parent)
        self.notebook.pack(side=tk.TOP, fill=tk.BOTH, expand=True)


        # TAB Logs
        # text box to display the application logs in the notebook, with horizontal/vertical scroll bars
        # a frame is used to encapsulate the text box and scroll bars
        self.frm_logs = ttk.Frame(self.notebook)
        self.frm_logs.pack(fill=tk.BOTH, expand=True)
        scrollbarv = ttk.Scrollbar(self.frm_logs, orient=tk.VERTICAL)
        scrollbarh = ttk.Scrollbar(self.frm_logs, orient=tk.HORIZONTAL)
        scrollbarv.pack(side=tk.RIGHT, fill=tk.Y)
        scrollbarh.pack(side=tk.BOTTOM, fill=tk.X)
        self.txt_events_log = tk.Text(self.frm_logs, fg=TEXT_COLOR, bg=COLOR_BG_LOGS, bd=3, wrap=tk.NONE, font=TEXT_FONT, selectbackground=COLOR_BG_TEXT_SEL)
        self.txt_events_log.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        self.txt_events_log.bind('<Control-Key-a>', self.logs_select_all)
        self.txt_events_log.bind('<Control-Key-A>', self.logs_select_all)
        self.txt_events_log.config(xscrollcommand=scrollbarh.set, yscrollcommand=scrollbarv.set)
        scrollbarv.config(command=self.txt_events_log.yview)
        scrollbarh.config(command=self.txt_events_log.xview)


        # TAB Help
        # text box to display the help in the notebook, with vertical scroll bar and search widgets
        # a main frame is used to encapsulate two other frames, one for the search widgets, one for the text box and his scroll bar
        self.frm_help = ttk.Frame(self.notebook)
        self.frm_help.pack(fill=tk.BOTH, expand=True)
        # widgets to search a text
        self.frm_help_top = ttk.Frame(self.frm_help)
        self.frm_help_top.pack(side=tk.TOP, fill=tk.X)
        self.lab_search = ttk.Label(self.frm_help_top, text='Search :', borderwidth=0, relief=tk.SOLID, anchor=tk.E)
        self.lab_search.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X)
        self.cmb_search_text = ttk.Combobox(self.frm_help_top, height=24, width=20, values=['[Organ]', '[Button]', '[Coupler999]', '[Divisional999]', '[DivisionalCoupler999]', '[DrawStop]', '[Enclosure999]', '[General999]', '[Image999]', '[Label999]', '[Manual999]', '[Panel999]', '[Panel999Element999]', '[Panel999Image999]', '[Panel999xxxxx999]', '[Piston]', '[PushButton]', '[Rank999]', '[ReversiblePiston999]', '[SetterElement999]', '[Stop999]', '[Switch999]', '[Tremulant999]', '[WindchestGroup999]'])
        self.cmb_search_text.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X)
        self.cmb_search_text.bind('<KeyRelease>', self.help_search_text_key_pressed)
        self.cmb_search_text.bind('<<ComboboxSelected>>', self.help_search_text_key_pressed)
        self.btn_search_prev = ttk.Button(self.frm_help_top, text='<', width=5, state=tk.NORMAL, command=self.help_search_previous)
        self.btn_search_prev.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X)
        self.btn_search_next = ttk.Button(self.frm_help_top, text='>', width=5, state=tk.NORMAL, command=self.help_search_next)
        self.btn_search_next.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X)
        self.btn_search_clear = ttk.Button(self.frm_help_top, text='Clear', width=10, state=tk.NORMAL, command=self.help_search_clear)
        self.btn_search_clear.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X)
        self.lab_search_occur_nb = ttk.Label(self.frm_help_top, text='', borderwidth=0, relief=tk.SOLID, anchor=tk.W)
        self.lab_search_occur_nb.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X)
        # help text box
        self.frm_help_bottom = ttk.Frame(self.frm_help)
        self.frm_help_bottom.pack(side=tk.BOTTOM, fill=tk.BOTH, expand=True)
        scrollbarv = ttk.Scrollbar(self.frm_help_bottom, orient=tk.VERTICAL)
        scrollbarv.pack(side=tk.RIGHT, fill=tk.Y)
        self.txt_help = tk.Text(self.frm_help_bottom, fg=TEXT_COLOR, bg=COLOR_BG_HELP, bd=3, wrap=tk.WORD, font=TEXT_FONT, selectbackground=COLOR_BG_TEXT_SEL)
        self.txt_help.pack(side=tk.BOTTOM, fill=tk.BOTH, expand=True)
        self.txt_help.config(yscrollcommand=scrollbarv.set)
        scrollbarv.config(command=self.txt_help.yview)
        # define the tags for the syntax highlighting
        self.txt_help.tag_config(TAG_FIELD, foreground=COLOR_TAG_FIELD)
        self.txt_help.tag_config(TAG_COMMENT, foreground=COLOR_TAG_COMMENT)
        self.txt_help.tag_config(TAG_OBJ_UID, foreground=COLOR_TAG_OBJ_UID, font=TEXT_FONT_BOLD)
        self.txt_help.tag_config(TAG_TITLE, foreground=COLOR_TAG_TITLE, font=TEXT_FONT_BOLD)


        # TAB Search & replace
        # list to search in the GO or HW ODF and display the search results, with vertical scroll bar
        # a main frame is used to encapsulate two other frames, one for the search widgets, one for the list box and his vertical scroll bar
        self.frm_search = ttk.Frame(self.notebook)
        self.frm_search.pack(fill=tk.BOTH, expand=True)

        # widgets to search a text
        # row 1 (radio buttons for search scope selection)
        self.frm_search_top2 = ttk.Frame(self.frm_search)
        self.frm_search_top2.pack(side=tk.TOP, fill=tk.X)

        self.lab_search = ttk.Label(self.frm_search_top2, text='Search in', borderwidth=0, relief=tk.SOLID, anchor=tk.E)
        self.lab_search.pack(side=tk.LEFT, padx=5, pady=1, fill=tk.X)

        self.odf_search_range = tk.StringVar(self.wnd_main)
        radiobutton = ttk.Radiobutton(self.frm_search_top2, text='whole ODF', variable=self.odf_search_range, value='odf')
        radiobutton.pack(side=tk.LEFT, padx=5, pady=1, fill=tk.X)
        radiobutton = ttk.Radiobutton(self.frm_search_top2, text='selected section', variable=self.odf_search_range, value='selected')
        radiobutton.pack(side=tk.LEFT, padx=5, pady=1, fill=tk.X)
        radiobutton = ttk.Radiobutton(self.frm_search_top2, text='children of selected section', variable=self.odf_search_range, value='children')
        radiobutton.pack(side=tk.LEFT, padx=5, pady=1, fill=tk.X)
        self.odf_search_range.set('odf')

        # row 2 (check buttons for regular expression or case sensitive slection)
        self.frm_search_top1 = ttk.Frame(self.frm_search)
        self.frm_search_top1.pack(side=tk.TOP, fill=tk.X)

        self.odf_search_regex = tk.BooleanVar(self.wnd_main)
        self.btn_odf_search_regex = ttk.Checkbutton(self.frm_search_top1, text='Regular expression', width=18, variable=self.odf_search_regex, command=self.gui_status_update_notebook)
        self.btn_odf_search_regex.pack(side=tk.LEFT, padx=2, pady=1, fill=tk.X)

        self.odf_search_case_sensitive = tk.BooleanVar(self.wnd_main)
        self.btn_odf_search_case_ins = ttk.Checkbutton(self.frm_search_top1, text='Case sensitive', width=16, variable=self.odf_search_case_sensitive)
        self.btn_odf_search_case_ins.pack(side=tk.LEFT, padx=2, pady=1, fill=tk.X)

        self.btn_odf_search_hw = ttk.Button(self.frm_search_top1, text='HW search', state=tk.NORMAL, command=self.odf_search_text_hw)

        # row 3 (entry for text to search, search button, found occurrences number)
        self.frm_search_top3 = ttk.Frame(self.frm_search)
        self.frm_search_top3.pack(side=tk.TOP, fill=tk.X)

        self.ent_odf_search_text = tk.Entry(self.frm_search_top3, width=45)
        self.ent_odf_search_text.pack(side=tk.LEFT, padx=2, pady=1, fill=tk.X)
        self.ent_odf_search_text.bind('<KeyPress-Return>', self.odf_search_text)
        self.ent_odf_search_text.bind('<KeyPress-KP_Enter>', self.odf_search_text)

        self.ent_odf_search_text.bind('<Button-1>', self.odf_search_text_selected)
        self.ent_odf_search_text.bind('<KeyPress>', self.odf_search_text_selected)

        self.btn_odf_search = ttk.Button(self.frm_search_top3, text='Search', width=7, state=tk.NORMAL, command=self.odf_search_text)
        self.btn_odf_search.pack(side=tk.LEFT, padx=2, pady=1, fill=tk.X)

        self.lab_search_results_nb = ttk.Label(self.frm_search_top3, text='', borderwidth=0, anchor=tk.W)
        self.lab_search_results_nb.pack(side=tk.LEFT, padx=10, pady=1, fill=tk.X)

        # row 4 (entry for text to replace, replace button, clear button)
        self.frm_search_top4 = ttk.Frame(self.frm_search)
        self.frm_search_top4.pack(side=tk.TOP, fill=tk.X)

        self.ent_odf_replace_text = tk.Entry(self.frm_search_top4, width=45)
        self.ent_odf_replace_text.pack(side=tk.LEFT, padx=2, pady=2, fill=tk.X)
        self.ent_odf_replace_text.bind('<Button-1>', self.odf_replace_text_selected)
        self.ent_odf_replace_text.bind('<KeyPress>', self.odf_replace_text_selected)

        self.btn_odf_replace = ttk.Button(self.frm_search_top4, text='Replace', width=7, state=tk.NORMAL, command=self.odf_replace_text)
        self.btn_odf_replace.pack(side=tk.LEFT, padx=2, pady=2, fill=tk.X)

        self.btn_odf_search_clear = ttk.Button(self.frm_search_top4, text='Clear', width=7, state=tk.NORMAL, command=self.odf_search_clear)
        self.btn_odf_search_clear.pack(side=tk.LEFT, padx=6, pady=2, fill=tk.X)

        # search results list box
        self.frm_search_bottom = ttk.Frame(self.frm_search)
        self.frm_search_bottom.pack(side=tk.BOTTOM, fill=tk.BOTH, expand=True)
        scrollbarv = ttk.Scrollbar(self.frm_search_bottom, orient=tk.VERTICAL)
        scrollbarv.pack(side=tk.RIGHT, fill=tk.Y)
        self.lst_odf_sresults = tk.Listbox(self.frm_search_bottom, bg=COLOR_BG_SEARCH, font=TEXT_FONT, fg=TEXT_COLOR, selectbackground=COLOR_SELECTED_ITEM,
                                           exportselection=0, selectmode=tk.SINGLE, activestyle=tk.DOTBOX)
        self.lst_odf_sresults.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        self.lst_odf_sresults.bind('<ButtonRelease-1>', self.odf_search_text_result_selected)
        self.lst_odf_sresults.bind('<Double-1>', self.odf_search_text_result_selected_dbl)
        self.lst_odf_sresults.config(yscrollcommand=scrollbarv.set)
        scrollbarv.config(command=self.lst_odf_sresults.yview)

        self.odf_search_clear()

        # TAB Viewer
        # viewer to show image or play wav file or show panel content
        self.frm_viewer = ttk.Frame(self.notebook)
        self.frm_viewer.pack(fill=tk.BOTH, expand=True)
        self.frm_viewer.bind('<Configure>', self.viewer_content_update)
        self.view_canvas = tk.Canvas(self.frm_viewer, bg=COLOR_BACKGROUND0)
        self.view_canvas.pack(side=tk.TOP, fill=tk.BOTH, expand=True)
        self.view_canvas.bind('<MouseWheel>', self.viewer_content_update)  # for Windows
        self.view_canvas.bind('<Button-4>', self.viewer_content_update)    # MouseWheel up in Linux
        self.view_canvas.bind('<Button-5>', self.viewer_content_update)    # MouseWheel down in Linux
        self.view_canvas.bind('<ButtonPress-1>', lambda event: self.view_canvas.scan_mark(event.x, event.y))
        self.view_canvas.bind('<B1-Motion>', self.viewer_content_drag)


        # TAB HW sections
        # list to navigate inside the Hauptwerk objects in the ODF, with vertical scroll bar
        # a main frame is used to encapsulate two other frames, one for the search widgets, one for the list box and his vertical scroll bar
        self.frm_hw_browser = ttk.Frame(self.notebook)
        self.frm_hw_browser.pack(fill=tk.BOTH, expand=True)
        # widgets to search an object UID
        self.frm_hw_browser_top = ttk.Frame(self.frm_hw_browser)
        self.frm_hw_browser_top.pack(side=tk.TOP, fill=tk.X)
        self.ent_hw_uid_search_text = tk.Entry(self.frm_hw_browser_top, width=40)
        self.ent_hw_uid_search_text.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X)
        self.ent_hw_uid_search_text.bind('<KeyPress-Return>', self.odf_search_uid_hw)
        self.ent_hw_uid_search_text.bind('<KeyPress-KP_Enter>', self.odf_search_uid_hw)

        self.btn_hw_uid_search = ttk.Button(self.frm_hw_browser_top, text='Go to UID', state=tk.NORMAL, command=self.odf_search_uid_hw)
        self.btn_hw_uid_search.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X)
        # browser list box
        self.frm_hw_browser_bottom = ttk.Frame(self.frm_hw_browser)
        self.frm_hw_browser_bottom.pack(side=tk.BOTTOM, fill=tk.BOTH, expand=True)
        scrollbarv = ttk.Scrollbar(self.frm_hw_browser_bottom, orient=tk.VERTICAL)
        scrollbarv.pack(side=tk.RIGHT, fill=tk.Y)
        self.lst_hw_browser = tk.Listbox(self.frm_hw_browser_bottom, bg='alice blue', font=TEXT_FONT, fg=TEXT_COLOR, selectbackground=COLOR_SELECTED_ITEM,
                                         exportselection=0, selectmode=tk.SINGLE, activestyle=tk.DOTBOX)
        self.lst_hw_browser.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        self.lst_hw_browser.bind('<ButtonRelease>', self.objects_list_selected_hw)
        self.lst_hw_browser.bind('<Double-1>', self.objects_list_selected_dbl_hw)
        self.lst_hw_browser.config(yscrollcommand=scrollbarv.set)
        scrollbarv.config(command=self.lst_hw_browser.yview)

        # create the notebook tabs, and attach the frames to them
        self.notebook.add(self.frm_logs, text='  Logs  ')
        self.notebook.add(self.frm_help, text='  Help  ')
        self.notebook.add(self.frm_search, text='  Search&replace  ')
        self.notebook.add(self.frm_viewer, text='  Viewer  ')
        self.notebook.add(self.frm_hw_browser, text='  HW sections  ')
        self.notebook.hide(self.frm_hw_browser)  # will be visible only if a Hauptwerk ODF is loaded

        # to initialize the content of the viewer
        self.viewer_file_show()

    #-------------------------------------------------------------------------------------------------
    def gui_status_update_notebook(self):
        # update the status of GUI widgets of the search function in the notebook

        # buttons to search previous or next or clear the search in the help
        is_search_text = self.cmb_search_text.get() != ''
        self.btn_search_prev['state']  = tk.NORMAL if is_search_text else tk.DISABLED
        self.btn_search_next['state']  = tk.NORMAL if is_search_text else tk.DISABLED
        self.btn_search_clear['state'] = tk.NORMAL if is_search_text else tk.DISABLED
        self.btn_odf_search_case_ins['state']  = tk.NORMAL if (not self.odf_search_regex.get()) else tk.DISABLED

        if self.is_loaded_hw_odf:
            self.btn_odf_search_hw.pack(side=tk.LEFT, padx=0, pady=5, fill=tk.X)
        else:
            self.btn_odf_search_hw.pack_forget()

        # HW objects list
        if self.selected_object_app != 'HW':
            # clear the selected items if it is not a HW object which is selected
            self.lst_hw_browser.selection_clear(0, 'end')

    #-------------------------------------------------------------------------------------------------
    def logs_update(self):
        # add in the logs text box widget the content of the logs buffer then clear it

        logs_list = logs.get()
        if len(logs_list) > 0:
            self.txt_events_log.insert('end', '\n' + '\n'.join(logs_list) + '\n')
            self.txt_events_log.see('end-1c linestart')  # to see the start of the last line of the text
        self.txt_events_log.update()

        # reset the logs buffer
        logs.clear()

    #-------------------------------------------------------------------------------------------------
    def logs_select_all(self, event):
        # (GUI event callback) the user has pressed the Ctrl+a or Ctrl+A keys combinaison in the logs text box to select all the text
        # in Windows it is managed natively by the text box widget, but not in Linux

        self.txt_events_log.tag_add('sel', '1.0', 'end')
        return 'break'  # do not process further the event in the widget

    #-------------------------------------------------------------------------------------------------
    def logs_clear(self):
        # (GUI event callback) the user has selected 'Clear all' in the context menu of the logs text box

        # clear the content of the logs text box
        self.txt_events_log.delete(1.0, "end")

        # reset the logs buffer
        logs.clear()

    #-------------------------------------------------------------------------------------------------
    def help_file_load(self):
        # load in the help text box widget the help file
        # done one time at the application start

        file_name = os.path.dirname(__file__) + os.path.sep + 'resources' + os.path.sep + 'Help.txt'

        try:
            with open(file_name, 'r') as f:   # errors='ignore'
                f.seek(0)
                # copy in the widget the help text
                self.txt_help.insert(1.0, f.read())
        except OSError as err:
            # it has not be possible to open the file
            logs.add(f'ERROR : error while opening the file "{file_name}"\nError number {err.errno} : {err.strerror}')
            return False
        except: # handle other exceptions such as attribute errors
            logs.add(f'ERROR : unexpected error while opening the file "{file_name}"\n{sys.exc_info()[0]}')
            return False

        # apply the ODF syntax highlighting
        self.odf_syntax_highlight(self.txt_help)
        # disable the text box to not permit its editing
        self.txt_help.configure(state='disabled')
        return True

    #-------------------------------------------------------------------------------------------------
    def help_search_next(self):
        # (GUI event callback) the user has clicked on the button '>'
        # show the next occurence of the text to search

        self.help_search_text(self.cmb_search_text.get(), True)

    #-------------------------------------------------------------------------------------------------
    def help_search_previous(self):
        # (GUI event callback) the user has clicked on the button '<'
        # show the previous occurence of the text to search

        self.help_search_text(self.cmb_search_text.get(), False)

    #-------------------------------------------------------------------------------------------------
    def help_search_clear(self):
        # (GUI event callback) the user has clicked on the button 'Clear'
        # clear the text to search (text box and highlighting)

        self.cmb_search_text.delete(0, tk.END)
        self.help_search_text('', False)
        self.lab_search_occur_nb.config(text='')

        # update the status of some GUI widgets
        self.gui_status_update_notebook()

    #-------------------------------------------------------------------------------------------------
    def help_search_object(self):
        # (GUI event callback) the user has clicked on the button "Show in help"
        # search and display in the help the part describing the selected object UID

        if self.edited_object_uid not in (None, 'Header'):
            # substitute the digits by the char 9 in the object UID to create a generic object ID
            gen_object_UID = '['
            for c in self.edited_object_uid: gen_object_UID += '9' if c.isdigit() else c
            gen_object_UID += ']'

            # put the generic object UID in the search text widget
            self.cmb_search_text.delete(0, tk.END)
            self.cmb_search_text.insert(0, gen_object_UID)

            # update the status of GUI widgets
            self.gui_status_update_notebook()

            # search the first occurence of the generic object ID
            self.help_search_next()

            # select the Help tab of the notebook
            self.notebook.select(self.frm_help)

    #-------------------------------------------------------------------------------------------------
    def help_search_text(self, text_to_find, search_next = True):
        # show in the help the next occurence (or previous if search_next=False) of the given text to find
        # highlight in yellow all the occurences of this text

        if text_to_find != '':
            text_len = len(text_to_find)
        else:
            text_to_find = None
            text_len = 0

        if text_to_find != self.text_to_search:
            # a new text has to be searched, highlight all its occurences in the entire help text

            # store the new text to search
            self.text_to_search = text_to_find

            # remove the highlight of the previous searched text occurences
            self.txt_help.tag_remove(TAG_FOUND, '1.0', tk.END)
            self.txt_help.tag_remove(TAG_FOUND2, '1.0', tk.END)
            nb_occurences = 0

            if text_to_find != None:
                # highlight the all occurences of the text to find in the entire help
                # configure the tag for the found text
                self.txt_help.tag_config(TAG_FOUND, foreground=TEXT_COLOR, background=COLOR_TAG_FOUND, font=TEXT_FONT)
                # get the lines of the text widget
                lines = self.txt_help.get('1.0', tk.END).splitlines()
                # scan all the lines
                for l, line in enumerate(lines):
                    idx = 0
                    while idx != -1:
                        # check the various occurences of the searched text in the current line (if any)
                        idx = line.find(text_to_find, idx)
                        if idx != -1:
                            # highlight the found occurence in the line
                            self.txt_help.tag_add(TAG_FOUND, f'{l+1}.{idx}', f'{l+1}.{idx} + {text_len} chars')
                            # move the search index after the found occurence
                            idx += text_len
                            nb_occurences += 1

                # display the number of occurences of the searched text
                if nb_occurences == 0:
                    self.lab_search_occur_nb.config(text='None occurence')
                elif nb_occurences == 1:
                    self.lab_search_occur_nb.config(text='1 occurence')
                else:
                    self.lab_search_occur_nb.config(text=f'{nb_occurences} occurences')

        if text_to_find != None:
            # search for the next/previous occurence of the text in the help

            # remove the highlight of the previous searched text occurence
            self.txt_help.tag_remove(TAG_FOUND2, '1.0', tk.END)
            # configure the tag for the highlighted found text
            self.txt_help.tag_config(TAG_FOUND2, foreground=TEXT_COLOR, background=COLOR_TAG_FOUND2, font=TEXT_FONT)

            if search_next and self.search_index != None:
                # if search upward, move the current search position after the previous found position
                self.search_index = f'{self.search_index} + {text_len} chars'

            # search for the next/previous occurence
            if self.search_index == None: self.search_index = '1.0'
            self.search_index = self.txt_help.search(text_to_find, self.search_index, backwards = not search_next)
            if self.search_index == '': self.search_index = None

            if self.search_index != None:
                # show and highlight the found text
                self.txt_help.see(self.search_index)
                self.txt_help.tag_add(TAG_FOUND2, self.search_index, f'{self.search_index} + {text_len} chars')

    #-------------------------------------------------------------------------------------------------
    def help_search_text_key_pressed(self, event):
        # (GUI event callback) the user has pressed a keyboard key in the help text search box

        # update the status of GUI widgets
        self.gui_status_update_notebook()

        self.search_index = None  # restart the search at the beginning of the help text
        self.help_search_text(self.cmb_search_text.get(), True)

    #-------------------------------------------------------------------------------------------------
    def odf_search_text(self, event=None, text_to_replace=None):
        # (GUI event callback) the user has clicked on the button "Search" of the "Search/replace" tab or pressed Enter in the search entry box
        # called by the odf_replace_text function, in this case text_to_replace provides the replacement text for the text to search
        # return True or False wether text has been found

        # recover the text to search
        text_to_search = self.ent_odf_search_text.get()

        # clear the search results list
        self.lst_odf_sresults.delete(0, tk.END)
        self.lab_search_results_nb.config(text='')

        if text_to_search != '':
            # fill the list of objects UID in which to do the search, depending on the user's choice
            objects_uid_search_list = []
            if self.odf_search_range.get() == 'odf':
                # the search has to be done in the whole ODF
                objects_uid_search_list = self.odf_data.objects_list_get()
            else:
                # the search range is depending on the selected object UID
                if self.edited_object_uid != None:
                    # an object is selected
                    if self.odf_search_range.get() == 'selected':
                        # the search has to be done inside the selected object
                        objects_uid_search_list.append(self.edited_object_uid)
                    else:
                        # the search has to be done in the children of the selected object
                        objects_uid_search_list = self.odf_data.object_kinship_list_get(self.edited_object_uid, TO_CHILD)
                        if len(objects_uid_search_list) == 0:
                            if self.edited_object_uid == 'Organ':
                                # the Organ object has no children functionnaly, take all the objects but the Header and Organ
                                objects_uid_search_list = list(self.odf_data.objects_list_get())
                                objects_uid_search_list.remove('Header')
                                objects_uid_search_list.remove('Organ')
                            else:
                                self.lst_odf_sresults.insert(tk.END, f'{self.edited_object_uid} has no children.')
                                return False
                else:
                    self.lst_odf_sresults.insert(tk.END, 'There is none selected object to search in.')
                    return False

            results_list = []
            reg_ex = self.odf_search_regex.get()
            case_sens = self.odf_search_case_sensitive.get()
            text_to_search_up = text_to_search.upper()
            for object_uid in objects_uid_search_list:
                # scan the objects to search in
                object_dic = self.odf_data.object_dic_get(object_uid)
                for i, line in enumerate(object_dic['lines']):
                    # scan the lines of the current object
                    if ((reg_ex and re.search(text_to_search, line)) or
                        (not reg_ex and ((not case_sens and text_to_search_up in line.upper()) or
                                         (case_sens and text_to_search in line)))):
                        # text found in the current line
                        if text_to_replace != None:
                            # the found text has to be replaced
                            if reg_ex:
                                line = re.sub(text_to_search, text_to_replace, line)
                            else:
                                line = line.replace(text_to_search, text_to_replace)
                            # update the current line of the current object
                            object_dic['lines'][i] = line
                        # add the found (and replaced) line to the search results list
                        results_list.append(f'{object_uid} : {line}')
            results_list.sort()

            if len(results_list) > 0:
                # items have been found
                self.lst_odf_sresults.insert(tk.END, *results_list)
                if len(results_list) > 1:
                    self.lab_search_results_nb.config(text=f'{len(results_list)} results')
                else:
                    self.lab_search_results_nb.config(text=f'{len(results_list)} result')
                if text_to_replace != None:
                    if self.odf_search_range.get() == 'odf':
                        logs.add(f'"{text_to_search}" has been replaced by "{text_to_replace}" in the whole ODF')
                    elif self.odf_search_range.get() == 'selected':
                        logs.add(f'"{text_to_search}" has been replaced by "{text_to_replace}" in {self.edited_object_uid}')
                    else:
                        logs.add(f'"{text_to_search}" has been replaced by "{text_to_replace}" in children of {self.edited_object_uid}')
                return True

            if case_sens:
                self.lst_odf_sresults.insert(tk.END, f'"{text_to_search}" is not found in the ODF (the search is case sensitive)')
            else:
                self.lst_odf_sresults.insert(tk.END, f'"{text_to_search}" is not found in the ODF')

        return False

    #-------------------------------------------------------------------------------------------------
    def odf_replace_text(self, event=None):
        # (GUI event callback) the user has clicked on the button "Replace" of the "Search/replace" tab

        # recover the replacement text
        replacement_text = self.ent_odf_replace_text.get()

        if replacement_text == 'replacement text':
            replacement_text = ''

        if self.odf_search_text(None, replacement_text):
            # the search and replace has been successful
            self.odf_data_changed = True
            # update the links between the objects and their displayed names
            self.odf_data.objects_kinship_update()
            # update the content of GUI widgets
            self.objects_list_update()
            self.objects_tree_update()
            self.object_links_list_update()
            self.object_text_update()
            self.gui_status_update_lists()
            self.gui_status_update_buttons()

            # update the events log text
            self.logs_update()

    #-------------------------------------------------------------------------------------------------
    def odf_search_clear(self, event=None):
        # (GUI event callback) the user has clicked on the button "Clear" in the ODF search tab

        # clear the search results list
        self.lst_odf_sresults.delete(0, tk.END)
        self.lab_search_results_nb.config(text='')

        self.ent_odf_search_text['fg'] = COLOR_BACKGROUND1
        self.ent_odf_search_text.delete(0, tk.END)
        self.ent_odf_search_text.insert(0, 'text to search')

        self.ent_odf_replace_text['fg'] = COLOR_BACKGROUND1
        self.ent_odf_replace_text.delete(0, tk.END)
        self.ent_odf_replace_text.insert(0, 'replacement text')

    #-------------------------------------------------------------------------------------------------
    def odf_search_text_selected(self, event=None):
        # (GUI event callback) the user has clicked on the entry widget to set a text to search

        if self.ent_odf_search_text['fg'] == COLOR_BACKGROUND1:
            self.ent_odf_search_text['fg'] = TEXT_COLOR
            self.ent_odf_search_text.delete(0, 'end')

    #-------------------------------------------------------------------------------------------------
    def odf_replace_text_selected(self, event=None):
        # (GUI event callback) the user has clicked on the entry widget to set the replacement text

        if self.ent_odf_replace_text['fg'] == COLOR_BACKGROUND1:
            self.ent_odf_replace_text['fg'] = TEXT_COLOR
            self.ent_odf_replace_text.delete(0, 'end')

    #-------------------------------------------------------------------------------------------------
    def odf_search_text_hw(self, event=None):
        # (GUI event callback) the user has clicked on the button "HW search" of the "Search in ODF" tab

        # recover the text to search
        text_to_search = self.ent_odf_search_text.get()

        self.lst_odf_sresults.delete(0, tk.END)
        self.lab_search_results_nb.config(text='')

        if text_to_search != '':
            results_list = []
            reg_ex = self.odf_search_regex.get()
            case_sens = self.odf_search_case_sensitive.get()
            text_to_search_up = text_to_search.upper()
            for object_type_dic in self.odf_hw2go.HW_odf_dic.values():
                # scan the various object types
                for HW_object_dic in object_type_dic.values():
                    # scan the objects of the current object type
                    if ((reg_ex and re.search(text_to_search, HW_object_dic['_uid'])) or
                        (not reg_ex and ((not case_sens and text_to_search_up in HW_object_dic['_uid'].upper()) or
                                         (case_sens and text_to_search in HW_object_dic['_uid'])))):
                        # text found in the object UID
                        results_list.append(f"{HW_object_dic['_uid']}")

                    for obj_attr_name, obj_attr_value in HW_object_dic.items():
                        # scan the attributes of the current object
                        if not isinstance(obj_attr_value, list) and not isinstance(obj_attr_value, dict): # skip the _parents / _children lists and HW dictionary reference
                            line = obj_attr_name + '=' + str(obj_attr_value)
                            if ((reg_ex and re.search(text_to_search, line)) or
                                (not reg_ex and ((not case_sens and text_to_search_up in line.upper()) or
                                                 (case_sens and text_to_search in line)))):
                                # text found in the current object attribute line
                                results_list.append(f"{HW_object_dic['_uid']} : {obj_attr_name}={obj_attr_value}")
            results_list.sort()

            if len(results_list) > 0:
                if len(results_list) > 1:
                    self.lab_search_results_nb.config(text=f'{len(results_list)} results')
                else:
                    self.lab_search_results_nb.config(text=f'{len(results_list)} result')
                self.lst_odf_sresults.insert(tk.END, *results_list)
            else:
                if case_sens:
                    self.lst_odf_sresults.insert(tk.END, f'"{text_to_search}" is not found in the HW ODF (the search is case sensitive)')
                else:
                    self.lst_odf_sresults.insert(tk.END, f'"{text_to_search}" is not found in the HW ODF')

    #-------------------------------------------------------------------------------------------------
    def odf_search_uid_hw(self, event=None):
        # (GUI event callback) the user has clicked on the button "UID search" of the HW ODF browser

        # recover the text to search (must be a HW object UID only)
        text_to_search = self.ent_hw_uid_search_text.get()

        if text_to_search != '':
            object_dic = self.odf_hw2go.HW_ODF_get_object_dic_from_uid(text_to_search)
            if object_dic != None:
                # the UID is existing in the HW ODF
                self.selected_object_app = 'HW'
                self.selected_object_uid = text_to_search
                self.selected_linked_uid = None
                self.edited_object_uid = text_to_search
                self.focused_objects_widget = self.lst_hw_browser
                self.focused_sel_item_id = text_to_search

                # update the status of GUI widgets
                self.object_links_list_update()
                self.objects_list_update_hw()
                self.object_text_update()
                self.gui_status_update_lists()
                self.gui_status_update_buttons()
            else:
                AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f'"{text_to_search}" is not a known HW UID.', ['Close'], 'Close')
        else:
            AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', 'Enter a HW object UID first.', ['Close'], 'Close')

    #-------------------------------------------------------------------------------------------------
    def odf_search_text_result_selected(self, event):
        # (GUI event callback) the user has clicked on an item of the ODF search results list

        # get the selected indice
        selected_indice = self.lst_odf_sresults.curselection()

        if self.can_i_make_change() and len(selected_indice) > 0 and not self.lst_odf_sresults.get(selected_indice[0]).startswith('"'):
            # the user has saved his modifications if he wanted and has not canceled the operation
            selected_text = self.lst_odf_sresults.get(selected_indice[0])

            self.selected_object_uid = selected_text.split(' ')[0]
            self.selected_linked_uid = None
            self.edited_object_uid = self.selected_object_uid

            if self.selected_object_uid == '_General' or self.selected_object_uid[-6:].isdigit():
                # the results are concerning HW UID
                self.selected_object_app = 'HW'
            else:
                self.selected_object_app = 'GO'

            # update the object text box and links list
            self.object_text_update()
            self.object_links_list_update()

            # select in the object edition text box the line which corresponds to the selected result
            if ':' in selected_text:
                selected_search_result = selected_text.split(':', maxsplit=1)[1].strip()
                idx = self.txt_object_text.search(selected_search_result, '1.0', stopindex=tk.END)
                if idx != '':
                    self.txt_object_text.tag_remove('sel', '1.0', 'end')
                    self.txt_object_text.tag_add('sel', idx, f'{idx} + {len(selected_search_result)} chars')
                    self.txt_object_text.see(idx)
                    self.txt_object_text.focus_set()

            # update the status of GUI widgets
            self.gui_status_update_lists(True)
            self.gui_status_update_buttons()

    #-------------------------------------------------------------------------------------------------
    def odf_search_text_result_selected_dbl(self, event):
        # (GUI event callback) the user has double-clicked on an item of the ODF search results list

        selected_indice = self.lst_odf_sresults.curselection()
        selected_text = self.lst_odf_sresults.get(selected_indice[0])
        self.selected_object_uid = selected_text.split(' ')[0]

        if self.selected_object_uid[-6:].isdigit():
            # the selected object is a HW object, display the "HW sections" tab of the notebook
            self.objects_list_update_hw()
            self.ent_hw_uid_search_text.delete(0, tk.END)
            self.ent_hw_uid_search_text.insert(0, self.selected_object_uid)
            self.notebook.select(self.frm_hw_browser)

    #-------------------------------------------------------------------------------------------------
    def viewer_file_show(self, line=None, app='GO', object_uid=None):
        # shows in the viewer tab the file defined in the given object line if any and if valid
        # app can be 'GO' or 'HW', it indicates which kind of ODF data it is (GrandOrgue or Hauptwerk)

        label_text = 'Click in the text editor on a line where is defined a bitmap or wave sample file'
        file_name = None
        pipe_pitch_tuning = None

        self.viewer_file_type = None

        if line != None:
            # read the content of the given line to determine if it contains a file to show in the viewer and which kind of file
            (error_msg, attr_name, attr_value, comment) = self.odf_data.object_line_split(line)
            if error_msg == None and attr_value != None and len(attr_value) > 0:
                # the line has been split with success
                if any(x in attr_name for x in ['Image', 'Mask', 'Bitmap']) and (attr_value.upper()[-4:] in ('.BMP', '.GIF', '.JPG', '.ICO', '.PNG')):
                    # it is a supported image file extension
                    self.viewer_file_type = 'image'
                    file_name = attr_value

                elif any(x in attr_name for x in ['Pipe', 'SampleFilename']):
                    if attr_value.upper()[-4:] == '.WAV':
                        # it is a wave file
                        self.viewer_file_type = 'sample'
                        file_name = attr_value
                    elif attr_value[:4] == 'REF:':
                        # it is the referencing REF:xx:xx:xx to borrow the wave file of another manual/stop/pipe
                        # read the three references
                        refs_list = attr_value[4:].split(':')
                        if len(refs_list) < 3 or not (refs_list[0].isdigit() and refs_list[1].isdigit() and refs_list[2].isdigit()):
                            label_text = f'ERROR : three numerical references are expected in the pipe referencing {attr_value}, at the format REF:aa:bb:cc'
                        else:
                            # recover the manual UID, the stop number in the manual, the pipe number in the stop or fist rank of the stop
                            manual_nb = int(refs_list[0])
                            manual_uid = 'Manual' + str(manual_nb).zfill(3)
                            stop_nb = int(refs_list[1])
                            pipe_nb = int(refs_list[2])
                            if self.odf_data.object_dic_get(manual_uid) == None:
                                label_text = f'ERROR : the section {manual_uid} does not exist'
                            elif stop_nb < 1:
                                label_text = f'ERROR : the stop number cannot be lower than 1, it is set at {stop_nb}'
                            elif pipe_nb < 1:
                                label_text = f'ERROR : the pipe number cannot be lower than 1, it is set at {pipe_nb}'
                            else:
                                # search for the appropriate stop UID in the list of the child stops of the manual
                                manual_stops_list = sorted(self.odf_data.object_kinship_list_get(manual_uid, TO_CHILD, 'Stop'))
                                if len(manual_stops_list) < stop_nb:
                                    label_text = f'ERROR : there is no stop number {stop_nb} in the manual {manual_nb}'
                                else:
                                    stop_uid = manual_stops_list[stop_nb - 1]
                                    if self.odf_data.object_dic_get(stop_uid) == None:
                                        label_text = f'ERROR : there is no section {stop_uid} defined'
                                    else:
                                        # proper stop object is found, recover the sample file in the stop object if defined
                                        pipe_id = 'Pipe' + str(pipe_nb).zfill(3)
                                        sample_file = self.odf_data.object_attr_value_get(stop_uid, pipe_id)
                                        pipe_pitch_tuning = myint(self.odf_data.object_attr_value_get(stop_uid, pipe_id + 'PitchTuning'))
                                        if sample_file == None:
                                            # there is no Pipe attribute in the stop, recover the sample file in the first rank
                                            rank_id = myint(self.odf_data.object_attr_value_get(stop_uid, 'Rank001'))
                                            if rank_id == None:
                                                label_text = f'ERROR : there is neither attribute {pipe_id} nor attribute Rank001 defined in the section {stop_uid}'
                                            else:
                                                # recover the sample file in the rank object if defined
                                                rank_uid = 'Rank' + str(rank_id).zfill(3)
                                                sample_file = self.odf_data.object_attr_value_get(rank_uid, pipe_id)
                                                pipe_pitch_tuning = myint(self.odf_data.object_attr_value_get(rank_uid, pipe_id + 'PitchTuning'))
                                        if sample_file != None:
                                            self.viewer_file_type = 'sample'
                                            file_name = sample_file

            if file_name != None:
                # a file name with expected extension has been extracted from the given line, check if it exists
                file_full_path = ''
                file_name = path2ospath(file_name)
                if app == 'GO':
                    file_full_path = os.path.dirname(self.odf_data.odf_file_name) + os.path.sep + file_name
                elif app == 'HW':
                    if object_uid.startswith('ImageSetElement'):
                        HW_image_set_elem_dic = self.odf_hw2go.HW_ODF_get_object_dic_from_uid(object_uid)
                        HW_image_set_dic = self.odf_hw2go.HW_ODF_get_linked_objects_dic_by_type(HW_image_set_elem_dic, 'ImageSet', TO_PARENT, FIRST_ONE)
                        HW_package_id = self.odf_hw2go.HW_ODF_get_attribute_value(HW_image_set_dic, 'InstallationPackageID').zfill(6)
                        HW_sample_set_path = self.odf_hw2go.HW_ODF_get_attribute_value(self.odf_hw2go.HW_general_dic, '_sample_set_path')
                        file_name = 'OrganInstallationPackages' + os.path.sep + HW_package_id + os.path.sep + file_name
                        file_full_path = HW_sample_set_path + os.path.sep + file_name
                    elif object_uid.startswith('ImageSet'):
                        HW_image_set_dic = self.odf_hw2go.HW_ODF_get_object_dic_from_uid(object_uid)
                        HW_package_id = self.odf_hw2go.HW_ODF_get_attribute_value(HW_image_set_dic, 'InstallationPackageID').zfill(6)
                        HW_sample_set_path = self.odf_hw2go.HW_ODF_get_attribute_value(self.odf_hw2go.HW_general_dic, '_sample_set_path')
                        file_name = 'OrganInstallationPackages' + os.path.sep + HW_package_id + os.path.sep + file_name
                        file_full_path = HW_sample_set_path + os.path.sep + file_name
                    elif object_uid.startswith('Sample'):
                        HW_sample_dic = self.odf_hw2go.HW_ODF_get_object_dic_from_uid(object_uid)
                        HW_package_id = self.odf_hw2go.HW_ODF_get_attribute_value(HW_sample_dic, 'InstallationPackageID').zfill(6)
                        HW_sample_set_path = self.odf_hw2go.HW_ODF_get_attribute_value(self.odf_hw2go.HW_general_dic, '_sample_set_path')
                        file_name = 'OrganInstallationPackages' + os.path.sep + HW_package_id + os.path.sep + file_name
                        file_full_path = HW_sample_set_path + os.path.sep + file_name

                if not os.path.isfile(file_full_path):
                    # the file is not found
                    if file_full_path != '':
                        label_text = f'File not found : {file_full_path}'
                    else:
                        label_text = f'File not found : {file_name}'
                    self.viewer_file_type = None
                    file_name = None

        if self.viewer_file_type == 'image':
            if file_name == self.viewer_file_name:
                # the image is already displayed in the viewer, no change of the label text
                label_text = self.viewer_text
            else:
                # the image is not already displayed in the viewer
                # open the image file
                self.viewer_orig_image = Image.open(file_full_path)
                # prepare the text to display with image file name and size
                label_text  = f'File : {file_name}\n'
                label_text += f'Image size {self.viewer_orig_image.size[0]} x {self.viewer_orig_image.size[1]} pixels'
        else:
            self.viewer_orig_image = None

        if self.viewer_file_type == 'sample':
            if file_name == self.viewer_file_name:
                # the audio sample is already displayed/played in the viewer, pause/resume its playback, no change of the label text
                audio_player.pause_resume()
                label_text = self.viewer_text
            else:
                # the audio sample is not already displayed/played in the viewer
                # start the playback of the sample and recover its metadata
                metadata_dic = audio_player.start(file_full_path)
                # prepare the text to display with audio file name and metadata
                label_text  = f'File : {file_name}\n'

                if pipe_pitch_tuning != None:
                    label_text += 'with pitch tuning ' + str(pipe_pitch_tuning) + '\n'

                if metadata_dic['file_format'] == 'wave':
                    label_text += "Wav format\n"
                elif metadata_dic['file_format'] == 'wavpack':
                    label_text += "WavPack format (sample data not decoded by OdfEdit)\n"

                if metadata_dic['error_msg'] != '':
                    label_text += f"ERROR : {metadata_dic['error_msg']}\n"

                if metadata_dic['metadata_recovered']:
                    label_text += "Mono" if metadata_dic['nb_of_channels'] == 1 else "Stereo"
                    label_text += f", sampling {metadata_dic['sampling_rate']} Hz"
                    label_text += f", resolution {metadata_dic['bits_per_sample']} bits"
                    label_text += f", duration {metadata_dic['audio_duration']:0.3f} sec. ({metadata_dic['nb_of_samples']} samples)"
                    label_text += '\n'

                    if 'midi_note' in metadata_dic.keys():
                        if metadata_dic['midi_note'] == 0 and metadata_dic['midi_note'] == 0:
                            label_text += 'Actual pitch : no pitch information provided.'
                        else:
                            label_text += f"Actual pitch : MIDI note {metadata_dic['midi_note']} + {metadata_dic['midi_pitch_fract']:0.2f} cents"
                            label_text += f" ({midi_nb_plus_cents_to_freq(metadata_dic['midi_note'], metadata_dic['midi_pitch_fract']):0.2f} Hz)"
                        label_text += '\n'

                    if 'info' in metadata_dic.keys():
                        label_text += '\n'
                        for key, value in metadata_dic['info'].items():
                            if key == 'IART':
                                label_text += f"Artist : {value}\n"
                            elif key == 'ICOP':
                                label_text += f"Copyright : {value}\n"
                            elif key == 'ISFT':
                                label_text += f"Software used : {value}\n"
                            elif key == 'ICMT':
                                label_text += f"Comments : {value}\n"
                            elif key == 'ICRD':
                                label_text += f"Date : {value}\n"
                            else:
                                label_text += f"{key} - {value}\n"

                    if 'loops_nb' in metadata_dic.keys():
                        label_text += '\n'
                        for l in range(1, metadata_dic['loops_nb']+1):
                            label_text += f"Loop {l} : {metadata_dic['loop'+str(l)+'_end_seconds'] - metadata_dic['loop'+str(l)+'_start_seconds']:0.3f} sec."
                            label_text += f" ({metadata_dic['loop'+str(l)+'_start_seconds']:0.3f} -> {metadata_dic['loop'+str(l)+'_end_seconds']:0.3f})"
                            label_text += f", {metadata_dic['loop'+str(l)+'_end_sample'] - metadata_dic['loop'+str(l)+'_start_sample']} samples"
                            label_text += f" ({metadata_dic['loop'+str(l)+'_start_sample']} -> {metadata_dic['loop'+str(l)+'_end_sample']})"
                            label_text += '\n'

                    if 'cue_points_nb' in metadata_dic.keys():
                        label_text += '\n'
                        for c in range(1, metadata_dic['cue_points_nb']+1):
                            label_text += f"Cue {c} : ID {metadata_dic['cue'+str(c)+'_id']}"
                            sample_nb = None
                            if metadata_dic['cue'+str(c)+'_sample_start'] > 0:
                                sample_nb = metadata_dic['cue'+str(c)+'_sample_start']
                            elif metadata_dic['cue'+str(c)+'_position'] > 0:
                                sample_nb = metadata_dic['cue'+str(c)+'_position']
                            if sample_nb != None:
                                sample_sec = int(sample_nb * 1000 / metadata_dic['sampling_rate']) / 1000
                                label_text += f", at {sample_sec} sec. (sample {sample_nb})"

                            label_text += '\n'

                    label_text = label_text[:-1] # remove the ending carriage return

        else:
            # it is not a sample file, if a sample playback is in progress stop it
            audio_player.stop()

        self.viewer_file_name = file_name
        self.viewer_text = label_text

        # reset the canvas viewing position
        self.view_canvas.xview(tk.MOVETO, 0)
        self.view_canvas.yview(tk.MOVETO, 0)

        # update the content of the viewer
        self.viewer_content_update()

    #-------------------------------------------------------------------------------------------------
    def viewer_content_update(self, event=None):
        # (GUI event callback) the user has turned the mouse wheel inside the viewer canvas or the viewer content has to be displayed/updated

        if event != None and event.type == '38' and self.viewer_file_type == 'image':  # 38 = MouseWheel event type
            if event.num == 4 or event.delta > 0:
                # mouse wheel rotation up in Linux or Windows
                self.viewer_zoom_factor = min(6, self.viewer_zoom_factor + 0.1)
            else:
                self.viewer_zoom_factor = max(0.2, self.viewer_zoom_factor - 0.1)

        # add to the text the image zoom if applicable and scale the image
        if self.viewer_file_type == 'image':
            label_text = self.viewer_text + f'\n\nZoom x{self.viewer_zoom_factor:0.1f} (use mouse drag/wheel to move/zoom the image)'
            # scale the image
            self.viewer_scaled_image = ImageTk.PhotoImage(ImageOps.scale(self.viewer_orig_image, self.viewer_zoom_factor))
        else:
            label_text = self.viewer_text

        self.view_canvas.delete('all')
        # show the text in the canvas
        text_widget_id = self.view_canvas.create_text(10, 10, anchor=tk.NW, text=label_text, font=TEXT_FONT, width=self.frm_viewer.winfo_width() - 10)
        # get the height of the text widget
        text_bounds = self.view_canvas.bbox(text_widget_id)
        text_widget_height = text_bounds[3] - text_bounds[1]

        if self.viewer_file_type == 'image':
            # show the image in the canvas
            self.view_canvas.create_image(10, text_widget_height + 10, anchor=tk.NW, image=self.viewer_scaled_image)

    #-------------------------------------------------------------------------------------------------
    def viewer_content_drag(self, event):
        # (GUI event callback) the user has dragged the viewer content

        if self.viewer_file_type == 'image':
            self.view_canvas.scan_dragto(event.x, event.y, gain=1)

    #-------------------------------------------------------------------------------------------------
    def viewer_sample_stop(self):
        # stop the sample file playback in progress

        audio_player.stop()

    #-------------------------------------------------------------------------------------------------
    def objects_list_update_hw(self, event=0):
        # do an update the Hauptwerk objects list widget

        tab = '      '

        # clear the HW objects list widget content
        self.lst_hw_browser.delete(0, tk.END)

        # update the HW objects list widgets
        if len(self.odf_hw2go.HW_odf_dic) > 0:
            # there are HW ODF data in the dictionary
            if self.selected_object_uid == None or self.selected_object_app != 'HW':
                center_object_UID = "_General"
            else:
                center_object_UID = self.selected_object_uid

            general_is_visible = center_object_UID == "_General"

            selected_object_dic = self.odf_hw2go.HW_ODF_get_object_dic_from_uid(center_object_UID)
            if selected_object_dic != None:
                # set the first element
                self.lst_hw_browser.insert(tk.END, '*** SECTIONS RELATIONSHIP (parents/current in red/children) ***')

                # display the parents of the selected object
                objects_uid_list = []
                for object_dic in selected_object_dic['_parents']:
                    objects_uid_list.append(object_dic['_uid'])
                for object_uid in sorted(objects_uid_list):
                    object_dic = self.odf_hw2go.HW_ODF_get_object_dic_from_uid(object_uid)
                    obj_name = self.odf_hw2go.HW_ODF_get_attribute_value(object_dic, 'Text')
                    if obj_name == None: obj_name = self.odf_hw2go.HW_ODF_get_attribute_value(object_dic, 'Name')
                    if obj_name != None:
                        self.lst_hw_browser.insert(tk.END, object_uid + ' (' + obj_name + ')')
                    else:
                        self.lst_hw_browser.insert(tk.END, object_uid)
                    if object_uid == '_General': general_is_visible = True

                # display the selected object
                item_str = tab + center_object_UID
                obj_name = self.odf_hw2go.HW_ODF_get_attribute_value(selected_object_dic, 'Text')
                if obj_name == None:
                    obj_name = self.odf_hw2go.HW_ODF_get_attribute_value(selected_object_dic, 'Name')
                if obj_name != None:
                    item_str += ' (' + obj_name + ')'
                children_nb = len(self.odf_hw2go.HW_ODF_get_attribute_value(selected_object_dic, '_children'))
                if children_nb == 1:
                    item_str += '  >> 1 child'
                elif children_nb >= 1:
                    item_str += '  >> ' + str(children_nb) + ' children'
                self.lst_hw_browser.insert(tk.END, item_str)
                self.lst_hw_browser.itemconfig(tk.END, foreground='red')
                self.lst_hw_browser.selection_set(tk.END)
                self.lst_hw_browser.see(tk.END)


                # display the children of the selected object
                objects_uid_list = []
                for object_dic in selected_object_dic['_children']:
                    objects_uid_list.append(object_dic['_uid'])
                for object_uid in sorted(objects_uid_list):
                    item_str = tab + tab + object_uid
                    object_dic = self.odf_hw2go.HW_ODF_get_object_dic_from_uid(object_uid)
                    obj_name = self.odf_hw2go.HW_ODF_get_attribute_value(object_dic, 'Text')
                    if obj_name == None:
                        obj_name = self.odf_hw2go.HW_ODF_get_attribute_value(object_dic, 'Name')
                    if obj_name != None:
                        item_str += ' (' + obj_name + ')'
                    children_nb = len(self.odf_hw2go.HW_ODF_get_attribute_value(object_dic, '_children'))
                    if children_nb == 1:
                        item_str += '  >> 1 child'
                    elif children_nb >= 1:
                        item_str += '  >> ' + str(children_nb) + ' children'
                    self.lst_hw_browser.insert(tk.END, item_str)

            if not general_is_visible:
                # add _General entry at the top of the list if it is not already visible
                self.lst_hw_browser.insert(0, '_General')

            self.lst_hw_browser.insert(tk.END, '*** SECTIONS WITHOUT PARENT ***')

            # add at the end all the objects which have no parent except some types
            objects_uid_list = []
            for HW_object_type, HW_object_type_dic in self.odf_hw2go.HW_odf_dic.items():
                # scan the HW object types
                if not HW_object_type.startswith(('Pi', 'Sa', 'TremulantWaveformP', 'SwitchL', '_General')):
                    # excluded objects types are : Pipe_xxx, Sample, TremulantWaveformPipe, SwitchLinkage, _General
                    # the current HW object type can be added in the list
                    for object_dic in HW_object_type_dic.values():
                        # scan the HW objects of the current HW objects type
                        if len(object_dic['_parents']) == 0:
                            # the current object has no parent
                            objects_uid_list.append(object_dic['_uid'])

            for object_uid in sorted(objects_uid_list):
                item_str = object_uid
                object_dic = self.odf_hw2go.HW_ODF_get_object_dic_from_uid(object_uid)
                obj_name = self.odf_hw2go.HW_ODF_get_attribute_value(object_dic, 'Text')
                if obj_name == None:
                    obj_name = self.odf_hw2go.HW_ODF_get_attribute_value(object_dic, 'Name')
                if obj_name != None:
                    item_str += ' (' + obj_name + ')'
                children_nb = len(self.odf_hw2go.HW_ODF_get_attribute_value(object_dic, '_children'))
                if children_nb == 1:
                    item_str += '  >> 1 child'
                elif children_nb >= 1:
                    item_str += '  >> ' + str(children_nb) + ' children'
                self.lst_hw_browser.insert(tk.END, item_str)

    #-------------------------------------------------------------------------------------------------
    def objects_list_selected_hw(self, event):
        # (GUI event callback) the user has selected an item in the Hauptwerk objects list widget

        if self.gui_events_blocked: return

        # get the line numbers of the selected item in the list
        cursel_tuple = self.lst_hw_browser.curselection()
        if len(cursel_tuple) > 0:
            selected_line_indice = cursel_tuple[0]
        else:
            selected_line_indice = None

        if selected_line_indice != None and self.can_i_make_change():
            # an item of the HW objects list widget is selected
            # recover in the objects list widget the UID of the selected object (before the first space in the selected item text)
            self.selected_object_app = 'HW'
            self.selected_object_uid = self.lst_hw_browser.get(selected_line_indice).strip().split(' ')[0]
            if self.selected_object_uid[0] == '*': # item without object UID
                self.selected_object_uid = None
            self.selected_linked_uid = None
            self.edited_object_uid = self.selected_object_uid

            self.focused_objects_widget = self.lst_hw_browser
            self.focused_sel_item_id = self.selected_object_uid

            # update the object text box and links list
            self.object_links_list_update()
            self.object_text_update()

        # update the status of GUI widgets
        self.gui_status_update_lists()
        self.gui_status_update_buttons()

    #-------------------------------------------------------------------------------------------------
    def objects_list_selected_dbl_hw(self, event):
        # (GUI event callback) the user has double-clicked an item in the Hauptwerk objects list widget

        if self.selected_object_uid != None:
            self.objects_list_update_hw()


#-------------------------------------------------------------------------------------------------
class C_GUI(C_GUI_NOTEBOOK):
    # class to manage the graphical user interface of the application

    odf_data = None             # one instance of the C_ODF_DATA class
    odf_hw2go = None            # one instance of the C_ODF_HW2GO class

    selected_object_app = 'GO'  # application associated to the object currently selected : 'GO' or 'HW', GO by default
    selected_object_uid = None  # UID of the object currently selected in the objects lists (GO or HW) or objects tree widgets (GO)
    selected_linked_uid = None  # UID of the linked object currently selected in the linked objects list if any (GO)
    edited_object_uid = None    # UID of the edited object in the edition text box

    focused_objects_widget = None  # objects widget which has the focus : self.lst_objects_list / self.trv_objects_tree / self.lst_links_list / self.lst_hw_browser / None
    focused_sel_item_id = None     # identifier of the selected item of the focused widget
    is_focus_on_objects_lists = False   # flag at True if one objects list or tree has currently the focus

    is_key_control_pressed = False # flag indicating if a Control key is currently pressed on the keyboard of the computer in OdfEdit

    ignore_b1_release = False        # flag permitting to ignore the processing of the mouse button 1 release event
    object_dragging_in_progress = False # flag indicating that a mouse cursor dragging is in progress
    drag_overflown_object_uid = None    # UID of the object currently overflown by the mouse cursor dragging
    dragged_object_drop_action = None   # action to do on dragged object drop
    dragged_object_uid = None           # UID of the dragged object
    dragged_object_type = None          # type of the dragged object
    dragged_object_parents_list = None  # parents list of the dragged object
    selected_object_uid_b4 = None       # selected object UID before the drag action
    selected_linked_uid_b4 = None       # selected linked object UID before the drag action
    selected_widget_b4 = None           # selected widget before the drag action

    opened_objects_iid_list = []   # list of the objects tree nodes iid which are opened

    odf_data_changed = False       # flag indicating that data have been changed in the odf_data and not saved in an ODF
    edited_object_changed = False  # flag indicating that data have been changed in the object currently edited (and not yet applied in odf_data)

    is_loading = False             # flag set at True if an ODF loading/conversion is in progress
    is_loaded_odf = False          # flag set at True if an ODF is loaded
    is_loaded_hw_odf = False       # flag set at True if a Hauptwerk ODF is loaded

    gui_events_blocked = False     # flag indicating that the GUI events are currently blocked

    text_to_search = None       # text which has to be searched in the help
    search_index = None         # last search result position in the help

    odf_check_files_names = None   # flag indicating if files names have to be checked or not during the ODF data check
                                   # None means that the question has not been asked to the user

    # application data which are saved at application close and restored at application start

    odf_save_encoding = ''      # StringVar with encoding type (ENCODING_ISO_8859_1 or ENCODING_UTF8_BOM) to use when saving data in an ODF
    odf_recent_opened_list = []   # list containing the file name of the last opened ODFs

    objects_list_width = 0
    objects_tree_width = 0
    object_editor_width = 0
    object_links_heigh = 0
    wnd_main_geometry = ''

    hw2go_warning_displayed_bool = False      # flag indicating that the HW to GO conversion warning has been displayed one time in the current life cycle of the application
    hw2go_convert_alt_ranks_bool = False      # flag BooleanVar set by the menu to ask the HW 2 GO conversion of the alternate ranks (for wave based tremulants)
    hw2go_alt_ranks_in_sep_ranks_bool = False # flag BooleanVar set by the menu to ask the HW 2 GO conversion to place the alternate ranks in their main rank
    hw2go_pitch_tuning_metadata_bool = False  # flag BooleanVar set by the menu to ask the HW 2 GO conversion to correct pipes pitch if needed by using MIDI note in metadata of sample files
    hw2go_pitch_tuning_filename_bool = False  # flag BooleanVar set by the menu to ask the HW 2 GO conversion to correct pipes pitch if needed by using MIDI note in sthe name of sample files
    hw2go_convert_alt_scr_layers_bool = False # flag BooleanVar set by the menu to ask the HW 2 GO conversion of the alternative screens layers
    hw2go_not_convert_keys_noises_bool = False# flag BooleanVar set by the menu to ask the HW 2 GO conversion of the keys noises
    hw2go_convert_unused_ranks_bool = False   # flag BooleanVar set by the menu to ask the HW 2 GO conversion of the HW ranks not used by the conversion

    #-------------------------------------------------------------------------------------------------
    def reset_all_data(self):
        # reset all the data of the class

        self.is_loaded_hw_odf = False
        self.is_loaded_odf = False

        self.selected_object_app = 'GO'
        self.selected_object_uid = None
        self.selected_linked_uid = None
        self.edited_object_uid = None

        self.focused_objects_widget = None
        self.focused_sel_item_id = None

        self.odf_data_changed = False
        self.edited_object_changed = False

        self.odf_data.reset_all_data()
        self.odf_hw2go.reset_all_data()

        self.lst_odf_sresults.delete(0, tk.END)
        self.lab_search_results_nb.config(text='')

        self.odf_check_files_names = None

    #-----------------------------------------------------------------------------------------------
    def wnd_main_build(self):
        # build the main window of the application with all its GUI widgets

        # create the main window
        self.wnd_main = tk.Tk(className='OdfEdit')

        # define Tkinter string or boolean variables used in the general menu
        self.odf_save_encoding = tk.StringVar(self.wnd_main)
        self.odf_save_encoding.set(ENCODING_ISO_8859_1)
        self.hw2go_convert_alt_ranks_bool = tk.BooleanVar(self.wnd_main)
        self.hw2go_alt_ranks_in_sep_ranks_bool = tk.BooleanVar(self.wnd_main)
        self.hw2go_pitch_tuning_metadata_bool = tk.BooleanVar(self.wnd_main)
        self.hw2go_pitch_tuning_filename_bool = tk.BooleanVar(self.wnd_main)
        self.hw2go_convert_alt_scr_layers_bool = tk.BooleanVar(self.wnd_main)
        self.hw2go_convert_unused_ranks_bool = tk.BooleanVar(self.wnd_main)
        self.hw2go_not_convert_keys_noises_bool = tk.BooleanVar(self.wnd_main)

        # load the application data
        self.app_data_load()

        # create the main window
        self.wnd_main.title(MAIN_WINDOW_TITLE)
        self.wnd_main.geometry(self.wnd_main_geometry)
        self.wnd_main.configure(background=COLOR_BACKGROUND0)

        # assign an image to the main window icon (needed for application icon in Linux which does not support .ico file)
        icon = tk.PhotoImage(file = os.path.dirname(__file__) + os.path.sep + 'resources' + os.path.sep + 'OdfEdit.png')
        self.wnd_main.iconphoto(True, icon)

        # define the style of tk widgets
        self.wnd_main.option_add('*TCombobox*tk.Listbox*Background', COLOR_BG_LIST)
        self.wnd_main.option_add('*TCombobox*tk.Listbox*Foreground', TEXT_COLOR)
        self.wnd_main.option_add('*Dialog.msg.font', TEXT_FONT)

        # define the style of ttk widgets
        self.wnd_main.style = ttk.Style()
        self.wnd_main.style.theme_use('alt')

        self.wnd_main.style.configure('Treeview', highlightthickness=3, font=TEXT_FONT, background=COLOR_BG_LIST, fieldbackground=COLOR_BG_LIST)
        self.wnd_main.style.map('Treeview', background=[('selected', COLOR_SELECTED_ITEM)])

        self.wnd_main.style.configure('TFrame', background=COLOR_BACKGROUND0)
        self.wnd_main.style.configure('TLabel',  font=TEXT_FONT, background=COLOR_BACKGROUND0)

        self.wnd_main.style.configure('TNotebook', background=COLOR_BACKGROUND0)
        self.wnd_main.style.map('TNotebook.Tab', background=[('active', COLOR_BACKGROUND2), ('selected', COLOR_BACKGROUND1), ('!selected', COLOR_BACKGROUND0)],
                                                 font=[('active', TEXT_FONT), ('selected', TEXT_FONT), ('!selected', TEXT_FONT)],
                                                 focuscolor=COLOR_BACKGROUND1)

        self.wnd_main.style.configure('TButton', font=TEXT_FONT, focuscolor=COLOR_BACKGROUND1, background=COLOR_BACKGROUND1)
        self.wnd_main.style.configure('Return.TButton', font=TEXT_FONT_BOLD)
        self.wnd_main.style.configure('RedText.TButton', foreground='red')
        self.wnd_main.style.configure('ReliefGroove.TButton', relief='groove')
        self.wnd_main.style.map('TButton', foreground=[('disabled', 'grey'), ('active', TEXT_COLOR)],
                                           background=[('disabled', COLOR_BACKGROUND0), ('active', COLOR_BACKGROUND2)],
                                           focuscolor=[('active', COLOR_BACKGROUND2)])

        self.wnd_main.style.configure('TCheckbutton', font=TEXT_FONT, focuscolor=COLOR_BACKGROUND1, background=COLOR_BACKGROUND0)
        self.wnd_main.style.map('TCheckbutton', background=[('active', COLOR_BACKGROUND2), ('selected', COLOR_BACKGROUND0), ('!selected', COLOR_BACKGROUND0)],
                                                indicatorcolor=[('selected', COLOR_SELECTED_ITEM), ('!selected', COLOR_BACKGROUND0)],
                                                focuscolor=COLOR_BACKGROUND0)

        self.wnd_main.style.configure('TRadiobutton', font=TEXT_FONT, focuscolor=COLOR_BACKGROUND1, background=COLOR_BACKGROUND0)
        self.wnd_main.style.map('TRadiobutton', background=[('active', COLOR_BACKGROUND2), ('selected', COLOR_BACKGROUND0), ('!selected', COLOR_BACKGROUND0)],
                                                indicatorcolor=[('selected', COLOR_SELECTED_ITEM), ('!selected', COLOR_BACKGROUND0)],
                                                focuscolor=COLOR_BACKGROUND0)

        self.wnd_main.style.configure('TPanedWindow', background=COLOR_BACKGROUND1)

        self.wnd_main.style.configure('Vertical.TScrollbar', background=COLOR_BACKGROUND1, troughcolor=COLOR_BACKGROUND0, arrowcolor='black')
        self.wnd_main.style.configure('Horizontal.TScrollbar', background=COLOR_BACKGROUND1, troughcolor=COLOR_BACKGROUND0, arrowcolor='black')
        self.wnd_main.style.map('Vertical.TScrollbar', foreground=[('disabled', COLOR_BACKGROUND0), ('active', COLOR_BACKGROUND1)],
                                                       background=[('disabled', COLOR_BACKGROUND0), ('active', COLOR_BACKGROUND2)])
        self.wnd_main.style.map('Horizontal.TScrollbar', foreground=[('disabled', COLOR_BACKGROUND0), ('active', COLOR_BACKGROUND1)],
                                                         background=[('disabled', COLOR_BACKGROUND0), ('active', COLOR_BACKGROUND2)])

        # link functions so some main window events
        # to react to ctrl + s or ctrl + S keyboard keys to save changed data
        self.wnd_main.bind('<Control-s>', self.wnd_main_key_ctrl_s)
        self.wnd_main.bind('<Control-S>', self.wnd_main_key_ctrl_s)
        # to react to Control key press or release
        self.wnd_main.bind('<KeyPress>', self.wnd_main_key_press)
        self.wnd_main.bind('<KeyRelease>', self.wnd_main_key_release)
        # to react to window delete, to ask the user to save his changes before to close the main window
        self.wnd_main.protocol('WM_DELETE_WINDOW', self.wnd_main_quit)

        #--- create the various widgets inside the main window

        #-- top buttons bar

        # top frame to encapsulate widgets bar
        self.frm_top = ttk.Frame(self.wnd_main)
        self.frm_top.pack(side=tk.TOP, fill=tk.X)

        # button 'New'
        self.btn_odf_new = ttk.Button(self.frm_top, text='New', width=7, command=self.file_new)
        self.btn_odf_new.pack(side=tk.LEFT, padx=7, pady=5)
        ToolTip(self.btn_odf_new, 'Clear all existing data to create a new ODF from scratch.')

        # button 'Open'
        self.btn_odf_open = ttk.Button(self.frm_top, text='Open...', width=10, command=self.file_open)
        self.btn_odf_open.pack(side=tk.LEFT, padx=0, pady=5)
        ToolTip(self.btn_odf_open, 'Load a GrandOrgue ODF (extension .organ) or a Hauptwerk ODF (extension .Organ_Hauptwerk_xml or .xml).')

        # button to open the recently opened ODFs list (showing the character ▼ unicode 25BC)
        self.btn_odf_open_last = ttk.Button(self.frm_top, text='\u25BC', width=2, command=self.recent_odf_list_open)
        self.btn_odf_open_last.pack(side=tk.LEFT, padx=0, pady=5)
        ToolTip(self.btn_odf_open_last, 'Show the list of recently opened ODF permitting to open one of them.')

        # list showing the last opened ODFs (is build in the function recent_odf_list_open)
        self.lst_recent_odf = None

        # button 'Save'
        self.btn_odf_save = ttk.Button(self.frm_top, text='Save', style='RedText.TButton', width=7, state=tk.DISABLED, command=self.file_save)
        self.btn_odf_save.pack(side=tk.LEFT, padx=7, pady=5)
        ToolTip(self.btn_odf_save, 'Save in a GrandOrgue ODF the changes done in the edited ODF.')

        # button 'Save as...'
        self.btn_odf_saveas = ttk.Button(self.frm_top, text='Save as...', style='RedText.TButton', width=10, state=tk.DISABLED, command=self.file_saveas)
        self.btn_odf_saveas.pack(side=tk.LEFT, padx=0, pady=5)
        ToolTip(self.btn_odf_saveas, 'Save in a new GrandOrgue ODF the changes done in the edited ODF.')

        # button 'Check ODF data'
        self.btn_data_check = ttk.Button(self.frm_top, text='ODF data check', width=15, state=tk.DISABLED, command=self.odf_data_check)
        self.btn_data_check.pack(side=tk.LEFT, padx=7, pady=5)
        ToolTip(self.btn_data_check, 'Execute checks in the edited ODF data (syntax, compliance with the specification). Not as exhaustively as GrandOrgue does.')

        # button 'Menu' and general menu
        self.btn_gen_menu = ttk.Button(self.frm_top, text='≡', width=3, command=self.gen_menu_open)
        self.btn_gen_menu.pack(side=tk.LEFT, padx=2, pady=5)

        self.general_menu = tk.Menu(self.btn_gen_menu, tearoff=0, background=COLOR_BACKGROUND1, foreground=TEXT_COLOR, activebackground=COLOR_BACKGROUND2, activeforeground=TEXT_COLOR)
        self.general_menu.add_checkbutton(label='Save ODF with ISO-8859-1 encoding (else UTF-8-BOM)', onvalue=ENCODING_ISO_8859_1, offvalue=ENCODING_UTF8_BOM, variable=self.odf_save_encoding, command=self.gen_menu_open)
        self.general_menu.add_command(label='Sort references in selected section...', command=self.gen_menu_references_sort)
        self.general_menu.add_command(label='Extend the compass of the selected Manual/Stop/Rank...', command=self.compass_extend)
        self.general_menu.add_separator()
        self.general_menu.add_checkbutton(label='HW to GO - do not convert keys noises', onvalue=True, offvalue=False, variable=self.hw2go_not_convert_keys_noises_bool, command=self.gen_menu_open)
        self.general_menu.add_checkbutton(label='HW to GO - convert alternate panels layouts', onvalue=True, offvalue=False, variable=self.hw2go_convert_alt_scr_layers_bool, command=self.gen_menu_open)
        self.general_menu.add_checkbutton(label='HW to GO - convert tremmed samples', onvalue=True, offvalue=False, variable=self.hw2go_convert_alt_ranks_bool, command=self.gen_menu_open)
        self.general_menu.add_checkbutton(label='HW to GO - place tremmed samples in separate ranks', onvalue=True, offvalue=False, variable=self.hw2go_alt_ranks_in_sep_ranks_bool, command=self.gen_menu_open)
        self.general_menu.add_checkbutton(label='HW to GO - convert unused ranks', onvalue=True, offvalue=False, variable=self.hw2go_convert_unused_ranks_bool, command=self.gen_menu_open)
        self.general_menu.add_checkbutton(label='HW to GO - correct pipes pitch from samples file name', onvalue=True, offvalue=False, variable=self.hw2go_pitch_tuning_filename_bool, command=self.gen_menu_open)
        self.general_menu.add_checkbutton(label='HW to GO - correct pipes pitch from samples metadata', onvalue=True, offvalue=False, variable=self.hw2go_pitch_tuning_metadata_bool, command=self.gen_menu_open)
        self.general_menu.add_separator()
        self.general_menu.add_command(label='Clear logs', command=self.logs_clear)
        self.general_menu.add_command(label='Clear last opened ODF list', command=self.recent_odf_list_clear)
        self.general_menu.add_command(label='About...', command=self.gen_menu_about)

        # button 'Quit'
        self.btn_quit_appli = ttk.Button(self.frm_top, text='Quit', style='ReliefGroove.TButton', width=7, command=self.wnd_main_quit)
        self.btn_quit_appli.pack(side=tk.LEFT, padx=7, pady=5)

        # label with loaded ODF file name or to display progression status
        self.lab_odf_file_name = tk.Label(self.frm_top, text='', fg=TEXT_COLOR, bg=COLOR_BACKGROUND0, borderwidth=1, relief=tk.SOLID, anchor=tk.W, height=1)
        self.lab_odf_file_name.pack(side=tk.LEFT, padx=5, pady=5, ipady=3, expand=True, fill=tk.X)

        #-- bottom area with horizontal paned window on the full window width
        """
            Composition of the horizontal paned window
                paned_wnd : bottom horizontal paned window
                    paned_wnd_frm_1 : first frame of paned_wnd (objects list)
                        lab_objects_nb : label showing the number of list objects
                        frm_objects_list : frame to contain the objects list and scroll bar
                            scrollbarv_obj_list : vertical scroll bar
                            scrollbarh : horizontal scroll bar
                            lst_objects_list : objects list
                    paned_wnd_frm_2 : second frame of paned_wnd (objects tree)
                        frm_top_paned_wnd_2 : frame to contain following buttons
                            btn_collapse_tree_node : button to collapse of the objects tree
                            btn_expand_tree_node : button to expand the objects tree
                            btn_unselect : button to unselect the current selected object
                            frm_object_tree : frame to contain the objects tree and its scroll bars
                                scrollbarv : vertical scroll bar
                                scrollbarh : horizontal scroll bar
                                trv_objects_tree : objects tree view
                    paned_wnd_frm_3 : third frame of paned_wnd (selected object parents/children list and text editor)
                        obj_paned_wnd : vertical paned window placed inside paned_wnd_frm_3
                            obj_paned_wnd_frm_1 : first frame of obj_paned_wnd (buttons bar and object perents/children list)
                                frm_top_obj_paned_wnd_1 : top frame of obj_paned_wnd_frm_1 for buttons bar
                                    btn_object_apply_chg : button to apply changes made in the object editor
                                    btn_object_add       : button to add an object
                                    btn_object_parents   : button to link the selected object to parents
                                    btn_object_children  : button to link the selected object to children
                                    btn_object_rename    : button to rename the selected object
                                    btn_object_delete    : button to delete the selected objects
                                    btn_show_help        : button to show the help related to the selected object
                                frm_bottom_obj_paned_wnd_1 : bottom frame of obj_paned_wnd_frm_1 for object parents/children list and its scroll bar
                                    scrollbarv : vertical scroll bar
                                    lst_links_list : object parents/children list
                            obj_paned_wnd_frm_2 : second frame of obj_paned_wnd (object text editor and its scroll bars)
                                scrollbarv : vertical scroll bar
                                scrollbarh : horizontal scroll bar
                                txt_object_text : object text editor
                    paned_wnd_frm_4 : fourth frame of paned_wnd containing the notebook
        """


        # horizontal paned window at the bottom of the main window
        self.paned_wnd = tk.PanedWindow(self.wnd_main, orient=tk.HORIZONTAL, bg=COLOR_BACKGROUND1, relief=tk.FLAT, sashrelief = tk.RAISED, sashwidth = 10)
        self.paned_wnd.pack(side=tk.BOTTOM, padx=5, pady=5, expand=True, fill=tk.BOTH)
        self.paned_wnd.bind('<Enter>', lambda event, widget=self.paned_wnd: self.widget_mouse_enter(event, widget))
        self.paned_wnd.bind('<Leave>', lambda event, widget=self.paned_wnd: self.widget_mouse_leave(event, widget))

        #-- paned window element #1 (objects list)

        # frame to occupy the full area of the element #1
        self.paned_wnd_frm_1 = ttk.Frame(self.wnd_main)
        self.paned_wnd.add(self.paned_wnd_frm_1, minsize=200, width=self.objects_list_width)

        # label with the number of objects in the objects list, placed at the top of the parent frame
        self.lab_objects_nb = ttk.Label(self.paned_wnd_frm_1, text='', borderwidth=0, relief=tk.SOLID, anchor=tk.CENTER)
        self.lab_objects_nb.pack(side = tk.TOP, pady=10, fill=tk.X)

        # frame to occupy the bottom area of the parent frame and to encapsulate the list box and its scroll bars
        self.frm_objects_list = ttk.Frame(self.paned_wnd_frm_1)
        self.frm_objects_list.pack(side = tk.BOTTOM, fill=tk.BOTH, expand=True)

        # list box with objects UIDs and names, with horizontal and vertical scroll bars, inside the parent frame
        self.scrollbarv_obj_list = ttk.Scrollbar(self.frm_objects_list, orient=tk.VERTICAL)
        self.scrollbarv_obj_list.pack(side=tk.RIGHT, fill=tk.Y)
        scrollbarh = ttk.Scrollbar(self.frm_objects_list, orient=tk.HORIZONTAL)
        scrollbarh.pack(side=tk.BOTTOM, fill=tk.X)
        self.lst_objects_list = tk.Listbox(self.frm_objects_list, bg=COLOR_BG_LIST, font=TEXT_FONT, fg=TEXT_COLOR, selectbackground=COLOR_SELECTED_ITEM,
                                           exportselection=0, selectmode=tk.SINGLE, activestyle=tk.DOTBOX)
        self.lst_objects_list.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        self.lst_objects_list.bind('<<ListboxSelect>>', self.objects_list_selected)
        self.lst_objects_list.bind('<Double-1>', self.objects_list_selected_dbl)
        self.lst_objects_list.bind('<B1-Motion>', self.object_b1_motion)
        self.lst_objects_list.bind('<ButtonRelease-1>', self.object_b1_release)
        self.lst_objects_list.bind('<B1-Leave>', lambda event: 'break')  # to avoid the auto scrolling in the list on drag
        self.lst_objects_list.bind('<FocusOut>', self.objects_widget_focus_out)

        self.lst_objects_list.bind('<Up>', self.objects_list_selected)
        self.lst_objects_list.bind('<Down>', self.objects_list_selected)

        self.lst_objects_list.config(yscrollcommand=self.scrollbarv_obj_list.set)
        self.lst_objects_list.config(xscrollcommand=scrollbarh.set)
        self.scrollbarv_obj_list.config(command=self.lst_objects_list.yview)
        scrollbarh.config(command=self.lst_objects_list.xview)

        #-- paned window element #2 (objects tree)

        # frame to occupy the full area of the element #2
        self.paned_wnd_frm_2 = ttk.Frame(self.wnd_main)
        self.paned_wnd.add(self.paned_wnd_frm_2, minsize=200, width=self.objects_tree_width)

        # frame to occupy the top area of the parent frame and to encamsulate collapse/expand buttons
        self.frm_top_paned_wnd_2 = ttk.Frame(self.paned_wnd_frm_2)
        self.frm_top_paned_wnd_2.pack(side=tk.TOP, fill=tk.X)

        # button 'Collapse' to collapse the selected object in objects tree
        self.btn_collapse_tree_node = ttk.Button(self.frm_top_paned_wnd_2, text='Collapse', width=5, state=tk.DISABLED, command=self.objects_tree_collapse_current)
        self.btn_collapse_tree_node.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)
        ToolTip(self.btn_collapse_tree_node, 'Collapse the entire tree under the selected section.')

        # button 'Expand' to expand the selected object in objects tree
        self.btn_expand_tree_node = ttk.Button(self.frm_top_paned_wnd_2, text='Expand', width=5, state=tk.DISABLED, command=self.objects_tree_expand_selected)
        self.btn_expand_tree_node.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)
        ToolTip(self.btn_expand_tree_node, 'Expand the entire tree under the selected section.')

        # button 'Unselect'
        self.btn_unselect = ttk.Button(self.frm_top_paned_wnd_2, text='Unselect', width=5, state=tk.DISABLED, command=self.object_unselect)
        self.btn_unselect.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)

        # frame to occupy the bottom area of the parent frame and to encapsulate the tree view and its scroll bars
        self.frm_object_tree = ttk.Frame(self.paned_wnd_frm_2)
        self.frm_object_tree.pack(side = 'bottom', fill=tk.BOTH, expand=True)

        # treeview to display the objects hierarchy, with horizontal and vertical scroll bars
        scrollbarv = ttk.Scrollbar(self.frm_object_tree, orient=tk.VERTICAL)
        scrollbarv.pack(side=tk.RIGHT, fill=tk.Y)
        scrollbarh = ttk.Scrollbar(self.frm_object_tree, orient=tk.HORIZONTAL)
        scrollbarh.pack(side=tk.BOTTOM, fill=tk.X)
        self.trv_objects_tree = ttk.Treeview(self.frm_object_tree, show='tree', selectmode=tk.EXTENDED)
        self.trv_objects_tree.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        self.trv_objects_tree.column('#0', width=500)
        self.trv_objects_tree.bind('<<TreeviewSelect>>', self.objects_tree_selected)
        self.trv_objects_tree.bind('<B1-Motion>', self.object_b1_motion)
        self.trv_objects_tree.bind('<ButtonRelease-1>', self.object_b1_release)
        self.trv_objects_tree.bind('<FocusOut>', self.objects_widget_focus_out)

        self.trv_objects_tree.config(yscrollcommand=scrollbarv.set)
        self.trv_objects_tree.config(xscrollcommand=scrollbarh.set)
        scrollbarv.config(command=self.trv_objects_tree.yview)
        scrollbarh.config(command=self.trv_objects_tree.xview)
        self.trv_objects_tree.tag_configure(TAG_SAME_UID, foreground=TEXT_COLOR, background=COLOR_SAME_UID_ITEM)

        #-- paned window element #3 (object editor)

        # frame to occupy the full area of the element #3
        self.paned_wnd_frm_3 = ttk.Frame(self.wnd_main)
        self.paned_wnd.add(self.paned_wnd_frm_3, minsize=400, width=self.object_editor_width)

        # vertical paned window placed inside the frame of the element #3
        self.obj_paned_wnd = tk.PanedWindow(self.paned_wnd_frm_3, orient=tk.VERTICAL, bg=COLOR_BACKGROUND1, relief=tk.FLAT, sashrelief=tk.RAISED, sashwidth=10)
        self.obj_paned_wnd.pack(side=tk.TOP, expand=True, fill=tk.BOTH)
        self.obj_paned_wnd.bind('<Enter>', lambda event, widget=self.obj_paned_wnd: self.widget_mouse_enter(event, widget))
        self.obj_paned_wnd.bind('<Leave>', lambda event, widget=self.obj_paned_wnd: self.widget_mouse_leave(event, widget))

        # frame to occupy the full area of the element #1 of the vertical paned window
        self.obj_paned_wnd_frm_1 = ttk.Frame(self.wnd_main)
        self.obj_paned_wnd.add(self.obj_paned_wnd_frm_1, minsize=120, height=self.object_links_heigh)

        # frame to occupy the top area of the parent frame and to encapsulate buttons
        self.frm_top_obj_paned_wnd_1 = ttk.Frame(self.obj_paned_wnd_frm_1)
        self.frm_top_obj_paned_wnd_1.pack(side=tk.TOP, fill=tk.X)

        # button 'Apply'
        self.btn_object_apply_chg = ttk.Button(self.frm_top_obj_paned_wnd_1, text='Apply', width=5, style='RedText.TButton', state=tk.DISABLED, command=self.object_text_changes_apply)
        self.btn_object_apply_chg.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)
        ToolTip(self.btn_object_apply_chg, 'Apply the changes done in the text box below.')

        # button 'Add'
        self.btn_object_add = ttk.Button(self.frm_top_obj_paned_wnd_1, text='Add', width=5, state=tk.NORMAL, command=self.object_add)
        self.btn_object_add.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)
        ToolTip(self.btn_object_add, 'Add a child section to the selected section or at the root.')

        # button 'Parents'
        self.btn_object_parents = ttk.Button(self.frm_top_obj_paned_wnd_1, text='Parents', width=8, state=tk.DISABLED, command=lambda type=TO_PARENT: self.object_link(type))
        self.btn_object_parents.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)
        ToolTip(self.btn_object_parents, 'Set/unset links between the selected section and parent sections.')

        # button 'Children'
        self.btn_object_children = ttk.Button(self.frm_top_obj_paned_wnd_1, text='Children', width=8, state=tk.DISABLED, command=lambda type=TO_CHILD: self.object_link(type))
        self.btn_object_children.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)
        ToolTip(self.btn_object_children, 'Set/unset links between the selected section and child sections.')

        # button 'Rename'
        self.btn_object_rename = ttk.Button(self.frm_top_obj_paned_wnd_1, text='Rename', width=8, state=tk.DISABLED, command=self.object_rename)
        self.btn_object_rename.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)
        ToolTip(self.btn_object_rename, 'Rename the last three digits of the selected section.')

        # button 'Delete'
        self.btn_object_delete = ttk.Button(self.frm_top_obj_paned_wnd_1, text='Delete', width=8, state=tk.DISABLED, command=self.object_delete)
        self.btn_object_delete.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)
        ToolTip(self.btn_object_delete, 'Delete the selected section.')

        # button 'Help'
        self.btn_show_help = ttk.Button(self.frm_top_obj_paned_wnd_1, text='Help', width=5, state=tk.DISABLED, command=self.help_search_object)
        self.btn_show_help.pack(side=tk.LEFT, padx=1, pady=5, ipadx=0, fill=tk.X, expand=True)
        ToolTip(self.btn_show_help, 'Show in the help tab the part describing the selected section type.')

        # frame to occupy the bottom area of the parent frame and to encapsulate the parent/children list and its vertical scroll bar
        self.frm_bottom_obj_paned_wnd_1 = ttk.Frame(self.obj_paned_wnd_frm_1)
        self.frm_bottom_obj_paned_wnd_1.pack(side=tk.TOP, fill=tk.BOTH, expand=True)

        # list of the selected object UID and its linked parent/children UID
        scrollbarv = ttk.Scrollbar(self.frm_bottom_obj_paned_wnd_1, orient=tk.VERTICAL)
        scrollbarv.pack(side='right', fill=tk.Y)
        self.lst_links_list = tk.Listbox(self.frm_bottom_obj_paned_wnd_1, bg=COLOR_BG_LIST, font=TEXT_FONT, fg=TEXT_COLOR, selectforeground='white', selectbackground=COLOR_SELECTED_ITEM,
                                      exportselection=0, selectmode=tk.SINGLE, activestyle=tk.DOTBOX)
        self.lst_links_list.pack(side=tk.TOP, fill=tk.BOTH, expand=True)
        self.lst_links_list.bind('<<ListboxSelect>>', self.object_links_list_selected)
        self.lst_links_list.bind('<Double-1>', self.object_links_list_selected_dbl)
        self.lst_links_list.bind('<B1-Motion>', self.object_b1_motion)
        self.lst_links_list.bind('<ButtonRelease-1>', self.object_b1_release)
        self.lst_links_list.bind('<B1-Leave>', lambda event: 'break')  # to avoid the auto scrolling in the list on drag
        self.lst_links_list.bind('<FocusOut>', self.objects_widget_focus_out)
        self.lst_links_list.config(yscrollcommand=scrollbarv.set)
        scrollbarv.config(command=self.lst_links_list.yview)

        # frame to occupy the full area of the element #2 of the vertical paned window
        self.obj_paned_wnd_frm_2 = ttk.Frame(self.wnd_main)
        self.obj_paned_wnd.add(self.obj_paned_wnd_frm_2, minsize=200, height=200)

        # text box with the object text and with horizontal and vertical scroll bars
        scrollbarv = ttk.Scrollbar(self.obj_paned_wnd_frm_2, orient=tk.VERTICAL)
        scrollbarv.pack(side=tk.RIGHT, fill=tk.Y)
        scrollbarh = ttk.Scrollbar(self.obj_paned_wnd_frm_2, orient=tk.HORIZONTAL)
        scrollbarh.pack(side=tk.BOTTOM, fill=tk.X)
        self.txt_object_text = tk.Text(self.obj_paned_wnd_frm_2, fg=TEXT_COLOR, bg=COLOR_BG_EDITOR, bd=1, wrap=tk.NONE, font=TEXT_FONT, selectbackground=COLOR_BG_TEXT_SEL, undo=True)
        self.txt_object_text.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        self.txt_object_text.bind('<<Modified>>', self.object_text_changed)
        self.txt_object_text.bind('<<Paste>>', self.object_text_paste)
        self.txt_object_text.bind('<KeyRelease>', self.object_text_key_pressed)
        self.txt_object_text.bind('<ButtonRelease-1>', self.object_text_click)
        self.txt_object_text.bind('<Double-1>', self.object_text_click_dbl)
        self.txt_object_text.bind('<Control-Key-a>', self.object_text_select_all)
        self.txt_object_text.bind('<Control-Key-A>', self.object_text_select_all)
        self.txt_object_text.config(xscrollcommand=scrollbarh.set, yscrollcommand=scrollbarv.set)
        scrollbarv.config(command=self.txt_object_text.yview)
        scrollbarh.config(command=self.txt_object_text.xview)
        # define the tags for the syntax highlighting in the text box
        self.txt_object_text.tag_config(TAG_FIELD, foreground=COLOR_TAG_FIELD)
        self.txt_object_text.tag_config(TAG_COMMENT, foreground=COLOR_TAG_COMMENT)
        self.txt_object_text.tag_config(TAG_OBJ_UID, foreground=COLOR_TAG_OBJ_UID, font=TEXT_FONT_BOLD)

        #-- paned window element #4 (notebook with several tabs)

        # frame to occupy the full area of the element #4
        self.paned_wnd_frm_4 = ttk.Frame(self.wnd_main)
        self.paned_wnd.add(self.paned_wnd_frm_4, minsize=400)  # no need to set its width, it has the remaining width in the width of the main window

        # build the notebook GUI
        self.wnd_notebook_build(self.paned_wnd_frm_4)

        # create an instance of the C_ODF_DATA class
        self.odf_data = C_ODF_DATA()

        # create an instance of the C_ODF_HW2GO class
        self.odf_hw2go = C_ODF_HW2GO()

        self.reset_all_data()

        # update the status of GUI widgets
        self.gui_status_update_lists()
        self.gui_status_update_buttons()
        self.gui_status_update_notebook()

        # display the logs resulting from the init of the application if any
        self.logs_update()

        # launch a timer to complete some initialization operations 200ms after the application start
        # to permit an application bring up as fast as possible
        self.wnd_main.after(20, self.init_complete)

        return self.wnd_main

    #-------------------------------------------------------------------------------------------------
    def wnd_main_quit(self):
        # (GUI event callback) the user has clicked on the button "Quit" or window top-right "X"

        if self.can_i_make_change(is_odf_changed=True):
            # the user has saved his modifications if he wanted and has not canceled the operation

            # save application data
            self.app_data_save()

            # destroy the main window
            self.wnd_main.destroy()

            # stop an eventual audio sample playback in progress in the file viewer
            self.viewer_sample_stop()

    #-------------------------------------------------------------------------------------------------
    def wnd_main_key_ctrl_s(self, event):
        # (GUI event callback) the user has pressed on the ctrl + s or ctrl + S keyboard keys

        if self.object_text_changes_apply() and self.odf_data_changed:
            # changes have been made in the ODF data : save them
            self.file_save()

    #-------------------------------------------------------------------------------------------------
    def wnd_main_key_press(self, event):
        # (GUI event callback) the user has pressed a keyboard key

        if event.keysym in ('Control_L', 'Control_R') and not self.is_key_control_pressed:
            # control left or right key is pressed
            self.is_key_control_pressed = True
            # update the mouse cursor aspect
            self.object_b1_motion()

        elif event.keysym == 'Delete' and self.is_focus_on_objects_lists:
            self.object_delete()

    #-------------------------------------------------------------------------------------------------
    def wnd_main_key_release(self, event):
        # (GUI event callback) the user has released a keyboard key

        if event.keysym in ('Control_L', 'Control_R') and self.is_key_control_pressed:
            # control left or right key is released
            self.is_key_control_pressed = False
            # update the mouse cursor aspect
            self.object_b1_motion()

    #-------------------------------------------------------------------------------------------------
    def widget_mouse_enter(self, event, widget):
        # (GUI event callback) the mouse is entered inside the area of the given widget
        # used for PanedWindow widgets only as they are Tk widgets and not a Ttk widgets.
        # Tk does not manage natively the background color change on mouse hovering
        self.entered_widget = widget
        # change the background color of the widget only 200ms later to avoid having all the paned window separators flashing too easily
        self.entered_widget.after(200, self.widget_mouse_enter_delay)

    def widget_mouse_enter_delay(self):
        if self.entered_widget != None:
            # change the background color of the widget to put it in evidence
            self.entered_widget.config(background=COLOR_BACKGROUND2)

    def widget_mouse_leave(self, event, widget):
        # (GUI event callback) the mouse has left the area of the given widget
        if self.entered_widget != None:
            self.entered_widget.after_cancel(self.entered_widget)
            self.entered_widget = None
        # restore the default background color of the widget
        widget.config(background=COLOR_BACKGROUND1)

    #-------------------------------------------------------------------------------------------------
    def init_complete(self):
        # function to complete the initialization of the application

        # load the help text
        self.help_file_load()

        # load the GO objects templates
        self.odf_data.objects_templates_load()

        self.logs_update()

    #-------------------------------------------------------------------------------------------------
    def app_data_load(self):
        # load application configuration data from a .cfg file located in the folder of the OdfEdit file (.py or .exe)

        # initialize the saved application data with default values
        self.odf_recent_opened_list = []
        self.odf_save_encoding.set(ENCODING_UTF8_BOM)
        self.hw2go_convert_alt_ranks_bool.set(False)
        self.hw2go_alt_ranks_in_sep_ranks_bool.set(False)
        self.hw2go_pitch_tuning_metadata_bool.set(False)
        self.hw2go_pitch_tuning_filename_bool.set(False)
        self.hw2go_convert_alt_scr_layers_bool.set(False)
        self.hw2go_convert_unused_ranks_bool.set(False)
        self.hw2go_not_convert_keys_noises_bool.set(False)

        self.objects_list_width = 250
        self.objects_tree_width = 250
        self.object_editor_width = 500
        self.object_links_heigh = 200
        self.wnd_main_geometry = '1600x800+50+50'

        try:
            # load the dictionary stored in the config file
            data_dic = {}
            with open('OdfEdit.cfg', 'r') as f:
                data_dic = eval(f.read())

            # recover each data from the dictionary if they are defined in the config file and are valid
            if 'recent_odf_dir_list' in data_dic.keys() and isinstance(data_dic['recent_odf_dir_list'], list):
                self.odf_recent_opened_list = data_dic['recent_odf_dir_list']
            if 'file_save_encoding' in data_dic.keys() and data_dic['file_save_encoding'] in (ENCODING_ISO_8859_1, ENCODING_UTF8_BOM):
                self.odf_save_encoding.set(data_dic['file_save_encoding'])
            if 'convert_alt_ranks' in data_dic.keys() and data_dic['convert_alt_ranks'] in (True, False):
                self.hw2go_convert_alt_ranks_bool.set(data_dic['convert_alt_ranks'])
            if 'separate_alt_ranks' in data_dic.keys() and data_dic['separate_alt_ranks'] in (True, False):
                self.hw2go_alt_ranks_in_sep_ranks_bool.set(data_dic['separate_alt_ranks'])
            if 'pitch_tuning_metadata' in data_dic.keys() and data_dic['pitch_tuning_metadata'] in (True, False):
                self.hw2go_pitch_tuning_metadata_bool.set(data_dic['pitch_tuning_metadata'])
            if 'pitch_tuning_filename' in data_dic.keys() and data_dic['pitch_tuning_filename'] in (True, False):
                self.hw2go_pitch_tuning_filename_bool.set(data_dic['pitch_tuning_filename'])
            if 'convert_alt_scr_layers' in data_dic.keys() and data_dic['convert_alt_scr_layers'] in (True, False):
                self.hw2go_convert_alt_scr_layers_bool.set(data_dic['convert_alt_scr_layers'])
            if 'convert_unused_ranks' in data_dic.keys() and data_dic['convert_unused_ranks'] in (True, False):
                self.hw2go_convert_unused_ranks_bool.set(data_dic['convert_unused_ranks'])
            if 'not_convert_keys_noise' in data_dic.keys() and data_dic['not_convert_keys_noise'] in (True, False):
                self.hw2go_not_convert_keys_noises_bool.set(data_dic['not_convert_keys_noise'])

            if 'objects_list_width' in data_dic.keys() and str(data_dic['objects_list_width']).isdigit():
                self.objects_list_width = data_dic['objects_list_width']
            if 'objects_tree_width' in data_dic.keys() and str(data_dic['objects_tree_width']).isdigit():
                self.objects_tree_width = data_dic['objects_tree_width']
            if 'object_editor_width' in data_dic.keys() and str(data_dic['object_editor_width']).isdigit():
                self.object_editor_width = data_dic['object_editor_width']
            if 'object_links_heigh' in data_dic.keys() and str(data_dic['object_links_heigh']).isdigit():
                self.object_links_heigh = data_dic['object_links_heigh']
            if 'wnd_main_geometry' in data_dic.keys() and isinstance(data_dic['wnd_main_geometry'], str):
                self.wnd_main_geometry = data_dic['wnd_main_geometry']

        except:
            # issue occured to read the config file : we keep the default values set before
            pass

    #-------------------------------------------------------------------------------------------------
    def app_data_save(self):
        # save application configuration data in a .cfg file located in the folder of the OdfEdit file (.py or .exe)

        # build a dictionary with the application data to save
        data_dic = {}

        if len(self.odf_recent_opened_list) > 21: # save maximum 21 recent opened files (last opened one + 20 previous one)
            self.odf_recent_opened_list = self.odf_recent_opened_list[:21]
        data_dic['recent_odf_dir_list'] = self.odf_recent_opened_list

        data_dic['file_save_encoding'] = self.odf_save_encoding.get()
        data_dic['convert_alt_ranks'] = self.hw2go_convert_alt_ranks_bool.get()
        data_dic['separate_alt_ranks'] = self.hw2go_alt_ranks_in_sep_ranks_bool.get()
        data_dic['pitch_tuning_metadata'] = self.hw2go_pitch_tuning_metadata_bool.get()
        data_dic['pitch_tuning_filename'] = self.hw2go_pitch_tuning_filename_bool.get()
        data_dic['convert_alt_scr_layers'] = self.hw2go_convert_alt_scr_layers_bool.get()
        data_dic['convert_unused_ranks'] = self.hw2go_convert_unused_ranks_bool.get()
        data_dic['not_convert_keys_noise'] = self.hw2go_not_convert_keys_noises_bool.get()

        data_dic['objects_list_width'] = self.paned_wnd_frm_1.winfo_width()
        data_dic['objects_tree_width'] = self.paned_wnd_frm_2.winfo_width()
        data_dic['object_editor_width'] = self.paned_wnd_frm_3.winfo_width()
        data_dic['object_links_heigh'] = self.obj_paned_wnd_frm_1.winfo_height()
        data_dic['wnd_main_geometry'] = self.wnd_main.winfo_geometry()

        # save the dictionnary in the config file
        with open('OdfEdit.cfg', 'w') as f:
            f.write(str(data_dic))

    #-------------------------------------------------------------------------------------------------
    def file_new(self, ignore_changes=False):
        # (GUI event callback) the user has clicked on the button "New"
        # do a reset of the objects list/tree, edit box and ODF data

        if ignore_changes or self.can_i_make_change(is_odf_changed=True):
            # the user has saved his modifications if he wanted and has not canceled the operation

            # reset the various data
            self.reset_all_data()

            # update the various GUI widgets
            self.objects_list_update()
            self.objects_list_update_hw()
            self.objects_tree_update()
            self.object_text_update()
            self.object_links_list_update()
            self.gui_status_update_buttons()
            self.gui_status_update_notebook()
            self.notebook.hide(self.frm_hw_browser)

    #-------------------------------------------------------------------------------------------------
    def file_open(self, file_name=None):
        # (GUI event callback) the user has clicked on the button "Open" or the provided file name has to be opened

        if self.can_i_make_change(is_odf_changed=True):
            # the user has saved his modifications if he wanted and has not canceled the operation

            if file_name == None:
                # let the user select the ODF file to open, using the directory of the last opened one
                if len(self.odf_recent_opened_list) > 0:
                    initial_dir = os.path.dirname(self.odf_recent_opened_list[0])
                else:
                    initial_dir = ''
                file_name = fdialog.askopenfilename(title='Open an Organ Definition File (ODF)', initialdir = initial_dir, filetypes=[('All supported ODF', '*.organ *.xml *.Organ_Hauptwerk_xml'), ('GrandOrgue ODF', '*.organ'), ('Hauptwerk ODF', '*.xml *.Organ_Hauptwerk_xml')])

            if len(file_name) > 0:
                # a file has been selected by the user or given in parameter of the function

                self.is_loading = True

                self.file_new(True)  # ignore the changes in the file_new function, already checked above

                # show the mouse cursor watch
                self.wnd_main['cursor'] = 'watch'
                self.txt_object_text['cursor'] = 'watch'
                self.txt_events_log['cursor'] = 'watch'
                self.recent_odf_list_close()
                self.wnd_main.update()

                file_name = path2ospath(file_name)

                # store the file extension of the selected file
                file_extension = os.path.splitext(file_name)[1]

                # store the file to open
                self.recent_odf_list_update(file_name)

                # select the logs tab of the notebook to show the file opening logs
                self.notebook.select(self.frm_logs)
                self.notebook.hide(self.frm_hw_browser)

                if file_extension in ('.xml', '.Organ_Hauptwerk_xml'):
                    # Hauptwerk ODF selected : build a GrandOrgue ODF which uses the Hauptwerk sample set

                    # define the name of the GO ODF to build according to the name of the HW ODF : same path and file name, only the extension is changed
                    if file_name.endswith('.Organ.Hauptwerk.xml'):
                        GO_odf_file_name = file_name[:-len('.Organ.Hauptwerk.xml')] + '.organ'
                    elif file_name.endswith('.Organ_Hauptwerk_xml'):
                        GO_odf_file_name = file_name[:-len('.Organ_Hauptwerk_xml')] + '.organ'
                    else:
                        GO_odf_file_name = file_name[:-len('.xml')] + '.organ'

                    # legal message displayed to the user before to start the ODF building
                    if not self.hw2go_warning_displayed_bool and not DEV_MODE:
                        answer = AskUserAnswerQuestion(self.wnd_main, "OdfEdit", HW_CONV_MSG, ['OK', 'Cancel'], 'OK')
                    else:
                        answer = 'OK'

                    if answer == 'OK':
                        self.hw2go_warning_displayed_bool = True
                        T0 = time.time()
                        if self.odf_hw2go.GO_ODF_build_from_HW_ODF(file_name, GO_odf_file_name, self.progress_status_update,
                                                                   self.hw2go_convert_alt_ranks_bool.get(),
                                                                   self.hw2go_alt_ranks_in_sep_ranks_bool.get(),
                                                                   self.hw2go_pitch_tuning_metadata_bool.get(),
                                                                   self.hw2go_pitch_tuning_filename_bool.get(),
                                                                   self.hw2go_convert_alt_scr_layers_bool.get(),
                                                                   self.hw2go_not_convert_keys_noises_bool.get(),
                                                                   self.hw2go_convert_unused_ranks_bool.get(),
                                                                   self.odf_save_encoding.get()):
                            # the GO ODF building has succeeded
                            if LOG_HW2GO_perfo: print(f'HW to GO conversion done in {time.time() - T0:0.3f} seconds')
                            self.is_loaded_hw_odf = True
                            # display the HW objects notebook tab and list inside this tab
                            self.notebook.add(self.frm_hw_browser)
                            # the built GO ODF will be then loaded
                            file_name = GO_odf_file_name
                        else:
                            logs.add('ERROR : something went wrong while converting the Hauptwerk ODF in a GrandOrgue ODF')
                            file_name = ''
                        self.logs_update()
                    else:
                        file_name = ''

                if file_name != '':
                    # GrandOrgue ODF selected or built from a Hauptwerk ODF
                    self.progress_status_update('Loading the ODF...')
                    if self.odf_data.load_from_file(file_name):
                        # the file has been loaded properly
                        self.is_loaded_odf = True
                        # store the opened file in the last opened ODFs list
                        self.recent_odf_list_update(file_name)
                        # reset the check files names flag
                        self.odf_check_files_names = None

                self.is_loading = False

                # update the objects list / tree / text
                self.objects_list_update()
                self.objects_list_update_hw()
                self.objects_tree_update()
                self.object_text_update()
                self.object_links_list_update()
                self.gui_status_update_buttons()
                self.gui_status_update_notebook()

                self.logs_update()

            self.wnd_main['cursor'] = ''
            self.txt_object_text['cursor'] = ''
            self.txt_events_log['cursor'] = ''

    #-------------------------------------------------------------------------------------------------
    def file_save(self):
        # (GUI event callback) the user has clicked on the button "Save"
        # return True or False whether the saving has been done or not in the current loaded ODF

        return self.file_saveas(self.odf_data.odf_file_name)

    #-------------------------------------------------------------------------------------------------
    def file_saveas(self, file_name = ''):
        # (GUI event callback) the user has clicked on the button "Save as"
        # return True or False whether the saving has been done or not

        if self.can_i_make_change():
            if file_name == '':
                # let the user select the ODF file in which to make the saving
                file_name = path2ospath(fdialog.asksaveasfilename(title='Save in ODF...', filetypes=[('ODF', '*.organ')]))

            if file_name != '' and self.odf_data.save_to_file(file_name, self.odf_save_encoding.get()):
                # a file has been selected by the user
                # and the ODF data have been correctly saved

                # store the file to save
                self.recent_odf_list_update(file_name)

                self.odf_data_changed = False
                self.edited_object_changed = False
                self.gui_status_update_buttons()
                data_saved = True
            else:
                data_saved = False
        else:
            data_saved = False

        self.logs_update()
        return data_saved

    #-------------------------------------------------------------------------------------------------
    def can_i_make_change(self, is_odf_changed=False):
        # before a file change or window closing (is_odf_changed to set at True) or a selected object change
        # ask to the user if he wants to save his modifications if any, if the answer is yes then do it
        # return True if the change to do can be done

        is_change_ok = True

        if is_odf_changed:
            # the coming change will be at ODF level
            if self.edited_object_changed:
                # the edited object have been chnaged
                # ask to the user if he wants to apply changes and to save the ODF data
                if self.edited_object_uid != None:
                    answer = AskUserAnswerQuestion(self.wnd_main, "OdfEdit", f"Do you want to apply and save changes made in {self.edited_object_uid} ?", ['Yes', 'No', 'Cancel'], 'Cancel')
                else:
                    answer = AskUserAnswerQuestion(self.wnd_main, "OdfEdit", "Do you want to apply and save changes ?", ['Yes', 'No', 'Cancel'], 'Cancel')
                if answer == 'Yes':
                    if not self.object_text_changes_apply():
                        is_change_ok = False
                    elif not self.file_save():
                        is_change_ok = False
                elif answer != 'No':
                    is_change_ok = False

            elif self.odf_data_changed:
                # data have changed in the ODF data
                # ask to the user if he wants to save the changes done
                answer = AskUserAnswerQuestion(self.wnd_main, "OdfEdit", "Do you want to save changes ?", ['Yes', 'No', 'Cancel'], 'Cancel')
                if answer == 'Yes':
                    if not self.file_save():
                        is_change_ok = False
                elif answer != 'No':
                    is_change_ok = False

        elif self.edited_object_changed:
            # the edited object have been changed
            # ask to the user if he wants to apply changes
            if self.edited_object_uid != None:
                answer = AskUserAnswerQuestion(self.wnd_main, "OdfEdit", f"Do you want to apply changes made in {self.edited_object_uid} ?", ['Yes', 'No', 'Cancel'], 'Cancel')
            else:
                answer = AskUserAnswerQuestion(self.wnd_main, "OdfEdit", "Do you want to apply changes ?", ['Yes', 'No', 'Cancel'], 'Cancel')
            if answer == 'Yes':
                if not self.object_text_changes_apply():
                    is_change_ok = False
            elif answer != 'No':
                is_change_ok = False

        return is_change_ok

    #-------------------------------------------------------------------------------------------------
    def gui_events_block(self):
        # set the flag which blocks the GUI events processing and launch a blocking timer of 100ms

        if self.gui_events_blocked == False:
            # the events are not yet blocked
            self.gui_events_blocked = True
            # launch a timer which will unblock the events 100ms later, the time for the modified widgets to have completed their update
            self.wnd_main.after(100, self.gui_events_unblock)

    #-------------------------------------------------------------------------------------------------
    def gui_events_unblock(self):
        # (GUI event callback) end of a timer started by the function gui_events_block
        # reset the flag which blocks the GUI events processing

        self.gui_events_blocked = False

    #-------------------------------------------------------------------------------------------------
    def gui_status_update_buttons(self):
        # update the status of some GUI widgets in a single time, according to some status of the application

        # recover the number of defined objects in the ODF data
        objects_nb = self.odf_data.objects_number_get()

        # recover the list of possible parents and children objects of the selected object
        (possible_parents_list, possible_children_list) = self.odf_data.object_poss_kinship_list_get(self.edited_object_uid)
        possible_children_type_list = self.odf_data.object_poss_children_type_list_get(self.edited_object_uid)

        root_object_not_def = bool('Header' not in self.odf_data.objects_list_get() or 'Organ' not in self.odf_data.objects_list_get())
        # True if the root object Header or Organ is not present in the ODF

        # get if an object is selected in the objects tree
        objects_tree_selected = len(self.trv_objects_tree.selection()) > 0

        # button "New"
        self.btn_odf_new['state'] = tk.NORMAL if objects_nb > 0 else tk.DISABLED

        # button "Open"
        self.btn_odf_open['state'] = tk.NORMAL if not self.is_loading else tk.DISABLED

        # button opening the last opened ODF list (normal button if more than 1 item in the list, the first item is the current opened ODF, not shown)
        min_list_len = 1 if not self.is_loaded_odf else 2
        self.btn_odf_open_last['state'] = tk.NORMAL if len(self.odf_recent_opened_list) >= min_list_len and not self.is_loading else tk.DISABLED

        # button "Save"
        self.btn_odf_save['state'] = tk.NORMAL if (self.odf_data.odf_file_name != '' and self.odf_data_changed) else tk.DISABLED

        # button "Save as"
        self.btn_odf_saveas['state'] = tk.NORMAL if objects_nb > 0 else tk.DISABLED
        self.btn_odf_saveas.configure(style='RedText.TButton' if (self.odf_data.odf_file_name == '' and self.odf_data_changed) else 'TButton')

        # button "Do check"
        self.btn_data_check['state'] = tk.NORMAL if objects_nb > 0 else tk.DISABLED

        # button for opening the general menu
        self.btn_gen_menu['state'] = tk.NORMAL if not self.is_loading else tk.DISABLED

        # button "Collapse"
        self.btn_collapse_tree_node['state'] = tk.NORMAL if objects_tree_selected else tk.DISABLED

        # button "Expand"
        self.btn_expand_tree_node['state'] = tk.NORMAL if objects_tree_selected else tk.DISABLED

        # button "Apply changes"
        self.btn_object_apply_chg['state'] = tk.NORMAL if self.edited_object_changed else tk.DISABLED

        # button "Add"
        self.btn_object_add['state'] = tk.NORMAL if (len(possible_children_type_list) > 0 or root_object_not_def) and self.selected_object_app == 'GO' and not self.is_loading else tk.DISABLED

        # button "Parents"
        self.btn_object_parents['state'] = tk.NORMAL if len(possible_parents_list) > 0 else tk.DISABLED

        # button "Children"
        self.btn_object_children['state'] = tk.NORMAL if len(possible_children_list) > 0 else tk.DISABLED

        # button "Rename"
        self.btn_object_rename['state'] = tk.NORMAL if (self.edited_object_uid != None and self.selected_object_app == 'GO' and self.edited_object_uid[-3:].isdigit()) else tk.DISABLED

        # button "Delete"
        self.btn_object_delete['state'] = tk.NORMAL if (self.edited_object_uid != None and self.edited_object_uid != 'Organ' and self.selected_object_app == 'GO') else tk.DISABLED

        # button "Unselect"
        self.btn_unselect['state'] = tk.NORMAL if (self.edited_object_uid != None and self.selected_object_app == 'GO') else tk.DISABLED

        # button "Show help"
        self.btn_show_help['state'] = tk.NORMAL if (self.edited_object_uid not in (None, 'Header') and self.selected_object_app == 'GO') else tk.DISABLED

        # label with the loaded ODF name
        if not self.is_loaded_odf:
            if objects_nb == 0:
                self.lab_odf_file_name.config(text='Click on the button "Open" to load a GrandOrgue or Hauptwerk ODF, or "Add" to create new sections')
            else:
                self.lab_odf_file_name.config(text='Click on the button "Save as" to define a file name')
        else:
            self.lab_odf_file_name.config(text=self.odf_recent_opened_list[0])
        self.lab_odf_file_name['foreground'] = TEXT_COLOR

        # label with the number of objects
        if 'Header' in self.odf_data.objects_list_get():
            objects_nb -= 1  # do not count the header in the sections number to display at the top of the sections list

        if objects_nb == 0:
            self.lab_objects_nb.config(text="")
        elif objects_nb == 1:
            self.lab_objects_nb.config(text="1 section")
        else:
            self.lab_objects_nb.config(text=f"{objects_nb} sections")

    #-------------------------------------------------------------------------------------------------
    def gui_status_update_lists(self, object_see_list=False, object_see_tree=False):
        # update the selections in the GUI lists/tree
        # if one object_see flag is enabled, the corresponding list or tree is moved to see the selected object

        # to block the GUI events triggered by the GUI updates done in this function
        self.gui_events_block()

        # GO objects list
        # highlight and make visible the item corresponding to the selected object UID
        for i in range(0, self.lst_objects_list.size()):
            object_uid = self.lst_objects_list.get(i).split(' ')[0]
            if object_uid == self.selected_object_uid and self.selected_object_app == 'GO':
                # the current item corresponds to the selected GO object UID : highlight it
                self.lst_objects_list.itemconfig(i, foreground=TEXT_COLOR, background=COLOR_SAME_UID_ITEM)
                if object_see_list:
                    self.lst_objects_list.see(i)
            else:
                self.lst_objects_list.itemconfig(i, foreground=TEXT_COLOR, background=COLOR_BG_LIST)

            if self.focused_objects_widget == self.lst_objects_list and self.focused_sel_item_id == object_uid:
                # the current item has the focus : select it
                self.lst_objects_list.selection_set(i)
            else:
                self.lst_objects_list.selection_clear(i)

        # GO linked objects list
        for i in range(0, self.lst_links_list.size()):
            object_uid = self.lst_links_list.get(i).strip().split(' ')[0]
            if object_uid == self.selected_object_uid and self.selected_object_app == 'GO':
                # the current item corresponds to the selected GO object UID : highlight it
                self.lst_links_list.itemconfig(i, foreground=TEXT_COLOR, background=COLOR_SAME_UID_ITEM)
            else:
                self.lst_links_list.itemconfig(i, foreground=TEXT_COLOR, background=COLOR_BG_LIST)

            if self.focused_objects_widget == self.lst_links_list and self.focused_sel_item_id == object_uid:
                # the current item has the focus : select it
                self.lst_links_list.selection_set(i)
            else:
                self.lst_links_list.selection_clear(i)

        # GO objects tree
        if self.selected_object_app == 'GO':
            object_uid = self.selected_object_uid
        else:
            object_uid = None
        # select the items corresponding to the selected GO object UID if any
        self.one_seen = False
        for iid in self.trv_objects_tree.get_children():
            self.objects_tree_nodes_select(iid, object_uid, object_see_tree)

    #-------------------------------------------------------------------------------------------------
    def gen_menu_open(self):
        # (GUI event callback) the user has clicked on the button Menu to open the general menu

        self.general_menu.tk_popup(self.btn_gen_menu.winfo_rootx() + self.btn_gen_menu.winfo_width(), self.btn_gen_menu.winfo_rooty())

    #-------------------------------------------------------------------------------------------------
    def gen_menu_close(self, event=None):
        # (GUI event callback) the menu has lost its focus

        self.general_menu.grab_release()

    #-------------------------------------------------------------------------------------------------
    def gen_menu_about(self):
        # (GUI event callback) the user has clicked on the item "About..." of the general Menu

        AskUserAnswerQuestion(self.wnd_main, 'About...', f'OdfEdit {APP_VERSION} - {RELEASE_DATE}\n\ngithub.com/GrandOrgue/OdfEdit', ['Close'], 'Close')

    #-------------------------------------------------------------------------------------------------
    def gen_menu_references_sort(self):
        # (GUI event callback) the user has clicked on the item "Sort references in selected section" of the general Menu

        if self.selected_object_app == 'GO':
            if self.edited_object_uid == None:
                AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', 'A section must be selected first.', ['Close'], 'Close')

            elif self.can_i_make_change():
                if 'OK' == AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f'This will sort in the section {self.edited_object_uid} the references to other sections.\nIs it ok to proceed ?', ['OK', 'Cancel'], 'OK'):
                    if self.odf_data.object_children_ref_all_sort(self.edited_object_uid):
                        # changes have been done in the edited object
                        self.odf_data_changed = True
                        self.object_text_update()
                        self.gui_status_update_buttons()
                    self.logs_update()

        else:  # self.selected_object_app is 'HW'
            AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', 'Action not applicable for a Hauptwerk section.', ['Close'], 'Close')

    #-------------------------------------------------------------------------------------------------
    def recent_odf_list_open(self):
        # (GUI event callback) the user has clicked on the button to show the list of recent opened ODFs

        if self.lst_recent_odf == None:
            # the list is not yet visible
            self.lst_recent_odf = tk.Listbox(self.wnd_main, bg=COLOR_BACKGROUND1, font=TEXT_FONT, fg=TEXT_COLOR, selectbackground=COLOR_SELECTED_ITEM,
                                             exportselection=0, selectmode=tk.SINGLE, activestyle=tk.NONE, height=20)
            self.lst_recent_odf.place(x=self.btn_odf_open.winfo_x(),
                                      y=self.btn_odf_open.winfo_y() + self.btn_odf_open.winfo_height(),
                                      width=900)

            self.lst_recent_odf.bind('<ButtonRelease-1>', self.recent_odf_list_selected)
            self.lst_recent_odf.bind('<Leave>', self.recent_odf_list_close)
            self.lst_recent_odf.bind('<Motion>', self.recent_odf_list_mouse_motion)
            self.lst_recent_odf.bind("<MouseWheel>", lambda event: "break") # to avoid mouse scrolling the list

            # add in the list widget the ODFs but the first one which is the currently opened
            for i, odf_name in enumerate(self.odf_recent_opened_list):
                if i > 0 or not self.is_loaded_odf:
                    # if an ODF is loaded, does not put the first ODF file name of the list, it is the current loaded ODF
                    if os.path.exists(odf_name):
                        # the file actually exists, add it in the list shown to the user
                        self.lst_recent_odf.insert(tk.END, odf_name)
            self.lst_recent_odf.focus_set()
        else:
            # the list is visible, close it
            self.recent_odf_list_close()

    #-------------------------------------------------------------------------------------------------
    def recent_odf_list_close(self, event=None):
        # (GUI event callback) the mouse has left the list area, or it has to be closed

        if self.lst_recent_odf != None:
            self.lst_recent_odf.destroy()
            self.lst_recent_odf = None

    #-------------------------------------------------------------------------------------------------
    def recent_odf_list_update(self, file_name):
        # update the list of recent opened ODFs with the given file name, place it at the top of the list

        if file_name in self.odf_recent_opened_list:
            # if the file name is already in the resent opened ODFs list, remove it
            self.odf_recent_opened_list.remove(file_name)
        # insert the file name at the top of the recent opened ODFs list
        self.odf_recent_opened_list.insert(0, file_name)

    #-------------------------------------------------------------------------------------------------
    def recent_odf_list_clear(self):
        # clear the content of the list of recent opened ODFs, but the first item which is the currently opened ODF

        if len(self.odf_recent_opened_list) > 1:
            del self.odf_recent_opened_list[1:]
            self.gui_status_update_buttons()

    #-------------------------------------------------------------------------------------------------
    def recent_odf_list_mouse_motion(self, event):
        # (GUI event callback) the mouse is moving in the recent opened ODFs list area

        # select the list item which is under the mouse cursor, and unselect other items
        index = self.lst_recent_odf.nearest(event.y)
        for i in range(0, self.lst_recent_odf.size()):
            if i == index:
                self.lst_recent_odf.selection_set(i)
            else:
                self.lst_recent_odf.selection_clear(i)

    #-------------------------------------------------------------------------------------------------
    def recent_odf_list_selected(self, event):
        # (GUI event callback) the user has clicked on an item of the recent opened ODFs list

        cursel_tuple = self.lst_recent_odf.curselection()
        if len(cursel_tuple) > 0:
            # an item is selected, recover its text
            selected_line_indice = cursel_tuple[0]
            file_name = self.lst_recent_odf.get(selected_line_indice)
            self.recent_odf_list_close()

            self.file_open(file_name)

    #-------------------------------------------------------------------------------------------------
    def objects_widget_focus_out(self, event):
        # (GUI event callback) an objects widget (list or tree) has left the focus

        if event.widget == self.focused_objects_widget:
            # the widget which has lost the focus is the currently selected list or tree widget
            self.is_focus_on_objects_lists = False

    #-------------------------------------------------------------------------------------------------
    def objects_list_update(self):
        # do an update of the objects list widget content

        # clear the objects list widget content
        self.lst_objects_list.delete(0, 'end')

        # update the widget with the current list of sorted objects UID and name
        # puting the 'Header' and 'Organ' objects in first position of the list and excluding the Header object
        organ_text = None
        header_text = None
        for object_uid in sorted(self.odf_data.objects_list_get()):
            item_text = self.odf_data.object_names_get(object_uid)
            if object_uid == 'Organ':
                organ_text = item_text
            elif object_uid =='Header':
                header_text = item_text
            else:
                self.lst_objects_list.insert(tk.END, item_text)

        if organ_text != None:
            self.lst_objects_list.insert(0, organ_text)
        if header_text != None:
            self.lst_objects_list.insert(0, header_text)

    #-------------------------------------------------------------------------------------------------
    def objects_list_selected(self, event):
        # (GUI event callback) the user has selected an item in the objects list widget

        if self.gui_events_blocked: return

        self.is_focus_on_objects_lists = True

        # get the line number of the selected item in the list
        cursel_tuple = self.lst_objects_list.curselection()
        if len(cursel_tuple) > 0:
            selected_line_indice = cursel_tuple[0]
        else:
            selected_line_indice = None

        if selected_line_indice != None:
            # an item has been selected in the list

            if event.keysym in ('Up', 'Down'):
                if event.keysym == 'Down' and selected_line_indice < self.lst_objects_list.size() - 1:
                    selected_line_indice += 1
                elif selected_line_indice > 0:
                    selected_line_indice -= 1
                self.lst_objects_list.selection_clear(0, tk.END)
                self.lst_objects_list.selection_set(selected_line_indice)

            # ignore the mouse button 1 release if the edited object has been changed : to let the user say if he wants to save the change or not or cancel
            self.ignore_b1_release = self.edited_object_changed

            if self.can_i_make_change():
                # the user has saved his modifications if he wanted and has not canceled the selection

                # recover in the objects list widget the UID of the selected object (before the first space in the selected item text)
                object_uid = self.lst_objects_list.get(selected_line_indice).split(' ')[0]
                self.selected_object_app = 'GO'
                self.selected_object_uid = object_uid
                self.selected_linked_uid = None
                self.edited_object_uid = self.selected_object_uid

                self.focused_objects_widget = self.lst_objects_list
                self.focused_sel_item_id = object_uid

                if self.ignore_b1_release:
                    # do now the processing of the mouse button 1 release
                    self.ignore_b1_release = False
                    self.object_b1_release(event)
                # else the widgets will be updated on mouse button release

            else:
                # the user had canceled the selection
                self.ignore_b1_release = False
                self.gui_status_update_lists()

    #-------------------------------------------------------------------------------------------------
    def objects_list_selected_dbl(self, event):
        # (GUI event callback) the user has double-clicked an item in the objects list widget

        # update the status of GUI widgets and for to see the objects of the tree having the selected UID
        self.gui_status_update_lists(False, True)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_update(self):
        # do an update of the objects tree widget

        # memorize the nodes of the tree which are currently opened to restore them in the updated list
        self.opened_objects_iid_list.clear()
        for iid in self.trv_objects_tree.get_children(''):
            self.objects_tree_node_and_children_opened_nodes_get(iid)

        # delete all the nodes of the tree
        for node_iid in self.trv_objects_tree.get_children():
            self.trv_objects_tree.delete(node_iid)

        odf_objects_list = self.odf_data.objects_list_get()
        if len(odf_objects_list) > 0:
            # there are existing objects

            # treeview insert syntax : insert(parent_iid or '', position (0 or 'end'), node_iid, keyword arguments...)

            if 'Header' in odf_objects_list:
                # place the Header object in first position of the tree
                iid = self.trv_objects_tree.insert('', 'end', 'Header', text='Header')

            if 'Organ' in odf_objects_list:
                # place the Organ object after the Header or in first position if no header
                organ_node_iid = self.trv_objects_tree.insert('', 'end', 'Organ', text=self.odf_data.object_names_get('Organ'), open=True)
            else:
                organ_node_iid = ''

            depth = 0
            for object_uid in sorted(odf_objects_list):
                if object_uid not in ('Header', 'Organ'):
                    # scan the objects UID of the objects list which are not Header or Organ
                    object_type = self.odf_data.object_type_get(object_uid)
                    if object_type in ('General', 'Manual', 'Panel', 'WindchestGroup', 'Image', 'tk.Label', 'ReversiblePiston', 'SetterElement'):
                        # put the current object under the 'Organ' node
                        self.objects_tree_child_add(organ_node_iid, object_uid, depth)
                    elif len(self.odf_data.object_kinship_list_get(object_uid, TO_PARENT)) == 0:
                        # the object has no parent, put it at the root of the tree
                        self.objects_tree_child_add('', object_uid, depth)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_child_add(self, parent_node_iid, child_object_uid, depth):
        # recursive function to insert in the objects tree widget the given child object UID under the given parent node ID

        # insert the given child object in the tree under the given parent node iid
        child_node_iid = self.trv_objects_tree.insert(parent_node_iid, 'end', parent_node_iid + child_object_uid, text=self.odf_data.object_names_get(child_object_uid))

        if child_node_iid in self.opened_objects_iid_list:
            # the given child node iid has to be opened
            self.trv_objects_tree.item(child_node_iid, open=True)
            self.opened_objects_iid_list.remove(child_node_iid)

        # the given child becomes the parent for the next recursive call
        new_parent_node_iid = child_node_iid
        new_parent_uid = child_object_uid
        new_parent_type = self.odf_data.object_type_get(new_parent_uid)

        # scan the children of new parent to add the corresponding children nodes
        if self.odf_data.object_dic_get(new_parent_uid) != None:
            # the new parent object UID exists in the ODF dictionary
            if not(new_parent_type == 'Manual' and depth > 0):
                # it is not a Manual object at depth higher than 1 (to avoid Manual childs in the tree when Manual is not child of Organ)
                for child_uid in sorted(self.odf_data.object_kinship_list_get(new_parent_uid, TO_CHILD)):
                    self.objects_tree_child_add(new_parent_node_iid, child_uid, depth + 1)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_expand_all(self):
        # expend all the nodes of the objects tree

        for iid in self.trv_objects_tree.get_children(''):
            self.objects_tree_node_and_children_open(iid, True)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_expand_selected(self):
        # (GUI event callback) the user has pressed the button "Expand"
        # expend the node of the selected object and its children

        # get the iid of the selected node in the tree
        cursel_tuple = self.trv_objects_tree.selection()
        if len(cursel_tuple) > 0:
            selected_node_iid = cursel_tuple[0]
            self.objects_tree_node_and_children_open(selected_node_iid, True)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_collapse_all(self):
        # collapse all the nodes of the objects tree widget except the root and the 'Organ' nodes

        for iid in self.trv_objects_tree.get_children(''):
            self.objects_tree_node_and_children_open(iid, False)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_collapse_current(self):
        # (GUI event callback) the user has pressed the button "Expand"
        # collapse the node of the selected object and its children, except the root and the 'Organ' nodes

        # get the iid of the selected node in the tree
        cursel_tuple = self.trv_objects_tree.selection()
        if len(cursel_tuple) > 0:
            selected_node_iid = cursel_tuple[0]
            self.objects_tree_node_and_children_open(selected_node_iid, False)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_node_and_parents_open(self, node_iid):
        # recursive function to open the given node and his parents in the objects tree

        if node_iid != '':
            # the node exists
            self.trv_objects_tree.item(node_iid, open=True)

            # apply the open status to the parent node of node_iid
            self.objects_tree_node_and_parents_open(self.trv_objects_tree.parent(node_iid))

    #-------------------------------------------------------------------------------------------------
    def objects_tree_node_and_children_open(self, node_iid, open_status):
        # recursive function to open (if True, else close) the given node and his children in the objects tree

        if self.trv_objects_tree.item(node_iid, option='text').startswith('Organ'):
            # the Organ node must stay always opened
            self.trv_objects_tree.item(node_iid, open=True)
        else:
            self.trv_objects_tree.item(node_iid, open=open_status)

        # apply the open status to the child nodes
        for iid in self.trv_objects_tree.get_children(node_iid):
            self.objects_tree_node_and_children_open(iid, open_status)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_nodes_select(self, node_iid, object_uid, object_see_tree=False):
        # recursive function to select and make visible the nodes of the objects tree which contain the given object UID text

        if object_uid != None and self.trv_objects_tree.item(node_iid)['text'].split(' ')[0] == object_uid:
            # the node node_iid corresponds to the object UID : tag it and open it parents nodes
            self.trv_objects_tree.item(node_iid, tags=TAG_SAME_UID)
            # open the parents of the node so that the object is visible if it is requested by the user in the menu
            if object_see_tree:
                self.objects_tree_node_and_parents_open(self.trv_objects_tree.parent(node_iid))
                if not self.one_seen:
                    self.trv_objects_tree.see(node_iid)
                    self.one_seen = True
        else:
            # remove the tag on the node_iid if any
            self.trv_objects_tree.item(node_iid, tags=())

        if self.focused_objects_widget == self.trv_objects_tree and self.focused_sel_item_id != None and node_iid.endswith(self.focused_sel_item_id):
            # the current node has the focus (use of endswith in case the focused_sel_item_id doesn't fit the full path of the node iid)
            self.trv_objects_tree.selection_add(node_iid)
            self.trv_objects_tree.see(node_iid)
            self.one_seen = True
        else:
            self.trv_objects_tree.selection_remove(node_iid)

        if self.trv_objects_tree.item(node_iid, 'open') or object_see_tree:
            # the node_iid is opened so its children are visible, or the tree has to be automatically expanded to show the selected object UID
            # search to select the object_uid in the children of node_iid
            for iid in self.trv_objects_tree.get_children(node_iid):
                self.objects_tree_nodes_select(iid, object_uid, object_see_tree)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_node_show(self, object_uid, parent_uid, node_iid=''):
        # recursive function to show in the objects tree the node which has the given object UID under the given parent object UID

        if node_iid != '' and node_iid.endswith(parent_uid + object_uid):
            # the given node is matching the given UIDs (use of endswith in case the given parent + object UID doesn't fit the full path of the node iid)
            self.trv_objects_tree.see(node_iid)
            return True

        for iid in self.trv_objects_tree.get_children(node_iid):
            if self.objects_tree_node_show(object_uid, parent_uid, iid):
                return True

        return False

    #-------------------------------------------------------------------------------------------------
    def objects_tree_node_and_children_opened_nodes_get(self, node_iid):
        # recursive function to build the list of the opened nodes from the given node iid and its children

        if self.trv_objects_tree.item(node_iid, 'open'):
            self.opened_objects_iid_list.append(node_iid)

        # check the children of the given node_iid
        for iid in self.trv_objects_tree.get_children(node_iid):
            self.objects_tree_node_and_children_opened_nodes_get(iid)

    #-------------------------------------------------------------------------------------------------
    def objects_tree_selected(self, event):
        # (GUI event callback) the user has selected an item in the objects tree widget

        if self.gui_events_blocked: return

        self.is_focus_on_objects_lists = True

        # get the iid of the selected node in the tree
        cursel_tuple = self.trv_objects_tree.selection()
        if len(cursel_tuple) > 0:
            selected_node_iid = cursel_tuple[0]
        else:
            selected_node_iid = None

        if selected_node_iid != None:
            # an item has been selected in the objects tree

            # ignore the mouse button 1 release if the edited object has been changed : to let the user say if he wants to save the change or not or cancel
            self.ignore_b1_release = self.edited_object_changed

            if self.can_i_make_change():
                # the user has saved his modifications if he wanted and has not canceled the operation

                # recover the object UID
                object_uid = self.trv_objects_tree.item(selected_node_iid, option='text').split(' ')[0]
                self.selected_object_app = 'GO'
                self.selected_object_uid = object_uid
                self.selected_linked_uid = None
                self.edited_object_uid = self.selected_object_uid

                self.focused_objects_widget = self.trv_objects_tree
                self.focused_sel_item_id = selected_node_iid

                if self.ignore_b1_release:
                    # do now the processing of the mouse button 1 release
                    self.ignore_b1_release = False
                    self.object_b1_release(event)
                # else the widgets will be updated on mouse button release

            else:
                # the user had canceled the selection
                self.ignore_b1_release = False
                self.gui_status_update_lists()

    #-------------------------------------------------------------------------------------------------
    def object_links_list_update(self):
        # update the selected object links list

        tab = '    '

        # clear the objects list widget content
        self.lst_links_list.delete(0, tk.END)

        if self.selected_object_app == 'GO' and self.selected_object_uid not in (None, 'Header', 'Organ'):

            # display the selected object
            self.lst_links_list.insert('end', self.odf_data.object_names_get(self.selected_object_uid))

            # display the parents
            parents_list = sorted(self.odf_data.object_kinship_list_get(self.selected_object_uid, TO_PARENT))
            if len(parents_list) > 0:
                self.lst_links_list.insert('end', f'Parents : #{len(parents_list)}')
                for parent_uid in parents_list:
                    self.lst_links_list.insert('end', tab + self.odf_data.object_names_get(parent_uid))

            # display the children
            children_list = sorted(self.odf_data.object_kinship_list_get(self.selected_object_uid, TO_CHILD))
            if len(children_list) > 0:
                self.lst_links_list.insert('end', f'Children : #{len(children_list)}')
                for child_uid in children_list:
                    self.lst_links_list.insert('end', tab + self.odf_data.object_names_get(child_uid))

    #-------------------------------------------------------------------------------------------------
    def object_links_list_selected(self, event):
        # (GUI event callback) the user has selected an item in the object links list widget

        if self.gui_events_blocked: return

        self.is_focus_on_objects_lists = True

        # get the line number of the selected item in the list
        cursel_tuple = self.lst_links_list.curselection()
        if len(cursel_tuple) > 0:
            selected_line_indice = cursel_tuple[0]
        else:
            selected_line_indice = None

        if selected_line_indice != None:
            # an item has been selected in the object links list

            # ignore the mouse button 1 release if the edited object has been changed : to let the user say if he wants to save the change or not or cancel
            self.ignore_b1_release = self.edited_object_changed

            if self.can_i_make_change():
                # the user has saved his modifications if he wanted and has not canceled the operation

                # recover in the list widget the UID of the selected object (before the first space in the selected item text)
                object_uid = self.lst_links_list.get(selected_line_indice).strip().split(' ')[0]
                if object_uid not in ('Parents', 'Children', self.selected_object_uid):
                    self.selected_linked_uid = object_uid
                    self.edited_object_uid = self.selected_linked_uid
                    self.focused_sel_item_id = self.selected_linked_uid
                else:
                    self.selected_linked_uid = None
                    self.edited_object_uid = self.selected_object_uid
                    self.focused_sel_item_id = self.selected_object_uid

                self.focused_objects_widget = self.lst_links_list

                if self.ignore_b1_release:
                    # do now the processing of the mouse button 1 release
                    self.ignore_b1_release = False
                    self.object_b1_release(event)
                # else the widgets will be updated on mouse button release

            else:
                # the user had canceled the selection
                self.ignore_b1_release = False
                self.gui_status_update_lists()

    #-------------------------------------------------------------------------------------------------
    def object_links_list_selected_dbl(self, event):
        # (GUI event callback) the user has double-clicked an item in the object links list widget

        if self.selected_linked_uid != None:
            # make the selected linked UID the selected object UID
            self.selected_object_uid = self.selected_linked_uid
            self.selected_linked_uid = None
            self.ignore_b1_release = True

            # update the status of GUI widgets
            self.object_links_list_update()
            self.gui_status_update_lists(True)
            self.gui_status_update_buttons()

    #-------------------------------------------------------------------------------------------------
    def object_b1_motion(self, event=None):
        # (GUI event callback) the user has moved the mouse cursor inside the objects lists or tree with the button 1 pressed
        # if event = None, the mouse cursor position must be ignored

        overflown_widget = None
        overflown_object_uid = None
        if event != None:
            # identify which object UID of an objects list or tree is overflown by the mouse cursor
            overflown_widget = event.widget.winfo_containing(event.x_root, event.y_root)
            if overflown_widget == self.lst_objects_list:
                # the objects list is overflown by the mouse cursor
                index = self.lst_objects_list.nearest(event.y)
                overflown_object_uid = self.lst_objects_list.get(index).split(' ')[0]
            elif overflown_widget == self.trv_objects_tree:
                # the objects tree is overflown by the mouse cursor
                index = self.trv_objects_tree.identify_row(event.y)
                overflown_object_uid = self.trv_objects_tree.item(index, option='text').split(' ')[0]
            elif overflown_widget == self.lst_links_list:
                # the linked objects list is overflown by the mouse cursor, no drag&drop possible with this list
                pass
        else:
            # event = None : the Control key has been pressed or released
            if self.object_dragging_in_progress:
                # keep active the current overflown object
                overflown_object_uid = self.drag_overflown_object_uid
            else:
                return

        if not self.object_dragging_in_progress:
            # start of a dragging operation
            self.selected_widget_b4 = overflown_widget
            if overflown_object_uid != self.edited_object_uid:
                # first time the mouse cursor is dragged outside the edited object UID
                self.object_dragging_in_progress = True
                # the dragged object UID is the edited object UID
                self.dragged_object_uid = self.edited_object_uid
                self.dragged_object_type = self.odf_data.object_type_get(self.dragged_object_uid)
                self.dragged_object_parents_list = self.odf_data.object_kinship_list_get(self.dragged_object_uid, TO_PARENT)
                # erase selections in objects lists/tree
                self.selected_object_uid_b4 = self.selected_object_uid
                self.selected_linked_uid_b4 = self.selected_linked_uid
                self.selected_object_uid = None
                self.selected_linked_uid = None

        if self.object_dragging_in_progress and (self.drag_overflown_object_uid != overflown_object_uid or event == None) and self.selected_widget_b4 != self.lst_links_list:
            # a new object UID is overflown by the dragged mouse cursor or no event if provided (Control key has been pressed/released)
            self.drag_overflown_object_uid = overflown_object_uid

            # highlight in the lists/tree the object UID which is overflown if the dragged object can be dropped on it
            # and update the mouse cursor aspect
            self.selected_object_uid = None
            self.dragged_object_drop_action = None
            self.wnd_main['cursor'] = 'X_cursor'
            if overflown_object_uid != None and self.dragged_object_uid != None and overflown_object_uid != self.dragged_object_uid:
                # overflown and dragged objects have not the same UID
                if overflown_object_uid[:-3] == self.dragged_object_uid[:-3]:
                    # dragged and overflown objects UID have the same type
                    if not self.is_key_control_pressed:
                        # the Control key is not pressed
                        self.selected_object_uid = overflown_object_uid
                        self.dragged_object_drop_action = 'reorder'
                        if overflown_object_uid < self.dragged_object_uid:
                            self.wnd_main['cursor'] = 'based_arrow_up'
                        else:
                            self.wnd_main['cursor'] = 'based_arrow_down'
                elif (self.dragged_object_type in self.odf_data.object_poss_children_type_list_get(overflown_object_uid) and
                      overflown_object_uid not in ('Organ', 'Header')):
                    # the dragged object can be child of the overflown object which is not Organ or Header
                    if not self.is_key_control_pressed:
                        # the Control key is not pressed
                        if overflown_object_uid not in self.dragged_object_parents_list:
                            # the overflown object is not parent of the dragged object
                            self.selected_object_uid = overflown_object_uid
                            self.dragged_object_drop_action = 'move'
                            self.wnd_main['cursor'] = 'sb_up_arrow'
                    else:
                        # the Control key is pressed
                        self.selected_object_uid = overflown_object_uid
                        self.dragged_object_drop_action = 'copy'
                        self.wnd_main['cursor'] = 'plus'

            # update the selections in the objects lists/tree
            self.gui_status_update_lists()

    #-------------------------------------------------------------------------------------------------
    def object_b1_release(self, event):
        # (GUI event callback) the user has released the mouse button 1 inside the GO objects lists or tree

        if self.ignore_b1_release:
            # the mouse button 1 release event has to be ignored (asked by a list/tree select function if the edited object has been changed)
            self.wnd_main['cursor'] = ''
            return

        if self.object_dragging_in_progress:
            # UID of the object on which the dragged object UID has been dropped

            self.wnd_main['cursor'] = 'watch'
            self.wnd_main.update()

            target_object_uid = self.drag_overflown_object_uid

            if self.dragged_object_drop_action == 'reorder':
                # the position of the dragged object has to be reordered
                # in this case the dragged object is moved before or after the overflown object, weither it has respectively a higher or lower UID than the target

                # rename temporarily the dragged object with the ID 999 to let its UID free in the ODF to shift objects before or after its new position
                temp_uid = self.dragged_object_uid[:-3] + '999'
                self.odf_data.object_rename(self.dragged_object_uid, temp_uid, True, 0)

                if target_object_uid < self.dragged_object_uid:
                    # object dragged over an object with lower UID, +1 shift is applied on target and next objects UID
                    self.odf_data.object_rename(temp_uid, target_object_uid, True, +1)
                else:
                    # object dragged over an object with higher UID, -1 shift is applied on target and previous objects UID
                    self.odf_data.object_rename(temp_uid, target_object_uid, True, -1)

                self.odf_data_changed = True

               # update the selection variables
                if self.focused_objects_widget == self.trv_objects_tree:
                    self.focused_sel_item_id = self.focused_sel_item_id.replace(self.dragged_object_uid, target_object_uid)
                else:
                    self.focused_sel_item_id = target_object_uid
                self.selected_object_uid = target_object_uid
                self.selected_linked_uid = None
                self.edited_object_uid = target_object_uid

            elif self.dragged_object_drop_action == 'move':
                # the dragged object has to be moved under a new parent : the target object UID
                target_object_type = self.odf_data.object_type_get(target_object_uid)
                parents_uid_list = self.odf_data.object_kinship_list_get(self.dragged_object_uid, TO_PARENT)

                # if the dragged object has already a parent of the type of the target object, remove it in its parents list
                for parent_uid in list(parents_uid_list):
                    if self.odf_data.object_type_get(parent_uid) == target_object_type:
                        parents_uid_list.remove(parent_uid)
                # add the target object UID in the parents list of the dragged object
                parents_uid_list.append(target_object_uid)
                # update the parent links of the dragged object
                object_uid = self.odf_data.object_link(self.dragged_object_uid, parents_uid_list, TO_PARENT)
                if object_uid != None:
                    # the links update has been done properly
                    self.odf_data_changed = True
                    self.selected_object_uid = object_uid
                    self.selected_linked_uid = None
                    self.edited_object_uid = object_uid
                    self.focused_objects_widget = self.trv_objects_tree
                    self.focused_sel_item_id = target_object_uid + object_uid

            elif self.dragged_object_drop_action == 'copy':
                # the dragged object UID as to be copied as a new child of the target object UID
                object_uid = self.odf_data.object_copy(self.dragged_object_uid, target_object_uid)
                if object_uid != None:
                    # the object copy has been done properly
                    self.odf_data_changed = True
                    self.selected_object_uid = object_uid
                    self.selected_linked_uid = None
                    self.edited_object_uid = object_uid
                    self.focused_objects_widget = self.trv_objects_tree
                    self.focused_sel_item_id = target_object_uid + object_uid

            else:
                # retore the lists selections
                self.selected_object_uid = self.selected_object_uid_b4
                self.selected_linked_uid = self.selected_linked_uid_b4

        # update the content of GUI widgets if a drag&drop action has been done
        if self.dragged_object_drop_action != None:
            self.objects_list_update()
            self.objects_tree_update()
            # update the events log text
            self.logs_update()

        if self.focused_objects_widget != self.lst_links_list:
            # update the content of the linked objects list only if it has not the focus
            self.object_links_list_update()

        # update the status of GUI widgets
        self.object_text_update()
        self.gui_status_update_lists(self.dragged_object_drop_action != None, self.dragged_object_drop_action != None)
        self.gui_status_update_buttons()

        # reset the drag&drop variables
        self.object_dragging_in_progress = False
        self.dragged_object_drop_action = None
        self.drag_overflown_object_uid = None
        self.dragged_object_uid = None

        # restore the default mouse cursor
        self.wnd_main['cursor'] = ''

    #-------------------------------------------------------------------------------------------------
    def object_unselect(self):
        # (GUI event callback) the user has selected 'Clear all' in the context menu of the object text box

        if self.can_i_make_change():
            # clear the current object selection
            self.selected_object_app = 'GO'
            self.selected_object_uid = None
            self.selected_linked_uid = None
            self.edited_object_uid = None

            self.focused_objects_widget = None
            self.focused_sel_item_id = None

            # update the object text and links list
            self.object_text_update()
            self.object_links_list_update()

            # reset the edit modified flag
            self.txt_object_text.edit_modified(False)
            self.edited_object_changed = False

            self.gui_status_update_lists()
            self.gui_status_update_buttons()

    #-------------------------------------------------------------------------------------------------
    def object_text_update(self):
        # update the content of the object editor text box widget

        # to block the GUI events triggered by the text box content change
        self.gui_events_block()

        # erase the content of the text box and the viewer
        self.txt_object_text.delete(1.0, "end")
        self.viewer_file_show()

        # choose which object lines to display in the text box
        object_lines_list = []
        if self.selected_object_app == 'GO':
            # get the data lines list of the selected GrandOrgue object UID
            object_lines_list = self.odf_data.object_lines_read(self.edited_object_uid)
        elif self.selected_object_app == 'HW':
            # get the data lines list of the selected Hauptwerk object UID
            object_lines_list = self.odf_hw2go.HW_ODF_get_object_attr_list(self.edited_object_uid)

        # write the object data lines in the object text box
        if len(object_lines_list) > 0:
            self.txt_object_text.insert(1.0, '\n'.join(object_lines_list))

        # place the insertion cursor at the beginning of the text
        self.txt_object_text.mark_set("insert", "1.0")

        # apply the syntax highlighting
        self.odf_syntax_highlight(self.txt_object_text)

        # reset the text modified flag
        self.txt_object_text.edit_modified(False)
        self.edited_object_changed = False

        self.txt_object_text['cursor'] = 'xterm'

    #-------------------------------------------------------------------------------------------------
    def object_text_changed(self, event):
        # (GUI event callback) the user has made a change in the object text box

        if self.gui_events_blocked:
            return

        if self.txt_object_text.edit_modified() and self.edited_object_changed == False and self.selected_object_app == 'GO':
            # update the status of GUI widgets
            self.edited_object_changed = True
            self.gui_status_update_buttons()

    #-------------------------------------------------------------------------------------------------
    def object_text_click(self, event):
        # (GUI event callback) the user has clicked with left button in the object text box

        cursor_pos = self.txt_object_text.index('insert')

        line = self.txt_object_text.get(cursor_pos + ' linestart', cursor_pos + ' lineend')
        self.viewer_file_show(line, self.selected_object_app, self.edited_object_uid)

    #-------------------------------------------------------------------------------------------------
    def object_text_click_dbl(self, event):
        # (GUI event callback) the user has double clicked with left button in the object text box

        # get the position of the cursor in the text box
        cursor_pos = self.txt_object_text.index('insert')
        line_nb = int(cursor_pos.split('.')[0])
        char_nb = int(cursor_pos.split('.')[1])
        # get the entire line in which is the cursor
        line = self.txt_object_text.get(cursor_pos + ' linestart', cursor_pos + ' lineend')

        pos = line.find('=')
        if pos >= 0:
            # there is an equal character in the line
            if char_nb <= pos:
                # the cursor is at the left of the cursor : select the text from the start of the line until the equal character
                self.txt_object_text.tag_add('sel', f'{line_nb}.0', f'{line_nb}.{pos}')
            else:
                # the cursor is at the right of the cursor
                posc = line.find(';', pos + 1)
                if posc >= 0:
                    # there is a comment at the end of the line : select the text from the equal char until the semi-colon
                    self.txt_object_text.tag_add('sel', f'{line_nb}.{pos+1}', f'{line_nb}.{posc}')
                else:
                    # there is no comment at the end of the line : select the text from the equal char until the end of the line
                    self.txt_object_text.tag_add('sel', f'{line_nb}.{pos+1}', f'{line_nb}.{pos+1} lineend')
        elif line[0] == '[':
            # the line starts with an opening bracket
            pos = line.find(']')
            if pos > 0:
                # there is a closing bracket : select the text between the bracket
                self.txt_object_text.tag_add('sel', f'{line_nb}.1', f'{line_nb}.{pos}')
            else:
                # there is no closing bracket : select the entire line
                self.txt_object_text.tag_add('sel', f'{line_nb}.0', f'{line_nb}.0 lineend')
        else:
            # in all other cases, select the entire line
            self.txt_object_text.tag_add('sel', f'{line_nb}.0', f'{line_nb}.0 lineend')

        return 'break'  # don't let tkinter manage the event

    #-------------------------------------------------------------------------------------------------
    def object_text_key_pressed(self, event):
        # (GUI event callback) the user has pressed a keyboard key in the object text box

        # update the syntax highlighting
        self.odf_syntax_highlight(self.txt_object_text)

    #-------------------------------------------------------------------------------------------------
    def object_text_select_all(self, event):
        # (GUI event callback) the user has pressed the Ctrl+a or Ctrl+A keys combinaison in the object text box to select all the text
        # in Windows it is managed natively by the text box widget, but not in Linux

        self.txt_object_text.tag_add('sel', '1.0', 'end')
        return 'break'  # do not process further the event in the widget

    #-------------------------------------------------------------------------------------------------
    def object_text_paste(self, event):
        # (GUI event callback) the user has requested a past in the object text box
        # in Windows it is managed correctly to past instead of a selected text of the text box widget, but not in Linux

        # delete the current selected if any
        try:
            event.widget.delete('sel.first', 'sel.last')
        except:
            pass

        # paste the text of the clipboard if any
        try:
            event.widget.insert('insert', event.widget.clipboard_get())
        except:
            pass

        return 'break'  # do not process further the event in the widget

    #-------------------------------------------------------------------------------------------------
    def object_text_changes_apply(self):
        # (GUI event callback) the user has clicked on the button "Apply" to apply the changes done in the object text box
        # return False if there is an error in the text to apply, else True

        if not self.edited_object_changed:
            # there is no change to apply
            return True

        # convert the object text lines in a list
        object_lines_list = self.txt_object_text.get(1.0, 'end').splitlines()

        if len(object_lines_list) <= 1 and object_lines_list[0] == '':
            # the text box is empty
            return True

        # apply the object data in the ODF data
        object_uid = self.odf_data.object_lines_write(object_lines_list, self.edited_object_uid)
        if object_uid != None:
            # the modification has been applied with success
            self.odf_data_changed = True
            self.edited_object_changed = False
            # reset the edit modified flag
            self.txt_object_text.edit_modified(False)

            if self.edited_object_uid not in (None, object_uid):
                # the object UID has been renamed inside the edited lines list
                if self.selected_linked_uid != None:
                    self.selected_linked_uid = object_uid
                else:
                    self.selected_object_uid = object_uid
                self.focused_sel_item_id = self.focused_sel_item_id.replace(self.edited_object_uid, object_uid)
                self.edited_object_uid = object_uid

            # update the various GUI widgets
            self.objects_list_update()
            self.objects_tree_update()
            self.object_links_list_update()
            self.gui_status_update_lists(True)
            self.gui_status_update_buttons()
            changes_applied = True
        else:
            changes_applied = False
            # select the Logs tab of the notebook to show error logs
            self.notebook.select(self.frm_logs)

        # update the events log text
        self.logs_update()

        return changes_applied

    #-------------------------------------------------------------------------------------------------
    def object_add(self):
        # (GUI event callback) the user has clicked on the button "Add" to create a new object


        if not self.can_i_make_change():
            # the user has answered Cancel if the edited object has been modified
            return

        # recover the list of the object types which can be child of the edited object
        object_types_list = self.odf_data.object_poss_children_type_list_get(self.edited_object_uid)

        # add to the objects types list Organ and Header if they are not already present in the ODF
        if 'Organ' not in self.odf_data.objects_list_get():
            object_types_list.insert(0, 'Organ')
        if 'Header' not in self.odf_data.objects_list_get():
            object_types_list.insert(0, 'Header')

        # let the user choose the object type to add with the proper invitation message
        if 'Header' in object_types_list or 'Organ' in object_types_list:
            if self.edited_object_uid in (None, 'Header'):
                msg = 'Choose a type of section to add at the root'
            else:
                msg = f'Choose a type of section to add as child of\n{self.edited_object_uid} or at the root'
        else:
            if len(object_types_list) == 0:
                return
            msg = f'Choose a type of section to add as child of\n{self.odf_data.object_names_get(self.edited_object_uid)}'
        chosen_type_list = AskUserChooseListItems(self.wnd_main, 'Section add', msg, object_types_list, multiselect_bool=False)

        if chosen_type_list == None or len(chosen_type_list) == 0:
            # no chosen object type
            return

        # recover the chosen object type and its parent UID
        chosen_object_type = chosen_type_list[0]
        if self.edited_object_uid in ('Header', 'Organ'):
            parent_object_uid = None
        else:
            parent_object_uid = self.edited_object_uid

        # create the object in the ODF
        new_object_uid = self.odf_data.object_add(chosen_object_type, parent_object_uid)
        if new_object_uid != None:
            # the object has been created successfully
            self.odf_data_changed = True
            # set the new object as the current selected and focused object
            self.selected_object_uid = new_object_uid
            self.selected_linked_uid = None
            self.edited_object_uid = new_object_uid

            # update the content of GUI widgets
            self.objects_list_update()
            self.objects_tree_update()
            self.object_links_list_update()
            self.object_text_update()

            if self.focused_objects_widget == self.trv_objects_tree:
                if new_object_uid != 'Header':
                    # in the objects tree the new object is child of the curent selected node
                    self.focused_sel_item_id += new_object_uid
                else:
                    self.focused_sel_item_id = 'Header'
                self.objects_tree_node_and_parents_open(self.focused_sel_item_id)
            else:
                self.focused_sel_item_id = new_object_uid

            self.gui_status_update_lists(True)
            self.gui_status_update_buttons()

        # update the events log text
        self.logs_update()

    #-------------------------------------------------------------------------------------------------
    def object_link(self, relationship):
        # (GUI event callback ) the user has clicked on the button "Parents" or "Children" to link the current selected object UID to another object
        # relationship must be TO_PARENT or TO_CHILD

        if not self.can_i_make_change():
            # the user has answered Cancel if the edited object has been modified
            return

        # recover the list of the parents/children objects of the edited object
        current_kinship_list = list(self.odf_data.object_kinship_list_get(self.edited_object_uid, relationship))

        # add in the current kinship list the name of each object to have same object names as in the possible kinship list
        for i, object_uid in enumerate(current_kinship_list):
            current_kinship_list[i] = self.odf_data.object_names_get(object_uid)

        # recover the list of the objects UID which can be possibly parents or children of the edited object
        (possible_parents_list, possible_children_list)  = self.odf_data.object_poss_kinship_list_get(self.edited_object_uid)

        # take the list corresponding to the kinship type, this will be the possible kinship objects list
        if relationship == TO_PARENT:
            if len(possible_parents_list) == 0:
                # no possible parent object
                return
            possible_kinship_list = possible_parents_list

        elif relationship == TO_CHILD:
            if len(possible_children_list) == 0:
                # no possible child object
                return

            possible_kinship_list = possible_children_list

        else:
            # wrong relationship value given
            return

        # add in the possible kinship objects list the name of each object to facilitate the objects selection
        for i, object_uid in enumerate(possible_kinship_list):
            possible_kinship_list[i] = self.odf_data.object_names_get(object_uid)

        # let the user choose the parent/children object(s) to which link the selected object
        if relationship == TO_PARENT:
            selected_kinship_list = AskUserChooseListItems(self.wnd_main, 'Link to parents',
                                                          f'Select/unselect parent(s) of\n{self.odf_data.object_names_get(self.edited_object_uid)}',
                                                          sorted(possible_kinship_list), current_kinship_list, True)
        else:
            selected_kinship_list = AskUserChooseListItems(self.wnd_main, 'Link to children',
                                                          f'Select/unselect child(ren) of\n{self.odf_data.object_names_get(self.edited_object_uid)}',
                                                          sorted(possible_kinship_list), current_kinship_list, True)

        if selected_kinship_list == None:
            # operation cancelled
            return

        # remove the object name in the selected kinship objects list before to give it to the function odf_data.object_link
        for i, kinship in enumerate(selected_kinship_list):
            selected_kinship_list[i] = kinship.split(' ')[0]

        # link the selected object to the one of the selected kinship objects list
        object_uid = self.odf_data.object_link(self.edited_object_uid, selected_kinship_list, relationship)
        if object_uid != None:
            # the link change has been done successfully
            self.odf_data_changed = True

            if self.focused_objects_widget != self.trv_objects_tree:
                self.focused_sel_item_id = object_uid
            # else if the focus is in the objects tree let unchanged the selected node iid
            # in case the object has not been renamed or has still the current parent node

            # update the object selection
            self.selected_object_uid = object_uid
            self.selected_linked_uid = None
            self.edited_object_uid = object_uid

            # update the content of GUI widgets
            self.objects_list_update()
            self.objects_tree_update()
            self.object_links_list_update()
            self.object_text_update()
            self.gui_status_update_lists(True)
            self.gui_status_update_buttons()

        # update the events log text
        self.logs_update()

    #-------------------------------------------------------------------------------------------------
    def object_rename(self):
        # (GUI event callback) the user has clicked on the button "Rename" to rename the current selected object UID

        if (not self.can_i_make_change() and not self.edited_object_uid[-3:].isdigit()):
            # the user has answered Cancel if the edited object has been modified
            # of the edited object UID has not 3 digits at the end
            return

        # ask to the user the new object ID to set
        answer = AskUserEnterString(self.wnd_main, 'OdfEdit', f"Enter a new value for\nthe 3 ending digits of {self.edited_object_uid}", int(self.edited_object_uid[-3:]))
        if answer == None:
            return

        # check the given number
        if not answer.isdigit():
            AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', 'Digits only are expected', ['Close'], 'Close')
        elif int(answer) not in range(0,1000):
            AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', 'A value between 0 and 999 is expected', ['Close'], 'Close')
        else:
            # the given number is correct
            # define the new UID of the object
            new_uid = self.edited_object_uid[:-3] + str(int(answer)).zfill(3)
            # check if the new UID can be applied
            object_type = self.odf_data.object_type_get(self.edited_object_uid)
            if new_uid == self.edited_object_uid:
                AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f'{new_uid} is not a new name.', ['Close'], 'Close')
            elif new_uid in self.odf_data.objects_list_get():
                AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f'A section named {new_uid} already exists', ['Close'], 'Close')
            elif int(answer) == 0 and object_type not in ('Panel', 'Manual'):
                AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', 'Only Manual and Panel can end by 000', ['Close'], 'Close')
            elif self.odf_data.object_rename(self.edited_object_uid, new_uid) != None:
                # the object has been renamed successfully
                self.odf_data_changed = True
                # update the UID of the selected object
                if self.selected_linked_uid != None:
                    self.selected_linked_uid = new_uid
                else:
                    self.selected_object_uid = new_uid
                self.focused_sel_item_id = self.focused_sel_item_id.replace(self.edited_object_uid, new_uid)
                self.edited_object_uid = new_uid

                # update the content of GUI widgets
                self.objects_list_update()
                self.objects_tree_update()
                self.object_links_list_update()
                self.object_text_update()
                self.gui_status_update_lists(True)
                self.gui_status_update_buttons()

            # update the events log text
            self.logs_update()

    #-------------------------------------------------------------------------------------------------
    def object_delete(self):
        # (GUI event callback) the user has clicked on the button "Delete" to delete the current selected object UID

        if self.edited_object_uid == None or self.selected_object_app != 'GO':
            return

        edited_object_type = self.odf_data.object_type_get(self.edited_object_uid)

        if len(self.odf_data.object_kinship_list_get(self.edited_object_uid, TO_CHILD)) > 0:
            # the edited object has children
            if edited_object_type == 'Panel':
                answer = AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f"The section {self.edited_object_uid} and all his children sections Panel999xxx will be deleted.\nIs it ok to proceed ?", ['OK', 'Cancel'], 'OK')
            else:
                answer = AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f"The section {self.edited_object_uid} has children which some may become orphan once this section is deleted.\nIs it ok to proceed ?", ['OK', 'Cancel'], 'OK')
        else:
            answer = AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f"The section {self.edited_object_uid} will be deleted.\nIs it ok to proceed ?", ['OK', 'Cancel'], 'OK')

        if answer != 'OK':
            return

        # the user is ok for the deletion

        # choose the object UID which will be selected after the removal of the current selected object
        if self.focused_objects_widget == self.lst_links_list and self.selected_linked_uid != None:
            # a parent/child object is selected in the linked objects list : the parent will be the selected object
            next_selected_object_uid = self.selected_object_uid
        else:
            next_selected_object_uid = None

        # identify the line in the objects list widget where is present the object to delete (to make visible this line after the deletion)
        for line_nb in range(self.lst_objects_list.size()):
            if self.lst_objects_list.get(line_nb).split(' ')[0] == self.edited_object_uid:
                break

        # remove the object in the ODF data
        if self.odf_data.object_delete(self.edited_object_uid):
            # the object has been removed without issue
            self.odf_data_changed = True
            self.edited_object_changed = False
            # update the current object UID
            self.selected_object_uid = next_selected_object_uid
            self.selected_linked_uid = None
            self.edited_object_uid = None
            self.focused_objects_widget = None

            # update the content of GUI widgets
            self.objects_list_update()
            self.lst_objects_list.see(line_nb)
            self.objects_tree_update()
            self.object_links_list_update()
            self.object_text_update()
            self.gui_status_update_lists()
            self.gui_status_update_buttons()

        # update the events log text
        self.logs_update()

    #-------------------------------------------------------------------------------------------------
    def compass_extend(self):
        # (GUI event callback) the user has clicked on the Menu item to extend the compass of the selected manual or stop or rank

        object_uid = self.edited_object_uid
        object_type = self.odf_data.object_type_get(object_uid)

        if object_uid == None or object_type not in ('Manual', 'Stop', 'Rank'):
            AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', 'Select a Manual or Stop or Rank section first.', ['Close'], 'Close')
            return

        # check if the selected object has a MIDI notes compass and if yes gets its first and last MIDI notes
        compass = self.odf_data.compass_get(object_uid)

        if compass != None:
            # the selected object has a compass, recover it
            midi_note_first, midi_note_last = compass
            midi_note_last_max = COMPASS_EXTEND_MAX
            prompt =  f"{object_uid} has a MIDI notes compass from {midi_note_first} to {midi_note_last} ({midi_nb_to_note2(midi_note_first)} to {midi_nb_to_note2(midi_note_last)})."

            if midi_note_last >= COMPASS_EXTEND_MAX:
                # the selected object has already the maximum extendable MIDI note
                prompt += '\nIt has already the highest possible MIDI note.'
                AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', prompt, ['Close'], 'Close')
            else:
                # ask the user to give the MIDI note to which to extend the compass of the selected object (maximum 12 notes of extension)
                prompt += f"\nEnter the MIDI note number between {midi_note_last + 1} and {midi_note_last_max} ({midi_nb_to_note2(midi_note_last + 1)} to {midi_nb_to_note2(midi_note_last_max)}) up to which to extend this {object_type}."
                if object_type == 'Manual':
                    prompt += '\nThe manual displayed keys will not be extended.'
                keyboard_image = os.path.dirname(__file__) + os.path.sep + 'resources' + os.path.sep + 'NotesMidiKeys.png'
                answer = AskUserEnterString(self.wnd_main, 'OdfEdit', prompt, '', keyboard_image)
                if answer != None:
                    # the user has given his choice
                    if answer.isdigit() and int(answer) in range(midi_note_last + 1, midi_note_last_max + 1):
                        # the user has given a proper MIDI note number
                        # do the compass extension
                        self.wnd_main['cursor'] = 'watch'
                        compass = self.odf_data.compass_extend(object_uid, int(answer))
                        self.wnd_main['cursor'] = ''
                        if compass != None:
                            # the extension has been done successfully
                            self.odf_data_changed = True
                            self.object_text_update()
                            self.gui_status_update_buttons()

                            new_midi_note_first, new_midi_note_last = compass
                            if new_midi_note_last > midi_note_last:
                                AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f'{object_uid} compass extended\nfrom MIDI {midi_note_first}-{midi_note_last} ({midi_nb_to_note2(midi_note_first)} to {midi_nb_to_note2(midi_note_last)}) to {new_midi_note_first}-{new_midi_note_last} ({midi_nb_to_note2(new_midi_note_first)} to {midi_nb_to_note2(new_midi_note_last)}).\nSee details in the logs tab.', ['Close'], 'Close')
                            else:
                                AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f'{object_uid} compass not extended.\nSee details in the logs tab.', ['Close'], 'Close')
                    else:
                        AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', f'{answer} is not in the proposed MIDI notes range {midi_note_last + 1}-{midi_note_last_max}.', ['Close'], 'Close')

        if compass == None:
            # an error has occurred
            AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', 'The extension cannot be done.\nSee details in the logs tab.', ['Close'], 'Close')

        # update the events log text
        self.logs_update()
        # select the logs tab of the notebook to show the logs generated by the operation
        self.notebook.select(self.frm_logs)

    #-------------------------------------------------------------------------------------------------
    images_ref=[]  # list needed to keep in memory the reference to the images opened by tk.PhotoImage and added in the text box (to prevent them being garbage collected)

    def odf_syntax_highlight(self, txt_widget):
        # apply syntax highlighting to the content of the given object text box widget

        # remove the tags previously set in the text box
        txt_widget.tag_remove(TAG_FIELD, '1.0', tk.END)
        txt_widget.tag_remove(TAG_COMMENT, '1.0', tk.END)
        txt_widget.tag_remove(TAG_OBJ_UID, '1.0', tk.END)
        txt_widget.tag_remove(TAG_TITLE, '1.0', tk.END)

        # put in a list the lines of the text box
        lines = txt_widget.get('1.0', tk.END).splitlines()

        # scan all the characters of the text box lines
        for l, line in enumerate(lines):
            c0 = None
            equal_seen = False
            for c, char in enumerate(line):
                if char == ';':
                    # comment : apply the comment color until the end of the line
                    txt_widget.tag_add(TAG_COMMENT, f'{l+1}.{c}', f'{l+1}.0 lineend')
                    break  # skip the rest of the line
                if char == '[' and c == 0 and line[c+1] != ' ':
                    # start of an object UID
                    c0 = c
                elif char == ']':
                    # end of an object UID : apply the UID color between the opening and closing brackets
                    if c0 != None:
                        txt_widget.tag_add(TAG_OBJ_UID, f'{l+1}.{c0}', f'{l+1}.{c+1}')
                        c0 = None
                elif txt_widget.get(f'{l + 1}.{c}') == '=' and not equal_seen:
                    # equal character (first seen in the line) : apply the field color before the '='
                    txt_widget.tag_add(TAG_FIELD, f'{l+1}.0', f'{l + 1}.{c}')
                    equal_seen = True
                elif line[:2] == '__' :
                    # start of line tag identifying an image file to insert (in the help)
                    # recover the file name after the '__' tag
                    file_name = line[2:]
                    # remove the file name in the widget
                    txt_widget.delete(f'{l+1}.0', f'{l+1}.0 lineend')
                    try:
                        # open the image file
                        photo = tk.PhotoImage(file = os.path.dirname(__file__) + os.path.sep + 'resources' + os.path.sep + file_name)
                        # add the reference of the image in the list to store these references
                        self.images_ref.append(photo)
                        # insert the image in the text box
                        txt_widget.image_create(f'{l+1}.0', image=photo, padx=10, pady=10)
                    except:
                        # insert a message indicating that the image has not been opened
                        txt_widget.insert(f'{l+1}.0', f'!!! cannot open the image resources{os.path.sep + file_name}')
                    break  # skip the rest of the line
                if line[:2] == '>>' :
                    # title line (in the help) : apply the title color to the whole line
                    txt_widget.delete(f'{l+1}.0', f'{l+1}.3')
                    txt_widget.tag_add(TAG_TITLE, f'{l+1}.0', f'{l+1}.0 lineend')
                    break  # skip the rest of the line

    #-------------------------------------------------------------------------------------------------
    def odf_data_check(self):
        # check the consistency of the ODF data

        # ask to the user to apply his object data change before to launch the check
        if self.can_i_make_change():
            # the user has not cancelled the operation

            if self.odf_check_files_names == None:
                # ask the user if he wants to check the files names (to make a faster check)
                self.odf_check_files_names = 'Yes' == AskUserAnswerQuestion(self.wnd_main, 'OdfEdit', "Do you want to check files names ?\nThe check will be faster if files names are not checked.\nThis choice will be memorized until the next ODF opening", ['Yes', 'No'], 'No')

            # do the check
            self.odf_data.check_odf_data(self.progress_status_update, self.odf_check_files_names)

            # update the events log text
            self.logs_update()

            # select the Logs tab of the notebook to show the check result
            self.notebook.select(self.frm_logs)

            self.gui_status_update_buttons()

    #-------------------------------------------------------------------------------------------------
    def progress_status_update(self, message):
        # callback function called by the C_ODF_DATA or C_ODF_HW2GO classes
        # to display in the ODF file name label a progress status message
        # if the given message contains a + in first character, the message is added to the current displayed text

        if len(message) > 0 and message[0] == '+':
            message = self.lab_odf_file_name.cget("text") + message[1:]

        self.lab_odf_file_name.config(text=message)
        self.lab_odf_file_name['foreground'] = 'red3'
        self.lab_odf_file_name.update()

#-------------------------------------------------------------------------------------------------
class ToolTip():
    # class to create a tooltip popup window for a given widget and with a given text to display

    # tk_ToolTip_class101.py
    # gives a Tkinter widget a tooltip as the mouse is above the widget
    # tested with Python27 and Python34  by  vegaseat  09sep2014
    # www.daniweb.com/programming/software-development/code/484591/a-tooltip-class-for-tkinter
    # Modified to include a delay time by Victor Zaccardo, 25mar16
    # example of usage : tooltip = ToolTip(parend_widget, "string to display")
    def __init__(self, widget, text='widget info'):
        self.waittime = 600     #miliseconds
        self.wraplength = 180   #pixels
        self.widget = widget
        self.text = text
        self.widget.bind("<Enter>", self.enter)
        self.widget.bind("<Leave>", self.leave)
        self.widget.bind("<ButtonPress>", self.leave)
        self.widget_id = None
        self.tw = None

    def enter(self, event=None):
        self.schedule()

    def leave(self, event=None):
        self.unschedule()
        self.hidetip()

    def schedule(self):
        self.unschedule()
        self.widget_id = self.widget.after(self.waittime, self.showtip)

    def unschedule(self):
        widget_id = self.widget_id
        self.widget_id = None
        if widget_id: self.widget.after_cancel(widget_id)

    def showtip(self, event=None):
        x = y = 0
        x, y, cx, cy = self.widget.bbox("insert")
        x += self.widget.winfo_rootx() + 20
        y += self.widget.winfo_rooty() + 35
        # creates a toplevel window
        self.tw = tk.Toplevel(self.widget)
        # Leaves only the label and removes the app window
        self.tw.wm_overrideredirect(True)
        self.tw.wm_geometry(f"+{x}+{y}")
        label = tk.Label(self.tw, text=self.text, justify=tk.LEFT,
                          background='#FFFFE5', relief=tk.SOLID, borderwidth=1,
                          wraplength = self.wraplength)
        label.pack(ipadx=1)

    def hidetip(self):
        tw = self.tw
        self.tw= None
        if tw: tw.destroy()

#-------------------------------------------------------------------------------------------------
class AskUserChooseListItems():
    # class to open and manage a dialog box to ask the user to choose items in the given list choice_items_list, with OK and Cancel buttons
    # items of preselect_items_list which are in choice_items_list are pre-selected in the list widget
    # Return key press is equivalent to clicking on OK button
    # returns a list containing the selected items in choice_items_list, or None if the user clicked on Cancel or close button or nothing has been selected

    # dialog box dimensions
    dialog_wnd_w = 300
    dialog_wnd_h = 450

    def __new__(cls, parent_wnd, title, message, choice_items_list, preselect_items_list = [], multiselect_bool=False, resizable_wnd_bool=True):

        # disable the parent window so that the dialog box is modal (only possible in Windows OS)
        if os.name == 'nt':
            parent_wnd.wm_attributes("-disabled", True)

        # create the toplevel modal dialog window
        dialog_wnd = tk.Toplevel(parent_wnd, takefocus=True)
        dialog_wnd.title(title)
        dialog_wnd.configure(background=COLOR_BACKGROUND0)
        if not resizable_wnd_bool:
            dialog_wnd.resizable(False, False)

        # place the dialog window at the center of the parent window with the defined dimensions
        dialog_wnd_x = max(0, parent_wnd.winfo_x() + int(parent_wnd.winfo_width()/2) - int(cls.dialog_wnd_w/2))
        dialog_wnd_y = max(0, parent_wnd.winfo_y() + int(parent_wnd.winfo_height()/2) - int(cls.dialog_wnd_h/2))
        dialog_wnd.geometry(f"{cls.dialog_wnd_w}x{cls.dialog_wnd_h}+{dialog_wnd_x}+{dialog_wnd_y}")

        # tell the window manager this is the child window of the parent window, permiting to have the child window flash if one clicks onto parent
        dialog_wnd.transient(parent_wnd)
        dialog_wnd.focus_force()

        # string var permitting to know the reason of the dialog box closure
        closure_reason = tk.StringVar()
        closure_reason.set(None)

        # link to the callback WM_DELETE_WINDOW or Escape key to manage the window closure
        dialog_wnd.protocol("WM_DELETE_WINDOW",  lambda reason=None: closure_reason.set(reason))
        dialog_wnd.bind('<Escape>', lambda event, reason=None: closure_reason.set(reason))
        dialog_wnd.bind('<KeyPress-Return>', lambda event, reason='ok': closure_reason.set(reason))
        dialog_wnd.bind('<KeyPress-KP_Enter>', lambda event, reason='ok': closure_reason.set(reason))

        # create the top label widget
        listdialog_label = ttk.Label(dialog_wnd, text=message, anchor='center', wraplength=cls.dialog_wnd_w - 10)
        listdialog_label.pack(side=tk.TOP, padx=5, pady=5, fill=tk.X)

        # frame to occupy the bottom area of the dialog window and to encapsulate OK and Cancel buttons
        listdialog_frm1 = ttk.Frame(dialog_wnd)
        listdialog_frm1.pack(side=tk.BOTTOM, fill=tk.X)

        # OK button
        toplevel_dialog_yes_button = ttk.Button(listdialog_frm1, text='OK', style='Return.TButton', command=lambda reason='ok': closure_reason.set(reason))
        toplevel_dialog_yes_button.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X, expand=True)

        # Cancel button
        toplevel_dialog_no_button = ttk.Button(listdialog_frm1, text='Cancel', command=lambda reason=None: closure_reason.set(reason))
        toplevel_dialog_no_button.pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X, expand=True)

        # frame to occupy the middle area of the window and to encapsulate the list and its vertical scroll bar
        listdialog_frm2 = ttk.Frame(dialog_wnd)
        listdialog_frm2.pack(side=tk.TOP, fill=tk.BOTH, padx=5, expand=True)

        # list box widget with its vertical scroll bar
        scrollbarv = ttk.Scrollbar(listdialog_frm2, orient=tk.VERTICAL)
        scrollbarv.pack(side=tk.RIGHT, fill=tk.Y)
        listdialog_list = tk.Listbox(listdialog_frm2, bg=COLOR_BG_LIST, font=TEXT_FONT, fg=TEXT_COLOR, selectbackground=COLOR_SELECTED_ITEM,
                                     exportselection=0, activestyle=tk.DOTBOX, selectmode=(tk.MULTIPLE if multiselect_bool else tk.SINGLE))
        listdialog_list.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        listdialog_list.config(yscrollcommand=scrollbarv.set)
        scrollbarv.config(command=listdialog_list.yview)

        # fill the list box
        see_used_bool = False
        for item in choice_items_list:
            # scan the items of the choice list
            # add the current item to the list widget
            listdialog_list.insert('end', item)
            if item in preselect_items_list:
                # the current has to be selected
                listdialog_list.selection_set('end')
                if not see_used_bool:
                    # make visible the first selected item
                    listdialog_list.see('end')
                    see_used_bool = True

        # wait for the change of the StringVar closure_reason before to continue in this function
        dialog_wnd.wait_variable(closure_reason)

        if closure_reason.get() == 'ok':
            # the user has clicked on the OK button, recover the list of the selected items
            selected_items_list = []
            for i in listdialog_list.curselection():
                selected_items_list.append(listdialog_list.get(i))
        else:
            selected_items_list = None

        # store the dialog window current dimensions
        cls.dialog_wnd_w = dialog_wnd.winfo_width()
        cls.dialog_wnd_h = dialog_wnd.winfo_height()

        # re-enable the parent window
        if os.name == 'nt':
            parent_wnd.wm_attributes("-disabled", False)
        # destroy this dialog box
        dialog_wnd.destroy()
        # restore the parent window if it was in icon state
        parent_wnd.deiconify()

        return selected_items_list

#-------------------------------------------------------------------------------------------------
class AskUserAnswerQuestion():
    # class to open and manage a dialog box to ask the user to answer a given question by one of the given possible answers (list of texts)
    # if on_return_answer is defined, this answer (a text) is considered as the answer if the Return key is pressed
    # returns the answer text selected by the user or None if the user has clicked windows close button or pressed Escape key

    def __new__(cls, parent_wnd, title, question, answers_list=[], on_return_answer=None):

        if len(answers_list) == 0:
            return None

        # disable the parent window so that the dialog box is modal (only possible in Windows OS)
        if os.name == 'nt':
            parent_wnd.wm_attributes("-disabled", True)

        # create the toplevel modal dialog window
        dialog_wnd = tk.Toplevel(parent_wnd, takefocus=True)
        dialog_wnd.title(title)
        dialog_wnd.configure(background=COLOR_BACKGROUND0)
        dialog_wnd.resizable(False, False)

        # tell the window manager this is the child window of the parent window, permiting to have the child window flash if one clicks onto parent
        dialog_wnd.transient(parent_wnd)
        dialog_wnd.focus_force()

        # string var permitting to know the reason of the dialog box closure
        closure_reason = tk.StringVar()
        closure_reason.set(None)

        # link to the callback WM_DELETE_WINDOW or Escape key to manage the window closure
        dialog_wnd.protocol("WM_DELETE_WINDOW",  lambda reason=None: closure_reason.set(reason))
        dialog_wnd.bind('<Escape>', lambda event, reason=None: closure_reason.set(reason))
        if on_return_answer != None:
            dialog_wnd.bind('<KeyPress-Return>', lambda event, reason=on_return_answer: closure_reason.set(reason))
            dialog_wnd.bind('<KeyPress-KP_Enter>', lambda event, reason=on_return_answer: closure_reason.set(reason))


        # create the top label widget containing the question asked to the user
        listdialog_label = ttk.Label(dialog_wnd, text=question, anchor='center', wraplength=int(parent_wnd.winfo_width()/3))
        listdialog_label.pack(side=tk.TOP, padx=20, pady=20, fill=tk.X)

        # frame to occupy the bottom area of the dialog window to encapsulate buttons
        listdialog_frm1 = ttk.Frame(dialog_wnd)
        listdialog_frm1.pack(side=tk.BOTTOM, fill=tk.X)

        # create buttons for the given possible answers
        for answer in answers_list:
            if answer == on_return_answer:
                ttk.Button(listdialog_frm1, text=answer, style='Return.TButton', command=lambda reason=answer: closure_reason.set(reason)).pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X, expand=True)
            else:
                ttk.Button(listdialog_frm1, text=answer, command=lambda reason=answer: closure_reason.set(reason)).pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X, expand=True)

        # place the dialog window at the center of the parent window
        dialog_wnd.update()
        dialog_wnd_x = max(0, parent_wnd.winfo_x() + int(parent_wnd.winfo_width()/2) - int(dialog_wnd.winfo_width()/2))
        dialog_wnd_y = max(0, parent_wnd.winfo_y() + int(parent_wnd.winfo_height()/2) - int(dialog_wnd.winfo_height()/2))
        dialog_wnd.geometry(f"+{dialog_wnd_x}+{dialog_wnd_y}")

        # wait for the change of the StringVar closure_reason before to continue in this function
        dialog_wnd.wait_variable(closure_reason)

        # re-enable the parent window
        if os.name == 'nt':
            parent_wnd.wm_attributes("-disabled", False)
        # destroy this dialog box
        dialog_wnd.destroy()
        # restore the parent window if it was in icon state
        parent_wnd.deiconify()

        return closure_reason.get()

#-------------------------------------------------------------------------------------------------
class AskUserEnterString():
    # class to open and manage a dialog box to ask the user to enter a string
    # returns the entered string or None if the user has clicked the windows close button or pressed Escape key or there is empty answer provided

    def __new__(cls, parent_wnd, title, message, initial_string='', image_path=None):

        # disable the parent window so that the dialog box is modal (only possible in Windows OS)
        if os.name == 'nt':
            parent_wnd.wm_attributes("-disabled", True)

        # create the toplevel modal dialog window
        dialog_wnd = tk.Toplevel(parent_wnd, takefocus=True)
        dialog_wnd.title(title)
        dialog_wnd.configure(background=COLOR_BACKGROUND0)
        dialog_wnd.resizable(False, False)

        # tell the window manager this is the child window of the parent window, permiting to have the child window flash if one clicks onto parent
        dialog_wnd.transient(parent_wnd)
        dialog_wnd.focus_force()

        # string var permitting to know the reason of the dialog box closure
        closure_reason = tk.StringVar()
        closure_reason.set(None)

        # link to the callback WM_DELETE_WINDOW or Escape key to manage the window closure
        dialog_wnd.protocol("WM_DELETE_WINDOW",  lambda reason=None: closure_reason.set(reason))
        dialog_wnd.bind('<Escape>', lambda event, reason=None: closure_reason.set(reason))
        dialog_wnd.bind('<KeyPress-Return>', lambda event, reason='OK': closure_reason.set(reason))
        dialog_wnd.bind('<KeyPress-KP_Enter>', lambda event, reason='OK': closure_reason.set(reason))

        # create the top label widget containing the message displayed to the user
        listdialog_label = ttk.Label(dialog_wnd, text=message, anchor='center')
        listdialog_label.pack(side=tk.TOP, padx=20, pady=10, fill=tk.X)

        # create the label widget to display the given image if any
        if image_path != None:
            try:
                image = tk.PhotoImage(file=image_path)
            except:
                image = None

            if image != None:
                label_image = tk.Label(dialog_wnd, image=image)
                label_image.pack(side=tk.TOP, padx=20, pady=0, fill=tk.X)

        # create the entry widget to let the user enter his string
        text_entry = tk.Entry(dialog_wnd)
        text_entry.pack(side=tk.TOP, padx=40, pady=10, fill=tk.X)
        text_entry.bind('<Return>', lambda event, reason='OK': closure_reason.set(reason))

        # frame to occupy the bottom area of the dialog window to encapsulate buttons
        listdialog_frm1 = ttk.Frame(dialog_wnd)
        listdialog_frm1.pack(side=tk.BOTTOM, fill=tk.X)

        # create OK and Cancel buttons
        ttk.Button(listdialog_frm1, text='OK', style='Return.TButton', command=lambda reason='OK': closure_reason.set(reason)).pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X, expand=True)
        ttk.Button(listdialog_frm1, text='Cancel', command=lambda reason=None: closure_reason.set(reason)).pack(side=tk.LEFT, padx=5, pady=5, fill=tk.X, expand=True)

        # place the dialog window at the center of the parent window
        dialog_wnd.update()
        dialog_wnd_x = max(0, parent_wnd.winfo_x() + int(parent_wnd.winfo_width()/2) - int(dialog_wnd.winfo_width()/2))
        dialog_wnd_y = max(0, parent_wnd.winfo_y() + int(parent_wnd.winfo_height()/2) - int(dialog_wnd.winfo_height()/2))
        dialog_wnd.geometry(f"+{dialog_wnd_x}+{dialog_wnd_y}")

        # place the focus on the entry widget and select its entire text coming from the given initial string
        text_entry.focus_set()
        text_entry.insert(0, initial_string)
        text_entry.select_range(0, tk.END)

        # wait for the change of the StringVar closure_reason before to continue in this function
        dialog_wnd.wait_variable(closure_reason)

        if closure_reason.get() == 'OK' and len(text_entry.get()) > 0:
            entered_text = text_entry.get()
        else:
            entered_text = None

        # re-enable the parent window
        if os.name == 'nt':
            parent_wnd.wm_attributes("-disabled", False)
        # destroy this dialog box
        dialog_wnd.destroy()
        # restore the parent window if it was in icon state
        parent_wnd.deiconify()

        return entered_text

#-------------------------------------------------------------------------------------------------

def midi_nb_to_note(midi_nb):
    # return in a tuple (note name (string), octave number (integer)) the note corresponding to the given MIDI note number
    assert 0 <= midi_nb <= 127, f'Out of range MIDI note number {midi_nb} given to midi_nb_to_note function'
    octave = int(midi_nb // NOTES_NB_IN_OCTAVE) - 1   # -1 to have MIDI note number 69 = note A4 and not A5
    note = NOTES_NAMES[midi_nb % NOTES_NB_IN_OCTAVE]
    return note, octave

def midi_nb_to_note2(midi_nb):
    # return in a string (note name + octave number concatenated, for example C#4) the note name corresponding to the given MIDI note number
    note, octave = midi_nb_to_note(midi_nb)
    return note + str(octave)

def note_to_midi_nb(note, octave):
    # return the MIDI note number corresponding to the given note (string in NOTES_NAMES list) and octave number (value in OCTAVES_RANGE list, -1 to 9)
    assert note in NOTES_NAMES, f'Wrong note name {note} given to note_to_midi_nb function'
    assert octave in OCTAVES_RANGE, f'Out of range octave number {octave} given to note_to_midi_nb function'
    midi_nb = NOTES_NAMES.index(note) + (NOTES_NB_IN_OCTAVE * (octave + 1))   # +1 to have note A4 = MIDI number 69 and not 57
    assert 0 <= midi_nb <= 127, f'Out of range MIDI number {midi_nb} calculated in note_to_midi_nb function'
    return midi_nb

def midi_nb_to_freq(midi_nb, a4_frequency=440.0):
    # return the frequency (float in Hz) corresponding to the given MIDI number
    # based on the given reference frequency of the A4 note (MIDI number 69), set at 440Hz by default if not provided
    return a4_frequency * math.pow(2, (midi_nb - 69) / 12)

def freq_to_midi_nb(frequency, a4_frequency=440.0):
    # return the MIDI number (integer) corresponding to the given frequency (in Hz)
    return round(12 * math.log2(frequency / a4_frequency) + 69)

def freq_diff_to_cents(ref_frequency, frequency):
    # return the number of cents (integer) from the given reference frequency (in Hz) to the given frequency (in Hz)
    return int(1200.0 * math.log2(frequency / ref_frequency))

def midi_nb_plus_cents_to_freq(midi_nb, cents, a4_frequency=440.0):
    # return the frequency (float in Hz) corresponding to the given MIDI number added by the given cents
    return midi_nb_to_freq(midi_nb, a4_frequency) * math.pow(2, cents / 1200)

#-------------------------------------------------------------------------------------------------
def myint(data, default_val=None):
    # return the given data in integer format, or the provided default value (or None if not defined) if it cannot be converted to integer or is not defined

    try:
        return int(data)
    except:
        return default_val

#-------------------------------------------------------------------------------------------------
def myfloat(data, default_val=None):
    # return the given data in float format, or the provided default value (or None if not defined) if it cannot be converted to float or is not defined

    try:
        return float(data)
    except:
        return default_val

#-------------------------------------------------------------------------------------------------
def mystr(data, default_val=''):
    # return the given data in string format, or the provided default value (or '' if not defined) if it cannot be converted to string or is not defined

    if data == None:
        return default_val

    try:
        return str(data)
    except:
        return default_val

#-------------------------------------------------------------------------------------------------
def mydickey(dic, key, default_val=None):
    # return the value corresponding to the given key in the given dic if it exists, else return the provided default value or None if not defined

    try:
        return dic[key]
    except:
        return default_val

#-------------------------------------------------------------------------------------------------
prev_file_name = '' # variable to store the previous file name processed by below function, to speed up the processing of the next one if there are common parts between them

def get_actual_file_name(file_name):
    # return the given file path/name with the actual characters case as they are defined in the files storage
    # return None if the given file name doesn't exist
    # the given file path/name must contain the path separator of the OS on which is running the script

    global prev_file_name

    # split in a list the elements of the given and previous file path/name separated by the OS path separator
    file_name_split_list = file_name.split(os.path.sep)
    prev_file_name_split_list = prev_file_name.split(os.path.sep)

    # find common path elements between the previous file name and the given one starting from the first element
    last_same_path_elem_id = None
    prev_file_name_elem_nb = len(prev_file_name_split_list)
    if prev_file_name_elem_nb > 0:
        for path_elem_id, path_elem_txt in enumerate(file_name_split_list):
            # scan the elements of the given file name
            if path_elem_id < prev_file_name_elem_nb:
                if path_elem_txt.lower() == prev_file_name_split_list[path_elem_id].lower():
                    # the current element is same as the one of the previous list with case unsensitive
                    last_same_path_elem_id = path_elem_id
                else:
                    # the current element is not the same between the given file name and the previous one, stop the check
                    break
            else:
                # the end of the previous list is reached, stop the check
                break

    if last_same_path_elem_id == None:
        # the previous file name is empty or has nothing similar to the given file name
        # take the first element of the given file name as starting point to search the actual name of elements
        actual_file_name =  file_name_split_list[0]
        # take the rest of the file name to check its elements
        file_name_split_list = file_name_split_list[1:]
    else:
        # recover from the last file name the elements which are identical to the given file name
        actual_file_name = os.path.sep.join(prev_file_name_split_list[:last_same_path_elem_id+1])
        # take the rest of the file name to check its elements
        file_name_split_list = file_name_split_list[last_same_path_elem_id+1:]

    # recover from the files storage the actual name of the elements of the file name not recovered from the previous file name
    for file_name_element in file_name_split_list:
        # scan the file name elements to recover their actual characters case in the files storage
        found_actual_element = None
        for actual_element in os.listdir(actual_file_name + os.path.sep):
            if actual_element.lower() == file_name_element.lower():
                # the current element is same as the expected file name element with case unsensitive
                found_actual_element = actual_element
                break

        if found_actual_element == None:
            # the current element of the given file name is not found in the actual_file_name directory, return None
            return None

        actual_file_name += os.path.sep + found_actual_element

    prev_file_name = actual_file_name

    return actual_file_name

#-------------------------------------------------------------------------------------------------
def path2ospath(file_name):
    # replace the / or \ in the given file name by the OS path separator

    # fix an issue observed in one sample set (Pipeloops Romantic Village Church) having // in the paths
    file_name = file_name.replace('//', '/')

    if os.path.sep == '/':  # Linux OS, replace \ by /
        return file_name.replace('\\', os.path.sep)

    # Windows OS, replace / by \
    return file_name.replace('/', os.path.sep)

#-------------------------------------------------------------------------------------------------
def main():
    # main function of the application

    # initiate a C_GUI class instance, display the main window based on this instance, start the main loop of this window
    C_GUI().wnd_main_build().mainloop()

#-------------------------------------------------------------------------------------------------
# first line of code executed at the launch of the script
# if we are in the main execution environment, call the main function of the application
if __name__ == '__main__': main()
