# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['thcrypto']

package_data = \
{'': ['*']}

install_requires = \
['bcrypt>=4.0.1,<5.0.0',
 'cryptography>=38.0.1,<39.0.0',
 'thresult>=0.9.14,<0.10.0']

setup_kwargs = {
    'name': 'thcrypto',
    'version': '0.9.20',
    'description': 'TangledHub Cryptography Library',
    'long_description': "[![Build][build-image]]()\n[![Status][status-image]][pypi-project-url]\n[![Stable Version][stable-ver-image]][pypi-project-url]\n[![Coverage][coverage-image]]()\n[![Python][python-ver-image]][pypi-project-url]\n[![License][bsd3-image]][bsd3-url]\n\n\n\n# thcrypto\n\n## Overview\n\nTangledHub library for cryptography.\n\n## Licencing\n\nthcrypto is licensed under the BSD license. Check the [LICENSE](https://opensource.org/licenses/BSD-3-Clause) for details.\n\n## Installation\n```bash\npip install thcrypto\n```\n\n---\n\n## Testing\n```bash\ndocker-compose build --no-cache thcrypto-test ; docker-compose run --rm thcrypto-test\n```\n\n## Building\n```bash\ndocker-compose build thcrypto-build ; docker-compose run --rm thcrypto-build\n```\n\n## Publish\n```bash\ndocker-compose build thcrypto-publish ; docker-compose run --rm -e PYPI_USERNAME=__token__ -e PYPI_PASSWORD=__SECRET__ thcrypto-publish\n```\n\n---\n\n## API\n\nthcrypto api is using the **thresult** library, so all api functions are returning result wrapped in **Ok** or **Err**\nobject.  \nTherefore, in order to reach into the result object, you should use **.unwrap()** as in examples.\n\n---\n\n## Bcrypt\n\n### cls.generate() -> Bcrypt\n\nGenerate bcrypt salt with custom parameters (4 < rounds: int > 31, prefix: bytes elem {b'2a', b'2b'}.  \nIf custom parameters are not passed, default values are used.\n\n#### Example:\n\n```python\nbcrypt_default: Bcrypt = Bcrypt.generate().unwrap()\n\nbcrypt_custom: Bcrypt = Bcrypt.generate(rounds=12, prefix=b'2b').unwrap()\n```\n\n### cls.load_or_generate() -> Bcrypt\n\nLoad bcrypt from file, if file does not exist, generate bcrypt key and writes it down to the file.  \nThis function can be called without parameters (in which case it is going to use default params  \n**(path: str='bcrypt.salt', rounds: int=12, prefix: bytes=b'2b'))**  \nor with custom params:  \n**(path: str='custom_path', 4 < rounds: int > 31, prefix: bytes elem {b'2a', b'2b'})**.  \nIn case that bcrypt is loaded from existing file, passed custom 'rounds' and 'prefix' params are ignored.\n\n#### Example:\n\n```python\nbcrypt_: Bcrypt = Bcrypt.load_or_generate_file().unwrap()\n\nbcrypt_: Bcrypt = Bcrypt.load_or_generate_file(path='bcrypt.salt', \n                                               rounds=12, \n                                               prefix=b'2b').unwrap()\n```\n\n### self.get_hashed_password() -> bytes\n\nHash password function accepts parameter(unhashed_password: bytes | str) which has to be max 72 bytes long.\n\n#### Example:\n\n```python\nhashed_password: bytes = bcrypt_.get_hashed_password(bcrypt_, 'test bcrypt').unwrap()\n```\n\n### cls.check_password() -> bool\n\nCheck password function accepts parameters(unhashed_password: bytes | str, hashed_password: bytes | str)\n\n#### Example:\n\n```python\nresult: bool = Bcrypt.check_password('test bcrypt', hashed_password).unwrap()\n```\n\n---\n\n## Edd25519\n\n### cls.generate() -> Ed25519\n\nGenerate Ed25519\n\n#### Example:\n\n```python\ngenerated_ed25519: Ed25519 = Ed25519.generate().unwrap()\n```\n\n### cls.load_or_generate_file() -> Ed25519\n\nLoad ed25519 keys from files - if files not exists then generates Ed25519, writes private and public keys to the files and returns Ed25519.  \nParameters:  \n**(private_path: str='custom_private_path.pem', public_path: str='custom_public_path.pem')**  \nare optional, if not passed then default params are used  \n**(private_path: str='sk_ed25519.pem', public_path: str='pk_ed25519.pem')**\n\n#### Example:\n\n```python\ned25519_: Ed25519 = Ed25519.load_or_generate_file().unwrap()\n\ned25519_: Ed25519 = Ed25519.load_or_generate_file(private_path='custom_path_to_sk_ed25519.pem',\n                                                  public_path='custom_path_to_pk_ed25519.pem').unwrap()\n```\n\n### self.sign(data: bytes) -> bytes\n\nSign data with ed25519 private key\n\n#### Example:\n\n```python\nsigned_data: bytes = ed25519_.sign(data=b'test_test_b').unwrap()\n```\n\n### self.verify(signature: bytes, data: bytes) -> bool\n\nVerify signature with ed25519 public key\n\n#### Example:\n\n```python\nverified: bool = ed25519_.verify(signature=signed_data, data=b'test_test_b').unwrap()\n```\n\n### self.get_raw_private_key_bytes() -> bytes\n\nGet serialized bytes from the private key.\n\n#### Example:\n\n```python\nraw_private_key_bytes: bytes = ed25519_.get_raw_private_key_bytes().unwrap()\n```\n\n### self.get_raw_private_key() -> str\n\nGet serialized bytes from the private key decoded to string\n\n#### Example:\n\n```python\nraw_private_key_string: str = ed25519_.get_raw_private_key().unwrap()\n```\n\n### self.get_raw_private_key_bytes() -> bytes\n\nGet serialized bytes from the private key.\n\n#### Example:\n\n```python\nraw_public_key_bytes: bytes = ed25519_.get_raw_public_key_bytes().unwrap()\n```\n\n### self.get_raw_private_key() -> str\n\nGet serialized bytes from the private key decoded to string\n\n#### Example:\n\n```python\nraw_private_key_string: str = ed25519_.get_raw_public_key().unwrap()\n```\n\n---\n\n## Secp2561\n\n### cls.generate() -> Secp2561\n\nGenerate Secp2561\n\n#### Example:\n\n```python\ngenerated_secp256k1: Secp2561 = Secp2561.generate().unwrap()\n```\n\n### cls.load_or_generate_file(private_path: str=None, public_path: str=None) -> Secp2561\n\nLoad secp256k1 keys from files - if files not exists then generates Secp2561, writes private and public keys to the \nfiles and returns Secp2561.  \nParameters:  \n**(private_path: str='custom_private_path.pem', public_path: str='custom_public_path.pem')**  \nare optional, if not passed then default params are used:  \n**(private_path: str='sk_secp256k1.pem', public_path: str='pk_secp256k1.pem')**\n\n#### Examples:\n\n```python\nsecp256k1_: Secp2561 = Secp2561.load_or_generate_file().unwrap()\n\nsecp256k1_: Secp2561 = Secp2561.load_or_generate_file(private_path='custom_path_to_sk_secp256k1.pem',\n                                                      public_path='custom_path_to_pk_secp256k1.pem').unwrap()\n```\n\n### self.sign(data: bytes) -> bytes\n\nSign data with SECP256K1 private key\n\n#### Example:\n\n```python\nsigned_data: bytes = secp256k1_.sign(data=b'test_test_b').unwrap()\n```\n\n### self.verify(signature: bytes, data: bytes) -> bool\n\nVerify signature with SECP256K1 public key\n\n#### Example:\n\n```python\nverified: bool = secp256k1_.verify(signature=b'signed_data', data=b'test_test_b').unwrap()\n```\n\n---\n\n## Rand\n\n### gen_random_int(bits: int) -> int\n\nGenerate a random integer, based on passed number of bits\n\n#### Example:\n\n```python\nrandom_int_bits: int = gen_random_int(bits=256).unwrap()\n```\n\n### gen_random_int_hex(bits: int) -> str\n\nGenerate random string, based on passed number of bits\n\n#### Example:\n\n```python\nrandom_int_hex: str = gen_random_int_hex(bits=256).unwrap()\n```\n\n### gen_random_int_hex_bytes(bits: int) -> bytes\n\nGenerate random string, based on passed number of bits\n\n#### Example:\n\n```python\nrandom_int_hex_bytes: bytes = gen_random_int_hex_bytes(bits=128).unwrap()\n```\n\n### gen_random_int128(bits: int) -> int\n\nGenerate a random integer, based on 128 bits\n\n#### Example:\n\n```python\nrandom_int_128: int = gen_random_int128().unwrap()\n```\n\n### gen_random_int128_hex(bits: int) -> str\n\nGenerate random string, based on 128 bits\n\n#### Example:\n\n```python\nrandom_int_128_hex: str = gen_random_int128_hex().unwrap()\n```\n\n### gen_random_int128_hex_bytes(bits: int) -> bytes\n\nGenerate random bytes, based on 128 bits\n\n#### Example:\n\n```python\nrandom_int_128_hex_bytes: bytes = gen_random_int128_hex_bytes().unwrap()\n```\n\n### gen_random_int256(bits: int) -> int\n\nGenerate a random integer, based on 256 bits\n\n#### Example:\n\n```python\nrandom_int_256: int = gen_random_int256().unwrap()\n```\n\n### gen_random_int256_hex(bits: int) -> str\n\nGenerate random string, based on 256 bits\n\n#### Example:\n\n```python\nrandom_int_256_hex: str = gen_random_int256_hex().unwrap()\n```\n\n### gen_random_int256_hex_bytes(bits: int) -> bytes\n\nGenerate random bytes, based on 256 bits\n\n#### Example:\n\n```python\nrandom_int_256_hex_bytes: bytes = gen_random_int256_hex_bytes().unwrap()\n```\n\n---\n\n## Fernet\n\n### cls.generate(cls) -> Fernet\n\nGenerate Fernet\n\n#### Example:\n\n```python\ngenerated_fernet: Fernet = Fernet.generate().unwrap()\n```\n\n### cls.load_or_generate_file(path: str='fernet.key') -> Fernet\n\nLoad Fernet from file or generate one This function tries to load the Fernet from file, if file does not exist then generate Fernet and writes it down to a file.  \nFunction accepts optional 'path' parameter **(path: str='custom_path.key')** or uses default value **(path: str='fernet.key')**\n\n#### Example:\n\n```python\nfernet_: Fernet = Fernet.load_or_generate_file().unwrap()\n\nfernet_: Fernet = Fernet.load_or_generate_file(path='custom_path_fernet.key').unwrap()\n```\n\n### self.encrypt_bytes(data_bytes: bytes, current_time: int | None=None) -> bytes\n\nEncrypt bytes with Fernet key This function takes required argument 'data_bytes' and optional argument 'current_time'\n\n#### Example:\n\n```python\nencrypted_bytes: bytes = fernet_.encrypt_bytes(data_bytes=b'test bytes', \n                                               current_time=int(time())).unwrap()\n```\n\n### self.decrypt_bytes(enc_data_bytes: bytes, ttl: int | None=None, current_time: int | None=None) -> bytes\n\nDecrypt bytes with Fernet key.  \nThis function takes required argument 'enc_data_bytes' and two optional arguments 'ttl' and 'current_time'\n\n#### Example:\n\n```python\ndecrypt_bytes: bytes = fernet_.decrypt_bytes(enc_data_bytes=encrypted_bytes, \n                                             ttl=100, \n                                             current_time=int(time())).unwrap()\n```\n\n### self.encrypt_dict(data_dict: dict, current_time: int | None=None) -> str\n\nEncrypt bytes with Fernet key This function takes required argument 'data_bytes' and optional argument 'current_time'\n\n#### Example:\n\n```python\nencrypted_dict: str = fernet_.encrypt_dict(data_dict={'test_key': 'test_value'}, current_time=int(time())).unwrap()\n```\n\n### self.decrypt_dict(enc_data_str: str, ttl: int | None=None, current_time: int | None=None) -> dict\n\nDecrypt bytes with Fernet key.  \nThis function takes required argument 'data_bytes' and two optional arguments 'ttl' and 'current_time'\n\n#### Example:\n\n```python\ndecrypted_dict: dict = fernet_.decrypt_dict(enc_data_str=encrypted_dict, ttl=100, current_time=int(time())).unwrap()\n```\n\n---\n\n## Multifernet\n\nCreate Multifernet instance\n\n#### Example:\n\n```python\nfernet_key_1: Fernet = Fernet.generate().unwrap()\nmulti_fernet_1: MultiFernet = MultiFernet([fernet_key_0])\n```\n\n### self.encrypt_bytes(data_bytes: bytes, current_time: int | None=None) -> bytes\n\nEncrypt bytes. This function takes required argument 'data_bytes' and optional argument 'current_time'\n\n#### Example:\n\n```python\nencrypted_bytes: bytes = multi_fernet_1.encrypt_bytes(data_bytes=b'12345', current_time=int(time())).unwrap()\n```\n\n### self.decrypt_bytes(enc_data_bytes: bytes, ttl: int | None=None, current_time: int | None=None) -> bytes\n\nDecrypt bytes. This function takes required argument 'enc_data_bytes' and two optional arguments 'ttl' and '\ncurrent_time'\n\n#### Example:\n\n```python\ndecrypted_bytes: bytes = multi_fernet_1.decrypt_bytes(enc_data_bytes=encrypted_bytes, ttl=100,\n                                                      current_time=int(time())).unwrap()\n```\n\n### self.encrypt_dict(data_dict: dict, current_time: int | None=None) -> str\n\nEncrypt dict. This function takes required argument 'data_dict' and optional argument 'current_time'\n\n#### Example:\n\n```python\nencrypted_dict: str = multi_fernet_1.encrypt_dict(data_dict={'test_key': 'test_value'},\n                                                  current_time=int(time())).unwrap()\n```\n\n### self.decrypt_dict(enc_data_str: str, ttl: int | None=None, current_time: int | None=None) -> dict\n\nDecrypt dict. This function takes required argument 'enc_data_bytes' and two optional arguments 'ttl' and 'current_time'\n\n#### Example:\n\n```python\ndecrypted_dict: dict = multi_fernet_1.decrypt_dict(enc_data_str=encrypted_dict, ttl=100,\n                                                   current_time=int(time())).unwrap()\n```\n\n### self.add_fernet(fernet: Fernet) -> MultiFernet\n\nAdd new fernet. This function takes required argument 'fernet' and returns new instance of MultiFernet with new fernet\nkey added to beginning of fernet key list\n\n#### Example:\n\n```python\nfernet_key_2: Fernet = Fernet.generate().unwrap()\nmulti_fernet_2: MultiFernet = multi_fernet_1.add_fernet(fernet_key_2).unwrap()\n```\n\n### self.rotate(enc_data_bytes: bytes) -> bytes\n\nRotate fernet keys (re-encrypt token)\n\n#### Example:\n\n```python\nrotated_msg_with_multi_fernet_2: bytes = multi_fernet_2.rotate(encrypted_bytes).unwrap()\n```\n\n<!-- Links -->\n\n<!-- Badges -->\n[bsd3-image]: https://img.shields.io/badge/License-BSD_3--Clause-blue.svg\n[bsd3-url]: https://opensource.org/licenses/BSD-3-Clause\n[build-image]: https://img.shields.io/badge/build-success-brightgreen\n[coverage-image]: https://img.shields.io/badge/Coverage-100%25-green\n\n[pypi-project-url]: https://pypi.org/project/thcrypto/\n[stable-ver-image]: https://img.shields.io/pypi/v/thcrypto?label=stable\n[python-ver-image]: https://img.shields.io/pypi/pyversions/thcrypto.svg?logo=python&logoColor=FBE072\n[status-image]: https://img.shields.io/pypi/status/thcrypto.svg\n\n",
    'author': 'TangledHub',
    'author_email': 'info@tangledgroup.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://gitlab.com/tangledlabs/thcrypto',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
