"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcProvider = exports.AmiProvider = exports.Cluster = void 0;
const cdk = require("@aws-cdk/core");
const ec2 = require("@aws-cdk/aws-ec2");
const s3 = require("@aws-cdk/aws-s3");
const iam = require("@aws-cdk/aws-iam");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const DEFAULT_INSTANCE_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M6G, ec2.InstanceSize.MEDIUM);
let PriceMap = new Map([
    ['m6g.medium', '0.0385'],
    ['m6g.large', '0.077'],
]);
/**
 * Represents the k3sCluster construct
 */
class Cluster extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        super(scope, id);
        // VPC configuration
        const vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // S3 bucket to host K3s token + kubeconfig file 
        const k3sBucket = new s3.Bucket(this, 'k3sBucket', {});
        // control plane node Security Group      
        const k3scontrolplanesg = new ec2.SecurityGroup(this, 'k3s-controlplane-SG', {
            vpc,
            securityGroupName: 'k3s-controlplane-SG',
            allowAllOutbound: true,
        });
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // worker nodes Security Group      
        const k3sworkersg = new ec2.SecurityGroup(this, 'k3s-worker-SG', {
            vpc,
            securityGroupName: 'k3-worker-SG',
            allowAllOutbound: true,
        });
        // for this prototype the workers are being placed in a public subnet 
        // ideally they should land on a private subnet 
        /// also ingress traffic - ssh (bastion style) or 6443 - should come from the control plane node only 
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // check if the user requires a particular instance type for workers and control plane
        // if not, the default instance type is used 
        this.controlPlaneInstanceType = (_b = props.controlPlaneInstanceType) !== null && _b !== void 0 ? _b : DEFAULT_INSTANCE_TYPE;
        this.workerInstanceType = (_c = props.workerInstanceType) !== null && _c !== void 0 ? _c : DEFAULT_INSTANCE_TYPE;
        // create control plane node
        const k3scontrolplane = new ec2.Instance(this, 'k3s-controlplane', {
            instanceType: this.controlPlaneInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnets: vpc.publicSubnets,
            },
            instanceName: 'k3s-controlplane',
            securityGroup: k3scontrolplanesg,
        });
        k3scontrolplane.addUserData(`
       #!/bin/bash
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.9%2Bk3s1/k3s-arm64
       chmod +x k3s
       ./k3s server &
       sleep 30
       ENDPOINT=$(curl http://169.254.169.254/latest/meta-data/public-hostname) 
       cp /etc/rancher/k3s/k3s.yaml /etc/rancher/k3s/kubeconfig.yaml
       sed -i s/127.0.0.1/$ENDPOINT/ /etc/rancher/k3s/kubeconfig.yaml
       aws s3 cp /var/lib/rancher/k3s/server/node-token s3://${k3sBucket.bucketName}/node-token
       aws s3 cp /etc/rancher/k3s/kubeconfig.yaml s3://${k3sBucket.bucketName}/kubeconfig.yaml
     `);
        this.endpointUri = k3scontrolplane.instancePublicIp;
        // create worker ASG
        const workerAsg = new autoscaling.AutoScalingGroup(this, 'WorkerAsg', {
            instanceType: this.workerInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
            minCapacity: (_d = props.workerMinCapacity) !== null && _d !== void 0 ? _d : 3,
            spotPrice: props.spotWorkerNodes ? PriceMap.get(this.workerInstanceType.toString()) : undefined,
        });
        workerAsg.addUserData(`
       #!/bin/bash
       LOGFILE='/var/log/k3s.log'
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.13%2Bk3s1/k3s-arm64
       chmod +x k3s
       echo the bucket name is ${k3sBucket.bucketName} 
       aws s3 cp s3://${k3sBucket.bucketName}/node-token /node-token 
       (./k3s agent --server https://${k3scontrolplane.instancePrivateIp}:6443 \
       --token $(cat /node-token) 2>&1 | tee -a $LOGFILE || echo "failed" > $LOGFILE &)
    `);
        workerAsg.addSecurityGroup(k3sworkersg);
        // enable the SSM session manager
        workerAsg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // grant the S3 write permission to the control plane node and read permissions to the worker nodes
        k3sBucket.grantWrite(k3scontrolplane.role);
        k3sBucket.grantRead(workerAsg.role);
        // endpoint info
        new cdk.CfnOutput(this, 'Endpoint', { value: `https://${k3scontrolplane.instancePublicIp}:6443` });
        // kubeconfig.yaml path
        new cdk.CfnOutput(this, 'Kubernetes configuration file', { value: `s3://${k3sBucket.bucketName}/kubeconfig.yaml` });
        workerAsg.node.addDependency(k3scontrolplane);
    }
}
exports.Cluster = Cluster;
/**
 * The AMI provider to get the latest Amazon Linux 2 AMI for ARM64
 */
class AmiProvider {
    get amiId() {
        return ec2.MachineImage.latestAmazonLinux({
            cpuType: ec2.AmazonLinuxCpuType.ARM_64,
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
    }
}
exports.AmiProvider = AmiProvider;
/**
 * The VPC provider to create or import the VPC
 */
class VpcProvider {
    static getOrCreate(scope) {
        const vpc = scope.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
            scope.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
        return vpc;
    }
}
exports.VpcProvider = VpcProvider;
//# sourceMappingURL=data:application/json;base64,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