"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubnetNetworkAclAssociation = exports.NetworkAclEntry = exports.TrafficDirection = exports.Action = exports.NetworkAcl = void 0;
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * A NetworkAclBase that is not created in this template
 *
 * @experimental
 */
class NetworkAclBase extends core_1.Resource {
    /**
     * Add a new entry to the ACL
     */
    addEntry(id, options) {
        return new NetworkAclEntry(this, id, {
            networkAcl: this,
            ...options,
        });
    }
}
/**
 * Define a new custom network ACL
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @experimental
 */
class NetworkAcl extends NetworkAclBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclName,
        });
        this.vpc = props.vpc;
        this.networkAcl = new ec2_generated_1.CfnNetworkAcl(this, 'Resource', {
            vpcId: props.vpc.vpcId,
        });
        this.networkAclId = this.networkAcl.ref;
        this.networkAclVpcId = this.networkAcl.vpcId;
        if (props.subnetSelection !== undefined) {
            this.associateWithSubnet('DefaultAssociation', props.subnetSelection);
        }
    }
    /**
     * Import an existing NetworkAcl into this app.
     */
    static fromNetworkAclId(scope, id, networkAclId) {
        class Import extends NetworkAclBase {
            constructor() {
                super(...arguments);
                this.networkAclId = networkAclId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Associate the ACL with a given set of subnets
     */
    associateWithSubnet(id, selection) {
        const subnets = this.vpc.selectSubnets(selection);
        for (const subnet of subnets.subnets) {
            subnet.associateNetworkAcl(id, this);
        }
    }
}
exports.NetworkAcl = NetworkAcl;
/**
 * What action to apply to traffic matching the ACL
 *
 * @experimental
 */
var Action;
(function (Action) {
    /**
     * Allow the traffic
     */
    Action["ALLOW"] = "allow";
    /**
     * Deny the traffic
     */
    Action["DENY"] = "deny";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Base class for NetworkAclEntries
 *
 * @experimental
 */
class NetworkAclEntryBase extends core_1.Resource {
}
/**
 * Direction of traffic the AclEntry applies to
 *
 * @experimental
 */
var TrafficDirection;
(function (TrafficDirection) {
    /**
     * Traffic leaving the subnet
     */
    TrafficDirection[TrafficDirection["EGRESS"] = 0] = "EGRESS";
    /**
     * Traffic entering the subnet
     */
    TrafficDirection[TrafficDirection["INGRESS"] = 1] = "INGRESS";
})(TrafficDirection = exports.TrafficDirection || (exports.TrafficDirection = {}));
/**
 * Define an entry in a Network ACL table
 *
 * @experimental
 */
class NetworkAclEntry extends NetworkAclEntryBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclEntryName,
        });
        this.networkAcl = props.networkAcl;
        new ec2_generated_1.CfnNetworkAclEntry(this, 'Resource', {
            networkAclId: this.networkAcl.networkAclId,
            ruleNumber: props.ruleNumber,
            ruleAction: props.ruleAction !== undefined ? props.ruleAction : Action.ALLOW,
            egress: props.direction !== undefined ? props.direction === TrafficDirection.EGRESS : undefined,
            ...props.traffic.toTrafficConfig(),
            ...props.cidr.toCidrConfig(),
        });
    }
}
exports.NetworkAclEntry = NetworkAclEntry;
/**
 * Associate a network ACL with a subnet
 *
 * @experimental
 */
class SubnetNetworkAclAssociationBase extends core_1.Resource {
}
class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.subnetNetworkAclAssociationName,
        });
        this.association = new ec2_generated_1.CfnSubnetNetworkAclAssociation(this, 'Resource', {
            networkAclId: props.networkAcl.networkAclId,
            subnetId: props.subnet.subnetId,
        });
        this.networkAcl = props.networkAcl;
        this.subnet = props.subnet;
        this.subnetNetworkAclAssociationAssociationId = this.association.attrAssociationId;
    }
    static fromSubnetNetworkAclAssociationAssociationId(scope, id, subnetNetworkAclAssociationAssociationId) {
        class Import extends SubnetNetworkAclAssociationBase {
            constructor() {
                super(...arguments);
                this.subnetNetworkAclAssociationAssociationId = subnetNetworkAclAssociationAssociationId;
            }
        }
        return new Import(scope, id);
    }
}
exports.SubnetNetworkAclAssociation = SubnetNetworkAclAssociation;
//# sourceMappingURL=data:application/json;base64,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