"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AclTraffic = exports.AclCidr = void 0;
/**
 * Either an IPv4 or an IPv6 CIDR
 *
 * @experimental
 */
class AclCidr {
    /**
     * An IP network range in CIDR notation (for example, 172.16.0.0/24).
     */
    static ipv4(ipv4Cidr) {
        return new AclCidrImpl({
            cidrBlock: ipv4Cidr,
        });
    }
    /**
     * The CIDR containing all IPv4 addresses (i.e., 0.0.0.0/0)
     */
    static anyIpv4() {
        return AclCidr.ipv4('0.0.0.0/0');
    }
    /**
     * An IPv6 network range in CIDR notation (for example, 2001:db8::/48)
     */
    static ipv6(ipv6Cidr) {
        return new AclCidrImpl({
            ipv6CidrBlock: ipv6Cidr,
        });
    }
    /**
     * The CIDR containing all IPv6 addresses (i.e., ::/0)
     */
    static anyIpv6() {
        return AclCidr.ipv6('::/0');
    }
}
exports.AclCidr = AclCidr;
class AclCidrImpl extends AclCidr {
    constructor(config) {
        super();
        this.config = config;
    }
    toCidrConfig() {
        return this.config;
    }
}
/**
 * The traffic that is configured using a Network ACL entry
 *
 * @experimental
 */
class AclTraffic {
    /**
     * Apply the ACL entry to all traffic
     */
    static allTraffic() {
        return new AclTrafficImpl({
            protocol: -1,
        });
    }
    /**
     * Apply the ACL entry to ICMP traffic of given type and code
     */
    static icmp(props) {
        return new AclTrafficImpl({
            protocol: 1,
            icmp: props,
        });
    }
    /**
     * Apply the ACL entry to ICMPv6 traffic of given type and code
     *
     * Requires an IPv6 CIDR block.
     */
    static icmpv6(props) {
        return new AclTrafficImpl({
            protocol: 58,
            icmp: props,
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port
     */
    static tcpPort(port) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: port,
                to: port,
            },
        });
    }
    /**
     * Apply the ACL entry to TCP traffic on a given port range
     */
    static tcpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 6,
            portRange: {
                from: startPort,
                to: endPort,
            },
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port
     */
    static udpPort(port) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: port,
                to: port,
            },
        });
    }
    /**
     * Apply the ACL entry to UDP traffic on a given port range
     */
    static udpPortRange(startPort, endPort) {
        return new AclTrafficImpl({
            protocol: 17,
            portRange: {
                from: startPort,
                to: endPort,
            },
        });
    }
}
exports.AclTraffic = AclTraffic;
class AclTrafficImpl extends AclTraffic {
    constructor(config) {
        super();
        this.config = config;
    }
    toTrafficConfig() {
        return this.config;
    }
}
//# sourceMappingURL=data:application/json;base64,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