import msaf
import librosa
import numpy as np
from .helpers import get_label_groups, clean_labels, clean_tempos, get_beats_from_boundaries

def get_clean_labels(y, sections, dirty_labels):
    '''
    Cleans dirty labels to produce more easily readable labels.
    
    Parameters
    ----------
    y: audio_signal
    sections: the sections list generated by `get_boundaries()`
    dirty_labels: the labels list generated by `get_labels()`
        
    Returns
    -------
    labels: cleaned label output.
        If the intro/outro sections are unique, they will be replaced by 'intro' and 'outro'
        A dirty_labels list that looks like [0, 1, 2, 0, 1] will be converted to ['A', 'B', 'C', 'A', 'B']
    '''
    label_groups = get_label_groups(y, sections, dirty_labels)

    if len(label_groups[dirty_labels[0]]) == 1:
        dirty_labels[0] = 'intro'
    if len(label_groups[dirty_labels[-1]]) == 1:
        dirty_labels[-1] = 'outro'

    labels = clean_labels(dirty_labels)

    return labels

def get_boundaries(audio_file: str, sample_rate: int, boundary_detection_id: str = "olda", label_detection_id: str = "scluster"):
    '''
    Parses labels and sections to generate a mapping of labels to actual sections and a measure of how percussive each section is
    
    Parameters
    ----------
    audio_file: filename of desired audio file
    sample_rate: sample_rate of audio file
    boundary_detection_id: 
        Algorithm for boundary detection. 
        Other algorithms are documented here: https://github.com/urinieto/msaf/blob/master/examples/Run%20MSAF.ipynb
    label_detection_id: 
        Algorithm for label_detection. 
        Other algorithms are documented here: https://github.com/urinieto/msaf/blob/master/examples/Run%20MSAF.ipynb
        
    Returns
    -------
    sections:
        A list of tuples designating the start/end of each section in samples
    dirty_labels:
        A list of integers corresponding to the label each section has been assigned 
        (e.g. a song structure of ABCAB could have a list of [0, 1, 2, 0, 1])
    '''

    bounds, labels = msaf.process(audio_file, boundaries_id=boundary_detection_id, labels_id=label_detection_id, out_sr=sample_rate)
    sections = np.array([( int(bounds[i] * sample_rate), int(bounds[i+1] * sample_rate) ) for i in range(len(bounds)-1)])
    boolarr = [True if end - start > sample_rate else False for start, end in sections]
    sections = sections[boolarr]
    labels = np.array(labels)[boolarr]
    dirty_labels = [int(i) for i in labels]

    return sections, dirty_labels

def read_audio(audio_file: str, sample_rate: int):
    '''
    Wrapper for librosa's load function to read in an audio signal
    
    Parameters
    ----------
    audio_file: filename of desired audio file
    sample_rate: sample_rate of audio file
        
    Returns
    -------
    y: audio signal
    sr: sample_rate
    '''

    y, sr = librosa.load(audio_file, sr = sample_rate)

    return y, sr

def get_section_beats(y, sr: int, sections):
    '''
    Gets the beat numbers of each section boundary based on estimated beat values
    
    Parameters
    ----------
    y: audio_signal
    sr: sample_rate
    sections: the sections list generated by `get_boundaries()`
        
    Returns
    -------
    section_beats: list of tuples corresponding to the section boundaries for each section in beats
    '''

    beats = librosa.beat.beat_track(y, sr)
    section_beats = get_beats_from_boundaries(beats[1], sections, sr)

    return section_beats

def get_tempos(y, sr: int, sections):
    '''
    Gets and cleans estimated tempo values for each section to analyse change in tempo
    
    Parameters
    ----------
    y: audio_signal
    sr: sample_rate
    sections: the sections list generated by `get_boundaries()`
        
    Returns
    -------
    tempos: list of cleaned tempos corresponding to each section
    '''

    tempos = clean_tempos(y, sr, sections)
    
    return tempos

def get_structure_array(sr: int, sections, section_beats, labels, tempos):
    '''
    Formats basic structure information in an array format
    
    Parameters
    ----------
    sr: sample_rate
    sections: the sections list generated by `get_boundaries()`
    labels: the labels list generated by `get_boundaries()`
    section_beats: the section_beats list generated by `get_section_beats()`
    tempos: the tempos list generated by `get_tempos()`
        
    Returns
    -------
    output: A list where each value corresponds to a section, and each section contains the metadata:
        length_seconds: length of the section in seconds
        length_beats: length of the section in beats
        label: the label that this section was assigned
        tempo: the estimated tempo of the section
    '''

    output = []
    for section_no in range(len(sections)):
        output.append({
            'length_seconds': (sections[section_no][1] - sections[section_no][0])/sr,
            'length_beats': section_beats[section_no][1] - section_beats[section_no][0],
            'label' : labels[section_no],
            'tempo' : tempos[section_no],
        })
    
    return output

def process(audio_file: str, sample_rate: int, boundary_detection_id: str = "olda", label_detection_id: str = "scluster"):
    '''
    Formats basic structure information in regular 
    
    Parameters
    ----------
    audio_file: filename of the audio file
    sample_rate: sample rate of the audio file
        
    Returns
    -------
    output: A list where each value corresponds to a section, and each section contains the metadata:
        length_seconds: length of the section in seconds
        length_beats: length of the section in beats
        label: the label that this section was assigned
        tempo: the estimated tempo of the section
    '''

    y, sr = read_audio(audio_file, sample_rate)
    sections, labels = get_boundaries(audio_file, sample_rate, boundary_detection_id, label_detection_id)

    section_beats = get_section_beats(y, sr, sections)
    tempos = get_tempos(y, sr, sections)
    labels = get_clean_labels(y, sections, labels)

    return get_structure_array(sr, sections, section_beats, labels, tempos)
