# asteroid-spinprops

## Overview
**asteroid-spinprops** is a Python package providing tools to fit SHG1G2 and SOCCA photometric models to sparse asteroid photometry.  
It supports multiband modeling, residual analysis and shape, period and pole orientation estimation for small solar system objects.

---

## Installation
Install the package via pip:

```bash
pip install asteroid_spinprops
```

## Quick Start
```python
import numpy as np
import pandas as pd
from asteroid_spinprops.ssolib import dataprep, periodest, modelfit

# Suppose `pdf` is your initial asteroid DataFrame 
# Ensure all columns are converted to the required single row format.
pdf_s = pd.DataFrame({col: [np.array(pdf[col])] for col in pdf.columns})

# Convert filter IDs to numeric
unique_vals, inv = np.unique(pdf_s["cfid"].values[0], return_inverse=True)
numeric_filter = inv + 1
pdf_s["cfid"].values[0] = numeric_filter

# --- Data cleaning and filtering ---
clean_data, errorbar_rejects = dataprep.errorbar_filtering(data=pdf_s, mlimit=0.7928)
clean_data, projection_rejects = dataprep.projection_filtering(data=clean_data)
clean_data, iterative_rejects = dataprep.iterative_filtering(data=clean_data)

# --- Fit SHG1G2 model ---
shg1g2_params = modelfit.get_fit_params(
    data=clean_data,
    flavor="SHG1G2",
)

# Compute residuals for period analysis
residuals_dataframe = modelfit.make_residuals_df(
    clean_data, model_parameters=shg1g2_params
)

# --- Estimate rotation period ---
p_in, k_val, p_rms, signal_peak, window_peak = periodest.get_multiband_period_estimate(
    residuals_dataframe,
    k_free=True,
)

# Assess period robustness via bootstrap resampling
_, Nbs = periodest.perform_residual_resampling(
    resid_df=residuals_dataframe,
    p_min=0.03,
    p_max=2,
    k=int(k_val)
)

# --- Fit SSHG1G2 (spin + multiband) model ---
SOCCA_params = modelfit.get_fit_params(
    data=clean_data,
    flavor="SSHG1G2",
    shg1g2_constrained=True,
    blind_scan=True,
    period_in=p_in,
)
```

## Models
Photometric models from Carry et al.(2024) {2024A&A...687A..38C}
and https://github.com/astrolabsoftware

## Project status
Under development
