"""Simple tox plugin to facilitate working with abstract and concrete dependencies"""
import dataclasses
import itertools
import logging
import os
import sys
import textwrap
from collections import defaultdict
from pathlib import Path
from typing import Dict, Iterator, Set, Optional

import pkg_resources
import toml
import tox  # type: ignore
import tox.config  # type: ignore
import tox.reporter  # type: ignore
import tox.venv  # type: ignore

logger = logging.getLogger(__name__)

REQ_PATH = Path("requirements")
ARG_NAME = "--requirements-file"


def _header(name, value):
    return textwrap.dedent(
        f"""\
        #
        # This file is autogenerated by tox-constraints
        # To update, run:
        #
        #    tox {name} {value}
        #
        """
    )


@dataclasses.dataclass
class Config:  # pylint: disable=too-few-public-methods
    """A collection of configurable parameters for this tool.

    This class serves as a single source of documentation for what parameters exist and
    what types and values they may take.
    """

    @staticmethod
    def read():
        """Read from default path"""
        pyproject_toml = toml.load("pyproject.toml")
        section = pyproject_toml["tool"]["tox-constraints"]

        # If any configuration of this plugin exists, enable the plugin unless
        # explicitly disabled to reduce the configuration needed.
        plugin_enabled = section.get("plugin_enabled", True)

        # Additional concrete requirements that cannot be captured in the constraints
        # file. This could be because they point to a path relative to the package
        # under test.
        concrete = section.get("concrete", {})

        constraints_filename = section.get("constraints_filename", "constraints.txt")

        return Config(
            plugin_enabled=plugin_enabled,
            concrete=concrete,
            constraints_filename=constraints_filename,
        )

    plugin_enabled: bool
    concrete: Dict[str, str]
    constraints_filename: str


_SAMPLE = """
# alpha
b
c > 3
d ; python_version >= '3.7'
e-f
g # hotel
black==19.3b0 \\
    --hash=sha256:09a9dcb7c46ed496a9850b76e4e825d6049ecd38b611f1224857a79bd985a8cf \\
    --hash=sha256:68950ffd4d9169716bcb8719a56c07a2f4485354fec061cdd5910aa07369731c \\
    # via -r requirements/tox.txt
"""


def _parse_requirements(text: str) -> Iterator[pkg_resources.Requirement]:
    """
    >>> sorted(map(str,_parse_requirements(_SAMPLE)))
    ['b', 'black==19.3b0', 'c>3', 'd; python_version >= "3.7"', 'e-f', 'g']
    """
    # pylint: disable=no-member
    for line in text.splitlines():
        try:
            ham = line.split("#")[0].strip(" \\")
            if not ham:
                continue
            yield pkg_resources.Requirement.parse(ham)
        except pkg_resources.extern.packaging.requirements.InvalidRequirement:  # type: ignore
            pass


def _expands_requirements(tool_config: Config, envconfig) -> None:
    filenames = ["install_requires.txt"] + [
        f"extras_require-{extra}.txt" for extra in envconfig.extras
    ]

    dep_names: Set[str] = set()
    for filename in filenames:
        filepath = REQ_PATH / filename
        dep_names.update(map(str, _parse_requirements(filepath.read_text())))

    new = {
        dep_name: tox.config.DepConfig(tool_config.concrete.get(dep_name, dep_name))
        for dep_name in dep_names
    }
    old = {dep.name: i for i, dep in enumerate(envconfig.deps)}

    for dep_name in new:
        if dep_name in old:
            envconfig.deps[old[dep_name]] = new[dep_name]
        else:
            envconfig.deps.append(new[dep_name])


def _find_upwards(start: Path, filename: str) -> Path:
    for dirpath in itertools.chain([start], start.parents):
        filepath = dirpath / filename
        if filepath.exists():
            return filepath

    raise FileNotFoundError


def _patch_envconfigs(tool_config: Config, config):
    for name, envconfig in config.envconfigs.items():
        if name == ".package":
            # Don't patch isolated packaging environment because some parsing will fail
            # on -cconstraints.txt
            continue
        if envconfig.skip_install is True and not envconfig.deps:
            # Avoid attempting to install no packages as pip does not like this:
            # > ERROR: You must give at least one requirement to install
            continue

        for dep in envconfig.deps:
            if dep.name.startswith("-c"):
                break
        else:
            constraints_filepath = _find_upwards(
                Path(config.toxinidir), tool_config.constraints_filename
            )
            envconfig.deps.append(tox.config.DepConfig(f"-c{constraints_filepath}"))

        if envconfig.skip_install is True:
            pass
        elif envconfig.skip_install is False:
            _expands_requirements(tool_config, envconfig)
        else:
            raise ValueError


def _save_if_different(path: Path, new_content: str):
    """Save `new_content` to file if it is different to the existing content.

    Useful e.g. for only touching files when they change to avoid triggering make.
    """
    try:
        with path.open("r") as fp:
            old_content: Optional[str] = fp.read()
    except FileNotFoundError:
        old_content = None

    if old_content != new_content:
        with path.open("w") as fp:
            fp.write(new_content)


def _export_deps(envconfigs, path):
    dep2names = defaultdict(list)
    for name, envconfig in envconfigs.items():
        for dep in envconfig.deps:
            if dep.name.startswith("-r"):
                continue  # Potential loop

            if dep.name.startswith("-c"):
                continue  # Constraint override

            dep2names[str(dep)].append(name)

    dep_len = max(len(dep) for dep in dep2names)
    lines = [
        f"{dep:<{dep_len}}  # from {', '.join(sorted(dep2names[dep]))}\n"
        for dep in sorted(dep2names)
    ]
    _save_if_different(path, _header(ARG_NAME, path) + "".join(lines))


@tox.hookimpl
def tox_addoption(parser):
    """Add command line option to display fireworks on request."""
    parser.add_argument(
        ARG_NAME,
        dest="requirementsfile",
        metavar="PATH",
        help="write a requirements file with the packages used by each environment.",
        default=None,
    )


@tox.hookimpl
def tox_configure(config):
    """Apply concrete constraints and export abstract dependencies"""
    try:
        tool_config = Config.read()
    except (FileNotFoundError, KeyError):
        # Disable plugin by default to make it less disruptive in a development
        # environment that is shared by multiple projects
        return

    if tool_config.plugin_enabled:
        if config.option.requirementsfile is not None:
            _export_deps(config.envconfigs, Path(config.option.requirementsfile))
        _patch_envconfigs(tool_config, config)


def _reqs_from_deps(deps):
    paths = [Path(dep.name[2:]) for dep in deps if dep.name.startswith("-c")]

    # Poorly handled corner case: when skip_install is set and there are no deps listed
    # for an environment then the constraints file is not been appended making any
    # information needed to pin the seeds unavailable at this point
    if not paths:
        logger.warning("No constraints file specified, seeds will not be pinned")

    for path in paths:
        text = path.read_text()
        yield from _parse_requirements(text)


def _exact_version(req):
    for version_cmp, version in req.specs:
        if version_cmp == "==":
            return version
    raise LookupError(f"no == for {req}")


def _augmented_virtualenv_args(args, deps):
    yield from args
    constraints = {req.key: req for req in _reqs_from_deps(deps)}
    for distribution in ["pip", "setuptools", "wheel"]:
        try:
            req = constraints[distribution]
            version = _exact_version(req)
            yield f"--{distribution}"
            yield version
        except LookupError:
            logger.warning("Seed %s is not pinned", distribution)


_SKIP_VENV_CREATION = os.environ.get("_TOX_SKIP_ENV_CREATION_TEST", False) == "1"


@tox.hookimpl
def tox_testenv_create(venv, action):
    # pylint: disable=protected-access,missing-docstring
    config_interpreter = venv.getsupportedinterpreter()
    args = [sys.executable, "-m", "virtualenv"]
    if venv.envconfig.sitepackages:
        args.append("--system-site-packages")
    if venv.envconfig.alwayscopy:
        args.append("--always-copy")
    if not venv.envconfig.download:
        args.append("--no-download")
    else:
        args.append("--download")
    # add interpreter explicitly, to prevent using default (virtualenv.ini)
    args.extend(["--python", str(config_interpreter)])

    # Disables it for whole function but place here to consolidate changes
    args = list(_augmented_virtualenv_args(args, venv.envconfig.deps))

    tox.venv.cleanup_for_venv(venv)

    base_path = venv.path.dirpath()
    base_path.ensure(dir=1)
    args.append(venv.path.basename)
    if not tox.venv._SKIP_VENV_CREATION:
        try:
            venv._pcall(
                args,
                venv=False,
                action=action,
                cwd=base_path,
                redirect=tox.reporter.verbosity() < tox.reporter.Verbosity.DEBUG,
            )
        except KeyboardInterrupt:
            venv.status = "keyboardinterrupt"
            raise
    return True  # Return non-None to indicate plugin has completed
