# python-fixturify-project

[![Build status](https://github.com/scalvert/python-fixturify-project/workflows/build/badge.svg?branch=main&event=push)](https://github.com/scalvert/python-fixturify-project/actions?query=workflow%3Abuild)
[![Python Version](https://img.shields.io/pypi/pyversions/python-fixturify-project.svg)](https://pypi.org/project/python-fixturify-project/)
[![Dependencies Status](https://img.shields.io/badge/dependencies-up%20to%20date-brightgreen.svg)](https://github.com/scalvert/python-fixturify-project/pulls?utf8=%E2%9C%93&q=is%3Apr%20author%3Aapp%2Fdependabot)

[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)
[![Security: bandit](https://img.shields.io/badge/security-bandit-green.svg)](https://github.com/PyCQA/bandit)
[![Pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/scalvert/python-fixturify-project/blob/master/.pre-commit-config.yaml)
[![Semantic Versions](https://img.shields.io/badge/%20%20%F0%9F%93%A6%F0%9F%9A%80-semantic--versions-e10079.svg)](https://github.com/scalvert/python-fixturify-project/releases)
[![License](https://img.shields.io/github/license/scalvert/python-fixturify-project)](https://github.com/scalvert/python-fixturify-project/blob/master/LICENSE)
![Coverage Report](assets/images/coverage.svg)

> Dynamic fixture creation for your tests

_A Python port of [node-fixturify-project](https://github.com/stefanpenner/node-fixturify-project)_

## Installation

```bash
pip install -U python-fixturify-project
```

or install with `Poetry`

```bash
poetry add python-fixturify-project --dev
```

## Usage

`python-fixturify-project` is a Python package that provides a way to create dynamic fixtures for your tests. Fixtures are real directories and files, written to a temporary directory.

```python
from python_fixturify_project import Project

dir_json = {
    "valid_file.txt": "some text",
    "nested_dir": {
        "valid_empty_file.txt": "",
        "another_nested_empty_dir": {},
        "another_nested_dir": {
            "last_nested_empty_dir": {},
            "final_text_file.txt": "some text",
        },
    },
}

# create a new project with the given directory structure
project = Project(files=dir_json)
# add new files to the project, merging with the existing directory structure
p.write({
    "new_file.txt": "some text"
})

# read the actual contents on disc
actual_dir_json = p.read()
```

### Ignore Files

By default, the `read()` function will ignore all hidden files and directories in your Project file structure. This can be overridden by using the `ignore_patterns` constructor argument, which
takes a list of glob pattern strings. Any patterns provided to the `ignore_patterns` argument will be used in an _exclusive_ manner. For example:

```python
files = {
    ".git": {
        "a_nested_dir": {}
    },
    ".github": {
        "ignore_me": {},
        "do_not_ignore_me": {
            "a_file": "some text"
        }
    },
    "ignore_me": "some text",
    "do_not_ignore_me": "some text",
}

# Default ignore_patterns is ["**/.git", "**/.git/**"]
project = Project(ignore_patterns=["**/.git", "**/.git/**", "**/ignore_me"])

project.write(files)

assert project.read() == {
    '.github': {
        'do_not_ignore_me': {
            'a_file': 'some text',
        },
    },
    'do_not_ignore_me': 'some text',
}
```

### Recommended Usage Patterns

`python-fixutrify-project` becomes even more useful when combining it with tools like `pytest` and something like [`syrupy`](https://github.com/tophat/syrupy), which uses `jest`-like snapshots for testing. The example below combines `python-fixturify-project` with `pytest`'s fixtures, and  `syrupy` to create a snapshot test.

First, we define a fixture to setup and teardown our `Project` instance:

```python
# conftest.py
import pytest

from python_fixturify_project import Project


@pytest.fixture
def project():
    project = Project()

    yield project

    project.dispose()
```

This fixture uses `pytest`'s `yield` fixture pattern, which allows us to run some code after the test has completed. In this case, we use the `dispose()` method to remove the temporary directory created by `python-fixturify-project`.

```python
from python_fixturify_project import Project


def test_mutating_project(project, snapshot):
    project.write({
        "a_file.txt": "some text",
        "a_dir": {
            "another_file.txt": "some text",
        },
        "path": {
            "to": {
                "a_file.py": "# some python code",
            },
        },
    })

    mutate_files_for_some_reason(p.base_dir)

    # ensure mutations were as expected
    assert project.files == snapshot
```

Or you can use the `project.get` method to get the path to a file in the project.

```python
from python_fixturify_project import Project

def test_mutating_project(snapshot):
    project.write({
        "a_file.txt": "some text",
        "a_dir": {
            "another_file.txt": "some text",
        },
        "path": {
            "to": {
                "a_file.py": "# some python code",
            },
        },
    })

    mutate_files_for_some_reason(p.base_dir)

    # ensure mutations were as  for single file
    assert project.get('path/to/a_file.py') == snapshot(name='path/to/a_file.py')
```

## Skip Dispose (for debugging)

If you want to skip the `dispose()` call, you can set the `FIXTURIFY_SKIP_DISPOSE` environment variable to `1`.

```bash
FIXTURIFY_SKIP_DISPOSE=1 pytest
```

This can be useful if you want to inspect the contents of the temporary directory after the test has completed.

## 🛡 License

[![License](https://img.shields.io/github/license/scalvert/python-fixturify-project)](https://github.com/scalvert/python-fixturify-project/blob/master/LICENSE)

This project is licensed under the terms of the `MIT` license. See [LICENSE](https://github.com/scalvert/python-fixturify-project/blob/master/LICENSE) for more details.
