# AUTOGENERATED! DO NOT EDIT! File to edit: module_notebooks/02_ucrel_doc.ipynb (unless otherwise specified).

__all__ = ['UCREL_Doc']

# Cell

from collections import abc
import json
from typing import List, Tuple, Optional, Iterable, Any

from .ucrel_token import UCREL_Token

class UCREL_Doc(abc.Iterable, abc.Sized):
    '''
    Classs that holds text level lingustic information which is stored as
    a list of `UCREL_Token`s. A text here can be anything from one word
    to a whole book or larger. However becareful how much information
    you store in this class as it is all stored in memory.

    This class is inspired by the [Doc](https://spacy.io/api/doc)
    class from the [SpaCy API.](https://spacy.io/api)

    **inherits from**: [collections.abc.Iterable](https://docs.python.org/3/library/collections.abc.html#collections.abc.Iterable)
    and [collections.abc.Sized](https://docs.python.org/3/library/collections.abc.html#collections.abc.Sized)
    '''
    def __init__(self, text: str, tokens: List[UCREL_Token],
                 sentence_indexes: Optional[List[Tuple[int,int]]] = None
                 ) -> None:
        '''
        1. **text**: The text the Doc is representing.
        2. **tokens**: List of `UCREL_Token`s
        3. **sentence_indexes**: A List of Tuples. Where each tuple
           contains a start and an end token index representing
           the start and end of the sentence. These are used to
           create the `sentences` property. Can be accessed through
           `self._sentence_indexes`. **Optional**
        '''
        self.text = text
        self.tokens = tokens
        self._sentence_indexes = sentence_indexes

    def __repr__(self) -> str:
        '''
        String representation of the UCREL Doc instance:
        '''
        base_repr = 'UCREL Doc:'
        if self._sentence_indexes is not None:
            base_repr = f'UCREL Doc ({len(self._sentence_indexes)} sentences):'
        first_three_tokens = self.tokens[:3]
        first_three_token_strings = '\n'.join([str(token) for token in first_three_tokens])
        base_repr += f'\nFirst {len(first_three_tokens)} tokens:\n'
        base_repr += first_three_token_strings
        return base_repr

    def __iter__(self) -> Iterable[UCREL_Token]:
        '''
        **returns**: Yields each token in `self.tokens`.
        '''
        for token in self.tokens:
            yield token

    def __len__(self) -> int:
        '''
        **returns**: The number of tokens in the Doc.
        '''
        return len(self.tokens)

    def __getitem__(self, index: int) -> UCREL_Token:
        '''
        1. **index**: The index of the token to return.

        **returns**: The token at the given index.
        '''
        return self.tokens[index]

    def __eq__(self, other: Any) -> bool:
        '''
        Compare another instance with the current instance of this
        class.

        1. **other**: Another instance, if this instance is not of this
        class type it will raise a `NotImplementedError`.

        **returns** `True` if the two instances are the same based on
        the following attributes:

        1. text
        2. sentence_indexes
        3. tokens

        **raises NotImplementedError**: If the `other` instance is not of
        the same class type as `self`.
        '''
        # Reference:
        # https://stackoverflow.com/questions/1227121/compare-object-instances-for-equality-by-their-attributes
        if not isinstance(other, UCREL_Doc):
            error_msg = (f"Cannot compare this UCREL Doc {self}\n"
                         "With anything other than another UCREL Doc instance."
                         " The instance that is being compared is of type "
                         f"{type(other)}")
            raise NotImplementedError(error_msg)
        other: UCREL_Doc
        if self.text != other.text:
            return False
        if self._sentence_indexes != other._sentence_indexes:
            return False
        if len(self) != len(other):
            return False
        for token_index, token in enumerate(self):
            if token != other[token_index]:
                return False
        return True

    @property
    def sentences(self) -> Iterable[List[UCREL_Token]]:
        '''
        **returns**: An iterable of all sentences in the text represented
        as a list of `UCREL_Token`s.

        **raises ValueError**: If the `sentence_indexes` parameter is not set
        at contruction time.
        '''
        if self._sentence_indexes is None:
            error_msg = ('The `sentences` attribute cannot be accessed/created '
                         'without `sentence_indexes` at construction time')
            raise ValueError(error_msg)
        for start_index, end_index in self._sentence_indexes:
            yield self.tokens[start_index:end_index]

    def to_json(self) -> str:
        '''
        **returns** This UCREL_Doc as a JSON String.
        '''
        text_json = json.dumps(self.text)
        sentence_indexes_json = json.dumps(self._sentence_indexes)

        token_json = '['
        for token in self.tokens:
            token_json += f'{token.to_json()}, '
        token_json = token_json.rstrip(', ')
        token_json += ']'

        json_string = ('{"text": ' + text_json + ', "tokens": ' + token_json +
                       ', "sentence_indexes": '+ sentence_indexes_json + '}')
        return json_string

    @staticmethod
    def from_json(json_string: str) -> 'UCREL_Doc':
        '''
        A static method that given a `json_string` will
        return a `UCREL_Doc` representation of that string.

        1. **json_string**: A string that is the return of
        `UCREL_Doc.to_json` method

        **returns** The given `json_string` represented through the
        `UCREL_Doc`.
        '''

        json_ucrel_doc = json.loads(json_string)

        # Convert Sentence Indexes from a List of List objects into
        # a List of Tuples
        sentence_indexes = []
        json_sentence_indexes = json_ucrel_doc['sentence_indexes']
        if json_sentence_indexes is not None:
            for index in json_sentence_indexes:
                sentence_indexes.append(tuple(index))
        sentence_indexes = sentence_indexes if sentence_indexes else None
        json_ucrel_doc['sentence_indexes'] = sentence_indexes

        # Convert the Tokens from a Dict object into UCREL_Token object
        ucrel_tokens = []
        for token in json_ucrel_doc['tokens']:
            ucrel_tokens.append(UCREL_Token(**token))
        json_ucrel_doc['tokens'] = ucrel_tokens

        return UCREL_Doc(**json_ucrel_doc)
