import numpy as np
import warnings

def _sigmoidspace(start, stop, n, b=1):
    """Interpolate n points of a sigmoidal function between start and stop.

    Like numpy.linspace, but sigmoidally eased.

    Args:
        start (float): Start value
        stop (float): Stop value
        n (int): The number of points to interpolate
        b (int): start + (stop - start) / (1 + np.exp(-b * np.linspace(-10, 10, n)))
    """
    return start + (stop - start) / (1 + np.exp(-b * np.linspace(-10, 10, n)))


def _rotfunc(xs, ys, rotation_rad):
    """Rotate the coordinates xs and ys by the given angle in radians

    Args:
        xs (numpy.ndarray): X coordinates
        ys (numpy.ndarray): Y coordinates
        rotation_rad (float): Angle by which to rotate the points in radians
    """
    rot = np.array([[np.cos(rotation_rad), -np.sin(rotation_rad)],
                    [np.sin(rotation_rad), np.cos(rotation_rad)]])
    xy = np.column_stack([xs, ys])
    prod = np.matmul(xy, rot)
    return prod[:, 0], prod[:, 1]


class LineScanPattern:
    """
    Template class for all scan patterns for the generation of 2D or 3D images acquired with A-line scans or point exposures.
    Generates normalized signals for the steering of galvos and the triggering of a line camera in order to acquire a given
    geometry.
    """

    def __init__(self):
        
        # All these must be defined by `generate`
        self._line_trigger = np.array([]).astype(np.float64)
        self._frame_trigger = np.array([]).astype(np.float64)
        self._x = np.array([]).astype(np.float64)
        self._y = np.array([]).astype(np.float64)
        self._positions = np.array([]).astype(np.float64)
        self._image_mask = np.array([]).astype(bool)
        self._dimensions = []
        self._points_in_scan: int = 0
        self._points_in_image: int = 0
        self._sample_rate: int = 0
        self._pattern_rate: int = 0
        self._samples_on: int = 0
        self._samples_off: int = 0
        self._max_trigger_rate: int = 0

    @property
    def x(self) -> np.ndarray:
        """Drive signal for the first galvo, by default the fast axis."""
        return self._x

    @property
    def y(self) -> np.ndarray:
        """Drive signal for the second galvo, by default the slow axis."""
        return self._y

    @property
    def frame_trigger(self) -> np.ndarray:
        """
        A signal equal to `x` and `y` drive signals in length with pulses
        for the triggering of a frame acquisition. Depending on the value of
        `trigger_each_bline`, pulses trigger the start of the pattern or each
        line of a multidimensional scan.
        """
        return self._frame_trigger

    @property
    def line_trigger(self) -> np.ndarray:
        """
        A signal equal in length to `x` and `y` drive signals with pulses for
        the triggering of each point acquisition. Period of the pulse is defined
        by `samples_on` and `samples_off` parameters.
        """
        return self._line_trigger
    
    @property
    def positions(self) -> np.ndarray:
        """
        2D array of point exposure coordinates defined jointly by `x`, `y`, and `image_mask`.
        """
        return self._positions
    
    @property
    def points_in_scan(self) -> int:
        """The number of A-lines or point exposures in the scan. Includes points
        which do not form the desired image. """
        return self._points_in_scan
    
    @property
    def points_in_image(self) -> int:
        """The number of A-lines or point exposures in the resultant image."""
        return self._points_in_image 
    
    @property
    def image_mask(self) -> np.ndarray:
        """
        A boolean array with length `points_in_scan`. True at `points_in_image`
        positions which correspond to the points in the resultant image.
        """
        return self._image_mask

    @property
    def sample_rate(self) -> int:
        """The approximate samples per second that should be generated by the DAC such that lines are triggered at the rate specified."""
        return self._sample_rate

    @property
    def pattern_rate(self) -> int:
        """The approximate rate at which the pattern is to be completed by the scanner."""
        return self._pattern_rate
    
    @property
    def dimensions(self) -> list:
        """
        An array of image dimensions which orient the A-scans with respect to one another. i.e.
        a raster pattern's first and second dimension represent the height and width of the pattern in A-lines.
        """
        raise NotImplementedError()

    def generate(self):
        """Generate the pattern using some number of parameters. The pattern object is unuseable until this is called."""
        raise NotImplementedError()


class Figure8ScanPattern(LineScanPattern):

    def __init__(self, *args, **kwargs):
        super().__init__()

        self.aline_width = None
        self.aline_per_b = None
        self.rotation_rad = None
        self.h = None
        self.aline_per_b = None

        if len(args) > 0:
            self.generate(*args, **kwargs)

    @property
    def dimensions(self):
        return [self.aline_per_b, 2]

    @property
    def points_in_image(self):
        return int(self.aline_per_b * 2)

    def generate(self, aline_width: float, aline_per_b: int, max_trigger_rate: float,
                  samples_on: int = 1, samples_off: int = None, rotation_rad: float = 0, h: float = 5):
        """Generate a Figure-8 pattern: a pattern consisting of two perpendicular B-scans.

        Args:
            aline_width (float): The width of the A-line in units.
            aline_per_b (int): The number of A-lines in each B-line.
            max_trigger_rate (float): The maximum rate in Hz at which to toggle the line trigger signal. This is the primary constraint on imaging rate.
            samples_on (int): Optional. The number of trigger samples to drive high for each exposure.
            samples_off (int): Optional. The number of trigger samples to drive low after each exposure. By default, equivalent to `samples_on`.
            rotation_rad (float): Optional. Rotates the scan. Default 0.
            h (float): Optional. 1 / H is the proportion of the B-line length to exclude from the scan for flyback. Default 5.
        """

        self.aline_width = aline_width
        self.aline_per_b = aline_per_b
        self.max_trigger_rate = max_trigger_rate
        self.samples_on = samples_on
        self.samples_off = samples_off
        self.rotation_rad = rotation_rad
        self.h = h

        samples_on = int(samples_on)

        if samples_off is None:
            samples_off = int(samples_on)
        else:
            samples_off = int(samples_off)

        self.aline_width = aline_width
        self.aline_per_b = int(aline_per_b)

        # Build B-line trigger
        btrig = np.tile(np.append(np.ones(samples_on), np.zeros(samples_off)), self.aline_per_b)

        # TODO parameterize these, especially H
        G = np.pi / 2.5  # Cylical point at which B-scans begin in the figure
        A = 1 / 1.6289944852363252  # Factor such that B-scans given a G are normalized in length
        AY = 1.223251  # Stretch factor along second axis to achieve perpendicular B-scans
        H = h  # 1 / H is the proportion of the B-line length used for flyback in samples

        def fig8(t):
            x = A * np.cos(t)
            y = A * np.cos(t) * np.sin(t) * AY
            return [x, y]

        t_fb0 = _sigmoidspace(-G, G, int(1 / H * len(btrig)), b=0.2)
        [x_fb0, y_fb0] = fig8(t_fb0)

        t_fb1 = _sigmoidspace(-G + np.pi, G + np.pi, int(1 / H * len(btrig)), b=0.2)
        [x_fb1, y_fb1] = fig8(t_fb1)

        x_b1 = np.linspace(x_fb0[-1], x_fb1[0], len(btrig))
        y_b1 = np.linspace(y_fb1[-1], y_fb0[0], len(btrig))

        x_b0 = np.linspace(x_fb1[-1], x_fb0[0], len(btrig))
        x_b0 = np.linspace(x_fb1[-1], x_fb0[0], len(btrig))
        y_b0 = np.linspace(y_fb1[-1], y_fb0[0], len(btrig))

        x = np.concatenate([x_b0[0:-1], x_fb0[0:-1], x_b1[0:-1], x_fb1[0:-1]]) * self.aline_width
        y = np.concatenate([y_b0[0:-1], y_fb0[0:-1], y_b1[0:-1], y_fb1[0:-1]]) * self.aline_width

        x, y = _rotfunc(x, y, rotation_rad)

        fb = np.zeros(len(t_fb0))
        line_trigger = np.concatenate([btrig[0:-1], fb[0:-1], btrig[0:-1], fb[0:-1]])
        frame_start = np.zeros(len(line_trigger))
        frame_start[0:samples_on] = 1

        self._x = x
        self._y = y
        self._line_trigger = line_trigger
        self._frame_trigger = frame_start

        period_samples = samples_on + samples_off
        self._sample_rate = self.max_trigger_rate * period_samples

        self._pattern_rate = 1 / ((1 / self._sample_rate) * len(line_trigger))


class RoseScanPattern(LineScanPattern):

    def __init__(self, *args, **kwargs):
        super().__init__()

        self.p = None
        self.aline_width = None
        self.aline_per_b = None
        self.max_trigger_rate = None
        self.samples_on = None
        self.samples_off = None
        self.rotation_rad = None
        self.h = None

        if len(args) > 0:
            self.generate(*args, **kwargs)

    @property
    def dimensions(self):
        return [self.aline_per_b, self._p]

    @property
    def points_in_image(self):
        return int(self.aline_per_b * self._p)

    def generate(self, p: int, aline_width: float, aline_per_b: int, max_trigger_rate: float,
                  samples_on: int = 1, samples_off: int = None, rotation_rad: float = 0, h: float = 6):
        """Generate a Rose pattern: a pattern consisting of any number of orthogonal B-scans.

        For a rose pattern with 2 B-scans, use a `Figure8ScanPattern`.

        Args:
            p (int): The number of B-scans the pattern should have.
            aline_width (float): The width of the A-line in units.
            aline_per_b (int): The number of A-lines in each B-line.
            max_trigger_rate (float): The maximum rate in Hz at which to toggle the line trigger signal. This is the constraint on imaging rate.
            samples_on (int): Optional. The number of trigger samples to drive high for each exposure. Default 2.
            samples_off (int): Optional. The number of trigger samples to drive low after each exposure. By default, equivalent to `samples_on`.
            rotation_rad (float): Optional. Rotates the scan. Default 0.
            h (float): Optional. 1 / H is the proportion of the B-line length to exclude from the scan for flyback. Default 6.
        """

        self.p = p
        self.aline_width = aline_width
        self.max_trigger_rate = max_trigger_rate
        self.samples_on = samples_on
        self.samples_off = samples_off
        self.rotation_rad = rotation_rad
        self.h = h

        p = int(p)
        if p % 2 == 0:
            raise ValueError("'p' must be an odd integer")
        else:
            k = p
            p0 = 0
            period = np.pi
        
        samples_on = int(samples_on)

        if samples_off is None:
            samples_off = int(samples_on)
        else:
            samples_off = int(samples_off)

        self.aline_width = aline_width
        self.aline_per_b = int(aline_per_b)
        self._p = int(p)

        b_frac = 0.2
        H = h  # 1 / H is the proportion of the B-line length used for flyback in samples

        period_samples = samples_on + samples_off
        self._sample_rate = self.max_trigger_rate * period_samples

        b_trig = np.tile(np.append(np.ones(samples_on), np.zeros(samples_off)), self.aline_per_b)

        t = np.array([])
        lt = np.array([])

        period_p = period / k

        period_b = period_p * b_frac
        period_fb = period_p * (1 - b_frac)

        for i in range(p):  # Generate each petal

            p_start = p0 + i * period_p

            t_fb = _sigmoidspace(p_start, p_start + period_fb, int(1 / H * len(b_trig)), b=1)
            t_b = np.linspace(p_start + period_fb, p_start + period_fb + period_b, len(b_trig))

            t_p = np.concatenate([t_fb, t_b])

            trig_p = np.concatenate([np.zeros(len(t_fb)), b_trig])

            def xfunc(ts):
                return aline_width * np.cos(p * ts) * np.cos(ts)

            def yfunc(ts):
                return aline_width * np.cos(p * ts) * np.sin(ts)

            t = np.append(t, t_p)
            lt = np.append(lt, trig_p)

        t -= period_fb / 2

        x = xfunc(t)
        y = yfunc(t)

        x, y = _rotfunc(x, y, rotation_rad)

        self._x = x
        self._y = y
        self._line_trigger = lt

        ft = np.zeros(len(lt))
        ft[0:samples_on] = 1
        self._frame_trigger = ft
        self._pattern_rate = 1 / ((1 / self._sample_rate) * len(x))


class CircleScanPattern(LineScanPattern):

    def __init__(self, *args, **kwargs):
    
        super().__init__()
        
        self.alines = None
        self.diameter = None
        self.max_trigger_rate = None
        self.rotation_rad = None
        self.samples_on = None
        self.samples_off = None
        
        if len(args) > 0:
            self.generate(*args, **kwargs)
            
    @property
    def dimensions(self):
        """Returns [number of a-lines, number of b-lines]."""
        return [self.alines, 1]        

    def generate(self,
                  alines: int,
                  diameter: float,
                  max_trigger_rate: float, 
                  rotation_rad: float = 0.0,
                  samples_on: int = 1,
                  samples_off: int = None
                  ):
        """Generate a circle pattern of any number of A-lines.

        For a rose pattern with 2 B-scans, use a `Figure8ScanPattern`.

        Args:
            alines (int): The number of A-lines in the circle.
            diameter (float): The diameter of the circle.
            max_trigger_rate (float): The maximum rate in Hz at which to toggle the line trigger signal. This is the constraint on imaging rate.
            samples_on (int): Optional. The number of trigger samples to drive high for each exposure. Default 2.
            samples_off (int): Optional. The number of trigger samples to drive low after each exposure. By default, equivalent to `samples_on`.
            rotation_rad (float): Optional. Rotates the scan. Default 0.
        """
        self.alines = int(alines)
        self.diameter = diameter
        self.rotation_rad = rotation_rad
        
        self._points_in_scan = self.alines
        
        self.samples_on = int(samples_on)
        if samples_off is None:
            self.samples_off = int(self.samples_on)
        else:
            self.samples_off = int(samples_off)

        period_samples = self.samples_on + self.samples_off
        
        self.max_trigger_rate = max_trigger_rate
        self._sample_rate = self.max_trigger_rate * period_samples

        single_position_trig = np.concatenate([np.ones(self.samples_on), np.zeros(self.samples_off)])
        trigger = np.tile(single_position_trig, self.alines)
        
        self.radius = self.diameter / 2
        t = np.linspace(0, 2*np.pi, len(trigger))
        
        x = np.cos(t) * self.radius
        y = np.sin(t) * self.radius
        
        self._x, self._y = _rotfunc(x, y, self.rotation_rad)
        self._line_trigger = trigger
        self._points_in_image = self._points_in_scan
        self._image_mask = np.ones(self._points_in_scan).astype(bool)
        ft = np.zeros(len(self._x))
        ft[0:self.samples_on] = 1
        self._frame_trigger = ft


class RasterScanPattern(LineScanPattern):

    def __init__(self, *args, **kwargs):

        super().__init__()

        self.alines = 0
        self.blines = 0
        self.max_trigger_rate = 0
        self.exposure_fraction = 0
        self.flyback_duty = 0
        self.fov = (0, 0)
        self.samples_park = 0
        self.samples_step = 0
        self.rotation_rad = 0
        self.fast_axis_step = 0
        self.slow_axis_step = 0
        self.aline_repeat = 0
        self.bline_repeat = 0
        self.trigger_blines = 0
        self.true_exposure_fraction = 0
        self._image_mask = np.array([]).astype(bool)

        if len(args) > 0:
            self.generate(*args, **kwargs)

    @property
    def dimensions(self) -> (int, int):
        """Returns (number of a-lines, number of b-lines) excluding repeats."""
        return self.alines, self.blines

    def generate(self, alines: int,
                 blines: int,
                 max_trigger_rate: float,  
                 fov: list = [1, 1],
                 flyback_duty: float = 0.2,
                 exposure_fraction: float = 0.7,
                 samples_on: int = 1,
                 samples_off: int = None,
                 samples_settle: int = 1,
                 samples_step: int = 1,
                 rotation_rad: float = 0,
                 fast_axis_step: bool = False,
                 slow_axis_step: bool = False,
                 aline_repeat: int = 1,
                 bline_repeat: int = 1,
                 trigger_blines: bool = False,
                 bidirectional: bool = False
                 ):
        """Generate a raster pattern.

        This raster pattern implementation supports stepped slow and fast axis scanning,
        repeated A-lines and then repeated B-lines, bidirectional scanning, rotation, and
        variable flyback and settling time.

        Args:
            alines (int): The number of A-lines in each B-line.
            blines (int): The number of B-lines.
            max_trigger_rate (float): The maximum rate at which to toggle the line trigger signal. This is the constraint on imaging rate.
            fov (list): Optional. If provided, scales the height and width of the pattern by these values. Otherwise the scan region is 1.0-by-1.0
            exposure_fraction (float): Optional. The fraction of the scan sweep to use for exposure. Default 0.8.
            flyback_duty (float): Optional. The amount of time to dedicate to flyback as a proportion of the duration of the scan sweep duration. Default 0.1.
            samples_on (int): Optional. The number of trigger samples to drive high for each exposure. Default 2.
            samples_off (int): Optional. The number of trigger samples to drive low after each exposure. By default, equivalent to `samples_on`.
            samples_settle (int): Optional. The number of samples to wait after holding galvos constant before triggering a repeated A-line. Default 1.
            samples_step (int): Optional. The number of samples to interpolate between step positions. Default 1.
            rotation_rad (float): Optional. Rotates the scan. Default 0.
            fast_axis_step (bool): Optional. If True, the fast axis is stepped to better approximate a square FOV. Always True if A-line repeat > 1. Default False.
            slow_axis_step (bool): Optional. If True, the slow axis is stepped to better approximate a square FOV. Always True if B-line repeat > 1. Default False.
            aline_repeat (int): Optional. If > 1, the number of A-lines to scan at each position. Default 1.
            bline_repeat (int): Optional. If > 1, the number of B-lines to scan before advancing the slow axis. Default 1.
            trigger_blines (bool): Optional. If True, the frame trigger signal goes high for each B-line rather than at the start of the frame.
            bidirectional (bool): Optional. If True, reversed scan sweeps replace flyback. Default False.
        """

        self.alines = int(alines)
        self.blines = int(blines)

        self.aline_repeat = int(aline_repeat)
        self.bline_repeat = int(bline_repeat)
        
        self._points_in_image = self.alines * self.aline_repeat * self.blines * self.bline_repeat
        
        self.fast_axis_step = fast_axis_step
        self.slow_axis_step = slow_axis_step

        if aline_repeat < 2:
            self.aline_repeat = 1

        if bline_repeat < 2:
            self.bline_repeat = 1
        else:
            # B-line repeat must use stepped slow axis
            self.slow_axis_step = True

        self.bidirectional = bool(bidirectional)
        if self.bidirectional:  # Lots of settings are incompatible with bidirectional=True
            if self.aline_repeat != 1:
                self.aline_repeat = 1
                warnings.warn('Bidirectional repeated A-lines not supported. Using aline_repeat=1')
            if self.fast_axis_step:
                self.fast_axis_step = False
                warnings.warn('Bidirectional scans with stepped fast axis are not supported. Using fast_axis_step=False')
            if self.bline_repeat != 1:
                self.bline_repeat = 1
                warnings.warn('Bidirectional repeated B-lines not supported. Using bline_repeat=1')

        self.fov = fov

        self._samples_on = int(samples_on)

        self.rotation_rad = rotation_rad

        if samples_off is None:
            self._samples_off = int(samples_on)
        else:
            self._samples_off = int(samples_off)

        single_aline_trig = np.concatenate([np.ones(self._samples_on), np.zeros(self._samples_off)])

        period_samples = len(single_aline_trig)

        self.max_trigger_rate = max_trigger_rate
        self._sample_rate = self.max_trigger_rate * period_samples

        # Round these to the trigger period
        self.samples_settle = period_samples * round(samples_settle / period_samples)
        self.samples_step = period_samples * round(samples_step / period_samples)

        if self.fast_axis_step:  # If fast axis is stepped, trigger includes a settling time and step time
            single_position_trig = np.concatenate([np.zeros(self.samples_settle),
                                                np.tile(single_aline_trig, self.aline_repeat),
                                                np.zeros(self.samples_step)])
        else:
            single_position_trig = np.tile(single_aline_trig, self.aline_repeat)

        aline_trigs = np.concatenate([np.zeros(period_samples), np.tile(single_position_trig, self.alines)])  # Each line needs this many exposures

        self.exposure_fraction = exposure_fraction
        if not self.fast_axis_step:
            # Exposures are padded such that they comprise exposure_fraction of the sweep
            pad_size = period_samples * round(int((len(aline_trigs) * (1 - self.exposure_fraction)) / 2) / period_samples)
            bline_pad = np.zeros(pad_size)
            self.true_exposure_fraction = 1 - (2 * len(bline_pad)) / len(aline_trigs)
            bline_exp = np.concatenate([bline_pad, aline_trigs, bline_pad])
            bline_start_offset = len(bline_pad) + period_samples
        else:
            bline_exp = aline_trigs
            bline_start_offset = period_samples

        # Flyback duty is the percentage of the imaging sweep to spend on flyback
        self.flyback_duty = flyback_duty
        flyback_size = period_samples * round(int(len(bline_exp) * self.flyback_duty) / period_samples)
        flyback_pad = np.zeros(flyback_size)
        if not self.bidirectional: # No flyback if the scan is bidirectional            
            bline_trig = np.concatenate([bline_exp, flyback_pad])
        else:
            bline_trig = bline_exp

        # Apply B-line repeat
        bline_trig = np.tile(bline_trig, self.bline_repeat)

        if not self.fast_axis_step:

            # True peak of sweep is greater than desired field of view to compensate for the exposure fraction
            fast_axis_peak = (fov[0] + fov[0] * (1 - self.true_exposure_fraction)) / 2

            # Fast axis is a simple saw
            fast_axis_scan = np.linspace(-fast_axis_peak, fast_axis_peak, len(bline_exp))
            fast_axis_flyback = np.linspace(fast_axis_peak, -fast_axis_peak, len(flyback_pad) + 2)[1:-1]

        else:  # Fast axis stepped

            positions = np.linspace(-fov[0] / 2, fov[0] / 2, self.alines)  # Exposure fraction is ignored
            fast_axis_scan = positions[0] * np.ones(period_samples)  # Begin at first spot

            for i in range(len(positions) - 1):
                fast_axis_scan = np.append(fast_axis_scan, positions[i] * np.ones(len(single_position_trig) - self.samples_step))  # Scan spot
                fast_axis_scan = np.append(fast_axis_scan, np.linspace(positions[i], positions[i + 1], self.samples_step + 2)[1:-1])  # Transit to next
            fast_axis_scan = np.append(fast_axis_scan, positions[-1] * np.ones(len(single_position_trig)))  # Scan final spot
            fast_axis_flyback = np.linspace(positions[-1], positions[0], len(flyback_pad) + 2)[1:-1]

        # Apply B-line repeat
        if not self.bidirectional:
            bline_scan = np.tile(np.concatenate([fast_axis_scan, fast_axis_flyback]), self.bline_repeat)
            x = np.tile(bline_scan, self.blines)
            line_trig = np.tile(bline_trig, self.blines)
        else:
            remainder = self.blines % 2
            half = int((self.blines - remainder) / 2)
            bline_scan2 = np.concatenate([fast_axis_scan, fast_axis_scan[::-1]])
            bline_trig2 = np.concatenate([bline_trig, bline_trig[::-1]])
            x = np.tile(bline_scan2, half)
            line_trig = np.tile(bline_trig2, half)
            if bool(remainder):  # Odd-numered scans do not start where they left off
                x = np.concatenate([x, fast_axis_scan])
                line_trig = np.concatenate([line_trig, bline_trig])

        self.trigger_blines = trigger_blines

        if self.trigger_blines:
            f = np.zeros(len(bline_trig))
            f[bline_start_offset:bline_start_offset + 2 * samples_on] = 1
            frame_trig = np.tile(f, self.blines)
        else:
            frame_trig = np.zeros(len(line_trig))
            frame_trig[0:2 * samples_on] = 1

        # Just zero x if it's actually a 1x1 "scan"!
        if not self.alines > 1:
            x = np.zeros(len(x))

        if self.blines > 1:

            slow_axis_flyback = np.linspace(-fov[1] / 2, fov[1] / 2, len(flyback_pad) + 2)[1:-1]

            if not self.bidirectional:

                if self.slow_axis_step is False:
                    slow_axis_scan = np.linspace(fov[1] / 2, -fov[1] / 2, len(x) - len(fast_axis_flyback))
                else:
                    slow_axis_scan = np.array([])
                    positions = np.linspace(fov[1] / 2, -fov[1] / 2, self.blines)
                    if self.bline_repeat > 1:
                        scan_duration = self.bline_repeat * (len(fast_axis_scan) + len(fast_axis_flyback)) - len(fast_axis_flyback)
                    else:
                        scan_duration = len(fast_axis_scan)
                    for i in range(len(positions) - 1):
                        slow_axis_scan = np.append(slow_axis_scan, positions[i] * np.ones(scan_duration))  # Scan spot
                        slow_axis_scan = np.append(slow_axis_scan, np.linspace(positions[i], positions[i + 1], len(fast_axis_flyback) + 2)[1:-1])  # Transit to next
                    slow_axis_scan = np.append(slow_axis_scan, positions[-1] * np.ones(scan_duration))  # Scan final spot
                y = np.concatenate([slow_axis_scan, slow_axis_flyback])

            else:  # Bidirectional slow axis has different transit and flyback model based on exposure_fraction

                if self.slow_axis_step is False:
                    slow_axis_scan = np.linspace(fov[1] / 2, -fov[1] / 2, len(x))
                else:
                    slow_axis_scan = np.array([])
                    positions = np.linspace(fov[1] / 2, -fov[1] / 2, self.blines)
                    slow_axis_scan = np.append(slow_axis_scan, positions[0] * np.ones(len(aline_trigs) + len(bline_pad)))  # Scan first spot
                    slow_axis_scan = np.append(slow_axis_scan, np.linspace(positions[0], positions[1], 2 * len(bline_pad) + 2)[1:-1])  # Transit to second spot
                    for i in range(1, len(positions) - 1):
                        slow_axis_scan = np.append(slow_axis_scan, positions[i] * np.ones(len(aline_trigs)))  # Scan spot
                        slow_axis_scan = np.append(slow_axis_scan, np.linspace(positions[i], positions[i + 1], 2 * len(bline_pad) + 2)[1:-1])  # Transit to next
                    slow_axis_scan = np.append(slow_axis_scan, positions[-1] * np.ones(len(aline_trigs) + len(bline_pad)))  # Scan final spot
                y = np.concatenate([slow_axis_scan, slow_axis_flyback])
                if self.blines % 2 > 0:
                    x = np.concatenate([x, np.linspace(x[-1], x[0], len(slow_axis_flyback))])
                else:
                    x = np.concatenate([x, np.ones(len(slow_axis_flyback)) * x[-1]])
                line_trig = np.concatenate([line_trig, np.zeros(len(slow_axis_flyback))])
                frame_trig = np.concatenate([frame_trig, np.zeros(len(slow_axis_flyback))])
        else:
            # If only 1 B-line, don't scan second axis at all
            y = np.zeros(len(x))

        # Roll exposures so first sample is always an imaging exposure
        self._x = np.roll(x, -bline_start_offset)
        self._y = np.roll(y, -bline_start_offset)
        line_trig = np.roll(line_trig, -bline_start_offset)
        self._frame_trigger = np.roll(frame_trig, -bline_start_offset)

        assert len(self._x) == len(self._y), 'Scan signals generated are not equal in length'
        assert len(self._x) == len(line_trig)
        assert len(self._x) == len(self._frame_trigger)

        assert len(self._x) % period_samples == 0, 'Generated pattern is not divisible by the exposure period'

        points_in_scan = int(len(line_trig) / period_samples)
        mask = np.zeros(points_in_scan)

        self._x, self._y = _rotfunc(self._x, self._y, rotation_rad)
        
        pos = []
        # Build mask array that is high for triggers which form the image
        for i in range(points_in_scan):
            for j in range(period_samples):
                if line_trig[i * period_samples + j] > 0:
                    mask[i] = 1
                    pos.append([self._x[i * period_samples + j], self._y[i * period_samples + j]])
                    for k in range(self.aline_repeat - 1):
                        pos.append([self._x[i * period_samples + j + period_samples * k], self._y[i * period_samples + j + period_samples * k]])
                    break
        self._positions = np.array(pos)

        self._points_in_scan = points_in_scan
        self._image_mask = mask
        
        self._points_in_scan = points_in_scan
        self._line_trigger = np.tile(single_aline_trig, self._points_in_scan)

        self._pattern_rate = 1 / (len(self._x) * (1 / self._sample_rate))


if __name__ == '__main__':
    
    DPI = 96
    FOLDER = 'img/'
    
    import os
    import matplotlib
    import matplotlib.pyplot as plt

    print('Generating demo figures...')
    
    matplotlib.rcParams.update({'font.size': 16, 'font.family': 'monospace'})
    
    patterns = [
        RasterScanPattern(16, 16, 76000, samples_on=1, samples_off=10),
        RasterScanPattern(16, 16, 76000, samples_on=1, samples_off=10, fast_axis_step=True, slow_axis_step=True),
        RasterScanPattern(16, 16, 76000, samples_on=1, samples_off=10, bline_repeat=2),
        RasterScanPattern(16, 16, 76000, samples_on=1, samples_off=10, aline_repeat=2),
        RasterScanPattern(16, 16, 76000, samples_on=1, samples_off=10, fov=[1.5, 4.5], fast_axis_step=True, slow_axis_step=True),
        RasterScanPattern(16, 16, 76000, samples_on=1, samples_off=10, fast_axis_step=True, slow_axis_step=True, rotation_rad=np.pi/4),
        RasterScanPattern(15, 15, 76000, samples_on=1, samples_off=10, bidirectional=True, slow_axis_step=True),
        RasterScanPattern(15, 1, 76000, samples_on=1, samples_off=10),
        RasterScanPattern(15, 1, 76000, samples_on=1, samples_off=10, bidirectional=True, rotation_rad=np.pi/8),
        # CircleScanPattern(64, 1.0, 76000, samples_on=1),
        # Figure8ScanPattern(1, 16, 76000, samples_on=1, samples_off=10),
        # RoseScanPattern(3, 1, 16, 76000, samples_on=1, samples_off=10),
        # RoseScanPattern(5, 1, 16, 76000, samples_on=1, samples_off=10),
        ]
    
    titles = [
        'Classic raster',
        'Stepped raster',
        'B-line repeated raster',
        'A-line repeated raster',
        'Rectangular raster',
        'Rotated raster',
        'Bidirectional raster',
        'Line scan',
        'Bidirectional line scan',
        'Circle',
        'Figure-8',
        'Rose p=3',
        'Rose p=5',
        ]
    
    fnames = [
        'raster.png',
        'rasterstep.png',
        'rasterbrpt.png',
        'rasterarpt.png',
        'rectraster.png',
        'rotraster.png',
        'biraster.png',
        'line.png',
        'biline.png',
        'circle.png',
        'fig8.png',
        'rose3.png',
        'rose5.png',
        ]
    
    for i, (pattern, title, fname) in enumerate(zip(patterns, titles, fnames)):
    
        fig = plt.figure(i, constrained_layout=False, figsize=(4, 4))
    
        gs = fig.add_gridspec(nrows=3, ncols=1, hspace=0.01)
        ax1 = fig.add_subplot(gs[0, 0])
        ax2 = fig.add_subplot(gs[1:, 0])
        
        ax1.set_title(title, pad=2)
        ax1.plot(pattern.x)
        ax1.plot(pattern.y)
        ax1.stem(np.repeat(pattern.image_mask.astype(int), int(len(pattern.x) / len(pattern.image_mask))),
                 markerfmt='None', linefmt='black')
        ax1.set_frame_on(False)
        ax1.set_xticks([])
        ax1.set_yticks([])
    
        ax2.set_aspect('equal')
        ax2.plot(np.tile(pattern.x, 2), np.tile(pattern.y, 2), color='lightgrey')
        ax2.scatter(pattern.positions[:, 0], pattern.positions[:, 1],
                    marker='.', color='black', zorder=10)
        ax2.set_frame_on(False)
        ax2.set_xticks([])
        ax2.set_yticks([])
    
        fname = FOLDER + '/' + fname
        
        try:
            os.remove(fname)
        except FileNotFoundError:
            pass
        
        fig.savefig(fname, dpi=DPI)
    
        plt.close(i)