from ......Internal.Core import Core
from ......Internal.CommandsGroup import CommandsGroup
from ......Internal import Conversions
from ...... import enums
from ...... import repcap


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class Mcs:
	"""Mcs commands group definition. 1 total commands, 0 Sub-groups, 1 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._base = CommandsGroup("mcs", core, parent)

	def set(self, mcs: enums.McsIndex, station=repcap.Station.Default) -> None:
		"""SCPI: CONFigure:WLAN:SIGNaling<instance>:STA<s>:CONNection:HETF:MCS \n
		Snippet: driver.configure.sta.connection.hetf.mcs.set(mcs = enums.McsIndex.MCS, station = repcap.Station.Default) \n
		Specifies the modulation and coding scheme (MCS) used by the HE TB PPDU. \n
			:param mcs: MCS | MCS1 | MCS2 | MCS3 | MCS4 | MCS5 | MCS6 | MCS7 | MCS8 | MCS9 | MCS10 | MCS11 MCS, MCS1,...,MCS11: MCS 0 to MCS 11
			:param station: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Sta')"""
		param = Conversions.enum_scalar_to_str(mcs, enums.McsIndex)
		station_cmd_val = self._base.get_repcap_cmd_value(station, repcap.Station)
		self._core.io.write(f'CONFigure:WLAN:SIGNaling<Instance>:STA{station_cmd_val}:CONNection:HETF:MCS {param}')

	# noinspection PyTypeChecker
	def get(self, station=repcap.Station.Default) -> enums.McsIndex:
		"""SCPI: CONFigure:WLAN:SIGNaling<instance>:STA<s>:CONNection:HETF:MCS \n
		Snippet: value: enums.McsIndex = driver.configure.sta.connection.hetf.mcs.get(station = repcap.Station.Default) \n
		Specifies the modulation and coding scheme (MCS) used by the HE TB PPDU. \n
			:param station: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Sta')
			:return: mcs: MCS | MCS1 | MCS2 | MCS3 | MCS4 | MCS5 | MCS6 | MCS7 | MCS8 | MCS9 | MCS10 | MCS11 MCS, MCS1,...,MCS11: MCS 0 to MCS 11"""
		station_cmd_val = self._base.get_repcap_cmd_value(station, repcap.Station)
		response = self._core.io.query_str(f'CONFigure:WLAN:SIGNaling<Instance>:STA{station_cmd_val}:CONNection:HETF:MCS?')
		return Conversions.str_to_scalar_enum(response, enums.McsIndex)
