"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Operator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdk8s_1 = require("cdk8s");
/**
 * A CDK8s app which allows implementing Kubernetes operators using CDK8s constructs.
 */
class Operator extends cdk8s_1.App {
    constructor(props = {}) {
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdk8s'));
        super({ outdir });
        this.providers = [];
        this.inputFile = props.inputFile ?? process.argv[2] ?? '/dev/stdin';
        this.outputFile = props.outputFile;
    }
    /**
     * Adds a custom resource provider to this operator.
     * @param provider The provider to add
     */
    addProvider(provider) {
        this.providers.push(provider);
    }
    /**
     * Reads a Kubernetes manifest in JSON format from STDIN or the file specified
     * as the first positional command-line argument. This manifest is expected to
     * include a single Kubernetes resource. Then, we match `apiVersion` and
     * `kind` to one of the registered providers and if we do, we invoke
     * `apply()`, passing it the `spec` of the input manifest and a chart as a
     * scope. The chart is then synthesized and the output manifest is written to
     * STDOUT.
     */
    synth() {
        const input = JSON.parse(fs.readFileSync(this.inputFile, 'utf-8'));
        let write;
        if (this.outputFile) {
            const outfile = this.outputFile;
            write = (data) => fs.writeFileSync(outfile, data);
        }
        else {
            write = (data) => process.stdout.write(data);
        }
        if (typeof (input) !== 'object') {
            throw new Error('input must be a single kubernetes resource');
        }
        const provider = this.findProvider(input);
        const name = input.metadata?.name;
        if (!name) {
            throw new Error('"metadata.name" must be defined');
        }
        const namespace = input.metadata?.namespace;
        // TODO: namespace
        const spec = input.spec ?? {};
        const chart = new cdk8s_1.Chart(this, name, { namespace });
        console.error(`Synthesizing ${input.kind}.${input.apiVersion}`);
        provider.handler.apply(chart, name, spec);
        super.synth();
        for (const file of fs.readdirSync(this.outdir)) {
            const filepath = path.join(this.outdir, file);
            const manifest = fs.readFileSync(filepath);
            write(manifest);
        }
    }
    findProvider(input) {
        const { apiVersion, kind } = input;
        if (!apiVersion) {
            throw new Error('"apiVersion" is required');
        }
        if (!kind) {
            throw new Error('"kind" is required');
        }
        for (const p of this.providers) {
            if (p.apiVersion === apiVersion && p.kind === kind) {
                return p;
            }
        }
        throw new Error(`No custom resource provider found for ${kind}.${apiVersion}`);
    }
}
exports.Operator = Operator;
_a = JSII_RTTI_SYMBOL_1;
Operator[_a] = { fqn: "cdk8s-operator.Operator", version: "0.1.346" };
//# sourceMappingURL=data:application/json;base64,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