from typing import Optional

from enum import Enum

from pydantic import BaseModel, Field, Extra

from .oauth_flows import OAuthFlows


class SecuritySchemeType(str, Enum):
    userPassword = 'userPassword'
    apiKey = 'apiKey'
    X509 = 'X509'
    symmetricEncryption = 'symmetricEncryption'
    asymmetricEncryption = 'asymmetricEncryption'
    httpApiKey = 'httpApiKey'
    http = 'http'
    oauth2 = 'oauth2'
    openIdConnect = 'openIdConnect'
    plain = 'plain'
    scramSha256 = 'scramSha256'
    scramSha512 = 'scramSha512'
    gssapi = 'gssapi'


class SecuritySchemeLocation(str, Enum):
    user = 'user'
    password = 'password'
    apiKey = 'apiKey'
    query = 'query'
    header = 'header'
    cookie = 'cookie'
    httpApiKey = 'httpApiKey'


class SecurityScheme(BaseModel):
    """
    Defines a security scheme that can be used by the operations. Supported schemes are:
        User/Password.
        API key (either as user or as password).
        X.509 certificate.
        End-to-end encryption (either symmetric or asymmetric).
        HTTP authentication.
        HTTP API key.
        OAuth2's common flows (Implicit, Resource Owner Protected Credentials, Client
            Credentials and Authorization Code) as defined in RFC6749.
        OpenID Connect Discovery.
        SASL (Simple Authentication and Security Layer) as defined in RFC4422.
    """

    param_type: SecuritySchemeType = Field(None, alias="type")
    """
    **REQUIRED**. The type of the security scheme. Valid values are "userPassword",
    "apiKey", "X509", "symmetricEncryption", "asymmetricEncryption", "httpApiKey",
    "http", "oauth2", "openIdConnect", "plain", "scramSha256", "scramSha512", and "gssapi".
    """

    description: Optional[str] = None
    """
    A short description for security scheme. CommonMark syntax MAY be used for rich
    text representation.
    """

    name: Optional[str] = None
    """
    The name of the header, query or cookie parameter to be used.
    """

    param_in: Optional[SecuritySchemeLocation] = Field(None, alias="in")
    """
    The location of the API key. Valid values are "user" and "password"
    for apiKey and "query", "header" or "cookie" for httpApiKey.
    """

    scheme: Optional[str] = None
    """
    The name of the HTTP Authorization scheme to be used in the
    Authorization header as defined in RFC7235.
    """

    bearerFormat: Optional[str] = None
    """
    A hint to the client to identify how the bearer token is formatted. Bearer tokens
    are usually generated by an authorization server, so this information is primarily
    for documentation purposes.
    """

    flows: Optional[OAuthFlows] = None
    """
    An object containing configuration information for the flow types
    supported.
    """

    openIdConnectUrl: Optional[str] = None
    """
    OpenId Connect URL to discover OAuth2 configuration values. This MUST
    be in the form of a URL.
    """


    class Config:
        extra = Extra.forbid
        schema_extra = {
            "examples": [
                {
                   "type": "userPassword"
                },
                {
                    "apiKey": "user"
                },
                {
                    "type": "http",
                    "scheme": "basic"
                },
                {
                    "type": "http",
                    "scheme": "bearer",
                    "bearerFormat": "JWT"
                },
                {
                    "type": "oauth2",
                    "flows": {
                        "implicit": {
                        "authorizationUrl": "https://example.com/api/oauth/dialog",
                        "scopes": {
                            "write:pets": "modify pets in your account",
                            "read:pets": "read your pets"
                            }
                        }
                    }
                }
            ]
        }
