"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateStackSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const stack_status_1 = require("../../domain/stack-status");
const string_variable_1 = require("../../interface/variables/string-variable");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const delete_stack_step_1 = require("../../parent-steps/automation/delete-stack-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const wait_for_and_assert_1 = require("../../parent-steps/automation/sub-steps/wait-for-and-assert");
const prune_and_transform_record_1 = require("../../utils/prune-and-transform-record");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
const aws_service_1 = require("../../domain/aws-service");
/**
 * AutomationStep implementation for aws:createStack
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createstack.html
 */
class CreateStackSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:createStack';
        this.creatStackStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        this.verifyUrlProps(inputs);
        const stackName = this.creatStackStep.stackName.resolveToString(inputs);
        console.log(`CreateStack: Checking that ${stackName} is ready to be created`);
        this.preVerifyStackSubStep(stackName);
        console.log(`CreateStack: Creating ${stackName}`);
        const stackId = this.createStack(inputs);
        console.log(`CreateStack: Waiting for ${stackName} completion`);
        this.waitForStackCompletion(stackName);
        console.log('CreateStack: Stack is created');
        const stack = this.getStackState(stackName);
        return {
            StackId: stackId,
            StackStatus: stack?.StackStatus,
            StackStatusReason: stack?.StackStatusReason ?? '',
        };
    }
    verifyUrlProps(inputs) {
        return this.verifyUrlProp(this.creatStackStep.templateUrl, inputs)
            && this.verifyUrlProp(this.creatStackStep.stackPolicyUrl, inputs);
    }
    verifyUrlProp(urlToVerify, inputs) {
        return urlToVerify === undefined || new url_1.URL(urlToVerify.resolveToString(inputs));
    }
    preVerifyStackSubStep(stackName) {
        while (!this.preVerifyStack(stackName)) {
            new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
        }
    }
    preVerifyStack(stackName) {
        const state = this.getStackState(stackName);
        if (this.isStackOperationInProgress(state)) {
            return false;
        }
        if (state?.StackStatus !== 'ROLLBACK_COMPLETE') {
            return true;
        }
        console.log(`Deleting stack ${stackName} because we are retrying a create on an existing stack in ROLLBACK_COMPLETE state.`);
        new automation_step_simulation_1.AutomationStepSimulation(new delete_stack_step_1.DeleteStackStep(new aws_cdk_lib_1.Stack(), 'deleteStack', {
            stackNameVariable: new string_variable_1.HardCodedString(stackName),
        }), this.props).invoke({});
        return false;
    }
    isStackOperationInProgress(state) {
        return state !== null && CreateStackSimulation.InProgressStatuses.includes(stack_status_1.StackStatus[state.StackStatus]);
    }
    createStack(inputs) {
        const apiParams = prune_and_transform_record_1.pruneAndTransformRecord({
            StackName: this.creatStackStep.stackName,
            Capabilities: this.creatStackStep.capabilities,
            ClientRequestToken: this.creatStackStep.clientRequestToken,
            NotificationARNs: this.creatStackStep.notificationARNs,
            OnFailure: this.creatStackStep.onStackFailure,
            ResourceTypes: this.creatStackStep.resourceTypes,
            RoleARN: this.creatStackStep.roleArn,
            StackPolicyBody: this.creatStackStep.stackPolicyBody,
            StackPolicyURL: this.creatStackStep.stackPolicyUrl,
            Tags: this.creatStackStep.tags,
            TemplateBody: this.creatStackStep.templateBody,
            TemplateURL: this.creatStackStep.templateUrl,
            TimeoutInMinutes: this.creatStackStep.timeoutInMinutes,
        }, x => x.resolve(inputs));
        const parameters = this.resolveSecureStringsInParametersInput(this.creatStackStep.parameters?.resolveToMapList(inputs));
        if (parameters) {
            apiParams.Parameters = parameters;
        }
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'createStack', {
            service: aws_service_1.AwsService.CLOUD_FORMATION,
            pascalCaseApi: 'CreateStack',
            apiParams: apiParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING,
                    selector: '$.StackId',
                    name: 'StackId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Create stack failed for ${apiParams.StackName}: ${result.stackTrace}`);
        }
        return result.outputs?.['createStack.StackId'];
    }
    resolveSecureStringsInParametersInput(parameters) {
        if (!parameters) {
            return null;
        }
        for (const parameter of parameters) {
            const value = parameter.ParameterValue;
            if (!value) {
                continue;
            }
            parameter.ParameterValue = this.props.parameterResolver.resolve(value);
        }
        return parameters;
    }
    getStackState(stackName) {
        const describeResponse = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'describeStack', {
            service: aws_service_1.AwsService.CLOUD_FORMATION,
            pascalCaseApi: 'DescribeStacks',
            apiParams: {
                StackName: stackName,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING_MAP,
                    name: 'Stack',
                    selector: '$.Stacks[0]',
                }],
        }), this.props).invoke({});
        return describeResponse.outputs?.['describeStack.Stack'] ?? null;
    }
    waitForStackCompletion(stackName) {
        wait_for_and_assert_1.waitForAndAssertStackStatus({
            stackName: stackName,
            waitForStatus: [
                stack_status_1.StackStatus.CREATE_FAILED,
                stack_status_1.StackStatus.CREATE_COMPLETE,
                stack_status_1.StackStatus.ROLLBACK_FAILED,
                stack_status_1.StackStatus.ROLLBACK_COMPLETE,
                stack_status_1.StackStatus.DELETE_IN_PROGRESS,
                stack_status_1.StackStatus.DELETE_FAILED,
                stack_status_1.StackStatus.UPDATE_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_COMPLETE_CLEANUP_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_COMPLETE,
                stack_status_1.StackStatus.UPDATE_FAILED,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_FAILED,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_COMPLETE,
                stack_status_1.StackStatus.REVIEW_IN_PROGRESS,
                stack_status_1.StackStatus.IMPORT_IN_PROGRESS,
                stack_status_1.StackStatus.IMPORT_COMPLETE,
                stack_status_1.StackStatus.IMPORT_ROLLBACK_IN_PROGRESS,
                stack_status_1.StackStatus.IMPORT_ROLLBACK_FAILED,
                stack_status_1.StackStatus.IMPORT_ROLLBACK_COMPLETE,
            ],
            assertStatus: stack_status_1.StackStatus.CREATE_COMPLETE,
        }, this.props);
    }
}
exports.CreateStackSimulation = CreateStackSimulation;
_a = JSII_RTTI_SYMBOL_1;
CreateStackSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.CreateStackSimulation", version: "0.0.30" };
CreateStackSimulation.InProgressStatuses = [
    stack_status_1.StackStatus.ROLLBACK_IN_PROGRESS,
    stack_status_1.StackStatus.DELETE_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_COMPLETE_CLEANUP_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_ROLLBACK_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS,
    stack_status_1.StackStatus.REVIEW_IN_PROGRESS,
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3JlYXRlLXN0YWNrLXNpbXVsYXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvc2ltdWxhdGlvbi9hdXRvbWF0aW9uL2NyZWF0ZS1zdGFjay1zaW11bGF0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsNkJBQWlDO0FBQ2pDLDZDQUFvQztBQUNwQyxzREFBc0Q7QUFDdEQsOERBQTBEO0FBQzFELDREQUF3RDtBQUd4RCwrRUFBNkY7QUFDN0YsNkVBQXdFO0FBRXhFLHVGQUFrRjtBQUNsRix5RUFBcUU7QUFDckUscUdBQTBHO0FBQzFHLHVGQUFpRjtBQUNqRiw4RUFBeUU7QUFDekUsNkVBQXdFO0FBQ3hFLDBEQUFvRDtBQWdDcEQ7OztHQUdHO0FBQ0gsTUFBYSxxQkFBc0IsU0FBUSxxREFBd0I7SUFlakUsWUFBWSxJQUFxQixFQUFFLEtBQWlDO1FBQ2xFLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUxMLFdBQU0sR0FBVyxpQkFBaUIsQ0FBQztRQU0xQyxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQztRQUMzQixJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQztJQUNyQixDQUFDO0lBRU0sV0FBVyxDQUFDLE1BQTJCO1FBQzVDLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDNUIsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3hFLE9BQU8sQ0FBQyxHQUFHLENBQUMsOEJBQThCLFNBQVMseUJBQXlCLENBQUMsQ0FBQztRQUM5RSxJQUFJLENBQUMscUJBQXFCLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdEMsT0FBTyxDQUFDLEdBQUcsQ0FBQyx5QkFBeUIsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUNsRCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sQ0FBQyxHQUFHLENBQUMsNEJBQTRCLFNBQVMsYUFBYSxDQUFDLENBQUM7UUFDaEUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sQ0FBQyxHQUFHLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUM3QyxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzVDLE9BQU87WUFDTCxPQUFPLEVBQUUsT0FBTztZQUNoQixXQUFXLEVBQUUsS0FBSyxFQUFFLFdBQVc7WUFDL0IsaUJBQWlCLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixJQUFJLEVBQUU7U0FDbEQsQ0FBQztJQUNKLENBQUM7SUFFTyxjQUFjLENBQUMsTUFBMkI7UUFDaEQsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVyxFQUFFLE1BQU0sQ0FBQztlQUN2RCxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQzVFLENBQUM7SUFFTyxhQUFhLENBQUMsV0FBd0MsRUFBRSxNQUEyQjtRQUN6RixPQUFPLFdBQVcsS0FBSyxTQUFTLElBQUksSUFBSSxTQUFHLENBQUMsV0FBVyxDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO0lBQ25GLENBQUM7SUFFTyxxQkFBcUIsQ0FBQyxTQUFpQjtRQUM3QyxPQUFPLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUN0QyxJQUFJLHFEQUF3QixDQUFDLElBQUksc0JBQVMsQ0FBQyxJQUFJLG1CQUFLLEVBQUUsRUFBRSxPQUFPLEVBQUU7Z0JBQy9ELFlBQVksRUFBRSxDQUFDO2FBQ2hCLENBQUMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQzVCO0lBQ0gsQ0FBQztJQUVPLGNBQWMsQ0FBQyxTQUFpQjtRQUN0QyxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQzVDLElBQUksSUFBSSxDQUFDLDBCQUEwQixDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQUUsT0FBTyxLQUFLLENBQUM7U0FBRTtRQUM3RCxJQUFJLEtBQUssRUFBRSxXQUFXLEtBQUssbUJBQW1CLEVBQUU7WUFBRSxPQUFPLElBQUksQ0FBQztTQUFFO1FBRWhFLE9BQU8sQ0FBQyxHQUFHLENBQUMsa0JBQWtCLFNBQVMsb0ZBQW9GLENBQUMsQ0FBQztRQUM3SCxJQUFJLHFEQUF3QixDQUFDLElBQUksbUNBQWUsQ0FBQyxJQUFJLG1CQUFLLEVBQUUsRUFBRSxhQUFhLEVBQUU7WUFDM0UsaUJBQWlCLEVBQUUsSUFBSSxpQ0FBZSxDQUFDLFNBQVMsQ0FBQztTQUNsRCxDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUMzQixPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFTywwQkFBMEIsQ0FBQyxLQUFpQztRQUNsRSxPQUFPLEtBQUssS0FBSyxJQUFJLElBQUkscUJBQXFCLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUFDLDBCQUFXLENBQUMsS0FBSyxDQUFDLFdBQXVDLENBQUMsQ0FBQyxDQUFDO0lBQ3pJLENBQUM7SUFFTyxXQUFXLENBQUMsTUFBMkI7UUFDN0MsTUFBTSxTQUFTLEdBQUcsb0RBQXVCLENBQUM7WUFDeEMsU0FBUyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsU0FBUztZQUN4QyxZQUFZLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZO1lBQzlDLGtCQUFrQixFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsa0JBQWtCO1lBQzFELGdCQUFnQixFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsZ0JBQWdCO1lBQ3RELFNBQVMsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWM7WUFDN0MsYUFBYSxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYTtZQUNoRCxPQUFPLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxPQUFPO1lBQ3BDLGVBQWUsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLGVBQWU7WUFDcEQsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsY0FBYztZQUNsRCxJQUFJLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJO1lBQzlCLFlBQVksRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLFlBQVk7WUFDOUMsV0FBVyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsV0FBVztZQUM1QyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLGdCQUFnQjtTQUN2RCxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQzNCLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxxQ0FBcUMsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLFVBQVUsRUFBRSxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ3hILElBQUksVUFBVSxFQUFFO1lBQUUsU0FBUyxDQUFDLFVBQVUsR0FBRyxVQUFVLENBQUM7U0FBRTtRQUV0RCxNQUFNLE1BQU0sR0FBRyxJQUFJLHFEQUF3QixDQUFDLElBQUkseUJBQVUsQ0FBQyxJQUFJLG1CQUFLLEVBQUUsRUFBRSxhQUFhLEVBQUU7WUFDckYsT0FBTyxFQUFFLHdCQUFVLENBQUMsZUFBZTtZQUNuQyxhQUFhLEVBQUUsYUFBYTtZQUM1QixTQUFTLEVBQUUsU0FBUztZQUNwQixPQUFPLEVBQUUsQ0FBQztvQkFDUixVQUFVLEVBQUUsd0JBQVksQ0FBQyxNQUFNO29CQUMvQixRQUFRLEVBQUUsV0FBVztvQkFDckIsSUFBSSxFQUFFLFNBQVM7aUJBQ2hCLENBQUM7U0FDSCxDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUMzQixJQUFJLE1BQU0sQ0FBQyxZQUFZLEtBQUssNEJBQVksQ0FBQyxPQUFPLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsU0FBUyxDQUFDLFNBQVMsS0FBSyxNQUFNLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUN6RjtRQUNELE9BQU8sTUFBTSxDQUFDLE9BQU8sRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVPLHFDQUFxQyxDQUFDLFVBQTZDO1FBQ3pGLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFBRSxPQUFPLElBQUksQ0FBQztTQUFFO1FBQ2pDLEtBQUssTUFBTSxTQUFTLElBQUksVUFBVSxFQUFFO1lBQ2xDLE1BQU0sS0FBSyxHQUFHLFNBQVMsQ0FBQyxjQUFjLENBQUM7WUFDdkMsSUFBSSxDQUFDLEtBQUssRUFBRTtnQkFBRSxTQUFTO2FBQUU7WUFDekIsU0FBUyxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUN4RTtRQUNELE9BQU8sVUFBVSxDQUFDO0lBQ3BCLENBQUM7SUFFTyxhQUFhLENBQUMsU0FBaUI7UUFDckMsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLHFEQUF3QixDQUFDLElBQUkseUJBQVUsQ0FBQyxJQUFJLG1CQUFLLEVBQUUsRUFBRSxlQUFlLEVBQUU7WUFDakcsT0FBTyxFQUFFLHdCQUFVLENBQUMsZUFBZTtZQUNuQyxhQUFhLEVBQUUsZ0JBQWdCO1lBQy9CLFNBQVMsRUFBRTtnQkFDVCxTQUFTLEVBQUUsU0FBUzthQUNyQjtZQUNELE9BQU8sRUFBRSxDQUFDO29CQUNSLFVBQVUsRUFBRSx3QkFBWSxDQUFDLFVBQVU7b0JBQ25DLElBQUksRUFBRSxPQUFPO29CQUNiLFFBQVEsRUFBRSxhQUFhO2lCQUN4QixDQUFDO1NBQ0gsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDM0IsT0FBTyxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLElBQUksQ0FBQztJQUNuRSxDQUFDO0lBRU8sc0JBQXNCLENBQUMsU0FBaUI7UUFDOUMsaURBQTJCLENBQUM7WUFDMUIsU0FBUyxFQUFFLFNBQVM7WUFDcEIsYUFBYSxFQUFFO2dCQUNiLDBCQUFXLENBQUMsYUFBYTtnQkFDekIsMEJBQVcsQ0FBQyxlQUFlO2dCQUMzQiwwQkFBVyxDQUFDLGVBQWU7Z0JBQzNCLDBCQUFXLENBQUMsaUJBQWlCO2dCQUM3QiwwQkFBVyxDQUFDLGtCQUFrQjtnQkFDOUIsMEJBQVcsQ0FBQyxhQUFhO2dCQUN6QiwwQkFBVyxDQUFDLGtCQUFrQjtnQkFDOUIsMEJBQVcsQ0FBQyxtQ0FBbUM7Z0JBQy9DLDBCQUFXLENBQUMsZUFBZTtnQkFDM0IsMEJBQVcsQ0FBQyxhQUFhO2dCQUN6QiwwQkFBVyxDQUFDLDJCQUEyQjtnQkFDdkMsMEJBQVcsQ0FBQyxzQkFBc0I7Z0JBQ2xDLDBCQUFXLENBQUMsNENBQTRDO2dCQUN4RCwwQkFBVyxDQUFDLHdCQUF3QjtnQkFDcEMsMEJBQVcsQ0FBQyxrQkFBa0I7Z0JBQzlCLDBCQUFXLENBQUMsa0JBQWtCO2dCQUM5QiwwQkFBVyxDQUFDLGVBQWU7Z0JBQzNCLDBCQUFXLENBQUMsMkJBQTJCO2dCQUN2QywwQkFBVyxDQUFDLHNCQUFzQjtnQkFDbEMsMEJBQVcsQ0FBQyx3QkFBd0I7YUFDckM7WUFDRCxZQUFZLEVBQUUsMEJBQVcsQ0FBQyxlQUFlO1NBQzFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ2pCLENBQUM7O0FBaEtILHNEQWtLQzs7O0FBakt5Qix3Q0FBa0IsR0FBRztJQUMzQywwQkFBVyxDQUFDLG9CQUFvQjtJQUNoQywwQkFBVyxDQUFDLGtCQUFrQjtJQUM5QiwwQkFBVyxDQUFDLGtCQUFrQjtJQUM5QiwwQkFBVyxDQUFDLG1DQUFtQztJQUMvQywwQkFBVyxDQUFDLDJCQUEyQjtJQUN2QywwQkFBVyxDQUFDLDRDQUE0QztJQUN4RCwwQkFBVyxDQUFDLGtCQUFrQjtDQUMvQixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgVVJMIGFzIFVybCB9IGZyb20gJ3VybCc7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IERhdGFUeXBlRW51bSB9IGZyb20gJy4uLy4uL2RvbWFpbi9kYXRhLXR5cGUnO1xuaW1wb3J0IHsgUmVzcG9uc2VDb2RlIH0gZnJvbSAnLi4vLi4vZG9tYWluL3Jlc3BvbnNlLWNvZGUnO1xuaW1wb3J0IHsgU3RhY2tTdGF0dXMgfSBmcm9tICcuLi8uLi9kb21haW4vc3RhY2stc3RhdHVzJztcbmltcG9ydCB7IElBd3NJbnZva2VyIH0gZnJvbSAnLi4vLi4vaW50ZXJmYWNlL2F3cy1pbnZva2VyJztcbmltcG9ydCB7IElTbGVlcEhvb2sgfSBmcm9tICcuLi8uLi9pbnRlcmZhY2Uvc2xlZXAtaG9vayc7XG5pbXBvcnQgeyBIYXJkQ29kZWRTdHJpbmcsIElTdHJpbmdWYXJpYWJsZSB9IGZyb20gJy4uLy4uL2ludGVyZmFjZS92YXJpYWJsZXMvc3RyaW5nLXZhcmlhYmxlJztcbmltcG9ydCB7IEF3c0FwaVN0ZXAgfSBmcm9tICcuLi8uLi9wYXJlbnQtc3RlcHMvYXV0b21hdGlvbi9hd3MtYXBpLXN0ZXAnO1xuaW1wb3J0IHsgQ3JlYXRlU3RhY2tTdGVwLCBJUGFyYW1ldGVyUmVzb2x2ZXIgfSBmcm9tICcuLi8uLi9wYXJlbnQtc3RlcHMvYXV0b21hdGlvbi9jcmVhdGUtc3RhY2stc3RlcCc7XG5pbXBvcnQgeyBEZWxldGVTdGFja1N0ZXAgfSBmcm9tICcuLi8uLi9wYXJlbnQtc3RlcHMvYXV0b21hdGlvbi9kZWxldGUtc3RhY2stc3RlcCc7XG5pbXBvcnQgeyBTbGVlcFN0ZXAgfSBmcm9tICcuLi8uLi9wYXJlbnQtc3RlcHMvYXV0b21hdGlvbi9zbGVlcC1zdGVwJztcbmltcG9ydCB7IHdhaXRGb3JBbmRBc3NlcnRTdGFja1N0YXR1cyB9IGZyb20gJy4uLy4uL3BhcmVudC1zdGVwcy9hdXRvbWF0aW9uL3N1Yi1zdGVwcy93YWl0LWZvci1hbmQtYXNzZXJ0JztcbmltcG9ydCB7IHBydW5lQW5kVHJhbnNmb3JtUmVjb3JkIH0gZnJvbSAnLi4vLi4vdXRpbHMvcHJ1bmUtYW5kLXRyYW5zZm9ybS1yZWNvcmQnO1xuaW1wb3J0IHsgQXV0b21hdGlvblN0ZXBTaW11bGF0aW9uIH0gZnJvbSAnLi4vYXV0b21hdGlvbi1zdGVwLXNpbXVsYXRpb24nO1xuaW1wb3J0IHsgQXV0b21hdGlvblNpbXVsYXRpb25CYXNlIH0gZnJvbSAnLi9hdXRvbWF0aW9uLXNpbXVsYXRpb24tYmFzZSc7XG5pbXBvcnQge0F3c1NlcnZpY2V9IGZyb20gXCIuLi8uLi9kb21haW4vYXdzLXNlcnZpY2VcIjtcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBDcmVhdGVTdGFja1N0ZXBcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDcmVhdGVTdGFja1NpbXVsYXRpb25Qcm9wcyB7XG4gIC8qKlxuICAgICAqIChPcHRpb25hbCkgVXNlIHRoaXMgYXMgYSBob29rIHRvIGluamVjdCBhbiBhbHRlcm5hdGUgSUF3c0ludm9rZXIgKGZvciBtb2NraW5nIHRoZSBBV1MgQVBJIGNhbGwpLlxuICAgICAqIEBkZWZhdWx0IC0gd2lsbCBwZXJmb3JtIGEgcmVhbCBpbnZvY2F0aW9uIG9mIHRoZSBKYXZhU2NyaXB0IEFXUyBTREsgdXNpbmcgUmVmbGVjdGl2ZUF3c0ludm9rZXIgY2xhc3MuXG4gICAgICovXG4gIHJlYWRvbmx5IGF3c0ludm9rZXI6IElBd3NJbnZva2VyO1xuXG4gIC8qKlxuICAgICAqIChPcHRpb25hbCkgV2hldGhlciB0byByZWFsbHkgcGVyZm9ybSBhIHBhdXNlIG9mIHRoZSBydW50aW1lLlxuICAgICAqIFRvIG92ZXJyaWRlIHNsZWVwIGJlaGF2aW9yLCBpbmplY3QgYW4gSVNsZWVwSG9vayBpbXBsIG9yIHVzZSB0aGUgcHJvdmlkZWQgTW9ja1NsZWVwIGNsYXNzLlxuICAgICAqIEBkZWZhdWx0IFNsZWVwZXJJbXBsXG4gICAgICovXG4gIHJlYWRvbmx5IHNsZWVwSG9vazogSVNsZWVwSG9vaztcblxuICAvKipcbiAgICogKE9wdGlvbmFsKSBSZXNvbHZlciBmb3Igc2VjdXJlIHN0cmluZ3MgaW4gcGFyYW1ldGVycy5cbiAgICogUmVxdWlyZWQgdG8gc2ltdWxhdGUgaWYgdXNpbmcgdG9rZW5zIGluIHBhcmFtZXRlcnMgaW5wdXQuXG4gICAqIEBkZWZhdWx0IC0gVHJlYXRzIHBhcmFtZXRlcnMgYXMgbGl0ZXJhbFxuICAgKi9cbiAgcmVhZG9ubHkgcGFyYW1ldGVyUmVzb2x2ZXI6IElQYXJhbWV0ZXJSZXNvbHZlcjtcbn1cblxuaW50ZXJmYWNlIERlc2NyaWJlU3RhY2tSZXN1bHQge1xuICBTdGFja1N0YXR1czogc3RyaW5nO1xuICBTdGFja1N0YXR1c1JlYXNvbj86IHN0cmluZztcbn1cblxuLyoqXG4gKiBBdXRvbWF0aW9uU3RlcCBpbXBsZW1lbnRhdGlvbiBmb3IgYXdzOmNyZWF0ZVN0YWNrXG4gKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3lzdGVtcy1tYW5hZ2VyL2xhdGVzdC91c2VyZ3VpZGUvYXV0b21hdGlvbi1hY3Rpb24tY3JlYXRlc3RhY2suaHRtbFxuICovXG5leHBvcnQgY2xhc3MgQ3JlYXRlU3RhY2tTaW11bGF0aW9uIGV4dGVuZHMgQXV0b21hdGlvblNpbXVsYXRpb25CYXNlIHtcbiAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgSW5Qcm9ncmVzc1N0YXR1c2VzID0gW1xuICAgIFN0YWNrU3RhdHVzLlJPTExCQUNLX0lOX1BST0dSRVNTLFxuICAgIFN0YWNrU3RhdHVzLkRFTEVURV9JTl9QUk9HUkVTUyxcbiAgICBTdGFja1N0YXR1cy5VUERBVEVfSU5fUFJPR1JFU1MsXG4gICAgU3RhY2tTdGF0dXMuVVBEQVRFX0NPTVBMRVRFX0NMRUFOVVBfSU5fUFJPR1JFU1MsXG4gICAgU3RhY2tTdGF0dXMuVVBEQVRFX1JPTExCQUNLX0lOX1BST0dSRVNTLFxuICAgIFN0YWNrU3RhdHVzLlVQREFURV9ST0xMQkFDS19DT01QTEVURV9DTEVBTlVQX0lOX1BST0dSRVNTLFxuICAgIFN0YWNrU3RhdHVzLlJFVklFV19JTl9QUk9HUkVTUyxcbiAgXTtcblxuICByZWFkb25seSBhY3Rpb246IHN0cmluZyA9ICdhd3M6Y3JlYXRlU3RhY2snO1xuICBwcml2YXRlIGNyZWF0U3RhY2tTdGVwOiBDcmVhdGVTdGFja1N0ZXA7XG4gIHByaXZhdGUgcHJvcHM6IENyZWF0ZVN0YWNrU2ltdWxhdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHN0ZXA6IENyZWF0ZVN0YWNrU3RlcCwgcHJvcHM6IENyZWF0ZVN0YWNrU2ltdWxhdGlvblByb3BzKSB7XG4gICAgc3VwZXIoc3RlcCk7XG4gICAgdGhpcy5jcmVhdFN0YWNrU3RlcCA9IHN0ZXA7XG4gICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICB9XG5cbiAgcHVibGljIGV4ZWN1dGVTdGVwKGlucHV0czogUmVjb3JkPHN0cmluZywgYW55Pik6IFJlY29yZDxzdHJpbmcsIGFueT4ge1xuICAgIHRoaXMudmVyaWZ5VXJsUHJvcHMoaW5wdXRzKTtcbiAgICBjb25zdCBzdGFja05hbWUgPSB0aGlzLmNyZWF0U3RhY2tTdGVwLnN0YWNrTmFtZS5yZXNvbHZlVG9TdHJpbmcoaW5wdXRzKTtcbiAgICBjb25zb2xlLmxvZyhgQ3JlYXRlU3RhY2s6IENoZWNraW5nIHRoYXQgJHtzdGFja05hbWV9IGlzIHJlYWR5IHRvIGJlIGNyZWF0ZWRgKTtcbiAgICB0aGlzLnByZVZlcmlmeVN0YWNrU3ViU3RlcChzdGFja05hbWUpO1xuICAgIGNvbnNvbGUubG9nKGBDcmVhdGVTdGFjazogQ3JlYXRpbmcgJHtzdGFja05hbWV9YCk7XG4gICAgY29uc3Qgc3RhY2tJZCA9IHRoaXMuY3JlYXRlU3RhY2soaW5wdXRzKTtcbiAgICBjb25zb2xlLmxvZyhgQ3JlYXRlU3RhY2s6IFdhaXRpbmcgZm9yICR7c3RhY2tOYW1lfSBjb21wbGV0aW9uYCk7XG4gICAgdGhpcy53YWl0Rm9yU3RhY2tDb21wbGV0aW9uKHN0YWNrTmFtZSk7XG4gICAgY29uc29sZS5sb2coJ0NyZWF0ZVN0YWNrOiBTdGFjayBpcyBjcmVhdGVkJyk7XG4gICAgY29uc3Qgc3RhY2sgPSB0aGlzLmdldFN0YWNrU3RhdGUoc3RhY2tOYW1lKTtcbiAgICByZXR1cm4ge1xuICAgICAgU3RhY2tJZDogc3RhY2tJZCxcbiAgICAgIFN0YWNrU3RhdHVzOiBzdGFjaz8uU3RhY2tTdGF0dXMsXG4gICAgICBTdGFja1N0YXR1c1JlYXNvbjogc3RhY2s/LlN0YWNrU3RhdHVzUmVhc29uID8/ICcnLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIHZlcmlmeVVybFByb3BzKGlucHV0czogUmVjb3JkPHN0cmluZywgYW55Pikge1xuICAgIHJldHVybiB0aGlzLnZlcmlmeVVybFByb3AodGhpcy5jcmVhdFN0YWNrU3RlcC50ZW1wbGF0ZVVybCwgaW5wdXRzKVxuICAgICAgICAgICAgJiYgdGhpcy52ZXJpZnlVcmxQcm9wKHRoaXMuY3JlYXRTdGFja1N0ZXAuc3RhY2tQb2xpY3lVcmwsIGlucHV0cyk7XG4gIH1cblxuICBwcml2YXRlIHZlcmlmeVVybFByb3AodXJsVG9WZXJpZnk6IElTdHJpbmdWYXJpYWJsZSB8IHVuZGVmaW5lZCwgaW5wdXRzOiBSZWNvcmQ8c3RyaW5nLCBhbnk+KSB7XG4gICAgcmV0dXJuIHVybFRvVmVyaWZ5ID09PSB1bmRlZmluZWQgfHwgbmV3IFVybCh1cmxUb1ZlcmlmeS5yZXNvbHZlVG9TdHJpbmcoaW5wdXRzKSk7XG4gIH1cblxuICBwcml2YXRlIHByZVZlcmlmeVN0YWNrU3ViU3RlcChzdGFja05hbWU6IHN0cmluZykge1xuICAgIHdoaWxlICghdGhpcy5wcmVWZXJpZnlTdGFjayhzdGFja05hbWUpKSB7XG4gICAgICBuZXcgQXV0b21hdGlvblN0ZXBTaW11bGF0aW9uKG5ldyBTbGVlcFN0ZXAobmV3IFN0YWNrKCksICdzbGVlcCcsIHtcbiAgICAgICAgc2xlZXBTZWNvbmRzOiAyLFxuICAgICAgfSksIHRoaXMucHJvcHMpLmludm9rZSh7fSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBwcmVWZXJpZnlTdGFjayhzdGFja05hbWU6IHN0cmluZyk6IGJvb2xlYW4ge1xuICAgIGNvbnN0IHN0YXRlID0gdGhpcy5nZXRTdGFja1N0YXRlKHN0YWNrTmFtZSk7XG4gICAgaWYgKHRoaXMuaXNTdGFja09wZXJhdGlvbkluUHJvZ3Jlc3Moc3RhdGUpKSB7IHJldHVybiBmYWxzZTsgfVxuICAgIGlmIChzdGF0ZT8uU3RhY2tTdGF0dXMgIT09ICdST0xMQkFDS19DT01QTEVURScpIHsgcmV0dXJuIHRydWU7IH1cblxuICAgIGNvbnNvbGUubG9nKGBEZWxldGluZyBzdGFjayAke3N0YWNrTmFtZX0gYmVjYXVzZSB3ZSBhcmUgcmV0cnlpbmcgYSBjcmVhdGUgb24gYW4gZXhpc3Rpbmcgc3RhY2sgaW4gUk9MTEJBQ0tfQ09NUExFVEUgc3RhdGUuYCk7XG4gICAgbmV3IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbihuZXcgRGVsZXRlU3RhY2tTdGVwKG5ldyBTdGFjaygpLCAnZGVsZXRlU3RhY2snLCB7XG4gICAgICBzdGFja05hbWVWYXJpYWJsZTogbmV3IEhhcmRDb2RlZFN0cmluZyhzdGFja05hbWUpLFxuICAgIH0pLCB0aGlzLnByb3BzKS5pbnZva2Uoe30pO1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIHByaXZhdGUgaXNTdGFja09wZXJhdGlvbkluUHJvZ3Jlc3Moc3RhdGU6IERlc2NyaWJlU3RhY2tSZXN1bHQgfCBudWxsKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHN0YXRlICE9PSBudWxsICYmIENyZWF0ZVN0YWNrU2ltdWxhdGlvbi5JblByb2dyZXNzU3RhdHVzZXMuaW5jbHVkZXMoU3RhY2tTdGF0dXNbc3RhdGUuU3RhY2tTdGF0dXMgYXMga2V5b2YgdHlwZW9mIFN0YWNrU3RhdHVzXSk7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZVN0YWNrKGlucHV0czogUmVjb3JkPHN0cmluZywgYW55Pik6IHN0cmluZyB7XG4gICAgY29uc3QgYXBpUGFyYW1zID0gcHJ1bmVBbmRUcmFuc2Zvcm1SZWNvcmQoe1xuICAgICAgU3RhY2tOYW1lOiB0aGlzLmNyZWF0U3RhY2tTdGVwLnN0YWNrTmFtZSxcbiAgICAgIENhcGFiaWxpdGllczogdGhpcy5jcmVhdFN0YWNrU3RlcC5jYXBhYmlsaXRpZXMsXG4gICAgICBDbGllbnRSZXF1ZXN0VG9rZW46IHRoaXMuY3JlYXRTdGFja1N0ZXAuY2xpZW50UmVxdWVzdFRva2VuLFxuICAgICAgTm90aWZpY2F0aW9uQVJOczogdGhpcy5jcmVhdFN0YWNrU3RlcC5ub3RpZmljYXRpb25BUk5zLFxuICAgICAgT25GYWlsdXJlOiB0aGlzLmNyZWF0U3RhY2tTdGVwLm9uU3RhY2tGYWlsdXJlLFxuICAgICAgUmVzb3VyY2VUeXBlczogdGhpcy5jcmVhdFN0YWNrU3RlcC5yZXNvdXJjZVR5cGVzLFxuICAgICAgUm9sZUFSTjogdGhpcy5jcmVhdFN0YWNrU3RlcC5yb2xlQXJuLFxuICAgICAgU3RhY2tQb2xpY3lCb2R5OiB0aGlzLmNyZWF0U3RhY2tTdGVwLnN0YWNrUG9saWN5Qm9keSxcbiAgICAgIFN0YWNrUG9saWN5VVJMOiB0aGlzLmNyZWF0U3RhY2tTdGVwLnN0YWNrUG9saWN5VXJsLFxuICAgICAgVGFnczogdGhpcy5jcmVhdFN0YWNrU3RlcC50YWdzLFxuICAgICAgVGVtcGxhdGVCb2R5OiB0aGlzLmNyZWF0U3RhY2tTdGVwLnRlbXBsYXRlQm9keSxcbiAgICAgIFRlbXBsYXRlVVJMOiB0aGlzLmNyZWF0U3RhY2tTdGVwLnRlbXBsYXRlVXJsLFxuICAgICAgVGltZW91dEluTWludXRlczogdGhpcy5jcmVhdFN0YWNrU3RlcC50aW1lb3V0SW5NaW51dGVzLFxuICAgIH0sIHggPT4geC5yZXNvbHZlKGlucHV0cykpO1xuICAgIGNvbnN0IHBhcmFtZXRlcnMgPSB0aGlzLnJlc29sdmVTZWN1cmVTdHJpbmdzSW5QYXJhbWV0ZXJzSW5wdXQodGhpcy5jcmVhdFN0YWNrU3RlcC5wYXJhbWV0ZXJzPy5yZXNvbHZlVG9NYXBMaXN0KGlucHV0cykpO1xuICAgIGlmIChwYXJhbWV0ZXJzKSB7IGFwaVBhcmFtcy5QYXJhbWV0ZXJzID0gcGFyYW1ldGVyczsgfVxuXG4gICAgY29uc3QgcmVzdWx0ID0gbmV3IEF1dG9tYXRpb25TdGVwU2ltdWxhdGlvbihuZXcgQXdzQXBpU3RlcChuZXcgU3RhY2soKSwgJ2NyZWF0ZVN0YWNrJywge1xuICAgICAgc2VydmljZTogQXdzU2VydmljZS5DTE9VRF9GT1JNQVRJT04sXG4gICAgICBwYXNjYWxDYXNlQXBpOiAnQ3JlYXRlU3RhY2snLFxuICAgICAgYXBpUGFyYW1zOiBhcGlQYXJhbXMsXG4gICAgICBvdXRwdXRzOiBbe1xuICAgICAgICBvdXRwdXRUeXBlOiBEYXRhVHlwZUVudW0uU1RSSU5HLFxuICAgICAgICBzZWxlY3RvcjogJyQuU3RhY2tJZCcsXG4gICAgICAgIG5hbWU6ICdTdGFja0lkJyxcbiAgICAgIH1dLFxuICAgIH0pLCB0aGlzLnByb3BzKS5pbnZva2Uoe30pO1xuICAgIGlmIChyZXN1bHQucmVzcG9uc2VDb2RlICE9PSBSZXNwb25zZUNvZGUuU1VDQ0VTUykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDcmVhdGUgc3RhY2sgZmFpbGVkIGZvciAke2FwaVBhcmFtcy5TdGFja05hbWV9OiAke3Jlc3VsdC5zdGFja1RyYWNlfWApO1xuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0Lm91dHB1dHM/LlsnY3JlYXRlU3RhY2suU3RhY2tJZCddO1xuICB9XG5cbiAgcHJpdmF0ZSByZXNvbHZlU2VjdXJlU3RyaW5nc0luUGFyYW1ldGVyc0lucHV0KHBhcmFtZXRlcnM6IFJlY29yZDxzdHJpbmcsIGFueT5bXSB8IHVuZGVmaW5lZCk6IFJlY29yZDxzdHJpbmcsIGFueT5bXSB8IG51bGwge1xuICAgIGlmICghcGFyYW1ldGVycykgeyByZXR1cm4gbnVsbDsgfVxuICAgIGZvciAoY29uc3QgcGFyYW1ldGVyIG9mIHBhcmFtZXRlcnMpIHtcbiAgICAgIGNvbnN0IHZhbHVlID0gcGFyYW1ldGVyLlBhcmFtZXRlclZhbHVlO1xuICAgICAgaWYgKCF2YWx1ZSkgeyBjb250aW51ZTsgfVxuICAgICAgcGFyYW1ldGVyLlBhcmFtZXRlclZhbHVlID0gdGhpcy5wcm9wcy5wYXJhbWV0ZXJSZXNvbHZlci5yZXNvbHZlKHZhbHVlKTtcbiAgICB9XG4gICAgcmV0dXJuIHBhcmFtZXRlcnM7XG4gIH1cblxuICBwcml2YXRlIGdldFN0YWNrU3RhdGUoc3RhY2tOYW1lOiBzdHJpbmcpOiBEZXNjcmliZVN0YWNrUmVzdWx0IHwgbnVsbCB7XG4gICAgY29uc3QgZGVzY3JpYmVSZXNwb25zZSA9IG5ldyBBdXRvbWF0aW9uU3RlcFNpbXVsYXRpb24obmV3IEF3c0FwaVN0ZXAobmV3IFN0YWNrKCksICdkZXNjcmliZVN0YWNrJywge1xuICAgICAgc2VydmljZTogQXdzU2VydmljZS5DTE9VRF9GT1JNQVRJT04sXG4gICAgICBwYXNjYWxDYXNlQXBpOiAnRGVzY3JpYmVTdGFja3MnLFxuICAgICAgYXBpUGFyYW1zOiB7XG4gICAgICAgIFN0YWNrTmFtZTogc3RhY2tOYW1lLFxuICAgICAgfSxcbiAgICAgIG91dHB1dHM6IFt7XG4gICAgICAgIG91dHB1dFR5cGU6IERhdGFUeXBlRW51bS5TVFJJTkdfTUFQLFxuICAgICAgICBuYW1lOiAnU3RhY2snLFxuICAgICAgICBzZWxlY3RvcjogJyQuU3RhY2tzWzBdJyxcbiAgICAgIH1dLFxuICAgIH0pLCB0aGlzLnByb3BzKS5pbnZva2Uoe30pO1xuICAgIHJldHVybiBkZXNjcmliZVJlc3BvbnNlLm91dHB1dHM/LlsnZGVzY3JpYmVTdGFjay5TdGFjayddID8/IG51bGw7XG4gIH1cblxuICBwcml2YXRlIHdhaXRGb3JTdGFja0NvbXBsZXRpb24oc3RhY2tOYW1lOiBzdHJpbmcpOiB2b2lkIHtcbiAgICB3YWl0Rm9yQW5kQXNzZXJ0U3RhY2tTdGF0dXMoe1xuICAgICAgc3RhY2tOYW1lOiBzdGFja05hbWUsXG4gICAgICB3YWl0Rm9yU3RhdHVzOiBbXG4gICAgICAgIFN0YWNrU3RhdHVzLkNSRUFURV9GQUlMRUQsXG4gICAgICAgIFN0YWNrU3RhdHVzLkNSRUFURV9DT01QTEVURSxcbiAgICAgICAgU3RhY2tTdGF0dXMuUk9MTEJBQ0tfRkFJTEVELFxuICAgICAgICBTdGFja1N0YXR1cy5ST0xMQkFDS19DT01QTEVURSxcbiAgICAgICAgU3RhY2tTdGF0dXMuREVMRVRFX0lOX1BST0dSRVNTLFxuICAgICAgICBTdGFja1N0YXR1cy5ERUxFVEVfRkFJTEVELFxuICAgICAgICBTdGFja1N0YXR1cy5VUERBVEVfSU5fUFJPR1JFU1MsXG4gICAgICAgIFN0YWNrU3RhdHVzLlVQREFURV9DT01QTEVURV9DTEVBTlVQX0lOX1BST0dSRVNTLFxuICAgICAgICBTdGFja1N0YXR1cy5VUERBVEVfQ09NUExFVEUsXG4gICAgICAgIFN0YWNrU3RhdHVzLlVQREFURV9GQUlMRUQsXG4gICAgICAgIFN0YWNrU3RhdHVzLlVQREFURV9ST0xMQkFDS19JTl9QUk9HUkVTUyxcbiAgICAgICAgU3RhY2tTdGF0dXMuVVBEQVRFX1JPTExCQUNLX0ZBSUxFRCxcbiAgICAgICAgU3RhY2tTdGF0dXMuVVBEQVRFX1JPTExCQUNLX0NPTVBMRVRFX0NMRUFOVVBfSU5fUFJPR1JFU1MsXG4gICAgICAgIFN0YWNrU3RhdHVzLlVQREFURV9ST0xMQkFDS19DT01QTEVURSxcbiAgICAgICAgU3RhY2tTdGF0dXMuUkVWSUVXX0lOX1BST0dSRVNTLFxuICAgICAgICBTdGFja1N0YXR1cy5JTVBPUlRfSU5fUFJPR1JFU1MsXG4gICAgICAgIFN0YWNrU3RhdHVzLklNUE9SVF9DT01QTEVURSxcbiAgICAgICAgU3RhY2tTdGF0dXMuSU1QT1JUX1JPTExCQUNLX0lOX1BST0dSRVNTLFxuICAgICAgICBTdGFja1N0YXR1cy5JTVBPUlRfUk9MTEJBQ0tfRkFJTEVELFxuICAgICAgICBTdGFja1N0YXR1cy5JTVBPUlRfUk9MTEJBQ0tfQ09NUExFVEUsXG4gICAgICBdLFxuICAgICAgYXNzZXJ0U3RhdHVzOiBTdGFja1N0YXR1cy5DUkVBVEVfQ09NUExFVEUsXG4gICAgfSwgdGhpcy5wcm9wcyk7XG4gIH1cblxufSJdfQ==