# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pypfopt', 'pypfopt.efficient_frontier']

package_data = \
{'': ['*']}

install_requires = \
['cvxpy>=1.1.10,<2.0.0',
 'numpy>=1.22.4,<2.0.0',
 'pandas>=0.19',
 'scipy>=1.3,<2.0']

extras_require = \
{'optionals': ['matplotlib>=3.5.2,<4.0.0', 'scikit-learn>=1.1.1,<2.0.0']}

setup_kwargs = {
    'name': 'pyportfolioopt',
    'version': '1.5.4',
    'description': 'Financial portfolio optimization in python',
    'long_description': '<p align="center">\n    <img width=60% src="https://github.com/robertmartin8/PyPortfolioOpt/blob/master/media/logo_v1.png?raw=true">\n</p>\n\n<!-- buttons -->\n<p align="center">\n    <a href="https://www.python.org/">\n        <img src="https://img.shields.io/badge/python-v3-brightgreen.svg"\n            alt="python"></a> &nbsp;\n    <a href="https://pypi.org/project/PyPortfolioOpt/">\n        <img src="https://img.shields.io/badge/pypi-v1.5.4-brightgreen.svg"\n            alt="pypi"></a> &nbsp;\n    <a href="https://opensource.org/licenses/MIT">\n        <img src="https://img.shields.io/badge/license-MIT-brightgreen.svg"\n            alt="MIT license"></a> &nbsp;\n    <a href="https://github.com/robertmartin8/PyPortfolioOpt/actions">\n        <img src="https://github.com/robertmartin8/PyPortfolioOpt/workflows/pytest/badge.svg?event=push"\n            alt="build"></a> &nbsp;\n    <a href="https://pepy.tech/project/pyportfolioopt">\n        <img src="https://pepy.tech/badge/pyportfolioopt"\n            alt="downloads"></a> &nbsp;\n    <a href="https://mybinder.org/v2/gh/robertmartin8/pyportfolioopt/master/?filepath=cookbook">\n      <img src="https://mybinder.org/badge_logo.svg"\n            alt="binder"></a> &nbsp;\n</p>\n\n<!-- content -->\n**PyPortfolioOpt is looking for maintainers! Please reach out to the email address at the bottom of the readme if you\'re interested**\n\nPyPortfolioOpt is a library that implements portfolio optimization methods, including\nclassical mean-variance optimization techniques and Black-Litterman allocation, as well as more\nrecent developments in the field like shrinkage and Hierarchical Risk Parity.\n\nIt is **extensive** yet easily **extensible**, and can be useful for either a casual investors, or a professional looking for an easy prototyping tool. Whether you are a fundamentals-oriented investor who has identified a\nhandful of undervalued picks, or an algorithmic trader who has a basket of\nstrategies, PyPortfolioOpt can help you combine your alpha sources\nin a risk-efficient way.\n\n**PyPortfolioOpt has been [published](https://joss.theoj.org/papers/10.21105/joss.03066) in the Journal of Open Source Software 🎉**\n\nHead over to the **[documentation on ReadTheDocs](https://pyportfolioopt.readthedocs.io/en/latest/)** to get an in-depth look at the project, or check out the [cookbook](https://github.com/robertmartin8/PyPortfolioOpt/tree/master/cookbook) to see some examples showing the full process from downloading data to building a portfolio.\n\n<center>\n<img src="https://github.com/robertmartin8/PyPortfolioOpt/blob/master/media/conceptual_flowchart_v2.png?raw=true" style="width:70%;"/>\n</center>\n\n## Table of contents\n\n-   [Table of contents](#table-of-contents)\n-   [Getting started](#getting-started)\n    -   [For development](#for-development)\n-   [A quick example](#a-quick-example)\n-   [An overview of classical portfolio optimization methods](#an-overview-of-classical-portfolio-optimization-methods)\n-   [Features](#features)\n    -   [Expected returns](#expected-returns)\n    -   [Risk models (covariance)](#risk-models-covariance)\n    -   [Objective functions](#objective-functions)\n    -   [Adding constraints or different objectives](#adding-constraints-or-different-objectives)\n    -   [Black-Litterman allocation](#black-litterman-allocation)\n    -   [Other optimizers](#other-optimizers)\n-   [Advantages over existing implementations](#advantages-over-existing-implementations)\n-   [Project principles and design decisions](#project-principles-and-design-decisions)\n-   [Testing](#testing)\n-   [Citing PyPortfolioOpt](#citing-pyportfolioopt)\n-   [Contributing](#contributing)\n-   [Getting in touch](#getting-in-touch)\n\n## Getting started\n\nIf you would like to play with PyPortfolioOpt interactively in your browser, you may launch Binder [here](https://mybinder.org/v2/gh/robertmartin8/pyportfolioopt/master). It takes a\nwhile to set up, but it lets you try out the cookbook recipes without having to deal with all of the requirements.\n\n_Note: macOS users will need to install [Command Line Tools](https://osxdaily.com/2014/02/12/install-command-line-tools-mac-os-x/)._\n\n_Note: if you are on windows, you first need to installl C++. ([download](https://visualstudio.microsoft.com/thank-you-downloading-visual-studio/?sku=BuildTools&rel=16), [install instructions](https://drive.google.com/file/d/0B4GsMXCRaSSIOWpYQkstajlYZ0tPVkNQSElmTWh1dXFaYkJr/view))_\n\nThis project is available on PyPI, meaning that you can just:\n\n```bash\npip install PyPortfolioOpt\n```\n\n(you may need to follow separate installation instructions for [cvxopt](https://cvxopt.org/install/index.html#) and [cvxpy](https://www.cvxpy.org/install/))).\n\nHowever, it is best practice to use a dependency manager within a virtual environment.\nMy current recommendation is to get yourself set up with [poetry](https://github.com/sdispater/poetry) then just run\n\n```bash\npoetry add PyPortfolioOpt\n```\n\nOtherwise, clone/download the project and in the project directory run:\n\n```bash\npython setup.py install\n```\n\nPyPortfolioOpt supports Docker. Build your first container with `docker build -f docker/Dockerfile . -t pypfopt`. You can use the image to run tests or even launch a Jupyter server.\n\n```bash\n# iPython interpreter:\ndocker run -it pypfopt poetry run ipython\n\n# Jupyter notebook server:\ndocker run -it -p 8888:8888 pypfopt poetry run jupyter notebook --allow-root --no-browser --ip 0.0.0.0\n# click on http://127.0.0.1:8888/?token=xxx\n\n# Pytest\ndocker run -t pypfopt poetry run pytest\n\n# Bash\ndocker run -it pypfopt bash\n```\n\nFor more information, please read [this guide](https://docker-curriculum.com/#introduction).\n\n### For development\n\nIf you would like to make major changes to integrate this with your proprietary system, it probably makes sense to clone this repository and to just use the source code.\n\n```bash\ngit clone https://github.com/robertmartin8/PyPortfolioOpt\n```\n\nAlternatively, you could try:\n\n```bash\npip install -e git+https://github.com/robertmartin8/PyPortfolioOpt.git\n```\n\n## A quick example\n\nHere is an example on real life stock data, demonstrating how easy it is to find the long-only portfolio that maximises the Sharpe ratio (a measure of risk-adjusted returns).\n\n```python\nimport pandas as pd\nfrom pypfopt import EfficientFrontier\nfrom pypfopt import risk_models\nfrom pypfopt import expected_returns\n\n# Read in price data\ndf = pd.read_csv("tests/resources/stock_prices.csv", parse_dates=True, index_col="date")\n\n# Calculate expected returns and sample covariance\nmu = expected_returns.mean_historical_return(df)\nS = risk_models.sample_cov(df)\n\n# Optimize for maximal Sharpe ratio\nef = EfficientFrontier(mu, S)\nraw_weights = ef.max_sharpe()\ncleaned_weights = ef.clean_weights()\nef.save_weights_to_file("weights.csv")  # saves to file\nprint(cleaned_weights)\nef.portfolio_performance(verbose=True)\n```\n\nThis outputs the following weights:\n\n```txt\n{\'GOOG\': 0.03835,\n \'AAPL\': 0.0689,\n \'FB\': 0.20603,\n \'BABA\': 0.07315,\n \'AMZN\': 0.04033,\n \'GE\': 0.0,\n \'AMD\': 0.0,\n \'WMT\': 0.0,\n \'BAC\': 0.0,\n \'GM\': 0.0,\n \'T\': 0.0,\n \'UAA\': 0.0,\n \'SHLD\': 0.0,\n \'XOM\': 0.0,\n \'RRC\': 0.0,\n \'BBY\': 0.01324,\n \'MA\': 0.35349,\n \'PFE\': 0.1957,\n \'JPM\': 0.0,\n \'SBUX\': 0.01082}\n\nExpected annual return: 30.5%\nAnnual volatility: 22.2%\nSharpe Ratio: 1.28\n```\n\nThis is interesting but not useful in itself. However, PyPortfolioOpt provides a method which allows you to convert the above continuous weights to an actual allocation that you could buy. Just enter the most recent prices, and the desired portfolio size ($10,000 in this example):\n\n```python\nfrom pypfopt.discrete_allocation import DiscreteAllocation, get_latest_prices\n\n\nlatest_prices = get_latest_prices(df)\n\nda = DiscreteAllocation(weights, latest_prices, total_portfolio_value=10000)\nallocation, leftover = da.greedy_portfolio()\nprint("Discrete allocation:", allocation)\nprint("Funds remaining: ${:.2f}".format(leftover))\n```\n\n```txt\n12 out of 20 tickers were removed\nDiscrete allocation: {\'GOOG\': 1, \'AAPL\': 4, \'FB\': 12, \'BABA\': 4, \'BBY\': 2,\n                      \'MA\': 20, \'PFE\': 54, \'SBUX\': 1}\nFunds remaining: $11.89\n```\n\n_Disclaimer: nothing about this project constitues investment advice, and the author bears no responsibiltiy for your subsequent investment decisions. Please refer to the [license](https://github.com/robertmartin8/PyPortfolioOpt/blob/master/LICENSE.txt) for more information._\n\n## An overview of classical portfolio optimization methods\n\nHarry Markowitz\'s 1952 paper is the undeniable classic, which turned portfolio optimization from an art into a science. The key insight is that by combining assets with different expected returns and volatilities, one can decide on a mathematically optimal allocation which minimises the risk for a target return – the set of all such optimal portfolios is referred to as the **efficient frontier**.\n\n<center>\n<img src="https://github.com/robertmartin8/PyPortfolioOpt/blob/master/media/efficient_frontier_white.png?raw=true" style="width:60%;"/>\n</center>\n\nAlthough much development has been made in the subject, more than half a century later, Markowitz\'s core ideas are still fundamentally important and see daily use in many portfolio management firms.\nThe main drawback of mean-variance optimization is that the theoretical treatment requires knowledge of the expected returns and the future risk-characteristics (covariance) of the assets. Obviously, if we knew the expected returns of a stock life would be much easier, but the whole game is that stock returns are notoriously hard to forecast. As a substitute, we can derive estimates of the expected return and covariance based on historical data – though we do lose the theoretical guarantees provided by Markowitz, the closer our estimates are to the real values, the better our portfolio will be.\n\nThus this project provides four major sets of functionality (though of course they are intimately related)\n\n-   Estimates of expected returns\n-   Estimates of risk (i.e covariance of asset returns)\n-   Objective functions to be optimized\n-   Optimizers.\n\nA key design goal of PyPortfolioOpt is **modularity** – the user should be able to swap in their\ncomponents while still making use of the framework that PyPortfolioOpt provides.\n\n## Features\n\nIn this section, we detail some of PyPortfolioOpt\'s available functionality. More examples are offered in the Jupyter notebooks [here](https://github.com/robertmartin8/PyPortfolioOpt/tree/master/cookbook). Another good resource is the [tests](https://github.com/robertmartin8/PyPortfolioOpt/tree/master/tests).\n\nA far more comprehensive version of this can be found on [ReadTheDocs](https://pyportfolioopt.readthedocs.io/en/latest/), as well as possible extensions for more advanced users.\n\n### Expected returns\n\n-   Mean historical returns:\n    -   the simplest and most common approach, which states that the expected return of each asset is equal to the mean of its historical returns.\n    -   easily interpretable and very intuitive\n-   Exponentially weighted mean historical returns:\n    -   similar to mean historical returns, except it gives exponentially more weight to recent prices\n    -   it is likely the case that an asset\'s most recent returns hold more weight than returns from 10 years ago when it comes to estimating future returns.\n-   Capital Asset Pricing Model (CAPM):\n    -   a simple model to predict returns based on the beta to the market\n    -   this is used all over finance!\n\n### Risk models (covariance)\n\nThe covariance matrix encodes not just the volatility of an asset, but also how it correlated to other assets. This is important because in order to reap the benefits of diversification (and thus increase return per unit risk), the assets in the portfolio should be as uncorrelated as possible.\n\n-   Sample covariance matrix:\n    -   an unbiased estimate of the covariance matrix\n    -   relatively easy to compute\n    -   the de facto standard for many years\n    -   however, it has a high estimation error, which is particularly dangerous in mean-variance optimization because the optimizer is likely to give excess weight to these erroneous estimates.\n-   Semicovariance: a measure of risk that focuses on downside variation.\n-   Exponential covariance: an improvement over sample covariance that gives more weight to recent data\n-   Covariance shrinkage: techniques that involve combining the sample covariance matrix with a structured estimator, to reduce the effect of erroneous weights. PyPortfolioOpt provides wrappers around the efficient vectorised implementations provided by `sklearn.covariance`.\n    -   manual shrinkage\n    -   Ledoit Wolf shrinkage, which chooses an optimal shrinkage parameter. We offer three shrinkage targets: `constant_variance`, `single_factor`, and `constant_correlation`.\n    -   Oracle Approximating Shrinkage\n-   Minimum Covariance Determinant:\n    -   a robust estimate of the covariance\n    -   implemented in `sklearn.covariance`\n\n<p align="center">\n    <img width=60% src="https://github.com/robertmartin8/PyPortfolioOpt/blob/master/media/corrplot_white.png?raw=true">\n</p>\n\n(This plot was generated using `plotting.plot_covariance`)\n\n### Objective functions\n\n-   Maximum Sharpe ratio: this results in a _tangency portfolio_ because on a graph of returns vs risk, this portfolio corresponds to the tangent of the efficient frontier that has a y-intercept equal to the risk-free rate. This is the default option because it finds the optimal return per unit risk.\n-   Minimum volatility. This may be useful if you\'re trying to get an idea of how low the volatility _could_ be, but in practice it makes a lot more sense to me to use the portfolio that maximises the Sharpe ratio.\n-   Efficient return, a.k.a. the Markowitz portfolio, which minimises risk for a given target return – this was the main focus of Markowitz 1952\n-   Efficient risk: the Sharpe-maximising portfolio for a given target risk.\n-   Maximum quadratic utility. You can provide your own risk-aversion level and compute the appropriate portfolio.\n\n### Adding constraints or different objectives\n\n-   Long/short: by default all of the mean-variance optimization methods in PyPortfolioOpt are long-only, but they can be initialised to allow for short positions by changing the weight bounds:\n\n```python\nef = EfficientFrontier(mu, S, weight_bounds=(-1, 1))\n```\n\n-   Market neutrality: for the `efficient_risk` and `efficient_return` methods, PyPortfolioOpt provides an option to form a market-neutral portfolio (i.e weights sum to zero). This is not possible for the max Sharpe portfolio and the min volatility portfolio because in those cases because they are not invariant with respect to leverage. Market neutrality requires negative weights:\n\n```python\nef = EfficientFrontier(mu, S, weight_bounds=(-1, 1))\nef.efficient_return(target_return=0.2, market_neutral=True)\n```\n\n-   Minimum/maximum position size: it may be the case that you want no security to form more than 10% of your portfolio. This is easy to encode:\n\n```python\nef = EfficientFrontier(mu, S, weight_bounds=(0, 0.1))\n```\n\nOne issue with mean-variance optimization is that it leads to many zero-weights. While these are\n"optimal" in-sample, there is a large body of research showing that this characteristic leads\nmean-variance portfolios to underperform out-of-sample. To that end, I have introduced an\nobjective function that can reduce the number of negligible weights for any of the objective functions. Essentially, it adds a penalty (parameterised by `gamma`) on small weights, with a term that looks just like L2 regularisation in machine learning. It may be necessary to try several `gamma` values to achieve the desired number of non-negligible weights. For the test portfolio of 20 securities, `gamma ~ 1` is sufficient\n\n```python\nef = EfficientFrontier(mu, S)\nef.add_objective(objective_functions.L2_reg, gamma=1)\nef.max_sharpe()\n```\n\n### Black-Litterman allocation\n\nAs of v0.5.0, we now support Black-Litterman asset allocation, which allows you to combine\na prior estimate of returns (e.g the market-implied returns) with your own views to form a\nposterior estimate. This results in much better estimates of expected returns than just using\nthe mean historical return. Check out the [docs](https://pyportfolioopt.readthedocs.io/en/latest/BlackLitterman.html) for a discussion of the theory, as well as advice\non formatting inputs.\n\n```python\nS = risk_models.sample_cov(df)\nviewdict = {"AAPL": 0.20, "BBY": -0.30, "BAC": 0, "SBUX": -0.2, "T": 0.131321}\nbl = BlackLittermanModel(S, pi="equal", absolute_views=viewdict, omega="default")\nrets = bl.bl_returns()\n\nef = EfficientFrontier(rets, S)\nef.max_sharpe()\n```\n\n### Other optimizers\n\nThe features above mostly pertain to solving mean-variance optimization problems via quadratic programming (though this is taken care of by `cvxpy`). However, we offer different optimizers as well:\n\n-   Mean-semivariance optimization\n-   Mean-CVaR optimization\n-   Hierarchical Risk Parity, using clustering algorithms to choose uncorrelated assets\n-   Markowitz\'s critical line algorithm (CLA)\n\nPlease refer to the [documentation](https://pyportfolioopt.readthedocs.io/en/latest/OtherOptimizers.html) for more.\n\n## Advantages over existing implementations\n\n-   Includes both classical methods (Markowitz 1952 and Black-Litterman), suggested best practices\n    (e.g covariance shrinkage), along with many recent developments and novel\n    features, like L2 regularisation, shrunk covariance, hierarchical risk parity.\n-   Native support for pandas dataframes: easily input your daily prices data.\n-   Extensive practical tests, which use real-life data.\n-   Easy to combine with your proprietary strategies and models.\n-   Robust to missing data, and price-series of different lengths (e.g FB data\n    only goes back to 2012 whereas AAPL data goes back to 1980).\n\n## Project principles and design decisions\n\n-   It should be easy to swap out individual components of the optimization process\n    with the user\'s proprietary improvements.\n-   Usability is everything: it is better to be self-explanatory than consistent.\n-   There is no point in portfolio optimization unless it can be practically\n    applied to real asset prices.\n-   Everything that has been implemented should be tested.\n-   Inline documentation is good: dedicated (separate) documentation is better.\n    The two are not mutually exclusive.\n-   Formatting should never get in the way of coding: because of this,\n    I have deferred **all** formatting decisions to [Black](https://github.com/ambv/black).\n\n## Testing\n\nTests are written in pytest (much more intuitive than `unittest` and the variants in my opinion), and I have tried to ensure close to 100% coverage. Run the tests by navigating to the package directory and simply running `pytest` on the command line.\n\nPyPortfolioOpt provides a test dataset of daily returns for 20 tickers:\n\n```python\n[\'GOOG\', \'AAPL\', \'FB\', \'BABA\', \'AMZN\', \'GE\', \'AMD\', \'WMT\', \'BAC\', \'GM\',\n\'T\', \'UAA\', \'SHLD\', \'XOM\', \'RRC\', \'BBY\', \'MA\', \'PFE\', \'JPM\', \'SBUX\']\n```\n\nThese tickers have been informally selected to meet several criteria:\n\n-   reasonably liquid\n-   different performances and volatilities\n-   different amounts of data to test robustness\n\nCurrently, the tests have not explored all of the edge cases and combinations\nof objective functions and parameters. However, each method and parameter has\nbeen tested to work as intended.\n\n## Citing PyPortfolioOpt\n\nIf you use PyPortfolioOpt for published work, please cite the [JOSS paper](https://joss.theoj.org/papers/10.21105/joss.03066).\n\nCitation string:\n\n```text\nMartin, R. A., (2021). PyPortfolioOpt: portfolio optimization in Python. Journal of Open Source Software, 6(61), 3066, https://doi.org/10.21105/joss.03066\n```\n\nBibTex::\n\n```bibtex\n@article{Martin2021,\n  doi = {10.21105/joss.03066},\n  url = {https://doi.org/10.21105/joss.03066},\n  year = {2021},\n  publisher = {The Open Journal},\n  volume = {6},\n  number = {61},\n  pages = {3066},\n  author = {Robert Andrew Martin},\n  title = {PyPortfolioOpt: portfolio optimization in Python},\n  journal = {Journal of Open Source Software}\n}\n```\n\n## Contributing\n\nContributions are _most welcome_. Have a look at the [Contribution Guide](https://github.com/robertmartin8/PyPortfolioOpt/blob/master/CONTRIBUTING.md) for more.\n\nI\'d like to thank all of the people who have contributed to PyPortfolioOpt since its release in 2018.\nSpecial shout-outs to:\n\n-   Philipp Schiele\n-   Carl Peasnell\n-   Felipe Schneider\n-   Dingyuan Wang\n-   Pat Newell\n-   Aditya Bhutra\n-   Thomas Schmelzer\n-   Rich Caputo\n-   Nicolas Knudde\n\n## Getting in touch\n\nIf you are having a problem with PyPortfolioOpt, please raise a GitHub issue. For anything else, you can reach me at:\n\n<center>\n<img src="https://github.com/robertmartin8/ReasonableDeviations/blob/gh-pages/assets/images/contact.png?raw=true" style="width:60%;"/>\n</center>\n',
    'author': 'Robert Andrew Martin',
    'author_email': 'martin.robertandrew@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/robertmartin8/PyPortfolioOpt',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
