# Shadertoy API for Python

![Python Versions][pyversions-shield]
[![license][license-button]][license-link]

[pyversions-shield]: https://img.shields.io/badge/python-3.5%20%7C%203.6%20%7C%203.7-blue
[license-button]: https://img.shields.io/badge/license-CC%20BY%20SA%204.0-green
[license-link]: https://creativecommons.org/licenses/by-sa/4.0/

shadertoy-api is a library for easily accessing the Shadertoy API.\
For more info about the Shadertoy API, refer to [shadertoy.com/howto](https://www.shadertoy.com/howto)\
Note: only shaders listed as Public+API are available through the Shadertoy API.\
Also, I (the author of this library) don't have any connections with the shadertoy
development team, in case you are wondering.

## Using the Shadertoy API

To start using the Shadertoy API you first need to create an instance of the App class provided by shadertoy-api

``` python
>>> import shadertoy
>>> app = shadertoy.App(api_key)
```

To get an API key for your application you must have an account on [shadertoy](https://www.shadertoy.com) (go to [shadertoy.com/signin](https://www.shadertoy.com/signin)).\
Once logged in, you can create an API key at [shadertoy.com/myapps](https://www.shadertoy.com/myapps)

The App class provides access to all documented features of the Shadertoy API as of the time this was written.

## Search the Shadertoy database

``` python
query(keywords=[], sort_by=None, filter=None, start_index=0, num_shaders="all")
```

> Queries the shadertoy database for shaders matching the given filter
  and returns a list with the given number of their IDs, starting at the
  given index and sorted by the given classifier.

Classifiers: "name", "love", "popular", "newest", "hot"\
Filters: "vr", "soundoutput", "soundinput", "webcam", "multipass", "musicstream"

A list of all classifiers and filters is provided by shadertoy-api

``` python
>>> shadertoy.classifiers
('name', 'love', 'popular', 'newest', 'hot')
>>> shadertoy.filters
('vr', 'soundoutput', 'soundinput', 'webcam', 'multipass', 'musicstream')
```

### Example

``` python
>>> app.query(keywords=["fluid", "simulation"], sort_by="newest", num_shaders=10)
['DsSSzm', 'fdcyzr', 'NldXWf', 'ftySzR', 'NtKGRc', '7tKGzK', '7tyGW1', 'slKGRw', '7ttGR4', 'sdd3zj']
```

## Get shader data (name, creator, description, likes, views, code, input media...)

``` python
get_shader(shader_id)
```

> Returns a dictionary containing data about the shader with the given ID.

### Example

``` python
>>> shader = app.get_shader("WdVXWy")
>>> shader["info"]["name"]
'molten bismuth'
```

## Get all available shader IDs

``` python
get_all_shaders()
```

> Returns a list of the IDs of all available shaders.

### Example

``` python
>>> all_shaders = app.get_all_shaders()
>>> len(all_shaders)
25859
```

## Non-API features

shadertoy-api also provides some non-API utilities through the App class including
downloading shadertoy media files and shader icons, getting URLs for embedding
shaders in webapps, and possibly more in the future!

## Download Shadertoy media files

``` python
download_media_file(path)
```

> Downloads a shadertoy media file from the given path (relative to shadertoy.com)
  and returns it as a file-like io.BytesIO object.

### Example

``` python
>>> Image_input0 = shader["renderpass"][0]["inputs"][0]
>>> media = app.download_media_file(Image_input0["src"])
>>> type(media)
<class '_io.BytesIO'>
```

If you have [pillow](https://pypi.org/project/Pillow/) installed, try this as well:

``` python
>>> from PIL import Image
>>> img = Image.open(media)
>>> img.show()
```

## Download shader icons

``` python
download_shader_icon(shader_id)
```

> Downloads the icon for the shader with the given id and returns it as a
  file-like io.BytesIO object (note that shader icons are in JPEG format).

### Example

``` python
>>> icon = app.download_shader_icon("WdVXWy")
>>> type(icon)
<class '_io.BytesIO'>
```

With pillow installed, try this:
``` python
>>> from PIL import Image
>>> img = Image.open(icon)
>>> img.show()
```

## Get embeddable shader URLs

``` python
get_embeddable_url(shader_id, enable_gui=True, start_time=10, paused=True, muted=False)
```

> Returns an embeddable URL for the shader with the given id, and configures
  it with the given initial settings.

### Example

``` python
>>> embeddable_url = app.get_embeddable_url("WdVXWy")
>>> embeddable_url
'https://www.shadertoy.com/embed/WdVXWy?gui=true&t=10&paused=true&muted=false'
```