#
# Copyright The NOMAD Authors.
#
# This file is part of NOMAD.
# See https://nomad-lab.eu for further info.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

from phonopy.structure.atoms import PhonopyAtoms

def read_aims(filename):
    """Method to read FHI-aims geometry files in phonopy context."""

    lines = open(filename, 'r').readlines()

    cell = []
    is_frac = []
    positions = []
    symbols = []
    magmoms = []
    for line in lines:
        fields = line.split()
        if not len(fields):
            continue
        if fields[0] == "lattice_vector":
            vec = [float(x) for x in fields[1:4]]
            cell.append(vec)
        elif fields[0][0:4] == "atom":
            if fields[0] == "atom":
                frac = False
            elif fields[0] == "atom_frac":
                frac = True
            pos = [float(x) for x in fields[1:4]]
            sym = fields[4]
            is_frac.append(frac)
            positions.append(pos)
            symbols.append(sym)
            magmoms.append(None)
        # implicitly assuming that initial_moments line adhere to FHI-aims geometry.in specification,
        # i.e. two subsequent initial_moments lines do not occur
        # if they do, the value specified in the last line is taken here - without any warning
        elif fields[0] == "initial_moment":
            magmoms[-1] = float(fields[1])

    for (n,frac) in enumerate(is_frac):
        if frac:
            pos = [ sum( [ positions[n][l] * cell[l][i] for l in range(3) ] ) for i in range(3) ]
            positions[n] = pos
    if None in magmoms:
        atoms = PhonopyAtoms(cell=cell, symbols=symbols, positions=positions)
    else:
        atoms = PhonopyAtoms(cell=cell, symbols=symbols, positions=positions, magmoms=magmoms)

    return atoms


def write_aims(filename, atoms):
    """Method to write FHI-aims geometry files in phonopy context."""

    lines = ""
    lines += "# geometry.in for FHI-aims \n"
    lines += "# | generated by phonopy.FHIaims.write_aims() \n"

    lattice_vector_line = "lattice_vector " + "%16.16f "*3 + "\n"
    for vec in atoms.get_cell():
        lines += lattice_vector_line % tuple(vec)

    N = atoms.get_number_of_atoms()

    atom_line = "atom " + "%16.16f "*3 + "%s \n"
    positions = atoms.get_positions()
    symbols = atoms.get_chemical_symbols()

    initial_moment_line = "initial_moment %16.6f\n"
    magmoms = atoms.get_magnetic_moments()

    for n in range(N):
        lines += atom_line % (tuple(positions[n]) + (symbols[n],))
        if magmoms is not None:
            lines += initial_moment_line % magmoms[n]

    f = open(filename, 'w')
    f.write(lines)
    f.close()



class Atoms_with_forces(PhonopyAtoms):
    """ Hack to phonopy.atoms to maintain ASE compatibility also for forces."""

    def get_forces(self):
        return self.forces


def read_aims_output(filename):
    """ Read FHI-aims output and 
        return geometry, energy and forces from last self-consistency iteration"""

    lines = open(filename, 'r').readlines()

    l = 0
    N = 0
    while l < len(lines):
        line = lines[l]
        if "Number of atoms" in line:
            N = int(line.split()[5])
        elif "| Unit cell:" in line:
            cell = []
            for i in range(3):
                l += 1
                vec = [float(x) for x in lines[l].split()[1:4]]
                cell.append(vec)
        elif ("Atomic structure:" in line) or ("Updated atomic structure:" in line):
            if "Atomic structure:" in line:
                i_sym = 3
                i_pos_min = 4 ; i_pos_max = 7
            elif "Updated atomic structure:" in line:
                i_sym = 4
                i_pos_min = 1 ; i_pos_max = 4
            l += 1
            symbols = []
            positions = []
            for n in range(N):
               l += 1
               fields = lines[l].split()
               sym = fields[i_sym]
               pos = [float(x) for x in fields[i_pos_min:i_pos_max]]
               symbols.append(sym)
               positions.append(pos)
        elif "Total atomic forces" in line:
            forces = []
            for i in range(N):
                l += 1
                force = [float(x) for x in lines[l].split()[2:5]]
                forces.append(force)
        l += 1
    
    atoms = Atoms_with_forces(cell=cell, symbols=symbols, positions=positions)
    atoms.forces = forces

    return atoms
