"""Mapping Of EBUS Circuit Names To Human-readable Names."""


class CircuitMap(dict):
    """
    Mapping Of EBUS Circuit Names To Human-readable Names.

    >>> c = CircuitMap({
    ...     'broadcast': '*',
    ...     'bai': 'Heater',
    ...     'mc': 'Mixer',
    ...     'hwc': 'Water',
    ... })
    >>> for circuitname, humanname in c:
    ...     print(circuitname, '=', humanname)
    broadcast = *
    bai = Heater
    mc = Mixer
    hwc = Water
    >>> tuple(c.iter_circuits())
    ('broadcast', 'bai', 'mc', 'hwc')

    Custom mappigns are added via :any:`add()`:

    >>> c = CircuitMap()
    >>> c.add('bai', 'Heater')
    >>> c.add('boo', 'My Boo')
    >>> c.add('mc.4', 'Mixer Unit 2')
    >>> c.get_humanname('bai')
    'Heater'
    >>> c.get_humanname('bai.3')
    'Heater#3'
    >>> c.get_humanname('mc.4')
    'Mixer Unit 2'
    >>> c.get_humanname('unknown')
    'unknown'
    >>> c.get_humanname('unknown.4')
    'unknown.4'
    """

    def get_humanname(self, circuitname):
        """Return human-readable name for `circuitname`."""
        # lookup full name
        humanname = self._map.get(circuitname, None)
        # loopup basename
        if humanname is None and "." in circuitname:
            basename, suffix = circuitname.split(".")
            humanname = self._map.get(basename, None)
            if humanname is not None:
                humanname = f"{humanname}#{suffix}"
        # use circuitname as default
        if humanname is None:
            humanname = circuitname
        return humanname
