"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cloudformation_1 = require("@aws-cdk/aws-cloudformation");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * Fargate profiles allows an administrator to declare which pods run on
 * Fargate. This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 */
class FargateProfile extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        const role = (_a = props.podExecutionRole) !== null && _a !== void 0 ? _a : new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')]
        });
        let subnets;
        if (props.vpc) {
            const selection = (_b = props.subnetSelection) !== null && _b !== void 0 ? _b : { subnetType: ec2.SubnetType.PRIVATE };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error(`Fargate profile requires at least one selector`);
        }
        if (props.selectors.length > 5) {
            throw new Error(`Fargate profile supports up to five selectors`);
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new aws_cloudformation_1.CustomResource(this, 'Resource', {
            provider: provider.provider,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster._getKubectlCreationRoleArn(),
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: role.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.anyValue({ produce: () => this.tags.renderTags() })
                }
            }
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
    }
}
exports.FargateProfile = FargateProfile;
//# sourceMappingURL=data:application/json;base64,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