"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const k8s_resource_1 = require("./k8s-resource");
/**
 * Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 */
class AwsAuth extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_resource_1.KubernetesResource(this, 'manifest', {
            cluster: props.cluster,
            manifest: [
                {
                    apiVersion: "v1",
                    kind: "ConfigMap",
                    metadata: {
                        name: "aws-auth",
                        namespace: "kube-system"
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    }
                }
            ]
        });
    }
    /**
     * Adds the specified IAM role to the `system:masters` RBAC group, which means
     * that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add
     * @param username Optional user (defaults to the role ARN)
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters']
        });
    }
    /**
     * Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map
     * @param mapping Mapping to k8s user name and groups
     */
    addRoleMapping(role, mapping) {
        this.roleMappings.push({ role, mapping });
    }
    /**
     * Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map
     * @param mapping Mapping to k8s user name and groups
     */
    addUserMapping(user, mapping) {
        this.userMappings.push({ user, mapping });
    }
    /**
     * Additional AWS account to add to the aws-auth configmap.
     * @param accountId account number
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    synthesizeMapRoles() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => {
                var _a;
                return ({
                    rolearn: m.role.roleArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.role.roleArn,
                    groups: m.mapping.groups
                });
            }))
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => {
                var _a;
                return ({
                    userarn: m.user.userArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.user.userArn,
                    groups: m.mapping.groups
                });
            }))
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.accounts)
        });
    }
}
exports.AwsAuth = AwsAuth;
//# sourceMappingURL=data:application/json;base64,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