# AUTOGENERATED! DO NOT EDIT! File to edit: 00_core.ipynb (unless otherwise specified).

__all__ = ['logger', 'tc', 'get_query_features_df', 'Curator', 'AbsBoundariesCurator', 'RelBoundariesCurator']

# Cell
import pandas as pd
import numpy as np
from scipy.optimize import linprog

import logging
logger = logging.getLogger(__name__)

import unittest
tc = unittest.TestCase('__init__')


# Cell
def get_query_features_df(df_samples, queries):
    """Apply queries on df_samples and return a boolean feature dataframe.

    If X = df_bool.values, then X[i, j] is True iff condition j is true for sample i
    """
    df_bool = df_samples[[]].copy()
    for query in queries:
        try:
            df_bool[query] = df_samples.eval(query)
        except Exception as e:
            print(query, e)
            raise
    df_bool.sum()
    return df_bool


# Cell
class Curator(object):
    def __init__(self, df, df_cond, dedup=True, allow_violations=True):
        """Init a curator object.
        :param df: dataframe of samples (one row per samples).
        :param df_cond: dataframe of indexed by queries that can be applied to df,
                        and the columns 'min', 'max', and 'index_ref' representing
                        required for each query boundaries.
        :param dedup: whether to combine rows that match the same set of queries.
                      If True (default), works faster, but slightly less accurate.
        :param allow_violations: Allow infeasible solutions (should generally be True).
        """

        self.df_bool = get_query_features_df(df, df_cond.index)
        self.df_cond = df_cond
        self.dedup = dedup
        self.allow_violations = allow_violations

        A = self.df_bool.values.astype('float').T
        b = df_cond.values

        if dedup:
            A, self.ix, self.cnt = np.unique(A, return_inverse=True, return_counts=True, axis=1)
            A = A * self.cnt

        self.n_constraints, self.n_samples = A.shape
        logger.info('#constraints=%d, #samples=%d' % A.shape)

        self.linprog_params = self.get_LP_params(A, b)

        for key, val in self.linprog_params.items():
            logger.debug(key, np.shape(val))

    @staticmethod
    def get_abs_bounds(b, cnt=None):
        """Convert bounds from relative fractions to absolute quantities.

           :param b: a matrix of shape (n_constraints, 3) where each row (l, u, j)
                     if j = -1:  the constraint is    "between l and u"
                     otherwise:  the constraint is    "between l*y_j and u*y_j"
                                 where y_j is the number of samples that satisfy query j.

           :param cnt: if cnt is not None, y_j = cnt[j]. Otherwise, infer the bounds by looking
                       at the l and u values of b[j, :].
        """

        for i, j in enumerate(b[:, 2]):
            j = int(j)
            if  j != -1:
                b[i, 0] = (b[j, 0] if cnt is None else cnt[j]) * b[i, 0]
                b[i, 1] = (b[j, 1] if cnt is None else cnt[j]) * b[i, 1]
        return b.round().astype('int')

    def get_LP_params(self, A, b):
        """Returns a dictionary with the arguments to scipy.optimize.linprog"""
        raise NotImplementedError

    def decode_solution(self):
        """Returns a boolean vector of size n_samples, indicating chosen samples."""

        if not self.dedup:
            # Original samples, just round.
            included = self.solution.x[:self.n_samples].round().astype('int')
        else:
            # Original x counts fraction (between 0-1) of samples to take from each group.
            # Convert it to (integer) number of samples to take from each group:
            x = (self.solution.x[:self.n_samples] * self.cnt).round().astype('int')

            # Randomly choose from each group:
            included = np.zeros((len(self.ix),), dtype='bool')
            for g, cnt_g in enumerate(x):
                all_members = (self.ix == g).nonzero()[0]
                included_members = np.random.choice(all_members, cnt_g, replace=False)
                included[included_members] = True

        self.included = included
        print('included:', included.sum())
        return included

    def get_summary(self, included):
        """Get summary of the queries, boundaries, and the violations."""

        cnt = self.df_bool[included.astype('bool')].sum()
        summary_df = cnt.to_frame('cnt')

        b = self.get_abs_bounds(self.df_cond.values, cnt=cnt)
        summary_df['min'] = b[:, 0]
        summary_df['max'] = b[:, 1]
        summary_df['violation'] = pd.DataFrame([summary_df['min'] - summary_df['cnt'],
                                                summary_df['cnt'] - summary_df['max']]).max().clip(0, None)
        print('actual violations:', summary_df['violation'].sum())
        return summary_df

    def run(self, method='revised simplex'):
        """Apply the LP. Use method='interior-point' for faster and less accurate solution."""

        included = summary_df = None
        self.solution = linprog(method=method, **self.linprog_params)
        logger.info(self.solution.message)

        if self.solution.success:
            print("Theoretical violations:", self.solution.fun)
            included = self.decode_solution()
            summary_df = self.get_summary(included)
        else:
            logger.error("Could not find solution.")

        return included, summary_df

# Cell
class AbsBoundariesCurator(Curator):
    def get_LP_params(self, A, b):
        n_constraints, n_samples = A.shape
        b = self.get_abs_bounds(b)

        bounds = [(0, 1)] * n_samples
        c = [0] * n_samples

        # Upper bound
        b_ub = b[:, 1]
        b_lb = b[:, 0]
        b_ub = np.hstack((b_ub, -b_lb))

        A_ub = np.vstack([A, # A * x <= b
                         -A])# A * x >= a ==> -A * x <= -a

        if self.allow_violations: # Support non-feasible scenarios (pay penalty)
            bounds += [(0, None)] * n_constraints
            c += [1] * n_constraints
            C = np.eye(n_constraints)
            A_ub = np.hstack((A_ub, np.vstack((-C, -C))))

        return dict(c=c, A_ub=A_ub, b_ub=b_ub, bounds=bounds)


# Cell
class RelBoundariesCurator(Curator):
    def get_LP_params(self, A, b):
        n_constraints, n_samples = A.shape

        #             X's                          Y's
        bounds = [(0, 1)] * n_samples + [(0, None)] * n_constraints
        c = [0] * (n_samples          +    n_constraints)

        # Equalities
        Y = np.eye(n_constraints)
        A_eq = np.hstack((A, -Y))
        b_eq = np.zeros((n_constraints,))

        # Upper bounds
        Y_ub = Y.copy()
        Y_lb = Y.copy()

        b_ub = b[:, 1]
        b_lb = b[:, 0]
        for i, (a_i, b_i, j) in enumerate(b):
            if j != -1:
                Y_ub[i, int(j)] = -b_i
                Y_lb[i, int(j)] = -a_i
                b_ub[i] = b_lb[i] = 0
        b_ub = np.hstack((b_ub, -b_lb))

        A_ub = np.zeros((n_constraints*2, n_samples))
        A_ub = np.hstack((A_ub, np.vstack((Y_ub, -Y_lb))))


        if self.allow_violations: # Support non-feasible scenarios (pay penalty)
            bounds += [(0, None)] * n_constraints
            c += [1] * n_constraints
            C = np.eye(n_constraints)
            A_ub = np.hstack((A_ub, np.vstack((-C, -C))))
            A_eq = np.hstack((A_eq, np.zeros((n_constraints, n_constraints))))

        return dict(c=c, A_eq=A_eq, b_eq=b_eq, A_ub=A_ub, b_ub=b_ub, bounds=bounds)
