import { aws_lambda as lambda, aws_logs as logs, aws_iam as iam, aws_s3 as s3 } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * The properties of a new AZ ID to name mapping
 */
export interface AzIdToNameMappingProps {
    /**
     * The target AZ IDs for mapping. Defaults to values for 3 zones in us-east-1.
     *
     * @default ['use1-az2', 'use1-az4', 'use1-az6']
     */
    readonly azIds?: string[];
    /**
     * The prefix to use for the SSM parameter names. Defaults to `/az-mapping/`.
     *
     * @default '/az-mapping/'
     */
    readonly ssmParameterPrefix?: string;
    /**
     * The number of days to retain log events in CloudWatch logs. Defaults to 30 days.
     *
     * @default logs.RetentionDays.ONE_MONTH
     */
    readonly logRetention?: logs.RetentionDays;
    /**
     * The lambda function code to use for the custom resource.
     *
     * For most use cases this should be left as the default, but
     * in cases where the custom resource needs to be deployed through
     * something like CloudFormation StackSets you may need to source
     * the lambda function code from S3 or some other location because
     * the CDK cant upload the local code to the correct asset location
     * for the StackSet target accounts.
     *
     * You can use the included `AccessKeyFunctionCodeCache` class to
     * cache the lambda function code in S3 and create a cross
     * account access policy to allow the StackSet target accounts
     * to access the code.
     *
     * @default - default lambda function code
     */
    readonly lambdaCode?: lambda.Code;
}
export declare class AzIdToNameMapping extends Construct {
    /**
     * The resulting mapping of AZ IDs to names
     */
    readonly mapping: string;
    constructor(scope: Construct, id: string, props: AzIdToNameMappingProps);
}
export interface AzIdToNameMappingFunctionCodeCacheProps extends s3.BucketProps {
}
export declare class AzIdToNameMappingFunctionCodeCache extends s3.Bucket {
    /**
     * The lambda.Code object that represents the contents of the bucket.
     */
    readonly lambdaCode: lambda.Code;
    constructor(scope: Construct, id: string, props: AzIdToNameMappingFunctionCodeCacheProps);
    /**
     * Add access to the whole organization to get
     * the lambda function code from the bucket.
     *
     * @param principalOrgId The organization ID to require for any accounts accessing the bucket.
     */
    addOrgWideAccessPolicy(principalOrgId: string): iam.AddToResourcePolicyResult;
    /**
     * Add access to the specified accounts to get
     * the lambda function code from the bucket.
     *
     * @param principalAccountId The account ID to add access for.
     * @param principalOrgId (Optional) The organization ID to require for the account accessing the bucket.
     */
    addAccountAccessPolicy(principalAccountId: string, principalOrgId?: string): iam.AddToResourcePolicyResult;
    /**
     * Add access to the specified organizational units to get
     * the lambda function code from the bucket.
     *
     * @param principalOrgPaths The organizational unit paths to add access for.
     */
    addOrgOuAccessPolicy(principalOrgPaths: string[]): iam.AddToResourcePolicyResult;
}
