#!/usr/bin/env python

__author__ = "Vanessa Sochat"
__copyright__ = "Copyright 2022, Vanessa Sochat"
__license__ = "MPL 2.0"

import usrse
import usrse.main.endpoints as endpoints
from usrse.logger import setup_logger
import argparse
import sys
import os


def get_parser():
    parser = argparse.ArgumentParser(
        description="USRSE Client",
        formatter_class=argparse.RawTextHelpFormatter,
    )

    # Global Variables
    parser.add_argument(
        "--debug",
        dest="debug",
        help="use verbose logging to debug.",
        default=False,
        action="store_true",
    )

    parser.add_argument(
        "--quiet",
        dest="quiet",
        help="suppress additional output.",
        default=False,
        action="store_true",
    )

    parser.add_argument(
        "--baseurl",
        dest="baseurl",
        help="change the default baseurl to something else",
    )

    parser.add_argument(
        "--version",
        dest="version",
        help="show software version.",
        default=False,
        action="store_true",
    )

    description = "actions"
    subparsers = parser.add_subparsers(
        help="actions",
        title="actions",
        description=description,
        dest="command",
    )

    # print version and exit
    subparsers.add_parser("version", description="show software version")

    # List endpoints available
    subparsers.add_parser("list", description="list endpoints available")

    # Local shell with client loaded
    shell = subparsers.add_parser(
        "shell",
        description="shell into a Python session with a client.",
        formatter_class=argparse.RawTextHelpFormatter,
    )
    shell.add_argument(
        "--interpreter",
        "-i",
        dest="interpreter",
        help="python interpreter",
        choices=["ipython", "python", "bpython"],
        default="ipython",
    )

    # Basic argument is "get" to get a table or json of results!
    get = subparsers.add_parser(
        "get",
        description="get USRSE content",
        formatter_class=argparse.RawTextHelpFormatter,
    )
    get.add_argument(
        "content_type",
        help="content type\n%s" % "\n".join(endpoints.register_names),
    )
    get.add_argument("--json", help="output json", default=False, action="store_true")
    get.add_argument("--all", help="output json", default=False, action="store_true")
    get.add_argument(
        "--live",
        "-k",
        help="get a live and interactive table!",
        default=False,
        action="store_true",
    )
    get.add_argument(
        "--limit",
        help="limit of posts to show (defaults to show, use --all for all posts)",
        default=25,
        type=int,
    )
    get.add_argument("--outfile", "-o", help="write content to output file")
    return parser


def run():

    parser = get_parser()

    def help(return_code=0):
        version = usrse.__version__

        print("\nUSRSE Client v%s" % version)
        parser.print_help()
        sys.exit(return_code)

    # If the user didn't provide any arguments, show the full help
    if len(sys.argv) == 1:
        help()

    # If an error occurs while parsing the arguments, the interpreter will exit with value 2
    args, extra = parser.parse_known_args()

    if args.debug is True:
        os.environ["MESSAGELEVEL"] = "DEBUG"

    # Show the version and exit
    if args.command == "version" or args.version:
        print(usrse.__version__)
        sys.exit(0)

    setup_logger(
        quiet=args.quiet,
        debug=args.debug,
    )

    # retrieve subparser (with help) from parser
    helper = None
    subparsers_actions = [
        action
        for action in parser._actions
        if isinstance(action, argparse._SubParsersAction)
    ]
    for subparsers_action in subparsers_actions:
        for choice, subparser in subparsers_action.choices.items():
            if choice == args.command:
                helper = subparser
                break

    if args.command == "get":
        from .get import main
    elif args.command == "shell":
        from .shell import main
    elif args.command == "list":
        from .listing import main

    # Pass on to the correct parser
    return_code = 0
    try:
        main(args=args, parser=parser, extra=extra, subparser=helper)
        sys.exit(return_code)
    except UnboundLocalError:
        return_code = 1

    help(return_code)


if __name__ == "__main__":
    run()
